﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/logs/CloudWatchLogs_EXPORTS.h>
#include <aws/logs/model/OpenSearchResourceStatus.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace CloudWatchLogs {
namespace Model {

/**
 * <p>This structure contains information about the OpenSearch Service data access
 * policy used for this integration. The access policy defines the access controls
 * for the collection. This data access policy was automatically created as part of
 * the integration setup. For more information about OpenSearch Service data access
 * policies, see <a
 * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-data-access.html">Data
 * access control for Amazon OpenSearch Serverless</a> in the OpenSearch Service
 * Developer Guide.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/OpenSearchDataAccessPolicy">AWS
 * API Reference</a></p>
 */
class OpenSearchDataAccessPolicy {
 public:
  AWS_CLOUDWATCHLOGS_API OpenSearchDataAccessPolicy() = default;
  AWS_CLOUDWATCHLOGS_API OpenSearchDataAccessPolicy(Aws::Utils::Json::JsonView jsonValue);
  AWS_CLOUDWATCHLOGS_API OpenSearchDataAccessPolicy& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_CLOUDWATCHLOGS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the data access policy.</p>
   */
  inline const Aws::String& GetPolicyName() const { return m_policyName; }
  inline bool PolicyNameHasBeenSet() const { return m_policyNameHasBeenSet; }
  template <typename PolicyNameT = Aws::String>
  void SetPolicyName(PolicyNameT&& value) {
    m_policyNameHasBeenSet = true;
    m_policyName = std::forward<PolicyNameT>(value);
  }
  template <typename PolicyNameT = Aws::String>
  OpenSearchDataAccessPolicy& WithPolicyName(PolicyNameT&& value) {
    SetPolicyName(std::forward<PolicyNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This structure contains information about the status of this OpenSearch
   * Service resource.</p>
   */
  inline const OpenSearchResourceStatus& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = OpenSearchResourceStatus>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = OpenSearchResourceStatus>
  OpenSearchDataAccessPolicy& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_policyName;

  OpenSearchResourceStatus m_status;
  bool m_policyNameHasBeenSet = false;
  bool m_statusHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchLogs
}  // namespace Aws
