"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2025.12.3"
__checksum__ = "cb8cf4df8c36dee74873a7161bbc4505810eca2c7ec139e1a389e334deb81a16"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'amazon', b'android', b'app', b'audible', b'azure', b'bank', b'bing', b'boo', b'channel', b'chrome', b'dad', b'day', b'dev', b'eat', b'esq', b'fire', b'fly', b'foo', b'fujitsu', b'gle', b'gmail', b'google', b'hangout', b'hotmail', b'imdb', b'ing', b'insurance', b'kindle', b'meet', b'meme', b'microsoft', b'mov', b'new', b'nexus', b'office', b'page', b'phd', b'play', b'prime', b'prof', b'rsvp', b'search', b'silk', b'skype', b'windows', b'xbox', b'xn--cckwcxetd', b'xn--jlq480n2rg', b'youtube', b'zappos', b'zip'}  # noqa: E501
_JUMPTABLE = [[(0, 4), (4, 10), (14, 19), (33, 42), (75, 26), (101, 12), (113, 8), (121, 19), (140, 22), (162, 7), (169, 13), (182, 25), None, (207, 23), (230, 37), (267, 19), (286, 9), (295, 36), (331, 14), (345, 10), (355, 18), None, (373, 62), (435, 8), (443, 28), (471, 19), (490, 13), (503, 14), (517, 7), None, None, (524, 33), (557, 20), (577, 25), (602, 14), (616, 14), (630, 28), None, (658, 18), (676, 30), (706, 8), (714, 13), (727, 10), (737, 8), (745, 17), None, (762, 26), (788, 16), (804, 5), (809, 10), (819, 14), (833, 11), (844, 12), (856, 27), (883, 7), (890, 11), (901, 5), (906, 5), (911, 20), (931, 8), (939, 27), (966, 46), (1012, 25), (1037, 16), (1053, 8), (1061, 5), (1066, 22), (1088, 18), None, (1106, 32), (1138, 15), (1153, 24), (1177, 11), None, (1188, 10), (1198, 19), (1217, 18), (1235, 18), (1253, 5), (1258, 20), (1278, 18), (1296, 35), (1331, 19), (1350, 20), (1370, 46), (1416, 12), (1428, 14), (1442, 8), (1450, 13), (1463, 10), (1473, 13), (1486, 20), (1506, 19), None, (1525, 13), (1538, 26), (1564, 11), (1575, 4), (1579, 22), (1601, 19), (1620, 15), (1635, 14), (1649, 19), (1668, 11), (1679, 21), (1700, 12), (1712, 25), None, (1737, 10), (1747, 14), (1761, 26), (1787, 45), (1832, 15), None, (1847, 11), (1858, 30), (1888, 21), (1909, 26), (1935, 6), (1941, 6), (1947, 20), (1967, 5), (1972, 38), (2010, 23), (2033, 6), (2039, 13), (2052, 8), (2060, 19), (2079, 12), (2091, 70), (2161, 55), None, (2216, 23), (2239, 16), (2255, 21), None, (2276, 24), (2300, 32), (2332, 6), (2338, 41), (2379, 22), (2401, 23), (2424, 23), (2447, 12), (2459, 8), (2467, 24), (2491, 12), (2503, 32), (2535, 25), (2560, 15), None, (2575, 46), (2621, 28), (2649, 13), (2662, 14), (2676, 26), (2702, 5), (2707, 42), (2749, 30), (2779, 16), (2795, 33), (2828, 17), (2845, 23), (2868, 14), (2882, 25), (2907, 19), (2926, 14), (2940, 7), (2947, 37), None, (2984, 18), (3002, 32), (3034, 27), (3061, 17), (3078, 24), (3102, 12), (3114, 28), (3142, 35), (3177, 12), (3189, 57), (3246, 32), (3278, 32), None, (3310, 8), (3318, 25), (3343, 18), (3361, 6), (3367, 23), None, (3390, 36), (3426, 33), (3459, 14), (3473, 16), (3489, 22), None, (3511, 20), (3531, 45), (3576, 34), (3610, 15), (3625, 20), (3645, 16), (3661, 21), (3682, 32), (3714, 24), (3738, 20), (3758, 10), (3768, 44), (3812, 6), (3818, 9), (3827, 12), (3839, 18), (3857, 5), (3862, 10), (3872, 39), (3911, 36), None, (3947, 25), (3972, 17), None, (3989, 14), (4003, 8), (4011, 7), None, (4018, 25), (4043, 17), None, (4060, 21), (4081, 35), (4116, 21), (4137, 10), (4147, 48), (4195, 11), (4206, 37), (4243, 41), (4284, 23), (4307, 12), (4319, 14), (4333, 23), (4356, 29), (4385, 14), (4399, 8), (4407, 47), (4454, 35), None, None, (4489, 47), (4536, 42), None, (4578, 18), None, (4596, 20), (4616, 8), (4624, 34), (4658, 6), (4664, 27), (4691, 22)], [(4713, 5319), (10032, 5069), (15101, 5625), (20726, 6010), (26736, 6007), (32743, 5314), (38057, 5998), (44055, 5585), (49640, 5272), (54912, 4871), (59783, 5183), (64966, 5419), (70385, 4820), (75205, 5783), (80988, 5478), (86466, 5626), (92092, 5303), (97395, 4930), (102325, 5339), (107664, 5198), (112862, 5286), (118148, 5320), (123468, 5754), (129222, 5145), (134367, 5523), (139890, 5079), (144969, 5410), (150379, 5511), (155890, 5339), (161229, 5319), (166548, 5223), (171771, 6002), (177773, 5467), (183240, 4985), (188225, 5075), (193300, 6018), (199318, 5430), (204748, 5815), (210563, 5835), (216398, 5923), (222321, 6313), (228634, 4869), (233503, 5604), (239107, 5398), (244505, 4748), (249253, 5439), (254692, 5009), (259701, 6182), (265883, 5111), (270994, 5128), (276122, 5824), (281946, 5270), (287216, 5252), (292468, 5533), (298001, 4982), (302983, 4964), (307947, 5942), (313889, 5716), (319605, 5439), (325044, 5074), (330118, 4896), (335014, 5621), (340635, 5604), (346239, 5607), (351846, 5957), (357803, 4668), (362471, 5471), (367942, 5899), (373841, 5809), (379650, 5044), (384694, 5219), (389913, 5519), (395432, 5108), (400540, 5486), (406026, 5305), (411331, 6025), (417356, 5179), (422535, 5819), (428354, 5570), (433924, 4978), (438902, 5132), (444034, 5236), (449270, 5114), (454384, 5450), (459834, 5680), (465514, 5591), (471105, 5733), (476838, 5220), (482058, 4970), (487028, 5457), (492485, 6088), (498573, 5879), (504452, 4900), (509352, 5312), (514664, 5446), (520110, 4853), (524963, 5826), (530789, 4703), (535492, 5892), (541384, 5185), (546569, 5514), (552083, 5125), (557208, 5451), (562659, 5368), (568027, 4679), (572706, 5457), (578163, 5764), (583927, 5142), (589069, 5157), (594226, 4524), (598750, 5582), (604332, 5335), (609667, 4761), (614428, 5527), (619955, 5480), (625435, 5220), (630655, 5626), (636281, 5002), (641283, 5700), (646983, 5626), (652609, 5692), (658301, 5045), (663346, 5690), (669036, 5474), (674510, 5498), (680008, 5974), (685982, 5350), (691332, 5348), (696680, 5285), (701965, 5543), (707508, 6070), (713578, 5335), (718913, 5426), (724339, 5118), (729457, 5134), (734591, 5878), (740469, 5799), (746268, 6038), (752306, 5457), (757763, 5591), (763354, 5971), (769325, 5512), (774837, 5774), (780611, 5881), (786492, 5924), (792416, 4892), (797308, 5541), (802849, 5540), (808389, 5305), (813694, 5297), (818991, 5060), (824051, 5475), (829526, 5074), (834600, 5126), (839726, 5519), (845245, 5272), (850517, 5408), (855925, 5273), (861198, 5418), (866616, 5082), (871698, 5502), (877200, 5435), (882635, 5441), (888076, 5201), (893277, 4987), (898264, 5851), (904115, 5789), (909904, 4956), (914860, 5813), (920673, 5173), (925846, 4780), (930626, 5429), (936055, 5264), (941319, 5727), (947046, 5211), (952257, 5078), (957335, 6254), (963589, 4708), (968297, 5158), (973455, 4859), (978314, 7009), (985323, 5642), (990965, 5423), (996388, 5545), (1001933, 5689), (1007622, 5058), (1012680, 4919), (1017599, 6276), (1023875, 5461), (1029336, 5500), (1034836, 5621), (1040457, 5099), (1045556, 5207), (1050763, 4864), (1055627, 5406), (1061033, 4991), (1066024, 5166), (1071190, 5459), (1076649, 4966), (1081615, 5460), (1087075, 5733), (1092808, 5968), (1098776, 5604), (1104380, 5096), (1109476, 5621), (1115097, 5861), (1120958, 5990), (1126948, 5622), (1132570, 6420), (1138990, 5048), (1144038, 6568), (1150606, 5252), (1155858, 5484), (1161342, 4854), (1166196, 5930), (1172126, 5587), (1177713, 5295), (1183008, 5869), (1188877, 4985), (1193862, 5772), (1199634, 5064), (1204698, 4840), (1209538, 5310), (1214848, 5352), (1220200, 5647), (1225847, 5159), (1231006, 6168), (1237174, 5446), (1242620, 5582), (1248202, 5765), (1253967, 5610), (1259577, 5117), (1264694, 5241), (1269935, 5567), (1275502, 5036), (1280538, 5715), (1286253, 5081), (1291334, 4726), (1296060, 4737), (1300797, 5656), (1306453, 4978), (1311431, 5439), (1316870, 5211), (1322081, 5872), (1327953, 4957), (1332910, 6119), (1339029, 5862), (1344891, 5255), (1350146, 5199), (1355345, 5197), (1360542, 5466), (1366008, 5559), (1371567, 4931), (1376498, 5275), (1381773, 5324), (1387097, 4978)], [(1392075, 786), (1392861, 470), (1393331, 748), (1394079, 710), (1394789, 633), (1395422, 628), (1396050, 539), (1396589, 540), (1397129, 553), (1397682, 613), (1398295, 442), (1398737, 546), (1399283, 689), (1399972, 745), (1400717, 589), (1401306, 706), (1402012, 1219), (1403231, 739), (1403970, 602), (1404572, 575), (1405147, 690), (1405837, 514), (1406351, 400), (1406751, 683), (1407434, 649), (1408083, 603), (1408686, 710), (1409396, 1155), (1410551, 644), (1411195, 575), (1411770, 471), (1412241, 654), (1412895, 523), (1413418, 616), (1414034, 679), (1414713, 705), (1415418, 454), (1415872, 634), (1416506, 478), (1416984, 653), (1417637, 570), (1418207, 542), (1418749, 448), (1419197, 657), (1419854, 544), (1420398, 570), (1420968, 541), (1421509, 533), (1422042, 428), (1422470, 476), (1422946, 593), (1423539, 496), (1424035, 824), (1424859, 847), (1425706, 1087), (1426793, 625), (1427418, 772), (1428190, 593), (1428783, 415), (1429198, 616), (1429814, 750), (1430564, 570), (1431134, 891), (1432025, 784), (1432809, 786), (1433595, 650), (1434245, 773), (1435018, 556), (1435574, 477), (1436051, 744), (1436795, 632), (1437427, 601), (1438028, 239), (1438267, 608), (1438875, 527), (1439402, 544), (1439946, 422), (1440368, 699), (1441067, 745), (1441812, 489), (1442301, 547), (1442848, 571), (1443419, 659), (1444078, 668), (1444746, 647), (1445393, 683), (1446076, 632), (1446708, 524), (1447232, 692), (1447924, 588), (1448512, 958), (1449470, 847), (1450317, 584), (1450901, 528), (1451429, 617), (1452046, 375), (1452421, 704), (1453125, 538), (1453663, 408), (1454071, 655), (1454726, 557), (1455283, 576), (1455859, 433), (1456292, 532), (1456824, 499), (1457323, 451), (1457774, 564), (1458338, 451), (1458789, 503), (1459292, 583), (1459875, 603), (1460478, 864), (1461342, 908), (1462250, 664), (1462914, 550), (1463464, 472), (1463936, 559), (1464495, 726), (1465221, 552), (1465773, 566), (1466339, 862), (1467201, 569), (1467770, 639), (1468409, 825), (1469234, 648), (1469882, 767), (1470649, 542), (1471191, 581), (1471772, 513), (1472285, 743), (1473028, 608), (1473636, 528), (1474164, 581), (1474745, 571), (1475316, 443), (1475759, 619), (1476378, 732), (1477110, 585), (1477695, 650), (1478345, 774), (1479119, 644), (1479763, 471), (1480234, 583), (1480817, 396), (1481213, 620), (1481833, 607), (1482440, 741), (1483181, 358), (1483539, 682), (1484221, 711), (1484932, 502), (1485434, 450), (1485884, 665), (1486549, 668), (1487217, 737), (1487954, 567), (1488521, 372), (1488893, 724), (1489617, 733), (1490350, 696), (1491046, 740), (1491786, 515), (1492301, 764), (1493065, 618), (1493683, 730), (1494413, 743), (1495156, 570), (1495726, 669), (1496395, 515), (1496910, 537), (1497447, 553), (1498000, 466), (1498466, 1001), (1499467, 665), (1500132, 855), (1500987, 698), (1501685, 609), (1502294, 608), (1502902, 696), (1503598, 593), (1504191, 1706), (1505897, 600), (1506497, 586), (1507083, 601), (1507684, 735), (1508419, 494), (1508913, 534), (1509447, 488), (1509935, 611), (1510546, 789), (1511335, 451), (1511786, 423), (1512209, 397), (1512606, 935), (1513541, 518), (1514059, 633), (1514692, 759), (1515451, 703), (1516154, 719), (1516873, 649), (1517522, 582), (1518104, 799), (1518903, 558), (1519461, 603), (1520064, 637), (1520701, 677), (1521378, 749), (1522127, 732), (1522859, 647), (1523506, 670), (1524176, 636), (1524812, 615), (1525427, 714), (1526141, 938), (1527079, 590), (1527669, 657), (1528326, 615), (1528941, 457), (1529398, 422), (1529820, 283), (1530103, 489), (1530592, 594), (1531186, 579), (1531765, 668), (1532433, 428), (1532861, 553), (1533414, 540), (1533954, 671), (1534625, 661), (1535286, 551), (1535837, 811), (1536648, 531), (1537179, 766), (1537945, 604), (1538549, 543), (1539092, 548), (1539640, 758), (1540398, 636), (1541034, 754), (1541788, 656), (1542444, 546), (1542990, 547), (1543537, 659), (1544196, 513), (1544709, 614), (1545323, 431), (1545754, 508), (1546262, 751), (1547013, 524), (1547537, 592), (1548129, 1938), (1550067, 487), (1550554, 812), (1551366, 595), (1551961, 590), (1552551, 747)], [(1553298, 48), None, None, (1553346, 42), None, (1553388, 27), (1553415, 25), None, None, None, None, None, None, None, None, None, None, (1553440, 42), (1553482, 20), None, (1553502, 44), None, None, (1553546, 18), None, (1553564, 23), None, None, None, None, None, None, (1553587, 21), (1553608, 25), None, None, (1553633, 26), None, None, None, None, (1553659, 44), (1553703, 21), (1553724, 23), None, None, None, None, None, None, None, None, None, None, (1553747, 31), None, None, None, None, (1553778, 42), (1553820, 19), (1553839, 16), None, (1553855, 21), None, None, (1553876, 42), None, None, (1553918, 42), (1553960, 27), None, None, None, None, (1553987, 37), (1554024, 21), None, None, (1554045, 20), (1554065, 42), None, None, None, (1554107, 25), (1554132, 17), None, (1554149, 21), None, None, None, (1554170, 24), None, (1554194, 24), (1554218, 21), None, None, (1554239, 22), None, None, (1554261, 17), (1554278, 19), None, None, None, None, (1554297, 31), None, None, None, None, None, None, None, None, (1554328, 42), (1554370, 42), (1554412, 17), (1554429, 17), None, None, (1554446, 19), None, (1554465, 25), None, None, (1554490, 20), None, None, (1554510, 42), (1554552, 63), None, None, None, (1554615, 21), None, None, None, None, (1554636, 21), (1554657, 16), None, (1554673, 31), None, None, None, None, (1554704, 42), None, (1554746, 25), None, (1554771, 9), None, (1554780, 21), (1554801, 42), None, (1554843, 18), (1554861, 65), (1554926, 56), (1554982, 21), None, (1555003, 42), None, None, (1555045, 24), None, None, None, None, None, None, (1555069, 42), (1555111, 21), (1555132, 21), None, (1555153, 42), (1555195, 25), None, (1555220, 37), (1555257, 21), (1555278, 42), None, None, (1555320, 21), (1555341, 16), None, None, (1555357, 16), None, (1555373, 51), None, None, (1555424, 21), None, (1555445, 22), (1555467, 21), (1555488, 21), None, None, (1555509, 63), (1555572, 41), (1555613, 39), (1555652, 42), None, None, None, None, None, None, (1555694, 21), (1555715, 21), None, None, (1555736, 21), None, None, (1555757, 21), None, None, None, (1555778, 50), None, None, None, (1555828, 50), None, (1555878, 21), (1555899, 21), (1555920, 19), None, (1555939, 16), (1555955, 26), None, (1555981, 58), (1556039, 42), None, None, None, None, None, None, (1556081, 41), None, None, None, (1556122, 21), None, None, (1556143, 44), None, (1556187, 254), (1556441, 21), None, None, (1556462, 21), None], [None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)

except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path),
            "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 5 deep.
            if layer > 4:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
