﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/vpc-lattice/VPCLatticeRequest.h>
#include <aws/vpc-lattice/VPCLattice_EXPORTS.h>
#include <aws/vpc-lattice/model/DnsOptions.h>

#include <utility>

namespace Aws {
namespace VPCLattice {
namespace Model {

/**
 */
class CreateServiceNetworkVpcAssociationRequest : public VPCLatticeRequest {
 public:
  AWS_VPCLATTICE_API CreateServiceNetworkVpcAssociationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateServiceNetworkVpcAssociation"; }

  AWS_VPCLATTICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. If you retry a request that completed successfully
   * using the same client token and parameters, the retry succeeds without
   * performing any actions. If the parameters aren't identical, the retry fails.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateServiceNetworkVpcAssociationRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID or ARN of the service network. You must use an ARN if the resources
   * are in different accounts.</p>
   */
  inline const Aws::String& GetServiceNetworkIdentifier() const { return m_serviceNetworkIdentifier; }
  inline bool ServiceNetworkIdentifierHasBeenSet() const { return m_serviceNetworkIdentifierHasBeenSet; }
  template <typename ServiceNetworkIdentifierT = Aws::String>
  void SetServiceNetworkIdentifier(ServiceNetworkIdentifierT&& value) {
    m_serviceNetworkIdentifierHasBeenSet = true;
    m_serviceNetworkIdentifier = std::forward<ServiceNetworkIdentifierT>(value);
  }
  template <typename ServiceNetworkIdentifierT = Aws::String>
  CreateServiceNetworkVpcAssociationRequest& WithServiceNetworkIdentifier(ServiceNetworkIdentifierT&& value) {
    SetServiceNetworkIdentifier(std::forward<ServiceNetworkIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the VPC.</p>
   */
  inline const Aws::String& GetVpcIdentifier() const { return m_vpcIdentifier; }
  inline bool VpcIdentifierHasBeenSet() const { return m_vpcIdentifierHasBeenSet; }
  template <typename VpcIdentifierT = Aws::String>
  void SetVpcIdentifier(VpcIdentifierT&& value) {
    m_vpcIdentifierHasBeenSet = true;
    m_vpcIdentifier = std::forward<VpcIdentifierT>(value);
  }
  template <typename VpcIdentifierT = Aws::String>
  CreateServiceNetworkVpcAssociationRequest& WithVpcIdentifier(VpcIdentifierT&& value) {
    SetVpcIdentifier(std::forward<VpcIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Indicates if private DNS is enabled for the VPC association. </p>
   */
  inline bool GetPrivateDnsEnabled() const { return m_privateDnsEnabled; }
  inline bool PrivateDnsEnabledHasBeenSet() const { return m_privateDnsEnabledHasBeenSet; }
  inline void SetPrivateDnsEnabled(bool value) {
    m_privateDnsEnabledHasBeenSet = true;
    m_privateDnsEnabled = value;
  }
  inline CreateServiceNetworkVpcAssociationRequest& WithPrivateDnsEnabled(bool value) {
    SetPrivateDnsEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IDs of the security groups. Security groups aren't added by default. You
   * can add a security group to apply network level controls to control which
   * resources in a VPC are allowed to access the service network and its services.
   * For more information, see <a
   * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html">Control
   * traffic to resources using security groups</a> in the <i>Amazon VPC User
   * Guide</i>.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSecurityGroupIds() const { return m_securityGroupIds; }
  inline bool SecurityGroupIdsHasBeenSet() const { return m_securityGroupIdsHasBeenSet; }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  void SetSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds = std::forward<SecurityGroupIdsT>(value);
  }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  CreateServiceNetworkVpcAssociationRequest& WithSecurityGroupIds(SecurityGroupIdsT&& value) {
    SetSecurityGroupIds(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  template <typename SecurityGroupIdsT = Aws::String>
  CreateServiceNetworkVpcAssociationRequest& AddSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds.emplace_back(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags for the association.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateServiceNetworkVpcAssociationRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateServiceNetworkVpcAssociationRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> DNS options for the service network VPC association. </p>
   */
  inline const DnsOptions& GetDnsOptions() const { return m_dnsOptions; }
  inline bool DnsOptionsHasBeenSet() const { return m_dnsOptionsHasBeenSet; }
  template <typename DnsOptionsT = DnsOptions>
  void SetDnsOptions(DnsOptionsT&& value) {
    m_dnsOptionsHasBeenSet = true;
    m_dnsOptions = std::forward<DnsOptionsT>(value);
  }
  template <typename DnsOptionsT = DnsOptions>
  CreateServiceNetworkVpcAssociationRequest& WithDnsOptions(DnsOptionsT&& value) {
    SetDnsOptions(std::forward<DnsOptionsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::String m_serviceNetworkIdentifier;

  Aws::String m_vpcIdentifier;

  bool m_privateDnsEnabled{false};

  Aws::Vector<Aws::String> m_securityGroupIds;

  Aws::Map<Aws::String, Aws::String> m_tags;

  DnsOptions m_dnsOptions;
  bool m_clientTokenHasBeenSet = true;
  bool m_serviceNetworkIdentifierHasBeenSet = false;
  bool m_vpcIdentifierHasBeenSet = false;
  bool m_privateDnsEnabledHasBeenSet = false;
  bool m_securityGroupIdsHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_dnsOptionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace VPCLattice
}  // namespace Aws
