﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/personalize/PersonalizeRequest.h>
#include <aws/personalize/Personalize_EXPORTS.h>
#include <aws/personalize/model/Domain.h>

#include <utility>

namespace Aws {
namespace Personalize {
namespace Model {

/**
 */
class CreateSchemaRequest : public PersonalizeRequest {
 public:
  AWS_PERSONALIZE_API CreateSchemaRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateSchema"; }

  AWS_PERSONALIZE_API Aws::String SerializePayload() const override;

  AWS_PERSONALIZE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name for the schema.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateSchemaRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A schema in Avro JSON format.</p>
   */
  inline const Aws::String& GetSchema() const { return m_schema; }
  inline bool SchemaHasBeenSet() const { return m_schemaHasBeenSet; }
  template <typename SchemaT = Aws::String>
  void SetSchema(SchemaT&& value) {
    m_schemaHasBeenSet = true;
    m_schema = std::forward<SchemaT>(value);
  }
  template <typename SchemaT = Aws::String>
  CreateSchemaRequest& WithSchema(SchemaT&& value) {
    SetSchema(std::forward<SchemaT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The domain for the schema. If you are creating a schema for a dataset in a
   * Domain dataset group, specify the domain you chose when you created the Domain
   * dataset group.</p>
   */
  inline Domain GetDomain() const { return m_domain; }
  inline bool DomainHasBeenSet() const { return m_domainHasBeenSet; }
  inline void SetDomain(Domain value) {
    m_domainHasBeenSet = true;
    m_domain = value;
  }
  inline CreateSchemaRequest& WithDomain(Domain value) {
    SetDomain(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_schema;

  Domain m_domain{Domain::NOT_SET};
  bool m_nameHasBeenSet = false;
  bool m_schemaHasBeenSet = false;
  bool m_domainHasBeenSet = false;
};

}  // namespace Model
}  // namespace Personalize
}  // namespace Aws
