﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/notifications/NotificationsRequest.h>
#include <aws/notifications/Notifications_EXPORTS.h>
#include <aws/notifications/model/MemberAccountNotificationConfigurationStatus.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace Notifications {
namespace Model {

/**
 */
class ListMemberAccountsRequest : public NotificationsRequest {
 public:
  AWS_NOTIFICATIONS_API ListMemberAccountsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListMemberAccounts"; }

  AWS_NOTIFICATIONS_API Aws::String SerializePayload() const override;

  AWS_NOTIFICATIONS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the notification configuration used to
   * filter the member accounts.</p>
   */
  inline const Aws::String& GetNotificationConfigurationArn() const { return m_notificationConfigurationArn; }
  inline bool NotificationConfigurationArnHasBeenSet() const { return m_notificationConfigurationArnHasBeenSet; }
  template <typename NotificationConfigurationArnT = Aws::String>
  void SetNotificationConfigurationArn(NotificationConfigurationArnT&& value) {
    m_notificationConfigurationArnHasBeenSet = true;
    m_notificationConfigurationArn = std::forward<NotificationConfigurationArnT>(value);
  }
  template <typename NotificationConfigurationArnT = Aws::String>
  ListMemberAccountsRequest& WithNotificationConfigurationArn(NotificationConfigurationArnT&& value) {
    SetNotificationConfigurationArn(std::forward<NotificationConfigurationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return in a single call. Valid values are
   * 1-100.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListMemberAccountsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next page of results. Use the value returned in the
   * previous response.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListMemberAccountsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The member account identifier used to filter the results.</p>
   */
  inline const Aws::String& GetMemberAccount() const { return m_memberAccount; }
  inline bool MemberAccountHasBeenSet() const { return m_memberAccountHasBeenSet; }
  template <typename MemberAccountT = Aws::String>
  void SetMemberAccount(MemberAccountT&& value) {
    m_memberAccountHasBeenSet = true;
    m_memberAccount = std::forward<MemberAccountT>(value);
  }
  template <typename MemberAccountT = Aws::String>
  ListMemberAccountsRequest& WithMemberAccount(MemberAccountT&& value) {
    SetMemberAccount(std::forward<MemberAccountT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status used to filter the member accounts.</p>
   */
  inline MemberAccountNotificationConfigurationStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(MemberAccountNotificationConfigurationStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline ListMemberAccountsRequest& WithStatus(MemberAccountNotificationConfigurationStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The organizational unit ID used to filter the member accounts.</p>
   */
  inline const Aws::String& GetOrganizationalUnitId() const { return m_organizationalUnitId; }
  inline bool OrganizationalUnitIdHasBeenSet() const { return m_organizationalUnitIdHasBeenSet; }
  template <typename OrganizationalUnitIdT = Aws::String>
  void SetOrganizationalUnitId(OrganizationalUnitIdT&& value) {
    m_organizationalUnitIdHasBeenSet = true;
    m_organizationalUnitId = std::forward<OrganizationalUnitIdT>(value);
  }
  template <typename OrganizationalUnitIdT = Aws::String>
  ListMemberAccountsRequest& WithOrganizationalUnitId(OrganizationalUnitIdT&& value) {
    SetOrganizationalUnitId(std::forward<OrganizationalUnitIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_notificationConfigurationArn;

  int m_maxResults{0};

  Aws::String m_nextToken;

  Aws::String m_memberAccount;

  MemberAccountNotificationConfigurationStatus m_status{MemberAccountNotificationConfigurationStatus::NOT_SET};

  Aws::String m_organizationalUnitId;
  bool m_notificationConfigurationArnHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_memberAccountHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_organizationalUnitIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Notifications
}  // namespace Aws
