#
# Copyright (2016-2017) Hewlett Packard Enterprise Development LP
# GNU General Public License v3.0+ (see LICENSES/GPL-3.0-or-later.txt or https://www.gnu.org/licenses/gpl-3.0.txt)
# SPDX-License-Identifier: GPL-3.0-or-later

from __future__ import annotations

import unittest
from .oneview_module_loader import FcoeNetworkModule
from .hpe_test_utils import OneViewBaseTestCase

FAKE_MSG_ERROR = "Fake message error"

DEFAULT_FCOE_NETWORK_TEMPLATE = dict(name="New FCoE Network 2", vlanId="201", connectionTemplateUri=None)

PARAMS_FOR_PRESENT = dict(config="config.json", state="present", data=dict(name=DEFAULT_FCOE_NETWORK_TEMPLATE["name"]))

PARAMS_WITH_CHANGES = dict(
    config="config.json",
    state="present",
    data=dict(name=DEFAULT_FCOE_NETWORK_TEMPLATE["name"], fabricType="DirectAttach", newName="New Name"),
)

PARAMS_FOR_ABSENT = dict(config="config.json", state="absent", data=dict(name=DEFAULT_FCOE_NETWORK_TEMPLATE["name"]))


class FcoeNetworkSpec(unittest.TestCase, OneViewBaseTestCase):
    """
    OneViewBaseTestCase provides the mocks used in this test case
    """

    def setUp(self):
        self.configure_mocks(self, FcoeNetworkModule)
        self.resource = self.mock_ov_client.fcoe_networks

    def test_should_create_new_fcoe_network(self):
        self.resource.get_by.return_value = []
        self.resource.create.return_value = DEFAULT_FCOE_NETWORK_TEMPLATE

        self.mock_ansible_module.params = PARAMS_FOR_PRESENT

        FcoeNetworkModule().run()

        self.mock_ansible_module.exit_json.assert_called_once_with(
            changed=True,
            msg=FcoeNetworkModule.MSG_CREATED,
            ansible_facts=dict(fcoe_network=DEFAULT_FCOE_NETWORK_TEMPLATE),
        )

    def test_should_not_update_when_data_is_equals(self):
        self.resource.get_by.return_value = [DEFAULT_FCOE_NETWORK_TEMPLATE]
        self.mock_ansible_module.params = PARAMS_FOR_PRESENT.copy()

        FcoeNetworkModule().run()

        self.mock_ansible_module.exit_json.assert_called_once_with(
            changed=False,
            msg=FcoeNetworkModule.MSG_ALREADY_PRESENT,
            ansible_facts=dict(fcoe_network=DEFAULT_FCOE_NETWORK_TEMPLATE),
        )

    def test_update_when_data_has_modified_attributes(self):
        data_merged = DEFAULT_FCOE_NETWORK_TEMPLATE.copy()
        data_merged["fabricType"] = "DirectAttach"

        self.resource.get_by.return_value = [DEFAULT_FCOE_NETWORK_TEMPLATE]
        self.resource.update.return_value = data_merged

        self.mock_ansible_module.params = PARAMS_WITH_CHANGES

        FcoeNetworkModule().run()

        self.mock_ansible_module.exit_json.assert_called_once_with(
            changed=True, msg=FcoeNetworkModule.MSG_UPDATED, ansible_facts=dict(fcoe_network=data_merged)
        )

    def test_should_remove_fcoe_network(self):
        self.resource.get_by.return_value = [DEFAULT_FCOE_NETWORK_TEMPLATE]

        self.mock_ansible_module.params = PARAMS_FOR_ABSENT

        FcoeNetworkModule().run()

        self.mock_ansible_module.exit_json.assert_called_once_with(changed=True, msg=FcoeNetworkModule.MSG_DELETED)

    def test_should_do_nothing_when_fcoe_network_not_exist(self):
        self.resource.get_by.return_value = []

        self.mock_ansible_module.params = PARAMS_FOR_ABSENT

        FcoeNetworkModule().run()

        self.mock_ansible_module.exit_json.assert_called_once_with(
            changed=False, msg=FcoeNetworkModule.MSG_ALREADY_ABSENT
        )

    def test_update_scopes_when_different(self):
        params_to_scope = PARAMS_FOR_PRESENT.copy()
        params_to_scope["data"]["scopeUris"] = ["test"]
        self.mock_ansible_module.params = params_to_scope

        resource_data = DEFAULT_FCOE_NETWORK_TEMPLATE.copy()
        resource_data["scopeUris"] = ["fake"]
        resource_data["uri"] = "rest/fcoe/fake"
        self.resource.get_by.return_value = [resource_data]

        patch_return = resource_data.copy()
        patch_return["scopeUris"] = ["test"]
        self.resource.patch.return_value = patch_return

        FcoeNetworkModule().run()

        self.resource.patch.assert_called_once_with(
            "rest/fcoe/fake", operation="replace", path="/scopeUris", value=["test"]
        )

        self.mock_ansible_module.exit_json.assert_called_once_with(
            changed=True, ansible_facts=dict(fcoe_network=patch_return), msg=FcoeNetworkModule.MSG_UPDATED
        )

    def test_should_do_nothing_when_scopes_are_the_same(self):
        params_to_scope = PARAMS_FOR_PRESENT.copy()
        params_to_scope["data"]["scopeUris"] = ["test"]
        self.mock_ansible_module.params = params_to_scope

        resource_data = DEFAULT_FCOE_NETWORK_TEMPLATE.copy()
        resource_data["scopeUris"] = ["test"]
        self.resource.get_by.return_value = [resource_data]

        FcoeNetworkModule().run()

        self.resource.patch.not_been_called()

        self.mock_ansible_module.exit_json.assert_called_once_with(
            changed=False, ansible_facts=dict(fcoe_network=resource_data), msg=FcoeNetworkModule.MSG_ALREADY_PRESENT
        )
