// @HEADER
// *****************************************************************************
//            NOX: An Object-Oriented Nonlinear Solver Package
//
// Copyright 2002 NTESS and the NOX contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

// Finite Element Problem Class
/* Provides function (RHS) and Jacobian evaluations for the
 * 2 equation test for quadratic and cubic line searches
 * from Dennis & Schnabel's book, chp 6. example 6.5.1
 *
 *    U0**2 + U1**2 - 2 = 0
 *    exp(U0-1) + U1**3 -2 = 0
 */

#ifndef _NOX_EXAMPLE_DENNIS_SCHNABEL_651_H
#define _NOX_EXAMPLE_DENNIS_SCHNABEL_651_H

#include "NOX_Epetra_Interface_Required.H"
#include "Teuchos_RCP.hpp"

// Forward Declarations
class Epetra_Comm;
class Epetra_Map;
class Epetra_Vector;
class Epetra_Import;
class Epetra_CrsGraph;
class Epetra_CrsMatrix;

// Flag to tell the evaluate routine what objects to fill
enum FillType {F_ONLY, MATRIX_ONLY, ALL};

// Finite Element Problem Class
class DennisSchnabel {

public:

  // Constructor
  DennisSchnabel(int NumGlobalElements, Epetra_Comm& Comm);

  // Destructor
  ~DennisSchnabel();

  // Evaluates the function (RHS) and/or the Jacobian using the solution
  // values in solnVector.
  bool evaluate( NOX::Epetra::Interface::Required::FillType fillType,
                 const Epetra_Vector *solnVector, Epetra_Vector *rhsVector);

  // Return a reference to the Epetra_Vector with the initial guess
  // that is generated by this class.
  Teuchos::RCP<Epetra_Vector> getSolution();

  // Return a reference to the Epetra_Vector with the Jacobian
  // that is generated by this class.
  Teuchos::RCP<Epetra_CrsMatrix> getJacobian();

private:

  // inserts the global column indices into the Graph
  Epetra_CrsGraph& generateGraph(Epetra_CrsGraph& AA);

private:

  FillType flag;
  Epetra_Map* StandardMap;
  Epetra_Map* OverlapMap;
  Epetra_Import* Importer;
  Teuchos::RCP<Epetra_Vector> initialSolution;
  Epetra_Vector* soln;
  Epetra_Vector* rhs;
  Epetra_CrsGraph* AA;
  Teuchos::RCP<Epetra_CrsMatrix> A;
  Epetra_Comm* Comm;

  int MyPID;              // Process number
  int NumProc;            // Total number of processes
  int NumMyElements;      // Number of elements owned by this process
  int NumGlobalElements;  // Total Number of elements

};
#endif




