# RustFS Project AI Coding Rules

## 🚨🚨🚨 CRITICAL DEVELOPMENT RULES - ZERO TOLERANCE 🚨🚨🚨

### ⛔️ ABSOLUTE PROHIBITION: NEVER COMMIT DIRECTLY TO MASTER/MAIN BRANCH ⛔️

**🔥 THIS IS THE MOST CRITICAL RULE - VIOLATION WILL RESULT IN IMMEDIATE REVERSAL 🔥**

- **🚫 ZERO DIRECT COMMITS TO MAIN/MASTER BRANCH - ABSOLUTELY FORBIDDEN**
- **🚫 ANY DIRECT COMMIT TO MAIN BRANCH MUST BE IMMEDIATELY REVERTED**
- **🚫 NO EXCEPTIONS FOR HOTFIXES, EMERGENCIES, OR URGENT CHANGES**
- **🚫 NO EXCEPTIONS FOR SMALL CHANGES, TYPOS, OR DOCUMENTATION UPDATES**
- **🚫 NO EXCEPTIONS FOR ANYONE - MAINTAINERS, CONTRIBUTORS, OR ADMINS**

### 📋 MANDATORY WORKFLOW - STRICTLY ENFORCED

**EVERY SINGLE CHANGE MUST FOLLOW THIS WORKFLOW:**

1. **Check current branch**: `git branch` (MUST NOT be on main/master)
2. **Switch to main**: `git checkout main`
3. **Pull latest**: `git pull origin main`
4. **Create feature branch**: `git checkout -b feat/your-feature-name`
5. **Make changes ONLY on feature branch**
6. **Test thoroughly before committing**
7. **Commit and push to feature branch**: `git push origin feat/your-feature-name`
8. **Create Pull Request**: Use `gh pr create` (MANDATORY)
9. **Wait for PR approval**: NO self-merging allowed
10. **Merge through GitHub interface**: ONLY after approval

### 🔒 ENFORCEMENT MECHANISMS

- **Branch protection rules**: Main branch is protected
- **Pre-commit hooks**: Will block direct commits to main
- **CI/CD checks**: All PRs must pass before merging
- **Code review requirement**: At least one approval needed
- **Automated reversal**: Direct commits to main will be automatically reverted

## 🎯 Core AI Development Principles

### Five Execution Steps

#### 1. Task Analysis and Planning
- **Clear Objectives**: Deeply understand task requirements and expected results before starting coding
- **Plan Development**: List specific files, components, and functions that need modification, explaining the reasons for changes
- **Risk Assessment**: Evaluate the impact of changes on existing functionality, develop rollback plans

#### 2. Precise Code Location
- **File Identification**: Determine specific files and line numbers that need modification
- **Impact Analysis**: Avoid modifying irrelevant files, clearly state the reason for each file modification
- **Minimization Principle**: Unless explicitly required by the task, do not create new abstraction layers or refactor existing code

#### 3. Minimal Code Changes
- **Focus on Core**: Only write code directly required by the task
- **Avoid Redundancy**: Do not add unnecessary logs, comments, tests, or error handling
- **Isolation**: Ensure new code does not interfere with existing functionality, maintain code independence

#### 4. Strict Code Review
- **Correctness Check**: Verify the correctness and completeness of code logic
- **Style Consistency**: Ensure code conforms to established project coding style
- **Side Effect Assessment**: Evaluate the impact of changes on downstream systems

#### 5. Clear Delivery Documentation
- **Change Summary**: Detailed explanation of all modifications and reasons
- **File List**: List all modified files and their specific changes
- **Risk Statement**: Mark any assumptions or potential risk points

### Core Principles
- **🎯 Precise Execution**: Strictly follow task requirements, no arbitrary innovation
- **⚡ Efficient Development**: Avoid over-design, only do necessary work
- **🛡️ Safe and Reliable**: Always follow development processes, ensure code quality and system stability
- **🔒 Cautious Modification**: Only modify when clearly knowing what needs to be changed and having confidence

### Additional AI Behavior Rules

1. **Use English for all code comments and documentation** - All comments, variable names, function names, documentation, and user-facing text in code should be in English
2. **Clean up temporary scripts after use** - Any temporary scripts, test files, or helper files created during AI work should be removed after task completion
3. **Only make confident modifications** - Do not make speculative changes or "convenient" modifications outside the task scope. If uncertain about a change, ask for clarification rather than guessing

## Project Overview

RustFS is a high-performance distributed object storage system written in Rust, compatible with S3 API. The project adopts a modular architecture, supporting erasure coding storage, multi-tenant management, observability, and other enterprise-level features.

## Core Architecture Principles

### 1. Modular Design

- Project uses Cargo workspace structure, containing multiple independent crates
- Core modules: `rustfs` (main service), `ecstore` (erasure coding storage), `common` (shared components)
- Functional modules: `iam` (identity management), `madmin` (management interface), `crypto` (encryption), etc.
- Tool modules: `cli` (command line tool), `crates/*` (utility libraries)

### 2. Asynchronous Programming Pattern

- Comprehensive use of `tokio` async runtime
- Prioritize `async/await` syntax
- Use `async-trait` for async methods in traits
- Avoid blocking operations, use `spawn_blocking` when necessary

### 3. Error Handling Strategy

- **Use modular, type-safe error handling with `thiserror`**
- Each module should define its own error type using `thiserror::Error` derive macro
- Support error chains and context information through `#[from]` and `#[source]` attributes
- Use `Result<T>` type aliases for consistency within each module
- Error conversion between modules should use explicit `From` implementations
- Follow the pattern: `pub type Result<T> = core::result::Result<T, Error>`
- Use `#[error("description")]` attributes for clear error messages
- Support error downcasting when needed through `other()` helper methods
- Implement `Clone` for errors when required by the domain logic

## Code Style Guidelines

### 1. Formatting Configuration

```toml
max_width = 130
fn_call_width = 90
single_line_let_else_max_width = 100
```

### 2. **🔧 MANDATORY Code Formatting Rules**

**CRITICAL**: All code must be properly formatted before committing. This project enforces strict formatting standards to maintain code consistency and readability.

#### Pre-commit Requirements (MANDATORY)

Before every commit, you **MUST**:

1. **Format your code**:
   ```bash
   cargo fmt --all
   ```

2. **Verify formatting**:
   ```bash
   cargo fmt --all --check
   ```

3. **Pass clippy checks**:
   ```bash
   cargo clippy --all-targets --all-features -- -D warnings
   ```

4. **Ensure compilation**:
   ```bash
   cargo check --all-targets
   ```

#### Quick Commands

Use these convenient Makefile targets for common tasks:

```bash
# Format all code
make fmt

# Check if code is properly formatted
make fmt-check

# Run clippy checks
make clippy

# Run compilation check
make check

# Run tests
make test

# Run all pre-commit checks (format + clippy + check + test)
make pre-commit

# Setup git hooks (one-time setup)
make setup-hooks
```

### 3. Naming Conventions

- Use `snake_case` for functions, variables, modules
- Use `PascalCase` for types, traits, enums
- Constants use `SCREAMING_SNAKE_CASE`
- Global variables prefix `GLOBAL_`, e.g., `GLOBAL_Endpoints`
- Use meaningful and descriptive names for variables, functions, and methods
- Avoid meaningless names like `temp`, `data`, `foo`, `bar`, `test123`
- Choose names that clearly express the purpose and intent

### 4. Type Declaration Guidelines

- **Prefer type inference over explicit type declarations** when the type is obvious from context
- Let the Rust compiler infer types whenever possible to reduce verbosity and improve maintainability
- Only specify types explicitly when:
  - The type cannot be inferred by the compiler
  - Explicit typing improves code clarity and readability
  - Required for API boundaries (function signatures, public struct fields)
  - Needed to resolve ambiguity between multiple possible types

### 5. Documentation Comments

- Public APIs must have documentation comments
- Use `///` for documentation comments
- Complex functions add `# Examples` and `# Parameters` descriptions
- Error cases use `# Errors` descriptions
- Always use English for all comments and documentation
- Avoid meaningless comments like "debug 111" or placeholder text

### 6. Import Guidelines

- Standard library imports first
- Third-party crate imports in the middle
- Project internal imports last
- Group `use` statements with blank lines between groups

## Asynchronous Programming Guidelines

### 1. Trait Definition

```rust
#[async_trait::async_trait]
pub trait StorageAPI: Send + Sync {
    async fn get_object(&self, bucket: &str, object: &str) -> Result<ObjectInfo>;
}
```

### 2. Error Handling

```rust
// Use ? operator to propagate errors
async fn example_function() -> Result<()> {
    let data = read_file("path").await?;
    process_data(data).await?;
    Ok(())
}
```

### 3. Concurrency Control

- Use `Arc` and `Mutex`/`RwLock` for shared state management
- Prioritize async locks from `tokio::sync`
- Avoid holding locks for long periods

## Logging and Tracing Guidelines

### 1. Tracing Usage

```rust
#[tracing::instrument(skip(self, data))]
async fn process_data(&self, data: &[u8]) -> Result<()> {
    info!("Processing {} bytes", data.len());
    // Implementation logic
}
```

### 2. Log Levels

- `error!`: System errors requiring immediate attention
- `warn!`: Warning information that may affect functionality
- `info!`: Important business information
- `debug!`: Debug information for development use
- `trace!`: Detailed execution paths

### 3. Structured Logging

```rust
info!(
    counter.rustfs_api_requests_total = 1_u64,
    key_request_method = %request.method(),
    key_request_uri_path = %request.uri().path(),
    "API request processed"
);
```

## Error Handling Guidelines

### 1. Error Type Definition

```rust
// Use thiserror for module-specific error types
#[derive(thiserror::Error, Debug)]
pub enum MyError {
    #[error("IO error: {0}")]
    Io(#[from] std::io::Error),

    #[error("Storage error: {0}")]
    Storage(#[from] ecstore::error::StorageError),

    #[error("Custom error: {message}")]
    Custom { message: String },

    #[error("File not found: {path}")]
    FileNotFound { path: String },

    #[error("Invalid configuration: {0}")]
    InvalidConfig(String),
}

// Provide Result type alias for the module
pub type Result<T> = core::result::Result<T, MyError>;
```

### 2. Error Helper Methods

```rust
impl MyError {
    /// Create error from any compatible error type
    pub fn other<E>(error: E) -> Self
    where
        E: Into<Box<dyn std::error::Error + Send + Sync>>,
    {
        MyError::Io(std::io::Error::other(error))
    }
}
```

### 3. Error Context and Propagation

```rust
// Use ? operator for clean error propagation
async fn example_function() -> Result<()> {
    let data = read_file("path").await?;
    process_data(data).await?;
    Ok(())
}

// Add context to errors
fn process_with_context(path: &str) -> Result<()> {
    std::fs::read(path)
        .map_err(|e| MyError::Custom {
            message: format!("Failed to read {}: {}", path, e)
        })?;
    Ok(())
}
```

## Performance Optimization Guidelines

### 1. Memory Management

- Use `Bytes` instead of `Vec<u8>` for zero-copy operations
- Avoid unnecessary cloning, use reference passing
- Use `Arc` for sharing large objects

### 2. Concurrency Optimization

```rust
// Use join_all for concurrent operations
let futures = disks.iter().map(|disk| disk.operation());
let results = join_all(futures).await;
```

### 3. Caching Strategy

- Use `LazyLock` for global caching
- Implement LRU cache to avoid memory leaks

## Testing Guidelines

### 1. Unit Tests

```rust
#[cfg(test)]
mod tests {
    use super::*;
    use test_case::test_case;

    #[tokio::test]
    async fn test_async_function() {
        let result = async_function().await;
        assert!(result.is_ok());
    }

    #[test_case("input1", "expected1")]
    #[test_case("input2", "expected2")]
    fn test_with_cases(input: &str, expected: &str) {
        assert_eq!(function(input), expected);
    }
}
```

### 2. Integration Tests

- Use `e2e_test` module for end-to-end testing
- Simulate real storage environments

### 3. Test Quality Standards

- Write meaningful test cases that verify actual functionality
- Avoid placeholder or debug content like "debug 111", "test test", etc.
- Use descriptive test names that clearly indicate what is being tested
- Each test should have a clear purpose and verify specific behavior
- Test data should be realistic and representative of actual use cases

## Cross-Platform Compatibility Guidelines

### 1. CPU Architecture Compatibility

- **Always consider multi-platform and different CPU architecture compatibility** when writing code
- Support major architectures: x86_64, aarch64 (ARM64), and other target platforms
- Use conditional compilation for architecture-specific code:

```rust
#[cfg(target_arch = "x86_64")]
fn optimized_x86_64_function() { /* x86_64 specific implementation */ }

#[cfg(target_arch = "aarch64")]
fn optimized_aarch64_function() { /* ARM64 specific implementation */ }

#[cfg(not(any(target_arch = "x86_64", target_arch = "aarch64")))]
fn generic_function() { /* Generic fallback implementation */ }
```

### 2. Platform-Specific Dependencies

- Use feature flags for platform-specific dependencies
- Provide fallback implementations for unsupported platforms
- Test on multiple architectures in CI/CD pipeline

### 3. Endianness Considerations

- Use explicit byte order conversion when dealing with binary data
- Prefer `to_le_bytes()`, `from_le_bytes()` for consistent little-endian format
- Use `byteorder` crate for complex binary format handling

### 4. SIMD and Performance Optimizations

- Use portable SIMD libraries like `wide` or `packed_simd`
- Provide fallback implementations for non-SIMD architectures
- Use runtime feature detection when appropriate

## Security Guidelines

### 1. Memory Safety

- Disable `unsafe` code (workspace.lints.rust.unsafe_code = "deny")
- Use `rustls` instead of `openssl`

### 2. Authentication and Authorization

```rust
// Use IAM system for permission checks
let identity = iam.authenticate(&access_key, &secret_key).await?;
iam.authorize(&identity, &action, &resource).await?;
```

## Configuration Management Guidelines

### 1. Environment Variables

- Use `RUSTFS_` prefix
- Support both configuration files and environment variables
- Provide reasonable default values

### 2. Configuration Structure

```rust
#[derive(Debug, Deserialize, Clone)]
pub struct Config {
    pub address: String,
    pub volumes: String,
    #[serde(default)]
    pub console_enable: bool,
}
```

## Dependency Management Guidelines

### 1. Workspace Dependencies

- Manage versions uniformly at workspace level
- Use `workspace = true` to inherit configuration

### 2. Feature Flags

```rust
[features]
default = ["file"]
gpu = ["dep:nvml-wrapper"]
kafka = ["dep:rdkafka"]
```

## Deployment and Operations Guidelines

### 1. Containerization

- Provide Dockerfile and docker-compose configuration
- Support multi-stage builds to optimize image size

### 2. Observability

- Integrate OpenTelemetry for distributed tracing
- Support Prometheus metrics collection
- Provide Grafana dashboards

### 3. Health Checks

```rust
// Implement health check endpoint
async fn health_check() -> Result<HealthStatus> {
    // Check component status
}
```

## Code Review Checklist

### 1. **Code Formatting and Quality (MANDATORY)**

- [ ] **Code is properly formatted** (`cargo fmt --all --check` passes)
- [ ] **All clippy warnings are resolved** (`cargo clippy --all-targets --all-features -- -D warnings` passes)
- [ ] **Code compiles successfully** (`cargo check --all-targets` passes)
- [ ] **Pre-commit hooks are working** and all checks pass
- [ ] **No formatting-related changes** mixed with functional changes (separate commits)

### 2. Functionality

- [ ] Are all error cases properly handled?
- [ ] Is there appropriate logging?
- [ ] Is there necessary test coverage?

### 3. Performance

- [ ] Are unnecessary memory allocations avoided?
- [ ] Are async operations used correctly?
- [ ] Are there potential deadlock risks?

### 4. Security

- [ ] Are input parameters properly validated?
- [ ] Are there appropriate permission checks?
- [ ] Is information leakage avoided?

### 5. Cross-Platform Compatibility

- [ ] Does the code work on different CPU architectures (x86_64, aarch64)?
- [ ] Are platform-specific features properly gated with conditional compilation?
- [ ] Is byte order handling correct for binary data?
- [ ] Are there appropriate fallback implementations for unsupported platforms?

### 6. Code Commits and Documentation

- [ ] Does it comply with [Conventional Commits](https://www.conventionalcommits.org/en/v1.0.0/)?
- [ ] Are commit messages concise and under 72 characters for the title line?
- [ ] Commit titles should be concise and in English, avoid Chinese
- [ ] Is PR description provided in copyable markdown format for easy copying?

## Common Patterns and Best Practices

### 1. Resource Management

```rust
// Use RAII pattern for resource management
pub struct ResourceGuard {
    resource: Resource,
}

impl Drop for ResourceGuard {
    fn drop(&mut self) {
        // Clean up resources
    }
}
```

### 2. Dependency Injection

```rust
// Use dependency injection pattern
pub struct Service {
    config: Arc<Config>,
    storage: Arc<dyn StorageAPI>,
}
```

### 3. Graceful Shutdown

```rust
// Implement graceful shutdown
async fn shutdown_gracefully(shutdown_rx: &mut Receiver<()>) {
    tokio::select! {
        _ = shutdown_rx.recv() => {
            info!("Received shutdown signal");
            // Perform cleanup operations
        }
        _ = tokio::time::sleep(SHUTDOWN_TIMEOUT) => {
            warn!("Shutdown timeout reached");
        }
    }
}
```

## Domain-Specific Guidelines

### 1. Storage Operations

- All storage operations must support erasure coding
- Implement read/write quorum mechanisms
- Support data integrity verification

### 2. Network Communication

- Use gRPC for internal service communication
- HTTP/HTTPS support for S3-compatible API
- Implement connection pooling and retry mechanisms

### 3. Metadata Management

- Use FlatBuffers for serialization
- Support version control and migration
- Implement metadata caching

## Branch Management and Development Workflow

### Branch Management

- **🚨 CRITICAL: NEVER modify code directly on main or master branch - THIS IS ABSOLUTELY FORBIDDEN 🚨**
- **⚠️ ANY DIRECT COMMITS TO MASTER/MAIN WILL BE REJECTED AND MUST BE REVERTED IMMEDIATELY ⚠️**
- **🔒 ALL CHANGES MUST GO THROUGH PULL REQUESTS - NO DIRECT COMMITS TO MAIN UNDER ANY CIRCUMSTANCES 🔒**
- **Always work on feature branches - NO EXCEPTIONS**
- Always check the .rules.md file before starting to ensure you understand the project guidelines
- **MANDATORY workflow for ALL changes:**
   1. `git checkout main` (switch to main branch)
   2. `git pull` (get latest changes)
   3. `git checkout -b feat/your-feature-name` (create and switch to feature branch)
   4. Make your changes ONLY on the feature branch
   5. Test thoroughly before committing
   6. Commit and push to the feature branch
   7. **Create a pull request for code review - THIS IS THE ONLY WAY TO MERGE TO MAIN**
   8. **Wait for PR approval before merging - NEVER merge your own PRs without review**
- Use descriptive branch names following the pattern: `feat/feature-name`, `fix/issue-name`, `refactor/component-name`, etc.
- **Double-check current branch before ANY commit: `git branch` to ensure you're NOT on main/master**
- **Pull Request Requirements:**
  - All changes must be submitted via PR regardless of size or urgency
  - PRs must include comprehensive description and testing information
  - PRs must pass all CI/CD checks before merging
  - PRs require at least one approval from code reviewers
  - Even hotfixes and emergency changes must go through PR process
- **Enforcement:**
  - Main branch should be protected with branch protection rules
  - Direct pushes to main should be blocked by repository settings
  - Any accidental direct commits to main must be immediately reverted via PR

### Development Workflow

## 🎯 **Core Development Principles**

- **🔴 Every change must be precise - don't modify unless you're confident**
  - Carefully analyze code logic and ensure complete understanding before making changes
  - When uncertain, prefer asking users or consulting documentation over blind modifications
  - Use small iterative steps, modify only necessary parts at a time
  - Evaluate impact scope before changes to ensure no new issues are introduced

- **🚀 GitHub PR creation prioritizes gh command usage**
  - Prefer using `gh pr create` command to create Pull Requests
  - Avoid having users manually create PRs through web interface
  - Provide clear and professional PR titles and descriptions
  - Using `gh` commands ensures better integration and automation

## 📝 **Code Quality Requirements**

- Use English for all code comments, documentation, and variable names
- Write meaningful and descriptive names for variables, functions, and methods
- Avoid meaningless test content like "debug 111" or placeholder values
- Before each change, carefully read the existing code to ensure you understand the code structure and implementation, do not break existing logic implementation, do not introduce new issues
- Ensure each change provides sufficient test cases to guarantee code correctness
- Do not arbitrarily modify numbers and constants in test cases, carefully analyze their meaning to ensure test case correctness
- When writing or modifying tests, check existing test cases to ensure they have scientific naming and rigorous logic testing, if not compliant, modify test cases to ensure scientific and rigorous testing
- **Before committing any changes, run `cargo clippy --all-targets --all-features -- -D warnings` to ensure all code passes Clippy checks**
- After each development completion, first git add . then git commit -m "feat: feature description" or "fix: issue description", ensure compliance with [Conventional Commits](https://www.conventionalcommits.org/en/v1.0.0/)
- **Keep commit messages concise and under 72 characters** for the title line, use body for detailed explanations if needed
- After each development completion, first git push to remote repository
- After each change completion, summarize the changes, do not create summary files, provide a brief change description, ensure compliance with [Conventional Commits](https://www.conventionalcommits.org/en/v1.0.0/)
- Provide change descriptions needed for PR in the conversation, ensure compliance with [Conventional Commits](https://www.conventionalcommits.org/en/v1.0.0/)
- **Always provide PR descriptions in English** after completing any changes, including:
  - Clear and concise title following Conventional Commits format
  - Detailed description of what was changed and why
  - List of key changes and improvements
  - Any breaking changes or migration notes if applicable
  - Testing information and verification steps
- **Provide PR descriptions in copyable markdown format** enclosed in code blocks for easy one-click copying

## 🚫 AI Documentation Generation Restrictions

### Forbidden Summary Documents

- **Strictly forbidden to create any form of AI-generated summary documents**
- **Do not create documents containing large amounts of emoji, detailed formatting tables and typical AI style**
- **Do not generate the following types of documents in the project:**
  - Benchmark summary documents (BENCHMARK*.md)
  - Implementation comparison analysis documents (IMPLEMENTATION_COMPARISON*.md)
  - Performance analysis report documents
  - Architecture summary documents
  - Feature comparison documents
  - Any documents with large amounts of emoji and formatted content
- **If documentation is needed, only create when explicitly requested by the user, and maintain a concise and practical style**
- **Documentation should focus on actually needed information, avoiding excessive formatting and decorative content**
- **Any discovered AI-generated summary documents should be immediately deleted**

### Allowed Documentation Types

- README.md (project introduction, keep concise)
- Technical documentation (only create when explicitly needed)
- User manual (only create when explicitly needed)
- API documentation (generated from code)
- Changelog (CHANGELOG.md)

These rules should serve as guiding principles when developing the RustFS project, ensuring code quality, performance, and maintainability.