#include <boost/hash2/blake2.hpp>

#include <boost/core/lightweight_test.hpp>

#include <stdlib.h>

std::string from_hex( char const* str )
{
    auto f = []( char c ) { return ( c >= 'a' ? c - 'a' + 10 : c - '0' ); };

    std::string s;
    while( *str != '\0' )
    {
        s.push_back( static_cast<char>( ( f( str[ 0 ] ) << 4 ) + f( str[ 1 ] ) ) );
        str += 2;
    }
    return s;
}

template<class H> std::string digest( std::string const & s )
{
    std::string s1;
    std::string s2;

    {
        H h;

        h.update( s.data(), s.size() );
        s1 = to_string( h.result() );
    }

    {
        auto const n = s.size() / 3;

        H h;
        h.update( s.data(), n );
        h.update( s.data() + n , s.size() - n );
        s2 = to_string( h.result() );
    }

    BOOST_TEST_EQ( s1, s2 );

    return s1;
}

template<class H> std::string digest( std::string const& key, std::string const & s )
{
    std::string s1;
    std::string s2;

    {
        H h( key.data(), key.size() );

        h.update( s.data(), s.size() );
        s1 = to_string( h.result() );
    }

    {
        auto const n = s.size() / 3;

        H h( key.data(), key.size() );
        h.update( s.data(), n );
        h.update( s.data() + n , s.size() - n );
        s2 = to_string( h.result() );
    }

    BOOST_TEST_EQ( s1, s2 );

    return s1;
}

static void blake2b_512()
{
    using boost::hash2::blake2b_512;

    // https://en.wikipedia.org/wiki/BLAKE_(hash_function)#Example_digests_2
    BOOST_TEST_EQ( digest<blake2b_512>( "" ), std::string( "786a02f742015903c6c6fd852552d272912f4740e15847618a86e217f71f5419d25e1031afee585313896444934eb04b903a685b1448b755d56f701afe9be2ce" ));
    BOOST_TEST_EQ( digest<blake2b_512>( "The quick brown fox jumps over the lazy dog" ), std::string( "a8add4bdddfd93e4877d2746e62817b116364a1fa7bc148d95090bc7333b3673f82401cf7aa2e4cb1ecd90296e3f14cb5413f8ed77be73045b13914cdcd6a918" ));
    BOOST_TEST_EQ( digest<blake2b_512>( "The quick brown fox jumps over the lazy dof" ), std::string( "ab6b007747d8068c02e25a6008db8a77c218d94f3b40d2291a7dc8a62090a744c082ea27af01521a102e42f480a31e9844053f456b4b41e8aa78bbe5c12957bb" ));

    // selected samples from the download available here:
    // https://raw.githubusercontent.com/BLAKE2/BLAKE2/refs/heads/master/testvectors/blake2-kat.json
    std::pair<char const*, char const*> inputs[] =
    {
          /* expected hash */                                                                                                                 /* input message */
        { "2fa3f686df876995167e7c2e5d74c4c7b6e48f8068fe0e44208344d480f7904c36963e44115fe3eb2a3ac8694c28bcb4f5a0f3276f2e79487d8219057a506e4b", "00" },
        { "40a374727302d9a4769c17b5f409ff32f58aa24ff122d7603e4fda1509e919d4107a52c57570a6d94e50967aea573b11f86f473f537565c66f7039830a85d186", "000102" },
        { "10f0dc91b9f845fb95fad6860e6ce1adfa002c7fc327116d44d047cd7d5870d772bb12b5fac00e02b08ac2a0174d0446c36ab35f14ca31894cd61c78c849b48a", "000102030405060708090a0b" },
        { "b6292669ccd38d5f01caae96ba272c76a879a45743afa0725d83b9ebb26665b731f1848c52f11972b6644f554c064fa90780dbbbf3a89d4fc31f67df3e5857ef", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e" },
        { "2319e3789c47e2daa5fe807f61bec2a1a6537fa03f19ff32e87eecbfd64b7e0e8ccff439ac333b040f19b0c4ddd11a61e24ac1fe0f10a039806c5dcc0da3d115", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f" },
        { "f59711d44a031d5f97a9413c065d1e614c417ede998590325f49bad2fd444d3e4418be19aec4e11449ac1a57207898bc57d76a1bcf3566292c20c683a5c4648f", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f80" },
        { "a3eb6e6c7bf2fb8b28bfe8b15e15bb500f781ecc86f778c3a4e655fc5869bf2846a245d4e33b7b14436a17e63be79b36655c226a50ffbc7124207b0202342db5", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182" },
        { "d400601f9728ccc4c92342d9787d8d28ab323af375ca5624b4bb91d17271fbae862e413be73f1f68e615b8c5c391be0dbd9144746eb339ad541547ba9c468a17", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a" },
        { "5b21c5fd8868367612474fa2e70e9cfa2201ffeee8fafab5797ad58fefa17c9b5b107da4a3db6320baaf2c8617d5a51df914ae88da3867c2d41f0cc14fa67928", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9fa0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebfc0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedfe0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfe" },
    };

    for( auto const& input : inputs )
    {
        BOOST_TEST_EQ( digest<blake2b_512>( from_hex( input.second ) ), std::string( input.first ));
    }

    {
        auto const block_size = blake2b_512::block_size;

        auto s = from_hex( "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9fa0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebfc0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedfe0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfe" );
        BOOST_TEST_GT( s.size(), block_size );
        BOOST_TEST_LT( s.size(), 2 * block_size );

        auto p = s.data();
        auto n = s.size();

        blake2b_512 h;
        h.update( p, block_size );

        p += block_size;
        n -= block_size;

        h.update( p, n );

        BOOST_TEST_EQ( to_string( h.result() ), std::string( "5b21c5fd8868367612474fa2e70e9cfa2201ffeee8fafab5797ad58fefa17c9b5b107da4a3db6320baaf2c8617d5a51df914ae88da3867c2d41f0cc14fa67928" ) );
    }

    {
        unsigned char buf[ 1234 ] = {};
        for( unsigned i = 0; i < 1234; ++i )
        {
            buf[ i ] = i & 0xff;
        }

        blake2b_512 h;
        {
            auto p = buf;
            auto n = 1234;

            auto const block_size = blake2b_512::block_size;

            h.update( p, block_size - 4 );
            p += block_size - 4;
            n -= block_size - 4;

            h.update( p , 500 );
            p += 500;
            n -= 500;

            h.update( p, n );
        }

        // independently calculated using OpenSSL's EVP_blake2b512
        //
        BOOST_TEST_EQ( to_string( h.result() ), std::string( "90efbea1169c454157ac945187c160f5f1c524bae3f87cd9c5db14a45f1916facddf78dfa37678321e00137c6d111116b561454896a79879e87e284adf33fc66" ) );
    }
}

static void blake2s_256()
{
    using boost::hash2::blake2s_256;

    // https://en.wikipedia.org/wiki/BLAKE_(hash_function)#Example_digests_2
    BOOST_TEST_EQ( digest<boost::hash2::blake2s_256>( "" ), std::string( "69217a3079908094e11121d042354a7c1f55b6482ca1a51e1b250dfd1ed0eef9" ));

    // selected samples from the download available here:
    // https://raw.githubusercontent.com/BLAKE2/BLAKE2/refs/heads/master/testvectors/blake2-kat.json
    std::pair<char const*, char const*> inputs[] =
    {
          /* expected hash */                                                 /* input message */
        { "e34d74dbaf4ff4c6abd871cc220451d2ea2648846c7757fbaac82fe51ad64bea", "00" },
        { "e8f91c6ef232a041452ab0e149070cdd7dd1769e75b3a5921be37876c45c9900", "000102" },
        { "1ac7ba754d6e2f94e0e86c46bfb262abbb74f450ef456d6b4d97aa80ce6da767", "000102030405060708090a0b" },
        { "56f34e8b96557e90c1f24b52d0c89d51086acf1b00f634cf1dde9233b8eaaa3e", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f" },
        { "f18417b39d617ab1c18fdf91ebd0fc6d5516bb34cf39364037bce81fa04cecb1", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e" },
        { "1fa877de67259d19863a2a34bcc6962a2b25fcbf5cbecd7ede8f1fa36688a796", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f" },
        { "5bd169e67c82c2c2e98ef7008bdf261f2ddf30b1c00f9e7f275bb3e8a28dc9a2", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f80" },
        { "e76d3fbda5ba374e6bf8e50fadc3bbb9ba5c206ebdec89a3a54cf3dd84a07016", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182" },
        { "ca9ffac4c43f0b48461dc5c263bea3f6f00611ceacabf6f895ba2b0101dbb68d", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a" },
        { "f03f5789d3336b80d002d59fdf918bdb775b00956ed5528e86aa994acb38fe2d", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9fa0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebfc0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedfe0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfe" },
    };

    for( auto const& input : inputs )
    {
        BOOST_TEST_EQ( digest<boost::hash2::blake2s_256>( from_hex( input.second ) ), std::string( input.first ));
    }

    {
        auto const block_size = blake2s_256::block_size;

        auto s = from_hex( "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142" );
        BOOST_TEST_GT( s.size(), block_size );
        BOOST_TEST_LT( s.size(), 2 * block_size );

        auto p = s.data();
        auto n = s.size();

        boost::hash2::blake2s_256 h;
        h.update( p, block_size );

        p += block_size;
        n -= block_size;

        h.update( p, n );

        BOOST_TEST_EQ( to_string( h.result() ), std::string( "d8b2866e8a309db53e529ec32911d82f5ca16cff76216891a9676aa31aaa6c42" ) );
    }

    {
        unsigned char buf[ 1234 ] = {};
        for( unsigned i = 0; i < 1234; ++i )
        {
            buf[ i ] = i & 0xff;
        }

        boost::hash2::blake2s_256 h;
        {
            auto p = buf;
            auto n = 1234;

            auto const block_size = blake2s_256::block_size;

            h.update( p, block_size - 4 );
            p += block_size - 4;
            n -= block_size - 4;

            h.update( p , 500 );
            p += 500;
            n -= 500;

            h.update( p, n );
        }

        // independently calculated using OpenSSL's EVP_blake2s256
        //
        BOOST_TEST_EQ( to_string( h.result() ), std::string( "0b4c48cbdfccf5f9b593e0e177be958359dcc2246cdcdb8b7dfd0da9e409cd25" ) );
    }
}

static void blake2b_512_seeded()
{
    // selected samples from the download available here:
    // https://raw.githubusercontent.com/BLAKE2/BLAKE2/refs/heads/master/testvectors/blake2-kat.json
    std::pair<char const*, char const*> inputs[] =
    {
        /* expected hash */                                                                                                                   /* input message */
        { "10ebb67700b1868efb4417987acf4690ae9d972fb7a590c2f02871799aaa4786b5e996e8f0f4eb981fc214b005f42d2ff4233499391653df7aefcbc13fc51568", "" },
        { "961f6dd1e4dd30f63901690c512e78e4b45e4742ed197c3c5e45c549fd25f2e4187b0bc9fe30492b16b0d0bc4ef9b0f34c7003fac09a5ef1532e69430234cebd", "00" },
        { "33d0825dddf7ada99b0e7e307104ad07ca9cfd9692214f1561356315e784f3e5a17e364ae9dbb14cb2036df932b77f4b292761365fb328de7afdc6d8998f5fc1", "000102" },
        { "962452a8455cc56c8511317e3b1f3b2c37df75f588e94325fdd77070359cf63a9ae6e930936fdf8e1e08ffca440cfb72c28f06d89a2151d1c46cd5b268ef8563", "000102030405060708090a0b" },
        { "76d2d819c92bce55fa8e092ab1bf9b9eab237a25267986cacf2b8ee14d214d730dc9a5aa2d7b596e86a1fd8fa0804c77402d2fcd45083688b218b1cdfa0dcbcb", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e" },
        { "72065ee4dd91c2d8509fa1fc28a37c7fc9fa7d5b3f8ad3d0d7a25626b57b1b44788d4caf806290425f9890a3a2a35a905ab4b37acfd0da6e4517b2525c9651e4", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f" },
        { "64475dfe7600d7171bea0b394e27c9b00d8e74dd1e416a79473682ad3dfdbb706631558055cfc8a40e07bd015a4540dcdea15883cbbf31412df1de1cd4152b91", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f80" },
        { "60756966479dedc6dd4bcff8ea7d1d4ce4d4af2e7b097e32e3763518441147cc12b3c0ee6d2ecabf1198cec92e86a3616fba4f4e872f5825330adbb4c1dee444", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182" },
        { "e6fec19d89ce8717b1a087024670fe026f6c7cbda11caef959bb2d351bf856f8055d1c0ebdaaa9d1b17886fc2c562b5e99642fc064710c0d3488a02b5ed7f6fd", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a" },
        { "142709d62e28fcccd0af97fad0f8465b971e82201dc51070faa0372aa43e92484be1c1e73ba10906d5d1853db6a4106e0a7bf9800d373d6dee2d46d62ef2a461", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9fa0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebfc0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedfe0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfe" },
      };


    std::string const key = from_hex( "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f" );

    for( auto const& input : inputs )
    {
        BOOST_TEST_EQ( digest<boost::hash2::blake2b_512>( key, from_hex( input.second ) ), std::string( input.first ));
    }
}

static void blake2s_256_seeded()
{
    // selected samples from the download available here:
    // https://raw.githubusercontent.com/BLAKE2/BLAKE2/refs/heads/master/testvectors/blake2-kat.json
    std::pair<char const*, char const*> inputs[] =
    {
          /* expected hash */                                                 /* input message */
          { "48a8997da407876b3d79c0d92325ad3b89cbb754d86ab71aee047ad345fd2c49", "" },
          { "40d15fee7c328830166ac3f918650f807e7e01e177258cdc0a39b11f598066f1", "00" },
          { "1d220dbe2ee134661fdf6d9e74b41704710556f2f6e5a091b227697445dbea6b", "000102" },
          { "fba16169b2c3ee105be6e1e650e5cbf40746b6753d036ab55179014ad7ef6651", "000102030405060708090a0b" },
          { "8975b0577fd35566d750b362b0897a26c399136df07bababbde6203ff2954ed4", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f" },
          { "ddbfea75cc467882eb3483ce5e2e756a4f4701b76b445519e89f22d60fa86e06", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e" },
          { "0c311f38c35a4fb90d651c289d486856cd1413df9b0677f53ece2cd9e477c60a", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f" },
          { "46a73a8dd3e70f59d3942c01df599def783c9da82fd83222cd662b53dce7dbdf", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f80" },
          { "ab70c5dfbd1ea817fed0cd067293abf319e5d7901c2141d5d99b23f03a38e748", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182" },
          { "c7a197b3a05b566bcc9facd20e441d6f6c2860ac9651cd51d6b9d2cdeeea0390", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a" },
          { "3fb735061abc519dfe979e54c1ee5bfad0a9d858b3315bad34bde999efd724dd", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9fa0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebfc0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedfe0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfe" },
      };


    std::string const key = from_hex( "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f" );

    for( auto const& input : inputs )
    {
        BOOST_TEST_EQ( digest<boost::hash2::blake2s_256>( key, from_hex( input.second ) ), std::string( input.first ));
    }
}

int main()
{
    blake2b_512();
    blake2s_256();

    blake2b_512_seeded();
    blake2s_256_seeded();

    return boost::report_errors();
}
