/* Modifications Copyright (C) 2021, Adviced Micro Devices, Inc. All rights reserved 

Original code is based on Tiny-AES from https://github.com/kokke/tiny-AES-c

*/

/*

This is an implementation of the AES algorithm, specifically ECB, CTR and CBC mode.
Block size can be chosen in aes.h - available choices are AES128, AES192, AES256.

The implementation is verified against the test vectors in:
  National Institute of Standards and Technology Special Publication 800-38A 2001 ED

ECB-AES128
----------

  plain-text:
    6bc1bee22e409f96e93d7e117393172a
    ae2d8a571e03ac9c9eb76fac45af8e51
    30c81c46a35ce411e5fbc1191a0a52ef
    f69f2445df4f9b17ad2b417be66c3710

  key:
    2b7e151628aed2a6abf7158809cf4f3c

  resulting cipher
    3ad77bb40d7a3660a89ecaf32466ef97 
    f5d3d58503b9699de785895a96fdbaaf 
    43b1cd7f598ece23881b00e3ed030688 
    7b0c785e27e8ad3f8223207104725dd4 


NOTE:   String length must be evenly divisible by 16byte (str_len % 16 == 0)
        You should pad the end of the string with zeros if this is not the case.
        For AES192/256 the key size is proportionally larger.

*/


/*****************************************************************************/
/* Includes:                                                                 */
/*****************************************************************************/
#include <string.h> // CBC mode, for memset
#include "aes.h"

// iOS is BIG_ENDIAN, Android and Windows(server) is LITTLE_ENDIAN
// Forced to Apple to use LITTLE_ENDIAN, otherwise encryption Decrypt will error
#ifdef __APPLE__
    #define LITTLE_ENDIAN   1
#else
    #ifndef BIG_ENDIAN
        #define LITTLE_ENDIAN   1
    #else
        #define LITTLE_ENDIAN   0
    #endif
#endif



#if (LITTLE_ENDIAN == 1)
#define MASK32_BYTE0  0x000000FF
#define MASK32_BYTE1  0x0000FF00
#define MASK32_BYTE2  0x00FF0000
#define MASK32_BYTE3  0xFF000000

#define MASK64_BYTE0  0x00000000000000FF
#define MASK64_BYTE1  0x000000000000FF00
#define MASK64_BYTE2  0x0000000000FF0000
#define MASK64_BYTE3  0x00000000FF000000
#define MASK64_BYTE4  0x000000FF00000000
#define MASK64_BYTE5  0x0000FF0000000000
#define MASK64_BYTE6  0x00FF000000000000
#define MASK64_BYTE7  0xFF00000000000000


#define INVMASK32_BYTE0  0xFFFFFF00
#define INVMASK32_BYTE1  0xFFFF00FF
#define INVMASK32_BYTE2  0xFF00FFFF
#define INVMASK32_BYTE3  0x00FFFFFF

#define INVMASK64_BYTE0  0xFFFFFFFFFFFFFF00
#define INVMASK64_BYTE1  0xFFFFFFFFFFFF00FF
#define INVMASK64_BYTE2  0xFFFFFFFFFF00FFFF
#define INVMASK64_BYTE3  0xFFFFFFFF00FFFFFF
#define INVMASK64_BYTE4  0xFFFFFF00FFFFFFFF
#define INVMASK64_BYTE5  0xFFFF00FFFFFFFFFF
#define INVMASK64_BYTE6  0xFF00FFFFFFFFFFFF
#define INVMASK64_BYTE7  0x00FFFFFFFFFFFFFF


#define OFS32_BYTE0   0
#define OFS32_BYTE1   8
#define OFS32_BYTE2   16
#define OFS32_BYTE3   24

#define OFS64_BYTE0   0
#define OFS64_BYTE1   8
#define OFS64_BYTE2   16
#define OFS64_BYTE3   24
#define OFS64_BYTE4   32
#define OFS64_BYTE5   40
#define OFS64_BYTE6   48
#define OFS64_BYTE7   56

#else
#define MASK32_BYTE0  0xFF000000
#define MASK32_BYTE1  0x00FF0000
#define MASK32_BYTE2  0x0000FF00
#define MASK32_BYTE3  0x000000FF

#define INVMASK32_BYTE0  0x00FFFFFF
#define INVMASK32_BYTE1  0xFF00FFFF
#define INVMASK32_BYTE2  0xFFFF00FF
#define INVMASK32_BYTE3  0xFFFFFF00

#define INVMASK64_BYTE0  0x00FFFFFFFFFFFFFF
#define INVMASK64_BYTE1  0xFF00FFFFFFFFFFFF
#define INVMASK64_BYTE2  0xFFFF00FFFFFFFFFF
#define INVMASK64_BYTE3  0xFFFFFF00FFFFFFFF
#define INVMASK64_BYTE4  0xFFFFFFFF00FFFFFF
#define INVMASK64_BYTE5  0xFFFFFFFFFF00FFFF
#define INVMASK64_BYTE6  0xFFFFFFFFFFFF00FF
#define INVMASK64_BYTE7  0xFFFFFFFFFFFFFF00


#define OFS32_BYTE0   24
#define OFS32_BYTE1   16
#define OFS32_BYTE2   8
#define OFS32_BYTE3   0

#define OFS64_BYTE0   56
#define OFS66_BYTE1   48
#define OFS64_BYTE2   40
#define OFS64_BYTE3   32
#define OFS64_BYTE4   24
#define OFS66_BYTE5   16
#define OFS64_BYTE6   8
#define OFS64_BYTE7   0

#endif

#define USE32_ARITHMETIC 1


/*****************************************************************************/
/* Defines:                                                                  */
/*****************************************************************************/
// The number of columns comprising a state in AES. This is a constant in AES. Value=4
#define Nb 4

#if defined(AES256) && (AES256 == 1)
    #define Nk 8
    #define Nr 14
#elif defined(AES192) && (AES192 == 1)
    #define Nk 6
    #define Nr 12
#else
    #define Nk 4        // The number of 32 bit words in a key.
    #define Nr 10       // The number of rounds in AES Cipher.
#endif

// jcallan@github points out that declaring Multiply as a function 
// reduces code size considerably with the Keil ARM compiler.
// See this link for more information: https://github.com/kokke/tiny-AES-C/pull/3
//#ifndef MULTIPLY_AS_A_FUNCTION  	/*Multiply is removed*/
//  #define MULTIPLY_AS_A_FUNCTION 0
//#endif




/*****************************************************************************/
/* Private variables:                                                        */
/*****************************************************************************/
// state - array holding the intermediate results during decryption.
typedef uint8_t state_t[4][4];



// The lookup-tables are marked const so they can be placed in read-only storage instead of RAM
// The numbers below can be computed dynamically trading ROM for RAM - 
// This can be useful in (embedded) bootloader applications, where ROM is often limited.
static const uint32_t sbox[256] = {
  //0     1    2      3     4    5     6     7      8    9     A      B    C     D     E     F
  0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76,
  0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0,
  0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,
  0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75,
  0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84,
  0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,
  0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8,
  0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2,
  0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,
  0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb,
  0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79,
  0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,
  0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a,
  0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e,
  0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,
  0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16 };

static const uint32_t rsbox[256] = {
  0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38, 0xbf, 0x40, 0xa3, 0x9e, 0x81, 0xf3, 0xd7, 0xfb,
  0x7c, 0xe3, 0x39, 0x82, 0x9b, 0x2f, 0xff, 0x87, 0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb,
  0x54, 0x7b, 0x94, 0x32, 0xa6, 0xc2, 0x23, 0x3d, 0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e,
  0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2, 0x76, 0x5b, 0xa2, 0x49, 0x6d, 0x8b, 0xd1, 0x25,
  0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xd4, 0xa4, 0x5c, 0xcc, 0x5d, 0x65, 0xb6, 0x92,
  0x6c, 0x70, 0x48, 0x50, 0xfd, 0xed, 0xb9, 0xda, 0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d, 0x84,
  0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a, 0xf7, 0xe4, 0x58, 0x05, 0xb8, 0xb3, 0x45, 0x06,
  0xd0, 0x2c, 0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02, 0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b,
  0x3a, 0x91, 0x11, 0x41, 0x4f, 0x67, 0xdc, 0xea, 0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73,
  0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85, 0xe2, 0xf9, 0x37, 0xe8, 0x1c, 0x75, 0xdf, 0x6e,
  0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89, 0x6f, 0xb7, 0x62, 0x0e, 0xaa, 0x18, 0xbe, 0x1b,
  0xfc, 0x56, 0x3e, 0x4b, 0xc6, 0xd2, 0x79, 0x20, 0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd, 0x5a, 0xf4,
  0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31, 0xb1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xec, 0x5f,
  0x60, 0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d, 0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef,
  0xa0, 0xe0, 0x3b, 0x4d, 0xae, 0x2a, 0xf5, 0xb0, 0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61,
  0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6, 0x26, 0xe1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0c, 0x7d };

// The round constant word array, Rcon[i], contains the values given by 
// x to the power (i-1) being powers of x (x is denoted as {02}) in the field GF(2^8)
static const unsigned int Rcon[11] = {
  0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36 };

/*
 * Jordan Goulder points out in PR #12 (https://github.com/kokke/tiny-AES-C/pull/12),
 * that you can remove most of the elements in the Rcon array, because they are unused.
 *
 * From Wikipedia's article on the Rijndael key schedule @ https://en.wikipedia.org/wiki/Rijndael_key_schedule#Rcon
 * 
 * "Only the first some of these constants are actually used – up to rcon[10] for AES-128 (as 11 round keys are needed), 
 *  up to rcon[8] for AES-192, up to rcon[7] for AES-256. rcon[0] is not used in AES algorithm."
 */


/*****************************************************************************/
/* Private functions:                                                        */
/*****************************************************************************/
#define getSBoxValue(num) (sbox[(num)])
#define getSBoxInvert(num) (rsbox[(num)])

// This function produces Nb(Nr+1) round keys. The round keys are used in each round to decrypt the states. 
static void KeyExpansion(uint8_t* RoundKey, const uint8_t* Key)
{
   unsigned i, j, k;
  union
  {
      uint8_t array[4]; // Used for the column/row operations
      uint32_t word;
  } tempa;
  
  // The first round key is the key itself.
  for (i = 0; i < Nk; ++i)
  {
/*  GK
    RoundKey[(i * 4) + 0] = Key[(i * 4) + 0];
    RoundKey[(i * 4) + 1] = Key[(i * 4) + 1];
    RoundKey[(i * 4) + 2] = Key[(i * 4) + 2];
    RoundKey[(i * 4) + 3] = Key[(i * 4) + 3];
*/
      *((uint32_t*)(RoundKey + i * 4)) = *((const uint32_t*)(Key + i * 4));
  }

  // All other round keys are found from the previous round keys.
  for (i = Nk; i < Nb * (Nr + 1); ++i)
  {
    {
      k = (i - 1) * 4;
/*  GK
      tempa[0]=RoundKey[k + 0];
      tempa[1]=RoundKey[k + 1];
      tempa[2]=RoundKey[k + 2];
      tempa[3]=RoundKey[k + 3];
*/

      tempa.word = *((const uint32_t*)(RoundKey + k));
    }

    if (i % Nk == 0)
    {
      // This function shifts the 4 bytes in a word to the left once.
      // [a0,a1,a2,a3] becomes [a1,a2,a3,a0]

      // Function RotWord()
      {
#if 0
         uint8_t u8tmp = tempa[0];
        tempa.array[0] = tempa[1];
        tempa.array[1] = tempa[2];
        tempa.array[2] = tempa[3];
        tempa.array[3] = u8tmp;
#else
         uint32_t firstByte = ((tempa.word & MASK32_BYTE0) >> OFS32_BYTE0) << OFS32_BYTE3;
            //        *((uint16_t*)tempa) = *((const uint16_t*)(tempa + 1));
//        tempa[2] = tempa[3];
#if defined(LITTLE_ENDIAN) && (LITTLE_ENDIAN == 1)
        tempa.word >>= 8;
#else
        tempa.word <<= 8;
#endif
        tempa.word |= firstByte;
#endif
      }

      // SubWord() is a function that takes a four-byte input word and 
      // applies the S-box to each of the four bytes to produce an output word.

      // Function Subword()
      {
          
           uint32_t res = (getSBoxValue((tempa.word & MASK32_BYTE0) >> OFS32_BYTE0) ^ Rcon[i / Nk]) << OFS32_BYTE0;
          res |= getSBoxValue((tempa.word & MASK32_BYTE1) >> OFS32_BYTE1) << OFS32_BYTE1;
          res |= getSBoxValue((tempa.word & MASK32_BYTE2) >> OFS32_BYTE2) << OFS32_BYTE2;
          res |= getSBoxValue((tempa.word & MASK32_BYTE3) >> OFS32_BYTE3) << OFS32_BYTE3;
          tempa.word = res;
          /* GK
        tempa.array[0] = getSBoxValue(tempa.array[0]);
        tempa.array[1] = getSBoxValue(tempa.array[1]);
        tempa.array[2] = getSBoxValue(tempa.array[2]);
        tempa.array[3] = getSBoxValue(tempa.array[3]);
*/
      }

//      tempa.array[0] = tempa.array[0] ^ Rcon[i/Nk];
    }
#if defined(AES256) && (AES256 == 1)
    if (i % Nk == 4)
    {
/* GK
      // Function Subword()
      {
        tempa.array[0] = getSBoxValue(tempa.array[0]);
        tempa.array[1] = getSBoxValue(tempa.array[1]);
        tempa.array[2] = getSBoxValue(tempa.array[2]);
        tempa.array[3] = getSBoxValue(tempa.array[3]);
      }
*/
         uint32_t res = getSBoxValue((tempa.word & MASK32_BYTE0) >> OFS32_BYTE0) << OFS32_BYTE0;
        res |= getSBoxValue((tempa.word & MASK32_BYTE1) >> OFS32_BYTE1) << OFS32_BYTE1;
        res |= getSBoxValue((tempa.word & MASK32_BYTE2) >> OFS32_BYTE2) << OFS32_BYTE2;
        res |= getSBoxValue((tempa.word & MASK32_BYTE3) >> OFS32_BYTE3) << OFS32_BYTE3;
        tempa.word = res;

    }
#endif
    j = i * 4; k=(i - Nk) * 4;
/* GK
    RoundKey[j + 0] = RoundKey[k + 0] ^ tempa[0];
    RoundKey[j + 1] = RoundKey[k + 1] ^ tempa[1];
    RoundKey[j + 2] = RoundKey[k + 2] ^ tempa[2];
    RoundKey[j + 3] = RoundKey[k + 3] ^ tempa[3];
*/
    *((uint32_t*)(RoundKey + j)) = *((const uint32_t*)(RoundKey + k)) ^ tempa.word;
  }
}

void AES_init_ctx(struct AES_ctx* ctx, const uint8_t* key)
{
  KeyExpansion(ctx->RoundKey, key);
}
#if (defined(CBC) && (CBC == 1)) || (defined(CTR) && (CTR == 1))
void AES_init_ctx_iv(struct AES_ctx* ctx, const uint8_t* key, const uint8_t* iv)
{
  KeyExpansion(ctx->RoundKey, key);
  memcpy (ctx->Iv, iv, AES_BLOCKLEN);
}
void AES_ctx_set_iv(struct AES_ctx* ctx, const uint8_t* iv)
{
  memcpy (ctx->Iv, iv, AES_BLOCKLEN);
}
#endif

// This function adds the round key to state.
// The round key is added to the state by an XOR function.
static void AddRoundKey(uint32_t round, state_t* state, const uint8_t* RoundKey)
{
  unsigned int i;
  for (i = 0; i < 4; ++i)
  {
#if 0
    unsigned int j;
    for (j = 0; j < 4; ++j)
    {
      (*state)[i][j] ^= RoundKey[(round * Nb * 4) + (i * Nb) + j];
    }
#else
      *((uint32_t*)(state)+i) ^= *(const uint32_t*)(RoundKey + (round * Nb * 4) + (i * Nb));
#endif
  }
}

// The SubBytes Function Substitutes the values in the
// state matrix with values in an S-box.
static void SubBytes(state_t* state)
{
    unsigned int i;
    for (i = 0; i < 4; i++)
    {
         uint32_t* pLine = ((uint32_t*)state + i);
         uint32_t line = *pLine;

        *pLine = (getSBoxValue((line & MASK32_BYTE0) >> OFS32_BYTE0) << OFS32_BYTE0) | 
            (getSBoxValue((line & MASK32_BYTE1) >> OFS32_BYTE1) << OFS32_BYTE1) | 
            (getSBoxValue((line & MASK32_BYTE2) >> OFS32_BYTE2) << OFS32_BYTE2) | 
            (getSBoxValue((line & MASK32_BYTE3) >> OFS32_BYTE3) << OFS32_BYTE3);
    }


}

// The ShiftRows() function shifts the rows in the state to the left.
// Each row is shifted with different offset.
// Offset = Row number. So the first row is not shifted.
static void ShiftRows(state_t* state)
{
#if 1
     uint32_t  line0 = *((uint32_t*)state);
     uint32_t  line1 = *((uint32_t*)state + 1);
     uint32_t  line2 = *((uint32_t*)state + 2);
     uint32_t  line3 = *((uint32_t*)state + 3);

     uint32_t temp;

    temp = line0 & MASK32_BYTE1;
    line0 &= INVMASK32_BYTE1;
    line0 |= line1 & MASK32_BYTE1;
    line1 &= INVMASK32_BYTE1;
    line1 |= line2 & MASK32_BYTE1;
    line2 &= INVMASK32_BYTE1;
    line2 |= line3 & MASK32_BYTE1;
    line3 &= INVMASK32_BYTE1;
    line3 |= temp;

    temp = line0 & MASK32_BYTE2;
    line0 &= INVMASK32_BYTE2;
    line0 |= line2 & MASK32_BYTE2;
    line2 &= INVMASK32_BYTE2;
    line2 |= temp;

    temp = line1 & MASK32_BYTE2;
    line1 &= INVMASK32_BYTE2;
    line1 |= line3 & MASK32_BYTE2;
    line3 &= INVMASK32_BYTE2;
    line3 |= temp;

    temp = line0 & MASK32_BYTE3;
    line0 &= INVMASK32_BYTE3;
    line0 |= line3 & MASK32_BYTE3;
    line3 &= INVMASK32_BYTE3;
    line3 |= line2 & MASK32_BYTE3;
    line2 &= INVMASK32_BYTE3;
    line2 |= line1 & MASK32_BYTE3;
    line1 &= INVMASK32_BYTE3;
    line1 |= temp;

    *((uint32_t*)state) = line0;
    *((uint32_t*)state + 1) = line1;
    *((uint32_t*)state + 2) = line2;
    *((uint32_t*)state + 3) = line3;
#else

  uint8_t temp;

  // Rotate first row 1 columns to left  
  temp           = (*state)[0][1];
  (*state)[0][1] = (*state)[1][1];
  (*state)[1][1] = (*state)[2][1];
  (*state)[2][1] = (*state)[3][1];
  (*state)[3][1] = temp;

  // Rotate second row 2 columns to left  
  temp           = (*state)[0][2];
  (*state)[0][2] = (*state)[2][2];
  (*state)[2][2] = temp;

  temp           = (*state)[1][2];
  (*state)[1][2] = (*state)[3][2];
  (*state)[3][2] = temp;

  // Rotate third row 3 columns to left
  temp           = (*state)[0][3];
  (*state)[0][3] = (*state)[3][3];
  (*state)[3][3] = (*state)[2][3];
  (*state)[2][3] = (*state)[1][3];
  (*state)[1][3] = temp;
#endif
}

#if 0 /*removing because xtime has been redefined*/
#ifndef _DEBUG
static inline uint32_t xtime(uint32_t x)
{
  return ((x<<1) ^ (((x>>7)/* & 1*/) * 0x1b)) & 0xFF;
}
static inline uint64_t xtime64(uint64_t x)
{
    return ((x << 1) ^ (((x >> 7)/* & 1*/) * 0x1b)) & 0xFF;
}
#else
#define xtime(x) ((((x)<<1) ^ ((((x)>>7)) * 0x1b)) & 0xFF)
#endif
#endif

/* xtime
 * original xtime function operated on bytes, this optimized version performs the same
 * calculation on all bytes in a dword simultaneously, reducing operations and memory accesses
 */
static inline uint32_t xtime(uint32_t x)
{
  return ((x&0x7f7f7f7f)<<1)^(((x&0x80808080)>>7)*0x1b);
}
static inline uint64_t xtime64(uint64_t x)
{
    return ((x << 1) ^ (((x >> 7)/* & 1*/) * 0x1b)) & 0xFF;
}

/* MixColumns
 * replaced byte-wise operations with word-based operations
 * eliminated repeated calculations
 */
static void MixColumns(state_t* state)
{
  unsigned int *sp = (unsigned int *) state;

  for (int i=4;i;--i,sp++)
    *sp = xtime((*sp) ^ (((*sp)>>8)|((*sp)<<24))) ^
            (((*sp)<<8)|((*sp)>>24)) ^
            (((*sp)<<16)|((*sp)>>16)) ^ (((*sp)<<24)|((*sp)>>8));
}

/*removed old Multiply function - now integrated into InvMixColumns */
// Multiply is used to multiply numbers in the field GF(2^8)
// Note: The last call to xtime() is unneeded, but often ends up generating a smaller binary
//       The compiler seems to be able to vectorize the operation better this way.
//       See https://github.com/kokke/tiny-AES-c/pull/34
// Multiply is used to multiply numbers in the field GF(2^8)
// Note: The last call to xtime() is unneeded, but often ends up generating a smaller binary
//       The compiler seems to be able to vectorize the operation better this way.
//       See https://github.com/kokke/tiny-AES-c/pull/34
//#if MULTIPLY_AS_A_FUNCTION
#ifndef _DEBUG
static inline uint32_t Multiply(uint32_t x, uint32_t y)
{
    uint32_t xtimeX = xtime(x);
    uint32_t xtimeXX = xtime(xtimeX);
    uint32_t xtimeXXX = xtime(xtimeXX);

    return ((~((y & 1)-1) & x) ^
            (~((y>>1 & 1)-1) & xtimeX) ^
            (~((y>>2 & 1)-1) & xtimeXX) ^
            (~((y>>3 & 1)-1) & xtimeXXX)
#if defined(_MSC_VER) && defined(_M_AMD64)
            ^
       (~((y>>4 & 1)-1) & xtime(xtimeXXX))
#endif
    ); /* this last call to xtime() can be omitted */
}

#else
#define Multiply(x, y)                                \
      ((~((y & 1)-1) & x) ^                              \
      (~((y>>1 & 1)-1) & xtime(x)) ^                       \
      (~((y>>2 & 1)-1) & xtime(xtime(x))) ^                \
      (~((y>>3 & 1)-1) & xtime(xtime(xtime(x)))) )   \


#endif

#if (defined(CBC) && CBC == 1) || (defined(ECB) && ECB == 1)
// MixColumns function mixes the columns of the state matrix.
// The method used to multiply may be difficult to understand for the inexperienced.
// Please use the references to gain more information.
/* InvMixColumns
 * this a more optimal version which performs parallel computation on all bytes in
 * a dword and applies the Multiply & xor and unrolls the multiple calls to Multiply
 * for each 0x9,0xb,0xd and 0xe perturbation
 */
static void InvMixColumns(state_t* state)
{
#if !defined(_M_X64)    // This approach runs a bit faster on arm64-v8a and possibly others
    for (int i = 0; i < 4; ++i)
    {
        uint32_t *pLine = ((uint32_t *) state + i);
        uint32_t line = *pLine;

        uint32_t byte0 = (line & MASK32_BYTE0) >> OFS32_BYTE0;
        uint32_t byte1 = (line & MASK32_BYTE1) >> OFS32_BYTE1;
        uint32_t byte2 = (line & MASK32_BYTE2) >> OFS32_BYTE2;
        uint32_t byte3 = (line & MASK32_BYTE3) >> OFS32_BYTE3;

        line = (Multiply(byte0, 0x0e) ^ Multiply(byte1, 0x0b) ^ Multiply(byte2, 0x0d) ^
                Multiply(byte3, 0x09)) << OFS32_BYTE0;
        line |= (Multiply(byte0, 0x09) ^ Multiply(byte1, 0x0e) ^ Multiply(byte2, 0x0b) ^
                 Multiply(byte3, 0x0d)) << OFS32_BYTE1;
        line |= (Multiply(byte0, 0x0d) ^ Multiply(byte1, 0x09) ^ Multiply(byte2, 0x0e) ^
                 Multiply(byte3, 0x0b)) << OFS32_BYTE2;
        line |= (Multiply(byte0, 0x0b) ^ Multiply(byte1, 0x0d) ^ Multiply(byte2, 0x09) ^
                 Multiply(byte3, 0x0e)) << OFS32_BYTE3;

        *pLine = line;
    }
#else   //  This way is more efficient on the x64 Intel/AMD architecture
    uint32_t *sp=(uint32_t*)state;
    uint32_t xtimeX;
    uint32_t xtimeXX;
    uint32_t xtimeXXX;
    uint32_t xtime_x9;
    uint32_t xtime_xb;
    uint32_t xtime_xd;
    uint32_t xtime_xe;
    //*sp++ = i;

    for (int i=4; i; --i, sp++) 
    {
        uint32_t spVal = *sp;
        xtimeX = xtime(spVal);
        xtimeXX = xtime(xtimeX);
        xtimeXXX = xtime(xtimeXX);

        xtime_x9 = xtimeXXX ^ spVal;
        xtime_xb = xtimeXXX ^ xtimeX ^ spVal;
        xtime_xd = xtimeXXX ^ xtimeXX ^ spVal;
        xtime_xe = xtimeXXX ^ xtimeXX ^ xtimeX;

        uint32_t xtime_xb_r8 =  xtime_xb >> 8;
        uint32_t xtime_xd_r16 = xtime_xd >> 16;
        uint32_t xtime_x9_l8 =  xtime_x9 << 8;
        uint32_t xtime_xd_l16 = xtime_xd << 16;

        //this next assignment incorporates all of the Multiply calls, eliminating the repeated re-calculations
        *sp =
        /* byte 0:*/ (((xtime_xe         ^ xtime_xb_r8  ^ xtime_xd_r16 ^ (xtime_x9 >> 24)) & 0x000000ff) |
        /* byte 1:*/  ((xtime_x9_l8      ^ xtime_xe     ^ xtime_xb_r8  ^ xtime_xd_r16    ) & 0x0000ff00) |
        /* byte 2:*/  ((xtime_xd_l16     ^ xtime_x9_l8  ^ xtime_xe     ^ xtime_xb_r8     ) & 0x00ff0000) |
        /* byte 3:*/  (((xtime_xb << 24) ^ xtime_xd_l16 ^ xtime_x9_l8  ^ xtime_xe        ) & 0xff000000));
    }
#endif
}


// The SubBytes Function Substitutes the values in the
// state matrix with values in an S-box.
static void InvSubBytes(state_t* state)
{
#if 0
  unsigned int i, j;
  for (i = 0; i < 4; ++i)
  {
    for (j = 0; j < 4; ++j)
    {
//      (*state)[j][i] = getSBoxInvert((*state)[j][i]);
        (*state)[i][j] = getSBoxInvert((*state)[i][j]);
    }
  }
#else
    unsigned int i;
    for (i = 0; i < 4; i++)
    {
        uint32_t* pLine = ((uint32_t*)state + i);
        uint32_t line = *pLine;

        uint32_t byte0 = (line & MASK32_BYTE0) >> OFS32_BYTE0;
        uint32_t byte1 = (line & MASK32_BYTE1) >> OFS32_BYTE1;
        uint32_t byte2 = (line & MASK32_BYTE2) >> OFS32_BYTE2;
        uint32_t byte3 = (line & MASK32_BYTE3) >> OFS32_BYTE3;

        *pLine = (getSBoxInvert(byte0) << OFS32_BYTE0) | (getSBoxInvert(byte1) << OFS32_BYTE1) | (getSBoxInvert(byte2) << OFS32_BYTE2) | (getSBoxInvert(byte3) << OFS32_BYTE3);
    }
#endif
}

static void InvShiftRows(state_t* state)
{
#if 1// GK
     uint32_t  line0 = *((uint32_t*)state);
     uint32_t  line1 = *((uint32_t*)state + 1);
     uint32_t  line2 = *((uint32_t*)state + 2);
     uint32_t  line3 = *((uint32_t*)state + 3);

     uint32_t temp;

    temp = line3 & MASK32_BYTE1;
    line3 &= INVMASK32_BYTE1;
    line3 |= line2 & MASK32_BYTE1;
    line2 &= INVMASK32_BYTE1;
    line2 |= line1 & MASK32_BYTE1;
    line1 &= INVMASK32_BYTE1;
    line1 |= line0 & MASK32_BYTE1;
    line0 &= INVMASK32_BYTE1;
    line0 |= temp;

    temp = line0 & MASK32_BYTE2;
    line0 &= INVMASK32_BYTE2;
    line0 |= line2 & MASK32_BYTE2;
    line2 &= INVMASK32_BYTE2;
    line2 |= temp;

    temp = line1 & MASK32_BYTE2;
    line1 &= INVMASK32_BYTE2;
    line1 |= line3 & MASK32_BYTE2;
    line3 &= INVMASK32_BYTE2;
    line3 |= temp;

    temp = line0 & MASK32_BYTE3;
    line0 &= INVMASK32_BYTE3;
    line0 |= line1 & MASK32_BYTE3;
    line1 &= INVMASK32_BYTE3;
    line1 |= line2 & MASK32_BYTE3;
    line2 &= INVMASK32_BYTE3;
    line2 |= line3 & MASK32_BYTE3;
    line3 &= INVMASK32_BYTE3;
    line3 |= temp;

    *((uint32_t*)state) = line0;
    *((uint32_t*)state + 1) = line1;
    *((uint32_t*)state + 2) = line2;
    *((uint32_t*)state + 3) = line3;
#else
  uint8_t temp;

  // Rotate first row 1 columns to right  
  temp = (*state)[3][1];
  (*state)[3][1] = (*state)[2][1];
  (*state)[2][1] = (*state)[1][1];
  (*state)[1][1] = (*state)[0][1];
  (*state)[0][1] = temp;

  // Rotate second row 2 columns to right 
  temp = (*state)[0][2];
  (*state)[0][2] = (*state)[2][2];
  (*state)[2][2] = temp;

  temp = (*state)[1][2];
  (*state)[1][2] = (*state)[3][2];
  (*state)[3][2] = temp;

  // Rotate third row 3 columns to right
  temp = (*state)[0][3];
  (*state)[0][3] = (*state)[1][3];
  (*state)[1][3] = (*state)[2][3];
  (*state)[2][3] = (*state)[3][3];
  (*state)[3][3] = temp;
#endif
}
#endif // #if (defined(CBC) && CBC == 1) || (defined(ECB) && ECB == 1)

// Cipher is the main function that encrypts the PlainText.
static void Cipher(state_t* state, const uint8_t* RoundKey)
{
  uint32_t round = 0;

  // Add the First round key to the state before starting the rounds.
  AddRoundKey(0, state, RoundKey);

  // There will be Nr rounds.
  // The first Nr-1 rounds are identical.
  // These Nr rounds are executed in the loop below.
  // Last one without MixColumns()
  for (round = 1; ; ++round)
  {
    SubBytes(state);
    ShiftRows(state);
    if (round == Nr) {
      break;
    }
    MixColumns(state);
    AddRoundKey(round, state, RoundKey);
  }
  // Add round key to last round
  AddRoundKey(Nr, state, RoundKey);
}

#if (defined(CBC) && CBC == 1) || (defined(ECB) && ECB == 1)
static void InvCipher(state_t* state, const uint8_t* RoundKey)
{
  uint8_t round = 0;

  // Add the First round key to the state before starting the rounds.
  AddRoundKey(Nr, state, RoundKey);

  // There will be Nr rounds.
  // The first Nr-1 rounds are identical.
  // These Nr rounds are executed in the loop below.
  // Last one without InvMixColumn()
  for (round = (Nr - 1); ; --round)
  {
    InvShiftRows(state);
    InvSubBytes(state);
    AddRoundKey(round, state, RoundKey);
    if (round == 0) {
      break;
    }
    InvMixColumns(state);
  }

}
#endif // #if (defined(CBC) && CBC == 1) || (defined(ECB) && ECB == 1)

/*****************************************************************************/
/* Public functions:                                                         */
/*****************************************************************************/
#if defined(ECB) && (ECB == 1)


void AES_ECB_encrypt(const struct AES_ctx* ctx, uint8_t* buf)
{
  // The next function call encrypts the PlainText with the Key using AES algorithm.
  Cipher((state_t*)buf, ctx->RoundKey);
}

void AES_ECB_decrypt(const struct AES_ctx* ctx, uint8_t* buf)
{
  // The next function call decrypts the PlainText with the Key using AES algorithm.
  InvCipher((state_t*)buf, ctx->RoundKey);
}


#endif // #if defined(ECB) && (ECB == 1)





#if defined(CBC) && (CBC == 1)


static inline void XorWithIv(uint8_t* buf, const uint8_t* Iv)
{
    *((uint64_t*)buf) ^= *((uint64_t*)Iv);
    *((uint64_t*)buf + 1) ^= *((uint64_t*)Iv + 1);
    /* GK
  unsigned int i;
  for (i = 0; i < AES_BLOCKLEN; ++i) // The block in AES is always 128bit no matter the key size
  {
    buf[i] ^= Iv[i];
  }
*/
}

void AES_CBC_encrypt_buffer(struct AES_ctx *ctx, uint8_t* buf, uint32_t length)
{
  uintptr_t i;
  uint8_t *Iv = ctx->Iv;
  for (i = 0; i < length; i += AES_BLOCKLEN)
  {
    XorWithIv(buf, Iv);
    Cipher((state_t*)buf, ctx->RoundKey);
    Iv = buf;
    buf += AES_BLOCKLEN;
  }
  /* store Iv in ctx for next call */
  memcpy(ctx->Iv, Iv, AES_BLOCKLEN);
}

void AES_CBC_decrypt_buffer(struct AES_ctx* ctx, uint8_t* buf,  uint32_t length)
{
  uintptr_t i;
  uint8_t storeNextIv[AES_BLOCKLEN];
  for (i = 0; i < length; i += AES_BLOCKLEN)
  {
    memcpy(storeNextIv, buf, AES_BLOCKLEN);
    InvCipher((state_t*)buf, ctx->RoundKey);
    XorWithIv(buf, ctx->Iv);
    memcpy(ctx->Iv, storeNextIv, AES_BLOCKLEN);
    buf += AES_BLOCKLEN;
  }

}

#endif // #if defined(CBC) && (CBC == 1)



#if defined(CTR) && (CTR == 1)

/* Symmetrical operation: same function for encrypting as for decrypting. Note any IV/nonce should never be reused with the same key */
void AES_CTR_xcrypt_buffer(struct AES_ctx* ctx, uint8_t* buf, uint32_t length)
{
  uint8_t buffer[AES_BLOCKLEN];
  
  unsigned i;
  int bi;
  for (i = 0, bi = AES_BLOCKLEN; i < length; ++i, ++bi)
  {
    if (bi == AES_BLOCKLEN) /* we need to regen xor compliment in buffer */
    {
      
      memcpy(buffer, ctx->Iv, AES_BLOCKLEN);
      Cipher((state_t*)buffer,ctx->RoundKey);

      /* Increment Iv and handle overflow */
      for (bi = (AES_BLOCKLEN - 1); bi >= 0; --bi)
      {
	/* inc will overflow */
        if (ctx->Iv[bi] == 255)
	{
          ctx->Iv[bi] = 0;
          continue;
        } 
        ctx->Iv[bi] += 1;
        break;   
      }
      bi = 0;
    }

    buf[i] = (buf[i] ^ buffer[bi]);
  }
}

#endif // #if defined(CTR) && (CTR == 1)

