/*
 * 86Box    A hypervisor and IBM PC system emulator that specializes in
 *          running old operating systems and software designed for IBM
 *          PC systems and compatibles from 1981 through fairly recent
 *          system designs based on the PCI bus.
 *
 *          This file is part of the 86Box distribution.
 *
 *          Emulation of Tandy models 1000, 1000HX and 1000SL2.
 *
 *
 *
 * Authors: Sarah Walker, <https://pcem-emulator.co.uk/>
 *          Miran Grca, <mgrca8@gmail.com>
 *
 *          Copyright 2008-2019 Sarah Walker.
 *          Copyright 2016-2019 Miran Grca.
 */
#include <stdarg.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <wchar.h>
#include <math.h>
#define HAVE_STDARG_H
#include <86box/86box.h>
#include <86box/timer.h>
#include <86box/io.h>
#include <86box/pic.h>
#include <86box/pit.h>
#include <86box/nmi.h>
#include <86box/mem.h>
#include <86box/rom.h>
#include <86box/device.h>
#include <86box/nvr.h>
#include <86box/fdd.h>
#include <86box/fdc.h>
#include <86box/fdc_ext.h>
#include <86box/gameport.h>
#include <86box/keyboard.h>
#include <86box/sound.h>
#include <86box/snd_sn76489.h>
#include <86box/video.h>
#include <86box/vid_cga_comp.h>
#include <86box/machine.h>
#include <86box/m_tandy.h>
#include <86box/plat_unused.h>


enum {
    TYPE_TANDY = 0,
    TYPE_TANDY1000SX,
    TYPE_TANDY1000HX,
    TYPE_TANDY1000SL2
};

enum {
    EEPROM_IDLE = 0,
    EEPROM_GET_OPERATION,
    EEPROM_READ,
    EEPROM_WRITE
};


static const scancode scancode_tandy[512] = {
  // clang-format off
    { .mk = {            0 }, .brk = {                   0 } }, /* 000 */
    { .mk = {      0x01, 0 }, .brk = {             0x81, 0 } }, /* 001 */
    { .mk = {      0x02, 0 }, .brk = {             0x82, 0 } }, /* 002 */
    { .mk = {      0x03, 0 }, .brk = {             0x83, 0 } }, /* 003 */
    { .mk = {      0x04, 0 }, .brk = {             0x84, 0 } }, /* 004 */
    { .mk = {      0x05, 0 }, .brk = {             0x85, 0 } }, /* 005 */
    { .mk = {      0x06, 0 }, .brk = {             0x86, 0 } }, /* 006 */
    { .mk = {      0x07, 0 }, .brk = {             0x87, 0 } }, /* 007 */
    { .mk = {      0x08, 0 }, .brk = {             0x88, 0 } }, /* 008 */
    { .mk = {      0x09, 0 }, .brk = {             0x89, 0 } }, /* 009 */
    { .mk = {      0x0a, 0 }, .brk = {             0x8a, 0 } }, /* 00a */
    { .mk = {      0x0b, 0 }, .brk = {             0x8b, 0 } }, /* 00b */
    { .mk = {      0x0c, 0 }, .brk = {             0x8c, 0 } }, /* 00c */
    { .mk = {      0x0d, 0 }, .brk = {             0x8d, 0 } }, /* 00d */
    { .mk = {      0x0e, 0 }, .brk = {             0x8e, 0 } }, /* 00e */
    { .mk = {      0x0f, 0 }, .brk = {             0x8f, 0 } }, /* 00f */
    { .mk = {      0x10, 0 }, .brk = {             0x90, 0 } }, /* 010 */
    { .mk = {      0x11, 0 }, .brk = {             0x91, 0 } }, /* 011 */
    { .mk = {      0x12, 0 }, .brk = {             0x92, 0 } }, /* 013 */
    { .mk = {      0x13, 0 }, .brk = {             0x93, 0 } }, /* 013 */
    { .mk = {      0x14, 0 }, .brk = {             0x94, 0 } }, /* 014 */
    { .mk = {      0x15, 0 }, .brk = {             0x95, 0 } }, /* 015 */
    { .mk = {      0x16, 0 }, .brk = {             0x96, 0 } }, /* 016 */
    { .mk = {      0x17, 0 }, .brk = {             0x97, 0 } }, /* 017 */
    { .mk = {      0x18, 0 }, .brk = {             0x98, 0 } }, /* 018 */
    { .mk = {      0x19, 0 }, .brk = {             0x99, 0 } }, /* 019 */
    { .mk = {      0x1a, 0 }, .brk = {             0x9a, 0 } }, /* 01a */
    { .mk = {      0x1b, 0 }, .brk = {             0x9b, 0 } }, /* 01b */
    { .mk = {      0x1c, 0 }, .brk = {             0x9c, 0 } }, /* 01c */
    { .mk = {      0x1d, 0 }, .brk = {             0x9d, 0 } }, /* 01d */
    { .mk = {      0x1e, 0 }, .brk = {             0x9e, 0 } }, /* 01e */
    { .mk = {      0x1f, 0 }, .brk = {             0x9f, 0 } }, /* 01f */
    { .mk = {      0x20, 0 }, .brk = {             0xa0, 0 } }, /* 020 */
    { .mk = {      0x21, 0 }, .brk = {             0xa1, 0 } }, /* 021 */
    { .mk = {      0x22, 0 }, .brk = {             0xa2, 0 } }, /* 022 */
    { .mk = {      0x23, 0 }, .brk = {             0xa3, 0 } }, /* 023 */
    { .mk = {      0x24, 0 }, .brk = {             0xa4, 0 } }, /* 024 */
    { .mk = {      0x25, 0 }, .brk = {             0xa5, 0 } }, /* 025 */
    { .mk = {      0x26, 0 }, .brk = {             0xa6, 0 } }, /* 026 */
    { .mk = {      0x27, 0 }, .brk = {             0xa7, 0 } }, /* 027 */
    { .mk = {      0x28, 0 }, .brk = {             0xa8, 0 } }, /* 028 */
    { .mk = {      0x29, 0 }, .brk = {             0xa9, 0 } }, /* 029 */
    { .mk = {      0x2a, 0 }, .brk = {             0xaa, 0 } }, /* 02a */
    { .mk = {      0x2b, 0 }, .brk = {             0xab, 0 } }, /* 02b */
    { .mk = {      0x2c, 0 }, .brk = {             0xac, 0 } }, /* 02c */
    { .mk = {      0x2d, 0 }, .brk = {             0xad, 0 } }, /* 02d */
    { .mk = {      0x2e, 0 }, .brk = {             0xae, 0 } }, /* 02e */
    { .mk = {      0x2f, 0 }, .brk = {             0xaf, 0 } }, /* 02f */
    { .mk = {      0x30, 0 }, .brk = {             0xb0, 0 } }, /* 030 */
    { .mk = {      0x31, 0 }, .brk = {             0xb1, 0 } }, /* 031 */
    { .mk = {      0x32, 0 }, .brk = {             0xb2, 0 } }, /* 032 */
    { .mk = {      0x33, 0 }, .brk = {             0xb3, 0 } }, /* 033 */
    { .mk = {      0x34, 0 }, .brk = {             0xb4, 0 } }, /* 034 */
    { .mk = {      0x35, 0 }, .brk = {             0xb5, 0 } }, /* 035 */
    { .mk = {      0x36, 0 }, .brk = {             0xb6, 0 } }, /* 036 */
    { .mk = {      0x37, 0 }, .brk = {             0xb7, 0 } }, /* 037 */
    { .mk = {      0x38, 0 }, .brk = {             0xb8, 0 } }, /* 038 */
    { .mk = {      0x39, 0 }, .brk = {             0xb9, 0 } }, /* 039 */
    { .mk = {      0x3a, 0 }, .brk = {             0xba, 0 } }, /* 03a */
    { .mk = {      0x3b, 0 }, .brk = {             0xbb, 0 } }, /* 03b */
    { .mk = {      0x3c, 0 }, .brk = {             0xbc, 0 } }, /* 03c */
    { .mk = {      0x3d, 0 }, .brk = {             0xbd, 0 } }, /* 03d */
    { .mk = {      0x3e, 0 }, .brk = {             0xbe, 0 } }, /* 03e */
    { .mk = {      0x3f, 0 }, .brk = {             0xbf, 0 } }, /* 03f */
    { .mk = {      0x40, 0 }, .brk = {             0xc0, 0 } }, /* 040 */
    { .mk = {      0x41, 0 }, .brk = {             0xc1, 0 } }, /* 041 */
    { .mk = {      0x42, 0 }, .brk = {             0xc2, 0 } }, /* 042 */
    { .mk = {      0x43, 0 }, .brk = {             0xc3, 0 } }, /* 043 */
    { .mk = {      0x44, 0 }, .brk = {             0xc4, 0 } }, /* 044 */
    { .mk = {      0x45, 0 }, .brk = {             0xc5, 0 } }, /* 045 */
    { .mk = {      0x46, 0 }, .brk = {             0xc6, 0 } }, /* 046 */
    { .mk = {      0x47, 0 }, .brk = {             0xc7, 0 } }, /* 047 */
    { .mk = {      0x48, 0 }, .brk = {             0xc8, 0 } }, /* 048 */
    { .mk = {      0x49, 0 }, .brk = {             0xc9, 0 } }, /* 049 */
    { .mk = {      0x4a, 0 }, .brk = {             0xca, 0 } }, /* 04a */
    { .mk = {      0x4b, 0 }, .brk = {             0xcb, 0 } }, /* 04b */
    { .mk = {      0x4c, 0 }, .brk = {             0xcc, 0 } }, /* 04c */
    { .mk = {      0x4d, 0 }, .brk = {             0xcd, 0 } }, /* 04d */
    { .mk = {      0x4e, 0 }, .brk = {             0xce, 0 } }, /* 04e */
    { .mk = {      0x4f, 0 }, .brk = {             0xcf, 0 } }, /* 04f */
    { .mk = {      0x50, 0 }, .brk = {             0xd0, 0 } }, /* 050 */
    { .mk = {      0x51, 0 }, .brk = {             0xd1, 0 } }, /* 051 */
    { .mk = {      0x52, 0 }, .brk = {             0xd2, 0 } }, /* 052 */
    { .mk = {      0x56, 0 }, .brk = {             0xd6, 0 } }, /* 053 */
    { .mk = {      0x54, 0 }, .brk = {             0xd4, 0 } }, /* 054 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 055 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 056 */
    { .mk = {      0x59, 0 }, .brk = {             0xd9, 0 } }, /* 057 */
    { .mk = {      0x5a, 0 }, .brk = {             0xda, 0 } }, /* 058 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 059 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 05a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 05b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 05c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 05d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 05e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 05f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 060 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 061 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 062 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 063 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 064 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 065 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 066 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 067 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 068 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 069 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 06a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 06b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 06c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 06d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 06e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 06f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 070 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 071 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 072 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 073 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 074 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 075 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 076 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 077 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 078 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 079 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 07a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 07b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 07c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 07d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 07e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 07f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 080 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 081 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 082 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 083 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 084 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 085 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 086 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 087 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 088 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 089 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 08a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 08b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 08c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 08d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 08e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 08f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 090 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 091 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 092 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 093 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 094 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 095 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 096 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 097 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 098 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 099 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 09a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 09b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 09c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 09d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 09e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 09f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0a0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0a1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0a2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0a3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0a4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0a5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0a6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0a7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0a8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0a9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0aa */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ab */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ac */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ad */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ae */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0af */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0b0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0b1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0b2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0b3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0b4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0b5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0b6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0b7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0b8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0b9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ba */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0bb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0bc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0bd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0be */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0bf */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0c0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0c1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0c2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0c3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0c4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0c5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0c6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0c7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0c8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0c9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ca */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0cb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0cc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0cd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ce */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0cf */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0d0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0d1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0d2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0d3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0d4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0d5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0d6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0d7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0d8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0d9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0da */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0db */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0dc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0dd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0de */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0df */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0e0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0e1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0e2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0e3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0e4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0e5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0e6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0e7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0e8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0e9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ea */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0eb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ec */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ed */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ee */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ef */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0fa */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0fb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0fc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0fd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0fe */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0ff */
    { .mk = {            0 }, .brk = {                   0 } }, /* 100 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 101 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 102 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 103 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 104 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 105 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 106 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 107 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 108 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 109 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 10a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 10b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 10c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 10d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 10e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 10f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 110 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 111 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 112 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 113 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 114 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 115 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 116 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 117 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 118 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 119 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 11a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 11b */
    { .mk = {      0x57, 0 }, .brk = {             0xd7, 0 } }, /* 11c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 11d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 11e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 11f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 120 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 121 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 122 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 123 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 124 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 125 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 126 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 127 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 128 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 129 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 12a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 12b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 12c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 12d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 12e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 12f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 130 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 131 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 132 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 133 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 134 */
    { .mk = {      0x35, 0 }, .brk = {             0xb5, 0 } }, /* 135 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 136 */
    { .mk = {      0x37, 0 }, .brk = {             0xb7, 0 } }, /* 137 */
    { .mk = {      0x38, 0 }, .brk = {             0xb8, 0 } }, /* 138 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 139 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 13a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 13b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 13c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 13d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 13e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 13f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 140 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 141 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 142 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 143 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 144 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 145 */
    { .mk = {      0x46, 0 }, .brk = {             0xc6, 0 } }, /* 146 */
    { .mk = {      0x58, 0 }, .brk = {             0xd8, 0 } }, /* 147 */
    { .mk = {      0x29, 0 }, .brk = {             0xa9, 0 } }, /* 148 */
    { .mk = {      0x49, 0 }, .brk = {             0xc9, 0 } }, /* 149 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 14a */
    { .mk = {      0x2b, 0 }, .brk = {             0xab, 0 } }, /* 14b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 14c */
    { .mk = {      0x4e, 0 }, .brk = {             0xce, 0 } }, /* 14d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 14e */
    { .mk = {      0x4f, 0 }, .brk = {             0xcf, 0 } }, /* 14f */
    { .mk = {      0x4a, 0 }, .brk = {             0xca, 0 } }, /* 150 */
    { .mk = {      0x51, 0 }, .brk = {             0xd1, 0 } }, /* 151 */
    { .mk = {      0x55, 0 }, .brk = {             0xd5, 0 } }, /* 152 */
    { .mk = {      0x53, 0 }, .brk = {             0xd3, 0 } }, /* 153 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 154 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 155 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 156 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 157 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 158 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 159 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 15a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 15b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 15c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 15d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 15e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 15f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 160 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 161 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 162 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 163 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 164 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 165 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 166 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 167 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 168 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 169 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 16a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 16b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 16c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 16d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 16e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 16f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 170 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 171 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 172 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 173 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 174 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 175 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 176 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 177 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 178 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 179 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 17a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 17b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 17c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 17d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 17e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 17f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 180 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 181 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 182 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 183 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 184 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 185 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 186 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 187 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 188 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 189 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 190 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 191 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 192 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 193 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 194 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 195 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 196 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 197 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 198 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 199 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1aa */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ab */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ac */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ad */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ae */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1af */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ba */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1be */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bf */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ca */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ce */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cf */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1da */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1db */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1dc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1dd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1de */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1df */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ea */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1eb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ec */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ed */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ee */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ef */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fa */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fe */
    { .mk = {            0 }, .brk = {                   0 } }  /* 1ff */
  // clang-format on
};

static int eep_data_out;

#ifdef ENABLE_TANDY_LOG
int tandy_do_log = ENABLE_TANDY_LOG;

static void
tandy_log(const char *fmt, ...)
{
    va_list ap;

    if (tandy_do_log) {
        va_start(ap, fmt);
        pclog_ex(fmt, ap);
        va_end(ap);
    }
}
#else
#    define tandy_log(fmt, ...)
#endif



static void
eep_write(UNUSED(uint16_t addr), uint8_t val, void *priv)
{
    t1keep_t *eep = (t1keep_t *) priv;

    if ((val & 4) && !eep->clk)
        switch (eep->state) {
            case EEPROM_IDLE:
                switch (eep->count) {
                    case 0:
                        if (!(val & 3))
                            eep->count = 1;
                        else
                            eep->count = 0;
                        break;

                    case 1:
                        if ((val & 3) == 2)
                            eep->count = 2;
                        else
                            eep->count = 0;
                        break;

                    case 2:
                        if ((val & 3) == 3)
                            eep->state = EEPROM_GET_OPERATION;
                        eep->count = 0;
                        break;

                    default:
                        break;
                }
                break;

            case EEPROM_GET_OPERATION:
                eep->data = (eep->data << 1) | (val & 1);
                eep->count++;
                if (eep->count == 8) {
                    eep->count = 0;
                    eep->addr  = eep->data & 0x3f;
                    switch (eep->data & 0xc0) {
                        case 0x40:
                            eep->state = EEPROM_WRITE;
                            break;

                        case 0x80:
                            eep->state = EEPROM_READ;
                            eep->data  = eep->store[eep->addr];
                            break;

                        default:
                            eep->state = EEPROM_IDLE;
                            break;
                    }
                }
                break;

            case EEPROM_READ:
                eep_data_out = eep->data & 0x8000;
                eep->data <<= 1;
                eep->count++;
                if (eep->count == 16) {
                    eep->count = 0;
                    eep->state = EEPROM_IDLE;
                }
                break;

            case EEPROM_WRITE:
                eep->data = (eep->data << 1) | (val & 1);
                eep->count++;
                if (eep->count == 16) {
                    eep->count            = 0;
                    eep->state            = EEPROM_IDLE;
                    eep->store[eep->addr] = eep->data;
                }
                break;

            default:
                break;
        }

    eep->clk = val & 4;
}

static void *
eep_init(const device_t *info)
{
    t1keep_t *eep;
    FILE     *fp = NULL;

    eep = (t1keep_t *) calloc(1, sizeof(t1keep_t));

    switch (info->local) {
        case TYPE_TANDY1000HX:
            eep->path = "tandy1000hx.bin";
            break;

        case TYPE_TANDY1000SL2:
            eep->path = "tandy1000sl2.bin";
            break;

        default:
            break;
    }

    fp = nvr_fopen(eep->path, "rb");
    if (fp != NULL) {
        if (fread(eep->store, 1, 128, fp) != 128)
            fatal("eep_init(): Error reading Tandy EEPROM\n");
        (void) fclose(fp);
    } else
        memset(eep->store, 0x00, 128);

    io_sethandler(0x037c, 1, NULL, NULL, NULL, eep_write, NULL, NULL, eep);

    return eep;
}

static void
eep_close(void *priv)
{
    t1keep_t *eep = (t1keep_t *) priv;
    FILE     *fp  = NULL;

    fp = nvr_fopen(eep->path, "wb");
    if (fp != NULL) {
        (void) fwrite(eep->store, 128, 1, fp);
        (void) fclose(fp);
    }

    free(eep);
}

static const device_t eep_1000hx_device = {
    .name          = "Tandy 1000HX EEPROM",
    .internal_name = "eep_1000hx",
    .flags         = 0,
    .local         = TYPE_TANDY1000HX,
    .init          = eep_init,
    .close         = eep_close,
    .reset         = NULL,
    .available     = NULL,
    .speed_changed = NULL,
    .force_redraw  = NULL,
    .config        = NULL
};

static const device_t eep_1000sl2_device = {
    .name          = "Tandy 1000SL2 EEPROM",
    .internal_name = "eep_1000sl2",
    .flags         = 0,
    .local         = TYPE_TANDY1000SL2,
    .init          = eep_init,
    .close         = eep_close,
    .reset         = NULL,
    .available     = NULL,
    .speed_changed = NULL,
    .force_redraw  = NULL,
    .config        = NULL
};

static void
tandy_write(uint16_t addr, uint8_t val, void *priv)
{
    tandy_t *dev = (tandy_t *) priv;

    switch (addr) {
        case 0x00a0:
            if (dev->is_hx && (val & 0x10)) {
                dev->base = (mem_size - 256) * 1024;
                dev->mask = 0x3ffff;
                mem_mapping_set_addr(&ram_low_mapping, 0, dev->base);
                mem_mapping_set_addr(&dev->ram_mapping,
                                     (((val >> 1) & 7) - 1) * 128 * 1024, 0x40000);
            } else {
                dev->base = (mem_size - 128) * 1024;
                dev->mask = 0x1ffff;
                mem_mapping_set_addr(&ram_low_mapping, 0, dev->base);
                mem_mapping_set_addr(&dev->ram_mapping,
                                     ((val >> 1) & 7) * 128 * 1024, 0x20000);
            }
            if (dev->is_hx) {
                io_removehandler(0x03d0, 16,
                                 tandy_vid_in, NULL, NULL, tandy_vid_out, NULL, NULL, dev);
                if (val & 0x01)
                    mem_mapping_disable(&dev->vid->mapping);
                else {
                    io_sethandler(0x03d0, 16,
                                  tandy_vid_in, NULL, NULL, tandy_vid_out, NULL, NULL, dev);
                    mem_mapping_set_addr(&dev->vid->mapping, 0xb8000, 0x8000);
                }
            } else {
                if (val & 0x01)
                    mem_mapping_set_addr(&dev->vid->mapping, 0xc0000, 0x10000);
                else
                    mem_mapping_set_addr(&dev->vid->mapping, 0xb8000, 0x8000);
            }
            dev->ram_bank = val;
            break;

        case 0xffe8:
            if ((val & 0xe) == 0xe)
                mem_mapping_disable(&dev->ram_mapping);
            else
                mem_mapping_set_addr(&dev->ram_mapping,
                                     ((val >> 1) & 7) * 128 * 1024,
                                     0x20000);
            tandy_recalc_address_sl(dev);
            dev->ram_bank = val;
            break;

        case 0xffea:
            dev->rom_bank   = val;
            dev->rom_offset = ((val ^ 4) & 7) * 0x10000;
            mem_mapping_set_exec(&dev->rom_mapping,
                                 &dev->rom[dev->rom_offset]);
            break;

        default:
            break;
    }
}

static uint8_t
tandy_read(uint16_t addr, void *priv)
{
    const tandy_t *dev = (tandy_t *) priv;
    uint8_t        ret = 0xff;

    switch (addr) {
        case 0x00a0:
            ret = dev->ram_bank;
            break;

        case 0xffe8:
            ret = dev->ram_bank;
            break;

        case 0xffea:
            ret = (dev->rom_bank ^ 0x10);
            break;

        default:
            break;
    }

    return ret;
}

static void
write_ram(uint32_t addr, uint8_t val, void *priv)
{
    const tandy_t *dev = (tandy_t *) priv;

    ram[dev->base + (addr & dev->mask)] = val;
}

static uint8_t
read_ram(uint32_t addr, void *priv)
{
    const tandy_t *dev = (tandy_t *) priv;

    return (ram[dev->base + (addr & dev->mask)]);
}

static uint8_t
read_rom(uint32_t addr, void *priv)
{
    const tandy_t *dev   = (tandy_t *) priv;
    uint32_t       addr2 = (addr & 0xffff) + dev->rom_offset;

    return (dev->rom[addr2]);
}

static uint16_t
read_romw(uint32_t addr, void *priv)
{
    tandy_t *dev   = (tandy_t *) priv;
    uint32_t addr2 = (addr & 0xffff) + dev->rom_offset;

    return (*(uint16_t *) &dev->rom[addr2]);
}

static uint32_t
read_roml(uint32_t addr, void *priv)
{
    tandy_t *dev = (tandy_t *) priv;

    return (*(uint32_t *) &dev->rom[addr]);
}

static void
init_rom(tandy_t *dev)
{
    dev->rom = (uint8_t *) malloc(0x80000);

#if 1
    if (!rom_load_interleaved("roms/machines/tandy1000sl2/8079047.hu1",
                              "roms/machines/tandy1000sl2/8079048.hu2",
                              0x000000, 0x80000, 0, dev->rom)) {
        tandy_log("TANDY: unable to load BIOS for 1000/SL2 !\n");
        free(dev->rom);
        dev->rom = NULL;
        return;
    }
#else
    f  = rom_fopen("roms/machines/tandy1000sl2/8079047.hu1", "rb");
    ff = rom_fopen("roms/machines/tandy1000sl2/8079048.hu2", "rb");
    for (c = 0x0000; c < 0x80000; c += 2) {
        dev->rom[c]     = getc(f);
        dev->rom[c + 1] = getc(ff);
    }
    fclose(ff);
    fclose(f);
#endif

    mem_mapping_add(&dev->rom_mapping, 0xe0000, 0x10000,
                    read_rom, read_romw, read_roml, NULL, NULL, NULL,
                    dev->rom, MEM_MAPPING_EXTERNAL, dev);
}

static void
machine_tandy1k_init(const machine_t *model, int type)
{
    tandy_t *dev;

    dev = calloc(1, sizeof(tandy_t));

    machine_common_init(model);

    nmi_init();

    /*
     * Base 128K mapping is controlled via ports 0xA0 or
     * 0xFFE8 (SL2), so we remove it from the main mapping.
     */
    dev->base = (mem_size - 128) * 1024;
    dev->mask = 0x1ffff;
    mem_mapping_add(&dev->ram_mapping, 0x60000, 0x20000,
                    read_ram, NULL, NULL, write_ram, NULL, NULL, NULL,
                    MEM_MAPPING_INTERNAL, dev);
    mem_mapping_set_addr(&ram_low_mapping, 0, dev->base);

    device_add(&kbc_tandy_device);

    if (fdc_current[0] == FDC_INTERNAL)
        device_add(&fdc_xt_tandy_device);

    video_reset(gfxcard[0]);

    switch (type) {
        case TYPE_TANDY:
        case TYPE_TANDY1000SX:
            keyboard_set_table(scancode_tandy);
            io_sethandler(0x00a0, 1,
                          tandy_read, NULL, NULL, tandy_write, NULL, NULL, dev);
            device_context(&tandy_1000_video_device);
            tandy_vid_init(dev);
            device_context_restore();
            device_add_ex(&tandy_1000_video_device, dev);
            device_add((type == TYPE_TANDY1000SX) ? &ncr8496_device : &sn76489_device);
            break;

        case TYPE_TANDY1000HX:
            dev->is_hx = 1;
            keyboard_set_table(scancode_tandy);
            io_sethandler(0x00a0, 1,
                          tandy_read, NULL, NULL, tandy_write, NULL, NULL, dev);
            device_context(&tandy_1000hx_video_device);
            tandy_vid_init(dev);
            device_context_restore();
            device_add_ex(&tandy_1000hx_video_device, dev);
            device_add(&ncr8496_device);
            device_add(&eep_1000hx_device);
            break;

        case TYPE_TANDY1000SL2:
            dev->is_sl2 = 1;
            init_rom(dev);
            io_sethandler(0xffe8, 8,
                          tandy_read, NULL, NULL, tandy_write, NULL, NULL, dev);
            device_context(&tandy_1000sl_video_device);
            tandy_vid_init(dev);
            device_context_restore();
            device_add_ex(&tandy_1000sl_video_device, dev);
            device_add(&pssj_device);
            device_add(&eep_1000sl2_device);
            break;

        default:
            break;
    }

    standalone_gameport_type = &gameport_200_device;

    eep_data_out = 0x0000;
}

int
tandy1k_eeprom_read(void)
{
    return eep_data_out;
}

int
machine_tandy1000sx_init(const machine_t *model)
{
    int ret;

    ret = bios_load_linearr("roms/machines/tandy/tandy1t1.020",
                            0x000f0000, 131072, 0);

    if (bios_only || !ret)
        return ret;

    machine_tandy1k_init(model, TYPE_TANDY1000SX);

    return ret;
}

int
machine_tandy1000hx_init(const machine_t *model)
{
    int ret;

    ret = bios_load_linear("roms/machines/tandy1000hx/v020000.u12",
                           0x000e0000, 131072, 0);

    if (bios_only || !ret)
        return ret;

    machine_tandy1k_init(model, TYPE_TANDY1000HX);

    return ret;
}

int
machine_tandy1000sl2_init(const machine_t *model)
{
    int ret;

    ret = bios_load_interleaved("roms/machines/tandy1000sl2/8079047.hu1",
                                "roms/machines/tandy1000sl2/8079048.hu2",
                                0x000f0000, 65536, 0x18000);

    if (bios_only || !ret)
        return ret;

    machine_tandy1k_init(model, TYPE_TANDY1000SL2);

    return ret;
}
