#!/usr/bin/python
from __future__ import absolute_import, division, print_function

# Copyright: (c) 2022 Fortinet
# GNU General Public License v3.0+ (see LICENSE or https://www.gnu.org/licenses/gpl-3.0.txt)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

__metaclass__ = type

ANSIBLE_METADATA = {
    "status": ["preview"],
    "supported_by": "community",
    "metadata_version": "1.1",
}

DOCUMENTATION = """
---
module: fortios_system_vdom_property
short_description: Configure VDOM property in Fortinet's FortiOS and FortiGate.
description:
    - This module is able to configure a FortiGate or FortiOS (FOS) device by allowing the
      user to set and modify system feature and vdom_property category.
      Examples include all parameters and values need to be adjusted to datasources before usage.
      Tested with FOS v6.0.0
version_added: "2.0.0"
author:
    - Link Zheng (@chillancezen)
    - Jie Xue (@JieX19)
    - Hongbin Lu (@fgtdev-hblu)
    - Frank Shen (@frankshen01)
    - Miguel Angel Munoz (@mamunozgonzalez)
    - Nicolas Thomas (@thomnico)
notes:
    - Legacy fortiosapi has been deprecated, httpapi is the preferred way to run playbooks

requirements:
    - ansible>=2.14
options:
    access_token:
        description:
            - Token-based authentication.
              Generated from GUI of Fortigate.
        type: str
        required: false
    enable_log:
        description:
            - Enable/Disable logging for task.
        type: bool
        required: false
        default: false
    vdom:
        description:
            - Virtual domain, among those defined previously. A vdom is a
              virtual instance of the FortiGate that can be configured and
              used as a different unit.
        type: str
        default: root
    member_path:
        type: str
        description:
            - Member attribute path to operate on.
            - Delimited by a slash character if there are more than one attribute.
            - Parameter marked with member_path is legitimate for doing member operation.
    member_state:
        type: str
        description:
            - Add or delete a member under specified attribute path.
            - When member_state is specified, the state option is ignored.
        choices:
            - 'present'
            - 'absent'

    state:
        description:
            - Indicates whether to create or remove the object.
        type: str
        required: true
        choices:
            - 'present'
            - 'absent'
    system_vdom_property:
        description:
            - Configure VDOM property.
        default: null
        type: dict
        suboptions:
            custom_service:
                description:
                    - Maximum guaranteed number of firewall custom services.
                type: list
                elements: str
            description:
                description:
                    - Description.
                type: str
            dialup_tunnel:
                description:
                    - Maximum guaranteed number of dial-up tunnels.
                type: list
                elements: str
            firewall_address:
                description:
                    - Maximum guaranteed number of firewall addresses (IPv4, IPv6, multicast).
                type: list
                elements: str
            firewall_addrgrp:
                description:
                    - Maximum guaranteed number of firewall address groups (IPv4, IPv6).
                type: list
                elements: str
            firewall_policy:
                description:
                    - Maximum guaranteed number of firewall policies (policy, DoS-policy4, DoS-policy6, multicast).
                type: list
                elements: str
            ipsec_phase1:
                description:
                    - Maximum guaranteed number of VPN IPsec phase 1 tunnels.
                type: list
                elements: str
            ipsec_phase1_interface:
                description:
                    - Maximum guaranteed number of VPN IPsec phase1 interface tunnels.
                type: list
                elements: str
            ipsec_phase2:
                description:
                    - Maximum guaranteed number of VPN IPsec phase 2 tunnels.
                type: list
                elements: str
            ipsec_phase2_interface:
                description:
                    - Maximum guaranteed number of VPN IPsec phase2 interface tunnels.
                type: list
                elements: str
            log_disk_quota:
                description:
                    - Log disk quota in megabytes (MB). Range depends on how much disk space is available.
                type: list
                elements: str
            name:
                description:
                    - VDOM name. Source system.vdom.name.
                required: true
                type: str
            onetime_schedule:
                description:
                    - Maximum guaranteed number of firewall one-time schedules.
                type: list
                elements: str
            proxy:
                description:
                    - Maximum guaranteed number of concurrent proxy users.
                type: list
                elements: str
            recurring_schedule:
                description:
                    - Maximum guaranteed number of firewall recurring schedules.
                type: list
                elements: str
            service_group:
                description:
                    - Maximum guaranteed number of firewall service groups.
                type: list
                elements: str
            session:
                description:
                    - Maximum guaranteed number of sessions.
                type: list
                elements: str
            snmp_index:
                description:
                    - Permanent SNMP Index of the virtual domain (1 - 2147483647).
                type: int
            sslvpn:
                description:
                    - Maximum guaranteed number of SSL-VPNs.
                type: list
                elements: str
            user:
                description:
                    - Maximum guaranteed number of local users.
                type: list
                elements: str
            user_group:
                description:
                    - Maximum guaranteed number of user groups.
                type: list
                elements: str
"""

EXAMPLES = """
- name: Configure VDOM property.
  fortinet.fortios.fortios_system_vdom_property:
      vdom: "{{ vdom }}"
      state: "present"
      access_token: "<your_own_value>"
      system_vdom_property:
          custom_service: "<your_own_value>"
          description: "<your_own_value>"
          dialup_tunnel: "<your_own_value>"
          firewall_address: "<your_own_value>"
          firewall_addrgrp: "<your_own_value>"
          firewall_policy: "<your_own_value>"
          ipsec_phase1: "<your_own_value>"
          ipsec_phase1_interface: "<your_own_value>"
          ipsec_phase2: "<your_own_value>"
          ipsec_phase2_interface: "<your_own_value>"
          log_disk_quota: "<your_own_value>"
          name: "default_name_14 (source system.vdom.name)"
          onetime_schedule: "<your_own_value>"
          proxy: "<your_own_value>"
          recurring_schedule: "<your_own_value>"
          service_group: "<your_own_value>"
          session: "<your_own_value>"
          snmp_index: "0"
          sslvpn: "<your_own_value>"
          user: "<your_own_value>"
          user_group: "<your_own_value>"
"""

RETURN = """
build:
  description: Build number of the fortigate image
  returned: always
  type: str
  sample: '1547'
http_method:
  description: Last method used to provision the content into FortiGate
  returned: always
  type: str
  sample: 'PUT'
http_status:
  description: Last result given by FortiGate on last operation applied
  returned: always
  type: str
  sample: "200"
mkey:
  description: Master key (id) used in the last call to FortiGate
  returned: success
  type: str
  sample: "id"
name:
  description: Name of the table used to fulfill the request
  returned: always
  type: str
  sample: "urlfilter"
path:
  description: Path of the table used to fulfill the request
  returned: always
  type: str
  sample: "webfilter"
revision:
  description: Internal revision number
  returned: always
  type: str
  sample: "17.0.2.10658"
serial:
  description: Serial number of the unit
  returned: always
  type: str
  sample: "FGVMEVYYQT3AB5352"
status:
  description: Indication of the operation's result
  returned: always
  type: str
  sample: "success"
vdom:
  description: Virtual domain used
  returned: always
  type: str
  sample: "root"
version:
  description: Version of the FortiGate
  returned: always
  type: str
  sample: "v5.6.3"
"""
from ansible.module_utils.basic import AnsibleModule
from ansible.module_utils.connection import Connection
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    FortiOSHandler,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    check_legacy_fortiosapi,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    schema_to_module_spec,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    check_schema_versioning,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortimanager.common import (
    FAIL_SOCKET_MSG,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.data_post_processor import (
    remove_invalid_fields,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.comparison import (
    is_same_comparison,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.comparison import (
    serialize,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.comparison import (
    find_current_values,
)


def filter_system_vdom_property_data(json):
    option_list = [
        "custom_service",
        "description",
        "dialup_tunnel",
        "firewall_address",
        "firewall_addrgrp",
        "firewall_policy",
        "ipsec_phase1",
        "ipsec_phase1_interface",
        "ipsec_phase2",
        "ipsec_phase2_interface",
        "log_disk_quota",
        "name",
        "onetime_schedule",
        "proxy",
        "recurring_schedule",
        "service_group",
        "session",
        "snmp_index",
        "sslvpn",
        "user",
        "user_group",
    ]

    json = remove_invalid_fields(json)
    dictionary = {}

    for attribute in option_list:
        if attribute in json and json[attribute] is not None:
            dictionary[attribute] = json[attribute]

    return dictionary


def flatten_single_path(data, path, index):
    if (
        not data
        or index == len(path)
        or path[index] not in data
        or not data[path[index]]
    ):
        return

    if index == len(path) - 1:
        data[path[index]] = " ".join(str(elem) for elem in data[path[index]])
    elif isinstance(data[path[index]], list):
        for value in data[path[index]]:
            flatten_single_path(value, path, index + 1)
    else:
        flatten_single_path(data[path[index]], path, index + 1)


def flatten_multilists_attributes(data):
    multilist_attrs = [
        ["session"],
        ["ipsec_phase1"],
        ["ipsec_phase2"],
        ["ipsec_phase1_interface"],
        ["ipsec_phase2_interface"],
        ["dialup_tunnel"],
        ["firewall_policy"],
        ["firewall_address"],
        ["firewall_addrgrp"],
        ["custom_service"],
        ["service_group"],
        ["onetime_schedule"],
        ["recurring_schedule"],
        ["user"],
        ["user_group"],
        ["sslvpn"],
        ["proxy"],
        ["log_disk_quota"],
    ]

    for attr in multilist_attrs:
        flatten_single_path(data, attr, 0)

    return data


def underscore_to_hyphen(data):
    if isinstance(data, list):
        for i, elem in enumerate(data):
            data[i] = underscore_to_hyphen(elem)
    elif isinstance(data, dict):
        new_data = {}
        for k, v in data.items():
            new_data[k.replace("_", "-")] = underscore_to_hyphen(v)
        data = new_data

    return data


def system_vdom_property(data, fos, check_mode=False):
    vdom = data["vdom"]

    state = data["state"]

    system_vdom_property_data = data["system_vdom_property"]
    system_vdom_property_data = flatten_multilists_attributes(system_vdom_property_data)
    filtered_data = underscore_to_hyphen(
        filter_system_vdom_property_data(system_vdom_property_data)
    )

    # check_mode starts from here
    if check_mode:
        diff = {
            "before": "",
            "after": filtered_data,
        }
        mkey = fos.get_mkey("system", "vdom-property", filtered_data, vdom=vdom)
        current_data = fos.get("system", "vdom-property", vdom=vdom, mkey=mkey)
        is_existed = (
            current_data
            and current_data.get("http_status") == 200
            and isinstance(current_data.get("results"), list)
            and len(current_data["results"]) > 0
        )

        # 2. if it exists and the state is 'present' then compare current settings with desired
        if state == "present" or state is True:
            if mkey is None:
                return False, True, filtered_data, diff

            # if mkey exists then compare each other
            # record exits and they're matched or not
            if is_existed:
                is_same = is_same_comparison(
                    serialize(current_data["results"][0]), serialize(filtered_data)
                )

                current_values = find_current_values(
                    current_data["results"][0], filtered_data
                )

                return (
                    False,
                    not is_same,
                    filtered_data,
                    {"before": current_values, "after": filtered_data},
                )

            # record does not exist
            return False, True, filtered_data, diff

        if state == "absent":
            if mkey is None:
                return (
                    False,
                    False,
                    filtered_data,
                    {"before": current_data["results"][0], "after": ""},
                )

            if is_existed:
                return (
                    False,
                    True,
                    filtered_data,
                    {"before": current_data["results"][0], "after": ""},
                )
            return False, False, filtered_data, {}

        return True, False, {"reason: ": "Must provide state parameter"}, {}

    if state == "present" or state is True:
        return fos.set("system", "vdom-property", data=filtered_data, vdom=vdom)

    elif state == "absent":
        return fos.delete(
            "system", "vdom-property", mkey=filtered_data["name"], vdom=vdom
        )
    else:
        fos._module.fail_json(msg="state must be present or absent!")


def is_successful_status(resp):
    return (
        "status" in resp
        and resp["status"] == "success"
        or "http_status" in resp
        and resp["http_status"] == 200
        or "http_method" in resp
        and resp["http_method"] == "DELETE"
        and resp["http_status"] == 404
    )


def fortios_system(data, fos, check_mode):
    fos.do_member_operation("system", "vdom-property")
    if data["system_vdom_property"]:
        resp = system_vdom_property(data, fos, check_mode)
    else:
        fos._module.fail_json(msg="missing task body: %s" % ("system_vdom_property"))
    if isinstance(resp, tuple) and len(resp) == 4:
        return resp
    return (
        not is_successful_status(resp),
        is_successful_status(resp)
        and (resp["revision_changed"] if "revision_changed" in resp else True),
        resp,
        {},
    )


versioned_schema = {
    "type": "list",
    "elements": "dict",
    "children": {
        "name": {"v_range": [["v6.0.0", ""]], "type": "string", "required": True},
        "description": {"v_range": [["v6.0.0", ""]], "type": "string"},
        "snmp_index": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "session": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "ipsec_phase1": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "ipsec_phase2": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "ipsec_phase1_interface": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "ipsec_phase2_interface": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "dialup_tunnel": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "firewall_policy": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "firewall_address": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "firewall_addrgrp": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "custom_service": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "service_group": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "onetime_schedule": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "recurring_schedule": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "user": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "user_group": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "sslvpn": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "proxy": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
        "log_disk_quota": {
            "v_range": [["v6.0.0", ""]],
            "type": "list",
            "multiple_values": True,
            "elements": "str",
        },
    },
    "v_range": [["v6.0.0", ""]],
}


def main():
    module_spec = schema_to_module_spec(versioned_schema)
    mkeyname = "name"
    fields = {
        "access_token": {"required": False, "type": "str", "no_log": True},
        "enable_log": {"required": False, "type": "bool", "default": False},
        "vdom": {"required": False, "type": "str", "default": "root"},
        "member_path": {"required": False, "type": "str"},
        "member_state": {
            "type": "str",
            "required": False,
            "choices": ["present", "absent"],
        },
        "state": {"required": True, "type": "str", "choices": ["present", "absent"]},
        "system_vdom_property": {
            "required": False,
            "type": "dict",
            "default": None,
            "options": {},
        },
    }
    for attribute_name in module_spec["options"]:
        fields["system_vdom_property"]["options"][attribute_name] = module_spec[
            "options"
        ][attribute_name]
        if mkeyname and mkeyname == attribute_name:
            fields["system_vdom_property"]["options"][attribute_name]["required"] = True

    module = AnsibleModule(argument_spec=fields, supports_check_mode=True)
    check_legacy_fortiosapi(module)

    is_error = False
    has_changed = False
    result = None
    diff = None

    versions_check_result = None
    if module._socket_path:
        connection = Connection(module._socket_path)
        if "access_token" in module.params:
            connection.set_option("access_token", module.params["access_token"])

        if "enable_log" in module.params:
            connection.set_option("enable_log", module.params["enable_log"])
        else:
            connection.set_option("enable_log", False)
        fos = FortiOSHandler(connection, module, mkeyname)
        versions_check_result = check_schema_versioning(
            fos, versioned_schema, "system_vdom_property"
        )

        is_error, has_changed, result, diff = fortios_system(
            module.params, fos, module.check_mode
        )

    else:
        module.fail_json(**FAIL_SOCKET_MSG)

    if versions_check_result and versions_check_result["matched"] is False:
        module.warn(
            "Ansible has detected version mismatch between FortOS system and your playbook, see more details by specifying option -vvv"
        )

    if not is_error:
        if versions_check_result and versions_check_result["matched"] is False:
            module.exit_json(
                changed=has_changed,
                version_check_warning=versions_check_result,
                meta=result,
                diff=diff,
            )
        else:
            module.exit_json(changed=has_changed, meta=result, diff=diff)
    else:
        if versions_check_result and versions_check_result["matched"] is False:
            module.fail_json(
                msg="Error in repo",
                version_check_warning=versions_check_result,
                meta=result,
            )
        else:
            module.fail_json(msg="Error in repo", meta=result)


if __name__ == "__main__":
    main()
