/*!
 * \file   src/UnicodeSupport/UnicodeSupport.cxx
 * \brief
 * \author Thomas Helfer
 * \date   15/07/2019
 * \copyright Copyright (C) 2006-2025 CEA/DEN, EDF R&D. All rights
 * reserved.
 * This project is publicly released under either the GNU GPL Licence with
 * linking exception or the CECILL-A licence. A copy of thoses licences are
 * delivered with the sources of TFEL. CEA or EDF may also distribute this
 * project under specific licensing conditions.
 */

#include <cstring>
#include "TFEL/UnicodeSupport/UnicodeSupport.hxx"

namespace tfel::unicode {

  const std::vector<UnicodeCharacterDescription>&
  getSupportedUnicodeCharactersDescriptions() {
    static std::vector<UnicodeCharacterDescription> ucds{
        {"\u0391", "tfel_unicode_mangling_0391", "Alpha (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u0392", "tfel_unicode_mangling_0392", "Beta (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u0393", "tfel_unicode_mangling_0393", "Gamma (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u0394", "tfel_unicode_mangling_0394", "Delta (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u0395", "tfel_unicode_mangling_0395", "Epsilon (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u0396", "tfel_unicode_mangling_0396", "Zeta (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u0397", "tfel_unicode_mangling_0397", "Eta (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u0398", "tfel_unicode_mangling_0398", "Theta (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u0399", "tfel_unicode_mangling_0399", "Iota (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u039A", "tfel_unicode_mangling_039A", "Kappa (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u039B", "tfel_unicode_mangling_039B", "Lambda (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u039C", "tfel_unicode_mangling_039C", "Mu (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u039D", "tfel_unicode_mangling_039D", "Nu (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u039E", "tfel_unicode_mangling_039E", "Xi (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u039F", "tfel_unicode_mangling_039F", "Omicron (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u03A0", "tfel_unicode_mangling_03A0", "Pi (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u03A1", "tfel_unicode_mangling_03A1", "Rho (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u03A3", "tfel_unicode_mangling_03A3", "Sigma (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u03A4", "tfel_unicode_mangling_03A4", "Tau (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u03A5", "tfel_unicode_mangling_03A5", "Upsilon (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u03A6", "tfel_unicode_mangling_03A6", "Psi (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u03A7", "tfel_unicode_mangling_03A7", "Chi (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u03A8", "tfel_unicode_mangling_03A8", "Psi (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u03A9", "tfel_unicode_mangling_03A9", "Omega (capital letter)",
         UnicodeCharacterCategory::GREEK_LETTER_CAPITAL},
        {"\u03B1", "tfel_unicode_mangling_03B1", "Alpha (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03B2", "tfel_unicode_mangling_03B2", "Beta (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03B3", "tfel_unicode_mangling_03B3", "Gamma (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03B4", "tfel_unicode_mangling_03B4", "Delta (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03B5", "tfel_unicode_mangling_03B5", "Epsilon (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03B6", "tfel_unicode_mangling_03B6", "Zeta (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03B7", "tfel_unicode_mangling_03B7", "Eta (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03B8", "tfel_unicode_mangling_03B8", "Theta (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03B9", "tfel_unicode_mangling_03B9", "Iota (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03BA", "tfel_unicode_mangling_03BA", "Kappa (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03BB", "tfel_unicode_mangling_03BB", "Lambda (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03BC", "tfel_unicode_mangling_03BC", "Mu (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03BD", "tfel_unicode_mangling_03BD", "Nu (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03BE", "tfel_unicode_mangling_03BE", "Xi (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03BF", "tfel_unicode_mangling_03BF", "Omicron (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03C0", "tfel_unicode_mangling_03C0", "Pi (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03C1", "tfel_unicode_mangling_03C1", "Rho (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03C3", "tfel_unicode_mangling_03C3", "Sigma (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03C4", "tfel_unicode_mangling_03C4", "Tau (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03C5", "tfel_unicode_mangling_03C5", "Upsilon (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03C6", "tfel_unicode_mangling_03C6", "Psi (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03C7", "tfel_unicode_mangling_03C7", "Chi (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03C8", "tfel_unicode_mangling_03C8", "Psi (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u03C9", "tfel_unicode_mangling_03C9", "Omega (small letter)",
         UnicodeCharacterCategory::GREEK_LETTER_SMALL},
        {"\u2215", "tfel_unicode_mangling_2215", "Division sign",
         UnicodeCharacterCategory::MATHEMATICAL_SYMBOL},
        {"\u2202", "tfel_unicode_mangling_2202", "Partial differentiation sign",
         UnicodeCharacterCategory::MATHEMATICAL_SYMBOL},
        {"\u2207", "tfel_unicode_mangling_2207", "Nabla",
         UnicodeCharacterCategory::MATHEMATICAL_SYMBOL},
        {"\u207A", "tfel_unicode_mangling_207A", "Superscript + sign",
         UnicodeCharacterCategory::MATHEMATICAL_SYMBOL},
        {"\u207B", "tfel_unicode_mangling_207B", "Superscript - sign",
         UnicodeCharacterCategory::MATHEMATICAL_SYMBOL},
        {"\u208A", "tfel_unicode_mangling_208A", "Superscript + sign",
         UnicodeCharacterCategory::MATHEMATICAL_SYMBOL},
        {"\u208B", "tfel_unicode_mangling_208B", "Subscript - sign",
         UnicodeCharacterCategory::MATHEMATICAL_SYMBOL},
        {"\u2297", "tfel_unicode_mangling_2297", "Tensor product",
         UnicodeCharacterCategory::MATHEMATICAL_OPERATOR},
        {"\u22C5", "tfel_unicode_mangling_22C5", "Dot sign",
         UnicodeCharacterCategory::MATHEMATICAL_OPERATOR},
        {"\u1D2C", "tfel_unicode_mangling_1D2C", "A",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D30", "tfel_unicode_mangling_1D30", "D",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D31", "tfel_unicode_mangling_1D31", "E",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D33", "tfel_unicode_mangling_1D33", "G",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D34", "tfel_unicode_mangling_1D34", "H",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D35", "tfel_unicode_mangling_1D35", "I",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D36", "tfel_unicode_mangling_1D36", "J",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D37", "tfel_unicode_mangling_1D37", "K",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D38", "tfel_unicode_mangling_1D38", "L",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D39", "tfel_unicode_mangling_1D39", "M",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D3A", "tfel_unicode_mangling_1D3A", "N",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D3C", "tfel_unicode_mangling_1D3C", "O",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D3E", "tfel_unicode_mangling_1D3E", "P",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D3F", "tfel_unicode_mangling_1D3F", "R",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D40", "tfel_unicode_mangling_1D40", "T",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D41", "tfel_unicode_mangling_1D41", "U",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u2C7D", "tfel_unicode_mangling_2C7D", "V",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D42", "tfel_unicode_mangling_1D42", "W",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_CAPITAL},
        {"\u1D43", "tfel_unicode_mangling_1D43", "a",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1D47", "tfel_unicode_mangling_1D47", "b",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1D9C", "tfel_unicode_mangling_1D9C", "c",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1D48", "tfel_unicode_mangling_1D48", "d",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1D49", "tfel_unicode_mangling_1D49", "e",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1DA0", "tfel_unicode_mangling_1DA0", "f",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1D4D", "tfel_unicode_mangling_1D4D", "g",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u02B0", "tfel_unicode_mangling_02B0", "h",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u2071", "tfel_unicode_mangling_2071", "i",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u02B2", "tfel_unicode_mangling_02B2", "j",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1D4F", "tfel_unicode_mangling_1D4F", "k",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u02E1", "tfel_unicode_mangling_02E1", "l",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1D50", "tfel_unicode_mangling_1D50", "m",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u207F", "tfel_unicode_mangling_207F", "n",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1D52", "tfel_unicode_mangling_1D52", "o",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1D56", "tfel_unicode_mangling_1D56", "p",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u02B3", "tfel_unicode_mangling_02B3", "r",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u02E2", "tfel_unicode_mangling_02E2", "s",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1D57", "tfel_unicode_mangling_1D57", "t",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1D58", "tfel_unicode_mangling_1D58", "u",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1D5B", "tfel_unicode_mangling_1D5B", "v",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u02B7", "tfel_unicode_mangling_02B7", "w",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u02E3", "tfel_unicode_mangling_02E3", "x",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u02B8", "tfel_unicode_mangling_02B8", "y",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u1DBB", "tfel_unicode_mangling_1DBB", "z",
         UnicodeCharacterCategory::LATIN_LETTER_SUPERSCRIPT_SMALL},
        {"\u2090", "tfel_unicode_mangling_2090", "a",
         UnicodeCharacterCategory::LATIN_LETTER_SUBSCRIPT_SMALL},
        {"\u2091", "tfel_unicode_mangling_2091", "e",
         UnicodeCharacterCategory::LATIN_LETTER_SUBSCRIPT_SMALL},
        {"\u2092", "tfel_unicode_mangling_2092", "o",
         UnicodeCharacterCategory::LATIN_LETTER_SUBSCRIPT_SMALL},
        {"\u2093", "tfel_unicode_mangling_2093", "x",
         UnicodeCharacterCategory::LATIN_LETTER_SUBSCRIPT_SMALL},
        {"\u2095", "tfel_unicode_mangling_2095", "h",
         UnicodeCharacterCategory::LATIN_LETTER_SUBSCRIPT_SMALL},
        {"\u2096", "tfel_unicode_mangling_2096", "k",
         UnicodeCharacterCategory::LATIN_LETTER_SUBSCRIPT_SMALL},
        {"\u2097", "tfel_unicode_mangling_2097", "l",
         UnicodeCharacterCategory::LATIN_LETTER_SUBSCRIPT_SMALL},
        {"\u2098", "tfel_unicode_mangling_2098", "m",
         UnicodeCharacterCategory::LATIN_LETTER_SUBSCRIPT_SMALL},
        {"\u209A", "tfel_unicode_mangling_209A", "p",
         UnicodeCharacterCategory::LATIN_LETTER_SUBSCRIPT_SMALL},
        {"\u209B", "tfel_unicode_mangling_209B", "s",
         UnicodeCharacterCategory::LATIN_LETTER_SUBSCRIPT_SMALL},
        {"\u209C", "tfel_unicode_mangling_209C", "t",
         UnicodeCharacterCategory::LATIN_LETTER_SUBSCRIPT_SMALL},
        {"\u1D66", "tfel_unicode_mangling_1D66", "beta",
         UnicodeCharacterCategory::GREEK_LETTER_SUBSCRIPT_SMALL},
        {"\u1D67", "tfel_unicode_mangling_1D67", "gamma",
         UnicodeCharacterCategory::GREEK_LETTER_SUBSCRIPT_SMALL},
        {"\u1D68", "tfel_unicode_mangling_1D68", "rho",
         UnicodeCharacterCategory::GREEK_LETTER_SUBSCRIPT_SMALL},
        {"\u1D69", "tfel_unicode_mangling_1D69", "phi",
         UnicodeCharacterCategory::GREEK_LETTER_SUBSCRIPT_SMALL},
        {"\u1D6A", "tfel_unicode_mangling_1D6A", "chi",
         UnicodeCharacterCategory::GREEK_LETTER_SUBSCRIPT_SMALL},
        {"\u2070", "tfel_unicode_mangling_2070", "0",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUPERSCRIPT},
        {"\u00B9", "tfel_unicode_mangling_00B9", "1",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUPERSCRIPT},
        {"\u00B2", "tfel_unicode_mangling_00B2", "2",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUPERSCRIPT},
        {"\u00B3", "tfel_unicode_mangling_00B3", "3",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUPERSCRIPT},
        {"\u2074", "tfel_unicode_mangling_2074", "4",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUPERSCRIPT},
        {"\u2075", "tfel_unicode_mangling_2075", "5",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUPERSCRIPT},
        {"\u2076", "tfel_unicode_mangling_2076", "6",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUPERSCRIPT},
        {"\u2077", "tfel_unicode_mangling_2077", "7",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUPERSCRIPT},
        {"\u2078", "tfel_unicode_mangling_2078", "8",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUPERSCRIPT},
        {"\u2079", "tfel_unicode_mangling_2079", "9",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUPERSCRIPT},
        {"\u2080", "tfel_unicode_mangling_2080", "0",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUBSCRIPT},
        {"\u2081", "tfel_unicode_mangling_2081", "1",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUBSCRIPT},
        {"\u2082", "tfel_unicode_mangling_2082", "2",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUBSCRIPT},
        {"\u2083", "tfel_unicode_mangling_2083", "3",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUBSCRIPT},
        {"\u2084", "tfel_unicode_mangling_2084", "4",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUBSCRIPT},
        {"\u2085", "tfel_unicode_mangling_2085", "5",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUBSCRIPT},
        {"\u2086", "tfel_unicode_mangling_2086", "6",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUBSCRIPT},
        {"\u2087", "tfel_unicode_mangling_2087", "7",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUBSCRIPT},
        {"\u2088", "tfel_unicode_mangling_2088", "8",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUBSCRIPT},
        {"\u2089", "tfel_unicode_mangling_2089", "9",
         UnicodeCharacterCategory::ARABIC_NUMBER_SUBSCRIPT}};
    return ucds;
  }  // end of getSupportedUnicodeCharactersDescriptions

  std::string getMangledString(const std::string& s) {
    auto replace_all = [](std::string& v, const char* const s1,
                          const char* const s2) {
      auto pos = std::string::size_type{};
      auto rs = std::string::size_type{};
      auto p = std::string::size_type{};
      auto r = std::string{};
      if ((s1 == nullptr) || (s2 == nullptr)) {
        return;
      }
      const auto ss1 = ::strlen(s1);
      const auto ss2 = ::strlen(s2);
      if (ss1 == 0) {
        return;
      }
      p = v.find(s1, pos);
      if (p == std::string::npos) {
        return;
      }
      while (p != std::string::npos) {
        rs = r.size();
        r.resize(rs + p - pos + ss2);
        std::copy(&v[0] + pos, v.data() + p, r.data() + rs);
        std::copy(s2, s2 + ss2, r.data() + rs + p - pos);
        pos = p + ss1;
        p = v.find(s1, pos);
      }
      rs = r.size();
      r.resize(rs + v.size() - pos);
      std::copy(v.data() + pos, v.data() + v.size(), r.data() + rs);
      v.swap(r);
    };  // end of replace_all
    const auto& ucds = getSupportedUnicodeCharactersDescriptions();
    auto r = s;
    for (const auto& ucd : ucds) {
      replace_all(r, ucd.uc, ucd.m);
    }
    return r;
  }  // end of getMangledString

}  // end of namespace tfel::unicode
