module QMMM_mmforces_m
   !! This module contains the subroutines needed to calculate
   !! electron and nuclei forces over point charges (i.e. atoms).
   implicit none
   public :: QMMM_mmforces

   private
contains
  subroutine QMMM_mmforces( ucell, Rho, ntpl, ntml, ntm, dvol, lattice_volume )
    !! Computes forces exerted by electrons and nuclei over point charges.
    !! This subroutine is an interface for existing methods.
    !
    ! Dev Note: An MPI barrier is called at the end of the routine to
    ! properly compute timings in cases where grid distribution is unbalanced.
    use precision     , only : dp, grid_p
    use mesh          , only : meshLim, nsm
    use QMMM_core     , only : doing_QMMM, QMMM_coulomb_type, &
                               COULOMB_EWALD, COULOMB_REAL_CUTOFF
    use QMMM_structure, only : mm_charges
#ifdef MPI
    use mpi_siesta    , only : MPI_Barrier, MPI_Comm_World
#endif
    implicit none
    integer     , intent(in)    :: ntpl
      !! Total number of grid points for this node.
    integer     , intent(in)    :: ntml(3)
      !! Number of mesh points along each axis for this node.
    integer     , intent(in)    :: ntm(3)
      !! Number of mesh points along each axis.
    real(dp)    , intent(in)    :: ucell(3,3)
      !! Unit cell vectors.
    real(dp)    , intent(in)    :: dvol
      !! Volume of a point of the unit cell.
    real(dp)    , intent(in)    :: lattice_volume
      !! Total lattice volume.
    real(grid_p), intent(in)    :: Rho(ntpl)
      !! Electron density in the grid.

    integer :: box_offset(3)
#ifdef MPI
    integer :: MPIerr
#endif

    if ( mm_charges%n < 1 ) return

    box_offset(:) = ( meshLim(1,:) -1 ) * nsm

    mm_charges%f(:,:)      = 0.0_dp
    mm_charges%stress(:,:) = 0.0_dp
    select case ( QMMM_coulomb_type() )
    case ( COULOMB_EWALD )
      call mmforce_ewald( mm_charges%n, mm_charges%r, mm_charges%f,      &
                          mm_charges%pc, mm_charges%stress, ucell, Rho,  &
                          ntpl, ntml, ntm, box_offset, lattice_volume, dvol )
    case ( COULOMB_REAL_CUTOFF )
      call mmforce_direct( mm_charges%n, mm_charges%r, mm_charges%f,      &
                           mm_charges%pc, mm_charges%stress, ucell, Rho,  &
                           ntpl, ntml, ntm, box_offset, lattice_volume, dvol )
    case default
      call die( "ERROR: QMMM_mmforces - Wrong type of Coulomb interaction.")
    end select

#ifdef MPI
    call MPI_Barrier( MPI_COMM_WORLD, MPIerr)
#endif
  end subroutine QMMM_mmforces

  subroutine mmforce_direct( na_mm, rmm, fmm, pc, stress, ucell, Rho, &
                             ntpl, ntml, ntm, box0, lattice_volume, dvol )
    !! Computes forces exerted by electrons and nuclei over point charges using
    !! a cut-off scheme.
    use precision     , only : dp, grid_p
    use QMMM_core     , only : QMMM_density_rcut
    use QMMM_helper   , only : get_lattice_type, pbc_displ_vector
    use QMMM_neighbour, only : num_mmvecs, grid_veclist, grid_nr
#ifdef MPI
    use alloc         , only : re_alloc, de_alloc
    use mpi_siesta    , only : MPI_AllReduce, MPI_double_precision, &
                               MPI_sum, MPI_Comm_World
#endif

    implicit none
    integer     , intent(in) :: na_mm
      !! Number of MM atoms.
    integer     , intent(in) :: ntpl
      !! Total number of grid points.
    integer     , intent(in) :: ntml(3)
      !! Number of mesh points along each axis for this node.
    integer     , intent(in) :: ntm(3)
      !! Number of mesh points along each axis.
    integer     , intent(in) :: box0(3)
      !! The starting mesh cell index for this node.
    real(dp)    , intent(in) :: rmm(3,na_mm)
      !! Atomic positions.
    real(dp)    , intent(in) :: pc(na_mm)
      !! Atomic (classical) partial charges.
    real(dp)    , intent(in) :: ucell(3,3)
      !! Unit cell vectors.
    real(dp)    , intent(in) :: dvol
      !! Volume of a point of the unit cell.
    real(dp)    , intent(in) :: lattice_volume
      !! Total lattice volume.
    real(grid_p), intent(in) :: Rho(ntpl)
      !! Electron density in the grid.
    real(dp)    , intent(inout) :: fmm(3,na_mm)
      !! Atomic forces.
    real(dp)    , intent(inout) :: stress(3,3)
      !! Cell stress.

    character :: lattice_type
    integer   :: ix, iy, iz, imesh, iat, icrd, ivec, js, ix0, iy0, iz0
    real(dp)  :: drij(3), rcut_rho2, stress_fact, dist, xm, ym, zm, dE, &
                 kcell(3,3)
#ifdef MPI
    integer :: MPIerr
    real(dp), pointer :: flocal(:,:), stresslocal(:,:)
#endif

    if ( num_mmvecs == 0 ) return
    rcut_rho2   = ( QMMM_density_rcut() ) ** 2
    stress_fact = 1.0_dp / lattice_volume
    call reclat( ucell, kcell, 0 )

    lattice_type = get_lattice_type( ucell )
    ! For now, we put a range of orbital equals to rmax0 (bohrs).
    ! Loop over the mesh points
    do iz0 = 0, ntml(3) -1
    do iy0 = 0, ntml(2) -1
    do ix0 = 0, ntml(1) -1
      imesh = 1 + ix0 + ntml(1) * iy0 + ntml(1) * ntml(2) * iz0

      ! We use only grid points where there is density of charge.
      if ( abs( Rho(imesh) ) > 0.0_dp ) then
        ix = ix0 + box0(1)
        iy = iy0 + box0(2)
        iz = iz0 + box0(3)

        xm = ucell(1,1) * ix / ntm(1) + ucell(1,2) * iy / ntm(2) &
           + ucell(1,3) * iz / ntm(3)
        ym = ucell(2,1) * ix / ntm(1) + ucell(2,2) * iy / ntm(2) &
           + ucell(2,3) * iz / ntm(3)
        zm = ucell(3,1) * ix / ntm(1) + ucell(3,2) * iy / ntm(2) &
           + ucell(3,3) * iz / ntm(3)

        do iat = 1, num_mmvecs ! Loop over MM atoms
          js = grid_veclist(iat)

          if ( lattice_type == 'D' ) then
            drij(1) = xm - rmm(1,js) + grid_nr(1,iat) * ucell(1,1)
            drij(2) = ym - rmm(2,js) + grid_nr(2,iat) * ucell(2,2)
            drij(3) = zm - rmm(3,js) + grid_nr(3,iat) * ucell(3,3)
          else
            drij(1) = xm - rmm(1,js)
            drij(2) = ym - rmm(2,js)
            drij(3) = zm - rmm(3,js)

            do icrd = 1, 3
            do ivec = 1, 3
              drij(icrd) = drij(icrd) + grid_nr(ivec,iat) * ucell(icrd,ivec)
            enddo
            enddo
          endif
          call pbc_displ_vector( ucell, kcell, drij )
          dist = drij(1) * drij(1) + drij(2) * drij(2) + drij(3) * drij(3)

          ! Forces exerted on point charges by electrons.
          De = 0.0_dp
          if ( dist < rcut_rho2 ) cycle

          ! If our density point and point charge are too close,
          ! this might diverge. Thus, we avoid it.
          dist = ( 1.0_dp / sqrt( dist ) ) ** 3
          De   = -2.0_dp * dvol * Rho(imesh) * pc(js) * dist

          do ivec = 1, 3
            fmm(ivec,js) = fmm(ivec,js) - De * drij(ivec)

            do icrd = 1, 3
              stress(icrd,ivec) = stress(icrd,ivec) &
                               - stress_fact * rmm(icrd,js) * De * drij(ivec)
            enddo
          enddo
        enddo ! MM atoms
      endif   ! abs(Rho(imesh)) > 0
    enddo     ! grid X
    enddo     ! grid Y
    enddo     ! grid Z

#ifdef MPI
    nullify( flocal, stresslocal )

    call re_alloc( flocal, 1, 3, 1, na_mm, 'flocal', 'mmforce_direct' )
    call re_alloc( stresslocal, 1, 3, 1, 3,  'stresslocal', 'mmforce_direct' )

    flocal(:,:)      = fmm(:,:)
    stresslocal(:,:) = stress(:,:)

    fmm(:,:)    = 0.0_dp
    stress(:,:) = 0.0_dp
    call MPI_AllReduce( flocal(1,1), fmm(1,1), 3*na_mm, MPI_double_precision, &
                        MPI_sum, MPI_Comm_World, MPIerr )
    call MPI_AllReduce( stresslocal(1,1), stress(1,1), 9, MPI_double_precision,&
                        MPI_sum, MPI_Comm_World, MPIerr )

    call de_alloc( flocal     , 'flocal'     , 'mmforce_direct' )
    call de_alloc( stresslocal, 'stresslocal', 'mmforce_direct' )
#endif

  end subroutine mmforce_direct

  subroutine mmforce_ewald( na_mm, rmm, fmm, pc, stress, ucell, Rho, ntpl, &
                            ntml, ntm, box0, lattice_volume, dvol )
    !! Computes forces exerted by electrons and nuclei over point charges using
    !! the Ewald method.
    use alloc         , only : re_alloc, de_alloc
    use precision     , only : dp, grid_p
    use QMMM_core     , only : QMMM_density_rcut
    use QMMM_ewald_m  , only : QMMM_ewald
    use QMMM_helper   , only : get_lattice_type, pbc_displ_vector
    use sys           , only : die
#ifdef MPI
    use alloc         , only : re_alloc, de_alloc
    use mpi_siesta    , only : MPI_AllReduce, MPI_double_precision, &
                               MPI_sum, MPI_Comm_World
#endif

    implicit none
    integer     , intent(in) :: na_mm
      !! Number of MM atoms.
    integer     , intent(in) :: ntpl
      !! Total number of grid points.
    integer     , intent(in) :: ntml(3)
      !! Number of mesh points along each axis for this node.
    integer     , intent(in) :: ntm(3)
      !! Number of mesh points along each axis.
    integer     , intent(in) :: box0(3)
      !! The starting mesh cell index for this node.
    real(dp)    , intent(in) :: rmm(3,na_mm)
      !! Atomic positions.
    real(dp)    , intent(in) :: pc(na_mm)
      !! Atomic (classical) partial charges.
    real(dp)    , intent(in) :: ucell(3,3)
      !! Unit cell vectors.
    real(dp)    , intent(in) :: dvol
      !! Volume of a point of the unit cell.
    real(dp)    , intent(in) :: lattice_volume
      !! Total lattice volume.
    real(grid_p), intent(in) :: Rho(ntpl)
      !! Electron density in the grid.
    real(dp)    , intent(inout) :: fmm(3,na_mm)
      !! Atomic forces.
    real(dp)    , intent(inout) :: stress(3,3)
      !! Cell stress.

    character :: lattice_type
    integer   :: iat, icrd, ivec, imesh, ix, iy, iz, n1, n2, n3, &
                 ikew, ix0, iy0, iz0
    real(dp)  :: rcut_rho2, kcell(3,3), dist, dist2, krecip(3), kronij, twopi, &
                 kr, drij(3), kmod2,  stress_fact, dE, xm, ym, zm, &
                 const1, const2, const3, const4, const6, const7, rcut2

    real(dp), pointer   :: S_qm_real(:), S_qm_imag(:), S_mm_real(:), &
                           S_mm_imag(:), stress_atom(:,:,:)

#ifdef MPI
    integer :: MPIerr, buff_size
    real(dp), pointer :: frc_buf(:,:), str_buf(:,:), S_buffer(:)
#endif

    rcut_rho2      = ( QMMM_density_rcut() ) ** 2
    stress_fact    = 1.0_dp / lattice_volume
    lattice_type   = get_lattice_type( ucell )
    rcut2          = QMMM_ewald%rcut * QMMM_ewald%rcut

    twopi  = 2.0_dp * acos( -1.0_dp )
    const1 = QMMM_ewald%sqalpha / sqrt( 0.5_dp * twopi )
    const2 = 2.0_dp * twopi / lattice_volume

    ! We calculate the reciprocal lattice vectors
    call reclat( ucell, kcell, 0 )

    nullify( S_qm_real, S_qm_imag, S_mm_real, S_mm_imag )
    call re_alloc( S_qm_real, 1, QMMM_ewald%n_kpoints, 'S_qm_real', &
                   'mmforce_ewald' )
    call re_alloc( S_qm_imag, 1, QMMM_ewald%n_kpoints, 'S_qm_imag', &
                   'mmforce_ewald' )
    call re_alloc( S_mm_real, 1, QMMM_ewald%n_kpoints, 'S_mm_real', &
                   'mmforce_ewald' )
    call re_alloc( S_mm_imag, 1, QMMM_ewald%n_kpoints, 'S_mm_imag', &
                   'mmforce_ewald' )


    ! We use this to store each MM atom's contribution to stress, to then
    ! add it up properly in a reduction.
    nullify( stress_atom )
    call re_alloc( stress_atom, 1, na_mm, 1, 3, 1, 3, 'stress_atom', &
                   'mmforce_ewald' )
    stress_atom(:,:,:) = 0.0_dp

    ! Real part of the Ewald summation
    ! Loop over the mesh points
!$OMP PARALLEL DEFAULT(FIRSTPRIVATE) SHARED(Rho,fmm,stress_atom) &
!$OMP& PRIVATE(imesh,ix,iy,iz,iz0,iy0,ix0,iat)
!$OMP DO SCHEDULE(DYNAMIC,2)
    do iz0 = 0, ntml(3) -1
    do iy0 = 0, ntml(2) -1
    do ix0 = 0, ntml(1) -1
      imesh = 1 + ix0 + ntml(1) * iy0 + ntml(1) * ntml(2) * iz0

      ! We use only grid points where there is density of charge.
      if ( .not. (abs(Rho(imesh)) > 0.0_dp) ) cycle
      ix = ix0 + box0(1)
      iy = iy0 + box0(2)
      iz = iz0 + box0(3)

      xm = ucell(1,1) * ix / ntm(1) + ucell(1,2) * iy / ntm(2) &
         + ucell(1,3) * iz / ntm(3)
      ym = ucell(2,1) * ix / ntm(1) + ucell(2,2) * iy / ntm(2) &
         + ucell(2,3) * iz / ntm(3)
      zm = ucell(3,1) * ix / ntm(1) + ucell(3,2) * iy / ntm(2) &
         + ucell(3,3) * iz / ntm(3)

      do iat = 1, na_mm
        drij(1) = xm - rmm(1,iat)
        drij(2) = ym - rmm(2,iat)
        drij(3) = zm - rmm(3,iat)
        call pbc_displ_vector( ucell, kcell, drij )
        dist2 = drij(1) * drij(1) + drij(2) * drij(2) + drij(3) * drij(3)

        ! Forces exerted on point charges by electrons.
        De = 0.0_dp
        if ( dist2 > rcut2 ) cycle
        if ( dist2 < rcut_rho2 ) cycle

        ! If our density point and point charge are too close,
        ! this might diverge. Thus, we avoid it.
        dist = sqrt( dist2 )
        De   = -2.0_dp * dvol * Rho(imesh) * pc(iat) / dist2 *   &
               ( 2.0_dp * const1 * exp( -QMMM_ewald%alpha * dist2 ) + &
              erfc( QMMM_ewald%sqalpha * dist ) / dist )

        fmm(:,iat) = fmm(:,iat) - De * drij(:)

        do icrd = 1, 3
          stress_atom(iat,icrd,:) = stress_atom(iat,icrd,:) &
                                 - stress_fact * rmm(icrd,iat) * De * drij(:)
        enddo
      enddo   ! mm atom
    enddo     ! grid X
    enddo     ! grid Y
    enddo     ! grid Z
!$OMP END DO
!$OMP END PARALLEL
!$OMP PARALLEL DEFAULT(FIRSTPRIVATE) SHARED(stress_atom,stress)
!$OMP DO SCHEDULE(DYNAMIC,2)
    do ivec = 1, 3
    do icrd = 1, 3
      De = 0.0_dp
      do iat = 1, na_mm
        De = De + stress_atom(iat,icrd,ivec)
      enddo
      stress(icrd,ivec) = stress(icrd,ivec) + De
    enddo
    enddo
!$OMP END DO
!$OMP END PARALLEL
    call de_alloc( stress_atom, 'stress_atom', 'mmforce_ewald' )

#ifdef MPI
    nullify( frc_buf, str_buf )

    call re_alloc( frc_buf, 1, 3, 1, na_mm, 'frc_buf', 'mmforce_ewald' )
    call re_alloc( str_buf, 1, 3, 1,     3, 'str_buf', 'mmforce_ewald' )

    frc_buf(:,:) = fmm(:,:)
    str_buf(:,:) = stress(:,:)

    fmm(:,:)    = 0.0_dp
    stress(:,:) = 0.0_dp
    call MPI_AllReduce( frc_buf(1,1), fmm(1,1), 3*na_mm, MPI_double_precision,&
                        MPI_sum, MPI_Comm_World, MPIerr )
    call MPI_AllReduce( str_buf(1,1), stress(1,1),    9, MPI_double_precision,&
                        MPI_sum, MPI_Comm_World, MPIerr )

    call de_alloc( frc_buf, 'frc_buf', 'mmforce_ewald' )
    call de_alloc( str_buf, 'str_buf', 'mmforce_ewald' )
#endif

    ! Reciprocal space part of the ewald summation.
    ! Calculate structure factors for QM atoms.
    ! We first initialize and then loop over mesh points.
    S_qm_real(:) = 0.0_dp
    S_qm_imag(:) = 0.0_dp

!$OMP PARALLEL DEFAULT(FIRSTPRIVATE) SHARED(S_qm_real,S_qm_imag,Rho) &
!$OMP& PRIVATE(imesh)
!$OMP DO SCHEDULE(DYNAMIC,2)
    do iz0 = 0, ntml(3) -1
    do iy0 = 0, ntml(2) -1
    do ix0 = 0, ntml(1) -1
      imesh = 1 + ix0 + ntml(1) * iy0 + ntml(1) * ntml(2) * iz0

      ! We use only grid points where there is density of charge.
      if ( .not. (abs(Rho(imesh)) > 0.0_dp) ) cycle

      ix = ix0 + box0(1)
      iy = iy0 + box0(2)
      iz = iz0 + box0(3)
      xm = ucell(1,1) * ix / ntm(1) + ucell(1,2) * iy / ntm(2) &
         + ucell(1,3) * iz / ntm(3)
      ym = ucell(2,1) * ix / ntm(1) + ucell(2,2) * iy / ntm(2) &
         + ucell(2,3) * iz / ntm(3)
      zm = ucell(3,1) * ix / ntm(1) + ucell(3,2) * iy / ntm(2) &
         + ucell(3,3) * iz / ntm(3)

      do ikew = 1, QMMM_ewald%n_kpoints
        n1 = QMMM_ewald%point_x(ikew)
        n2 = QMMM_ewald%point_y(ikew)
        n3 = QMMM_ewald%point_z(ikew)

        if ( lattice_type == 'D' ) then
          krecip(1) = n1 * twopi * kcell(1,1)
          krecip(2) = n2 * twopi * kcell(2,2)
          krecip(3) = n3 * twopi * kcell(3,3)
        else
          krecip(1) = twopi * ( n1 * kcell(1,1) + n2 * kcell(1,2) &
                    + n3 * kcell(1,3) )
          krecip(2) = twopi * ( n1 * kcell(2,1) + n2 * kcell(2,2) &
                    + n3 * kcell(2,3) )
          krecip(3) = twopi * ( n1 * kcell(3,1) + n2 * kcell(3,2) &
                    + n3 * kcell(3,3) )
        endif

        kr = krecip(1) * xm + krecip(2) * ym + krecip(3) * zm
        S_qm_real(ikew) = S_qm_real(ikew) + &
                              2.0_dp * dvol * Rho(imesh) * cos(kr)
        S_qm_imag(ikew) = S_qm_imag(ikew) + &
                              2.0_dp * dvol * Rho(imesh) * sin(kr)
      enddo
    enddo
    enddo
    enddo
!$OMP END DO
!$OMP END PARALLEL

    ! Calculate structure factors for all MM atoms.
    S_mm_real(:) = 0.0_dp
    S_mm_imag(:) = 0.0_dp

    ! Reciprocal-space sum depends on the lattice type.
    if ( lattice_type == 'D' ) then
!$OMP  PARALLEL DEFAULT(FIRSTPRIVATE) SHARED(S_mm_real,S_mm_imag,Rho)
!$OMP DO SCHEDULE(DYNAMIC,2)
      do ikew = 1, QMMM_ewald%n_kpoints
        n1 = QMMM_ewald%point_x(ikew)
        n2 = QMMM_ewald%point_y(ikew)
        n3 = QMMM_ewald%point_z(ikew)

        krecip(1) = n1 * twopi * kcell(1,1)
        krecip(2) = n2 * twopi * kcell(2,2)
        krecip(3) = n3 * twopi * kcell(3,3)

        ! Loop over MM atoms
        do iat = 1, na_mm
          kr = krecip(1) * rmm(1,iat) + krecip(2) * rmm(2,iat) &
             + krecip(3) * rmm(3,iat)
          S_mm_real(ikew) = S_mm_real(ikew) + pc(iat) * cos(kr)
          S_mm_imag(ikew) = S_mm_imag(ikew) + pc(iat) * sin(kr)
        enddo
      enddo
!$OMP END DO
!$OMP END PARALLEL
    else ! Lattice
!$OMP PARALLEL DEFAULT(FIRSTPRIVATE) SHARED(S_mm_real,S_mm_imag,Rho)
!$OMP DO SCHEDULE(DYNAMIC,2)
      do ikew = 1, QMMM_ewald%n_kpoints
        n1 = QMMM_ewald%point_x(ikew)
        n2 = QMMM_ewald%point_y(ikew)
        n3 = QMMM_ewald%point_z(ikew)

        krecip(1) = twopi * ( n1 * kcell(1,1) + n2 * kcell(1,2) &
                            + n3 * kcell(1,3) )
        krecip(2) = twopi * ( n1 * kcell(2,1) + n2 * kcell(2,2) &
                            + n3 * kcell(2,3) )
        krecip(3) = twopi * ( n1 * kcell(3,1) + n2 * kcell(3,2) &
                            + n3 * kcell(3,3) )

        ! Loop over MM atoms
        do iat = 1, na_mm
          kr = krecip(1) * rmm(1,iat) + krecip(2) * rmm(2,iat) &
             + krecip(3) * rmm(3,iat)
          S_mm_real(ikew) = S_mm_real(ikew) + pc(iat) * cos(kr)
          S_mm_imag(ikew) = S_mm_imag(ikew) + pc(iat) * sin(kr)
        enddo
      enddo
!$OMP END DO
!$OMP END PARALLEL
    endif ! Lattice

#ifdef MPI
    nullify( S_buffer )
    buff_size = QMMM_ewald%n_kpoints

    call re_alloc( S_buffer, 1, buff_size, 'S_buffer', 'mmforce_ewald' )

    S_buffer  = S_qm_real
    S_qm_real = 0.0_dp
    call MPI_AllReduce( S_buffer(1), S_qm_real(1), buff_size, &
                        MPI_double_precision, MPI_sum, MPI_Comm_World, MPIerr )

    S_buffer  = S_qm_imag
    S_qm_imag = 0.0_dp
    call MPI_AllReduce( S_buffer(1), S_qm_imag(1), buff_size, &
                        MPI_double_precision, MPI_sum, MPI_Comm_World, MPIerr )

    call de_alloc( S_buffer, 'S_buffer', 'mmforce_ewald' )
#endif
    const4 = 1.0_dp / lattice_volume
    const6 = 1.0_dp / ( 4.0_dp * QMMM_ewald%alpha )

    ! Calculate the reciprocal part of the force on classical atoms
    ! due to QM grid points
    if ( lattice_type == 'D' ) then
      do ikew = 1, QMMM_ewald%n_kpoints
        n1 = QMMM_ewald%point_x(ikew)
        n2 = QMMM_ewald%point_y(ikew)
        n3 = QMMM_ewald%point_z(ikew)

        ! Loop over MM atoms
        krecip(1) = n1 * twopi * kcell(1,1)
        krecip(2) = n2 * twopi * kcell(2,2)
        krecip(3) = n3 * twopi * kcell(3,3)
        kmod2 = krecip(1) * krecip(1) + krecip(2) * krecip(2) &
              + krecip(3) * krecip(3)

        const3 = ( const2 / kmod2 ) * exp( -kmod2 / (4.0_dp * QMMM_ewald%alpha))

        ! Loop over MM atoms.
        do iat = 1, na_mm
          kr = krecip(1) * rmm(1,iat) + krecip(2) * rmm(2,iat) &
             + krecip(3) * rmm(3,iat)
          De = const3 * pc(iat) * ( cos(kr) * S_qm_imag(ikew) &
                                  - sin(kr) * S_qm_real(ikew) )
          do iCrd = 1, 3
            fmm(iCrd,iat) = fmm(iCrd,iat) + De * krecip(iCrd)
          enddo
        enddo

        De = const3 * const4 * ( S_mm_imag(ikew) * S_qm_imag(ikew) &
                               + S_mm_real(ikew) * S_qm_real(ikew) )
        const7 = 2.0_dp * ( 1.0_dp + kmod2 * const6 ) / kmod2
        do iVec = 1, 3
        do iCrd = 1, 3
          kronij = real( int( ( iCrd + iVec - abs( iCrd - iVec ) ) &
                            / ( iCrd + iVec + abs( iCrd - iVec ) ) ), kind=dp )
          stress(iCrd,iVec) = stress(iCrd,iVec) + De * &
                              ( kronij - const7 * krecip(iCrd) * krecip(iVec) )
        enddo
        enddo
      enddo
    else ! Lattice
      do ikew = 1, QMMM_ewald%n_kpoints
        n1 = QMMM_ewald%point_x(ikew)
        n2 = QMMM_ewald%point_y(ikew)
        n3 = QMMM_ewald%point_z(ikew)

        krecip(1) = twopi * ( n1 * kcell(1,1) + n2 * kcell(1,2) &
                            + n3 * kcell(1,3) )
        krecip(2) = twopi * ( n1 * kcell(2,1) + n2 * kcell(2,2) &
                            + n3 * kcell(2,3) )
        krecip(3) = twopi * ( n1 * kcell(3,1) + n2 * kcell(3,2) &
                            + n3 * kcell(3,3) )
        kmod2 = krecip(1) * krecip(1) + krecip(2) * krecip(2) &
              + krecip(3) * krecip(3)
        const3 = ( const2 / kmod2 ) * exp( -kmod2 / (4.0_dp * QMMM_ewald%alpha))

        ! Loop over MM atoms.
        do iat = 1, na_mm
          kr = krecip(1) * rmm(1,iat) + krecip(2) * rmm(2,iat) &
             + krecip(3) * rmm(3,iat)
          De = const3 * pc(iat) * ( cos(kr) * S_qm_imag(ikew) &
                                  - sin(kr) * S_qm_real(ikew) )
          do iCrd = 1, 3
            fmm(iCrd,iat) = fmm(iCrd,iat) + De * krecip(iCrd)
          enddo
        enddo

        De = const3 * const4 * ( S_mm_imag(ikew) * S_qm_imag(ikew) &
                               + S_mm_real(ikew) * S_qm_real(ikew) )
        const7 = 2.0_dp * ( 1.0_dp + kmod2 * const6 ) / kmod2
        do iVec = 1, 3
        do iCrd = 1, 3
          kronij = real( int( ( iCrd + iVec - abs( iCrd - iVec ) ) &
                            / ( iCrd + iVec + abs( iCrd - iVec ) ) ), kind=dp )
          stress(iCrd,iVec) = stress(iCrd,iVec) + De * &
                              ( kronij - const7 * krecip(iCrd) * krecip(iVec) )
        enddo
        enddo
      enddo
    endif ! Lattice

    call de_alloc( S_qm_real, 'S_qm_real', 'mmforce_ewald' )
    call de_alloc( S_qm_imag, 'S_qm_imag', 'mmforce_ewald' )
    call de_alloc( S_mm_real, 'S_mm_real', 'mmforce_ewald' )
    call de_alloc( S_mm_imag, 'S_mm_imag', 'mmforce_ewald' )
  end subroutine mmforce_ewald
end module QMMM_mmforces_m
