/*----------------------------------------------------------------------------*/
/*  CP2K: A general program to perform molecular dynamics simulations         */
/*  Copyright 2000-2025 CP2K developers group <https://cp2k.org>              */
/*                                                                            */
/*  SPDX-License-Identifier: MIT                                              */
/*----------------------------------------------------------------------------*/

/*
 *  libgrpp - a library for the evaluation of integrals over
 *            generalized relativistic pseudopotentials.
 *
 *  Copyright (C) 2021-2023 Alexander Oleynichenko
 */

#include <math.h>
#include <string.h>
#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

#include "grpp_factorial.h"
#include "grpp_specfunc.h"

extern const double boys_table[301][40];

/**
 * Boys function
 */
double libgrpp_boys_function(int n, double x) {
  if (x <= 30.0) {
    double step = 0.1;
    int idx = (int)(round(x / step) + 1e-6);
    double x0 = idx * step;
    double dx = x - x0;
    double dx2 = dx * dx;
    double dx3 = dx * dx2;
    double dx4 = dx * dx3;
    double dx5 = dx * dx4;
    double dx6 = dx * dx5;
    const double *boys_values = boys_table[idx];

    return +boys_values[n] - boys_values[n + 1] * dx +
           boys_values[n + 2] * dx2 / 2.0 - boys_values[n + 3] * dx3 / 6.0 +
           boys_values[n + 4] * dx4 / 24.0 - boys_values[n + 5] * dx5 / 120.0 +
           boys_values[n + 6] * dx6 / 720.0;
  } else {
    // asymptotic behaviour
    return libgrpp_double_factorial(2 * n - 1) / pow(2.0, n + 1) *
           sqrt(M_PI / pow(x, 2 * n + 1));
  }
}

void libgrpp_boys_values(double x, int nmax, double *b) {
  memset(b, 0, (nmax + 1) * sizeof(double));

  for (int n = 0; n <= nmax; n++) {
    b[n] = libgrpp_boys_function(n, x);
  }
}

const double boys_table[301][40] = {
    // x = 0.0000
    {1.0000000000000000e+00, 3.3333333333333331e-01, 2.0000000000000001e-01,
     1.4285714285714285e-01, 1.1111111111111110e-01, 9.0909090909090912e-02,
     7.6923076923076927e-02, 6.6666666666666666e-02, 5.8823529411764705e-02,
     5.2631578947368418e-02, 4.7619047619047616e-02, 4.3478260869565216e-02,
     4.0000000000000001e-02, 3.7037037037037035e-02, 3.4482758620689655e-02,
     3.2258064516129031e-02, 3.0303030303030304e-02, 2.8571428571428571e-02,
     2.7027027027027029e-02, 2.5641025641025640e-02, 2.4390243902439025e-02,
     2.3255813953488372e-02, 2.2222222222222223e-02, 2.1276595744680851e-02,
     2.0408163265306121e-02, 1.9607843137254902e-02, 1.8867924528301886e-02,
     1.8181818181818181e-02, 1.7543859649122806e-02, 1.6949152542372881e-02,
     1.6393442622950821e-02, 1.5873015873015872e-02, 1.5384615384615385e-02,
     1.4925373134328358e-02, 1.4492753623188406e-02, 1.4084507042253521e-02,
     1.3698630136986301e-02, 1.3333333333333334e-02, 1.2987012987012988e-02,
     1.2658227848101266e-02},
    // x = 0.1000
    {9.6764331263559222e-01, 3.1402947299816136e-01, 1.8625500479262172e-01,
     1.3218802963573911e-01, 1.0239394707106531e-01, 8.3540528018141616e-02,
     7.0541950817983701e-02, 6.1039712989141452e-02, 5.3791384005818628e-02,
     4.8080550314777717e-02, 4.3465189724101037e-02, 3.9657830850815971e-02,
     3.6463457664022141e-02, 3.3745117822999306e-02, 3.1403815925109733e-02,
     2.9366218961129207e-02, 2.7576848795227756e-02, 2.5992961032803887e-02,
     2.4581090560947576e-02, 2.3314663595510904e-02, 2.2172310944786246e-02,
     2.1136653501373413e-02, 2.0193412615459261e-02, 1.9330748298555210e-02,
     1.8538759980634943e-02, 1.7809105075818767e-02, 1.7134704153943029e-02,
     1.6509510615140950e-02, 1.5928328983990853e-02, 1.5386670257497745e-02,
     1.4880635781912741e-02, 1.4406823303446590e-02, 1.3962250405762433e-02,
     1.3544291692841780e-02, 1.3150626922038184e-02, 1.2779197923369966e-02,
     1.2428172616374387e-02, 1.2095914796859068e-02, 1.1780958642279890e-02,
     1.1481987097850902e-02},
    // x = 0.2000
    {9.3715002879797882e-01, 2.9604818929999216e-01, 1.7353453705498575e-01,
     1.2235483049236805e-01, 9.4382650921487468e-02, 7.6782763038513460e-02,
     6.4699100864166220e-02, 5.5893895390438972e-02, 4.9194194446507196e-02,
     4.3926381281596337e-02, 3.9676228180875482e-02, 3.6175096801002102e-02,
     3.3241183362660763e-02, 3.0747077471354994e-02, 2.8600846621503247e-02,
     2.6734497364043396e-02, 2.5096663018394909e-02, 2.3647816322619475e-02,
     2.2357045534259823e-02, 2.1199829224056735e-02, 2.0156466650558556e-02,
     1.9210948987320266e-02, 1.8350133441956861e-02, 1.7563129525190457e-02,
     1.6840836514920243e-02, 1.6175590382811619e-02, 1.5560891113512934e-02,
     1.4991189845529041e-02, 1.4461721065300080e-02, 1.3968369110257468e-02,
     1.3507561068017697e-02, 1.3076180177726731e-02, 1.2671495296964329e-02,
     1.2291103061777966e-02, 1.1932880152817191e-02, 1.1594943666038783e-02,
     1.1275618026940453e-02, 1.0973407221703866e-02, 1.0686971374507005e-02,
     1.0415106897668704e-02},
    // x = 0.3000
    {9.0839287703275051e-01, 2.7929109391838775e-01, 1.6175843512240917e-01,
     1.1328992488387975e-01, 8.7018755842399959e-02, 7.0584303166468840e-02,
     5.9348523582398928e-02, 5.1187643149114115e-02, 4.4994044258324983e-02,
     4.0134219516344737e-02, 3.6219916881388786e-02, 3.3000056379078609e-02,
     3.0305126728485825e-02, 2.8016579217384933e-02, 2.6049030312792845e-02,
     2.4339430648799478e-02, 2.2840215718440623e-02, 2.1514830044711327e-02,
     2.0334718138643654e-02, 1.9277250746827981e-02, 1.8324264074286934e-02,
     1.7461010606754600e-02, 1.6675392347874227e-02, 1.5957391621043347e-02,
     1.5298642512210485e-02, 1.4692104027664691e-02, 1.4131807881949360e-02,
     1.3612661769335560e-02, 1.3130294386236026e-02, 1.2680932222867225e-02,
     1.2261300779087065e-02, 1.1868544737642107e-02, 1.1500162982869939e-02,
     1.1153955341370607e-02, 1.0827978650178514e-02, 1.0520510301017463e-02,
     1.0230017817549276e-02, 9.9551333322904483e-03, 9.6946320667852803e-03,
     9.4474141012498153e-03},
    // x = 0.4000
    {8.8125403493992438e-01, 2.6366748613035634e-01, 1.5085301544428781e-01,
     1.0493128898224920e-01, 8.0248721050131103e-02, 6.4898054269425381e-02,
     5.4448188660049628e-02, 4.6883008181258533e-02, 4.1156345854049314e-02,
     3.6672291853997611e-02, 3.3066873987894811e-02, 3.0105384637690892e-02,
     2.7629750789064208e-02, 2.5529654613709991e-02, 2.3725785668160446e-02,
     2.2159672926269792e-02, 2.0787268348401888e-02, 1.9574761827026808e-02,
     1.8495772387876707e-02, 1.7529415394748549e-02, 1.6658942949443322e-02,
     1.5870768614418797e-02, 1.5153755480458645e-02, 1.4498688231251980e-02,
     1.3897876041504753e-02, 1.3344849997619781e-02, 1.2834129803704802e-02,
     1.2361041950898708e-02, 1.1921576579737175e-02, 1.1512273761709959e-02,
     1.1130132381561993e-02, 1.0772536549548246e-02, 1.0437195732359570e-02,
     1.0122095709665260e-02, 9.8254581399106911e-03, 9.5457070227437813e-03,
     9.2814407239480198e-03, 9.0314085157230783e-03, 8.7944908044875116e-03,
     8.5696823873858945e-03},
    // x = 0.5000
    {8.5562439189214867e-01, 2.4909373217951528e-01, 1.4075053682591263e-01,
     9.7222024416930147e-02, 7.4023511205877635e-02, 5.9680941140265384e-02,
     4.9959692830285299e-02, 4.2945347081074947e-02, 3.7649546503491026e-02,
     3.3511630846713547e-02, 3.0190326374923766e-02, 2.7466194160764287e-02,
     2.5191805984945748e-02, 2.3264489911013504e-02, 2.1610567884730893e-02,
     2.0175808944564600e-02, 1.8919417568866954e-02, 1.7810120059975461e-02,
     1.6823542386511507e-02, 1.5940408588290489e-02, 1.5145275230694080e-02,
     1.4425624745820856e-02, 1.3771204357660789e-02, 1.3173536382103366e-02,
     1.2625550246229664e-02, 1.2121302352627433e-02, 1.1655760271357930e-02,
     1.1224634669338504e-02, 1.0824247100982653e-02, 1.0451425043366051e-02,
     1.0103417845964559e-02, 9.7778288912010389e-03, 9.4725604330210249e-03,
     9.1857684337338149e-03, 8.9158253475207488e-03, 8.6612892662976167e-03,
     8.4208781944899620e-03, 8.1934484851396028e-03, 7.9779766728302840e-03,
     7.7735440953114847e-03},
    // x = 0.6000
    {8.3140287121402101e-01, 2.3549269593332897e-01, 1.3138870975496711e-01,
     9.0109927234007625e-02, 6.8298212120022447e-02, 5.4893560821812945e-02,
     4.5847944121596894e-02, 3.9343031238943359e-02, 3.4444860408437326e-02,
     3.0625825707839067e-02, 2.7565876962430041e-02, 2.5059816764169525e-02,
     2.2970124568227335e-02, 2.1201231759715989e-02, 1.9684684515253563e-02,
     1.8370179040275089e-02, 1.7219928462083728e-02, 1.6205002628945763e-02,
     1.5302879932566300e-02, 1.4495767842436270e-02, 1.3769424800822556e-02,
     1.3112317283081596e-02, 1.2515005898733226e-02, 1.1969691124140067e-02,
     1.1469872283797240e-02, 1.1010088176702319e-02, 1.0585717431486640e-02,
     1.0192823145639619e-02, 9.8280307634616409e-03, 9.4884311860676529e-03,
     9.1715032366371692e-03, 8.8750511173651247e-03, 8.5971535833075235e-03,
     8.3361223507993579e-03, 8.0904678412718339e-03, 7.8588707947809533e-03,
     7.6401586128469869e-03, 7.4332855365102377e-03, 7.2373159535339283e-03,
     7.0514102734073741e-03},
    // x = 0.7000
    {8.0849580691258371e-01, 2.2279321651512429e-01, 1.2271024696711658e-01,
     8.3547093602981057e-02, 6.3031679592469886e-02, 5.0499866100585378e-02,
     4.2080873796449772e-02, 3.6047182544598044e-02, 3.1516024555400805e-02,
     2.7990795464573968e-02, 2.5171292882496907e-02, 2.2865604815018325e-02,
     2.0945433538580145e-02, 1.9321810480782253e-02, 1.7931127992651161e-02,
     1.6726719996769501e-02, 1.5673582934602359e-02, 1.4744952178905745e-02,
     1.3920016050211041e-02, 1.3182350047427596e-02, 1.2518820041617092e-02,
     1.1918798510636663e-02, 1.1373594404260116e-02, 1.0876031714497704e-02,
     1.0420133421416665e-02, 1.0000881327150562e-02, 9.6140313523285920e-03,
     9.2559699157206462e-03, 8.9236011237309610e-03, 8.6142573294630074e-03,
     8.3256276049345282e-03, 8.0557000782787470e-03, 7.8027151001021696e-03,
     7.5651269394495851e-03, 7.3415722512177655e-03, 7.1308439590149746e-03,
     6.9318694990360089e-03, 6.7436925987309328e-03, 6.5654579381490552e-03,
     6.3963981757657390e-03},
    // x = 0.8000
    {7.8681637046172048e-01, 2.1092962896531176e-01, 1.1466245173669609e-01,
     7.7489559103911809e-02, 5.8186218506350453e-02, 4.6466876524957769e-02,
     3.8629173535821397e-02, 3.3031432405285138e-02, 2.8839076226285214e-02,
     2.5584582331016802e-02, 2.2986312607560823e-02, 2.0864750400972971e-02,
     1.9100184440723450e-02, 1.7609779313041349e-02, 1.6334423334308720e-02,
     1.5230820361083107e-02, 1.4266541922721965e-02, 1.3416824582877025e-02,
     1.2662435177174967e-02, 1.1988210898907630e-02, 1.1382038087609490e-02,
     1.0834123421729678e-02, 1.0336464385721063e-02, 9.8824582751420257e-03,
     9.4666092590326850e-03, 9.0843059846143232e-03, 8.7316506863157790e-03,
     8.4053264109479678e-03, 8.1024928030721931e-03, 7.8207035361768287e-03,
     7.5578403232599489e-03, 7.3120597510221317e-03, 7.0817501232286065e-03,
     6.8654961828989542e-03, 6.6620500856282025e-03, 6.4703073694548459e-03,
     6.2892869462938782e-03, 6.1181143513984347e-03, 5.9560076485353783e-03,
     5.8022655125031242e-03},
    // x = 0.9000
    {7.6628404352069612e-01, 1.9984132432227608e-01, 1.0719684068123844e-01,
     7.1896968703107397e-02, 5.3727289545084760e-02, 4.2764414536202208e-02,
     3.5466055643125112e-02, 3.0271702011126132e-02, 2.6392150236829413e-02,
     2.3387163491944762e-02, 2.0992470336862110e-02, 1.9040120740835994e-02,
     1.7418398499238326e-02, 1.6050168189088689e-02, 1.4880489647108489e-02,
     1.3869188903082776e-02, 1.2986220141647815e-02, 1.2208669407655120e-02,
     1.1518760848518687e-02, 1.0902495363662374e-02, 1.0348699690128209e-02,
     9.8483486414760214e-03, 9.3940732460386870e-03, 8.9797979617469957e-03,
     8.6004691452826968e-03, 8.2518490990318126e-03, 7.9303579500112249e-03,
     7.6329508944423244e-03, 7.3570219187391022e-03, 7.1003275708452669e-03,
     6.8609260773750428e-03, 6.6371283218203004e-03, 6.4274580744831831e-03,
     6.2306195004493553e-03, 6.0454704386131734e-03, 5.8710002909500552e-03,
     5.7063116204745176e-03, 5.5506047522477569e-03, 5.4031648210981801e-03,
     5.2633508244251939e-03},
    // x = 1.0000
    {7.4682413281242721e-01, 1.8947234582049235e-01, 1.0026879814501738e-01,
     6.6732274776822240e-02, 4.9623241133156748e-02, 3.9364864513484144e-02,
     3.2567034238441726e-02, 2.7746001964150033e-02, 2.4155294145404189e-02,
     2.1380279650214294e-02, 1.9172936091314634e-02, 1.7376108373082448e-02,
     1.5885525704727104e-02, 1.4629350723368210e-02, 1.3556514179749532e-02,
     1.2629735020647933e-02, 1.1821172234321877e-02, 1.1109621280589770e-02,
     1.0478651824601551e-02, 9.9153381694070623e-03, 9.4093737177159235e-03,
     8.9524406274551927e-03, 8.5377529045650383e-03, 8.1597197669928279e-03,
     7.8136939386101705e-03, 7.4957809102297367e-03, 7.2026926251350498e-03,
     6.9316339803580642e-03, 6.6802138741261880e-03, 6.4463748268721805e-03,
     6.2283368070088010e-03, 6.0245520280462518e-03, 5.8336682977321056e-03,
     5.6544990905717449e-03, 5.4859989484303817e-03, 5.3272431351280031e-03,
     5.1774107113216642e-03, 5.0357703775219930e-03, 4.9016685713517477e-03,
     4.7745194113223205e-03},
    // x = 1.1000
    {7.2836732370309676e-01, 1.7977101818409724e-01, 9.3837259479187363e-02,
     6.1961460771753282e-02, 4.5845064410997012e-02, 3.6242952727678905e-02,
     2.9909725593813018e-02, 2.5434249555222530e-02, 2.2110299831935819e-02,
     1.9547278838558765e-02, 1.7512370106607739e-02, 1.5858494791219424e-02,
     1.4488316590894036e-02, 1.3334923215578585e-02, 1.2350837782973717e-02,
     1.1501460003709057e-02, 1.0760989280409458e-02, 1.0109801161559982e-02,
     9.5327077075106903e-03, 9.0177733999160826e-03, 8.5554904084762535e-03,
     8.1381922952029097e-03, 7.7596295434752627e-03, 7.4146571628670530e-03,
     7.0990013439417394e-03, 6.8090827977591150e-03, 6.5418813580142132e-03,
     6.2948310348523518e-03, 6.0657378267273193e-03, 5.8527147388056904e-03,
     5.6541299506618747e-03, 5.4685651328613555e-03, 5.2947816691726263e-03,
     5.1316930900633965e-03, 4.9783424255283211e-03, 4.8338834833534525e-03,
     4.6975652818240579e-03, 4.5687190341268452e-03, 4.4467472097419183e-03,
     4.3311142963866718e-03},
    // x = 1.2000
    {7.1084926931384862e-01, 1.7068960725068616e-01, 8.7864420766606763e-02,
     5.7553288300346582e-02, 4.2366169245926691e-02, 3.3375546375474213e-02,
     2.7473665924172674e-02, 2.3318102125851052e-02, 2.0240549989818366e-02,
     1.7872974131129080e-02, 1.5996790241354326e-02, 1.4474326315099435e-02,
     1.3214705556285414e-02, 1.2155594581222494e-02, 1.1252850742002003e-02,
     1.0474358169107195e-02, 9.7962047208835779e-03, 9.2002266153981705e-03,
     8.6723831778068080e-03, 8.2016523611041364e-03, 7.7792625711907383e-03,
     7.3981472944245324e-03, 7.0525507283549149e-03, 6.7377378599044312e-03,
     6.4497781263772530e-03, 6.1853817834528471e-03, 5.9417746016204509e-03,
     5.7166008223678904e-03, 5.5078472158467092e-03, 5.3137830796065912e-03,
     5.1329124102446185e-03, 4.9639354636324875e-03, 4.8057176235999574e-03,
     4.6572640090800661e-03, 4.5176986233992182e-03, 4.3862471259773009e-03,
     4.2622225134106208e-03, 4.1450131528229022e-03, 4.0340727289631464e-03,
     3.9289117574842023e-03},
    // x = 1.3000
    {6.9421021226000634e-01, 1.6218400739461297e-01, 8.2315472749933186e-02,
     5.3479065659866674e-02, 3.9162179455790071e-02, 3.0741470026191587e-02,
     2.5240145097728959e-02, 2.1380805090947579e-02, 1.8530878203923528e-02,
     1.6343514012571902e-02, 1.4613451232635952e-02, 1.3211801096670016e-02,
     1.2053704688229935e-02, 1.1081086219898607e-02, 1.0252898039711493e-02,
     9.5393269683160583e-03, 8.9182088399174780e-03, 8.3727302627938146e-03,
     7.8899100629899414e-03, 7.4595689602363520e-03, 7.0736140058475123e-03,
     6.7255312329754392e-03, 6.4100192245888574e-03, 6.1227200278794868e-03,
     5.8600185678162695e-03, 5.6188910726869198e-03, 5.3967891050067055e-03,
     5.1915498197473778e-03, 5.0013257892666816e-03, 4.8245295977632904e-03,
     4.6597897053933732e-03, 4.5059149980697221e-03, 4.3618660939898496e-03,
     4.2267319520482476e-03, 4.0997106743150902e-03, 3.9800936514352908e-03,
     3.8672523914970516e-03, 3.7606275174140225e-03, 3.6597195277063626e-03,
     3.5640809997610941e-03},
    // x = 1.4000
    {6.7839463635527919e-01, 1.5421345443345458e-01, 7.7158356913841902e-02,
     4.9712435938429646e-02, 3.6210745581214711e-02, 2.8321337960473526e-02,
     2.3192054865572329e-02, 1.9607053325297788e-02, 1.6967441406378805e-02,
     1.4946264273868855e-02, 1.3350734736393548e-02, 1.2060166258092253e-02,
     1.0995307140898487e-02, 1.0102040921734452e-02, 9.3421931947227609e-03,
     8.6880852519126313e-03, 8.1191710241731701e-03, 7.6198856628957885e-03,
     7.1782265213389063e-03, 6.7847919099758075e-03, 6.4321144812318227e-03,
     6.1141892103207911e-03, 5.8261328936383494e-03, 5.5639343829002388e-03,
     5.3242685909661996e-03, 5.1043560770503375e-03, 4.9018557099848176e-03,
     4.7147816741390305e-03, 4.5414386200147219e-03, 4.3803704997246468e-03,
     4.2303198364818315e-03, 4.0901950299224296e-03, 3.9590439083937680e-03,
     3.8360321799953915e-03, 3.7204257564584379e-03, 3.6115761621524655e-03,
     3.5089084182924201e-03, 3.4119109263371257e-03, 3.3201269763122564e-03,
     3.2331475837285130e-03},
    // x = 1.5000
    {6.6335094584033472e-01, 1.4674026189730166e-01, 7.2363541847825050e-02,
     4.6229183030231824e-02, 3.3491373687730933e-02, 2.6097401013716204e-02,
     2.1313750334149489e-02, 1.7982864731837724e-02, 1.5537603609712087e-02,
     1.3669700405558465e-02, 1.2198049185726959e-02, 1.1009624250612011e-02,
     1.0030399205215527e-02, 9.2099399939863189e-03, 8.5127398964000324e-03,
     7.9130989490573452e-03, 7.3919690907826158e-03, 6.9349399491322036e-03,
     6.5309126903996990e-03, 6.1712031321194457e-03, 5.8489206680759622e-03,
     5.5585290808945805e-03, 5.2955301100123357e-03, 5.0562316007085797e-03,
     4.8375750282907354e-03, 4.6370054126061910e-03, 4.4523719648275617e-03,
     4.2818513291439696e-03, 4.1238876514964233e-03, 3.9771453289539193e-03,
     3.8404714199507936e-03, 3.7128654895223039e-03, 3.5934552304898973e-03,
     3.4814766111376000e-03, 3.3762575992626730e-03, 3.2772047335648418e-03,
     3.1837919782239854e-03, 3.0955514206415786e-03, 3.0120654665616044e-03,
     2.9329602589381687e-03},
    // x = 1.6000
    {6.4903116989786180e-01, 1.3972957871975200e-01, 6.7903818176437661e-02,
     4.3007054027354059e-02, 3.0985268811507184e-02, 2.4053406659034129e-02,
     1.9590923517100064e-02, 1.6495464914889089e-02, 1.4229829915212861e-02,
     1.2503309551238384e-02, 1.1145738587148124e-02, 1.0051247604829647e-02,
     9.1506802863833073e-03, 8.3970278640399486e-03, 7.7572607295071667e-03,
     7.2075134878290354e-03, 6.7301250400138186e-03, 6.3117526018128421e-03,
     5.9421322089987447e-03, 5.6132417932180795e-03, 5.3187224815151556e-03,
     5.0534699210832406e-03, 4.8133401912260699e-03, 4.5949345657867935e-03,
     4.3954395616637235e-03, 4.2125064146464703e-03, 4.0441591100972858e-03,
     3.8887233876565941e-03, 3.7447713520179170e-03, 3.6110778344883910e-03,
     3.4865857000499975e-03, 3.3703780338731606e-03, 3.2616556685467549e-03,
     3.1597188940258132e-03, 3.0639524703346506e-03, 2.9738132682609010e-03,
     2.8888200162086918e-03, 2.8085447464318439e-03, 2.7326056211657702e-03,
     2.6606608859715947e-03},
    // x = 1.7000
    {6.3539069040215268e-01, 1.3314916657335804e-01, 6.3754110490394200e-02,
     4.0025596588010671e-02, 2.8675191783335302e-02, 2.2174471175671520e-02,
     1.8010487905780043e-02, 1.5133181977178198e-02, 1.3033589883805432e-02,
     1.1437501168222810e-02, 1.0184999453970269e-02, 9.1769013178355004e-03,
     8.3485900757300554e-03, 7.6562434825050612e-03, 7.0691323455594306e-03,
     6.5650923436734541e-03, 6.1277466473949500e-03, 5.7447397974408143e-03,
     5.4065790757927687e-03, 5.1058534563522429e-03, 4.8366943367652900e-03,
     4.5943952219536623e-03, 4.3751383797861832e-03, 4.1757950110721731e-03,
     3.9937769022521654e-03, 3.8269247522423177e-03, 3.6734230328296868e-03,
     3.5317343197761634e-03, 3.4005480985162670e-03, 3.2787404596143540e-03,
     3.1653420777978105e-03, 3.0595125567443550e-03, 2.9605197123989228e-03,
     2.8677227215281424e-03, 2.7805583204845159e-03, 2.6985294296175532e-03,
     2.6211957206204472e-03, 2.5481657507529852e-03, 2.4790903687463141e-03,
     2.4136571590391426e-03},
    // x = 1.8000
    {6.2238799102131492e-01, 1.2696919522214670e-01, 5.9891304845792713e-02,
     3.7266010002049210e-02, 2.6545328275766090e-02, 2.0446962850085605e-02,
     1.6560473091487584e-02, 1.3885350546597786e-02, 1.1939269438161213e-02,
     1.0463525618653840e-02, 9.3078051480101291e-03, 8.3791721907294364e-03,
     7.6172422681084614e-03, 6.9811579114237802e-03, 6.4423264963487199e-03,
     5.9801611590354162e-03, 5.5794743634752873e-03, 5.2288238258604621e-03,
     4.9194293565366001e-03, 4.6444438806297998e-03, 4.3984508674931120e-03,
     4.1771103737864501e-03, 3.9769049586750350e-03, 3.7949541441084406e-03,
     3.6288768198638414e-03, 3.4766877643731581e-03, 3.3367188226229344e-03,
     3.2075581603971002e-03, 3.0880029445152350e-03, 2.9770221154942727e-03,
     2.8737268312712797e-03, 2.7773468016555171e-03, 2.6872111896411787e-03,
     2.6027330847469368e-03, 2.5233967934603623e-03, 2.4487473686608308e-03,
     2.3783819314811593e-03, 2.3119424379278021e-03, 2.2491096174994630e-03,
     2.1895978682981102e-03},
    // x = 1.9000
    {6.0998442594601110e-01, 1.2116205440088847e-01, 5.6294090521060609e-02,
     3.4711008784912623e-02, 2.4581169018882503e-02, 1.8858395249291435e-02,
     1.5229928557781835e-02, 1.2742224218033884e-02, 1.0938090538914097e-02,
     9.5733999839222282e-03, 8.5068369662861902e-03, 7.6513044919407237e-03,
     6.9503642347373221e-03, 6.3659175383680381e-03, 5.8713563982372994e-03,
     5.4475569279599156e-03, 5.0804330379269607e-03, 4.7593871128828180e-03,
     4.4762972969118655e-03, 4.2248370429221575e-03, 4.0000066977180365e-03,
     3.7978040483696938e-03, 3.6149881203318837e-03, 3.4489068927106570e-03,
     3.2973696670435091e-03, 3.1585511743417376e-03, 3.0309185970506655e-03,
     2.9131753739606824e-03, 2.8042174592641379e-03, 2.7030989356363732e-03,
     2.6090047315554836e-03, 2.5212287900653763e-03, 2.4391564609160281e-03,
     2.3622501939226156e-03, 2.2900378342575471e-03, 2.2221029845098032e-03,
     2.1580770204105092e-03, 2.0976324387719618e-03, 2.0404772855951549e-03,
     1.9863504653140208e-03},
    // x = 2.0000
    {5.9814400666130396e-01, 1.1570218085617275e-01, 5.2942814832976463e-02,
     3.2344697732067405e-02, 2.2769400221964796e-02, 1.7397329690267610e-02,
     1.4008835839082771e-02, 1.1694895667865808e-02, 1.0022037945343652e-02,
     8.7598404585573289e-03, 7.7754213689941428e-03, 6.9871413780660582e-03,
     6.3422421147266844e-03, 5.8051924078887079e-03, 5.3512279440955570e-03,
     4.9625817855397910e-03, 4.6261880287801760e-03, 4.3322304282833067e-03,
     4.0731954383262051e-03, 3.8432369953641837e-03, 3.6377398956474608e-03,
     3.4530131212333099e-03, 3.2860702441049440e-03, 3.1344694370275946e-03,
     2.9961950759209727e-03, 2.8695688708789951e-03, 2.7531822945537029e-03,
     2.6458445936834566e-03, 2.5465423539947039e-03, 2.4544077352709232e-03,
     2.3686932860931434e-03, 2.2887518037670810e-03, 2.2140201001775812e-03,
     2.1440058187322791e-03, 2.0782766546121908e-03, 2.0164514829075432e-03,
     1.9581930124556355e-03, 1.9032016681627508e-03, 1.8512104688980923e-03,
     1.8019807171352961e-03},
    // x = 2.1000
    {5.8683320530884686e-01, 1.1056589929901559e-01, 4.9819349915253441e-02,
     3.0152457457925125e-02, 2.1097803322022320e-02, 1.6053286106004521e-02,
     1.2888028312635205e-02, 1.0735223764589442e-02, 9.1837924323475862e-03,
     8.0162007373635758e-03, 7.1074727992681093e-03, 6.3810715551543690e-03,
     5.7876708370401420e-03, 5.2941292078623843e-03, 4.8773953236433912e-03,
     4.5209609839707123e-03, 4.2127052976452493e-03, 3.9435348974550758e-03,
     3.7064983709398426e-03, 3.4961932075695205e-03, 3.3083587719592213e-03,
     3.1395908088919342e-03, 2.9871372688979061e-03, 2.8487497255772075e-03,
     2.7225735355110312e-03, 2.6070654731095867e-03, 2.5009311608584928e-03,
     2.4030769696472220e-03, 2.3125726375276823e-03, 2.2286219252605193e-03,
     2.1505393660450497e-03, 2.0777316847060108e-03, 2.0096828294034696e-03,
     1.9459418233912641e-03, 1.8861128367219659e-03, 1.8298470192464073e-03,
     1.7768357413124622e-03, 1.7268049673403699e-03, 1.6795105470343626e-03,
     1.6347342544443223e-03},
    // x = 2.2000
    {5.7602077330637458e-01, 1.0573127612364563e-01, 4.6906970456500705e-02,
     2.8120839527311262e-02, 1.9555163256555649e-02, 1.4816661578787955e-02,
     1.1859117955530388e-02, 9.8557670589911623e-03, 8.4166698914849326e-03,
     7.3364158620249521e-03, 6.4974415945773080e-03, 5.8279807099521613e-03,
     5.2819086287649768e-03, 4.8283084901797925e-03, 4.4457206528454645e-03,
     4.1188046750420754e-03, 3.8363151281751023e-03, 3.5898274698737808e-03,
     3.3729097916475877e-03, 3.1805690746878547e-03, 3.0088717160209398e-03,
     2.8546777260282622e-03, 2.7154508765638622e-03, 2.5891207006909834e-03,
     2.4739805841230726e-03, 2.3686114226584777e-03, 2.2718236802833176e-03,
     2.1826128847006062e-03, 2.1001250673181098e-03, 2.0236296533627991e-03,
     1.9524979968344439e-03, 1.8861862374015322e-03, 1.8242214986271929e-03,
     1.7661906928258340e-03, 1.7117313765898476e-03, 1.6605242323558987e-03,
     1.6122868488485942e-03, 1.5667685462762999e-03, 1.5237460473607972e-03,
     1.4830198373744473e-03},
    // x = 2.3000
    {5.6567757400110730e-01, 1.0117798484310944e-01, 4.4190241479679253e-02,
     2.6237470364259292e-02, 1.8131184527611109e-02, 1.3678655875151390e-02,
     1.0914428457361220e-02, 9.0497230919330540e-03, 7.7145657948244021e-03,
     6.7149510411328283e-03, 5.9402665345043588e-03, 5.3232072829973668e-03,
     4.8206356056816986e-03, 4.4037057433128409e-03, 4.0524372492702077e-03,
     3.7525731534854148e-03, 3.4936791380965455e-03, 3.2679495292135727e-03,
     3.0694325651461382e-03, 2.8935133016528732e-03, 2.7365597916648157e-03,
     2.5956755946638251e-03, 2.4685232277696948e-03, 2.3531959840941558e-03,
     2.2481233760046440e-03, 2.1520003698750757e-03, 2.0637337262661148e-03,
     1.9824008194131589e-03, 1.9072176836782979e-03, 1.8375139667082070e-03,
     1.7727131115175766e-03, 1.7123165390800880e-03, 1.6558909215739065e-03,
     1.6030578651086899e-03, 1.5534854868428578e-03, 1.5068814933378945e-03,
     1.4629874574317301e-03, 1.4215740586335820e-03, 1.3824371031987226e-03,
     1.3453941790214313e-03},
    // x = 2.4000
    {5.5577642823416273e-01, 9.6887182280156303e-02, 4.1654915323136761e-02,
     2.4490963192973201e-02, 1.6816414387791623e-02, 1.2631203375148371e-02,
     1.0046934132754079e-02, 8.3108730075813538e-03, 7.0719045467308034e-03,
     6.1467550010439803e-03, 5.4313316105048361e-03, 4.8625021939977192e-03,
     4.3999160776114939e-03, 4.0166559689323346e-03, 3.6941162232834112e-03,
     3.4190452470431181e-03, 3.1817602851925138e-03, 2.9750283587375681e-03,
     2.7933415138340543e-03, 2.6324339005098329e-03, 2.4889518396813927e-03,
     2.3602233619843247e-03, 2.2440940158153476e-03, 2.1388077963079757e-03,
     2.0429194035546262e-03, 1.9552286426594376e-03, 1.8747307262954194e-03,
     1.8005781675509689e-03, 1.7320512345606488e-03, 1.6685348084464449e-03,
     1.6095000851933875e-03, 1.5544899807050151e-03, 1.5031073947919442e-03,
     1.4550057025132519e-03, 1.4098809956196773e-03, 1.3674657100720864e-03,
     1.3275233595218009e-03, 1.2898441574333626e-03, 1.2542413579351593e-03,
     1.2205481815824155e-03},
    // x = 2.5000
    {5.4629197178514788e-01, 9.2841394632249760e-02, 3.9287837054570125e-02,
     2.2870837329790401e-02, 1.5602172536926775e-02, 1.1666910841688435e-02,
     9.2502041269348235e-03, 7.6335310052507632e-03, 6.4835932909725787e-03,
     5.6272174645270135e-03, 4.9664266404228913e-03, 4.4419921649963990e-03,
     4.0161642342036894e-03, 3.6638214462387281e-03, 3.3676360849093580e-03,
     3.1152895676945964e-03, 2.8977955949267579e-03, 2.7084512017367880e-03,
     2.5421586873779472e-03, 2.3949745618170224e-03, 2.2638018573929472e-03,
     2.1461755058423716e-03, 2.0401096254646367e-03, 1.9439869044020426e-03,
     1.8564771765994211e-03, 1.7764766058947989e-03, 1.7030616553470261e-03,
     1.6354538218987774e-03, 1.5729923161068820e-03, 1.5151126788384161e-03,
     1.4613298855136090e-03, 1.4112248784861577e-03, 1.3644337441454382e-03,
     1.3206389491108442e-03, 1.2795621933053003e-03, 1.2409585428335176e-03,
     1.2046115834560963e-03, 1.1703293936794845e-03, 1.1379411804124597e-03,
     1.1072944535723563e-03},
    // x = 2.6000
    {5.3720052375048144e-01, 8.9024412603105341e-02, 3.7076857614419617e-02,
     2.1367444203416214e-02, 1.4480486771072982e-02, 1.0779000524100576e-02,
     8.5183514520716513e-03, 7.0124982043456626e-03, 5.9449797790098570e-03,
     5.1521303901603159e-03, 4.5417113843677246e-03, 4.0581463187285239e-03,
     3.6661129070042725e-03, 3.3421623963025217e-03, 3.0701550934296440e-03,
     2.8386383644473617e-03, 2.6392713622181447e-03, 2.4658416805509115e-03,
     2.3136308855670413e-03, 2.1789931830089662e-03, 2.0590684467337787e-03,
     1.9515823272598255e-03, 1.8547042034305281e-03, 1.7669444115462475e-03,
     1.6870786785269222e-03, 1.6140917372088604e-03, 1.5471346890994870e-03,
     1.4854923669113487e-03, 1.4285580703443912e-03, 1.3758138067875045e-03,
     1.3268146896402509e-03, 1.2811765103310580e-03, 1.2385657570233154e-03,
     1.1986915369580455e-03, 1.1612989926643063e-03, 1.1261638999046199e-03,
     1.0930882074795287e-03, 1.0618963330140344e-03, 1.0324320695612424e-03,
     1.0045559888235572e-03},
    // x = 2.7000
    {5.2847996498634731e-01, 8.5421194860480998e-02, 3.5010754044758015e-02,
     1.9971899534081568e-02, 1.3444034073855762e-02, 9.9612581342503840e-03,
     7.8459864327786056e-03, 6.4430205345133501e-03, 5.4518139403612166e-03,
     4.7176526382205390e-03, 4.1536828493408352e-03, 3.7077457585940185e-03,
     3.3467851310949388e-03, 3.0489102847451868e-03, 2.7990861015500536e-03,
     2.5866637417040952e-03, 2.4039006024217228e-03, 2.2450383592901523e-03,
     2.1057092287789388e-03, 1.9825423564946085e-03, 1.8728961413962185e-03,
     1.7746720476842803e-03, 1.6861824649396587e-03, 1.6060552189879546e-03,
     1.5331634356822086e-03, 1.4665732608665080e-03, 1.4055043637855005e-03,
     1.3492997297929778e-03, 1.2974022960859750e-03, 1.2493366920648087e-03,
     1.2046948318656179e-03, 1.1631244451949245e-03, 1.1243198717646062e-03,
     1.0880146157313073e-03, 1.0539752804162152e-03, 1.0219965942536646e-03,
     9.9189730597407403e-04, 9.6351677710351715e-04, 9.3671213759510515e-04,
     9.1135589908784235e-04},
    // x = 2.8000
    {5.2010962581807518e-01, 8.2017779141581634e-02, 3.3079156214201245e-02,
     1.8676021151033631e-02, 1.2486086684288819e-02, 9.2079852738181037e-03,
     7.2281741762109438e-03, 5.9207502974150334e-03, 5.0002128278585139e-03,
     4.3202777586386944e-03, 3.7991454980209481e-03, 3.3878558630753420e-03,
     3.0554682545562571e-03, 2.7815435247658333e-03, 2.5520736684749071e-03,
     2.3571560286704805e-03, 2.1896025470655267e-03, 2.0440752549900679e-03,
     1.9165305891848199e-03, 1.8038516383250571e-03, 1.7035984409748354e-03,
     1.6138345454911234e-03, 1.5330040769464502e-03, 1.4598430066736872e-03,
     1.3933140515080835e-03, 1.3325581961926394e-03, 1.2768581036796160e-03,
     1.2256101553217338e-03, 1.1783028424066900e-03, 1.1344998914218900e-03,
     1.0938269586916924e-03, 1.0559610455312765e-03, 1.0206220077233433e-03,
     9.8756569228556279e-04, 9.5657834962758548e-04, 9.2747205340816183e-04,
     9.0008092263586890e-04, 8.7425798700020024e-04, 8.4987257139216448e-04,
     8.2680810213905671e-04},
    // x = 2.9000
    {5.1207018228274814e-01, 7.8801200383851869e-02, 3.1272479499163543e-02,
     1.7472271972312145e-02, 1.1600462715478922e-02, 8.5139559280867257e-03,
     6.6603957159563519e-03, 5.4417110777629819e-03, 4.5866286396616489e-03,
     3.9568046237656567e-03, 3.4751841026103790e-03, 3.0958010514501198e-03,
     2.7896903667147545e-03, 2.5377653640451408e-03, 2.3269732366916368e-03,
     2.1481041047673463e-03, 1.9944839985138988e-03, 1.8611641197502210e-03,
     1.7444007129053807e-03, 1.6413114346709664e-03, 1.5496423958207611e-03,
     1.4676065814213754e-03, 1.3937694732261345e-03, 1.3269665928912243e-03,
     1.2662430706000182e-03, 1.2108086901714552e-03, 1.1600039900579772e-03,
     1.1132743821837792e-03, 1.0701501661553329e-03, 1.0302309335251171e-03,
     9.9317327958189905e-04, 9.5868103415316008e-04, 9.2649743021387307e-04,
     8.9639877715420955e-04, 8.6818931257311323e-04, 8.4169698467897352e-04,
     8.1676997513783361e-04, 7.9327381528537864e-04, 7.7108898103373012e-04,
     7.5010887641210097e-04},
    // x = 3.0000
    {5.0434356023143878e-01, 7.5759415310595796e-02, 2.9581862927320599e-02,
     1.6353707711456500e-02, 1.0781480935388586e-02, 7.8743766751055485e-03,
     6.1385125097161968e-03, 5.0022657097410885e-03, 4.2078195463754195e-03,
     3.6243106534196859e-03, 3.1791390078516894e-03, 2.8291417995036003e-03,
     2.5471988367864787e-03, 2.3154837586330480e-03, 2.1218321858713697e-03,
     1.9576775037343534e-03, 1.8168223746501580e-03, 1.6946783325985527e-03,
     1.5877788788476684e-03, 1.4934583582499430e-03, 1.4096346006472976e-03,
     1.3346583764458895e-03, 1.2672069698848646e-03, 1.2062075461592154e-03,
     1.1507810502698640e-03, 1.1002005158933365e-03, 1.0538596571159319e-03,
     1.0112489098801152e-03, 9.7193694592374695e-04, 9.3555625829813249e-04,
     9.0179181195436235e-04, 8.7037202689203526e-04, 8.4106155438886808e-04,
     8.1365544456875381e-04, 7.8797440304031584e-04, 7.6386090698636405e-04,
     7.4117600469461305e-04, 7.1979666247390666e-04, 6.9961355294642527e-04,
     6.8052920150189474e-04},
    // x = 3.1000
    {4.9691284667213048e-01, 7.2881232948156890e-02, 2.7999112330792410e-02,
     1.5313928912968429e-02, 1.0023919354390502e-02, 7.2848502896704359e-03,
     5.6587339988414644e-03, 4.5990870308679314e-03, 3.8608230757195632e-03,
     3.3201274022055887e-03, 2.9085835884432752e-03, 2.5856537038307908e-03,
     2.3259407733145754e-03, 2.1127930547268918e-03, 1.9348725942045419e-03,
     1.7842101352216453e-03, 1.6550502900504976e-03, 1.5431382545336436e-03,
     1.4452639540516476e-03, 1.3589619203795461e-03, 1.2823084679426439e-03,
     1.2137814180791201e-03, 1.1521610619103808e-03, 1.0964589342595914e-03,
     1.0458657284908117e-03, 9.9971262943427341e-04, 9.5744221090156561e-04,
     9.1858625552017749e-04, 8.8274865484717826e-04, 8.4959208592429831e-04,
     8.1882752838325420e-04, 7.9020594158393266e-04, 7.6351160100467433e-04,
     7.3855672124935713e-04, 7.1517708550787465e-04, 6.9322846878802896e-04,
     6.7258369199869666e-04, 6.5313018102384991e-04, 6.3476793277909317e-04,
     6.1740781136010599e-04},
    // x = 3.2000
    {4.8976220778451218e-01, 7.0156250594710298e-02, 2.6516648094650735e-02,
     1.4347036952326175e-02, 9.3229772949870275e-03, 6.7413424494557764e-03,
     5.2175879633824043e-03, 4.2291311790007774e-03, 3.5429318291633721e-03,
     3.0418182995954777e-03, 2.6613037053043586e-03, 2.3633084114102107e-03,
     2.1240452318857170e-03, 1.9279573154339007e-03, 1.7644755528670425e-03,
     1.6261854773090992e-03, 1.5077415340962155e-03, 1.4051979135638942e-03,
     1.3155817181829022e-03, 1.2366124366251581e-03, 1.1665126640648268e-03,
     1.1038773825455804e-03, 1.0475817923583752e-03, 9.9671510590014479e-04,
     9.5053218733446405e-04, 9.0841768765984987e-04, 8.6985907691961950e-04,
     8.3442610912090424e-04, 8.0175500363805180e-04, 7.7153612953160806e-04,
     7.4350432249982988e-04, 7.1743120220678206e-04, 6.9311902510320160e-04,
     6.7039572708466597e-04, 6.4911089629783147e-04, 6.2913247909136545e-04,
     6.1034406830007585e-04, 5.9264265742814328e-04, 5.7593677011631554e-04,
     5.6014489384226067e-04},
    // x = 3.3000
    {4.8287681308381275e-01, 6.7574794800389804e-02, 2.5127457121201430e-02,
     1.3447593667388966e-02, 8.6742406470428426e-03, 6.2401512760826613e-03,
     4.8118934296468565e-03, 3.8896132097225918e-03, 3.2516713249392259e-03,
     2.7871583519283125e-03, 2.4352789826360675e-03, 2.1602562475935504e-03,
     1.9398070141532918e-03, 1.7593951443321948e-03, 1.6091668932923175e-03,
     1.4822231067026467e-03, 1.3735983191730530e-03, 1.2796328987076863e-03,
     1.1975733414438660e-03, 1.1253100351792389e-03, 1.0612006016288255e-03,
     1.0039480705366667e-03, 9.5251509573282284e-04, 9.0606241011171598e-04,
     8.6390392030464148e-04, 8.2547343843757321e-04, 7.9029969076906267e-04,
     7.5798730447280808e-04, 7.2820217344919680e-04, 7.0066007353993085e-04,
     6.7511771782059965e-04, 6.5136566451765444e-04, 6.2922264596537312e-04,
     6.0853099795592724e-04, 5.8915294875856771e-04, 5.7096758531840488e-04,
     5.5386835702521929e-04, 5.3776100933355475e-04, 5.2256186345096107e-04,
     5.0819637643701441e-04},
    // x = 3.4000
    {4.7624276525303211e-01, 6.5127866954809696e-02, 2.3825048662368106e-02,
     1.2610584316399186e-02, 8.0736500374217861e-03, 5.7778794671279471e-03,
     4.4387359085413785e-03, 3.5779848309870281e-03, 2.9847797800704951e-03,
     2.5541156324812176e-03, 2.2286657436495698e-03, 1.9748103906345373e-03,
     1.7716719153335735e-03, 1.6056658710313745e-03, 1.4676041996354290e-03,
     1.3510664454561034e-03, 1.2514396836489795e-03, 1.1653293529544442e-03,
     1.0901849107470428e-03, 1.0240546672521296e-03, 9.6542089154513322e-04,
     9.1308626368006088e-04, 8.6609402616419579e-04, 8.2367076721513472e-04,
     7.8518472040960249e-04, 7.5011490290362889e-04, 7.1802795408212026e-04,
     6.8856053029797118e-04, 6.6140576559740546e-04, 6.3630274687136504e-04,
     6.1302825074771475e-04, 5.9139019636532788e-04, 5.7122241333657745e-04,
     5.5238042743400702e-04, 5.3473804084586672e-04, 5.1818453794688548e-04,
     5.0262238733858784e-04, 4.8796534049870723e-04, 4.7413684957007250e-04,
     4.6106874361317909e-04},
    // x = 3.5000
    {4.6984703520162341e-01, 6.2807093111329274e-02, 2.2603413701667050e-02,
     1.1831383583716673e-02, 7.5174716662426084e-03, 5.3514087962664219e-03,
     4.0954447623731674e-03, 3.2919140697903792e-03, 2.7401896606481769e-03,
     2.3408344012429269e-03, 2.0397814573281611e-03, 1.8054324545104118e-03,
     1.6182232902030018e-03, 1.4654569761080958e-03, 1.3385649903714326e-03,
     1.2315716140647499e-03, 1.1401909448126781e-03, 1.0612739652142665e-03,
     9.9245790859731455e-04, 9.3193702796887481e-04, 8.7830866692393335e-04,
     8.3046741736611529e-04, 7.8753078920348917e-04, 7.4878601311981045e-04,
     7.1365131347322637e-04, 6.8164727683857026e-04, 6.5237538520689379e-04,
     6.2550171337812045e-04, 6.0074440192549772e-04, 5.7786392677631798e-04,
     5.5665546535493826e-04, 5.3694285204750438e-04, 5.1857375095340171e-04,
     5.0141576995035180e-04, 4.8535330919356116e-04, 4.7028498743390619e-04,
     4.5612152649839748e-04, 4.4278400172355689e-04, 4.3020238670687694e-04,
     4.1831433630159403e-04},
    // x = 3.6000
    {4.6367740194420626e-01, 6.0604677707904682e-02, 2.1456987593947423e-02,
     1.1105724378117316e-02, 7.0022705832678590e-03, 4.9578767780719670e-03,
     3.7795725154859809e-03, 3.0292667019479387e-03, 2.5160108447120325e-03,
     2.1456197101127682e-03, 1.8670905617847280e-03, 1.6507193541925992e-03,
     1.4781698193246184e-03, 1.3375726438643026e-03, 1.2209359634782757e-03,
     1.1226972907746630e-03, 1.0388741064891796e-03, 9.6654487039588444e-04,
     9.0352055785608955e-04, 8.4813030463649474e-04, 7.9907769910149754e-04,
     7.5534211331513125e-04, 7.1610950350806666e-04, 6.8072294591259254e-04,
     6.4864666813879343e-04, 6.1943948493177578e-04, 5.9273490058719003e-04,
     5.6822601164286041e-04, 5.4565391570349057e-04, 5.2479871497306809e-04,
     5.0547246334982289e-04, 4.8751358570091666e-04, 4.7078242387008779e-04,
     4.5515765336990276e-04, 4.4053337895705254e-04, 4.2681676399227412e-04,
     4.1392608279982977e-04, 4.0178911070770163e-04, 3.9034178552569120e-04,
     3.7952708863694636e-04},
    // x = 3.7000
    {4.5772239692548344e-01, 5.8513360872316768e-02, 2.0380615695487964e-02,
     1.0429669190148708e-02, 6.5248861973920989e-03, 4.5946553116472328e-03,
     3.4888759402405684e-03, 2.7880892909172895e-03, 2.3105152558675649e-03,
     1.9669233620823157e-03, 1.7091915417871124e-03, 1.5093913388094537e-03,
     1.3503343678754123e-03, 1.2209233414251421e-03, 1.1137032092080074e-03,
     1.0234954860395950e-03, 9.4659913471458639e-04, 8.8030337503269384e-04,
     8.2257995348716979e-04, 7.7188267684944010e-04, 7.2701323334980550e-04,
     6.8702920229765272e-04, 6.5117962546749843e-04, 6.1885901022947745e-04,
     5.8957392033052612e-04, 5.6291832781847130e-04, 5.3855516870300059e-04,
     5.1620236093509009e-04, 4.9562207852576362e-04, 4.7661243319307674e-04,
     4.5900095784489101e-04, 4.4263945381066357e-04, 4.2739988104484111e-04,
     4.1317105372638927e-04, 3.9985596342275001e-04, 3.8736959538250955e-04,
     3.7563713538088045e-04, 3.6459248817098758e-04, 3.5417704628166839e-04,
     3.4433866126341503e-04},
    // x = 3.8000
    {4.5197125244745961e-01, 5.6526379025170270e-02, 1.9369521739387524e-02,
     9.7995837948384266e-03, 6.0824098299609762e-03, 4.2593311333530498e-03,
     3.2212987645681536e-03, 2.5665936951605756e-03, 2.1221228383214528e-03,
     1.8033311046446174e-03, 1.5648051489581779e-03, 1.3802810884152800e-03,
     1.2336438391297153e-03, 1.1145163318522842e-03, 1.0159433031376333e-03,
     9.3310314931920567e-04, 8.6255602272759758e-04, 8.0178643340067187e-04,
     7.4891490958659695e-04, 7.0251049980768704e-04, 6.6146547846502050e-04,
     6.2490957380265783e-04, 5.9214997596692580e-04, 5.6262856083502493e-04,
     5.3589085566848715e-04, 5.1156316731454539e-04, 4.8933548379946732e-04,
     4.6894852436925229e-04, 4.5018381370305995e-04, 4.3285599011951118e-04,
     4.1680678432704947e-04, 4.0190026155056176e-04, 3.8801902914727566e-04,
     3.7506118926409191e-04, 3.6293787164844160e-04, 3.5157122204959433e-04,
     3.4089275123096315e-04, 3.3084197153881702e-04, 3.2136526437437498e-04,
     3.1241493429753496e-04},
    // x = 3.9000
    {4.4641385388252741e-01, 5.4637428517528584e-02, 1.8419278731638643e-02,
     9.2121131041524151e-03, 5.6721641388798082e-03, 3.9496879236043426e-03,
     2.9749558607491521e-03, 2.3631429158890525e-03, 1.9493887554527449e-03,
     1.6535509483195196e-03, 1.4327636631110943e-03, 1.2623237794267385e-03,
     1.1271199334628983e-03, 1.0174470372779642e-03, 9.2681520008981689e-04,
     8.5073453292313001e-04, 7.8600757369392421e-04, 7.3029980590962116e-04,
     6.8186945654263539e-04, 6.3939210849654694e-04, 6.0184369045652227e-04,
     5.6842049524525977e-04, 5.3848331406945510e-04, 5.1151765222066481e-04,
     4.8710489853421570e-04, 4.6490110030414443e-04, 4.4462111150084981e-04,
     4.2602659791546305e-04, 4.0891685122386125e-04, 3.9312167614811121e-04,
     3.7849582653002608e-04, 3.6491461186244551e-04, 3.5227039763194746e-04,
     3.4046979490668251e-04, 3.2943138627475294e-04, 3.1908387271201161e-04,
     3.0936455342926494e-04, 3.0021807109387866e-04, 2.9159537003029494e-04,
     2.8345282647800925e-04},
    // x = 4.0000
    {4.4104069538121077e-01, 5.2840632061559575e-02, 1.7525782161993075e-02,
     8.6641589901538946e-03, 5.2916842552928854e-03, 3.6636899261127259e-03,
     2.7481187873132245e-03, 2.1762381682922170e-03, 1.7909917044561373e-03,
     1.5164025108775163e-03, 1.3120011022423313e-03, 1.1545480322943463e-03,
     1.0298707317544721e-03, 9.2889117564096081e-04, 8.4555285669646340e-04,
     7.7567424443292048e-04, 7.1628283608578876e-04, 6.6521183776210716e-04,
     6.2084692911747803e-04, 5.8196218607656301e-04, 5.4761079603146820e-04,
     5.1705046856950678e-04, 4.8969140746932336e-04, 4.6505930592318471e-04,
     4.4276856120692616e-04, 4.2250257630068025e-04, 4.0399906282504578e-04,
     3.8703893012416001e-04, 3.7143778351185014e-04, 3.5703934643011758e-04,
     3.4371031883036445e-04, 3.3133631998974640e-04, 3.1981865882743806e-04,
     3.0907174188115504e-04, 2.9902097716288174e-04, 2.8960106693809489e-04,
     2.8075460798380797e-04, 2.7243093676050629e-04, 2.6458517103795394e-04,
     2.5717741014854429e-04},
    // x = 4.1000
    {4.3584283880651753e-01, 5.1130507732287354e-02, 1.6685225340865958e-02,
     8.1528599149473802e-03, 4.9387004881549287e-03, 3.3994669501991588e-03,
     2.5392025671255457e-03, 2.0045070696183983e-03, 1.6457232490871652e-03,
     1.3908072966732346e-03, 1.2015442969549031e-03, 1.0560676627184976e-03,
     9.4108302936149382e-04, 8.4809760149708534e-04, 7.7145851690976124e-04,
     7.0727092544169547e-04, 6.5277113255260682e-04, 6.0594780152130659e-04,
     5.6530459164448035e-04, 5.2970664501030104e-04, 4.9827850654152379e-04,
     4.7033455688307306e-04, 4.4533055417206075e-04, 4.2282921170506521e-04,
     4.0247531077766295e-04, 3.8397741784688098e-04, 3.6709425712555294e-04,
     3.5162441779183756e-04, 3.3739848497438800e-04, 3.2427295631445570e-04,
     3.1212549034045987e-04, 3.0085115963496560e-04, 2.9035947015137439e-04,
     2.8057197049731881e-04, 2.7142031970231413e-04, 2.6284471435347615e-04,
     2.5479259967505554e-04, 2.4721760664854150e-04, 2.4007867035112464e-04,
     2.3333929454578519e-04},
    // x = 4.2000
    {4.3081187564812423e-01, 4.9501940336624597e-02, 1.5894076689213815e-02,
     7.6755722173323148e-03, 4.6111224643867231e-03, 3.1553006379765238e-03,
     2.3467535949123828e-03, 1.8466928468313447e-03, 1.5124780811895824e-03,
     1.2757798285410924e-03, 1.1005047525955989e-03, 9.6607416476545522e-04,
     8.6001535346758690e-04, 7.7438178764428452e-04, 7.0389660070451225e-04,
     6.4493149999442754e-04, 5.9491662849398387e-04, 5.5198475235996799e-04,
     5.1474875144301728e-04, 4.8215797415641497e-04, 4.5340287757409531e-04,
     4.2785013810240296e-04, 4.0499751403875702e-04, 3.8444182276980890e-04,
     3.6585581544085590e-04, 3.4897120668147229e-04, 3.3356603812824463e-04,
     3.1945514289515346e-04, 3.0648286175665265e-04, 2.9451741662514416e-04,
     2.8344651909594279e-04, 2.7317391004460872e-04, 2.6361660861099514e-04,
     2.5470270705201409e-04, 2.4636958952465373e-04, 2.3856248294326933e-04,
     2.3123327005884685e-04, 2.2433951116884428e-04, 2.1784363299827015e-04,
     2.1171225242727846e-04},
    // x = 4.3000
    {4.2593989168916341e-01, 4.7950154962437500e-02, 1.5149058822687392e-02,
     7.2298529187483787e-03, 4.3070245836090322e-03, 2.9296118884046957e-03,
     2.1694385767733367e-03, 1.7016444750991234e-03, 1.3902451295681290e-03,
     1.1704195570299155e-03, 1.0080712292287724e-03, 8.8382986065154703e-04,
     7.8599160264937603e-04, 7.0711989000389781e-04, 6.4228814161678045e-04,
     5.8811594124253276e-04, 5.4221339143227954e-04, 5.0284684942607509e-04,
     4.6873031601300997e-04, 4.3889100933492911e-04, 4.1258027347223928e-04,
     3.8921304653033873e-04, 3.6832581262832706e-04, 3.4954680884579108e-04,
     3.3257453529665946e-04, 3.1716200201576604e-04, 3.0310501053522159e-04,
     2.9023331932159839e-04, 2.7840390121942443e-04, 2.6749573922162442e-04,
     2.5740576765987045e-04, 2.4804567616873473e-04, 2.3933937051500364e-04,
     2.3122093852025896e-04, 2.2363300798327446e-04, 2.1652541147035835e-04,
     2.0985409327841235e-04, 2.0358020896782609e-04, 1.9766937911463683e-04,
     1.9209106739839196e-04},
    // x = 4.4000
    {4.2121943421632779e-01, 4.6470692535597651e-02, 1.4447129284514155e-02,
     6.8134439226707182e-03, 4.0246326767757502e-03, 2.7209493395355996e-03,
     2.0060344127071764e-03, 1.5683076661505513e-03, 1.2780994419533921e-03,
     1.0739034784249110e-03, 9.2350297579600420e-04, 8.0866165780087021e-04,
     7.1839525299449865e-04, 6.4574334338568930e-04, 5.8610572367557938e-04,
     5.3633250949130151e-04, 4.9420089672294338e-04, 4.5810110099870938e-04,
     4.2684075362346534e-04, 3.9951908874997176e-04, 3.7544369979322721e-04,
     3.5407406686975345e-04, 3.3498238321942359e-04, 3.1782583429609720e-04,
     3.0232662600546297e-04, 2.8825736036356363e-04, 2.7543016766739484e-04,
     2.6368852082994676e-04, 2.5290099347485454e-04, 2.4295644602250350e-04,
     2.3376027412037523e-04, 2.2523145662208731e-04, 2.1730021183212793e-04,
     2.0990612113862183e-04, 2.0299661513853209e-04, 1.9652574335117688e-04,
     1.9045316759829528e-04, 1.8474333313719046e-04, 1.7936478207052832e-04,
     1.7428958140481340e-04},
    // x = 4.5000
    {4.1664348158051584e-01, 4.5059387226919274e-02, 1.3785462793612835e-02,
     6.4242574922024308e-03, 3.7623117674638585e-03, 2.5279788187702666e-03,
     1.8554189409145136e-03, 1.4457166326273777e-03, 1.1751947723520394e-03,
     9.8547939908248260e-04, 8.4612356048053823e-04, 7.3995535909433374e-04,
     6.5666408010303570e-04, 5.8973394048151574e-04, 5.3486887275095438e-04,
     4.8913341905949326e-04, 4.5045993917799607e-04, 4.1735349495905771e-04,
     3.8870842059164732e-04, 3.6369055818317656e-04, 3.4165947010016579e-04,
     3.2211574842938768e-04, 3.0466451602459085e-04, 2.8898963142936791e-04,
     2.7483512654865932e-04, 2.6199162918245775e-04, 2.5028628334031693e-04,
     2.3957516431049729e-04, 2.2973749987056263e-04, 2.2067121715327781e-04,
     2.1228947486678625e-04, 2.0451793651461643e-04, 1.9729260690868887e-04,
     1.9055810120249388e-04, 1.8426624914718353e-04, 1.7837496143482495e-04,
     1.7284730262557161e-04, 1.6765072815828633e-04, 1.6275645262544628e-04,
     1.5813892376856622e-04},
    // x = 4.6000
    {4.1220541493039259e-01, 4.3712345563669465e-02, 1.3161434885475526e-02,
     6.0603629002982679e-03, 3.5185548432015495e-03, 2.3494736787152580e-03,
     1.7165624696993756e-03, 1.3329865610280771e-03, 1.0807568120421284e-03,
     9.0445978913941086e-04, 7.7531524445817836e-04, 6.7715047706393020e-04,
     6.0028535085182447e-04, 5.3861935072414152e-04, 4.8813986140415098e-04,
     4.4611089522683656e-04, 4.1060891384764168e-04, 3.8024547960201834e-04,
     3.5399522189535107e-04, 3.3108559407547496e-04, 3.1092417655542476e-04,
     2.9304951023248742e-04, 2.7709708645253645e-04, 2.6277534192724084e-04,
     2.4984840499421476e-04, 2.3812348913946496e-04, 2.2744154363902458e-04,
     2.1767022480813201e-04, 2.0869854563192956e-04, 2.0043275612894202e-04,
     1.9279313771457013e-04, 1.8571148434295463e-04, 1.7912910532308702e-04,
     1.7299522840946374e-04, 1.6726571291308739e-04, 1.6190200504017212e-04,
     1.5687028404549671e-04, 1.5214075985737902e-04, 1.4768709181192698e-04,
     1.4348590487879223e-04},
    // x = 4.7000
    {4.0789899195518325e-01, 4.2425927112073131e-02, 1.2572606833459956e-02,
     5.7199741559153146e-03, 3.2919725520969573e-03, 2.1843059433166805e-03,
     1.5885200292327301e-03, 1.2293067317372002e-03, 9.9407700791087259e-04,
     8.3021617370095579e-04, 7.1051385091727053e-04, 6.1973550718796323e-04,
     5.4879144293908078e-04, 4.9196903955119346e-04, 4.4551989001982467e-04,
     4.0689358605097269e-04, 3.7430043254088542e-04, 3.4645076299503872e-04,
     3.2239357480113293e-04, 3.0141331552617837e-04, 2.8296193657713971e-04,
     2.6661301042200773e-04, 2.5203003685643645e-04, 2.3894410179190116e-04,
     2.2713783856633419e-04, 2.1643372213347507e-04, 2.0668539650120014e-04,
     1.9777116094338342e-04, 1.8958901597769431e-04, 1.8205285202474371e-04,
     1.7508948593235210e-04, 1.6863633406145049e-04, 1.6263956852929456e-04,
     1.5705264390514117e-04, 1.5183511063282426e-04, 1.4695165233714254e-04,
     1.4237129938736724e-04, 1.3806678229597412e-04, 1.3401399686193338e-04,
     1.3019155922054781e-04},
    // x = 4.8000
    {4.0371832248582623e-01, 4.1196726608000653e-02, 1.2016711747393952e-02,
     5.4014387174947647e-03, 3.0812837472336779e-03, 2.0314381954253209e-03,
     1.4704242813185918e-03, 1.1339342300126754e-03, 9.1450691678855299e-04,
     7.6217401420680794e-04, 6.5120408551138829e-04, 5.6724361944990721e-04,
     5.0175585399248003e-04, 4.4939055216583571e-04, 4.0664561036015865e-04,
     3.7114329702339920e-04, 3.4121824569847335e-04, 3.1567240198224748e-04,
     2.9362364795403468e-04, 2.7440915888325321e-04, 2.5752189035695828e-04,
     2.4256775579325449e-04, 2.2923608855102769e-04, 2.1727884747669120e-04,
     2.0649570649837169e-04, 1.9672318431253045e-04, 1.8782659905405261e-04,
     1.7969403133799640e-04, 1.7223173693435483e-04, 1.6536062044146047e-04,
     1.5901349552355820e-04, 1.5313293519968399e-04, 1.4766956964165008e-04,
     1.4258072684241409e-04, 1.3782933848141427e-04, 1.3338305272891343e-04,
     1.2921350986798950e-04, 1.2529574701492465e-04, 1.2160770594782994e-04,
     1.1812982385029883e-04},
    // x = 4.9000
    {3.9965784581539265e-01, 4.0021557422904919e-02, 1.1491641754876577e-02,
     5.1032271125978105e-03, 2.8853068078837059e-03, 1.8899161428601037e-03,
     1.3614790306670204e-03, 1.0461881967088731e-03, 8.4145304894987089e-04,
     6.9980803685953724e-04, 5.9691526830682354e-04, 5.1924873097131986e-04,
     4.5878956545061344e-04, 4.1052612911643085e-04, 3.7118596073666471e-04,
     3.3855201943254736e-04, 3.1107444198822499e-04, 2.8764015455990348e-04,
     2.6743085088493557e-04, 2.4983249100186391e-04, 2.3437592634166802e-04,
     2.2069692949836856e-04, 2.0850866301076070e-04, 1.9758232291427563e-04,
     1.8773327612720185e-04, 1.7881096523557588e-04, 1.7069144449897237e-04,
     1.6327178444094004e-04, 1.5646582380892112e-04, 1.5020090675346804e-04,
     1.4441534974798749e-04, 1.3905645547988220e-04, 1.3407894125592345e-04,
     1.2944368476639352e-04, 1.2511671514529879e-04, 1.2106839531646248e-04,
     1.1727275474943309e-04, 1.1370694140656650e-04, 1.1035076883347277e-04,
     1.0718633971970244e-04},
    // x = 5.0000
    {3.9571230961051351e-01, 3.8897436261142802e-02, 1.0995436178434296e-02,
     4.8239233893086026e-03, 2.7029516726074744e-03, 1.7588618054381801e-03,
     1.2609532860734507e-03, 9.6544457198693935e-04, 7.7437215807186273e-04,
     6.4263796881362052e-04, 5.4721744083733096e-04, 4.7536192584985050e-04,
     4.1953772954610917e-04, 3.7504962395672951e-04, 3.3883928477462176e-04,
     3.0883922593786216e-04, 2.8360690049882855e-04, 2.6210807173758554e-04,
     2.4358355117301125e-04, 2.2746443943159463e-04, 2.1331661387467214e-04,
     2.0080341697760779e-04, 1.8965999309516823e-04, 1.7967526901971288e-04,
     1.7067906448410645e-04, 1.6253271606358607e-04, 1.5512215201573254e-04,
     1.4835270577484051e-04, 1.4214518185308057e-04, 1.3643283665400370e-04,
     1.3115903635008087e-04, 1.2627542182694386e-04, 1.2174045760118777e-04,
     1.1751827449917261e-04, 1.1357773923590568e-04, 1.0989170081920906e-04,
     1.0643637590783091e-04, 1.0319084421863287e-04, 1.0013663173119287e-04,
     9.7257364421647190e-05},
    // x = 5.1000
    {3.9187675029551905e-01, 3.7821568993137593e-02, 1.0526270628813447e-02,
     4.5622163312305491e-03, 2.5332125248135494e-03, 1.6374672703731667e-03,
     1.1681758243714899e-03, 8.9113128934448411e-04, 7.1276693869133580e-04,
     5.9022464629775292e-04, 5.0171781511192679e-04, 4.3522819135635645e-04,
     3.8367665055691628e-04, 3.4266369592228285e-04, 3.0933070827313688e-04,
     2.8174940925542629e-04, 2.5857697268652963e-04, 2.3885230717057135e-04,
     2.2187099857396209e-04, 2.0710591977656453e-04, 1.9415532409513124e-04,
     1.8270801199850588e-04, 1.7251940690393003e-04, 1.6339477893737923e-04,
     1.5517725926874289e-04, 1.4773913124047335e-04, 1.4097540468121879e-04,
     1.3479900809695684e-04, 1.2913714508010297e-04, 1.2392850039707069e-04,
     1.1912107430505308e-04, 1.1467048696986189e-04, 1.1053863858681478e-04,
     1.0669264143405206e-04, 1.0310396182017340e-04, 9.9747725497685710e-05,
     9.6602151452940990e-05, 9.3648087308738928e-05, 9.0868625748993120e-05,
     8.8248785995770774e-05},
    // x = 5.2000
    {3.8814647480016234e-01, 3.6791337536480923e-02, 1.0082446941219422e-02,
     4.3168913735900337e-03, 2.3751610763816785e-03, 1.5249889679494551e-03,
     1.0825302141041567e-03, 8.2272388101858386e-04, 6.5618209562672914e-04,
     5.4216646200900122e-04, 4.6005753436637297e-04, 3.9852344239740811e-04,
     3.5091103407496429e-04, 3.1309725299166833e-04, 2.8240975096290884e-04,
     2.5704984203496511e-04, 2.3576737330030154e-04, 2.1766912482203574e-04,
     2.0210143730870601e-04, 1.8857584227512900e-04, 1.7672052192011959e-04,
     1.6624778634270716e-04, 1.5693176845919289e-04, 1.4859280383681898e-04,
     1.4108628457401194e-04, 1.3429456955441502e-04, 1.2812102178022617e-04,
     1.2248555130685350e-04, 1.1732124049194412e-04, 1.1257175839230331e-04,
     1.0818935811395925e-04, 1.0413331001834185e-04, 1.0036866446102789e-04,
     9.6865266269811716e-05, 9.3596963395823203e-05, 9.0540966687605503e-05,
     8.7677328274923122e-05, 8.4988513779688099e-05, 8.2459049299594251e-05,
     8.0075228395003791e-05},
    // x = 5.3000
    {3.8451704357023420e-01, 3.5804287704087168e-02, 9.6623838872972930e-03,
     4.0868231631675713e-03, 2.2279403995530923e-03, 1.4207424234969446e-03,
     1.0034502595807706e-03, 7.5974145921130329e-04, 6.0420075294899259e-04,
     4.9809612200213676e-04, 4.2190871803116916e-04, 3.6495180865512449e-04,
     3.2097148039223226e-04, 2.8610312291468045e-04, 2.5784815205529720e-04,
     2.3452853798994853e-04, 2.1498026139416814e-04, 1.9837308670729535e-04,
     1.8410038941935833e-04, 1.7170948128358827e-04, 1.6085621350468840e-04,
     1.5127460818698197e-04, 1.4275606086132316e-04, 1.3513479545748289e-04,
     1.2827749807466508e-04, 1.2207580176872045e-04, 1.1644075314098804e-04,
     1.1129868014737208e-04, 1.0658806615049778e-04, 1.0225715694981826e-04,
     9.8262108785761068e-05, 9.4565540473696410e-05, 9.1135390842693578e-05,
     8.7944009232531864e-05, 8.4967425629183782e-05, 8.2184760519198938e-05,
     7.9577744335173386e-05, 7.7130323543161286e-05, 7.4828335738379449e-05,
     7.2659240089155734e-05},
    // x = 5.4000
    {3.8098425474813663e-01, 3.4858117944955913e-02, 9.2646086011347328e-03,
     3.8709687095426831e-03, 2.0907592614987161e-03, 1.3240974454514593e-03,
     9.3041582938457315e-04, 7.0174304068396228e-04, 5.5644117293025569e-04,
     4.5767768492608132e-04, 3.8697176583174763e-04, 3.3424316109759335e-04,
     2.9361220024370211e-04, 2.6145593180369579e-04, 2.3543789037843343e-04,
     2.1399239614462444e-04, 1.9603549424728359e-04, 1.8079540440256128e-04,
     1.6770909365528257e-04, 1.5635699283636680e-04, 1.4642053500052093e-04,
     1.3765379559339510e-04, 1.2986409887993586e-04, 1.2289847286892891e-04,
     1.1663400761360733e-04, 1.1097087319019997e-04, 1.0582718426735595e-04,
     1.0113516884788928e-04, 9.6838272594559728e-05, 9.2888944007141794e-05,
     8.9246921648955931e-05, 8.5877896108666970e-05, 8.2752454836411735e-05,
     7.9845242755005204e-05, 7.7134289071538120e-05, 7.4600463270690429e-05,
     7.2227032370952325e-05, 6.9999298191381260e-05, 6.7904298309337497e-05,
     6.5930558074656922e-05},
    // x = 5.5000
    {3.7754412943762905e-01, 3.3950668909014999e-02, 8.8877486625982671e-03,
     3.6683610795024786e-03, 1.9628869198230260e-03, 1.2344737127221064e-03,
     8.6294903649809958e-04, 6.4832418509193021e-04, 5.1255375799226228e-04,
     4.2060385885494482e-04, 3.5497289816180779e-04, 3.0615085663035388e-04,
     2.6860893309400613e-04, 2.3895017171691863e-04, 2.1498938162661064e-04,
     1.9526551170069722e-04, 1.7876903856886727e-04, 1.6478243948259410e-04,
     1.5278308576607101e-04, 1.4238206680732322e-04, 1.3328446972922693e-04,
     1.2526289276675075e-04, 1.1813935913692512e-04, 1.1177270206341818e-04,
     1.0604959622877700e-04, 1.0087807061328038e-04, 9.6182742073922434e-05,
     9.1901262859440852e-05, 8.7981638073198279e-05, 8.4380175609833860e-05,
     8.1059902046920582e-05, 7.7989326036190126e-05, 7.5141463801435643e-05,
     7.2493064420841585e-05, 7.0023988884749562e-05, 6.7716708598517794e-05,
     6.5555897457330136e-05, 6.3528097811010287e-05, 6.1621445214693601e-05,
     5.9825440278852422e-05},
    // x = 5.6000
    {3.7419289797353578e-01, 3.3079913772951150e-02, 8.5305247859259418e-03,
     3.4781035904595330e-03, 1.8436483407798039e-03, 1.1513367277263663e-03,
     8.0061074004527636e-04, 5.9911392000943460e-04, 4.7221831104094522e-04,
     3.8659353314402921e-04, 3.2566191189764500e-04, 2.8044968155068064e-04,
     2.4575704992702794e-04, 2.1839844032971219e-04, 1.9632983682315124e-04,
     1.7818763851682736e-04, 1.6303152478024371e-04, 1.5019433939866979e-04,
     1.3919090736344135e-04, 1.2966070142539265e-04, 1.2133068206315652e-04,
     1.1399055786665149e-04, 1.0747591712348750e-04, 1.0165647804232478e-04,
     9.6427745670207699e-05, 9.1704984049757561e-05, 8.7418791969166431e-05,
     8.3511808739543752e-05, 7.9936228945715442e-05, 7.6651904769890948e-05,
     7.3624880798271340e-05, 7.0826251090322812e-05, 6.8231259125652743e-05,
     6.5818582739687775e-05, 6.3569761346080067e-05, 6.1468733606842403e-05,
     5.9501461571682830e-05, 5.7655623058032579e-05, 5.5920358291917260e-05,
     5.4286059999531238e-05},
    // x = 5.7000
    {3.7092698712326211e-01, 3.2243949268929019e-02, 8.1917440657294557e-03,
     3.2993644623838603e-03, 1.7324198051943641e-03, 1.0741941043226316e-03,
     7.4299734123488390e-04, 5.5377192794580895e-04, 4.3514153172946130e-04,
     3.5538952473066371e-04, 2.9881013266766130e-04, 2.5693397618856259e-04,
     2.2486982411102304e-04, 1.9962983730739510e-04, 1.7930176752880523e-04,
     1.6261278954948153e-04, 1.4868693145286420e-04, 1.3690379653273873e-04,
     1.2681293168198473e-04, 1.1808008901422357e-04, 1.1045245737573854e-04,
     1.0373555218719143e-04, 9.7777481278766876e-05, 9.2458000006426081e-05,
     8.7680749371118070e-05, 8.3367654536276121e-05, 7.9454817884103070e-05,
     7.5889464068965416e-05, 7.2627637396653397e-05, 6.9632445099815973e-05,
     6.6872702054199205e-05, 6.4321874371480597e-05, 6.1957248064204503e-05,
     5.9759269008948257e-05, 5.7711014572649718e-05, 5.5797767372069481e-05,
     5.4006668942599384e-05, 5.2326436433204680e-05, 5.0747129387636632e-05,
     4.9259956611998441e-05},
    // x = 5.8000
    {3.6774300815254152e-01, 3.1440987362686697e-02, 7.8702937364383014e-03,
     3.1313718911048015e-03, 1.6286248700308449e-03, 1.0025921624915331e-03,
     6.8973784845095202e-04, 5.1198597280056619e-04, 4.0105472815798971e-04,
     3.2675652011293128e-04, 2.7420854627326573e-04, 2.3541592468644427e-04,
     2.0577685538037999e-04, 1.8248850337359440e-04, 1.6376162463027858e-04,
     1.4840796283640433e-04, 1.3561138815109595e-04, 1.2479491927675401e-04,
     1.1554029563022589e-04, 1.0753760283987342e-04, 1.0055273839476168e-04,
     9.4405821449088833e-05, 8.8956515253016651e-05, 8.4093831121546161e-05,
     7.9728906666969645e-05, 7.5789800112562334e-05, 7.2217677617655959e-05,
     6.8963980031030812e-05, 6.5988289338869154e-05, 6.3256702322383836e-05,
     6.0740576865934928e-05, 5.8415555469498789e-05, 5.6260797345045058e-05,
     5.4258369142422489e-05, 5.2392757514349536e-05, 5.0650476130543783e-05,
     4.9019746542479107e-05, 4.7490237260791777e-05, 4.6052849067544184e-05,
     4.4699537312508397e-05},
    // x = 5.9000
    {3.6463774569297541e-01, 3.0669347531712455e-02, 7.5651354047770345e-03,
     2.9734095089082035e-03, 1.5317306562363609e-03, 9.3611280401346378e-04,
     6.4049118859150253e-04, 4.7346954516281123e-04, 3.6971172531133893e-04,
     3.0047919589189752e-04, 2.5166609348963449e-04, 2.1572399529779209e-04,
     1.8832263331193725e-04, 1.6683228932458141e-04, 1.4957855872841765e-04,
     1.3545198172506517e-04, 1.2369208599225862e-04, 1.1376220499797823e-04,
     1.0527392848821340e-04, 9.7939875872502332e-05, 9.1543249174509436e-05,
     8.5917660795467455e-05, 8.0933440291246339e-05, 7.6488135113367384e-05,
     7.2499790810160112e-05, 6.8902112790636665e-05, 6.5640926572376973e-05,
     6.2671549963357634e-05, 5.9956816035281557e-05, 5.7465567393442390e-05,
     5.5171496393621477e-05, 5.3052242478181197e-05, 5.1088682826862987e-05,
     4.9264369913365897e-05, 4.7565081815855837e-05, 4.5978459875060232e-05,
     4.4493714606854182e-05, 4.3101385384070257e-05, 4.1793142799733712e-05,
     4.0561625153485355e-05},
    // x = 6.0000
    {3.6160814735365848e-01, 2.9927449598082675e-02, 7.2752997181318065e-03,
     2.8248122011660568e-03, 1.4412444359580029e-03, 8.7437064557963897e-04,
     5.9494374372580578e-04, 4.3795970764742658e-04, 3.4088695317041970e-04,
     2.7636050226923181e-04, 2.3100811387075321e-04, 1.9770151788495483e-04,
     1.7236522789063432e-04, 1.5253154338329254e-04, 1.3663329122354394e-04,
     1.2363443906803618e-04, 1.1282628620356350e-04, 1.0370960567093591e-04,
     9.5923668484702892e-05, 8.9201963105637371e-05, 8.3343698704457835e-05,
     7.8194955851368356e-05, 7.3635910411873435e-05, 6.9571982655663940e-05,
     6.5927584012487792e-05, 6.2641619995464540e-05, 5.9664203591857684e-05,
     5.6954217808508280e-05, 5.4477483566801084e-05, 5.2205365553437746e-05,
     5.0113699248873396e-05, 4.8181956459575988e-05, 4.6392590023906075e-05,
     4.4730514573961248e-05, 4.3182691649084801e-05, 4.1737795593376549e-05,
     4.0385942538613232e-05, 3.9118469054371353e-05, 3.7927750200956001e-05,
     3.6807049067273338e-05},
    // x = 6.1000
    {3.5865131402354594e-01, 2.9213807074103298e-02, 6.9998814346577121e-03,
     2.6849622503117023e-03, 1.3567104944832873e-03, 8.1701038777571970e-04,
     5.5280709393828857e-04, 4.0521512309114367e-04, 3.1437369892470067e-04,
     2.5422009526509062e-04, 2.1207492545499292e-04, 1.8120538648106976e-04,
     1.5777509586711421e-04, 1.3946800632721966e-04, 1.2481708617615659e-04,
     1.1285473603465179e-04, 1.0292041783511502e-04, 9.4549677792868698e-05,
     8.7407459283985774e-05, 8.1246579837841303e-05, 7.5881056900819764e-05,
     7.1168492905558444e-05, 6.6998153725673641e-05, 6.3282721161436173e-05,
     5.9952473368991152e-05, 5.6951104556951060e-05, 5.4232673190056236e-05,
     5.1759340949769079e-05, 4.9499674815698067e-05, 4.7427356148274587e-05,
     4.5520187972330021e-05, 4.3759323510354328e-05, 4.2128660792333775e-05,
     4.0614363279991647e-05, 3.9204477071608468e-05, 3.7888622824196331e-05,
     3.6657745986241040e-05, 3.5503912910641168e-05, 3.4420143345266747e-05,
     3.3400271975389738e-05},
    // x = 6.2000
    {3.5576449081523659e-01, 2.8527020982172649e-02, 6.7380348637275976e-03,
     2.5532857808340530e-03, 1.2777072443179544e-03, 7.6370440020692373e-04,
     5.1381594886938913e-04, 3.7501424991986525e-04, 2.8998250907276164e-04,
     2.3389290467267838e-04, 1.9472052842622260e-04, 1.6610487585926869e-04,
     1.4443399261834182e-04, 1.2753380477119488e-04, 1.1403081391342853e-04,
     1.0302120703175025e-04, 9.3889256587783689e-05, 8.6202808959767296e-05,
     7.9650619136786285e-05, 7.4003409013334096e-05, 6.9088896389054604e-05,
     6.4775331907700884e-05, 6.0960373849628547e-05, 5.7563402172383481e-05,
     5.4520102081151671e-05, 5.1778577877477856e-05, 4.9296518988355225e-05,
     4.7039102426378639e-05, 4.4977419125411576e-05, 4.3087278536506647e-05,
     4.1348290109529774e-05, 3.9743150031094863e-05, 3.8257081908322006e-05,
     3.6877394172999456e-05, 3.5593126878645417e-05, 3.4394767607323946e-05,
     3.3274021276147768e-05, 3.2223622327667080e-05, 3.1237180506391775e-05,
     3.0309053443260601e-05},
    // x = 6.3000
    {3.5294505860455216e-01, 2.7865774113295504e-02, 6.4889696478458419e-03,
     2.4292494811270085e-03, 1.2038445707031867e-03, 7.1415050470633097e-04,
     4.7772625196355027e-04, 3.4715369035692424e-04, 2.6753972843848901e-04,
     2.1522782590677813e-04, 1.7881142184126008e-04, 1.5228056203472621e-04,
     1.3223398014045997e-04, 1.1663053384782565e-04, 1.0418409816368092e-04,
     9.4050322993480726e-05, 8.5655177441983950e-05, 7.8596514171155784e-05,
     7.2585176108061128e-05, 6.7408471346773910e-05, 6.2906794086925891e-05,
     5.8958236550401337e-05, 5.5468205923677597e-05, 5.2362261074332337e-05,
     4.9581070909896917e-05, 4.7076801393338505e-05, 4.4810483653279847e-05,
     4.2750067983724318e-05, 4.0868965244122379e-05, 3.9144939832224533e-05,
     3.7559259767646721e-05, 3.6096037206153748e-05, 3.4741711663391728e-05,
     3.3484641356472366e-05, 3.2314777290056934e-05, 3.1223401268653697e-05,
     3.0202913733768905e-05, 2.9246660756845656e-05, 2.8348792042420437e-05,
     2.7504143669641291e-05},
    // x = 6.4000
    {3.5019052612371154e-01, 2.7228825691448244e-02, 6.2519468594664689e-03,
     2.3123575800123764e-03, 1.1347613896025544e-03, 6.6806994009758255e-04,
     4.4431344280464614e-04, 3.2144667838175558e-04, 2.4688616426190598e-04,
     1.9808652494363025e-04, 1.6422552349648747e-04, 1.3962333751971128e-04,
     1.2107652263901722e-04, 1.0666842131261864e-04, 9.5194539239590274e-05,
     8.5865965997983609e-05, 7.8147474434652503e-05, 7.1664795560124109e-05,
     6.6149263393002416e-05, 6.1403552528683689e-05, 5.7279787144118981e-05,
     5.3665156229293007e-05, 5.0472222241066262e-05, 4.7632244349534903e-05,
     4.5090485254233805e-05, 4.2802851897151720e-05, 4.0733451060997870e-05,
     3.8852783832730669e-05, 3.7136393564551742e-05, 3.5563840625428050e-05,
     3.4117915916119335e-05, 3.2784031071042274e-05, 3.1549740961069837e-05,
     3.0404366351218224e-05, 2.9338693152941822e-05, 2.8344730810863501e-05,
     2.7415516749794150e-05, 2.6544956996958033e-05, 2.5727695437334140e-05,
     2.4959005898501207e-05},
    // x = 6.5000
    {3.4749852256904484e-01, 2.6615006413543637e-02, 6.0262753882810265e-03,
     2.2021490575713505e-03, 1.0701234007709141e-03, 6.2520549338158861e-04,
     4.1337086417076151e-04, 2.9772169548017934e-04, 2.2787586455577836e-04,
     1.8234234649774285e-04, 1.5085118388304168e-04, 1.2803351296663824e-04,
     1.1087166194270095e-04, 9.7565565814612047e-05, 8.6987006462842454e-05,
     7.8398768803451772e-05, 7.1301741533033017e-05, 6.5347559816347314e-05,
     6.0286569276508256e-05, 5.5935682327172234e-05, 5.2157878290933507e-05,
     4.8848755150053936e-05, 4.5927482959596139e-05, 4.3330580015712021e-05,
     4.1007543673914376e-05, 3.8917726695712239e-05, 3.7028066807979957e-05,
     3.5311411372720693e-05, 3.3745264040160094e-05, 3.2310835177810069e-05,
     3.0992314039479206e-05, 2.9776304879281378e-05, 2.8651385724394449e-05,
     2.7607759931389549e-05, 2.6636978648116376e-05, 2.5731717980189362e-05,
     2.4885598739682305e-05, 2.4093039616866660e-05, 2.3349136791338625e-05,
     2.2649564611963295e-05},
    // x = 6.6000
    {3.4486679068710424e-01, 2.6023213837087604e-02, 5.8113085964935553e-03,
     2.0981950715848398e-03, 1.0096210199656052e-03, 5.8531978349564754e-04,
     3.8470830158365382e-04, 2.7582120326057654e-04, 2.1037500086066305e-04,
     1.6787931644571038e-04, 1.3858628597883358e-04, 1.1741999757633387e-04,
     1.0153726565968059e-04, 8.9247242723040595e-05, 7.9492993634408426e-05,
     7.1585513473482293e-05, 6.5059309100761495e-05, 5.9590088089183961e-05,
     5.4945836785876752e-05, 5.0956660873450133e-05, 4.7495586099746818e-05,
     4.4465984283464943e-05, 4.1793127775840613e-05, 3.9418387300374979e-05,
     3.7295164058312597e-05, 3.5385984947685953e-05, 3.3660393544247727e-05,
     3.2093395477064067e-05, 3.0664296491715141e-05, 2.9355822915139805e-05,
     2.8153448064043397e-05, 2.7044870784753990e-05, 2.6019607719059479e-05,
     2.5068671529619131e-05, 2.4184314767922499e-05, 2.3359824351422886e-05,
     2.2589355409327585e-05, 2.1867796010079075e-05, 2.1190656303638374e-05,
     2.0553977108505375e-05},
    // x = 6.7000
    {3.4229318030570582e-01, 2.5452408089778535e-02, 5.6064412213926960e-03,
     2.0000965824097014e-03, 9.5296747568615137e-04, 5.4819368496282679e-04,
     3.5815064417295668e-04, 2.5560048295335459e-04, 1.9426084639006258e-04,
     1.5459123029534203e-04, 1.2733742335358327e-04, 1.0769955132475888e-04,
     9.2998341626565070e-05, 8.1645271491839232e-05, 7.2650031910908355e-05,
     6.5368583786781383e-05, 5.9366730949010040e-05, 5.4342553630137634e-05,
     5.0080408535921680e-05, 4.6422627847434413e-05, 4.3251536072869170e-05,
     4.0477692262250398e-05, 3.8032004821140914e-05, 3.5860321961034533e-05,
     3.3919643992174493e-05, 3.2175421861424905e-05, 3.0599597929788706e-05,
     2.9169163254128642e-05, 2.7865080321164919e-05, 2.6671468330812742e-05,
     2.5574979764513294e-05, 2.4564318131480009e-05, 2.3629861164905059e-05,
     2.2763363660100658e-05, 2.1957721086062775e-05, 2.1206780019765446e-05,
     2.0505184979839803e-05, 1.9848253795137982e-05, 1.9231875519542000e-05,
     1.8652426293377084e-05},
    // x = 6.8000
    {3.3977564227891199e-01, 2.4901607877284355e-02, 5.4111065061770774e-03,
     1.9074821605176903e-03, 8.9989705704257561e-04, 5.1362487981899865e-04,
     3.3353665662971913e-04, 2.3692657267217245e-04, 1.7942084134101355e-04,
     1.4238082021709002e-04, 1.1701914972646381e-04, 9.8796102677274541e-05,
     8.5186412774449489e-05, 7.4697439082091073e-05, 6.6401154953797965e-05,
     5.9695466604069405e-05, 5.4175317417746271e-05, 4.9559582863295502e-05,
     4.5647812674305135e-05, 4.2293670669447614e-05, 3.9388088842915571e-05,
     3.6848271670200818e-05, 3.4610333380427792e-05, 3.2624253990768526e-05,
     3.0850352185390454e-05, 2.9256772738186870e-05, 2.7817666309022115e-05,
     2.6511850480394810e-05, 2.5321810924773326e-05, 2.4232946681415250e-05,
     2.3232993114609826e-05, 2.2311575893117035e-05, 2.1459862751583752e-05,
     2.0670289044715531e-05, 1.9936339569935440e-05, 1.9252373711819711e-05,
     1.8613484242233788e-05, 1.8015382488108510e-05, 1.7454304320831819e-05,
     1.6926932710238508e-05},
    // x = 6.9000
    {3.3731222281724860e-01, 2.4369886767260873e-02, 5.2247735415024725e-03,
     1.8200059622075250e-03, 8.5016350046406995e-04, 4.8142652718319697e-04,
     3.1071785289613470e-04, 2.1967729410153888e-04, 1.6575173786047658e-04,
     1.3115899381011505e-04, 1.0755329372055635e-04, 9.0640126020519703e-05,
     7.8038947059669688e-05, 6.8346974452408412e-05, 6.0694411678732572e-05,
     5.4518297799618856e-05, 4.9440710343454381e-05, 4.5199855962715765e-05,
     4.1609386206272049e-05, 3.8533468158228335e-05, 3.5871002110317945e-05,
     3.3545337498153886e-05, 3.1497397345804392e-05, 2.9680974747296310e-05,
     2.8059448121334037e-05, 2.6603444122961659e-05, 2.5289146465399712e-05,
     2.4097053160701020e-05, 2.3011050347105198e-05, 2.2017713096845061e-05,
     2.1105771280097923e-05, 2.0265697031699825e-05, 1.9489382894822574e-05,
     1.8769888341663132e-05, 1.8101238394414057e-05, 1.7478262330873462e-05,
     1.6896463510398692e-05, 1.6351913565985275e-05, 1.5841165826114965e-05,
     1.5361184026256667e-05},
    // x = 7.0000
    {3.3490105817655924e-01, 2.3856369729357479e-02, 5.0469448016084240e-03,
     1.7373458601776862e-03, 8.0353850397780623e-04, 4.5142604073183847e-04,
     2.8955746303540772e-04, 2.0374036099327016e-04, 1.5315881781032413e-04,
     1.2084413837292808e-04, 9.8868333109365547e-05, 8.3168073553011393e-05,
     7.1498837583195928e-05, 6.2542069573241928e-05, 5.5482422351643908e-05,
     4.9793448760225956e-05, 4.5122496143749348e-05, 4.1225743370657862e-05,
     3.7929932315608432e-05, 3.5108966437357018e-05, 3.2669123250171843e-05,
     3.0539434835894743e-05, 2.8665266599211424e-05, 2.7003930815000370e-05,
     2.5521627339321375e-05, 2.4191269576588824e-05, 2.2990913060821746e-05,
     2.1902601904931514e-05, 2.0911509944051694e-05, 2.0005292946886898e-05,
     1.9173594165129929e-05, 1.8407662751314944e-05, 1.7700056269878713e-05,
     1.7044406570543262e-05, 1.6435233905133233e-05, 1.5867798135691671e-05,
     1.5337978719970762e-05, 1.4842177214525851e-05, 1.4377237538207982e-05,
     1.3940380349108078e-05},
    // x = 7.1000
    {3.3254036968084461e-01, 2.3360229912505543e-02, 4.8771538601585030e-03,
     1.6592017167272263e-03, 7.5981035872004789e-04, 4.2346396515595240e-04,
     2.6992948545419531e-04, 1.8901256250976307e-04, 1.4155517706905221e-04,
     1.1136148499351984e-04, 9.0898823352885460e-05, 7.6321856841175218e-05,
     6.5513928456417510e-05, 5.7235442827079554e-05, 5.0721974159524007e-05,
     4.5481149814105586e-05, 4.1183853589532582e-05, 3.7602975013286617e-05,
     3.4577408605573386e-05, 3.1990084164810535e-05, 2.9754109800119833e-05,
     2.7803773136549564e-05, 2.6088543775051112e-05, 2.4568982155732414e-05,
     2.3213890003768660e-05, 2.1998287811180943e-05, 2.0901954584810803e-05,
     1.9908357023173994e-05, 1.9003853028779765e-05, 1.8177091505248131e-05,
     1.7418554615755995e-05, 1.6720204809521874e-05, 1.6075209840419263e-05,
     1.5477726504319958e-05, 1.4922729050952423e-05, 1.4405871919001462e-05,
     1.3923379083323433e-05, 1.3471954212444603e-05, 1.3048707230100156e-05,
     1.2651093905056529e-05},
    // x = 7.2000
    {3.3022845905623444e-01, 2.2880685642212344e-02, 4.7149632721014134e-03,
     1.5852937883423879e-03, 7.1878268819583566e-04, 3.9739294342957241e-04,
     2.5171781731587632e-04, 1.7539901505067465e-04, 1.3086107065162766e-04,
     1.0264252727090208e-04, 8.3584875678504160e-05, 7.0048373671660323e-05,
     6.0036582366076592e-05, 5.2383941026058376e-05, 4.6373652731034246e-05,
     4.1545147279397258e-05, 3.7591233144766534e-05, 3.4300339263931151e-05,
     3.1522643462564245e-05, 2.9149444426263492e-05, 2.7100175294971841e-05,
     2.5313984633136604e-05, 2.3744134086680346e-05, 2.2354182328051413e-05,
     2.1115330627898612e-05, 2.0004541138220022e-05, 1.9003179838370101e-05,
     1.8096022434509834e-05, 1.7270515661206059e-05, 1.6516221132781536e-05,
     1.5824391559544126e-05, 1.5187644219132991e-05, 1.4599706765880614e-05,
     1.4055217458719648e-05, 1.3549566760939968e-05, 1.3078770703346011e-05,
     1.2639368858394458e-05, 1.2228341547648127e-05, 1.1843042201175678e-05,
     1.1481141744017476e-05},
    // x = 7.3000
    {3.2796370405490027e-01, 2.2416997621897698e-02, 4.5599626089383059e-03,
     1.5153612513354581e-03, 6.8027328658591506e-04, 3.7307676740269799e-04,
     2.3481545659149569e-04, 1.6281247674627372e-04, 1.2100331342467548e-04,
     9.4624489933262993e-05, 7.6871680379325474e-05, 6.4299076217259719e-05,
     5.5023286150899029e-05, 4.7948176614975028e-05, 4.2401506397977916e-05,
     3.7952391119693143e-05, 3.4314065035386443e-05, 3.1289408970815192e-05,
     2.8739077998952072e-05, 2.6562130874477885e-05, 2.4683858144574671e-05,
     2.3047904707788928e-05, 2.1611036112402230e-05, 2.0339578757825955e-05,
     1.9206947015340267e-05, 1.8191892366975279e-05, 1.7277242159032804e-05,
     1.6448976659924229e-05, 1.5695543911095076e-05, 1.5007344365654550e-05,
     1.4376338519162867e-05, 1.3795744827060576e-05, 1.3259804719928344e-05,
     1.2763598054896807e-05, 1.2302896882481037e-05, 1.1874048609407711e-05,
     1.1473881922883082e-05, 1.1099630491549628e-05, 1.0748870662491101e-05,
     1.0419470261505019e-05},
    // x = 7.4000
    {3.2574455434925187e-01, 2.1968466323521774e-02, 4.4117666357726855e-03,
     1.4491608390360712e-03, 6.4411304811641393e-04, 3.5038950485933467e-04,
     2.1912376975156137e-04, 1.5117271930004917e-04, 1.1191473164670022e-04,
     8.7249843031373772e-05, 7.0709071382873547e-05, 5.9029576885863054e-05,
     5.0434291030086225e-05, 4.3892196933958418e-05, 3.8772740276168891e-05,
     3.4672750464819848e-05, 3.1324493464853954e-05, 2.8544292108824450e-05,
     2.6202531262114454e-05, 2.4205465916801084e-05, 2.2483811461193417e-05,
     2.0985372214821282e-05, 1.9670151628901366e-05, 1.8507031227769374e-05,
     1.7471466660512254e-05, 1.6543858461860453e-05, 1.5708379758466243e-05,
     1.4952119328995551e-05, 1.4264446078728630e-05, 1.3636531443104250e-05,
     1.3060986081998675e-05, 1.2531580396779663e-05, 1.2043027288346460e-05,
     1.1590811663036376e-05, 1.1171055425260032e-05, 1.0780409676578929e-05,
     1.0415967966724215e-05, 1.0075195975763916e-05, 9.7558741251833717e-06,
     9.4560504398346200e-06},
    // x = 7.5000
    {3.2356952767817865e-01, 2.1534429553868719e-02, 4.2700136194305550e-03,
     1.3864655818003298e-03, 6.1014498016363165e-04, 3.2921469675499000e-04,
     2.0455181961047048e-04, 1.4040595231921895e-04, 1.0353366097603005e-04,
     8.0465857762978331e-05, 6.5051128489916973e-05, 5.4199288542694872e-05,
     4.6233284422276485e-05, 4.0183182693938623e-05, 3.5457437505900387e-05,
     3.1678754501552345e-05, 2.8597134626685905e-05, 2.6041404835519915e-05,
     2.3890986606358260e-05, 2.2058808952495049e-05, 2.0480611933297979e-05,
     1.9108047941158907e-05, 1.7904112754799902e-05, 1.6840046921211092e-05,
     1.5893189009939287e-05, 1.5045459422613321e-05, 1.4282270693695828e-05,
     1.3591731774536485e-05, 1.2964058496778659e-05, 1.2391130944569439e-05,
     1.1866157038784403e-05, 1.1383413947867604e-05, 1.0938047237854249e-05,
     1.0525913354179311e-05, 1.0143454972144743e-05, 9.7876015286771763e-06,
     9.4556892258826669e-06, 9.1453962227741286e-06, 8.8546897706813737e-06,
     8.5817828129758671e-06},
    // x = 7.6000
    {3.2143720622840949e-01, 2.1114260183879532e-02, 4.1343637577761841e-03,
     1.3270636418052837e-03, 5.7822329336818261e-04, 3.0944461887322575e-04,
     1.9101574830032058e-04, 1.3044429568839184e-04, 9.5803486966136093e-05,
     7.4224200327875165e-05, 5.9855813999277388e-05, 4.9771096088435137e-05,
     4.2387090565355019e-05, 3.6791173071925309e-05, 3.2428305230353391e-05,
     2.8945356463134429e-05, 2.6108856376089106e-05, 2.3759264932258620e-05,
     2.1784397315029798e-05, 2.0103372843124607e-05, 1.8656586015871729e-05,
     1.7399249553298210e-05, 1.6297124825737661e-05, 1.5323630507736077e-05,
     1.4457842133090977e-05, 1.3683080992161876e-05, 1.2985901128923773e-05,
     1.2355350420549626e-05, 1.1782423663791252e-05, 1.1259652328650214e-05,
     1.0780793023010395e-05, 1.0340588221251639e-05, 9.9345805590941647e-06,
     9.5589672960865679e-06, 9.2104852234988910e-06, 8.8863188803173229e-06,
     8.5840267803888106e-06, 8.3014816794592792e-06, 8.0368218762390395e-06,
     7.7884112519603533e-06},
    // x = 7.7000
    {3.1934623323527611e-01, 2.0707364029375935e-02, 4.0044977211195461e-03,
     1.2707572352409697e-03, 5.4821256258441500e-04, 2.9097960262161929e-04,
     1.7843821077603996e-04, 1.2122529592218978e-04, 8.8672224412081211e-05,
     6.8480560527180642e-05, 5.5084640722703663e-05, 4.5711056642206459e-05,
     3.8865397395061704e-05, 3.3688815064269271e-05, 2.9660443107043615e-05,
     2.6449718650485082e-05, 2.3838577615470063e-05, 2.1678303793747504e-05,
     1.9864509733400859e-05, 1.8322056964222905e-05, 1.6995651864798599e-05,
     1.5843801530515900e-05, 1.4834823566583413e-05, 1.3944147896718119e-05,
     1.3152452484347607e-05, 1.2444349925080777e-05, 1.1807445668332238e-05,
     1.1231651787974953e-05, 1.0708679574794326e-05, 1.0231659277692786e-05,
     9.7948515907194537e-06, 9.3934262433173969e-06, 9.0232902884541240e-06,
     8.6809536274492273e-06, 8.3634227371618680e-06, 8.0681159725567563e-06,
     7.7927955301770416e-06, 7.5355123906580738e-06, 7.2945614553605235e-06,
     7.0684447516861113e-06},
    // x = 7.8000
    {3.1729530978828779e-01, 2.0313177872391543e-02, 3.8801152973201823e-03,
     1.2173616351039182e-03, 5.1998695299664368e-04, 2.7372741012756448e-04,
     1.6674785464252713e-04, 1.1269148278032461e-04, 8.2092132225966951e-05,
     6.3194312106516038e-05, 5.0702368656667852e-05, 4.1988125821169142e-05,
     3.5640507365839972e-05, 3.0851134946552108e-05, 2.7131132344687040e-05,
     2.4171016603598629e-05, 2.1767085623831430e-05, 1.9780695295297903e-05,
     1.8114702330490074e-05, 1.6699295336432201e-05, 1.5483175585965754e-05,
     1.4427898720820963e-05, 1.3504145257404493e-05, 1.2689202410475269e-05,
     1.1965226558496388e-05, 1.1318020665803931e-05, 1.0736158652320826e-05,
     1.0210348050847109e-05, 9.7329592190259279e-06, 9.2976728528639043e-06,
     8.8992127781528300e-06, 8.5331410568930992e-06, 8.1956992054144131e-06,
     7.8836839341119040e-06, 7.5943490131859230e-06, 7.3253271109000007e-06,
     7.0745670444937722e-06, 6.8402830300165333e-06, 6.6209133507337414e-06,
     6.4150864760712730e-06},
    // x = 7.9000
    {3.1528319182796044e-01, 1.9931167613132999e-02, 3.7609341328442981e-03,
     1.1667042483393926e-03, 4.9342950619725684e-04, 2.5760265919722930e-04,
     1.5587884245002755e-04, 1.0478996274628290e-04, 7.6019360806022475e-05,
     5.8328202104005882e-05, 4.6676727817533121e-05, 3.8573907829690347e-05,
     3.2687110102771492e-05, 2.8255329880392425e-05, 2.4819643437436941e-05,
     2.2090260710543408e-05, 1.9876869719478403e-05, 1.8050200017955305e-05,
     1.6519839251224704e-05, 1.5220918470646989e-05, 1.4105840499755682e-05,
     1.3138982280436212e-05, 1.2293208708839539e-05, 1.1547522242955095e-05,
     1.0885443351885882e-05, 1.0293872391349620e-05, 9.7622754113755858e-06,
     9.2820921736592677e-06, 8.8462993096312041e-06, 8.4490835563215331e-06,
     8.0855942635368793e-06, 7.7517537732063321e-06, 7.4441105856263531e-06,
     7.1597245320647725e-06, 6.8960761512177937e-06, 6.6509945553760345e-06,
     6.4225995552281114e-06, 6.2092548780109471e-06, 6.0095300880839220e-06,
     5.8221693875555985e-06},
    // x = 8.0000
    {3.1330868732130712e-01, 1.9560826543337788e-02, 3.6466885626319288e-03,
     1.1186237615785708e-03, 4.6843148144671778e-04, 2.4252629406987174e-04,
     1.4577041292912981e-04, 9.7472046261010991e-05, 7.0413629125790788e-05,
     5.3848066702245747e-05, 4.2978164965009861e-05, 3.5442427272668429e-05,
     2.9982074960553879e-05, 2.5880577881958377e-05, 2.2707060931897661e-05,
     2.0190133695157780e-05, 1.8151969790461402e-05, 1.6472023448919521e-05,
     1.5066137050604814e-05, 1.3874027685616655e-05, 1.2851528239783588e-05,
     1.1965626870538654e-05, 1.1191207970665551e-05, 1.0508858173590003e-05,
     9.9033566410136030e-06, 9.3626154691976349e-06, 8.8769225641602686e-06,
     8.4383917498741575e-06, 8.0405573962856718e-06, 7.6780714803603203e-06,
     7.3464743399219387e-06, 7.0420191770455035e-06, 6.7615362657093707e-06,
     6.5023268355373243e-06, 6.2620793799055295e-06, 6.0388030819359351e-06,
     5.8307744321834442e-06, 5.6364941029305827e-06, 5.4546518635798551e-06,
     5.2840978495714746e-06},
    // x = 8.1000
    {3.1137065360428079e-01, 1.9201673732481602e-02, 3.5371285221830833e-03,
     1.0729693501750957e-03, 4.4489174772511137e-04, 2.2842509823747743e-04,
     1.3636647793415963e-04, 9.0692905868222697e-05, 6.5237929008918200e-05,
     4.9722571300786520e-05, 3.9579612138029476e-05, 3.2569920791342750e-05,
     2.7504261735926894e-05, 2.3707864525883091e-05, 2.0776123711109576e-05,
     1.8454842564402068e-05, 1.6577838359110920e-05, 1.5032686899493288e-05,
     1.3741043419963152e-05, 1.2646880769121531e-05, 1.1709210612152544e-05,
     1.0897438087616484e-05, 1.0188314795595078e-05, 9.5638906001797817e-06,
     9.0101061808383190e-06, 8.5158064680380013e-06, 8.0720365303126815e-06,
     7.6715307424509291e-06, 7.3083365898726045e-06, 6.9775337990039311e-06,
     6.6750219791583925e-06, 6.3973581882588519e-06, 6.1416313445328388e-06,
     5.9053641552941999e-06, 5.6864358225826850e-06, 5.4830205974899771e-06,
     5.2935385396860623e-06, 5.1166157603838066e-06, 4.9510520956735050e-06,
     4.7957946474068617e-06},
    // x = 8.2000
    {3.0946799488028637e-01, 1.8853252518921601e-02, 3.4320185357800403e-03,
     1.0295999456663453e-03, 4.2271622254221188e-04, 2.1523124590901001e-04,
     1.2761525213579074e-04, 8.4411262670313182e-05, 6.0458254273326606e-05,
     4.5922972724049351e-05, 3.6456275108829006e-05, 2.9934646787394295e-05,
     2.5234347935239396e-05, 2.1719824902978183e-05, 1.9011079415138275e-05,
     1.6869983723589601e-05, 1.5141214967020521e-05, 1.3719909996312887e-05,
     1.2533126823098308e-05, 1.1528787956249662e-05, 1.0668851239121541e-05,
     9.9249591970632250e-06, 9.2755899696082892e-06, 8.7041450402578886e-06,
     8.1976370073157681e-06, 7.7457709381911760e-06, 7.3402895046097476e-06,
     6.9744984007424337e-06, 6.6429171993105780e-06, 6.3410189261311438e-06,
     6.0650333335120179e-06, 5.8117965470785202e-06, 5.5786349081584598e-06,
     5.3632743328143727e-06, 5.1637689223426004e-06, 4.9784442481402357e-06,
     4.8058519296224977e-06, 4.6447329811162416e-06, 4.4939880250958534e-06,
     4.3526529244047189e-06},
    // x = 8.3000
    {3.0759965986463939e-01, 1.8515129098646468e-02, 3.3311367752308103e-03,
     9.8838355717145200e-04, 4.0181735380073557e-04, 2.0288188898184747e-04,
     1.1946891275255242e-04, 7.8589098715375246e-05, 5.6043352627872200e-05,
     4.2422901660594795e-05, 3.3585440026707777e-05, 2.7516711653789889e-05,
     2.3154671140314160e-05, 1.9900599481921833e-05, 1.7397551741200968e-05,
     1.5422420083546967e-05, 1.3830011776024195e-05, 1.2522503704870230e-05,
     1.1431976057982586e-05, 1.0510017291409892e-05, 9.7213160998211388e-06,
     9.0395863243805158e-06, 8.4449027012293373e-06, 7.9219153281548690e-06,
     7.4586261033327141e-06, 7.0455332503225678e-06, 6.6750222083431971e-06,
     6.3409246948335386e-06, 6.0381946450540187e-06, 5.7626667265132330e-06,
     5.5108740817067189e-06, 5.2799091491659889e-06, 5.0673162222590171e-06,
     4.8710076710169534e-06, 4.6891980030229254e-06, 4.5203515120860766e-06,
     4.3631403765153591e-06, 4.2164108661249943e-06, 4.0791558946636945e-06,
     3.9504925771779906e-06},
    // x = 8.4000
    {3.0576463956555450e-01, 1.8186891204843788e-02, 3.2342741839495553e-03,
     9.4919664259338857e-04, 3.8211364130802806e-04, 1.9131877664246451e-04,
     1.1188328683858701e-04, 7.3191393138260822e-05, 5.1964498386611026e-05,
     3.9198163594853464e-05, 3.0946296674010037e-05, 2.5297911069962032e-05,
     2.1249085144659423e-05, 1.8235702645097505e-05, 1.5922419478498931e-05,
     1.4100169089144224e-05, 1.2633209380037157e-05, 1.1430272938236633e-05,
     1.0428108848776020e-05, 9.5817085253490331e-06, 8.8582921612954114e-06,
     8.2334913353729194e-06, 7.6888573358443683e-06, 7.2101938056040932e-06,
     6.7864157550323020e-06, 6.4087528462939693e-06, 6.0701827965560083e-06,
     5.7650216689655931e-06, 5.4886230722774415e-06, 5.2371542226762234e-06,
     5.0074270808958531e-06, 4.7967695092737744e-06, 4.6029258872257453e-06,
     4.4239796720730052e-06, 4.2582924910737219e-06, 4.1044558157883435e-06,
     3.9612523060788004e-06, 3.8276246526731299e-06, 3.7026502840259724e-06,
     3.5855206959020202e-06},
    // x = 8.5000
    {3.0396196519290036e-01, 1.7868146871993514e-02, 3.1412336615864647e-03,
     9.1192352581892240e-04, 3.6352919480716543e-04, 1.8048790495610854e-04,
     1.0481756385332643e-04, 6.8185880063682342e-05, 4.8195284232034741e-05,
     3.6226556643173295e-05, 2.8519776894685234e-05, 2.3261585045749730e-05,
     1.9502828649505866e-05, 1.6711902778058977e-05, 1.4573706235114536e-05,
     1.2892300694569387e-05, 1.1540761913000406e-05, 1.0433927889315764e-05,
     9.5128886538477040e-06, 8.7357947753953142e-06, 8.0722133664570877e-06,
     7.4995517067116060e-06, 7.0007267281149662e-06, 6.5626078679135694e-06,
     6.1749529871348747e-06, 5.8296663152334416e-06, 5.5202713568386882e-06,
     5.2415301706944921e-06, 4.9891641398561132e-06, 4.7596462918323679e-06,
     4.5500448357333702e-06, 4.3579038805347928e-06, 4.1811514978260448e-06,
     4.0180281381205062e-06, 3.8670303672603831e-06, 3.7268662547249154e-06,
     3.5964197102836414e-06, 3.4747217552979254e-06, 3.3609272139233174e-06,
     3.2542956742030233e-06},
    // x = 8.6000
    {3.0219070618656318e-01, 1.7558523278656722e-02, 3.0518293047850347e-03,
     8.7645585641032541e-04, 3.4599332565143594e-04, 1.7033919402298511e-04,
     9.8234031429375414e-05, 6.3542826448505913e-05, 4.4711430410465623e-05,
     3.3487704843624282e-05, 2.6288406881469841e-05, 2.1392485514144660e-05,
     1.7902405416148104e-05, 1.5317112891635089e-05, 1.3340479907358513e-05,
     1.1788844398012764e-05, 1.0543510620396398e-05, 9.5250033026453694e-06,
     8.6784489491287026e-06, 7.9649312471036261e-06, 7.3561933121780119e-06,
     6.8312867518442564e-06, 6.3743916663792611e-06, 5.9733622860168704e-06,
     5.6187345218147096e-06, 5.3030347617061926e-06, 5.0202894871763841e-06,
     4.7656714623703999e-06, 4.5352405094648124e-06, 4.3257508937157884e-06,
     4.1345063407938847e-06, 3.9592496000494147e-06, 3.7980773916006365e-06,
     3.6493742317711053e-06, 3.5117604570396462e-06, 3.3840510388464434e-06,
     3.2652226796811694e-06, 3.1543873226248958e-06, 3.0507706703073702e-06,
     2.9536946480284839e-06},
    // x = 8.7000
    {3.0044996835682453e-01, 1.7257665663553846e-02, 2.9658856999812593e-03,
     8.4269210855854402e-04, 3.2944016947828586e-04, 1.6082619047798504e-04,
     9.2097832429321934e-05, 5.9234828195031702e-05, 4.1490609881485176e-05,
     3.0962905574575586e-05, 2.4236172122373666e-05, 1.9676655378288151e-05,
     1.6435474868562866e-05, 1.4040290846347045e-05, 1.2212761026651525e-05,
     1.0780704527888547e-05, 9.6331051366040253e-06, 8.6957849724309915e-06,
     7.9176243130721348e-06, 7.2624303790825521e-06, 6.7039640112979537e-06,
     6.2227996250335914e-06, 5.8042857982074257e-06, 5.4371867776840706e-06,
     5.1127567565239007e-06, 4.8240959817264797e-06, 4.5656944873802519e-06,
     4.3331032668689574e-06, 4.1226936028829190e-06, 3.9314784124533757e-06,
     3.7569778935124993e-06, 3.5971172710707653e-06, 3.4501481086103959e-06,
     3.3145871305772670e-06, 3.1891682046575413e-06, 3.0728043180310542e-06,
     2.9645572179637567e-06, 2.8636129841221206e-06, 2.7692622311221885e-06,
     2.6808839545274047e-06},
    // x = 8.8000
    {2.9873889212967619e-01, 1.6965236309919356e-02, 2.8832372644694653e-03,
     8.1053711632112792e-04, 3.1380833745184197e-04, 1.5190579329381251e-04,
     8.6376741541844396e-05, 5.5236623008437521e-05, 3.8512288069948104e-05,
     2.8634989891684166e-05, 2.2348393911734235e-05, 1.8101318014258097e-05,
     1.5090752229116996e-05, 1.2871348331389123e-05, 1.1181439196138059e-05,
     9.8595830450299856e-06, 8.8019317784347585e-06, 7.9392428177767088e-06,
     7.2238877003813057e-06, 6.6222028306039677e-06, 6.1098201873908625e-06,
     5.6687245788379185e-06, 5.2853455565086675e-06, 4.9492883492849632e-06,
     4.6524703023246863e-06, 4.3885210067292638e-06, 4.1523577413474031e-06,
     3.9398798406780938e-06, 3.7477452353306389e-06, 3.5732047339981284e-06,
     3.4139775119552398e-06, 3.2681564280563555e-06, 3.1341352200039938e-06,
     3.0105519434534657e-06, 2.8962446088580816e-06, 2.7902160747575279e-06,
     2.6916060347899983e-06, 2.5996684911474303e-06, 2.5137535079874121e-06,
     2.4332923306564188e-06},
    // x = 8.9000
    {2.9705665089048200e-01, 1.6680913593483147e-02, 2.8037276322453609e-03,
     7.7990164240139298e-04, 2.9904059383863705e-04, 1.4353800101492818e-04,
     8.1040959813606655e-05, 5.1524918600835673e-05, 3.5757575984860857e-05,
     2.6488194677563130e-05, 2.0611616426499325e-05, 1.6654776318790695e-05,
     1.3857917351133403e-05, 1.1801067825636168e-05, 1.0238196899447470e-05,
     9.0179091911216480e-06, 8.0430482271213583e-06, 7.2489699445500926e-06,
     6.5912933470362204e-06, 6.0387037841756997e-06, 5.5685686011707689e-06,
     5.1641789980893712e-06, 4.8129646312264285e-06, 4.5053079732123062e-06,
     4.2337386662340217e-06, 3.9923746159246364e-06, 3.7765269061877556e-06,
     3.5824157048280912e-06, 3.4069627687378773e-06, 3.2476377155080967e-06,
     3.1023426254476187e-06, 2.9693243634996076e-06, 2.8471072145201646e-06,
     2.7344405877415451e-06, 2.6302580279028752e-06, 2.5336448001847625e-06,
     2.4438120410732774e-06, 2.3600759840641473e-06, 2.2818411417302593e-06,
     2.2085865972595011e-06},
    // x = 9.0000
    {2.9540244941984034e-01, 1.6404391089764096e-02, 2.7272090814003118e-03,
     7.5070197793971553e-04, 2.8508355786062938e-04, 1.3568567870327690e-04,
     7.6062925647187038e-05, 4.8078234962597314e-05, 3.3209095575126677e-05,
     2.4508045593915243e-05, 1.9013503455428313e-05, 1.5326320470961985e-05,
     1.2727531485858078e-05, 1.0821026836654046e-05, 9.3754400279432996e-06,
     8.2487753735376042e-06, 7.3501240273880471e-06, 6.6191271565069692e-06,
     6.0144247995036952e-06, 5.5068840830531027e-06, 5.0754819529106169e-06,
     4.7047197768141983e-06, 4.3829525731294097e-06, 4.1012812057858131e-06,
     3.8528006991807110e-06, 3.6320794540653940e-06, 3.4347915594806761e-06,
     3.2574526980997935e-06, 3.0972274616005486e-06, 2.9517867346970855e-06,
     2.8192007366915962e-06, 2.6978578250837938e-06, 2.5864021607552812e-06,
     2.4836853534674209e-06, 2.3887285886463336e-06, 2.3006926961065174e-06,
     2.2188542964934165e-06, 2.1425866420744708e-06, 2.0713441149390433e-06,
     2.0046495979793027e-06},
    // x = 9.1000
    {2.9377552241592036e-01, 1.6135376736671994e-02, 2.6535420000838392e-03,
     7.2285957098511451e-04, 2.7188742793437832e-04, 1.2831434301754344e-04,
     7.1417140917739714e-05, 4.4876759529697879e-05, 3.0850856288755693e-05,
     2.2681249913117150e-05, 1.7542743948302815e-05, 1.4106143649683737e-05,
     1.1690961288605017e-05, 9.9235287760994765e-06, 8.5862345310203949e-06,
     7.5458787312899361e-06, 6.7173863835094942e-06, 6.0443924266867315e-06,
     5.4883476068089348e-06, 5.0221457671327684e-06, 4.6262569465968223e-06,
     4.2863036439755487e-06, 3.9914971538919867e-06, 3.7336023865398426e-06,
     3.5062364657834047e-06, 3.3043834249051662e-06, 3.1240519879146917e-06,
     2.9620300477672475e-06, 2.8157057218178703e-06, 2.6829350359067059e-06,
     2.5619427817792123e-06, 2.4512473185943675e-06, 2.3496028890840063e-06,
     2.2559549066123838e-06, 2.1694049589512294e-06, 2.0891831690945302e-06,
     2.0146261821755637e-06, 1.9451594949837469e-06, 1.8802831666848123e-06,
     1.8195601837700976e-06},
    // x = 9.2000
    {2.9217513309792820e-01, 1.5873592048700601e-02, 2.5825943882752567e-03,
     6.9630068149669519e-04, 2.5940572655650944e-04, 1.2139196397671150e-04,
     6.7080010973192002e-05, 4.1902214174695807e-05, 2.8668141890399109e-05,
     2.0995598385852798e-05, 1.6188965624679881e-05, 1.2985265015281750e-05,
     1.0740309430129708e-05, 9.1015398867472500e-06, 7.8642486470153474e-06,
     6.9034678764321376e-06, 6.1395707789295544e-06, 5.5199148841076457e-06,
     5.0085662557975941e-06, 4.5803016101856964e-06, 4.2169761391385318e-06,
     3.9052510797600823e-06, 3.6351301409016609e-06, 3.3989920925804925e-06,
     3.1909362235972887e-06, 3.0063300608247874e-06, 2.8414908296178211e-06,
     2.6934571811223828e-06, 2.5598229959042336e-06, 2.4386146157307684e-06,
     2.3281989397294538e-06, 2.2272137764349976e-06, 2.1345144607776266e-06,
     2.0491325061659316e-06, 1.9702432650012082e-06, 1.8971404047821717e-06,
     1.8292155925239098e-06, 1.7659421965844160e-06, 1.7068621144965696e-06,
     1.6515750532142706e-06},
    // x = 9.3000
    {2.9060057188571942e-01, 1.5618771379260281e-02, 2.5142413928119717e-03,
     6.7095606089148836e-04, 2.4759506423452929e-04, 1.1488878207702097e-04,
     6.3029697385433184e-05, 3.9137733039379476e-05, 2.6647406672716062e-05,
     1.9439875374086011e-05, 1.4942655947820468e-05, 1.1955459323981543e-05,
     9.8683512351291595e-06, 8.3486316881750349e-06, 7.2037002205673364e-06,
     6.3162943504451286e-06, 5.6118759884744961e-06, 5.0412729108325170e-06,
     4.5709849677939097e-06, 4.1775382973226227e-06, 3.8440732321187054e-06,
     3.5582134967039487e-06, 3.3106961763493134e-06, 3.0944675514809516e-06,
     2.9040722280339100e-06, 2.7352315965317774e-06, 2.5845473088680187e-06,
     2.4492890264425576e-06, 2.3272400524822154e-06, 2.2165834146940879e-06,
     2.1158166660633791e-06, 2.0236873737468586e-06, 1.9391437133266450e-06,
     1.8612962305407619e-06, 1.7893879552718756e-06, 1.7227708298702188e-06,
     1.6608869592801429e-06, 1.6032535779182415e-06, 1.5494509067576629e-06,
     1.4991122764605827e-06},
    // x = 9.4000
    {2.8905115515090612e-01, 1.5370661227944121e-02, 2.4483648733125388e-03,
     6.4676065430883316e-04, 2.3641492098963829e-04, 1.0877713953989955e-04,
     5.9245982413950144e-05, 3.6567750309825525e-05, 2.4776180270784592e-05,
     1.8003776545037556e-05, 1.3795089829738382e-05, 1.1009192599354977e-05,
     9.0684768206665995e-06, 7.6589284553208892e-06, 6.5993086562251046e-06,
     5.7795683762710516e-06, 5.1299230908388696e-06, 4.6044359809068936e-06,
     4.1718720093143748e-06, 3.8103829142552382e-06, 3.5043014946447646e-06,
     3.2421433895639888e-06, 3.0153244784371582e-06, 2.8173157432468040e-06,
     2.6430731051045839e-06, 2.4886444996539314e-06, 2.3508938258359912e-06,
     2.2273035751422928e-06, 2.1158314402231049e-06, 2.0148046029830907e-06,
     1.9228407457111713e-06, 1.8387882942419352e-06, 1.7616806904577705e-06,
     1.6907010278256750e-06, 1.6251544312599615e-06, 1.5644462872503333e-06,
     1.5080649381989730e-06, 1.4555678155263087e-06, 1.4065702450978619e-06,
     1.3607363465906171e-06},
    // x = 9.5000
    {2.8752622403511485e-01, 1.5129019589748797e-02, 2.3848529968083527e-03,
     6.2365332390284546e-04, 2.2582744407537985e-04, 1.0303132456793253e-04,
     5.5710144229450350e-05, 3.4177897110271278e-05, 2.3042980356124653e-05,
     1.6677833482443080e-05, 1.2738263488353586e-05, 1.0139563335143396e-05,
     8.3346382537156704e-06, 7.0270592871153179e-06, 6.0462510981270010e-06,
     5.2889185241043478e-06, 4.6897181241860316e-06, 4.2057299058124873e-06,
     3.8078272008282883e-06, 3.4756724496287123e-06, 3.1947050340957045e-06,
     2.9542671847485762e-06, 2.7464031082361824e-06, 2.5650689464698805e-06,
     2.4056005577043681e-06, 2.2643472336744553e-06, 2.1384146857734245e-06,
     2.0254814978047800e-06, 1.9236659206085897e-06, 1.8314277677361214e-06,
     1.7474951794069025e-06, 1.6708092661115543e-06, 1.6004817830171288e-06,
     1.5357624214953628e-06, 1.4760132817098792e-06, 1.4206887658042597e-06,
     1.3693196044421579e-06, 1.3215000650830375e-06, 1.2768776312382251e-06,
     1.2351446167180274e-06},
    // x = 9.6000
    {2.8602514333134182e-01, 1.4893615343481822e-02, 2.3235998590601364e-03,
     6.0157659160467859e-04, 2.1579726066364043e-04, 9.7627427577182813e-05,
     5.2404842023862326e-05, 3.1954906761424829e-05, 2.1437232548464505e-05,
     1.5453344626720986e-05, 1.1764833927960653e-05, 9.3402487498083359e-06,
     7.6613012893092396e-06, 6.4481143615561119e-06, 5.5401224620396497e-06,
     4.8403549431404093e-06, 4.2876180597134828e-06, 3.8418051812338830e-06,
     3.4757523360589977e-06, 3.1705260387150586e-06, 2.9125926572413055e-06,
     2.6920605445853756e-06, 2.5015555690790142e-06, 2.3354825061303192e-06,
     2.1895281925661566e-06, 2.0603200492129871e-06, 1.9451867718232919e-06,
     1.8419876258218920e-06, 1.7489886942370362e-06, 1.6647718271174749e-06,
     1.5881667348477831e-06, 1.5181997049406514e-06, 1.4540544229377322e-06,
     1.3950417187551632e-06, 1.3405759721739924e-06, 1.2901565411016494e-06,
     1.2433530170500871e-06, 1.1997934246772376e-06, 1.1591547062467003e-06,
     1.1211549942782422e-06},
    // x = 9.7000
    {2.8454730042461540e-01, 1.4664227676781556e-02, 2.2645051306851259e-03,
     5.8047639991610351e-04, 2.0629130434842796e-04, 9.2543208457867462e-05,
     4.9314010205325763e-05, 2.9886527712165620e-05, 1.9949196939652688e-05,
     1.4322312006230594e-05, 1.0868063560059769e-05, 8.6054556550532595e-06,
     7.0434012893300338e-06, 5.9176050092798399e-06, 5.0768989792439932e-06,
     4.4302358425182331e-06, 3.9202997971568684e-06, 3.5096081573687741e-06,
     3.1728242502415348e-06, 2.8923197013255084e-06, 2.6555140875501324e-06,
     2.4532259039346904e-06, 2.2786195265963641e-06, 2.1265146208048850e-06,
     1.9929222744642678e-06, 1.8747266183262104e-06, 1.7694619835780212e-06,
     1.6751541276501461e-06, 1.5902052560586016e-06, 1.5133095124802486e-06,
     1.4433900094388104e-06, 1.3795513155951107e-06, 1.3210431870756455e-06,
     1.2672325828193411e-06, 1.2175818554470618e-06, 1.1716315965271029e-06,
     1.1289870257835749e-06, 1.0893071045865915e-06, 1.0522957624108658e-06,
     1.0176947758976976e-06},
    // x = 9.8000
    {2.8309210428842713e-01, 1.4440645545356885e-02, 2.2074737263591058e-03,
     5.6030188940629343e-04, 1.9727865440876926e-04, 8.7757973992180915e-05,
     4.6422760942949640e-05, 2.7961443511539203e-05, 1.8569900675556683e-05,
     1.3277383267973814e-05, 1.0041769523434978e-05, 7.9298755387733462e-06,
     6.4763029571229500e-06, 5.4314272701988285e-06, 4.6529049419995624e-06,
     4.0552369329495031e-06, 3.5847319127172335e-06, 3.2063547799740408e-06,
     2.8964702993324326e-06, 2.6386633491389110e-06, 2.4212383257264885e-06,
     2.2356720368678283e-06, 2.0756274567928387e-06, 1.9363079654847599e-06,
     1.8140242319187017e-06, 1.7058973434612578e-06, 1.6096512798135830e-06,
     1.5234652243848464e-06, 1.4458667300505215e-06, 1.3756532745255737e-06,
     1.3118338655526572e-06, 1.2535850187007601e-06, 1.2002171809167944e-06,
     1.1511488432354217e-06, 1.1058863808466852e-06, 1.0640082064410382e-06,
     1.0251522053640574e-06, 9.8900569180610895e-07, 9.5529731904488565e-07,
     9.2379051705504471e-07},
    // x = 9.9000
    {2.8165898453362015e-01, 1.4222667164220404e-02, 2.1524154954851026e-03,
     5.4100519168532029e-04, 1.8873038685560939e-04, 8.3252464628500450e-05,
     4.3717294386733816e-05, 2.6169199240977994e-05, 1.7291076088812860e-05,
     1.2311798558264808e-05, 9.2802773005482863e-06, 7.3086434977444008e-06,
     5.9557635551487693e-06, 4.9858286274012052e-06, 4.2647823678614803e-06,
     3.7123235662529200e-06, 3.2781489140234960e-06, 2.9295066720504808e-06,
     2.6443460336157822e-06, 2.4073798579600230e-06, 2.2077339598113789e-06,
     2.0374954694995541e-06, 1.8907890470861256e-06, 1.7631729829646775e-06,
     1.6512347547052767e-06, 1.5523141881002044e-06, 1.4643101786330241e-06,
     1.3855433036047872e-06, 1.3146565475802207e-06, 1.2505424826210006e-06,
     1.1922891120436312e-06, 1.1391390797215141e-06, 1.0904585841554749e-06,
     1.0457134299965016e-06, 1.0044503915954400e-06, 9.6628257393488695e-07,
     9.3087781278794227e-07, 8.9794940794675370e-07, 8.6724866362781887e-07,
     8.3855884056400895e-07},
    // x = 10.0000
    {2.8024739050664271e-01, 1.4010099528844012e-02, 2.0992449328384776e-03,
     5.2254123672149523e-04, 1.8061943636439910e-04, 7.9008749875855339e-05,
     4.1184815943596181e-05, 2.4500133875213286e-05, 1.6105103918285712e-05,
     1.1419341842418616e-05, 8.5783782621734554e-06, 6.7373006871578771e-06,
     5.4778993021073207e-06, 4.5773776395099048e-06, 3.9094633252141243e-06,
     3.3987253334362335e-06, 2.9980277787019223e-06, 2.6767493467339127e-06,
     2.4143148686601411e-06, 2.1964860188970244e-06, 2.0131512487249228e-06,
     1.8569635717618520e-06, 1.7224751911637053e-06, 1.6055726919940909e-06,
     1.5030993380618571e-06, 1.4125968901273355e-06, 1.3321255817004065e-06,
     1.2601363033818389e-06, 1.1953783461758322e-06, 1.1368317984767866e-06,
     1.0836573173822954e-06, 1.0351583298917465e-06, 9.9075225103466145e-07,
     9.4994832773840709e-07, 9.1233040979937466e-07, 8.7754342568358570e-07,
     8.4528267305245486e-07, 8.1528526851725314e-07, 7.8732326881538979e-07,
     7.6119809681502144e-07},
    // x = 10.1000
    {2.7885679043427003e-01, 1.3802757964309145e-02, 2.0478809078070368e-03,
     5.0486757345593496e-04, 1.7292046826565563e-04, 7.5010131641861368e-05,
     3.8813460041345258e-05, 2.2945318084761772e-05, 1.5004961190402269e-05,
     1.0594296287699889e-05, 7.9312908040097712e-06, 6.2117599831140849e-06,
     5.0391546725902471e-06, 4.2029362172997789e-06, 3.5841446852372924e-06,
     3.1119129032960780e-06, 2.7420665731127568e-06, 2.4459723607633572e-06,
     2.2044295743276120e-06, 2.0041751992485611e-06, 1.8358058190788464e-06,
     1.6924991760857524e-06, 1.5692034329894328e-06, 1.4621088742190115e-06,
     1.3682951417322938e-06, 1.2854904316625108e-06, 1.2119038014597355e-06,
     1.1461062501022466e-06, 1.0869449767486789e-06, 1.0334806163055783e-06,
     9.8494065033310265e-07, 9.4068437846628521e-07, 9.0017626822148871e-07,
     8.6296545589584013e-07, 8.2866981780788678e-07, 7.9696347541799119e-07,
     7.6756690739485880e-07, 7.4023906012496271e-07, 7.1477100416190704e-07,
     6.9098079679041836e-07},
    // x = 10.2000
    {2.7748667061208709e-01, 1.3600465700657007e-02, 1.9982464109454361e-03,
     4.8794420274720854e-04, 1.6560975983070258e-04, 7.1241054891774331e-05,
     3.6592219859087775e-05, 2.1496497033530130e-05, 1.3984173373471819e-05,
     9.8314033659261905e-06, 7.3346247680622960e-06, 5.7282745806461612e-06,
     4.6362743466046603e-06, 3.8596343127936091e-06, 3.2862650863382816e-06,
     2.8495769029256588e-06, 2.5081649660082723e-06, 2.2352512350071615e-06,
     2.0129154186825697e-06, 1.8288015591729525e-06, 1.6741638295891400e-06,
     1.5426665847562674e-06, 1.4296247284506137e-06, 1.3315095145172219e-06,
     1.2456190440285450e-06, 1.1698536506506329e-06, 1.1025596813262100e-06,
     1.0424188444197217e-06, 9.8836851759600835e-07, 9.3954347151199235e-07,
     8.9523265368044316e-07, 8.5484672501864376e-07, 8.1789338196309569e-07,
     7.8395838938598854e-07, 7.5269085317321905e-07, 7.2379167572674457e-07,
     6.9700442610154794e-07, 6.7210805986702314e-07, 6.4891106891662111e-07,
     6.2724674619867715e-07},
    // x = 10.3000
    {2.7613653463419668e-01, 1.3403053472767523e-02, 1.9502683166561575e-03,
     4.7173342175218781e-04, 1.5866509014949491e-04, 6.7687025056297821e-05,
     3.4510882545318300e-05, 2.0146037762301888e-05, 1.3036770448971328e-05,
     9.1258253614948374e-06, 6.7843488680914129e-06, 5.2834092739903198e-06,
     4.2662775784494280e-06, 3.5448468094911078e-06, 3.0134839160456679e-06,
     2.6096086592041533e-06, 2.2944064684276582e-06, 2.0428309840967531e-06,
     1.8381548183334157e-06, 1.6688656841076410e-06, 1.5268284706057728e-06,
     1.4061588402484387e-06, 1.3025114050953344e-06, 1.2126173807558956e-06,
     1.1339767820295205e-06, 1.0646488899868217e-06, 1.0031067089130405e-06,
     9.4813399935305197e-07, 8.9875120284947642e-07, 8.5416132896603263e-07,
     8.1370986520763100e-07, 7.7685468892944384e-07, 7.4314321440947286e-07,
     7.1219484227654510e-07, 6.8368734207811396e-07, 6.5734618532385349e-07,
     6.3293611515895001e-07, 6.1025442819831622e-07, 5.8912557908516994e-07,
     5.6939681572037060e-07},
    // x = 10.4000
    {2.7480590266177773e-01, 1.3210359143210063e-02, 1.9038771608952787e-03,
     4.5619967891673019e-04, 1.5206563795234173e-04, 6.4334531661666905e-05,
     3.2559969484131366e-05, 1.8886880782947314e-05, 1.2157246573836828e-05,
     8.4731109974433855e-06, 6.2767608626452310e-06, 4.8740141878435620e-06,
     3.9264347746153080e-06, 3.2561724210086055e-06, 2.7636621326359949e-06,
     2.3900826364442445e-06, 2.0990422462196131e-06, 1.8671111113866956e-06,
     1.6786733601024720e-06, 1.5230015055474919e-06, 1.3925276782667502e-06,
     1.2817861442564014e-06, 1.1867462112798760e-06, 1.1043796393841800e-06,
     1.0323730789736807e-06, 9.6893258948592936e-07, 9.1264803150856378e-07,
     8.6239724334375471e-07, 8.1727718151459233e-07, 7.7655367009265098e-07,
     7.3962420803188425e-07, 7.0599008084332115e-07, 6.7523519638100187e-07,
     6.4700984405583804e-07, 6.2101810304506553e-07, 5.9700798565894792e-07,
     5.7476365256641988e-07, 5.5409921293007586e-07, 5.3485374814191623e-07,
     5.1688728839057655e-07},
    // x = 10.5000
    {2.7349431072827185e-01, 1.3022227346615338e-02, 1.8590069328807751e-03,
     4.4130943881210137e-04, 1.4579188677785534e-04, 6.1170977697664323e-05,
     3.0730681205931445e-05, 1.7712495539398174e-05, 1.1340523035296446e-05,
     7.8691639166805968e-06, 5.8084602412944816e-06, 4.4972007484493837e-06,
     3.6142460887899289e-06, 2.9914144223810071e-06, 2.5348447645019073e-06,
     2.1892404200384860e-06, 1.9204763653069472e-06, 1.7066319383515243e-06,
     1.5331270710741040e-06, 1.3899643942854726e-06, 1.2701029536850474e-06,
     1.1684653214923692e-06, 1.0813123559249885e-06, 1.0058384127084473e-06,
     9.3990266893093488e-07, 8.8184673466044248e-07, 8.3036829133024437e-07,
     7.8443190908361579e-07, 7.4320503094532304e-07, 7.0601130543503647e-07,
     6.7229607956763222e-07, 6.4160054780372267e-07, 6.1354215056603892e-07,
     5.8779954462120141e-07, 5.6410095904158054e-07, 5.4221508686296813e-07,
     5.2194389607252817e-07, 5.0311690778801246e-07, 4.8558660639778665e-07,
     4.6922473061345119e-07},
    // x = 10.6000
    {2.7220131007911258e-01, 1.2838509154216086e-02, 1.8155948798545642e-03,
     4.2703105611043962e-04, 1.3982553693592329e-04, 5.8184614277915396e-05,
     2.9014846571960666e-05, 1.6616839419999318e-05, 1.0581914224928611e-05,
     7.3102137779378859e-06, 5.3763232098734277e-06, 4.1503197016905064e-06,
     3.3274218588386096e-06, 2.7485630537482087e-06, 2.3252449396084151e-06,
     2.0054761092990922e-06, 1.7572523423004100e-06, 1.5600621492646369e-06,
     1.4002908251301505e-06, 1.2686203206751086e-06, 1.1584991874917860e-06,
     1.0652102337575475e-06, 9.8528444905996118e-07, 9.1612219227335811e-07,
     8.5574213704454957e-07, 8.0261108413586358e-07, 7.5552620563327274e-07,
     7.1353203618209703e-07, 6.7586095559020005e-07, 6.4188984609140116e-07,
     6.1110807490044645e-07, 5.8309353006718010e-07, 5.5749446522303595e-07,
     5.3401558999972292e-07, 5.1240730181497801e-07, 4.9245726857219526e-07,
     4.7398378948688755e-07, 4.5683051420000693e-07, 4.4086220912720128e-07,
     4.2596133826846680e-07},
    // x = 10.7000
    {2.7092646654404517e-01, 1.2659061757295885e-02, 1.7735813240174971e-03,
     4.1333465804552683e-04, 1.3414942375726519e-04, 5.5364480182344598e-05,
     2.7404875892176555e-05, 1.5594320032013223e-05, 9.8770963816348700e-06,
     6.7927897464757304e-06, 4.9774797789638620e-06, 3.8309410021041600e-06,
     3.0638647259431477e-06, 2.5257794502507762e-06, 2.1332293104466680e-06,
     1.8373229948477243e-06, 1.6080408844423962e-06, 1.4261874426816267e-06,
     1.2790477841422811e-06, 1.1579359859837485e-06, 1.0567553990726198e-06,
     9.7112305835351101e-07, 8.9782026149479157e-07, 8.3443803056324695e-07,
     7.7914250108692559e-07, 7.3051610467512275e-07, 6.8744782360834639e-07,
     6.4905592233785978e-07, 6.1463260819487849e-07, 5.8360377354651036e-07,
     5.5549928626318496e-07, 5.2993077331038181e-07, 5.0657480398791858e-07,
     4.8516001616833122e-07, 4.6545715747969238e-07, 4.4727130620965540e-07,
     4.3043573961088110e-07, 4.1480705973749689e-07, 4.0026128818222307e-07,
     3.8669071387004617e-07},
    // x = 10.8000
    {2.6966935994020597e-01, 1.2483748168370128e-02, 1.7329094908194080e-03,
     4.0019203475397547e-04, 1.2874744166049333e-04, 5.2700345904317952e-05,
     2.5893717663718770e-05, 1.4639760473017230e-05, 9.2220788742632600e-06,
     6.3136961783010911e-06, 4.6092927766920726e-06, 3.5368354120074813e-06,
     2.8216532900463033e-06, 2.3213809648141498e-06, 1.9573047517967661e-06,
     1.6834414069877028e-06, 1.4716287132151300e-06, 1.3039001909688562e-06,
     1.1683797811452870e-06, 1.0569698375557135e-06, 9.6399630803245359e-07,
     8.8538635269252267e-07, 8.1815322937993813e-07, 7.6006444032061468e-07,
     7.0942246684708695e-07, 6.6491654927481166e-07, 6.2552039823346671e-07,
     5.9042026366675947e-07, 5.5896347641204387e-07, 5.3062105297748997e-07,
     5.0496012567130645e-07, 4.8162334512859067e-07, 4.6031330240411125e-07,
     4.4078061319884909e-07, 4.2281470709031212e-07, 4.0623663787313378e-07,
     3.9089342026946468e-07, 3.7665353094904919e-07, 3.6340330601880716e-07,
     3.5104403482761246e-07},
    // x = 10.9000
    {2.6842958350424229e-01, 1.2312436939002131e-02, 1.6935253478445238e-03,
     3.8757653693701096e-04, 1.2360447360382091e-04, 5.0182661854981996e-05,
     2.4474818642615658e-05, 1.3748367355881789e-05, 8.6131778138736821e-06,
     5.8699903137647704e-06, 4.2693386222821158e-06, 3.2659576638598083e-06,
     2.5990271684860092e-06, 2.1338277622187970e-06, 1.7961062194553645e-06,
     1.5426076315791267e-06, 1.3469083753748834e-06, 1.1921900179628682e-06,
     1.0673585611431171e-06, 9.6486388837223048e-07, 8.7942466287780988e-07,
     8.0725583404632028e-07, 7.4558563615646921e-07, 6.9234493721379618e-07,
     6.4596229602145223e-07, 6.0522561969133476e-07, 5.6918681690261781e-07,
     5.3709483027789785e-07, 5.0834778300477862e-07, 4.8245824016015473e-07,
     4.5902762264534594e-07, 4.3772710943053333e-07, 4.1828320634599675e-07,
     4.0046671636281522e-07, 3.8408422020769189e-07, 3.6897143112617477e-07,
     3.5498796395310805e-07, 3.4201318223838120e-07, 3.2994287486688216e-07,
     3.1868657655824254e-07},
    // x = 11.0000
    {2.6720674335186229e-01, 1.2145001893230544e-02, 1.6553774535864269e-03,
     3.7546298032463144e-04, 1.1870632552191698e-04, 4.7802510404863970e-05,
     2.3142086984693543e-05, 1.2915701364125930e-05, 8.0469918032565123e-06,
     5.4589618120506819e-06, 3.9553897108507849e-06, 3.0164310517100399e-06,
     2.3943733363220573e-06, 1.9617105735366258e-06, 1.6483856679656031e-06,
     1.4137037991253086e-06, 1.2328689537563155e-06, 1.0901352128960334e-06,
     9.7513780277798272e-07, 8.8083626875180236e-07, 8.0231425868519775e-07,
     7.3605380981125002e-07, 6.7948241052899966e-07, 6.3068216743451727e-07,
     5.8819823087165299e-07, 5.5090966011207938e-07, 5.1794053979409301e-07,
     4.8859762812914472e-07, 4.6232585258443757e-07, 4.3867603668484851e-07,
     4.1728115337092937e-07, 3.9783861660823512e-07, 3.8009691163966026e-07,
     3.6384538483328677e-07, 3.4890636334473024e-07, 3.3512901275185305e-07,
     3.2238450523344232e-07, 3.1056218599071777e-07, 2.9956650722990079e-07,
     2.8931455756622167e-07},
    // x = 11.1000
    {2.6600045796330180e-01, 1.1981321875652338e-02, 1.6184168154566292e-03,
     3.6382755646231047e-04, 1.1403966537911342e-04, 4.5551561468115570e-05,
     2.1889858213036769e-05, 1.2137650132865904e-05, 7.5203796474384442e-06,
     5.0781139723693021e-06, 3.6653982727550348e-06, 2.7865333291892204e-06,
     2.2062136374548181e-06, 1.8037395097529475e-06, 1.5130019343907460e-06,
     1.2957086611475969e-06, 1.1285875980054280e-06, 9.9689490605063318e-07,
     8.9094585098725714e-07, 8.0417444444475371e-07, 7.3200358168875151e-07,
     6.7116319952178759e-07, 6.1926548466584814e-07, 5.7453256712199502e-07,
     5.3561742499453442e-07, 5.0148333355304209e-07, 4.7132099961035218e-07,
     4.4449050267989365e-07, 4.2047990214140594e-07, 3.9887525235156290e-07,
     3.7933856166159825e-07, 3.6159137123885032e-07, 3.4540236793658293e-07,
     3.3057793225327302e-07, 3.1695484869882854e-07, 3.0439462794432632e-07,
     2.9277904343206179e-07, 2.8200659237323543e-07, 2.7198966703320729e-07,
     2.6265227665324296e-07},
    // x = 11.2000
    {2.6481035769328720e-01, 1.1821280513268818e-02, 1.5825967564169986e-03,
     3.5264774937586219e-04, 1.0959196649845328e-04, 4.3422031358053515e-05,
     2.0712863788969088e-05, 1.1410403267451658e-05, 7.0304398636649098e-06,
     4.7251465007420750e-06, 3.3974815825186831e-06, 2.5746838021076511e-06,
     2.0331933653033503e-06, 1.6587338422724486e-06, 1.3889115033783540e-06,
     1.1876891755487216e-06, 1.0332218025147044e-06, 9.1170193827251425e-07,
     8.1407909704718720e-07, 7.3422904129754827e-07, 6.6789002432693896e-07,
     6.1202209516622601e-07, 5.6440866189583191e-07, 5.2340150534069878e-07,
     4.8775333416659394e-07, 4.5650523698582679e-07, 4.2890942056231458e-07,
     4.0437514393400437e-07, 3.8243021654863611e-07, 3.6269313739245839e-07,
     3.4485263573544922e-07, 3.2865244259738204e-07, 3.1387981330150234e-07,
     3.0035677673735073e-07, 2.8793339177327255e-07, 2.7648249851228162e-07,
     2.6589559503083515e-07, 2.5607957015938421e-07, 2.4695409358359712e-07,
     2.3844951519000358e-07},
    // x = 11.3000
    {2.6363608430416080e-01, 1.1664765990261014e-02, 1.5478727896690823e-03,
     3.4190225770281519e-04, 1.0535145485212026e-04, 4.1406644664039554e-05,
     1.9606202081533044e-05, 1.0730429327351389e-05, 6.5744918428532105e-06,
     4.3979396932175381e-06, 3.1499084030683625e-06, 2.3794315133914799e-06,
     1.8740708206290149e-06, 1.5256126661034160e-06, 1.2751600762391158e-06,
     1.0887928296082347e-06, 9.4600236531270384e-07, 8.3385636254561751e-07,
     7.4389594811098991e-07, 6.7040822204948371e-07, 6.0942461938680701e-07,
     5.5811881119782227e-07, 5.1443294777513232e-07, 4.7683886672976758e-07,
     4.4418152542082791e-07, 4.1557391521383180e-07, 3.9032501832375818e-07,
     3.6788945616685726e-07, 3.4783167377827875e-07, 3.2980005060059069e-07,
     3.1350790812596284e-07, 2.9871938645555524e-07, 2.8523880906688267e-07,
     2.7290258086544385e-07, 2.6157294938921210e-07, 2.5113315248085997e-07,
     2.4148360904214924e-07, 2.3253890257915343e-07, 2.2422537308177115e-07,
     2.1647917988974247e-07},
    // x = 11.4000
    {2.6247729052091423e-01, 1.1511670834915421e-02, 1.5142025008729684e-03,
     3.3157092190887069e-04, 1.0130706002278526e-04, 3.9498598919406854e-05,
     1.8565311546968612e-05, 1.0094454617017593e-05, 6.1500585268716167e-06,
     4.0945399172906392e-06, 2.9210865607864555e-06, 2.1994444269265224e-06,
     1.7277077621367979e-06, 1.4033863688960081e-06, 1.1708748735790064e-06,
     9.9824063557895671e-07, 8.6622696755950449e-07, 7.6271952135406655e-07,
     6.7981133354392122e-07, 6.1217256572518349e-07, 5.5610724652154411e-07,
     5.0898738003475025e-07, 4.6890230258347797e-07, 4.3443503393270101e-07,
     4.0451586632657711e-07, 3.7832423716717342e-07, 3.5522154618134702e-07,
     3.3470425899212677e-07, 3.1637058780596692e-07, 2.9989643255916468e-07,
     2.8501774905262757e-07, 2.7151744954470624e-07, 2.5921554731250821e-07,
     2.4796165494395858e-07, 2.3762921222167569e-07, 2.2811100003091516e-07,
     2.1931562103525632e-07, 2.1116471460452473e-07, 2.0359073476966454e-07,
     1.9653516380146386e-07},
    // x = 11.5000
    {2.6133363960695355e-01, 1.1361891717971953e-02, 1.4815454374050969e-03,
     3.2163465623595019e-04, 9.7448369567522641e-05, 3.7691531848220557e-05,
     1.7585945944860672e-05, 9.4994436384590463e-06, 5.7548504773154338e-06,
     3.8131462832926810e-06, 2.7095515558230534e-06, 2.0334995249414535e-06,
     1.5930606728270751e-06, 1.2911488357411371e-06, 1.0752576072339109e-06,
     9.1532073961533608e-07, 7.9325431867150981e-07, 6.9770864858822143e-07,
     6.2129170008508758e-07, 5.5903040454424083e-07, 5.0748226863455300e-07,
     4.6420345284286752e-07, 4.2741977711358827e-07, 3.9581723354264496e-07,
     3.6840505990754855e-07, 3.4442410160171316e-07, 3.2328415578506307e-07,
     3.0452028947729260e-07, 2.8776184011393274e-07, 2.7271005599405590e-07,
     2.5912172630516212e-07, 2.4679703069496140e-07, 2.3557040587615865e-07,
     2.2530359927476395e-07, 2.1588032838167055e-07, 2.0720213303064082e-07,
     1.9918512376280822e-07, 1.9175741026324417e-07, 1.8485705091793439e-07,
     1.7843040530653183e-07},
    // x = 11.6000
    {2.6020480495948550e-01, 1.1215329261713331e-02, 1.4498630041984374e-03,
     3.1207538505413532e-04, 9.3765586536323262e-05, 3.5979490995287141e-05,
     1.6664151431547882e-05, 8.9425810721497814e-06, 5.3867512218103054e-06,
     3.5520984066606723e-06, 2.5139561202717727e-06, 1.8804737409249821e-06,
     1.4691727717684049e-06, 1.1880703257742471e-06, 9.8757806291625281e-07,
     8.3938259001395329e-07, 7.2649881699072774e-07, 6.3829194932958815e-07,
     5.6785045216758467e-07, 5.1053357732556152e-07, 4.6313455945901850e-07,
     4.2338056903328307e-07, 3.8962399707687513e-07, 3.6064621259533791e-07,
     3.3552949378160675e-07, 3.1357144220048652e-07, 2.9422654379211103e-07,
     2.7706547779026949e-07, 2.6174627337143544e-07, 2.4799352784154864e-07,
     2.3558320717257815e-07, 2.2433137505515856e-07, 2.1408572811323374e-07,
     2.0471916341002142e-07, 1.9612483673377501e-07, 1.8821232751650323e-07,
     1.8090463437172013e-07, 1.7413580055552095e-07, 1.6784902178518282e-07,
     1.6199512677635575e-07},
    // x = 11.7000
    {2.5909046972348559e-01, 1.1071887859159179e-02, 1.4191183657400333e-03,
     3.0287598331364999e-04, 9.0249489916016768e-05, 3.4356905559119377e-05,
     1.5796245384169045e-05, 8.4212551638222338e-06, 5.0438037733579543e-06,
     3.3098651703558914e-06, 2.3330606442736998e-06, 1.7393356567944593e-06,
     1.3551667070732999e-06, 1.0933909622254311e-06, 9.0716824014227926e-07,
     7.6983161552857687e-07, 6.6542568036874192e-07, 5.8398411501756843e-07,
     5.1904379764348403e-07, 4.6627356205348580e-07, 4.2268588715079340e-07,
     3.8616676121475094e-07, 3.5518596459303028e-07, 3.2861321563799797e-07,
     3.0559837496702931e-07, 2.8549150481313511e-07, 2.6778835832104117e-07,
     2.5209247137854129e-07, 2.3808832329327294e-07, 2.2552201995551803e-07,
     2.1418718019736106e-07, 2.0391447996930276e-07, 1.9456380672258557e-07,
     1.8601830240216894e-07, 1.7817979060632011e-07, 1.7096523038799002e-07,
     1.6430394003375250e-07, 1.5813540434644396e-07, 1.5240752842845513e-07,
     1.4707523624930800e-07},
    // x = 11.8000
    {2.5799032642326603e-01, 1.0931475502769107e-02, 1.3892763538301801e-03,
     2.9402022081507730e-04, 8.6891397787731525e-05, 3.2818560261631633e-05,
     1.4978796820460636e-05, 7.9330424046996418e-06, 4.7241982269244803e-06,
     3.0850344043491232e-06, 2.1657243969303592e-06, 1.6091378991720613e-06,
     1.2502378714356174e-06, 1.0064147826615936e-06, 8.3341700071127599e-07,
     7.0612436887924465e-07, 6.0954650509236098e-07, 5.3434223529538338e-07,
     4.7446696272294874e-07, 4.2587795363018064e-07, 3.8579162188559996e-07,
     3.5224146534884639e-07, 3.2380614808997829e-07, 2.9943723512593587e-07,
     2.7834712440008971e-07, 2.5993437205626062e-07, 2.4373284151662166e-07,
     2.2937638497051139e-07, 2.1657386687717137e-07, 2.0509120749668329e-07,
     1.9473827657743679e-07, 1.8535919305706617e-07, 1.7682505286093061e-07,
     1.6902841190184956e-07, 1.6187905433673369e-07, 1.5530071331177119e-07,
     1.4922850551096026e-07, 1.4360690623828035e-07, 1.3838813783025540e-07,
     1.3353087702766127e-07},
    // x = 11.9000
    {2.5690407661072145e-01, 1.0794003622097228e-02, 1.3603033807346348e-03,
     2.8549271003637793e-04, 8.3683133001985962e-05, 3.1359571101281259e-05,
     1.4208608290198085e-05, 7.4756934018989768e-06, 4.4262603454245870e-06,
     2.8763034060856517e-06, 2.0108974751364658e-06, 1.4890101752305192e-06,
     1.1536482866774141e-06, 9.2650430082167498e-07, 7.6576518129435914e-07,
     6.4776409454441004e-07, 5.5841521527299327e-07, 4.8896207128694602e-07,
     4.3375074317914400e-07, 3.8900725589280916e-07, 3.5213773833781763e-07,
     3.2131270859122353e-07, 2.9521183453962287e-07, 2.7286251037410351e-07,
     2.5353500757156536e-07, 2.3667271275745625e-07, 2.2184468669120100e-07,
     2.0871275576698509e-07, 1.9700826721868673e-07, 1.8651539596998007e-07,
     1.7705897289282990e-07, 1.6849548483542784e-07, 1.6070633349799484e-07,
     1.5359272562984393e-07, 1.4707175671512376e-07, 1.4107337840185837e-07,
     1.3553802769547731e-07, 1.3041475690716835e-07, 1.2565974624613613e-07,
     1.2123511149466948e-07},
    // x = 12.0000
    {2.5583143052938306e-01, 1.0659386929876237e-02, 1.3321673573864744e-03,
     2.7727885727412690e-04, 8.0616991190231660e-05, 2.9975362848281525e-05,
     1.3482699124073690e-05, 7.0471198441512422e-06, 4.1484410545391831e-06,
     2.6824702322432460e-06, 1.8676134191372276e-06, 1.3781528936897318e-06,
     1.0647210083973183e-06, 8.5307553569186341e-07, 7.0370112959800537e-07,
     5.9429668354224816e-07, 5.1162436818672851e-07, 4.4747465820140832e-07,
     3.9655836182171183e-07, 3.5535195975313074e-07, 3.2143808654349550e-07,
     2.9311454978979702e-07, 2.6915472031804268e-07, 2.4865625254098921e-07,
     2.3094297983742400e-07, 2.1549973577940571e-07, 2.0192809047589169e-07,
     1.8991568507891469e-07, 1.7921459691717611e-07, 1.6962581962294526e-07,
     1.6098795851773467e-07, 1.5316887984390101e-07, 1.4605946153489072e-07,
     1.3956886026832007e-07, 1.3362088686037953e-07, 1.2815120166825077e-07,
     1.2310512354656097e-07, 1.1843590273211673e-07, 1.1410334798252005e-07,
     1.1007272672191228e-07},
    // x = 12.1000
    {2.5477210679346429e-01, 1.0527543276042257e-02, 1.3048376163175670e-03,
     2.6936481687380934e-04, 7.7685710945248567e-05, 2.8661648151470533e-05,
     1.2798289934897754e-05, 6.6453824757033347e-06, 3.8893067724752013e-06,
     2.5024256979515817e-06, 1.7349824388194177e-06, 1.2758313212213896e-06,
     9.8283500605131481e-07, 7.8559346734019571e-07, 6.4675662712955061e-07,
     5.4530698120276181e-07, 4.6880178411716703e-07, 4.0954320802547085e-07,
     3.6258260492732598e-07, 3.2462988184549294e-07, 2.9343190703818640e-07,
     2.6740474987254076e-07, 2.4540871912682040e-07, 2.2660657516763635e-07,
     2.1037172691027718e-07, 1.9622732962618076e-07, 1.8380498220185916e-07,
     1.7281614938216421e-07, 1.6303202373425390e-07, 1.5426909550422115e-07,
     1.4637865260739404e-07, 1.3923903171078166e-07, 1.3274982463342521e-07,
     1.2682749419514278e-07, 1.2140201939769775e-07, 1.1641430152029046e-07,
     1.1181413910290737e-07, 1.0755863276290057e-07, 1.0361091799864817e-07,
     9.9939150588668768e-08},
    // x = 12.2000
    {2.5372583208112509e-01, 1.0398393509242539e-02, 1.2782848390213322e-03,
     2.6173744834014550e-04, 7.4882446015324045e-05, 2.7414408136508397e-05,
     1.2152788274363969e-05, 6.2686799983450891e-06, 3.6475305068879062e-06,
     2.3351460249993006e-06, 1.6121851995030846e-06, 1.1813702287890709e-06,
     9.0742047766546004e-07, 7.2356788256250167e-07, 5.9450316483918394e-07,
     5.0041541693544712e-07, 4.2960747204456747e-07, 3.7486028567046177e-07,
     3.3154321276043851e-07, 2.9658374036986873e-07, 2.6788156833251693e-07,
     2.4396265141482703e-07, 2.2376796736582425e-07, 2.0652061165371723e-07,
     1.9163988281201849e-07, 1.7868437092941069e-07, 1.6731341435608286e-07,
     1.5726046531807317e-07, 1.4831434366322504e-07, 1.4030581892181422e-07,
     1.3309785693752751e-07, 1.2657842893761373e-07, 1.2065513999828285e-07,
     1.1525116773676129e-07, 1.1030215701850423e-07, 1.0575382078546862e-07,
     1.0156006838757301e-07, 9.7681532179570942e-08, 9.4084397801488814e-08,
     9.0739468180461514e-08},
    // x = 12.3000
    {2.5269234084123032e-01, 1.0271861345397041e-02, 1.2524809874686197e-03,
     2.5438427613333389e-04, 7.2200739368709520e-05, 2.6229874384361884e-05,
     1.1543775356296645e-05, 5.9153388279989088e-06, 3.4218836567845679e-06,
     2.1796860854574972e-06, 1.4984671203499681e-06, 1.0941489863522806e-06,
     8.3795456191135026e-07, 6.6654957661384228e-07, 5.4654854087359778e-07,
     4.5927492773378495e-07, 3.9373082506764648e-07, 3.4314523431500385e-07,
     3.0318450154235362e-07, 2.7097894690991328e-07, 2.4457050676436497e-07,
     2.2258724854698178e-07, 2.0404500912345368e-07, 1.8822280274277349e-07,
     1.7458240917996292e-07, 1.6271518645264503e-07, 1.5230609943095774e-07,
     1.4310889458363921e-07, 1.3492864792751957e-07, 1.2760928734899316e-07,
     1.2102453213444635e-07, 1.1507121939503949e-07, 1.0966432352862126e-07,
     1.0473319375110211e-07, 1.0021867960327156e-07, 9.6070911770020376e-08,
     9.2247572056426082e-08, 8.8712532399833847e-08, 8.5434775077404890e-08,
     8.2387529181987763e-08},
    // x = 12.4000
    {2.5167137501291509e-01, 1.0147873242911595e-02, 1.2273992395172278e-03,
     2.4729345197091143e-04, 6.9634498995517925e-05, 2.5104512187585705e-05,
     1.0968993764351089e-05, 5.5838036382672775e-06, 3.2112284623577994e-06,
     2.0351731916349552e-06, 1.3931331424809855e-06, 1.0135970679260077e-06,
     7.7395741349848589e-07, 6.1412688023897072e-07, 5.0253375237566493e-07,
     4.2156814965155531e-07, 3.6088806176058514e-07, 3.1414182784530658e-07,
     2.7727319625201633e-07, 2.4760159491084204e-07, 2.2330135056399672e-07,
     2.0309543006403972e-07, 1.8606914456523932e-07, 1.7155333862500400e-07,
     1.5904912128384884e-07, 1.4817815465213675e-07, 1.3864907982754728e-07,
     1.3023437594049542e-07, 1.2275411166094933e-07, 1.1606434101364186e-07,
     1.1004868597859673e-07, 1.0461214262736874e-07, 9.9676462822114423e-08,
     9.5176668383135481e-08, 9.1058390086061187e-08, 8.7275814854942389e-08,
     8.3790086579238510e-08, 8.0568048901163513e-08, 7.7581248389174990e-08,
     7.4805137839952937e-08},
    // x = 12.5000
    {2.5066268375731304e-01, 1.0026358284165638e-02, 1.2030139279729933e-03,
     2.4045371946771558e-04, 6.7177975324077209e-05, 2.4035004989784647e-05,
     1.0426336068622097e-05, 5.2726286288003456e-06, 3.0145110503970587e-06,
     1.9008013873868538e-06, 1.2955429275308615e-06, 9.3918993224277780e-07,
     7.1498861078020809e-07, 5.6592248389706124e-07, 4.6213015572567981e-07,
     3.8700485375864192e-07, 3.3081989177756811e-07, 2.8761613026324378e-07,
     2.5359645548539414e-07, 2.2625662723523638e-07, 2.0389421160382174e-07,
     1.8532038014712144e-07, 1.6968492696990137e-07, 1.5636674166267527e-07,
     1.4490334744268149e-07, 1.3494443410451244e-07, 1.2622051869005260e-07,
     1.1852137273976556e-07, 1.1168089314433976e-07, 1.0556630948594302e-07,
     1.0007036350367737e-07, 9.5105560065826739e-08, 9.0599884482731918e-08,
     8.6493572771955377e-08, 8.2736648145688412e-08, 7.9287021998955093e-08,
     7.6109015593882948e-08, 7.3172198650993913e-08, 7.0450469069829918e-08,
     6.7921317851930361e-08},
    // x = 12.6000
    {2.4966602320083608e-01, 9.9072480629207142e-03, 1.1793004830765081e-03,
     2.3385438095826995e-04, 6.4825740137847227e-05, 2.3018239920892295e-05,
     9.9138342815744459e-06, 4.9804694613622498e-06, 2.8307550272339099e-06,
     1.7758261995570354e-06, 1.2051064506922417e-06, 8.7044524723801168e-07,
     6.6064386715615345e-07, 5.2159053352240807e-07, 4.2503687186372223e-07,
     3.5531960515510971e-07, 3.0328938593925515e-07, 2.6335454372445368e-07,
     2.3196007127844055e-07, 2.0676616679218744e-07, 1.8618512979190914e-07,
     1.6911012251305853e-07, 1.5475079499691673e-07, 1.4253057701278291e-07,
     1.3202070974054009e-07, 1.2289680726775300e-07, 1.1490960065540440e-07,
     1.0786482542052750e-07, 1.0160913349166323e-07, 9.6020054558950364e-08,
     9.0998729557101420e-08, 8.6464574160476285e-08, 8.2351307062328611e-08,
     7.8603957337785949e-08, 7.5176583630652182e-08, 7.2030517316502900e-08,
     6.9132995846527000e-08, 6.6456089787990980e-08, 6.3975853173019317e-08,
     6.1671645245369835e-08},
    // x = 12.7000
    {2.4868115618943198e-01, 9.7904765773178717e-03, 1.1562353782045502e-03,
     2.2748526635687854e-04, 6.2572666887406026e-05, 2.2051294347583372e-05,
     9.4296500891882160e-06, 4.7060758110791503e-06, 2.6590555751240480e-06,
     1.6595598117745038e-06, 1.1212799553416990e-06, 8.0691942929682154e-07,
     6.1055202030671131e-07, 4.8081397439493588e-07, 3.9097841537900965e-07,
     3.2626962551003399e-07, 2.7808003278640267e-07, 2.4116202850058548e-07,
     2.1218682832614536e-07, 1.8896799567050976e-07, 1.7002465642179021e-07,
     1.5432619508885748e-07, 1.4113782798363915e-07, 1.2992427957588076e-07,
     1.2028801504054924e-07, 1.1192862909254176e-07, 1.0461553250721210e-07,
     9.8169199403378376e-08, 9.2448047604310170e-08, 8.7339100606659612e-08,
     8.2751235344746317e-08, 7.8610228267444678e-08, 7.4855071764272876e-08,
     7.1435201048869995e-08, 6.8308382371568434e-08, 6.5439087622118587e-08,
     6.2797230832046066e-08, 6.0357176878070344e-08, 5.8096957000740893e-08,
     5.5997642953567604e-08},
    // x = 12.8000
    {2.4770785205327503e-01, 9.6759801281524581e-03, 1.1337960785892712e-03,
     2.2133670392087185e-04, 6.0413912299768179e-05, 2.1131423364292047e-05,
     8.9720657982490343e-06, 4.4482844845781350e-06, 2.4985740115872979e-06,
     1.5513666259744865e-06, 1.0435622391202365e-06, 7.4820447068311588e-07,
     5.6437227553415852e-07, 4.4330212173112382e-07, 3.5970252791809132e-07,
     2.9963284131200996e-07, 2.5499396518105882e-07, 2.2086047964600550e-07,
     1.9411500842081994e-07, 1.7271416951301342e-07, 1.5527656402227787e-07,
     1.4084244347172509e-07, 1.2872861317371146e-07, 1.1843808674921175e-07,
     1.0960224629592855e-07, 1.0194287095559973e-07, 9.5246634636654635e-08,
     8.9347619675995232e-08, 8.4115098052445587e-08, 7.9444844412190979e-08,
     7.5252861261018858e-08, 7.1470779878319907e-08, 6.8042443761595725e-08,
     6.4921338768224537e-08, 6.2068637712256603e-08, 5.9451696488614381e-08,
     5.7042885884937820e-08, 5.4818675686066791e-08, 5.2758910328819713e-08,
     5.0846231378199776e-08},
    // x = 12.9000
    {2.4674588638137812e-01, 9.5636972221338091e-03, 1.1119609928711457e-03,
     2.1539949279185511e-04, 5.8344899194461982e-05, 2.0256048155980275e-05,
     8.5394759453455950e-06, 4.2060130606056645e-06, 2.3485327745433455e-06,
     1.4506591798980711e-06, 9.7149124388359368e-07, 6.9392503084065213e-07,
     5.2179168152978197e-07, 4.0878843846425992e-07, 3.3097819816544167e-07,
     2.7520610158647908e-07, 2.3385034198892371e-07, 2.0228724650262904e-07,
     1.7759702719865836e-07, 1.5786975505750899e-07, 1.4181667137117069e-07,
     1.2854392249423904e-07, 1.1741621478238879e-07, 1.0797206741631313e-07,
     9.9869645065894362e-08, 9.2851251254350534e-08, 8.6719515042836136e-08,
     8.1321084162932999e-08, 7.6535244305996491e-08, 7.2265837192832554e-08,
     6.8435429012034364e-08, 6.4981040459979724e-08, 6.1850977640000782e-08,
     5.9002450416024407e-08, 5.6399761705696390e-08, 5.4012915962265796e-08,
     5.1816539048612765e-08, 4.9789031964425126e-08, 4.7911901994776018e-08,
     4.6169229757021328e-08},
    // x = 13.0000
    {2.4579504080564116e-01, 9.4535684798551598e-03, 1.0907094273137886e-03,
     2.0966487719853698e-04, 5.6361300422414530e-05, 1.9422745169028836e-05,
     8.1303795173975416e-06, 3.9782540122764234e-06, 2.2082107991217417e-06,
     1.3568943914649448e-06, 9.0464092426357293e-07, 6.4373576932899916e-07,
     4.8252281875330464e-07, 3.7702850237890680e-07, 3.0459385220190057e-07,
     2.5280355026438699e-07, 2.1448387120057487e-07, 1.8529378240915087e-07,
     1.6249819143612372e-07, 1.4431167985213587e-07, 1.2953177335585529e-07,
     1.1732589617726986e-07, 1.0710323571698234e-07, 9.8435238472429381e-08,
     9.1004876970119275e-08, 8.4573444790571883e-08, 7.8958318359155019e-08,
     7.4017748694391556e-08, 6.9640260431172881e-08, 6.5737132215221084e-08,
     6.2236976681422610e-08, 5.9081775791759584e-08, 5.6223941073065782e-08,
     5.3624106260316993e-08, 5.1249450479235620e-08, 4.9072410618701865e-08,
     4.7069682574874123e-08, 4.5221439268647242e-08, 4.3510713006439994e-08,
     4.1922903635186966e-08},
    // x = 13.1000
    {2.4485510279387740e-01, 9.3455365482157590e-03, 1.0700215425199526e-03,
     2.0412452221279544e-04, 5.4459023849810854e-05, 1.8629236031441761e-05,
     7.7433727374555690e-06, 3.7640692733740048e-06, 2.0769392548124650e-06,
     1.2695701033316189e-06, 8.4261837170905608e-07, 5.9731890005216350e-07,
     4.4630168231588858e-07, 3.4779814631197386e-07, 2.8035569946182467e-07,
     2.3225513968967279e-07, 1.9674346205558692e-07, 1.6974441310346820e-07,
     1.4869556618694243e-07, 1.3192768413715235e-07, 1.1831866629104795e-07,
     1.0709292722936909e-07, 9.7700963600738736e-08, 8.9744760973655628e-08,
     8.2930272566347943e-08, 7.7036363787313441e-08, 7.1894043077461327e-08,
     6.7372277045113003e-08, 6.3368114998389098e-08, 5.9799691999413086e-08,
     5.6601190971066608e-08, 5.3719161248532842e-08, 5.1109791379161509e-08,
     4.8736863172595061e-08, 4.6570198780166138e-08, 4.4584469133892236e-08,
     4.2758270381024263e-08, 4.1073401270660642e-08, 3.9514292777711082e-08,
     3.8067554174055494e-08},
    // x = 13.2000
    {2.4392586545138553e-01, 9.2395460170525742e-03, 1.0498783124984902e-03,
     1.9877049095813897e-04, 5.2634198314749650e-05, 1.7873378167998672e-05,
     7.3771423731213429e-06, 3.5625852141286193e-06, 1.9540976141798252e-06,
     1.1882219031619360e-06, 7.8506117282177568e-07, 5.5438194816952209e-07,
     4.1288574281504162e-07, 3.2089175654523244e-07, 2.5808622080073424e-07,
     2.1340527294088576e-07, 1.8049099483278586e-07, 1.5551521332954648e-07,
     1.3607694200576576e-07, 1.2061536578149393e-07, 1.0808326014758932e-07,
     9.7758048048077613e-08, 8.9128593503235518e-08, 8.1825208714534422e-08,
     7.5575136818227266e-08, 7.0173504034519032e-08, 6.5463920764337462e-08,
     6.1325250110909444e-08, 5.7662407519626501e-08, 5.4399849660484064e-08,
     5.1476891378283816e-08, 4.8844286988386476e-08, 4.6461700101757866e-08,
     4.4295807160316427e-08, 4.2318859172700378e-08, 4.0507578989941672e-08,
     3.8842307223633129e-08, 3.7306334459976534e-08, 3.5885374504405275e-08,
     3.4567145426414376e-08},
    // x = 13.3000
    {2.4300712733065752e-01, 9.1355433397536882e-03, 1.0302614859417905e-03,
     1.9359522317667364e-04, 5.0883160489747401e-05, 1.7153156060086176e-05,
     7.0304595282523934e-06, 3.3729879946558970e-06, 1.8391100267069240e-06,
     1.1124201971647907e-06, 7.3163498258258509e-07, 5.1465569266165494e-07,
     3.8205216999187218e-07, 2.9612071580310283e-07, 2.3762278636276322e-07,
     1.9611156372503271e-07, 1.6560019797149465e-07, 1.4249298209116718e-07,
     1.2453989337430776e-07, 1.1028130997801249e-07, 9.8739769913842716e-08,
     8.9242005903507060e-08, 8.1312520466787033e-08, 7.4607902690644927e-08,
     6.8875120940827736e-08, 6.3924350250614149e-08, 5.9610851629586943e-08,
     5.5822628832100137e-08, 5.2471856253054437e-08, 4.9488819435705456e-08,
     4.6817561551592274e-08, 4.4412708466648524e-08, 4.2237121201673832e-08,
     4.0260137920094858e-08, 3.8456241774888965e-08, 3.6804040339589620e-08,
     3.5285475739720388e-08, 3.3885207502457534e-08, 3.2590126062029814e-08,
     3.1388966065489279e-08},
    // x = 13.4000
    {2.4209869224883476e-01, 9.0334767576389034e-03, 1.0111535495822602e-03,
     1.8859151506713274e-04, 4.9202442587977085e-05, 1.6466673103718302e-05,
     6.7021738816700763e-06, 3.1945192667749164e-06, 1.7314419734880779e-06,
     1.0417675163117194e-06, 6.8203129469326185e-07, 4.7789227897071813e-07,
     3.5359620537997267e-07, 2.7331197844612207e-07, 2.1881639201127016e-07,
     1.8024370358893980e-07, 1.5195562310126445e-07, 1.3057430784322685e-07,
     1.1399092023767517e-07, 1.0084029614368376e-07, 9.0209979009716912e-08,
     8.1472575643849011e-08, 7.4185695542621038e-08, 6.8030305495324991e-08,
     6.2771650975262157e-08, 5.8233835285247485e-08, 5.4282891321057203e-08,
     5.0815265965402225e-08, 4.7749832684846937e-08, 4.5022251898991631e-08,
     4.2580923503629162e-08, 4.0384038118587437e-08, 3.8397398855512022e-08,
     3.6592791547203174e-08, 3.4946750802960640e-08, 3.3439615420188978e-08,
     3.2054797861572906e-08, 3.0778213871329231e-08, 2.9597833142030949e-08,
     2.8503322380341709e-08},
    // x = 13.5000
    {2.4120036911254680e-01, 8.9332962279059409e-03, 9.9253769350486827e-04,
     1.8375250031251695e-04, 4.7588760855601287e-05, 1.5812144022741755e-05,
     6.3912083393990819e-06, 3.0264721972519995e-06, 1.6305971804591080e-06,
     9.7589603634891645e-07, 6.3596539274982687e-07, 4.4386348745786229e-07,
     3.2732967130173159e-07, 2.5230676652441518e-07, 2.0153050406574493e-07,
     1.6568242709342684e-07, 1.3945170938933891e-07, 1.1966471568385534e-07,
     1.0434466527966138e-07, 9.2214575146791977e-08, 8.2422568308918677e-08,
     7.4383933862280819e-08, 6.7687039618295814e-08, 6.2035470238489548e-08,
     5.7211407956478633e-08, 5.3051848277163112e-08, 4.9432784287081840e-08,
     4.6258462253008786e-08, 4.3453938427090178e-08, 4.0959829776296405e-08,
     3.8728550756201201e-08, 3.6721574434970423e-08, 3.4907411222925795e-08,
     3.3260097892818800e-08, 3.1758054534620491e-08, 3.0383210240915992e-08,
     2.9120327434108786e-08, 2.7956474677995626e-08, 2.6880611646873228e-08,
     2.5883259645376572e-08},
    // x = 13.6000
    {2.4031197174978466e-01, 8.8349533549523795e-03, 9.7439777830063195e-04,
     1.7907163222144129e-04, 4.6039004796696038e-05, 1.5187887797437781e-05,
     6.0965540695536346e-06, 2.8681877876559027e-06, 1.5361147696647727e-06,
     9.1446529427736865e-07, 5.9317446732769451e-07, 4.1235914462959100e-07,
     3.0307960465161295e-07, 2.3295937633579465e-07, 1.8564000298197598e-07,
     1.5231856641619813e-07, 1.2799192937299382e-07, 1.0967788931441460e-07,
     9.5523200222345380e-08, 8.4333210598164052e-08, 7.5312504903370811e-08,
     6.7916089010349104e-08, 6.1760909834128513e-08, 5.6571539065406978e-08,
     5.2145855004316380e-08, 4.8332787325544707e-08, 4.5017539472280950e-08,
     4.2111562943903612e-08, 3.9545620660220102e-08, 3.7264900649845680e-08,
     3.5225516852360399e-08, 3.3391965001369987e-08, 3.1735246879762120e-08,
     3.0231469383375802e-08, 2.8860785615052703e-08, 2.7606585569188416e-08,
     2.6454871156456435e-08, 2.5393768914140867e-08, 2.4413146639846331e-08,
     2.3504309239391438e-08},
    // x = 13.7000
    {2.3943331874848084e-01, 8.7384013248947290e-03, 9.5671830395324653e-04,
     1.7454266691244519e-04, 4.4550227081090577e-05, 1.4592321071700028e-05,
     5.8172658913673703e-06, 2.7190514679759664e-06, 1.4475666297228161e-06,
     8.5716008540341328e-07, 5.5341588530768270e-07, 3.8318566519076625e-07,
     2.8068700489610526e-07, 2.1513608602803962e-07, 1.7103021742783670e-07,
     1.4005218759755184e-07, 1.1748800913466313e-07, 1.0053496117553071e-07,
     8.7455375033183596e-08, 7.7131478503410545e-08, 6.8820485270024352e-08,
     6.2014362439294095e-08, 5.6356613856032975e-08, 5.1591286798800615e-08,
     4.7530807091579805e-08, 4.4035152636974420e-08, 4.0998044498223497e-08,
     3.8337590991660816e-08, 3.5989822602446271e-08, 3.3904143179016985e-08,
     3.2040075997360571e-08, 3.0364900387031637e-08, 2.8851910917833540e-08,
     2.7479118409665614e-08, 2.6228268912891659e-08, 2.5084094516613125e-08,
     2.4033735235225371e-08, 2.3066288574348364e-08, 2.2172455395685186e-08,
     2.1344259132607156e-08},
    // x = 13.8000
    {2.3856423330148460e-01, 8.6435948431164335e-03, 9.3948438035790943e-04,
     1.7015964747530958e-04, 4.3119634088991469e-05, 1.4023952004707271e-05,
     5.5524579920571551e-06, 2.5784899429616133e-06, 1.3645549881674531e-06,
     8.0368852637116710e-07, 5.1646559891484364e-07, 3.5616471399308798e-07,
     2.6000568662458444e-07, 1.9871415560188861e-07, 1.5759604095103241e-07,
     1.2879180132526994e-07, 1.0785921630727850e-07, 9.2163864751365605e-08,
     8.0076224467221283e-08, 7.0550320082778838e-08, 6.2892427979198892e-08,
     5.6628915802342734e-08, 5.1427967699211916e-08, 4.7051705632682669e-08,
     4.3326039628028552e-08, 4.0121176477208002e-08, 3.7338714830981689e-08,
     3.4902913588389061e-08, 3.2754665810105904e-08, 3.0847263774402505e-08,
     2.9143372887219533e-08, 2.7612836054996199e-08, 2.6231057987761460e-08,
     2.4977800659492937e-08, 2.3836274390707063e-08, 2.2792444273779186e-08,
     2.1834495378108994e-08, 2.0952416362300147e-08, 2.0137672324999765e-08,
     1.9382945580524848e-08},
    // x = 13.9000
    {2.3770454305763764e-01, 8.5504900746863205e-03, 9.2268169937773613e-04,
     1.6591688904786991e-04, 4.1744577049539254e-05, 1.3481374535537961e-05,
     5.3012999472309210e-06, 2.4459682718023030e-06, 1.2867101697531144e-06,
     7.5378027078794932e-07, 4.8211668298824018e-07, 3.3113197787248512e-07,
     2.4090122781184190e-07, 1.8358091141719724e-07, 1.4524112411389835e-07,
     1.1845364177716160e-07, 9.9031709971009123e-08, 8.4498743566379355e-08,
     7.3326426867817099e-08, 6.4535843029182564e-08, 5.7479011519430397e-08,
     5.1714320661823302e-08, 4.6932893185627598e-08, 4.2913627174650544e-08,
     3.9494932349302109e-08, 3.6556486590571510e-08, 3.4007174756156884e-08,
     3.1776938999221985e-08, 2.9811161405009462e-08, 2.8066720941998714e-08,
     2.6509179053236100e-08, 2.5110739724800877e-08, 2.3848749811671512e-08,
     2.2704582009377935e-08, 2.1662792688141915e-08, 2.0710479769203595e-08,
     1.9836787975377003e-08, 1.9032523895848895e-08, 1.8289853751463725e-08,
     1.7602064063480793e-08},
    // x = 14.0000
    {2.3685407997867067e-01, 8.4590445874982694e-03, 9.0629650834913244e-04,
     1.6180896475094856e-04, 4.0422543733483429e-05, 1.2963263031508832e-05,
     5.0630130224104845e-06, 2.3209871632940267e-06, 1.2136885260823869e-06,
     7.0718486515346472e-07, 4.5017798995758063e-07, 3.0793603821448671e-07,
     2.2325000570820092e-07, 1.6963290798576630e-07, 1.3387713558971875e-07,
     1.0896100760708116e-07, 9.0937947025569618e-08, 7.7479411883579544e-08,
     6.7151810600775459e-08, 5.9038866897326038e-08, 5.2535253210433581e-08,
     4.7229166518721897e-08, 4.2833051471481213e-08, 3.9141378468324240e-08,
     3.6004145318131168e-08, 3.3309800053031556e-08, 3.0973967271465149e-08,
     2.8931840938717970e-08, 2.7132947590211904e-08, 2.5537474769231569e-08,
     2.4113653295753765e-08, 2.2835861854907881e-08, 2.1683234919843534e-08,
     2.0638626810223629e-08, 1.9687831327907186e-08, 1.8818987232930117e-08,
     1.8022120515516533e-08, 1.7288788518899221e-08, 1.6611800707637839e-08,
     1.5984997692305521e-08},
    // x = 14.1000
    {2.3601268020165578e-01, 8.3692172979913677e-03, 8.9031558495301011e-04,
     1.5783069242786648e-04, 3.9151150662973368e-05, 1.2468367293175314e-05,
     4.8368667349543275e-06, 2.2030804700421935e-06, 1.1451705230998751e-06,
     6.6367023381139902e-07, 4.2047291287943834e-07, 2.8643733585999238e-07,
     2.0693831296324120e-07, 1.5677516045619172e-07, 1.2342308628016859e-07,
     1.0024365967760555e-07, 8.3516140098912891e-08, 7.1050862554883126e-08,
     6.1502903907960593e-08, 5.4014508701352002e-08, 4.8020125536748406e-08,
     4.3135703822349723e-08, 3.9093509402291308e-08, 3.5702468932152136e-08,
     3.2823324134564385e-08, 3.0352644800611968e-08, 2.8212290270027438e-08,
     2.6342307982093959e-08, 2.4696051056693528e-08, 2.3236759255854068e-08,
     2.1935124002800785e-08, 2.0767527126043501e-08, 1.9714748571783785e-08,
     1.8761005601046688e-08, 1.7893229647293814e-08, 1.7100515831448808e-08,
     1.6373699461575429e-08, 1.5705027002787595e-08, 1.5087898084844875e-08,
     1.4516661465128777e-08},
    // x = 14.2000
    {2.3518018390676340e-01, 8.2809684193179221e-03, 8.7472621337392144e-04,
     1.5397712213856371e-04, 3.7928135804068606e-05, 1.1995507890923231e-05,
     4.6221756572449963e-06, 2.0918128665418066e-06, 1.0808589740749812e-06,
     6.2302128256609342e-07, 3.9283824768866675e-07, 2.6650722067128065e-07,
     1.9186154721978244e-07, 1.4492044176397625e-07, 1.1380471102921572e-07,
     9.2237270614423281e-08, 7.6709762487658189e-08, 6.5162817876587406e-08,
     5.6334524340948120e-08, 4.9421805148501867e-08, 4.3896206563166727e-08,
     3.9399518827190341e-08, 3.5682435745476944e-08, 3.2567305427775803e-08,
     2.9924831715064004e-08, 2.7659106325370790e-08, 2.5697756626628629e-08,
     2.3985315732876597e-08, 2.2478670102485947e-08, 2.1143875402959517e-08,
     1.9953891351302117e-08, 1.8886945001120040e-08, 1.7925331009609708e-08,
     1.7054520465739586e-08, 1.6262490732637390e-08, 1.5539215709660763e-08,
     1.4876273978460442e-08, 1.4266544578521202e-08, 1.3703968626459236e-08,
     1.3183360909850076e-08},
    // x = 14.3000
    {2.3435643519008545e-01, 8.1942594118342427e-03, 8.5951616167400670e-04,
     1.5024352436167492e-04, 3.6751351709985735e-05, 1.1543571809920964e-05,
     4.4182964434614877e-06, 1.9867776971631916e-06, 1.0204774066893645e-06,
     5.8503861145409594e-07, 3.6712314655579607e-07, 2.4802707872516308e-07,
     1.7792346799114329e-07, 1.3398863893868991e-07, 1.0495390297946060e-07,
     8.4882921687842804e-08, 7.0467096020666898e-08, 5.9769319669578899e-08,
     5.1605404276335858e-08, 4.5223368254978264e-08, 4.0129361392031307e-08,
     3.5989237445497758e-08, 3.2570824616235922e-08, 2.9708932923026702e-08,
     2.7283504239517173e-08, 2.5205597258892310e-08, 2.3408176016483701e-08,
     2.1839919676279049e-08, 2.0460977484730955e-08, 1.9240003164251487e-08,
     1.8152047571986067e-08, 1.7177037613954422e-08, 1.6298662361785090e-08,
     1.5503546412027254e-08, 1.4780628792789109e-08, 1.4120690929034928e-08,
     1.3515994049979495e-08, 1.2959997885191107e-08, 1.2447140393610849e-08,
     1.1972663782377436e-08},
    // x = 14.4000
    {2.3354128194129781e-01, 8.1090529357961284e-03, 8.4467366031316373e-04,
     1.4662537886099133e-04, 3.5618759085335750e-05, 1.1111508381901120e-05,
     4.2246250635987101e-06, 1.8875949811636731e-06, 9.6376855375644568e-07,
     5.4953732793715680e-07, 3.4318815491446240e-07, 2.3088753069216151e-07,
     1.6503551516146723e-07, 1.2390616353358397e-07, 9.6808196046434477e-08,
     7.8126642919345044e-08, 6.4740818098272126e-08, 5.4828355137969415e-08,
     4.7277849324981663e-08, 4.1385071380377108e-08, 3.6688451894626543e-08,
     3.2876255500355287e-08, 2.9732243654368583e-08, 2.7102798443650145e-08,
     2.4876429082711574e-08, 2.2970647770257736e-08, 2.1323356493530139e-08,
     1.9887066836376326e-08, 1.8624941205946152e-08, 1.7508030537133688e-08,
     1.6513313625744446e-08, 1.5622283399338954e-08, 1.4819912669753143e-08,
     1.4093887300850607e-08, 1.3434030551650078e-08, 1.2831865930361346e-08,
     1.2280281659242725e-08, 1.1773270550530539e-08, 1.1305726459039140e-08,
     1.0873283613769641e-08},
    // x = 14.5000
    {2.3273457572594794e-01, 8.0253128061477715e-03, 8.3018738175106026e-04,
     1.4311836417561151e-04, 3.4528420743679750e-05, 1.0698325483812062e-05,
     4.0405942296332690e-06, 1.7939095628505039e-06, 9.1049295793757471e-07,
     5.1634595249554770e-07, 3.2090432533956967e-07, 2.1498769550217497e-07,
     1.5311618393041845e-07, 1.1460541157560597e-07, 8.9310291378395605e-08,
     7.1918992669158193e-08, 5.9487624488828169e-08, 5.0301515364256570e-08,
     4.3317426730382442e-08, 3.7875762981250358e-08, 3.3545072196581931e-08,
     3.0034493016964446e-08, 2.7142604729709801e-08, 2.4726536216174287e-08,
     2.2682742744562093e-08, 2.0934714893643718e-08, 1.9424924034756100e-08,
     1.8109424526696028e-08, 1.6954161600013903e-08, 1.5932398228720291e-08,
     1.5022890790572950e-08, 1.4208575022657452e-08, 1.3475605650328371e-08,
     1.2812644989774241e-08, 1.2210329370585241e-08, 1.1660864275948171e-08,
     1.1157713828428306e-08, 1.0695360238185858e-08, 1.0269115699863483e-08,
     9.8749740110897531e-09},
    // x = 14.6000
    {2.3193617167216368e-01, 7.9430039492988675e-03, 8.1604642107054146e-04,
     1.3971834769575063e-04, 3.3478495932652875e-05, 1.0303085984831693e-05,
     3.8656709998752811e-06, 1.7053893959448036e-06, 8.6042768158841281e-07,
     4.8530540925385669e-07, 3.0015240202167401e-07, 2.0023451389262857e-07,
     1.4209045146370061e-07, 1.0602426882883964e-07, 8.2407624026995580e-08,
     6.6214673287413307e-08, 5.4667884764363583e-08, 4.6153683576493091e-08,
     3.9692681109139022e-08, 3.4667005624922731e-08, 3.0673307622657110e-08,
     2.7440170402004835e-08, 2.4779955153966072e-08, 2.2559772093769695e-08,
     2.0683446973177052e-08, 1.9080009071756610e-08, 1.7696158420177009e-08,
     1.6491224639691323e-08, 1.5433723232624211e-08, 1.4498958474971660e-08,
     1.3667325795648339e-08, 1.2923090299456886e-08, 1.2253494934087965e-08,
     1.1648100476894187e-08, 1.1098290930188396e-08, 1.0596898540842555e-08,
     1.0137916424979577e-08, 9.7162761038739311e-09, 9.3276736643879467e-09,
     8.9684327180641740e-09},
    // x = 14.7000
    {2.3114592836158837e-01, 7.8620923617907061e-03, 8.0224027756566432e-04,
     1.3642137628866444e-04, 3.2467235002689237e-05, 9.9249044245787696e-06,
     3.6993545485979286e-06, 1.6217239520471343e-06, 8.1336511357549959e-07,
     4.5626809487061786e-07, 2.8082207010049023e-07, 1.8654212688853633e-07,
     1.3188925091323157e-07, 9.8105657525287897e-08, 7.6051966380797446e-08,
     6.0972179709380934e-08, 5.0245327530730654e-08, 4.2352750575741055e-08,
     3.6374875121211252e-08, 3.1732838023724134e-08, 2.8049515011493665e-08,
     2.5071604553874626e-08, 2.2624287558819091e-08, 2.0583945529235840e-08,
     1.8861241438325071e-08, 1.7390336356823406e-08, 1.6121843966348781e-08,
     1.5018122062216319e-08, 1.4050060266482315e-08, 1.3194846721161700e-08,
     1.2434388263987334e-08, 1.1754174915506875e-08, 1.1142451635700470e-08,
     1.0589605943306357e-08, 1.0087709408646214e-08, 9.6301703268458049e-09,
     9.2114677421332708e-09, 8.8269457002862139e-09, 8.4726525827936962e-09,
     8.1452145335985826e-09},
    // x = 14.8000
    {2.3036370772435924e-01, 7.7825450707574746e-03, 7.8875883724102821e-04,
     1.3322366744145786e-04, 3.1492974397034341e-05, 9.5629439065986644e-06,
     3.5411740890066484e-06, 1.5426227438884426e-06, 7.6911186555196333e-07,
     4.2909701947279906e-07, 2.6281126459441404e-07, 1.7383130467885697e-07,
     1.2244898883868869e-07, 9.0797121046577393e-08, 7.0199065211792730e-08,
     5.6153478147076439e-08, 4.6186752857123071e-08, 3.8869354942450583e-08,
     3.3337752871528423e-08, 2.9049558049257633e-08, 2.5652122497720315e-08,
     2.2909023122229972e-08, 2.0657366765789284e-08, 1.8782147515945598e-08,
     1.7200371461517526e-08, 1.5850954852224269e-08, 1.4688133766043970e-08,
     1.3677065932831212e-08, 1.2790834064770174e-08, 1.2008364989978387e-08,
     1.1312959338520326e-08, 1.0691235866932916e-08, 1.0132362217169914e-08,
     9.6274866934973897e-09, 9.1693131917498261e-09, 8.7517794676425725e-09,
     8.3698109531788104e-09, 8.0191304592412270e-09, 7.6961096775189781e-09,
     7.3976522696094681e-09},
    // x = 14.9000
    {2.2958937493795381e-01, 7.7043300960941419e-03, 7.7559235617228315e-04,
     1.3012160088969883e-04, 3.0554131942265144e-05, 9.2164131923488522e-06,
     3.3906869385049297e-06, 1.4678139547709266e-06, 7.2748775077763189e-07,
     4.0366501392044516e-07, 2.4602553409724784e-07, 1.6202892173327953e-07,
     1.1371110239848844e-07, 8.4050443341333480e-08, 6.4808309457232557e-08,
     5.1723712277492323e-08, 4.2461769537308351e-08, 3.5676645850359052e-08,
     3.0557324039331953e-08, 2.6595524867946604e-08, 2.3461447028840431e-08,
     2.0934395295033031e-08, 1.8862572124024970e-08, 1.7138973060088994e-08,
     1.5686489444084213e-08, 1.4448444106364211e-08, 1.3382426209197774e-08,
     1.2456182573725173e-08, 1.1644821918269328e-08, 1.0928875870834328e-08,
     1.0292930469421384e-08, 9.7246446390682816e-09, 9.2140357003630798e-09,
     8.7529520850043854e-09, 8.3346792384166881e-09, 7.9536415792042753e-09,
     7.6051746219138395e-09, 7.2853489600418493e-09, 6.9908330071362651e-09,
     6.7187850053363661e-09},
    // x = 15.0000
    {2.2882279832973734e-01, 7.6274164142472282e-03, 7.6273144468070610e-04,
     1.2711171070276764e-04, 2.9649202419962391e-05, 8.8845639819719891e-06,
     3.2474767160396675e-06, 1.3970431662671286e-06, 6.8832483911683680e-07,
     3.7985399814947993e-07, 2.3037745481127646e-07, 1.5106747435116600e-07,
     1.0562165298583309e-07, 7.7821300138133348e-08, 5.9842426107592518e-08,
     4.7650934553945187e-08, 3.9042555022349184e-08, 3.2750066507856583e-08,
     2.8011666909105155e-08, 2.4350978504502168e-08, 2.1459528039125319e-08,
     1.9131277636743783e-08, 1.7224753929271850e-08, 1.5640386877180223e-08,
     1.4306528757521515e-08, 1.3170586287224510e-08, 1.2193252671553821e-08,
     1.1344669036350863e-08, 1.0601815883249239e-08, 9.9467061614456380e-09,
     9.3651114341156240e-09, 8.8456492326409086e-09, 8.3791193718178524e-09,
     7.9580146222112525e-09, 7.5761553062107923e-09, 7.2284131875574893e-09,
     6.9105005271584052e-09, 6.6188072660249239e-09, 6.3502741483346401e-09,
     6.1022929639981514e-09},
    // x = 15.1000
    {2.2806384928305334e-01, 7.5517739235492548e-03, 7.5016705227754681e-04,
     1.2419067779873778e-04, 2.8776753401500339e-05, 8.5666883691274489e-06,
     3.1111516620711204e-06, 1.3300721768566390e-06, 6.5146658235084447e-07,
     3.5755430576524093e-07, 2.1578609085051315e-07, 1.4088463715253958e-07,
     9.8130953266699674e-08, 7.2068939265344756e-08, 5.5267201798351842e-08,
     4.3905860473447806e-08, 3.5903636053735226e-08, 3.0067156421164929e-08,
     2.5680748638632090e-08, 2.2297875290573080e-08, 1.9629975843651337e-08,
     1.7484673633629940e-08, 1.5730102661986985e-08, 1.4273601123686526e-08,
     1.3048588972158491e-08, 1.2006258072241440e-08, 1.1110175358616559e-08,
     1.0332696296443666e-08, 9.6525308096133142e-09, 9.0530592812540433e-09,
     8.5211467463544516e-09, 8.0462942276007200e-09, 7.6200222014817586e-09,
     7.2354164649746205e-09, 6.8867893145432849e-09, 6.5694237367353931e-09,
     6.2793781276199156e-09, 6.0133356775556180e-09, 5.7684870848536510e-09,
     5.5424383999884476e-09},
    // x = 15.2000
    {2.2731240214670703e-01, 7.4773734110220331e-03, 7.3789045333647583e-04,
     1.2135532286332720e-04, 2.7935421329146630e-05, 8.2621164580620674e-06,
     2.9813430724161173e-06, 1.2666779047426567e-06, 6.1676700440483847e-07,
     3.3666406044899645e-07, 2.0217649708217670e-07, 1.3142285531226802e-07,
     9.1193224833861922e-08, 6.6755887618874544e-08, 5.1051227910422737e-08,
     4.0461643821365054e-08, 3.3021688198340609e-08, 2.7607370832647329e-08,
     2.3546261247035960e-08, 2.0419737793012140e-08, 1.7957833443905004e-08,
     1.5980905722609983e-08, 1.4366029896035124e-08, 1.3026964081869020e-08,
     1.1901831401811955e-08, 1.0945332284737611e-08, 1.0123694384501642e-08,
     9.4113212219023617e-09, 8.7885207227589659e-09, 8.2399356567266921e-09,
     7.7534396879645101e-09, 7.3193481491142858e-09, 6.9298452684681126e-09,
     6.5785626716338130e-09, 6.2602651896942159e-09, 5.9706138439520817e-09,
     5.7059850555208186e-09, 5.4633313098773070e-09, 5.2400727305273736e-09,
     5.0340119413766242e-09},
    // x = 15.3000
    {2.2656833414769759e-01, 7.4041865205779342e-03, 7.2589323345493553e-04,
     1.1860259964908175e-04, 2.7123907827803773e-05, 7.9702141319430527e-06,
     2.8577038378626463e-06, 1.2066513686090798e-06, 5.8408995151502040e-07,
     3.1708860009734562e-07, 1.8947926108081681e-07, 1.2262896960524779e-07,
     8.4766283926278677e-08, 6.1847682528771087e-08, 4.7165667173210743e-08,
     3.7293671086488285e-08, 3.0375352643940042e-08, 2.5351915832465682e-08,
     2.1591471939860353e-08, 1.8701517941119709e-08, 1.6429450553173156e-08,
     1.4607498689657776e-08, 1.3121059832637703e-08, 1.1889858813468150e-08,
     1.0856384034523875e-08, 9.9785883959182664e-09, 9.2251632488647259e-09,
     8.5724065167731047e-09, 8.0021027988874942e-09, 7.5000603516341726e-09,
     7.0550832669885554e-09, 6.6582374676382149e-09, 6.3023185517853053e-09,
     5.9814605584794577e-09, 5.6908445961290524e-09, 5.4264792273311582e-09,
     5.1850330837888138e-09, 4.9637059588240372e-09, 4.7601285665107167e-09,
     4.5722838838154303e-09},
    // x = 15.4000
    {2.2583152530706313e-01, 7.3321857225521299e-03, 7.1416727646749444e-04,
     1.1592958863246466e-04, 2.6340976232782186e-05, 7.6903809622557949e-06,
     2.7399070820536535e-06, 1.1497967405563736e-06, 5.5330839775274064e-07,
     2.9873994494270770e-07, 1.7763008105033291e-07, 1.1445387157453891e-07,
     7.8811252878350737e-08, 5.7312625467453742e-08, 4.3584039937014883e-08,
     3.4379373396501241e-08, 2.7945068757478779e-08, 2.3283597773883373e-08,
     1.9801086510543022e-08, 1.7129472186003226e-08, 1.5032368756264030e-08,
     1.3353073423559493e-08, 1.1984730508177439e-08, 1.0852610886584166e-08,
     9.9032550035150209e-09, 9.0976310912676773e-09, 8.4067119510863566e-09,
     7.8085479170903741e-09, 7.2862882428175625e-09, 6.8268172817986485e-09,
     6.4197974696405737e-09, 6.0569866261974899e-09, 5.7317435028973636e-09,
     5.4386646145174761e-09, 5.1733140133595925e-09, 4.9320197844358230e-09,
     4.7117190627840218e-09, 4.5098387669495936e-09, 4.3242029207799827e-09,
     4.1529599785346661e-09},
    // x = 15.5000
    {2.2510185835870916e-01, 7.2613442845023525e-03, 7.0270475207647723e-04,
     1.1333349100793953e-04, 2.5585448319977902e-05, 7.4220482497915974e-06,
     2.6276448906918057e-06, 1.0959304658945767e-06, 5.2430380168248560e-07,
     2.8153630620453735e-07, 1.6656937682660037e-07, 1.0685218635796807e-07,
     7.3292295160376361e-08, 5.3121556217671336e-08, 4.0282028439210563e-08,
     3.1698054466951894e-08, 2.5712921039158417e-08, 2.1386685742923579e-08,
     1.8161124669055714e-08, 1.5691047628821411e-08, 1.3755216814917302e-08,
     1.2207250101613287e-08, 1.0947503809928176e-08, 9.9064043608119456e-09,
     9.0342538289213923e-09, 8.2948161727596819e-09, 7.6611770499724128e-09,
     7.1130079802238848e-09, 6.6347194403458115e-09, 6.2141894141324778e-09,
     5.8418722313619401e-09, 5.5101635435961623e-09, 5.2129408704822926e-09,
     4.9452264619275300e-09, 4.7029366673401364e-09, 4.4826933478992096e-09,
     4.2816803690078575e-09, 4.0975332476123223e-09, 3.9282534615910861e-09,
     3.7721412993844421e-09},
    // x = 15.6000
    {2.2437921867109531e-01, 7.1916362432160996e-03, 6.9149810406715704e-04,
     1.1081162299944829e-04, 2.4856201225741616e-05, 7.1646771884190687e-06,
     2.5206271256285320e-06, 1.0448804448772870e-06, 4.9696551026153000e-07,
     2.6540163209764249e-07, 1.5624193130947566e-07, 9.9781980913443024e-08,
     6.8176372051590475e-08, 4.9247645778531905e-08, 3.7237297532717837e-08,
     2.9230733187475946e-08, 2.3662499224749665e-08, 1.9646785942858044e-08,
     1.6658806250014948e-08, 1.4374778149070093e-08, 1.2587615218402773e-08,
     1.1160559966504741e-08, 1.0000683511542844e-08, 9.0432052890981479e-09,
     8.2419197303861137e-09, 7.5631831342741709e-09, 6.9820380400132191e-09,
     6.4796558692672155e-09, 6.0416128144274496e-09, 5.6567044045767588e-09,
     5.3161156048224918e-09, 5.0128300927758921e-09, 4.7412032963240362e-09,
     4.4966493994069409e-09, 4.2754088705350208e-09, 4.0743736880562049e-09,
     3.8909544503981281e-09, 3.7229782653686470e-09, 3.5686095161242573e-09,
     3.4262878122437069e-09},
    // x = 15.7000
    {2.2366349417166412e-01, 7.1230363778683860e-03, 6.8054003907670074e-04,
     1.0836141047091552e-04, 2.4152164544574890e-05, 6.9177571434527740e-06,
     2.4185803178733635e-06, 9.9648527183429373e-07, 4.7119020639355139e-07,
     2.5026518827813612e-07, 1.4659655988941690e-07, 9.3204495570727982e-08,
     6.3433019151905961e-08, 4.5666206434418879e-08, 3.4429330476828452e-08,
     2.6959999588252491e-08, 2.1778770397418223e-08, 1.8050726954073377e-08,
     1.5282447345085141e-08, 1.3170189650039944e-08, 1.1520089159065636e-08,
     1.0204364928815127e-08, 9.1363406108966414e-09, 8.2556920615748954e-09,
     7.5194563737937726e-09, 6.8963938076114083e-09, 6.3633594723339023e-09,
     5.9029125884331797e-09, 5.5017068664628491e-09, 5.1493851793465598e-09,
     4.8378066504635119e-09, 4.5604970994070302e-09, 4.3122522454059061e-09,
     4.0888471081047209e-09, 3.8868203657999067e-09, 3.7033123720796223e-09,
     3.5359420911068401e-09, 3.3827126080862548e-09, 3.2419378614513112e-09,
     3.1121853049317579e-09},
    // x = 15.8000
    {2.2295457527389848e-01, 7.0555201842761282e-03, 6.6982351588783522e-04,
     1.0598038381855649e-04, 2.3472317593579214e-05, 6.6808040370069861e-06,
     2.3212466339963509e-06, 9.5059352750507683e-07, 4.4688139682860799e-07,
     2.3606117004095557e-07, 1.3758580563246884e-07, 8.7083897009167912e-08,
     5.9034141089200060e-08, 4.2354517545501969e-08, 3.1839278510645982e-08,
     2.4869883038499844e-08, 2.0047962069663146e-08, 1.6586454921099490e-08,
     1.4021365488808290e-08, 1.2067713616891378e-08, 1.0543989185652648e-08,
     9.3307842980892029e-09, 8.3472453172688667e-09, 7.5371919773721401e-09,
     6.8606724729225082e-09, 6.2886765310462915e-09, 5.7997383003550672e-09,
     5.3777011748948775e-09, 5.0102149312368419e-09, 4.6877050091251387e-09,
     4.4026526185520513e-09, 4.1490834474537550e-09, 3.9221988733367520e-09,
     3.7181061384414940e-09, 3.5336183064378569e-09, 3.3661041250656342e-09,
     3.2133740534025145e-09, 3.0735928198179438e-09, 2.9452116675380828e-09,
     2.8269153673511260e-09},
    // x = 15.9000
    {2.2225235480690081e-01, 6.9890638501980632e-03, 6.5934173521986885e-04,
     1.0366617312883601e-04, 2.2815686833317347e-05, 6.4533588332545850e-06,
     2.2283829108000042e-06, 9.0706312069307477e-07, 4.2394893735961730e-07,
     2.2272834379727070e-07, 1.2916565816942944e-07, 8.1387050918167670e-08,
     5.4953822916904695e-08, 3.9291665740953778e-08, 2.9449823039165565e-08,
     2.2945731628160831e-08, 1.8457455286562847e-08, 1.5242937801770037e-08,
     1.2865793103825913e-08, 1.1058608254128065e-08, 9.6514188537958278e-09,
     8.5326280077151598e-09, 7.6268050934422826e-09, 6.8816234856797268e-09,
     6.2599277190773910e-09, 5.7347753419655733e-09, 5.2862559773571391e-09,
     4.8994013974730074e-09, 4.5627822170041212e-09, 4.2675466669478628e-09,
     4.0067500311207040e-09, 3.7748789162917177e-09, 3.5675084706315903e-09,
     3.3810518311562121e-09, 3.2125745385904716e-09, 3.0596553711464649e-09,
     2.9202807858839424e-09, 2.7927639940013803e-09, 2.6756823015664153e-09,
     2.5678281402619021e-09},
    // x = 16.0000
    {2.2155672794739220e-01, 6.9236442316317967e-03, 6.4908812999127103e-04,
     1.0141650358692614e-04, 2.2181343435430116e-05, 6.2349861170047425e-06,
     2.1397597535104033e-06, 8.6576067565362452e-07, 4.0230859250265967e-07,
     2.1020971555706109e-07, 1.2129529440202818e-07, 7.6083312741354153e-08,
     5.1168156822871434e-08, 3.6458398307891469e-08, 2.7245049362306586e-08,
     2.1174101774613484e-08, 1.6995686884179985e-08, 1.4010077889333759e-08,
     1.1806798481481922e-08, 1.0134886534236645e-08, 8.8351687536240596e-09,
     7.8033357556039823e-09, 6.9690082116160009e-09, 6.2834435876081560e-09,
     5.7120835593226219e-09, 5.2299037402359264e-09, 4.8184348760241652e-09,
     4.4638085534381246e-09, 4.1554467412450757e-09, 3.8851652978657957e-09,
     3.6465493079633085e-09, 3.4345104083282112e-09, 3.2449681564192413e-09,
     3.0746173577497251e-09, 2.9207558828116030e-09, 2.7811556623356975e-09,
     2.6539649158304470e-09, 2.5376332542614317e-09, 2.4308537296983446e-09,
     2.3325175771097884e-09},
    // x = 16.1000
    {2.2086759215403401e-01, 6.8592388300620201e-03, 6.3905635602947623e-04,
     9.9229191121442533e-05, 2.1568400987986536e-05, 6.0252727594704215e-06,
     2.0551606930820943e-06, 8.2656096189863441e-07, 3.8188162706671782e-07,
     1.9845222432307462e-07, 1.1393683929214961e-07, 7.1144335037389519e-08,
     4.7655082885986284e-08, 3.3836988671380658e-08, 2.5210330968822291e-08,
     1.9542657179026285e-08, 1.5652060112379345e-08, 1.2878631887494387e-08,
     1.0836213637611888e-08, 9.2892505484608776e-09, 8.0886563496513725e-09,
     7.1369215343014498e-09, 6.3683723305516205e-09, 5.7375999361373559e-09,
     5.2124583871818574e-09, 4.7697026099599827e-09, 4.3921986390297779e-09,
     4.0670959918440175e-09, 3.7846038080814883e-09, 3.5371546624051075e-09,
     3.3188226133758248e-09, 3.1249112572896041e-09, 2.9516575241627668e-09,
     2.7960155943906818e-09, 2.6554971395333418e-09, 2.5280517297081515e-09,
     2.4119762695055080e-09, 2.3058456752393171e-09, 2.2084592767623619e-09,
     2.1187989869411200e-09},
    // x = 16.2000
    {2.2018484710397848e-01, 6.7958257706163617e-03, 6.2924028320496110e-04,
     9.7102138272112950e-05, 2.0976013330137186e-05, 5.8238266655212646e-06,
     1.9743813985305007e-06, 7.8934636335033479e-07, 3.6259442721942466e-07,
     1.8740645846865920e-07, 1.0705514514070563e-07, 6.6543890111393754e-08,
     4.4394242722728220e-08, 3.1411112954399527e-08, 2.3332223500713774e-08,
     1.8040076332562919e-08, 1.4416862900116970e-08, 1.1840137881425876e-08,
     9.9465684414890182e-09, 8.5150316046121265e-09, 7.4058711183398587e-09,
     6.5279230711812707e-09, 5.8198976455288164e-09, 5.2394872130597356e-09,
     4.7567867490168717e-09, 4.3502019245730271e-09, 4.0038361051716917e-09,
     3.7057810255691180e-09, 3.4469737055753314e-09, 3.2204164466706795e-09,
     3.0206346298737858e-09, 2.8432933356987928e-09, 2.6849219692394998e-09,
     2.5427135695958828e-09, 2.4143765684339733e-09, 2.2980239159346918e-09,
     2.1920891878302010e-09, 2.0952624187020611e-09, 2.0064405264503683e-09,
     1.9246886478709263e-09},
    // x = 16.3000
    {2.1950839463156066e-01, 6.7333837810875071e-03, 6.1963398696793335e-04,
     9.5033330267845607e-05, 2.0403372508190785e-05, 5.6302755971109643e-06,
     1.8972289405006435e-06, 7.5400638400640280e-07, 3.4437814884038085e-07,
     1.7702639332474987e-07, 1.0061758789173678e-07, 6.2257706681805056e-08,
     4.1366844962634626e-08, 2.9165736692215885e-08, 2.1598367570251578e-08,
     1.6655967841646248e-08, 1.3281193104030851e-08, 1.0886848605318746e-08,
     9.1310304689118748e-09, 7.8061355659543684e-09, 6.7813245144967097e-09,
     5.9713557421698581e-09, 5.3190242028735036e-09, 4.7849073996834237e-09,
     4.3411822050764843e-09, 3.9677858941447987e-09, 3.6499684877839497e-09,
     3.3766939249362267e-09, 3.1395723304253569e-09, 2.9321323599574929e-09,
     2.7493159919590552e-09, 2.5871217058243898e-09, 2.4423484529848683e-09,
     2.3124092498278394e-09, 2.1951936146881701e-09, 2.0889647703636771e-09,
     1.9922819262635050e-09, 1.9039408809082589e-09, 1.8229281646775679e-09,
     1.7483853000166931e-09},
    // x = 16.4000
    {2.1883813866904583e-01, 6.6718921717823874e-03, 6.1023174026718482e-04,
     9.3020831303427612e-05, 1.9849706845746793e-05, 5.4442660679336170e-06,
     1.8235211025540041e-06, 7.2043718749096383e-07, 3.2716839112396383e-07,
     1.6726914834173602e-07, 9.4593879115687235e-08, 5.8263319449115572e-08,
     3.8555541580219965e-08, 2.7087010853873681e-08, 1.9997399681589839e-08,
     1.5380792904501920e-08, 1.2236890138448709e-08, 1.0011670459462346e-08,
     8.3833500787420493e-09, 7.1569929699699360e-09, 6.2100043393410040e-09,
     5.4626705614188701e-09, 4.8615930073953959e-09, 4.3700335925106806e-09,
     3.9621035167548743e-09, 3.6191612445898291e-09, 3.3275195114524177e-09,
     3.0769497136748284e-09, 2.8596844741974507e-09, 2.6697387661831997e-09,
     2.5024391373892871e-09, 2.3540915817775699e-09, 2.2217434802909447e-09,
     2.1030104487947820e-09, 1.9959486759392503e-09, 1.8989596079684969e-09,
     1.8107179471744018e-09, 1.7301166660149650e-09, 1.6562245869719344e-09,
     1.5882533444633156e-09},
    // x = 16.5000
    {2.1817398518935302e-01, 6.6113308161611867e-03, 6.0102800583181201e-04,
     9.1062781003796862e-05, 1.9314279120994382e-05, 5.2654623047026065e-06,
     1.7530857369080396e-06, 6.8854116806185352e-07, 3.1090489355043793e-07,
     1.5809476232103319e-07, 8.8955892434432741e-08, 5.4539930526052705e-08,
     3.5944314191995861e-08, 2.5162176395046906e-08, 1.8518870572815686e-08,
     1.4205794328736564e-08, 1.1276472437196502e-08, 9.2081077777011149e-09,
     7.6978102562481836e-09, 6.5625135066010462e-09, 5.6873331210330897e-09,
     4.9977158848184365e-09, 4.4438105843589179e-09, 3.9913770464485631e-09,
     3.6163238612040656e-09, 3.3013283465348644e-09, 3.0336882396948391e-09,
     2.8039225133478131e-09, 2.6048395294176885e-09, 2.4309036185895606e-09,
     2.2777963555586854e-09, 2.1421073916888372e-09, 2.0211130882740705e-09,
     1.9126156658929216e-09, 1.8148247227719839e-09, 1.7262688517550913e-09,
     1.6457289306442923e-09, 1.5721872173844426e-09, 1.5047881072873342e-09,
     1.4428075908417199e-09},
    // x = 16.6000
    {2.1751584215067568e-01, 6.5516801322307921e-03, 5.9201742879756085e-04,
     8.9157391065495444e-05, 1.8796384844732295e-05, 5.0935452707600856e-06,
     1.6857601615965399e-06, 6.5822655082527761e-07, 2.9553125448865542e-07,
     1.4946598532985956e-07, 8.3677503250347232e-08, 5.1068281774141221e-08,
     3.3518369501489269e-08, 2.3379476631970726e-08, 1.7153170352632706e-08,
     1.3122931532245331e-08, 1.0393080245295221e-08, 8.4702118897270658e-09,
     7.0691808067663912e-09, 6.0180444727274844e-09, 5.2091301530291658e-09,
     4.5727024975419353e-09, 4.0622166879066151e-09, 3.6457571566263213e-09,
     3.3009028013745067e-09, 3.0115549370947900e-09, 2.7659243504828477e-09,
     2.5552222054152736e-09, 2.3727893958445087e-09, 2.2135054881726651e-09,
     2.0733798327223581e-09, 1.9492637482006480e-09, 1.8386446620733485e-09,
     1.7394966770771727e-09, 1.6501706026616326e-09, 1.5693119948147594e-09,
     1.4957993456638832e-09, 1.4286969541464512e-09, 1.3672186206379165e-09,
     1.3106994106419967e-09},
    // x = 16.7000
    {2.1686361944292756e-01, 6.4929210646590884e-03, 5.8319482966057905e-04,
     8.7302942065538018e-05, 1.8295350633079066e-05, 4.9282117480175568e-06,
     1.6213905962355667e-06, 6.2940701907596351e-07, 2.8099466982677454e-07,
     1.4134808602185578e-07, 7.8734440733235861e-08, 4.7830537169843415e-08,
     3.1264043143150168e-08, 2.1728076786035083e-08, 1.5891459859214512e-08,
     1.2124821017725693e-08, 9.5804232821650940e-09, 7.7925345624630513e-09,
     6.4926765197911226e-09, 5.5193328535056699e-09, 4.7715768671119337e-09,
     4.1841717852217630e-09, 3.7136548754297400e-09, 3.3302741158429878e-09,
     3.0131607640550313e-09, 2.7473522010034138e-09, 2.5219056385027295e-09,
     2.3286732010617432e-09, 2.1614883872220010e-09, 2.0156144954032618e-09,
     1.8873635120841470e-09, 1.7738281530524231e-09, 1.6726904106921906e-09,
     1.5820827176589716e-09, 1.5004858729930280e-09, 1.4266530332735723e-09,
     1.3595524382407438e-09, 1.2983237744429070e-09, 1.2422445855698876e-09,
     1.1907041665778210e-09},
    // x = 16.8000
    {2.1621722883594177e-01, 6.4350350675782238e-03, 5.7455519754229711e-04,
     8.5497780428511988e-05, 1.7810532669229184e-05, 4.7691734734993825e-06,
     1.5598316337800550e-06, 6.0200136683504054e-07, 2.6724569015006702e-07,
     1.3370867318654119e-07, 7.4104151103003790e-08, 4.4810174395229893e-08,
     2.9168711238301558e-08, 2.0197990103398321e-08, 1.4725607717511891e-08,
     1.1204681854895522e-08, 8.8327328576906576e-09, 7.1700854410844139e-09,
     5.9639189569821210e-09, 5.0624907120530855e-09, 4.3711852466284162e-09,
     3.8289667151312477e-09, 3.3952456924788207e-09, 3.0422840082795146e-09,
     2.7506558007673279e-09, 2.5064529986381959e-09, 2.2995175430712151e-09,
     2.1222951279589087e-09, 1.9690749569757622e-09, 1.8354749721506523e-09,
     1.7180866033789637e-09, 1.6142252774631673e-09, 1.5217523510958850e-09,
     1.4389461112463636e-09, 1.3644070229211087e-09, 1.2969872350655440e-09,
     1.2357375061397789e-09, 1.1798667995491162e-09, 1.1287112048460317e-09,
     1.0817098002913567e-09},
    // x = 16.9000
    {2.1557658392935689e-01, 6.3780040880464248e-03, 5.6609368375004664e-04,
     8.3740315543270394e-05, 1.7341315248970360e-05, 4.6161563270129804e-06,
     1.5009457458416661e-06, 5.7593317480021312e-07, 2.5423799509789161e-07,
     1.2651753044289051e-07, 6.9765671323090582e-08, 4.1991884913918473e-08,
     2.7220709033303747e-08, 1.8780010004086848e-08, 1.3648132615854642e-08,
     1.0356285744536285e-08, 8.1447180587623273e-09, 6.5982931433342955e-09,
     5.4789015479953351e-09, 4.6439635970819319e-09, 4.0047690115898676e-09,
     3.5042053781280087e-09, 3.1043622356522608e-09, 2.7793761215944887e-09,
     2.5111624266415161e-09, 2.2867920476620116e-09, 2.0968345188255768e-09,
     1.9342852604526030e-09, 1.7938550778235555e-09, 1.6714896964935618e-09,
     1.5640385916764052e-09, 1.4690226744912388e-09, 1.3844686626771609e-09,
     1.3087892124129632e-09, 1.2406949567221601e-09, 1.1791291218748984e-09,
     1.1232183419721727e-09, 1.0722352446601082e-09, 1.0255696939363785e-09,
     9.8270647155748494e-10},
    // x = 17.0000
    {2.1494160010412461e-01, 6.3218105501396281e-03, 5.5780559561887332e-04,
     8.2029017021093489e-05, 1.6887109405013717e-05, 4.4688995667039869e-06,
     1.4446028193104705e-06, 5.5113050805436070e-07, 2.4192818363610461e-07,
     1.1974646307723310e-07, 6.5699512390575802e-08, 3.9361481853360007e-08,
     2.5409256042336128e-08, 1.7465647760898573e-08, 1.2652150363423820e-08,
     9.5739112750423281e-09, 7.5115256570135468e-09, 6.0729696909880991e-09,
     5.0339577057862437e-09, 4.2605017037129193e-09, 3.6694173310868308e-09,
     3.2072568643149575e-09, 2.8386078816968138e-09, 2.5393522790736782e-09,
     2.2926523508415056e-09, 2.0864878824524267e-09, 1.9121030828594775e-09,
     1.7630025354029518e-09, 1.6342871252738868e-09, 1.5222055574340474e-09,
     1.4238456088458315e-09, 1.3369177926983769e-09, 1.2596012868277917e-09,
     1.1904325428222097e-09, 1.1282236229775146e-09, 1.0720015528704978e-09,
     1.0209627372339179e-09, 9.7443831265370430e-10, 9.3186753709339715e-10,
     8.9277715201000878e-10},
    // x = 17.1000
    {2.1431219447557845e-01, 6.2664373396454061e-03, 5.4968639062077918e-04,
     8.0362412087670574e-05, 1.6447351605500906e-05, 4.3271551094720806e-06,
     1.3906797221821619e-06, 5.2752563400015062e-07, 2.3027557907717269e-07,
     1.1336915610377143e-07, 6.1887551473938734e-08, 3.6905815069876066e-08,
     2.3724387165034985e-08, 1.6247075250377844e-08, 1.1731325327404951e-08,
     8.8523020155494428e-09, 6.9287034186865553e-09, 5.5902779898743821e-09,
     4.6257316983231263e-09, 3.9091335460527338e-09, 3.3624708401492382e-09,
     2.9357192655896588e-09, 2.5957959899825430e-09, 2.3202080112942045e-09,
     2.0932769288852379e-09, 1.9038264313574382e-09, 1.7437263870256384e-09,
     1.6069530102165535e-09, 1.4889681286245076e-09, 1.3863005195509872e-09,
     1.2962580435835443e-09, 1.2167261723872291e-09, 1.1460246578199558e-09,
     1.0828040115597997e-09, 1.0259696845483059e-09, 9.7462580909010517e-10,
     9.2803294977903810e-10, 8.8557601669353276e-10, 8.4673964003101551e-10,
     8.1108908536363562e-10},
    // x = 17.2000
    {2.1368828584800359e-01, 6.2118677893331477e-03, 5.4173167072837740e-04,
     7.8739083101704208e-05, 1.6021502522344982e-05, 4.1906868534256637e-06,
     1.3390598966401255e-06, 5.0505475910057165e-07, 2.1924204776879663e-07,
     1.0736104269789454e-07, 5.8312932209244970e-08, 3.4612692823609755e-08,
     2.2156889293053917e-08, 1.5117072356522638e-08, 1.0879825882678546e-08,
     8.1866281202240859e-09, 6.3921665251446995e-09, 5.1467020949943577e-09,
     4.2511520367102755e-09, 3.5871419214043851e-09, 3.0814997579302902e-09,
     2.6873996166619999e-09, 2.3739314027470030e-09, 2.1201154029548237e-09,
     1.9113511824682415e-09, 1.7372460662492645e-09, 1.5902501778123126e-09,
     1.4647766326187013e-09, 1.3566212654668411e-09, 1.2625717694663100e-09,
     1.1801392772251065e-09, 1.1073707165337776e-09, 1.0427154616179567e-09,
     9.8492912044178976e-10, 9.3300313394273682e-10, 8.8611258663023742e-10,
     8.4357704618535898e-10, 8.0483084620823143e-10, 7.6940029661765583e-10,
     7.3688603423226384e-10},
    // x = 17.3000
    {2.1306979467065232e-01, 6.1580856647757039e-03, 5.3393717702064061e-04,
     7.7157665193360050e-05, 1.5609045865334530e-05, 4.0592700397417659e-06,
     1.2896329775784331e-06, 4.8365778310907506e-07, 2.0879183045348236e-07,
     1.0169918221860012e-07, 5.4959972520455670e-08, 3.2470809537107922e-08,
     2.0698242959764349e-08, 1.4068978643019455e-08, 1.0092283538031494e-08,
     7.5724511461640945e-09, 5.8981668377298099e-09, 4.7390200202152685e-09,
     3.9074071607485277e-09, 3.2920419653774501e-09, 2.8242839220833770e-09,
     2.4602956028839004e-09, 2.1711935831899468e-09, 1.9374074652575721e-09,
     1.7453392462852705e-09, 1.5853239919752213e-09, 1.4503500186445317e-09,
     1.3352352035411748e-09, 1.2360844869690459e-09, 1.1499249367514551e-09,
     1.0744554428526004e-09, 1.0078719383922954e-09, 9.4874332870167315e-10,
     8.9592206416107450e-10, 8.4847876743222650e-10, 8.0565381523666551e-10,
     7.6682103861754196e-10, 7.3146019865450175e-10, 6.9913589460840375e-10,
     6.6948124101767978e-10},
    // x = 17.4000
    {2.1245664299515013e-01, 6.1050751506987857e-03, 5.2629878451907852e-04,
     7.5616844016177304e-05, 1.5209487278184441e-05, 3.9326906514723434e-06,
     1.2422944348785448e-06, 4.6327806956892740e-07, 1.9889138537676737e-07,
     9.6362147097202196e-08, 5.1814079380012483e-08, 3.0469679153961342e-08,
     1.9340568624100664e-08, 1.3096648941957160e-08, 9.3637554313343627e-09,
     7.0056918128243975e-09, 5.4432647636537025e-09, 4.3642788729357660e-09,
     3.5919232221363292e-09, 3.0215611148458881e-09, 2.5887945705914374e-09,
     2.2525788785011910e-09, 1.9859212457799926e-09, 1.7705638976348331e-09,
     1.5938411139778646e-09, 1.4467638552444794e-09, 1.3228196607776130e-09,
     1.2172014252530510e-09, 1.1263001714514369e-09, 1.0473642916759906e-09,
     9.7826611455391342e-10, 9.1733909670637875e-10, 8.6326237557738727e-10,
     8.1497764339434169e-10, 7.7162843922449329e-10, 7.3251522655410218e-10,
     6.9706174318774084e-10, 6.6478950661776159e-10, 6.3529829235413224e-10,
     6.0825103703623743e-10},
    // x = 17.5000
    {2.1184875443424242e-01, 6.0528208378357370e-03, 5.1881249724330437e-04,
     7.4115353606427536e-05, 1.4822353292955292e-05, 3.8107448470011472e-06,
     1.1969452378701476e-06, 4.4386223145012797e-07, 1.8950924229127077e-07,
     9.1329917925547516e-08, 4.8861669972227509e-08, 2.8599573653123949e-08,
     1.8076577213268878e-08, 1.2194412536408058e-08, 8.6896899121593656e-09,
     6.4826004541480519e-09, 5.0243035006042802e-09, 4.0197721132143326e-09,
     3.3023437830017647e-09, 2.7736208118178790e-09, 2.3731777172416377e-09,
     2.0625798528419271e-09, 1.8165983461360882e-09, 1.6181981148195473e-09,
     1.4555805668308301e-09, 1.3203844633506017e-09, 1.2065604592411644e-09,
     1.1096489366383426e-09, 1.0263057130762622e-09, 9.5398383108305978e-10,
     8.9071584218461313e-10, 8.3496213759489999e-10, 7.8550351745822214e-10,
     7.4136391649556207e-10, 7.0175402422178985e-10, 6.6602960325325410e-10,
     6.3366029352973168e-10, 6.0420599629233454e-10, 5.7729880469955768e-10,
     5.5262904012646283e-10},
    // x = 17.6000
    {2.1124605412183001e-01, 6.0013077102661948e-03, 5.1147444347553010e-04,
     7.2651974344253479e-05, 1.4447190339484279e-05, 3.6932384260065551e-06,
     1.1534915405154645e-06, 4.2535993087424161e-07, 1.8061586656778082e-07,
     8.6583786128537932e-08, 4.6090098764615969e-08, 2.6851466310488542e-08,
     1.6899524579928912e-08, 1.1357035641206940e-08, 8.0658949541150230e-09,
     5.9997299358408268e-09, 4.6383854569126974e-09, 3.7030187542721723e-09,
     3.0365112634030571e-09, 2.5463197959708664e-09, 2.1757389805433292e-09,
     1.8887738146175564e-09, 1.6618413096822830e-09, 1.4790454263626183e-09,
     1.3293941792984242e-09, 1.2051095130080703e-09, 1.1005717396498021e-09,
     1.0116432464119631e-09, 9.3522496093521244e-10, 8.6895917174909749e-10,
     8.1102645470053784e-10, 7.6000436957368785e-10, 7.1476748168759594e-10,
     6.7441552221464367e-10, 6.3822102445003575e-10, 6.0559064657143570e-10,
     5.7603624939864388e-10, 5.4915301927168931e-10, 5.2460274197834505e-10,
     5.0210088649413062e-10},
    // x = 17.7000
    {2.1064846867424880e-01, 5.9505211332182631e-03, 5.0428087122408166e-04,
     7.1225531011246052e-05, 1.4083563806678663e-05, 3.5799863259268449e-06,
     1.1118443859575617e-06, 4.0772369195340953e-07, 1.7218353268371573e-07,
     8.2106263653264494e-08, 4.3487590031480807e-08, 2.5216979332302053e-08,
     1.5803169557778035e-08, 1.0579686910927869e-08, 7.4885091609448584e-09,
     5.5539108296552672e-09, 4.2828506616301914e-09, 3.4117443371805602e-09,
     2.7924499859806615e-09, 2.3379188469966104e-09, 1.9949297382842727e-09,
     1.7297682760466324e-09, 1.5203873890508687e-09, 1.3519522658103023e-09,
     1.2142213049570568e-09, 1.0999582385675500e-09, 1.0039420302073656e-09,
     9.2233348314649429e-10, 8.5226043150996285e-10, 7.9154017792269514e-10,
     7.3849006214222755e-10, 6.9179580488903026e-10, 6.5041845792101748e-10,
     6.1352761206953471e-10, 5.8045276019925518e-10, 5.5064740554079898e-10,
     5.2366224564382983e-10, 4.9912490492642938e-10, 4.7672445118296966e-10,
     4.5619944755604982e-10},
    // x = 17.8000
    {2.1005592615274601e-01, 5.9004468413147958e-03, 4.9722814387654832e-04,
     6.9834890939423456e-05, 1.3731057152716214e-05, 3.4708121470555899e-06,
     1.0719194291669823e-06, 3.9090872583999588e-07, 1.6418620641384899e-07,
     7.7880999150801038e-08, 4.1043175410064729e-08, 2.3688335515293365e-08,
     1.4781735325416623e-08, 9.8579057266432603e-09, 6.9539751503497916e-09,
     5.1422286543041689e-09, 3.9552569948458886e-09, 3.1438635270505272e-09,
     2.5683506792093550e-09, 2.1468268501075988e-09, 1.8293344912157556e-09,
     1.5842914290149018e-09, 1.3910840500203730e-09, 1.2358663759550914e-09,
     1.1090949551397133e-09, 1.0040368970486072e-09, 9.1584108097088387e-10,
     8.4094488832981198e-10, 7.7668622447258661e-10, 7.2104425640510902e-10,
     6.7246269272432944e-10, 6.2972710644013590e-10, 5.9187832693291948e-10,
     5.5815033662147265e-10, 5.2792509232366284e-10, 5.0069921638813589e-10,
     4.7605913192813791e-10, 4.5366228550108989e-10, 4.3322281308050118e-10,
     4.1450048708659836e-10},
    // x = 17.9000
    {2.0946835602711042e-01, 5.8510709272452423e-03, 4.9031273603366004e-04,
     6.8478962246860443e-05, 1.3389271061377826e-05, 3.3655477045168625e-06,
     1.0336366765080654e-06, 3.7487276714827132e-07, 1.5659943509852883e-07,
     7.3892700166990278e-08, 3.8746636102683732e-08, 2.2258313616190529e-08,
     1.3829873812168842e-08, 9.1875730337285877e-09, 6.4590151177076622e-09,
     4.7620030088487324e-09, 3.6533620831724337e-09, 2.8974641912835948e-09,
     2.3625563127985580e-09, 1.9715880691846114e-09, 1.6776593296497851e-09,
     1.4511816150531747e-09, 1.2728792947080724e-09, 1.1298278649476433e-09,
     1.0131334903855243e-09, 9.1653101486461195e-10, 8.3551259947926566e-10,
     7.6677198535372723e-10, 7.0784157814853302e-10, 6.5685025859577900e-10,
     6.1235850727412372e-10, 5.7324408511129082e-10, 5.3862141523108494e-10,
     5.0778383500791136e-10, 4.8016162706123568e-10, 4.5529110532089665e-10,
     4.3279155812977041e-10, 4.1234785046749282e-10, 3.9369715330513498e-10,
     3.7661871809380206e-10},
    // x = 18.0000
    {2.0888568914041525e-01, 5.8023798308454297e-03, 4.8353122951545952e-04,
     6.7156692155487580e-05, 1.3057822641907450e-05, 3.2640326054839527e-06,
     9.9692024112718770e-07, 3.5957592096968688e-07, 1.4940024541112749e-07,
     7.0127060895679289e-08, 3.6588449368699821e-08, 2.0920207138832877e-08,
     1.2942632901345658e-08, 8.5648845219146851e-09, 6.0006083985273285e-09,
     4.4107684392383310e-09, 3.3751067186576606e-09, 2.6707928325275021e-09,
     2.1735491498263917e-09, 1.8108705221906621e-09, 1.5387214061312011e-09,
     1.3293777196296299e-09, 1.1648128388711525e-09, 1.0329610556802578e-09,
     9.2553305200720943e-10, 8.3669832787890772e-10, 7.6226763825310169e-10,
     6.9917236340838411e-10, 6.4512500674301607e-10, 5.9839293443442630e-10,
     5.5764453852552286e-10, 5.2184269737067321e-10, 4.9016972748997483e-10,
     4.6197368172599213e-10, 4.3672935919246321e-10, 4.1400961221020558e-10,
     3.9346396450587408e-10, 3.7480249067268273e-10, 3.5778352932606074e-10,
     3.4220422259428604e-10},
    // x = 18.1000
    {2.0830785767483326e-01, 5.7543603285684442e-03, 4.7688030953176193e-04,
     6.5867065386830415e-05, 1.2736344669952042e-05, 3.1661138501110334e-06,
     9.6169811314012199e-07, 3.4498051975891276e-07, 1.4256704808935761e-07,
     6.6570695082047873e-08, 3.4559738978175798e-08, 1.9667786270517845e-08,
     1.2115426206435490e-08, 7.9863259561309800e-09, 5.5759708637179761e-09,
     4.0862568922341291e-09, 3.1185996715555896e-09, 2.4622412600107407e-09,
     1.9999389102604985e-09, 1.6634553627263190e-09, 1.4114393258944966e-09,
     1.2179104091913995e-09, 1.0660080675769185e-09, 9.4446705779048569e-10,
     8.4556066760240636e-10, 7.6386235258486767e-10, 6.9547857549399146e-10,
     6.3756102077999731e-10, 5.8798896668381329e-10, 5.4515788817767150e-10,
     5.0783590755626823e-10, 4.7506449756869884e-10, 4.4608861320115570e-10,
     4.2030674887098229e-10, 3.9723471340911134e-10, 3.7647899658241098e-10,
     3.5771693945870753e-10, 3.4068179645139332e-10, 3.2515135869913122e-10,
     3.1093920085324659e-10},
    // x = 18.2000
    {2.0773479511848542e-01, 5.7069995233305654e-03, 4.7035676100931075e-04,
     6.4609102631696914e-05, 1.2424484867282763e-05, 3.0716454547461288e-06,
     9.2790194366543014e-07, 3.3105098941936377e-07, 1.3607954913474464e-07,
     6.3211073695189754e-08, 3.2652229324803694e-08, 1.8495262720745237e-08,
     1.1344005214049164e-08, 7.4486504825680773e-09, 5.1825359957029416e-09,
     3.7863816233416034e-09, 2.8821037785120515e-09, 2.2703343929985362e-09,
     1.8404519497032374e-09, 1.5282271800348591e-09, 1.2948243746046302e-09,
     1.1158941366219478e-09, 9.7566470024705566e-10, 8.6361699794962951e-10,
     7.7254797027147482e-10, 6.9740653180085990e-10, 6.3457363560694612e-10,
     5.8140521707190292e-10, 5.3593500420889221e-10, 4.9667699049878724e-10,
     4.6249147400212997e-10, 4.3249250352690146e-10, 4.0598282792427682e-10,
     3.8240744586063321e-10, 3.6131994744261370e-10, 3.4235779087334026e-10,
     3.2522391116087725e-10, 3.0967287717015162e-10, 2.9550035719808208e-10,
     2.8253501973905252e-10},
    // x = 18.3000
    {2.0716643623328762e-01, 5.6602848347169707e-03, 4.6395746506842673e-04,
     6.3381859090044757e-05, 1.2121905218137097e-05, 2.9804880960857478e-06,
     8.9546684181551161e-07, 3.1775372396464914e-07, 1.2991866702115409e-07,
     6.0036467018145427e-08, 3.0858202919378867e-08, 1.7397257235285781e-08,
     1.0624433604264381e-08, 6.9488577489375105e-09, 4.8179375061698414e-09,
     3.5092224367057041e-09, 2.6640231978792403e-09, 2.0937190993037805e-09,
     1.6939213655774792e-09, 1.4041651374554679e-09, 1.1879725099799613e-09,
     1.0225198473683558e-09, 8.9305210222247826e-10, 7.8974584985013318e-10,
     7.0588547670654850e-10, 6.3676890336405093e-10, 5.7903190098552025e-10,
     5.3021978843540798e-10, 4.8850934066805439e-10, 4.5252420553503895e-10,
     4.2120988063034732e-10, 3.9374743778565583e-10, 3.6949295314205144e-10,
     3.4793430215127486e-10, 3.2865988384258478e-10, 3.1133566911589481e-10,
     2.9568814239703570e-10, 2.8149147266359091e-10, 2.6855775831344930e-10,
     2.5672953264040728e-10},
    // x = 18.4000
    {2.0660271702375890e-01, 5.6142039895325582e-03, 4.5767939564230847e-04,
     6.2184423077467902e-05, 1.1828281320150861e-05, 2.8925087750172316e-06,
     8.6433118381700919e-07, 3.0505696817656314e-07, 1.2406645548708828e-07,
     5.7035890830350620e-08, 2.9170461006876743e-08, 1.6368769576652554e-08,
     9.9530635744405197e-09, 6.4841746912341121e-09, 4.4799933679272660e-09,
     3.2530121452797033e-09, 2.4628917331541621e-09, 1.9311539801763539e-09,
     1.5592779506134421e-09, 1.2903348763342327e-09, 1.0900570503651501e-09,
     9.3704832449384701e-10, 8.1750318558798411e-10, 7.2224681053971903e-10,
     6.4501737423285905e-10, 5.8143724494055615e-10, 5.2837877087964071e-10,
     4.8356288404954597e-10, 4.4529885595454913e-10, 4.1231179526662231e-10,
     3.8362595644383675e-10, 3.5848431031186462e-10, 3.3629214201221243e-10,
     3.1657686160859883e-10, 2.9895894033092958e-10, 2.8313060215317556e-10,
     2.6884000079558554e-10, 2.5587932974131732e-10, 2.4407578823372725e-10,
     2.3328464593476714e-10},
    // x = 18.5000
    {2.0604357470675796e-01, 5.5687450126840074e-03, 4.5151961623249624e-04,
     6.1015914694941058e-05, 1.1543301767970954e-05, 2.8075804989750431e-06,
     8.3443643348820250e-07, 2.9293070772120704e-07, 1.1850603151773336e-07,
     5.4199056382148548e-08, 2.7582287070239232e-08, 1.5405150778731174e-08,
     9.3265140067255726e-09, 6.0520378515180748e-09, 4.1666911440274972e-09,
     3.0161241490493745e-09, 2.2773621340151366e-09, 1.7815000205548372e-09,
     1.4355419204715863e-09, 1.1858811187967046e-09, 1.0003219992189441e-09,
     8.5880411637854457e-10, 7.4840884708937419e-10, 6.6056617451489884e-10,
     5.8943677135755959e-10, 5.3094465228513076e-10, 4.8218182715057048e-10,
     4.4103208586512557e-10, 4.0592743407057672e-10, 3.7568686702844116e-10,
     3.4940744574884964e-10, 3.2638931158674440e-10, 3.0608316129713243e-10,
     2.8805286006332504e-10, 2.7194843141275455e-10, 2.5748627312188147e-10,
     2.4443447918061505e-10, 2.3260181941120557e-10, 2.2182937280728707e-10,
     2.1198410930244806e-10},
    // x = 18.6000
    {2.0548894768211590e-01, 5.5238962183797261e-03, 4.4547527679433005e-04,
     5.9875484558643791e-05, 1.1266667567752825e-05, 2.7255819817116673e-06,
     8.0572697335287528e-07, 2.8134656622274208e-07, 1.1322150815160636e-07,
     5.1516323883761642e-08, 2.6087413002421948e-08, 1.4502077498642107e-08,
     8.7416503324568227e-09, 5.6500771024205893e-09, 3.8761745070962721e-09,
     2.7970610377531528e-09, 2.1061962921766967e-09, 1.6437120306574293e-09,
     1.3218153486998754e-09, 1.0900209086161519e-09, 9.1807594985632467e-10,
     7.8716999577243859e-10, 6.8521289561398518e-10, 6.0419866132405102e-10,
     5.3868137045311273e-10, 4.8486551211902999e-10, 4.4004707034128854e-10,
     4.0226087706219527e-10, 3.7005263809263628e-10, 3.4232823306198106e-10,
     3.1825203358285813e-10, 2.9717698782741698e-10, 2.7859570246646003e-10,
     2.6210566018670246e-10, 2.4738411642834049e-10, 2.3416972935971311e-10,
     2.2224894309582401e-10, 2.1144577270485690e-10, 2.0161405514756306e-10,
     1.9263150927386259e-10},
    // x = 18.7000
    {2.0493877550413292e-01, 5.4796462016349938e-03, 4.3954361074654706e-04,
     5.8762312586861429e-05, 1.0998091580853251e-05, 2.6463973594535019e-06,
     7.7814994571845592e-07, 2.7027770882945617e-07, 1.0819793177335768e-07,
     4.8978659252107351e-08, 2.4679987745234614e-08, 1.3655528302985611e-08,
     8.1955659585669467e-09, 5.2761006643291044e-09, 3.6067308507652594e-09,
     2.5944441324579510e-09, 1.9482562563618028e-09, 1.5168308112747150e-09,
     1.2172752480307442e-09, 1.0020374347292111e-09, 8.4268651968379927e-10,
     7.2158190344306598e-10, 6.2740742593012563e-10, 5.5268315637942623e-10,
     4.9232952490823346e-10, 4.4281183428444674e-10, 4.0161549278721074e-10,
     3.6691542779296986e-10, 3.3736268476866749e-10, 3.1194355383820769e-10,
     2.8988464059329206e-10, 2.7058767267181246e-10, 2.5358393743472979e-10,
     2.3850192018695980e-10, 2.2504397150430827e-10, 2.1296924907840190e-10,
     2.0208108466046385e-10, 1.9221751502537343e-10, 1.8324410450903301e-10,
     1.7504844729760526e-10},
    // x = 18.8000
    {2.0439299885391032e-01, 5.4359838300701933e-03, 4.3372193209945641e-04,
     5.7675606841123934e-05, 1.0737297995128863e-05, 2.5699159224770622e-06,
     7.5165510309110275e-07, 2.5969875183941812e-07, 1.0342122357896417e-07,
     4.6577593878196085e-08, 2.3354548209135844e-08, 1.2861761738554114e-08,
     7.6855651320352812e-09, 4.9280813106027686e-09, 3.3567799032993807e-09,
     2.4070038875921296e-09, 1.8024959971579264e-09, 1.3999759809572753e-09,
     1.1211672423214535e-09, 9.2127438679988820e-10, 7.7357526763238435e-10,
     6.6152433379547291e-10, 5.7452859922838664e-10, 5.0559882827285665e-10,
     4.4999664645229464e-10, 4.0442991100916121e-10, 3.6655995849067083e-10,
     3.3469166962200866e-10, 3.0757369167533387e-10, 2.8426673944882955e-10,
     2.6405496290593042e-10, 2.4638515199585320e-10, 2.3082429663617759e-10,
     2.1702947497695701e-10, 2.0472616492654042e-10, 1.9369240313067445e-10,
     1.8374706383137357e-10, 1.7474108078383822e-10, 1.6655079884493849e-10,
     1.5907288534154926e-10},
    // x = 18.9000
    {2.0385155951248826e-01, 5.3928982359904903e-03, 4.2800763269639657e-04,
     5.6614602418896834e-05, 1.0484021822343786e-05, 2.4960318612013594e-06,
     7.2619466734212397e-07, 2.4958567798319968e-07, 9.8878124922363247e-08,
     4.4305187195701346e-08, 2.2105992302530728e-08, 1.2117296049483634e-08,
     7.2091471284513081e-09, 4.6041436647782720e-09, 3.1248632610145320e-09,
     2.2335710816602479e-09, 1.6679538584339594e-09, 1.2923394086152556e-09,
     1.0327997782769814e-09, 8.4713079665567021e-10, 7.1021305256367954e-10,
     6.0652612360970500e-10, 5.2615279451199211e-10, 4.6256158916337599e-10,
     4.1133193142895940e-10, 3.6939727400409997e-10, 3.3458236221018742e-10,
     3.0531263265809261e-10, 2.8042717231563685e-10, 2.5905558569649007e-10,
     2.4053523474679557e-10, 2.2435454065847761e-10, 2.1011344917513544e-10,
     1.9749541041586812e-10, 1.8624721733330668e-10, 1.7616429400355240e-10,
     1.6707981988328288e-10, 1.5885659176761889e-10, 1.5138086507691123e-10,
     1.4455764360471396e-10},
    // x = 19.0000
    {2.0331440033476206e-01, 5.3503788087359361e-03, 4.2239817956345703e-04,
     5.5578560395285476e-05, 1.0238008420277912e-05, 2.4246440259490432e-06,
     7.0172319707899817e-07, 2.3991575698919578e-07, 9.4556146273694692e-08,
     4.2153991847770324e-08, 2.0929553912371017e-08, 1.1418890413743532e-08,
     6.7639916599622076e-09, 4.3025525016188911e-09, 2.9096347659519162e-09,
     2.0730687309228499e-09, 1.5437456373966078e-09, 1.1931791999092322e-09,
     9.5153883050752138e-10, 7.7905632345371361e-10, 6.5211579413572458e-10,
     5.5615660847440688e-10, 4.8189309807532218e-10, 4.2322086778558545e-10,
     3.7601537758908621e-10, 3.3741992274547254e-10, 3.0541104269689088e-10,
     2.7852602172099923e-10, 2.5568775676625545e-10, 2.3608962363524473e-10,
     2.1911819360374194e-10, 2.0430035190239386e-10, 1.9126646663983389e-10,
     1.7972431300136555e-10, 1.6944032983037161e-10, 1.6022595580943130e-10,
     1.5192753749821456e-10, 1.4441878420612227e-10, 1.3759506257688685e-10,
     1.3136903633902871e-10},
    // x = 19.1000
    {2.0278146522415016e-01, 5.3084151872913694e-03, 4.1689111236267663e-04,
     5.4566766811348711e-05, 9.9990130382088643e-06, 2.3556556995816085e-06,
     6.7819746271034981e-07, 2.3066747108304258e-07, 9.0443519538830257e-08,
     4.0117021264340108e-08, 1.9820779690056012e-08, 1.0763527581907172e-08,
     6.3479454063231067e-09, 4.0217019710930722e-09, 2.7098516585651999e-09,
     1.9245046658656678e-09, 1.4290582403013980e-09, 1.1018141902519332e-09,
     8.7680305749987916e-10, 7.1654694411448238e-10, 5.9884060099849557e-10,
     5.1002211462345650e-10, 4.4139610121691973e-10, 3.8725666734133756e-10,
     3.4375506551624580e-10, 3.0822979968517709e-10, 2.7879842726758441e-10,
     2.5410201002250530e-10, 2.3314111751087587e-10, 2.1516816324077607e-10,
     1.9961522955192529e-10, 1.8604474189385101e-10, 1.7411515384791674e-10,
     1.6355667899979623e-10, 1.5415386467706952e-10, 1.4573290053391305e-10,
     1.3815225328757897e-10, 1.3129567088144089e-10, 1.2506689669596204e-10,
     1.1938563306980905e-10},
    // x = 19.2000
    {2.0225269910798888e-01, 5.2669972531460196e-03, 4.1148404094417475e-04,
     5.3578531706748096e-05, 9.7668003845179682e-06, 2.2889743822634125e-06,
     6.5557632872788762e-07, 2.2182044509676802e-07, 8.6529153507939367e-08,
     3.8187719476258374e-08, 1.8775507507871393e-08, 1.0148397810381555e-08,
     5.9590095805241726e-09, 3.7601056710014797e-09, 2.5243664419373024e-09,
     1.7869647153524538e-09, 1.3231438653457962e-09, 1.0176189012959309e-09,
     8.0805936975807457e-10, 6.5914101391669990e-10, 5.4998223427353548e-10,
     4.6776275673352739e-10, 4.0433897898161894e-10, 3.5437688300847231e-10,
     3.1428468111329922e-10, 2.8158249031000403e-10, 2.5451888695736194e-10,
     2.3183123078366250e-10, 2.1259208193890518e-10, 1.9610851364245037e-10,
     1.8185470203794449e-10, 1.6942591348091290e-10, 1.5850653131901109e-10,
     1.4884746846584040e-10, 1.4024996459801300e-10, 1.3255379715144461e-10,
     1.2562858987252225e-10, 1.1936732588663553e-10, 1.1368145038672003e-10,
     1.0849713367525751e-10},
    // x = 19.3000
    {2.0172804791362889e-01, 5.2261151233931385e-03, 4.0617464299289447e-04,
     5.2613188194579907e-05, 9.5411442152427859e-06, 2.2245115876553717e-06,
     6.3382064275962136e-07, 2.1335538088568337e-07, 8.2802592217553166e-08,
     3.6359933005458682e-08, 1.7789846461554834e-08, 9.5708839897656280e-09,
     5.5953284475883726e-09, 3.5163875000520997e-09, 2.3521193993188723e-09,
     1.6596064479934998e-09, 1.2253146683963796e-09, 9.4001892148658891e-10,
     7.4481887472363707e-10, 6.0641566520923242e-10, 5.0516987708450340e-10,
     4.2904951483850869e-10, 3.7042682513879957e-10, 3.2431485605046014e-10,
     2.8736125768583992e-10, 2.5725512795098291e-10, 2.3236678336274422e-10,
     2.1152294897738915e-10, 1.9386291466471948e-10, 1.7874436394276730e-10,
     1.6568040893588341e-10, 1.5429666469923923e-10, 1.4430145565683647e-10,
     1.3546479093254479e-10, 1.2760329805135203e-10, 1.2056927144676160e-10,
     1.1424260582958088e-10, 1.0852478077361043e-10, 1.0333432306534892e-10,
     9.8603346848018940e-11},
    // x = 19.4000
    {2.0120745854521010e-01, 5.1857591440603954e-03, 4.0096066176583555e-04,
     5.1670091576350817e-05, 9.3218269424662289e-06, 2.1621826498822194e-06,
     6.1289313097779573e-07, 2.0525399577178881e-07, 7.9253976026053935e-08,
     3.4627884682396358e-08, 1.6860158304113208e-08, 9.0285478768154401e-09,
     5.2551787216705360e-09, 3.2892732287583786e-09, 2.1921317116118023e-09,
     1.5416534245052571e-09, 1.1349378709722855e-09, 8.6848667464296697e-10,
     6.8663316614859712e-10, 5.5798351498865420e-10, 4.6406418346956778e-10,
     3.9358156588438123e-10, 3.3939022080129131e-10, 2.9682714355978911e-10,
     2.6276311807659257e-10, 2.3504448504677186e-10, 2.1215468998574932e-10,
     1.9300339699243385e-10, 1.7679175434653492e-10, 1.6292431009830456e-10,
     1.5095019406859995e-10, 1.4052306887232659e-10, 1.3137336528397716e-10,
     1.2328871076083172e-10, 1.1609991894426151e-10, 1.0967081549525303e-10,
     1.0389075088207467e-10, 9.8669021867353250e-11, 9.3930666858585250e-11,
     8.9613262427132077e-11},
    // x = 19.5000
    {2.0069087886109258e-01, 5.1459198836621717e-03, 3.9583990391586397e-04,
     5.0748618495166679e-05, 9.1086392614960804e-06, 2.1019065406575692e-06,
     5.9275829947214770e-07, 1.9749896475175454e-07, 7.5874005216841599e-08,
     3.2986149252995175e-08, 1.5983040204805467e-08, 8.5191173456774273e-09,
     4.9369597725919414e-09, 3.0775827306488061e-09, 2.0434991258467359e-09,
     1.4323899187194941e-09, 1.0514312733540832e-09, 8.0253754362024839e-10,
     6.3309092839009305e-10, 5.1348965463944569e-10, 4.2635458234470038e-10,
     3.6108384760609391e-10, 3.1098301609146082e-10, 2.7169148473386355e-10,
     2.4028799905119291e-10, 2.1476523420547182e-10, 1.9371177243548698e-10,
     1.7611425947655755e-10, 1.6123119106963112e-10, 1.4851051465317682e-10,
     1.3753468064211162e-10, 1.2798327435060950e-10, 1.1960714011777846e-10,
     1.1221016123488670e-10, 1.0563623033954513e-10, 9.9759796767528533e-11,
     9.4478916692291263e-11, 8.9710079462623188e-11, 8.5384311287218429e-11,
     8.1444208964635649e-11},
    // x = 19.6000
    {2.0017825765192118e-01, 5.1065881269653389e-03, 3.9081023739837088e-04,
     4.9848166125305482e-05, 8.9013797958484370e-06, 2.0436056959886821e-06,
     5.7338234122438547e-07, 1.9007386622544454e-07, 7.2653905956685752e-08,
     3.1429630647552829e-08, 1.5155308735304009e-08, 8.0404745806581005e-09,
     4.6391845784579011e-09, 2.8802228209658388e-09, 1.9053861297574244e-09,
     1.3311560684535383e-09, 9.7425913883859892e-10, 7.4172631893079526e-10,
     5.8381482864773463e-10, 4.7260889745866236e-10, 3.9175681431890928e-10,
     3.3130483437471129e-10, 2.8498030608484634e-10, 2.4870494628141525e-10,
     2.1975134172550782e-10, 1.9624836390212438e-10, 1.7688231325055451e-10,
     1.6071129394624447e-10, 1.4704697161879643e-10, 1.3537748731338831e-10,
     1.2531611918120661e-10, 1.1656641302211636e-10, 1.0889806481649852e-10,
     1.0212995748688401e-10, 9.6118042653942142e-11, 9.0746557743251332e-11,
     8.5921574494493355e-11, 8.1566200472231918e-11, 7.7616968260986541e-11,
     7.4021088686462341e-11},
    // x = 19.7000
    {1.9966954461930350e-01, 5.0677548689603332e-03, 3.8586958945722408e-04,
     4.8968151396440336e-05, 8.6998547591043479e-06, 1.9872058519179715e-06,
     5.5473304834331652e-07, 1.8296313101756595e-07, 6.9585398449047541e-08,
     2.9953540793471875e-08, 1.4373984992511414e-08, 7.5906451388762407e-09,
     4.3604713660694834e-09, 2.6961806543330251e-09, 1.7770205912684216e-09,
     1.2373434207047803e-09, 9.0292841804142966e-10, 6.8564394478048965e-10,
     5.3845867249495611e-10, 4.3504326170805918e-10, 3.6001068110655927e-10,
     3.0401450645566166e-10, 2.6117658392328894e-10, 2.2768223110487656e-10,
     2.0098473327191210e-10, 1.7933973500781283e-10, 1.6152436838534492e-10,
     1.4666307495957934e-10, 1.3411682111068503e-10, 1.2341097543891319e-10,
     1.1418733959086287e-10, 1.0617160771641574e-10, 9.9150886170516326e-11,
     9.2957899240298001e-11, 8.7459717717869275e-11, 8.2549597754241709e-11,
     7.8140991612568342e-11, 7.4163096870602912e-11, 7.0557510007087604e-11,
     6.7275682729185664e-11},
    // x = 19.8000
    {1.9916469035508078e-01, 5.0294113090298487e-03, 3.8101594468663703e-04,
     4.8108010250870337e-05, 8.5038776327619402e-06, 1.9326358887913635e-06,
     5.3677972924205951e-07, 1.7615199448048826e-07, 6.6660667133532441e-08,
     2.8553379862389214e-08, 1.3636280774392848e-08, 7.1677878167376633e-09,
     4.0995358854931302e-09, 2.5245176368154036e-09, 1.6576888251155957e-09,
     1.1503908386089397e-09, 8.3698528478380963e-10, 6.3391453733402594e-10,
     4.9670480018314744e-10, 4.0051966887217636e-10, 3.3087798905496470e-10,
     2.7900249575291129e-10, 2.3938405550345740e-10, 2.0845413581356840e-10,
     1.8383448645958201e-10, 1.6389876558285974e-10, 1.4750854356786813e-10,
     1.3384985074895793e-10, 1.2232937049884876e-10, 1.1250695452010372e-10,
     1.0405079791029933e-10, 9.6707069522474729e-11, 9.0278956072667858e-11,
     8.4611955183968707e-11, 7.9583390855747053e-11, 7.5094829535564122e-11,
     7.1066519636028511e-11, 6.7433262979593645e-11, 6.4141313233110999e-11,
     6.1146020189679138e-11},
    // x = 19.9000
    {1.9866364632117203e-01, 4.9915488453076626e-03, 3.7624734316572730e-04,
     4.7267196932200880e-05, 8.3132688592554002e-06, 1.8798276835743063e-06,
     5.1949313045919999e-07, 1.6962645148061291e-07, 6.3872332793165534e-08,
     2.7224917850316799e-08, 1.2939585731527981e-08, 6.7701852593186511e-09,
     3.8551842694252155e-09, 2.3643638114177145e-09, 1.5467310519365053e-09,
     1.0697807403254594e-09, 7.7601195750964531e-10, 5.8619265217620418e-10,
     4.5826170313974310e-10, 3.6878783856696276e-10, 3.0414066992226620e-10,
     2.5607639260320491e-10, 2.1943110152594079e-10, 1.9086614390607967e-10,
     1.6816034478342675e-10, 1.4979723249202385e-10, 1.3471687979617702e-10,
     1.2216250221085483e-10, 1.1158318041185040e-10, 1.0257070960477684e-10,
     9.4817709182825428e-11, 8.8089276857964836e-11, 8.2203452282573402e-11,
     7.7017521536732626e-11, 7.2418263863206699e-11, 6.8314903652160066e-11,
     6.4633947690903406e-11, 6.1315355282176951e-11, 5.8309663441152801e-11,
     5.5575805119684387e-11},
    // x = 20.0000
    {1.9816636482997363e-01, 4.9541590692204996e-03, 3.7156187866269682e-04,
     4.6445183303996541e-05, 8.1278555493588313e-06, 1.8287159697651754e-06,
     5.0284536284486220e-07, 1.6337321408401926e-07, 6.1213426440946245e-08,
     2.5964177396841186e-08, 1.2281455422938597e-08, 6.3962352564818001e-09,
     3.6263064319160701e-09, 2.2149126793865799e-09, 1.4435372180249770e-09,
     9.9503564250714445e-10, 7.1962378238207288e-10, 5.4216077990424658e-10,
     4.2286184185525160e-10, 3.3961836315514432e-10, 2.7959906351530171e-10,
     2.3506019954222044e-10, 2.0116087394692316e-10, 1.7477714262932473e-10,
     1.5383430202849277e-10, 1.3691817942393983e-10, 1.2304183820455960e-10,
     1.1150159506007756e-10, 1.0178585264664286e-10, 9.3515999460502150e-11,
     8.6407258643276826e-11, 8.0242228870033692e-11, 7.4852669909339243e-11,
     7.0106748041714228e-11, 6.5899962408905069e-11, 6.2148594594398691e-11,
     5.8784914844092445e-11, 5.5753629029505457e-11, 5.3009213869357658e-11,
     5.0513896137548919e-11},
    // x = 20.1000
    {1.9767279902529536e-01, 4.9172337602061298e-03, 3.6695769690570822e-04,
     4.5641458197005456e-05, 7.9474712032367232e-06, 1.7792382044828009e-06,
     4.8680983185047013e-07, 1.5737967176950770e-07, 5.8677364866186877e-08,
     2.4767417756300353e-08, 1.1659600210144374e-08, 6.0444426739086852e-09,
     3.4118699646267896e-09, 2.0754164227305220e-09, 1.3475431465627193e-09,
     9.2571498329393252e-10, 6.6746655622410812e-10, 5.0152705058439795e-10,
     3.9025964797390928e-10, 3.1280094659532030e-10, 2.5707034814215739e-10,
     2.1579291920213156e-10, 1.8443001501962408e-10, 1.6005825258657498e-10,
     1.4073952611110087e-10, 1.2515492182938261e-10, 1.1238537540785440e-10,
     1.0177651678391811e-10, 9.2853221423202196e-11, 8.5264296000491613e-11,
     7.8745884132493075e-11, 7.3096766422371080e-11, 6.8161377181756862e-11,
     6.3817925246552791e-11, 5.9969951980503701e-11, 5.6540242904278928e-11,
     5.3466376226394131e-11, 5.0697426433433565e-11, 4.8191494044892191e-11,
     4.5913833819749127e-11},
    // x = 20.2000
    {1.9718290286381646e-01, 4.8807648806011529e-03, 3.6243299391766333e-04,
     4.4855526783625222e-05, 7.7719554444435404e-06, 1.7313344423300581e-06,
     4.7136117167755263e-07, 1.5163385400868007e-07, 5.6257927729522081e-08,
     2.3631119840207096e-08, 1.1071874928376889e-08, 5.7134119712476761e-09,
     3.2109144921086136e-09, 1.9451814961684855e-09, 1.2582269935406082e-09,
     8.6141220186062140e-10, 6.1921406931115677e-10, 4.6402312944949626e-10,
     3.6022969488177815e-10, 2.8814279339399156e-10, 2.3638710606082049e-10,
     1.9812726462834445e-10, 1.6910748815619514e-10, 1.4659176013663994e-10,
     1.2876937794340482e-10, 1.1441014014154226e-10, 1.0265810123044873e-10,
     9.2904690784783525e-11, 8.4708617219172315e-11, 7.7744092177829102e-11,
     7.1766623291668030e-11, 6.6589954536713645e-11, 6.2070229661001057e-11,
     5.8094927486547957e-11, 5.4575007031935156e-11, 5.1439248210272786e-11,
     4.8630117955959975e-11, 4.6100711714767644e-11, 4.3812463393540458e-11,
     4.1733411480051586e-11},
    // x = 20.3000
    {1.9669663109704430e-01, 4.8447445706922198e-03, 3.5798601441222892e-04,
     4.4086909978349405e-05, 7.6011537662139567e-06, 1.6849472156592724e-06,
     4.5647518305647446e-07, 1.4612439506342463e-07, 5.3949236103257205e-08,
     2.2551972256065167e-08, 1.0516269278475652e-08, 5.4018402632724658e-09,
     3.0225464506438445e-09, 1.8235645592849622e-09, 1.1751059837824481e-09,
     8.0175205346217653e-10, 5.7456584970938098e-10, 4.2940228771632933e-10,
     3.3256502243706668e-10, 2.6546713475480001e-10, 2.1739601222439858e-10,
     1.8192848133186552e-10, 1.5507351079171458e-10, 1.3427014350986576e-10,
     1.1782651689253910e-10, 1.0459505341398556e-10, 9.3778511452873383e-11,
     8.4810860785115332e-11, 7.7282196651594484e-11, 7.0890272167839614e-11,
     6.5408519674283735e-11, 6.0664520714401682e-11, 5.6525237583268690e-11,
     5.2886706497188613e-11, 4.9666728477463493e-11, 4.6799608517694989e-11,
     4.4232316394818927e-11, 4.1921648711706248e-11, 3.9832105769737766e-11,
     3.7934285348355538e-11},
    // x = 20.4000
    {1.9621393925376049e-01, 4.8091651439247035e-03, 3.5361505024856664e-04,
     4.3335143862996132e-05, 7.4349172894211229e-06, 1.6400214208872310e-06,
     4.4212877444040330e-07, 1.4084050086113146e-07, 5.1745732360928732e-08,
     2.1526858273154625e-08, 9.9908988872269210e-09, 5.1085108833501414e-09,
     2.8459342579895514e-09, 1.7099687220259621e-09, 1.0977334045005259e-09,
     7.4638814067244348e-10, 5.3324509239623061e-10, 3.9743763377441357e-10,
     3.0707560272119946e-10, 2.4461188013737355e-10, 1.9995663564406940e-10,
     1.6707327133155173e-10, 1.4221858029846053e-10, 1.2299518437135255e-10,
     1.0782208515622632e-10, 9.5628666215087137e-11, 8.5672288869850740e-11,
     7.7426439191729096e-11, 7.0510332462391282e-11, 6.4643538217567833e-11,
     5.9616082445023502e-11, 5.5268344070267210e-11, 5.1477281503610402e-11,
     4.8146831045469080e-11, 4.5201105121737173e-11, 4.2579501527459814e-11,
     4.0233137680639132e-11, 3.8122217147234763e-11, 3.6214061151054943e-11,
     3.4481620528228186e-11},
    // x = 20.5000
    {1.9573478362293809e-01, 4.7740190822630540e-03, 3.4931843894234855e-04,
     4.2599779135582356e-05, 7.2731025316148780e-06, 1.5965042105284756e-06,
     4.2829990641333754e-07, 1.3577191781724396e-07, 4.9642161326640777e-08,
     2.0552843650890400e-08, 9.4939969878178245e-09, 4.8322874116533549e-09,
     2.6803038439424492e-09, 1.6038400788335238e-09, 1.0256958356614248e-09,
     6.9500064311694025e-10, 4.9499675781069310e-10, 3.6792049125283242e-10,
     2.8358693481615595e-10, 2.2542838345880552e-10, 1.8394034362211410e-10,
     1.5344881029560829e-10, 1.3044258478840043e-10, 1.1267715729490922e-10,
     9.8674964060341381e-11, 8.7437082257804596e-11, 7.8271666555153541e-11,
     7.0688913683814563e-11, 6.4335058200562500e-11, 5.8949889049885862e-11,
     5.4338794818451956e-11, 5.0353990671678834e-11, 4.6881671851927639e-11,
     4.3833068389964864e-11, 4.1138115018070436e-11, 3.8740904142930831e-11,
     3.6596373847837162e-11, 3.4667864012326887e-11, 3.2925291086286685e-11,
     3.1343769445300679e-11},
    // x = 20.6000
    {1.9525912123711545e-01, 4.7392990316973381e-03, 3.4509456223075056e-04,
     4.1880380581763619e-05, 7.1155711865838238e-06, 1.5543448906350152e-06,
     4.1496753912362608e-07, 1.3090890348338271e-07, 4.7633552600945846e-08,
     1.9627165270007472e-08, 9.0239066756885980e-09, 4.5721081334390518e-09,
     2.5249345140127805e-09, 1.5046645087026184e-09, 9.5861059813589994e-10,
     6.4729422947013216e-10, 4.5958582577190450e-10, 3.4065891161896151e-10,
     2.6193875764301009e-10, 2.0778031410242100e-10, 1.6922929995686365e-10,
     1.4095184967135389e-10, 1.1965399136156760e-10, 1.0323409027909968e-10,
     9.0311095906457630e-11, 7.9952878826329870e-11, 7.1514847775999775e-11,
     6.4541306804765887e-11, 5.8703562681807927e-11, 5.3760145172121878e-11,
     4.9530666829697884e-11, 4.5878290894173320e-11, 4.2697748411681765e-11,
     3.9907003761836992e-11, 3.7441363113384200e-11, 3.5249246676017212e-11,
     3.3289112189940928e-11, 3.1527186846944333e-11, 2.9935774799737137e-11,
     2.8491979801168479e-11},
    // x = 20.7000
    {1.9478690985621269e-01, 4.7049977978906778e-03, 3.4094184468921342e-04,
     4.1176526567813272e-05, 6.9621899139153395e-06, 1.5134948233487992e-06,
     4.0211158256616001e-07, 1.2624219890730247e-07, 4.5715203985791719e-08,
     1.8747220511128013e-08, 8.5790736983978924e-09, 4.3269808953767231e-09,
     2.3791551206740385e-09, 1.4119647202679117e-09, 8.9612340216121258e-10,
     6.0299613683371356e-10, 4.2679569188612818e-10, 3.1547631001166225e-10,
     2.4198387083692651e-10, 1.9154262374895821e-10, 1.5571554899637583e-10,
     1.2948789640318834e-10, 1.0976910501395377e-10, 9.4591090045132980e-11,
     8.2662865564295604e-11, 7.3114536591571787e-11, 6.5345477497844098e-11,
     5.8931683807399163e-11, 5.3567729749940597e-11, 4.9029516859594204e-11,
     4.5149828428904371e-11, 4.1801954985597995e-11, 3.8888516095006285e-11,
     3.6333694406183529e-11, 3.4077751870891099e-11, 3.2073098582900164e-11,
     3.0281434821895496e-11, 2.8671645996111214e-11, 2.7218233138862973e-11,
     2.5900129322054072e-11},
    // x = 20.8000
    {1.9431810795177618e-01, 4.6711083419625037e-03, 3.3685875239787224e-04,
     4.0487808554166804e-05, 6.8128301380563828e-06, 1.4739073342905134e-06,
     3.8971284954744202e-07, 1.2176300259843003e-07, 4.3882665936403690e-08,
     1.7910557329250139e-08, 8.1580397411946090e-09, 4.0959783303612984e-09,
     2.2423405186599291e-09, 1.3252975227029199e-09, 8.3790618007024180e-10,
     5.6185440384567841e-10, 3.9642669464325143e-10, 2.9221021397044525e-10,
     2.2358705449300026e-10, 1.7660060082173637e-10, 1.4330017812478236e-10,
     1.1897046348700255e-10, 1.0071139201645037e-10, 8.6679726410660874e-11,
     7.5668536520295433e-11, 6.6865919890329403e-11, 5.9712160873798728e-11,
     5.3812704429196177e-11, 4.8883719268005147e-11, 4.4717210966815447e-11,
     4.1158159254455168e-11, 3.8089223376562340e-11, 3.5420313717948387e-11,
     3.3081355038698938e-11, 3.1017181863847527e-11, 2.9183882849728627e-11,
     2.7546145679662236e-11, 2.6075303186504395e-11, 2.4747877810842727e-11,
     2.3544484840820569e-11},
    // x = 20.9000
    {1.9385267469163833e-01, 4.6376237764027706e-03, 3.3284379165564077e-04,
     3.9813830628604716e-05, 6.6673678564044499e-06, 1.4355376245242365e-06,
     3.7775301117798208e-07, 1.1746294599968467e-07, 4.2131726972926357e-08,
     1.7114864976382777e-08, 7.7594361728307369e-09, 3.8782334235047259e-09,
     2.1139082826234624e-09, 1.2442513047521194e-09, 7.8365508854926521e-10,
     5.2363624499953925e-10, 3.6829476239324805e-10, 2.7071111559878559e-10,
     2.0662407940214232e-10, 1.6284900504620388e-10, 1.3189255191038284e-10,
     1.0932038520042469e-10, 9.2410862121299381e-11, 7.9437470538529332e-11,
     6.9271736563706505e-11, 6.1155802982623765e-11, 5.4568024525499134e-11,
     4.9141214662763448e-11, 4.4611585634067892e-11, 4.0786073066840346e-11,
     3.7520951770230368e-11, 3.4707548589380580e-11, 3.2262512779498474e-11,
     3.0121071859316863e-11, 2.8232279608284414e-11, 2.6555616412049998e-11,
     2.5058522447549643e-11, 2.3714584012516995e-11, 2.2502183588296797e-11,
     2.1403483477294577e-11},
    // x = 21.0000
    {1.9339056992497949e-01, 4.6045373611124625e-03, 3.2889550774003199e-04,
     3.9154209058193267e-05, 6.5256834559835720e-06, 1.3983426868526033e-06,
     3.6621455474609150e-07, 1.1333407037277136e-07, 4.0458399989256644e-08,
     1.6357965327965991e-08, 7.3819782187752992e-09, 3.6729353940830978e-09,
     1.9933156671695251e-09, 1.1684437056296888e-09, 7.3308866688596223e-10,
     4.8812655468813575e-10, 3.4223017029859576e-10, 2.5084141850148731e-10,
     1.9098080011335390e-10, 1.5019127527149781e-10, 1.2140961173326724e-10,
     1.0046519138744690e-10, 8.4803504449262600e-11, 7.2807182319657776e-11,
     6.3420988719826852e-11, 5.5937438201912575e-11, 4.9870316797770302e-11,
     4.4877874940253254e-11, 4.0714930450541456e-11, 3.7202261735468414e-11,
     3.4206604752415390e-11, 3.1627305883479798e-11, 2.9387243520667731e-11,
     2.7426542525052467e-11, 2.5698150152079357e-11, 2.4164674231006863e-11,
     2.2796091133578276e-11, 2.1568062141905915e-11, 2.0460681377422540e-11,
     1.9457533887389174e-11},
    // x = 21.1000
    {1.9293175416777980e-01, 4.5718424995659090e-03, 3.2501248371088353e-04,
     3.8508571859146392e-05, 6.3876615382839982e-06, 1.3622812262112800e-06,
     3.5508074383611625e-07, 1.0936880501017874e-07, 3.8858909400775159e-08,
     1.5637804771876972e-08, 7.0244595314145294e-09, 3.4793258701359188e-09,
     1.8800567908330993e-09, 1.0975194628165434e-09, 6.8594613876960509e-10,
     4.5512653043481494e-10, 3.1807639818743887e-10, 2.3247447156910272e-10,
     1.7655232381396336e-10, 1.3853880429258989e-10, 1.1177523524743287e-10,
     9.2338535667663603e-11, 7.7830772363511626e-11, 6.6736642568452056e-11,
     5.8069283430045112e-11, 5.1168162277319126e-11, 4.5580043511230555e-11,
     4.0986821472177169e-11, 3.7160586279630280e-11, 3.3934952084587291e-11,
     3.1186344384391057e-11, 2.8821530035035818e-11, 2.6769145787628350e-11,
     2.4973839721007078e-11, 2.3392152164410552e-11, 2.1989574866705816e-11,
     2.0738421221961203e-11, 1.9616263251986541e-11, 1.8604770235299571e-11,
     1.7688835738121080e-11},
    // x = 21.2000
    {1.9247618858864865e-01, 4.5395327350905915e-03, 3.2119333925623199e-04,
     3.7876558383811758e-05, 6.2531907518665407e-06, 1.3273135839447621e-06,
     3.4433558056253394e-07, 1.0555994669273588e-07, 3.7329679076653243e-08,
     1.4952446621749170e-08, 6.6857471290635325e-09, 3.2966953341838168e-09,
     1.7736600270114955e-09, 1.0311484229798825e-09, 6.4198584621501104e-10,
     4.2445240564374879e-10, 2.9568908099094007e-10, 2.1549368235284567e-10,
     1.6324224861814238e-10, 1.2781027495120072e-10, 1.0291965052255814e-10,
     8.4879672888785923e-11, 7.1439113048996091e-11, 6.1178126091142992e-11,
     5.3173688333569828e-11, 4.6809037438786820e-11, 4.1661636048060547e-11,
     3.7435357773746403e-11, 3.3918328693766346e-11, 3.0956065910479270e-11,
     2.8433970466460259e-11, 2.6265675884757451e-11, 2.4385142448098148e-11,
     2.2741196183632561e-11, 2.1293693475542884e-11, 2.0010785586605999e-11,
     1.8866939521430611e-11, 1.7841486889719181e-11, 1.6917546598787018e-11,
     1.6081215735052772e-11},
    // x = 21.3000
    {1.9202383499502093e-01, 4.5076017472603525e-03, 3.1743672957866586e-04,
     3.7257818924024329e-05, 6.1221636323539393e-06, 1.2934016657580517e-06,
     3.3396376979920472e-07, 1.0190064031683614e-07, 3.5867321006105638e-08,
     1.4300064019045868e-08, 6.3647766776861439e-09, 3.1243798201635509e-09,
     1.6736855861812188e-09, 9.6902370333541637e-10, 6.0098380510832191e-10,
     3.9593428299764318e-10, 2.7493504414284529e-10, 1.9979170338667051e-10,
     1.5096196540121450e-10, 1.1793105228646031e-10, 9.4778900157986424e-11,
     7.8032981550134182e-11, 6.5579537892348375e-11, 5.6088012035538690e-11,
     4.8694992329309664e-11, 4.2824524218179434e-11, 3.8082648889329157e-11,
     3.4193673678232792e-11, 3.0960614081245894e-11, 2.8240026060781419e-11,
     2.5925625472872634e-11, 2.3937400502429642e-11, 2.2214233314820192e-11,
     2.0708809225858787e-11, 1.9384045204009533e-11, 1.8210545226172139e-11,
     1.7164761000386708e-11, 1.6227644614707274e-11, 1.5383649146979105e-11,
     1.4619978682527542e-11},
    // x = 21.4000
    {1.9157465581970801e-01, 4.4760433483979865e-03, 3.1374134432055452e-04,
     3.6652014330106235e-05, 5.9944764494513571e-06, 1.2605088731503900e-06,
     3.2395068529039199e-07, 9.8384360620317430e-08, 3.4468624651384366e-08,
     1.3678933290841671e-08, 6.0605480911327367e-09, 2.9617578431840393e-09,
     1.5797232749467827e-09, 9.1085999177941538e-10, 5.6273237272274831e-10,
     3.6941506036096024e-10, 2.5569141695512091e-10, 1.8526968537402803e-10,
     1.3963001781729877e-10, 1.0883262695629581e-10, 8.7294351069741718e-11,
     7.1747527297869637e-11, 6.0207230100415948e-11, 5.1426428247853384e-11,
     4.4597380891086880e-11, 3.9182183134408068e-11, 3.4813483980005121e-11,
     3.1234589379198138e-11, 2.8262340895918321e-11, 2.5763532840111048e-11,
     2.3639584393830846e-11, 2.1816365011988055e-11, 2.0237307715996645e-11,
     1.8858662991336256e-11, 1.7646172778246144e-11, 1.6572703181971819e-11,
     1.5616535056294213e-11, 1.4760112934554013e-11, 1.3989117864040914e-11,
     1.3291772196742211e-11},
    // x = 21.5000
    {1.9112861410779317e-01, 4.4448514801834335e-03, 3.1010590652662230e-04,
     3.6058815644827286e-05, 5.8700290606570499e-06, 1.2286000381482690e-06,
     3.1428233753704863e-07, 9.5004894940552725e-08, 3.3130546943498256e-08,
     1.3087427732595470e-08, 5.7721214274755495e-09, 2.8082475450958444e-09,
     1.4913904185892831e-09, 8.5639197504341466e-10, 5.2703901833272033e-10,
     3.4474944171589882e-10, 2.3784481756575893e-10, 1.7183659166987942e-10,
     1.2917151559489646e-10, 1.0045210556153339e-10, 8.0412246024059841e-11,
     6.5976663933051712e-11, 5.5281186308278907e-11, 4.7156926651333024e-11,
     4.0848139877847032e-11, 3.5852402701145922e-11, 3.1827139532400227e-11,
     2.8533322263999609e-11, 2.5800632247784800e-11, 2.3505360471128025e-11,
     2.1556063486774946e-11, 1.9884054298326540e-11, 1.8436974021612605e-11,
     1.7174366826502112e-11, 1.6064582314195569e-11, 1.5082573065447984e-11,
     1.4208305808873820e-11, 1.3425599678664558e-11, 1.2721266000804367e-11,
     1.2084463818528117e-11},
    // x = 21.6000
    {1.9068567350386137e-01, 4.4140202103639262e-03, 3.0652917164240834e-04,
     3.5477903751673662e-05, 5.7487247713420419e-06, 1.1976413611652534e-06,
     3.0494534335829540e-07, 9.1756326942555985e-08, 3.1850202879604569e-08,
     1.2524011787357759e-08, 5.4986130606568276e-09, 2.6633040401474763e-09,
     1.4083299348141096e-09, 8.0537288497523888e-10, 4.9372518877104255e-10,
     3.2180302627170861e-10, 2.2129060358797584e-10, 1.5940856895788856e-10,
     1.1951759662277043e-10, 9.2731743880094701e-11, 7.4083293331048123e-11,
     6.0677668684040719e-11, 5.0763889212762405e-11, 4.3246186924812328e-11,
     3.7417385325087152e-11, 3.2808151423769798e-11, 2.9098981092357518e-11,
     2.6067274501678999e-11, 2.3554637920600226e-11, 2.1446171806754590e-11,
     1.9657046230928361e-11, 1.8123612978338288e-11, 1.6797404588261500e-11,
     1.5641008305893188e-11, 1.4625180950750670e-11, 1.3726799680078213e-11,
     1.2927385135720527e-11, 1.2212022580676150e-11, 1.1568563752969678e-11,
     1.0987029387876263e-11},
    // x = 21.7000
    {1.9024579823955284e-01, 4.3835437295625355e-03, 3.0300992654720731e-04,
     3.4908969036802520e-05, 5.6304702008942449e-06, 1.1676003518258257e-06,
     2.9592689703402573e-07, 8.8633021258870349e-08, 3.0624856683771752e-08,
     1.1987235594856190e-08, 5.2391921081775677e-09, 2.5264169461835896e-09,
     1.3302085473470372e-09, 7.5757315383213812e-10, 4.6262526143102738e-10,
     3.0045146945019964e-10, 2.0593218307822692e-10, 1.4790836945208186e-10,
     1.1060493372031632e-10, 8.5618519361533868e-11, 6.8262291422550381e-11,
     5.5811408777959254e-11, 4.6621008527444993e-11, 3.9663745999398394e-11,
     3.4277816926511300e-11, 3.0024751659859888e-11, 2.6606532886905965e-11,
     2.3815839530988064e-11, 2.1505354074728332e-11, 1.9568349652036424e-11,
     1.7926124933387403e-11, 1.6519696165334011e-11, 1.5304194724767454e-11,
     1.4245019579464338e-11, 1.3315149872418834e-11, 1.2493238106814773e-11,
     1.1762237286237539e-11, 1.1108398952401342e-11, 1.0520532582234154e-11,
     9.9894516467948199e-12},
    // x = 21.8000
    {1.8980895312143126e-01, 4.3534163481817618e-03, 2.9954698862015783e-04,
     3.4351711064091441e-05, 5.5151751546384812e-06, 1.1384457725996345e-06,
     2.8721474294022329e-07, 8.5629608986712360e-08, 2.9451913495377114e-08,
     1.1475729885760732e-08, 4.9930770969543774e-09, 2.3971080879311685e-09,
     1.2567151289030702e-09, 7.1277917019114130e-10, 4.3358557781921633e-10,
     2.8057970996592391e-10, 1.9168038088762959e-10, 1.3726482034515571e-10,
     1.0237528233572156e-10, 7.9063739545383174e-11, 6.2907785321719181e-11,
     5.1342036604300441e-11, 4.2821127792246111e-11, 3.6381751110277209e-11,
     3.1404493117890859e-11, 2.7479673406735409e-11, 2.4329287649094450e-11,
     2.1760225760527916e-11, 1.9635459050138682e-11, 1.7855843202177002e-11,
     1.6348356575354495e-11, 1.5058333689303552e-11, 1.3944234407870044e-11,
     1.2974055831847879e-11, 1.2122828864468314e-11, 1.1370843793700156e-11,
     1.0702374016277943e-11, 1.0104745434525661e-11, 9.5676492051930131e-12,
     9.0826281320386238e-12},
    // x = 21.9000
    {1.8937510351915646e-01, 4.3236324933989155e-03, 2.9613920483819280e-04,
     3.3805838262718332e-05, 5.4027525012586057e-06, 1.1101475851010530e-06,
     2.7879714959384179e-07, 8.2740973991292896e-08, 2.8328911551782636e-08,
     1.0988201198150477e-08, 4.7595328507658709e-09, 2.2749293599262434e-09,
     1.1875591638582127e-09, 6.7079212774295539e-10, 4.0646355131840038e-10,
     2.6208125769223151e-10, 1.7845285587089743e-10, 1.2741233658506273e-10,
     9.4775065734322528e-11, 7.3022683352878262e-11, 5.7981752270511053e-11,
     4.7236710999476877e-11, 3.9335494786841050e-11, 3.3374734340045209e-11,
     2.8774625860005898e-11, 2.5152346251744789e-11, 2.2248533276829159e-11,
     1.9883295944248379e-11, 1.7929163826935904e-11, 1.6294028237933328e-11,
     1.4910131614707742e-11, 1.3726806556696464e-11, 1.2705591592648665e-11,
     1.1816884753614021e-11, 1.1037611351102107e-11, 1.0349572646396834e-11,
     9.7382592826553733e-12, 9.1919869022825186e-12, 8.7012583643223357e-12,
     8.2582875054556884e-12},
    // x = 22.0000
    {1.8894421535395367e-01, 4.2941867062501558e-03, 2.9278545090462852e-04,
     3.3271067626734847e-05, 5.2931180554621498e-06, 1.0826768989170467e-06,
     2.7066288502905053e-07, 7.9962239967462958e-08, 2.7253514834151301e-08,
     1.0523427394801936e-08, 4.5378675839079504e-09, 2.1594607375631833e-09,
     1.1224693216969619e-09, 6.3142695984402617e-10, 3.8112684332958663e-10,
     2.4485753743797987e-10, 1.6617356480205650e-10, 1.1829047339047665e-10,
     8.7754994531358823e-11, 6.7454272463031494e-11, 5.3449314017530373e-11,
     4.3465342396178496e-11, 3.6137793494290535e-11, 3.0619406771731413e-11,
     2.6367393386011022e-11, 2.3023987877901094e-11, 2.0347194829228726e-11,
     1.8169420833232539e-11, 1.6372075830474933e-11, 1.4869579842049554e-11,
     1.3599054577137077e-11, 1.2513534543601590e-11, 1.1577406067272944e-11,
     1.0763286024678364e-11, 1.0049848962876344e-11, 9.4202902079904138e-12,
     8.8612226245551493e-12, 8.3618736887644525e-12, 7.9134935766008494e-12,
     7.5089135479857148e-12},
    // x = 22.1000
    {1.8851625508737008e-01, 4.2650736388001936e-03, 2.8948463040721873e-04,
     3.2747124426121787e-05, 5.1861904656412184e-06, 1.0560059228339427e-06,
     2.6280119343128243e-07, 7.7288758214841351e-08, 2.6223506147328859e-08,
     1.0080253461400589e-08, 4.3274301867856709e-09, 2.0503084256066646e-09,
     1.0611921329925059e-09, 5.9451135326081728e-10, 3.5745260242890378e-10,
     2.2881728415732332e-10, 1.5477226918987557e-10, 1.0984351521677313e-10,
     8.1269717690102690e-11, 6.2320770222834356e-11, 4.9278473484299069e-11,
     4.0000359777641210e-11, 3.3203935741041884e-11, 2.8094470535347591e-11,
     2.4163769332054675e-11, 2.1077447660320320e-11, 1.8609690497552389e-11,
     1.6604345829994468e-11, 1.4951074886089935e-11, 1.3570356511823654e-11,
     1.2403835373598986e-11, 1.1407973916403086e-11, 1.0549792897033352e-11,
     9.8039599009648409e-12, 9.1507647665186716e-12, 8.5746890996666995e-12,
     8.0633804387288020e-12, 7.6069063652191275e-12, 7.1972051291155788e-12,
     6.8276761144201758e-12},
    // x = 22.2000
    {1.8809118971030947e-01, 4.2362880513947552e-03, 2.8623567400454806e-04,
     3.2233741928838049e-05, 5.0818911062958332e-06, 1.0301079183407942e-06,
     2.5520177295985353e-07, 7.4716096084388737e-08, 2.5236780606635017e-08,
     9.6575875671522315e-09, 4.1276076901949597e-09, 1.9471031342986034e-09,
     1.0034907603070206e-09, 5.5988483505494037e-10, 3.3532676160466743e-10,
     2.1387598547971163e-10, 1.4418408150078611e-10, 1.0202009815967980e-10,
     7.5277502350284587e-11, 5.7587505759557931e-11, 4.5439873724923458e-11,
     3.6816497555984352e-11, 3.0511870803404734e-11, 2.5780447182816567e-11,
     2.2146366809824811e-11, 1.9297064019037357e-11, 1.7021800309341178e-11,
     1.5175069485561365e-11, 1.3654200235804463e-11, 1.2385294644249941e-11,
     1.1314190377806070e-11, 1.0400524365090374e-11, 9.6137545892913688e-12,
     8.9304440781031830e-12, 8.3323743692963587e-12, 7.8052139694885254e-12,
     7.3375644278722678e-12, 6.9202673873540253e-12, 6.5458946580042176e-12,
     6.2083684105894721e-12},
    // x = 22.3000
    {1.8766898673233878e-01, 4.2078248099930464e-03, 2.8303753863969191e-04,
     3.1730661133398668e-05, 4.9801439749980187e-06, 1.0049571552937947e-06,
     2.4785475469400932e-07, 7.2240026057022435e-08, 2.4291339506197660e-08,
     9.2543973705479686e-09, 3.9378228959975469e-09, 1.8494984739243885e-09,
     9.4914385698072120e-10, 5.2739792703132885e-10, 3.1464338903419551e-10,
     1.9995536778185899e-10, 1.3434904758558003e-10, 9.4772863035950308e-11,
     6.9739940074831890e-11, 5.3222621162511702e-11, 4.1906577273928223e-11,
     3.3890600657181616e-11, 2.8041411420135697e-11, 2.3659520963797045e-11,
     2.0299296107123412e-11, 1.7668534312293496e-11, 1.5570546435116838e-11,
     1.3869732783417815e-11, 1.2470547627067761e-11, 1.1304312462379043e-11,
     1.0320752384744522e-11, 9.4824443172344969e-12, 8.7611011897985606e-12,
     8.1350380810795575e-12, 7.5874141818738457e-12, 7.1049932652138914e-12,
     6.6772565674759694e-12, 6.2957589799325164e-12, 5.9536557079412665e-12,
     5.6453500132447307e-12},
    // x = 22.4000
    {1.8724961417125635e-01, 4.1796788835775150e-03, 2.7988920678010960e-04,
     3.1237630511538168e-05, 4.8808755936859171e-06, 9.8052886963248202e-07,
     2.4075068264105483e-07, 6.9856515417314077e-08, 2.3385284545126531e-08,
     8.8697065542211665e-09, 3.7575321627714166e-09, 1.7571694593785058e-09,
     8.9794450731390501e-10, 4.9691136260725308e-10, 2.9530408822216860e-10,
     1.8698292233036165e-10, 1.2521176237050283e-10, 8.8058136611247971e-11,
     6.4621677306941830e-11, 4.9196839783798265e-11, 3.8653864141193053e-11,
     3.1201446244335531e-11, 2.5774074787137133e-11, 2.1715395700412645e-11,
     1.8608034818676572e-11, 1.6178796162331874e-11, 1.4244084067773248e-11,
     1.2677518248868253e-11, 1.1390175572434592e-11, 1.0318222535403738e-11,
     9.4149886505834068e-12, 8.6457738759576648e-12, 7.9843775924698305e-12,
     7.4107344970503249e-12, 6.9092761888323340e-12, 6.4677772095256255e-12,
     6.0765299962864730e-12, 5.7277466894247655e-12, 5.4151197246466472e-12,
     5.1334951061408611e-12},
    // x = 22.5000
    {1.8683304054291636e-01, 4.1518453416383673e-03, 2.7678968568278458e-04,
     3.0754405760536238e-05, 4.7840149140880304e-06, 9.5679922304444438e-07,
     2.3388049474881391e-07, 6.7561716487607098e-08, 2.2516812389374328e-08,
     8.5025915739356007e-09, 3.5862233358336633e-09, 1.6698111168943428e-09,
     8.4969924214018570e-10, 4.6829536136095833e-10, 2.7721744365520806e-10,
     1.7489146830532260e-10, 1.1672101611563308e-10, 8.1835638656763998e-11,
     5.9890168016116134e-11, 4.5483253874106321e-11, 3.5659046862858523e-11,
     2.8729580647015267e-11, 2.3692937234669867e-11, 1.9933164073302914e-11,
     1.7059309315193885e-11, 1.4815919204066394e-11, 1.3031601936574968e-11,
     1.1588558008376907e-11, 1.0404019959982603e-11, 9.4186521212776950e-12,
     8.5891262865304600e-12, 7.8832646937076746e-12, 7.2767974098237997e-12,
     6.7511564306007572e-12, 6.2919486353053227e-12, 5.8878814127678851e-12,
     5.5299952898890545e-12, 5.2111080866752954e-12, 4.9254069830918515e-12,
     4.6681454541458380e-12},
    // x = 22.6000
    {1.8641923485130060e-01, 4.1243193517303397e-03, 2.7373800668366739e-04,
     3.0280749564802701e-05, 4.6894932270877016e-06, 9.3374526448085206e-07,
     2.2723550486802172e-07, 6.5351957390033081e-08, 2.1684209548577614e-08,
     8.1521786077661835e-09, 3.4234138117869360e-09, 1.5871371856800381e-09,
     8.0422712424453371e-10, 4.4142895688772996e-10, 2.6029850843320267e-10,
     1.6361874977378160e-10, 1.0882946876547084e-10, 7.6068212709570578e-11,
     5.5515446682456227e-11, 4.2057129921746296e-11, 3.2901301153985796e-11,
     2.6457170188616608e-11, 2.1782501400941733e-11, 1.8299188235717479e-11,
     1.5640986555105620e-11, 1.3569006336553712e-11, 1.1923231601245392e-11,
     1.0593850001726945e-11, 9.5038162678547441e-12, 8.5979706300834792e-12,
     7.8360843448446214e-12, 7.1883161559197823e-12, 6.6321828105087987e-12,
     6.1505006171488088e-12, 5.7299620920580046e-12, 5.3601356098242025e-12,
     5.0327516263179334e-12, 4.7411861489659807e-12, 4.4800819996571191e-12,
     4.2450676810983972e-12},
    // x = 22.7000
    {1.8600816657883143e-01, 4.0970961770993514e-03, 2.7073322451051028e-04,
     2.9816431366336240e-05, 4.5972440758473045e-06, 9.1134489343038990e-07,
     2.2080738561339770e-07, 6.3223733305906143e-08, 2.0885847548514552e-08,
     7.8176406924827304e-09, 3.2686487284392179e-09, 1.5088789077355109e-09,
     7.6135889849967770e-10, 4.1619937393084944e-10, 2.4446833061897707e-10,
     1.5310706391891977e-10, 1.0149335012015721e-10, 7.0721578588850487e-11,
     5.1469919924634150e-11, 3.8895730202204579e-11, 3.0361510833801830e-11,
     2.4367864717059470e-11, 2.0028574819431312e-11, 1.6800990767467690e-11,
     1.4341975331268857e-11, 1.2428103638784105e-11, 1.0909964747722770e-11,
     9.6851816210998069e-12, 8.6820287032844490e-12, 7.8492235695118448e-12,
     7.1494119949300225e-12, 6.5549173052721602e-12, 6.0449097841579595e-12,
     5.6034856814733809e-12, 5.2183404102938507e-12, 4.8798370546012970e-12,
     4.5803423974739471e-12, 4.3137468939255827e-12, 4.0751130179084297e-12,
     3.8604144569950313e-12},
    // x = 22.8000
    {1.8559980567691839e-01, 4.0701711743767431e-03, 2.6777441661823132e-04,
     2.9361227143690252e-05, 4.5072031725208515e-06, 8.8957682486418401e-07,
     2.1458815207511317e-07, 6.1173698203851110e-08, 2.0120178381072494e-08,
     7.4981950350300812e-09, 3.1214992715945357e-09, 1.4347838996189598e-09,
     7.2093620198191619e-10, 3.9250145106770223e-10, 2.2965351530629989e-10,
     1.4330291804872836e-10, 9.4672183742322255e-11, 6.5764104882367342e-11,
     4.7728175219784493e-11, 3.5978149172617187e-11, 2.8022125808583036e-11,
     2.2446672747614509e-11, 1.8418158931726767e-11, 1.5427155067259236e-11,
     1.3152136125010992e-11, 1.1384118185632796e-11, 9.9835777764602942e-12,
     8.8550601160463237e-12, 7.9317856474093143e-12, 7.1660723912642010e-12,
     6.5232322417522972e-12, 5.9775939887326155e-12, 5.5098583427515726e-12,
     5.1053050749598557e-12, 4.7525561219654817e-12, 4.4427081481345485e-12,
     4.1687148608962788e-12, 3.9249408768584690e-12, 3.7068351952565373e-12,
     3.5106892362735846e-12},
    // x = 22.9000
    {1.8519412255673215e-01, 4.0435397913388821e-03, 2.6486068254596805e-04,
     2.8914919199094564e-05, 4.4193083183919773e-06, 8.6842055576896186e-07,
     2.0857014633513631e-07, 5.9198657009769313e-08, 1.9385730214786071e-08,
     7.1931004878171533e-09, 2.9815610908119736e-09, 1.3646151003867678e-09,
     6.8281082968467587e-10, 3.7023710551920083e-10, 2.1578581964368831e-10,
     1.3415671310763020e-10, 8.8328532685568545e-11, 6.1166599766075645e-11,
     4.4266805294217206e-11, 3.3285163461758508e-11, 2.5867032001552058e-11,
     2.0679847221094698e-11, 1.6939347623789837e-11, 1.4167234360983513e-11,
     1.2062198812854321e-11, 1.0428743067518872e-11, 9.1365630372481386e-12,
     8.0966491623723611e-12, 7.2468208426415314e-12, 6.5427397094556703e-12,
     5.9521906882180593e-12, 5.4513607564149391e-12, 5.0223672121537879e-12,
     4.6515842675218146e-12, 4.3284898818005137e-12, 4.0448579149942765e-12,
     3.7941834634737401e-12, 3.5712681986044786e-12, 3.3719171519178494e-12,
     3.1927142218379143e-12},
    // x = 23.0000
    {1.8479108808019945e-01, 4.0171975647300146e-03, 2.6199114329502704e-04,
     2.8477295953398671e-05, 4.3334993272824854e-06, 8.4785633319013154e-07,
     2.0274602274554623e-07, 5.7295558193386612e-08, 1.8681103350097431e-08,
     6.9016551772899850e-09, 2.8484528167867972e-09, 1.2981497903522985e-09,
     6.4684405177795293e-10, 3.4931483691590879e-10, 2.0280177826983735e-10,
     1.2562245159800573e-10, 8.2427765287416469e-11, 5.6902118655820093e-11,
     4.1064246883406186e-11, 3.0799094312369785e-11, 2.3881432214465278e-11,
     1.9054780966870830e-11, 1.5581234462139715e-11, 1.3011668575636270e-11,
     1.1063687537780135e-11, 9.5543889790045250e-12, 8.3620661220046065e-12,
     7.4037110467223000e-12, 6.6214198098414728e-12, 5.9739594096509862e-12,
     5.4314088880954193e-12, 4.9716770838435281e-12, 4.5781926079374715e-12,
     4.2383418086721128e-12, 3.9423935840002786e-12, 3.6827469778043447e-12,
     3.4533965023280540e-12, 3.2495467033245501e-12, 3.0673305746157002e-12,
     2.9036012593128212e-12},
    // x = 23.1000
    {1.8439067355121283e-01, 3.9911401181463094e-03, 2.5916494072696111e-04,
     2.8048151748516855e-05, 4.2497179520842273e-06, 8.2786512371063487e-07,
     1.9710873392836109e-07, 5.5461486747662177e-08, 1.8004966404507169e-08,
     6.6231942759723015e-09, 2.7218146735232100e-09, 1.2351786756988304e-09,
     6.1290597866605286e-10, 3.2964926709848212e-10, 1.9064235681726338e-10,
     1.1765746798699540e-10, 7.6937839413320537e-11, 5.2945787314678078e-11,
     3.8100632669097046e-11, 2.8503681429764964e-11, 2.2051736992627360e-11,
     1.7559911038880360e-11, 1.4333827878147320e-11, 1.1951708394267965e-11,
     1.0148852117513966e-11, 8.7541217977648288e-12, 7.6538286793194127e-12,
     6.7705539675861348e-12, 6.0503710284802635e-12, 5.4549312104312923e-12,
     4.9564418769263373e-12, 4.5344075279449561e-12, 4.1734707263870845e-12,
     3.8619539072231814e-12, 3.5908568199676194e-12, 3.3531557116085838e-12,
     3.1433058186744552e-12, 2.9568830749304304e-12, 2.7903225956349066e-12,
     2.6407253931263056e-12},
    // x = 23.2000
    {1.8399285070704832e-01, 3.9653631599791189e-03, 2.5638123698103860e-04,
     2.7627286657069590e-05, 4.1679078142741965e-06, 8.0842858429533105e-07,
     1.9165151745871236e-07, 5.3693657538768790e-08, 1.7356052713731531e-08,
     6.3570879088156776e-09, 2.6013071789463210e-09, 1.1755050343423305e-09,
     5.8087497037683205e-10, 3.1116071325500462e-10, 1.7925263132535635e-10,
     1.1022217982859361e-10, 7.1829103670211490e-11, 4.9274639153629106e-11,
     3.5357655301474774e-11, 2.6383967279439175e-11, 2.0365464644602137e-11,
     1.6184631164657310e-11, 1.3187973614044732e-11, 1.0979345867315171e-11,
     9.3106054175694909e-12, 8.0216056265585429e-12, 7.0061362621960529e-12,
     6.1919839980129756e-12, 5.5289214547871756e-12, 4.9812792787560425e-12,
     4.5232395055607258e-12, 4.1357854622977458e-12, 3.8046836150036491e-12,
     3.5191222109191517e-12, 3.2707763737714399e-12, 3.0531552888284589e-12,
     2.8611392482377386e-12, 2.6906465675165237e-12, 2.5383906934299682e-12,
     2.4017008406361449e-12},
    // x = 23.3000
    {1.8359759170998671e-01, 3.9398624814155538e-03, 2.5363921391040312e-04,
     2.7214506298930075e-05, 4.0880143362797504e-06, 7.8952903443441323e-07,
     1.8636788319537433e-07, 5.1989409005703236e-08, 1.6733156935868074e-08,
     6.1027391853146164e-09, 2.4866099280446702e-09, 1.1189439187734334e-09,
     5.5063708707495466e-10, 2.9377479190510277e-10, 1.6858149157385623e-10,
     1.0327985797308489e-10, 6.7074114228960496e-11, 4.5867465567803051e-11,
     3.2818442505944920e-11, 2.4426190957594174e-11, 1.8811149640560070e-11,
     1.4919211613023386e-11, 1.2135283803947043e-11, 1.0087251010333551e-11,
     8.5424661671588770e-12, 7.3510508174830858e-12, 6.4137707638533059e-12,
     5.6632612958830884e-12, 5.0527359909334008e-12, 4.5490145362045156e-12,
     4.1281112347638022e-12, 3.7723800691956795e-12, 3.4686281172372222e-12,
     3.2068444962306020e-12, 2.9793284796432777e-12, 2.7800813519165997e-12,
     2.6043755770387233e-12, 2.4484451293509463e-12, 2.3092598842327793e-12,
     2.1843591628723202e-12},
    // x = 23.4000
    {1.8320486913913150e-01, 3.9146339544945319e-03, 2.5093807253624272e-04,
     2.6809621664396949e-05, 4.0099846765671441e-06, 7.7114942952282088e-07,
     1.8125160122467429e-07, 5.0346197189839638e-08, 1.6135131846395791e-08,
     5.8595823494114760e-09, 2.3774204530458693e-09, 1.0653214119164940e-09,
     5.2208557773094024e-10, 2.7742205143167008e-10, 1.5858136650392622e-10,
     9.6796414366293039e-11, 6.2647466006062790e-11, 4.2704678246340291e-11,
     3.0467442357918036e-11, 2.2617690832726955e-11, 1.7378258679594125e-11,
     1.3754725845983195e-11, 1.1168072117648109e-11, 9.2687138680089480e-12,
     7.8385067419887780e-12, 6.7371665403013745e-12, 5.8719670359834209e-12,
     5.1800601845251166e-12, 4.6178605529387928e-12, 4.1545003257727643e-12,
     3.7676940801121212e-12, 3.4410662952028483e-12, 3.1623876138126820e-12,
     2.9223880048392773e-12, 2.7139435908736275e-12, 2.5315100747635637e-12,
     2.3707217742920263e-12, 2.2281037019652114e-12, 2.1008619952146414e-12,
     1.9867294307730757e-12},
    // x = 23.5000
    {1.8281465598241875e-01, 3.8896735302165375e-03, 2.4827703251932267e-04,
     2.6412448943726955e-05, 3.9337676673328179e-06, 7.5327333541593870e-07,
     1.7629669038572540e-07, 4.8761590075919178e-08, 1.5560885312607630e-08,
     5.6270810397468207e-09, 2.2734531555097975e-09, 1.0144739323292910e-09,
     4.9512040419882149e-10, 2.6203763103705230e-10, 1.4920797003768795e-10,
     9.0740206060933437e-11, 5.8525637056592737e-11, 3.9768182481669842e-11,
     2.8290317885731202e-11, 2.0946815224318654e-11, 1.6057113777007437e-11,
     1.2682983375068670e-11, 1.0279294444656918e-11, 8.5175915697975234e-12,
     7.1933054798171485e-12, 6.1751174953609076e-12, 5.3763733198793307e-12,
     4.7384327600963157e-12, 4.2206884165153122e-12, 3.7944211385594759e-12,
     3.4389234243815169e-12, 3.1389975033658472e-12, 2.8833063081483135e-12,
     2.6632651872460170e-12, 2.4722834322639492e-12, 2.3052363956837166e-12,
     2.1580922949886336e-12, 2.0276444958530380e-12, 1.9113168289340607e-12,
     1.8070202042307417e-12},
    // x = 23.6000
    {1.8242692562881302e-01, 3.8649772367053939e-03, 2.4565533164825839e-04,
     2.6022809362773355e-05, 3.8593137546827113e-06, 7.3588490410497650e-07,
     1.7149740734672122e-07, 4.7233262227073626e-08, 1.5009377436792445e-08,
     5.4047266533044674e-09, 2.1744383055780537e-09, 9.6624758533132176e-10,
     4.6964779815931453e-10, 2.4756094416275683e-10, 1.4042006573927315e-10,
     8.5081854172419016e-11, 5.4686845123818781e-11, 3.7041260584433850e-11,
     2.6273850232085628e-11, 1.9402840446746297e-11, 1.4838821778016168e-11,
     1.1696468292328817e-11, 9.4624946413007688e-12, 7.8282599440210117e-12,
     6.6019031328017989e-12, 5.6604844069183357e-12, 4.9230151536460245e-12,
     4.3347757127647582e-12, 3.8579295487567569e-12, 3.4657541266608569e-12,
     3.1390064400899941e-12, 2.8635805793378593e-12, 2.6289658262191183e-12,
     2.4272116356673723e-12, 2.2522201290036276e-12, 2.0992542245870208e-12,
     1.9645902636638940e-12, 1.8452690624305428e-12, 1.7389150462192367e-12,
     1.6436031580114242e-12},
    // x = 23.7000
    {1.8204165186068522e-01, 3.8405411774203842e-03, 2.4307222534392757e-04,
     2.5640529024486319e-05, 3.7865749411902550e-06, 7.1896885045835748e-07,
     1.6684823620370859e-07, 4.5758989697511978e-08, 1.4479617858154761e-08,
     5.1920368059596440e-09, 2.0801211039470341e-09, 9.2049755689333189e-10,
     4.4557984857389930e-10, 2.3393538456125925e-10, 1.3217924888207181e-10,
     7.9794076612579877e-11, 5.1110915376501702e-11, 3.4508464583982292e-11,
     2.4405849668683211e-11, 1.7975895603322437e-11, 1.3715209754974247e-11,
     1.0788282991775291e-11, 8.7117549051169214e-12, 7.1955692950731166e-12,
     6.0597630954565637e-12, 5.1892279648433794e-12, 4.5082624485910709e-12,
     3.9658000766145838e-12, 3.5265826597047802e-12, 3.1657431531514123e-12,
     2.8653978880197860e-12, 2.6124532705783021e-12, 2.3971639219965723e-12,
     2.2121660081430737e-12, 2.0518172236664714e-12, 1.9117384449196332e-12,
     1.7884903680484872e-12, 1.6793419983912318e-12, 1.5821026104545817e-12,
     1.4949982038229394e-12},
    // x = 23.8000
    {1.8165880884636612e-01, 3.8163615294171532e-03, 2.4052698617945081e-04,
     2.5265438756043205e-05, 3.7155047307290953e-06, 7.0251042997824373e-07,
     1.6234387857485783e-07, 4.4336645207472758e-08, 1.3970663204089141e-08,
     4.9885538838669598e-09, 1.9902608024374816e-09, 8.7708754734745869e-10,
     4.2283411746940049e-10, 2.2110805334636405e-10, 1.2444974459881820e-10,
     7.4851533527681601e-11, 4.7779158443488414e-11, 3.2155517461912497e-11,
     2.2675073817623008e-11, 1.6656893567310278e-11, 1.2678765791533999e-11,
     9.9520966387565544e-12, 8.0216503785388585e-12, 6.6148039829026561e-12,
     5.5627350787466970e-12, 4.7576559119854236e-12, 4.1287994551087067e-12,
     3.6285036485857003e-12, 3.2239097315993992e-12, 2.8918751523757072e-12,
     2.6157780786152232e-12, 2.3834635577194652e-12, 2.1858950984925253e-12,
     2.0162517637380675e-12, 1.8693124101337869e-12, 1.7410285472930543e-12,
     1.6282233069269954e-12, 1.5283761335627788e-12, 1.4394666506118972e-12,
     1.3598599716183720e-12},
    // x = 23.9000
    {1.8127837113287176e-01, 3.7924345416558590e-03, 2.3801890341518597e-04,
     2.4897373961385786e-05, 3.6460580754814101e-06, 6.8649541752392430e-07,
     1.5797924416472941e-07, 4.2964193565949384e-08, 1.3481614682011279e-08,
     4.7938436800228053e-09, 1.9046298793165629e-09, 8.3588924218715945e-10,
     4.0133328203353692e-10, 2.0902950547566573e-10, 1.1719822089417102e-10,
     7.0230684452497024e-11, 4.4674257927318155e-11, 2.9969222228031359e-11,
     2.1071152487018765e-11, 1.5437467634269753e-11, 1.1722584699475316e-11,
     9.1820979827800839e-12, 7.3872076183670174e-12, 6.0816454754166428e-12,
     5.1070219297695005e-12, 4.3623930011946734e-12, 3.7815973631658607e-12,
     3.3201458398784619e-12, 2.9474128052660089e-12, 2.6418585941121839e-12,
     2.3880328023225696e-12, 2.1746508753154062e-12, 1.9933329716165365e-12,
     1.8377605450517720e-12, 1.7031018288047815e-12, 1.5856137465675673e-12,
     1.4823616507258824e-12, 1.3910190669746819e-12, 1.3097226134205114e-12,
     1.2369655257694496e-12},
    // x = 24.0000
    {1.8090031363879586e-01, 3.7687565333550930e-03, 2.3554728254820693e-04,
     2.4536174478951856e-05, 3.5781913250274490e-06, 6.7091008695628322e-07,
     1.5374944177445146e-07, 4.1639687327550543e-08, 1.3011615803496151e-08,
     4.6074941107081614e-09, 1.8230132657919225e-09, 7.9678181742057442e-10,
     3.8100480115063368e-10, 1.9765351423589690e-10, 1.1039361539430051e-10,
     6.5909656270665055e-11, 4.1780166644746377e-11, 2.7937378204871656e-11,
     1.9584518577660766e-11, 1.4309913373555362e-11, 1.0840318252622252e-11,
     8.4729521440566998e-12, 6.8038665989926518e-12, 5.5921385731641320e-12,
     4.6891493228317371e-12, 4.0003535703325886e-12, 3.4638893050868975e-12,
     3.0382247443086410e-12, 2.6948128227955054e-12, 2.4136038636781862e-12,
     2.1802350523796286e-12, 1.9842290156743009e-12, 1.8178142196810474e-12,
     1.6751370590932732e-12, 1.5517257815928856e-12, 1.4441194476482924e-12,
     1.3496069862549640e-12, 1.2660409282046108e-12, 1.1917025869282245e-12,
     1.1252032031392220e-12},
    // x = 24.1000
    {1.8052461164736411e-01, 3.7453238923901510e-03, 2.3311144487575978e-04,
     2.4181684444397698e-05, 3.5118621774264304e-06, 6.5574119165991569e-07,
     1.4964977073504424e-07, 4.0361262670656386e-08, 1.2559850232985640e-08,
     4.4291140068634445e-09, 1.7452076203416518e-09, 7.5965147712109645e-10,
     3.6178060464413638e-10, 1.8693685241007411e-10, 1.0400697479531456e-10,
     6.1868121143698821e-11, 3.9082010902838706e-11, 2.6048703938917812e-11,
     1.8206344562744497e-11, 1.3267135246549953e-11, 1.0026129557626757e-11,
     7.8197610357854452e-12, 6.2674459460775027e-12, 5.1426605318599378e-12,
     4.3059380717754895e-12, 3.6687165057917112e-12, 3.1731475489115913e-12,
     2.7804562272712819e-12, 2.4640303436116572e-12, 2.2052053868885057e-12,
     1.9906283794311894e-12, 1.8105705644702074e-12, 1.6578239755971890e-12,
     1.5269653210783973e-12, 1.4138557380583850e-12, 1.3152949375143167e-12,
     1.2287782287484969e-12, 1.1523232522987928e-12, 1.0843446884765777e-12,
     1.0235624704327375e-12},
    // x = 24.2000
    {1.8015124079964706e-01, 3.7221330737342742e-03, 2.3071072707220316e-04,
     2.3833752158116288e-05, 3.4470296322030267e-06, 6.4097594590164769e-07,
     1.4567571274236858e-07, 3.9127135484779088e-08, 1.2125539753805828e-08,
     4.2583319757689044e-09, 1.6710206477711795e-09, 7.2439102098575808e-10,
     3.4359680361959393e-10, 1.7683908890507778e-10, 9.8011306052748284e-11,
     5.8087183637306123e-11, 3.6566002175049721e-11, 2.4292766204391688e-11,
     1.6928484084083168e-11, 1.2302597595599051e-11, 9.2746512137247865e-12,
     7.2180271131949004e-12, 5.7741111236218466e-12, 4.7298928322086744e-12,
     3.9544788353248866e-12, 3.3649023852652121e-12, 2.9070626893479872e-12,
     2.5447548564330596e-12, 2.2531679673498959e-12, 2.0149253441086960e-12,
     1.8176117317836748e-12, 1.6521927304694412e-12, 1.5119825317250315e-12,
     1.3919561379761132e-12, 1.2882825157090164e-12, 1.1980021921454334e-12,
     1.1188009950050091e-12, 1.0488488667620278e-12, 9.8668341989812320e-13,
     9.3112470760479549e-13},
    // x = 24.3000
    {1.7978017708792635e-01, 3.6991805979415520e-03, 2.2834448077896651e-04,
     2.3492229957365180e-05, 3.3836539451578535e-06, 6.2660200698633983e-07,
     1.4182292407335753e-07, 3.7935597655742564e-08, 1.1707942344682696e-08,
     4.0947953287046617e-09, 1.6002704600993678e-09, 6.9089943986682015e-10,
     3.2639342041929199e-10, 1.6732239971035039e-10, 9.2381438428398996e-11,
     5.4549276334656406e-11, 3.4219355592902863e-11, 2.2659914609187758e-11,
     1.5743418246137903e-11, 1.1410279640901915e-11, 8.5809469431256266e-12,
     6.6636201668494149e-12, 5.3203453213015565e-12, 4.3507953682983686e-12,
     3.6321090072171648e-12, 3.0865526089376933e-12, 2.6635246604229041e-12,
     2.3292165111758176e-12, 2.0604943118395212e-12, 1.8411788315963701e-12,
     1.6597256482493122e-12, 1.5077444441139693e-12, 1.3790332390928161e-12,
     1.2689357197893777e-12, 1.1739055281197366e-12, 1.0912056971397691e-12,
     1.0186979411039957e-12, 9.5469269838112493e-13, 8.9784090163199802e-13,
     8.4705483264185848e-13},
    // x = 24.4000
    {1.7941139684921101e-01, 3.6764630496701946e-03, 2.2601207220707175e-04,
     2.3156974092825308e-05, 3.3216965849240475e-06, 6.1260745817280342e-07,
     1.3808722816428732e-07, 3.6785013537953274e-08, 1.1306350360362377e-08,
     3.9381690705439606e-09, 1.5327849765721340e-09, 6.5908153738523969e-10,
     3.1011413680830713e-10, 1.5835139214329017e-10, 8.7093895578969572e-11,
     5.1238063282274552e-11, 3.2030214715889186e-11, 2.1141221352648034e-11,
     1.4644206223925455e-11, 1.0584634153076288e-11, 7.9404764007142681e-12,
     6.1527469019296325e-12, 4.9029228108167067e-12, 4.0025828457301806e-12,
     3.3363916011106477e-12, 2.8315103450676809e-12, 2.4406054094945243e-12,
     2.1321025222969438e-12, 1.8844294080166588e-12, 1.6825203410211188e-12,
     1.5156396824821134e-12, 1.3759946109895497e-12, 1.2578314933857103e-12,
     1.1568353166998402e-12, 1.0697230041756226e-12, 9.9396319002047280e-13,
     9.2757997697407134e-13, 8.6901341606533030e-13, 8.1701890552829780e-13,
     7.7059369079016355e-13},
    // x = 24.5000
    {1.7904487675889888e-01, 3.6539770762449284e-03, 2.2371288175178442e-04,
     2.2827844609419872e-05, 3.2611201911957261e-06, 5.9898079231455256e-07,
     1.3446460853289020e-07, 3.5673816603655434e-08, 1.0920088810334181e-08,
     3.7881349474943803e-09, 1.4684013602844234e-09, 6.2884757586768248e-10,
     2.9470605911224969e-10, 1.4989294141530177e-10, 8.2126776933810040e-11,
     4.8138350665796633e-11, 2.9987582085372256e-11, 1.9728425721649570e-11,
     1.3624439832679166e-11, 9.8205494969933393e-12, 7.3490628964547890e-12,
     5.6819230673100187e-12, 4.5188845599566406e-12, 3.6827031988080249e-12,
     3.0650959568882004e-12, 2.5978031312462519e-12, 2.2365430864714969e-12,
     1.9518252066639567e-12, 1.7235313859196357e-12, 1.5376314396115057e-12,
     1.3841409485841505e-12, 1.2558214164730137e-12, 1.1473347098233548e-12,
     1.0546817894298505e-12, 9.7482309051723483e-13, 9.0541724059660910e-13,
     8.4463827807966677e-13, 7.9104583353796407e-13, 7.4349161344677134e-13,
     7.0105113834584102e-13},
    // x = 24.6000
    {1.7868059382458065e-01, 3.6317193862573632e-03, 2.2144630361897784e-04,
     2.2504705231229917e-05, 3.2018885345575551e-06, 5.8571089619187677e-07,
     1.3095120202709321e-07, 3.4600506259645424e-08, 1.0548513730020816e-08,
     3.6443905494450530e-09, 1.4069654890609512e-09, 6.0011294497200748e-10,
     2.8011949911989757e-10, 1.4191603862480145e-10, 7.7459647666919312e-11,
     4.5236004160232740e-11, 2.8081255107926958e-11, 1.8413882945211004e-11,
     1.2678201736284059e-11, 9.1133147679939284e-12, 6.8028637864746799e-12,
     5.2479479184604281e-12, 4.1655159113127950e-12, 3.3888178526802212e-12,
     2.8161801101374303e-12, 2.3836269865757164e-12, 2.0497276150031029e-12,
     1.7869346735356823e-12, 1.5764843371393016e-12, 1.4053095438969786e-12,
     1.2641236876567661e-12, 1.1462025871921515e-12, 1.0465931978840800e-12,
     9.6158902856211940e-13, 8.8837575724498273e-13, 8.2478759334745868e-13,
     7.6913702175733234e-13, 7.2009400266415408e-13, 6.7659903541874927e-13,
     6.3779975752103705e-13},
    // x = 24.7000
    {1.7831852537998139e-01, 3.6096867482031292e-03, 2.1921174546280615e-04,
     2.2187423250349546e-05, 3.1439664778478420e-06, 5.7278703550338525e-07,
     1.2754329238408687e-07, 3.3563644822465267e-08, 1.0191010639142382e-08,
     3.5066484636047615e-09, 1.3483314584037496e-09, 5.7279785148043519e-10,
     2.6630776964989397e-10, 1.3439164935024508e-10, 7.3073436621687731e-11,
     4.2517872442989900e-11, 2.6301766849948262e-11, 1.7190517058766199e-11,
     1.1800026998209996e-11, 8.4585877649499403e-12, 6.2983433094697328e-12,
     4.8478808166602506e-12, 3.8403261492135215e-12, 3.1187836714446496e-12,
     2.5877746804181338e-12, 2.1873319027377695e-12, 1.8786875241778028e-12,
     1.6361067902060890e-12, 1.4420872493136059e-12, 1.2844576895877076e-12,
     1.1545797639763552e-12, 1.0462065234274297e-12, 9.5474185487217124e-13,
     8.7675014618316475e-13, 8.0962543419087693e-13, 7.5136420271108753e-13,
     7.0040678233288383e-13, 6.5552493451534074e-13, 6.1574102719448679e-13,
     5.8026914471119086e-13},
    // x = 24.8000
    {1.7795864907903677e-01, 3.5878759891547173e-03, 2.1700862803430195e-04,
     2.1875869419530987e-05, 3.0873199389908051e-06, 5.6019884048678017e-07,
     1.2423730408430725e-07, 3.2561854643609915e-08, 9.8469930822823967e-09,
     3.3746354763277999e-09, 1.2923611144614181e-09, 5.4682702884572464e-10,
     2.5322699376027864e-10, 1.2729258207792593e-10, 6.8950341596011266e-11,
     3.9971716398033980e-11, 2.4640331358814777e-11, 1.6051777457642085e-11,
     1.0984867703137739e-11, 7.8523655660132193e-12, 5.8322476639652194e-12,
     4.4790197838857542e-12, 3.5410297936541698e-12, 2.8706364470281646e-12,
     2.3781681465419992e-12, 2.0074085935225278e-12, 1.7220779299961350e-12,
     1.4981322045958579e-12, 1.3192439175591037e-12, 1.1740752088513777e-12,
     1.0545900076691665e-12, 9.5498422431408456e-13, 8.7099260463602158e-13,
     7.9943037015122636e-13, 7.3788431165910249e-13, 6.8450089844027268e-13,
     6.3783852514957004e-13, 5.9676289100964063e-13, 5.6037185249762978e-13,
     5.2794071976767861e-13},
    // x = 24.9000
    {1.7760094289010062e-01, 3.5662839934689160e-03, 2.1483638484052602e-04,
     2.1569917848475470e-05, 3.0319158552364782e-06, 5.4793629214009291e-07,
     1.2102979648572990e-07, 3.1593815376763407e-08, 9.5159012469835252e-09,
     3.2480918202218572e-09, 1.2389236151113226e-09, 5.2212946517402312e-10,
     2.4083592665323498e-10, 1.2059336575621383e-10, 6.5073741440689721e-11,
     3.7586143576477439e-11, 2.3088793156613959e-11, 1.4991598845317985e-11,
     1.0228060399893651e-11, 7.2909574924626463e-12, 5.4015821392493542e-12,
     4.1388818481485594e-12, 3.2655294738747527e-12, 2.6425757962437456e-12,
     2.1857933885949953e-12, 1.8424763929664550e-12, 1.5786695656599169e-12,
     1.3719063314969934e-12, 1.2069537462029984e-12, 1.0732492341796304e-12,
     9.6331632821632205e-13, 8.7176193473434467e-13, 7.9462751439434518e-13,
     7.2896057759451953e-13, 6.7252624553027142e-13, 6.2360962389886366e-13,
     5.8087814508823322e-13, 5.4328419518103688e-13, 5.0999524030426882e-13,
     4.8034300867476503e-13},
    // x = 25.0000
    {1.7724538509027910e-01, 3.5449077015278219e-03, 2.1269446181391044e-04,
     2.1269445903632166e-05, 2.9777221487496256e-06, 5.3598970901605517e-07,
     1.1791745822465478e-07, 3.0658261379532947e-08, 9.1972006549825835e-09,
     3.1267704638167788e-09, 1.1878950173730764e-09, 4.9863814841939286e-10,
     2.2909578939562134e-10, 1.1427013582051139e-10, 6.1428114465776871e-11,
     3.5350547512851293e-11, 2.1639580580668520e-11, 1.4004364305941948e-11,
     9.5252961368600781e-12, 6.7709602639771792e-12, 5.0035901286029186e-12,
     3.8251850281551138e-12, 3.0119002469141178e-12, 2.4329513449234258e-12,
     2.0092153869287413e-12, 1.6912722018908868e-12, 1.4473387686294246e-12,
     1.2564202174479106e-12, 1.1043033618934217e-12, 9.8114695525922158e-13,
     8.7999452990660058e-13, 7.9583444918677422e-13, 7.2499252867605564e-13,
     6.6473140997959344e-13, 6.1298121207337521e-13, 5.6815519536197794e-13,
     5.2902150011472795e-13, 4.9461251286822319e-13, 4.6415989200305619e-13,
     4.3704735638542975e-13},
    // x = 25.1000
    {1.7689195425989040e-01, 3.5237441085124313e-03, 2.1058231699145344e-04,
     2.0974334111373231e-05, 2.9247076934916598e-06, 5.2434973456361868e-07,
     1.1489710186989117e-07, 2.9753979242576035e-08, 8.8903809224576057e-09,
     3.0104364412452336e-09, 1.1391578894898572e-09, 4.7628982764976883e-10,
     2.1797011363896575e-10, 1.0830052808178368e-10, 5.7998962688038980e-11,
     3.3255051527580231e-11, 2.0285662671043527e-11, 1.3084871252506659e-11,
     8.8725928798631506e-12, 6.2892351650664232e-12, 4.6357338679081281e-12,
     3.5358318190364726e-12, 2.7783752380471301e-12, 2.2402500885163032e-12,
     1.8471199779215921e-12, 1.5526403914254116e-12, 1.3270583405197927e-12,
     1.1507522067519249e-12, 1.0104589661903083e-12, 8.9700856183755425e-13,
     8.0392776653014563e-13, 7.2655901373977130e-13, 6.6149176488054214e-13,
     6.0618791730344274e-13, 5.5873026276351209e-13, 5.1765053510119463e-13,
     4.8180989489208004e-13, 4.5031456291871488e-13, 4.2245539741635666e-13,
     3.9766403052703516e-13},
    // x = 25.2000
    {1.7654062927704448e-01, 3.5027902632078171e-03, 2.0849942020343182e-04,
     2.0684466064418104e-05, 2.8728422833420628e-06, 5.1300732499190120e-07,
     1.1196565881793993e-07, 2.8879805439414285e-08, 8.5949545854062483e-09,
     2.8988662195469632e-09, 1.0926009461240680e-09, 4.5502478932274136e-10,
     2.0742459658102739e-10, 1.0266357991639719e-10, 5.4772741488163832e-11,
     3.1290456674950080e-11, 1.9020509329051306e-11, 1.2228300022210845e-11,
     8.2662701198038200e-12, 5.8428870574102122e-12, 4.2956767559470533e-12,
     3.2688940526698912e-12, 2.5633324905860836e-12, 2.0630848281172028e-12,
     1.6983035750444813e-12, 1.4255235801173193e-12, 1.2168892033078682e-12,
     1.0540603380168712e-12, 9.2465936213613490e-13, 8.2014081155740047e-13,
     7.3448057822658625e-13, 6.6334977247133004e-13, 6.0358231996879153e-13,
     5.5282468370445938e-13, 5.0930093390738173e-13, 4.7165233532425867e-13,
     4.3882597358642160e-13, 4.0999621691551869e-13, 3.8450852240867336e-13,
     3.6183868028176904e-13},
    // x = 25.3000
    {1.7619138931234130e-01, 3.4820432668389882e-03, 2.0644525277131468e-04,
     2.0399728331385998e-05, 2.8220966014083020e-06, 5.0195373763306027e-07,
     1.0912017441742686e-07, 2.8034624090606445e-08, 8.3104559865065942e-09,
     2.7918471013586776e-09, 1.0481187052164407e-09, 4.3478664758265381e-10,
     1.9742696546517911e-10, 9.7339638236492250e-11, 5.1736794279097238e-11,
     2.9448193522348805e-11, 1.7838054492684632e-11, 1.1430184909004403e-11,
     7.7029254931261602e-12, 5.4292450866145792e-12, 3.9812671248023386e-12,
     3.0225990168760034e-12, 2.3652829221877057e-12, 1.9001835896339234e-12,
     1.5616637717356573e-12, 1.3089542085390713e-12, 1.1159727820656608e-12,
     9.6557540684802047e-13, 8.4620959513183637e-13, 7.4991116718467363e-13,
     6.7107345851631530e-13, 6.0567270961511919e-13, 5.5076954234747693e-13,
     5.0418139109912744e-13, 4.6426306997374075e-13, 4.2975711393711386e-13,
     3.9968998471472185e-13, 3.7329895298822973e-13, 3.4997983421239317e-13,
     3.2924911801154186e-13},
    // x = 25.4000
    {1.7584421382368359e-01, 3.4615002719365004e-03, 2.0441930721456955e-04,
     2.0120010369359855e-05, 2.7724421904754293e-06, 4.9118051978172608e-07,
     1.0635780331164991e-07, 2.7217364836304684e-08, 8.0364402200275461e-09,
     2.6891766608916805e-09, 1.0056111651529729e-09, 4.1552214765829779e-10,
     1.8794685096397528e-10, 9.2310273758434625e-11, 4.8879291817063613e-11,
     2.7720277470943731e-11, 1.6732662095046311e-11, 1.0686387440268216e-11,
     7.1794132527260677e-12, 5.0458449444874320e-12, 3.6905233398446219e-12,
     2.7953167276127259e-12, 2.1828592936307903e-12, 1.7503799417811510e-12,
     1.4361907498584861e-12, 1.2020468419717086e-12, 1.0235240505837417e-12,
     8.8459463602431627e-13, 7.7447515374870769e-13, 6.8574245261364786e-13,
     6.1317790419886170e-13, 5.5304104301628941e-13, 5.0260272714809481e-13,
     4.5983878137250219e-13, 4.2322502324910629e-13, 3.9159762663947446e-13,
     3.6405638426180348e-13, 3.3989663150017142e-13, 3.1856063369310482e-13,
     2.9960239416271837e-13},
    // x = 25.5000
    {1.7549908255120200e-01, 3.4411584812309515e-03, 2.0242108696606647e-04,
     1.9845204439350364e-05, 2.7238514245487993e-06, 4.8067949798971434e-07,
     1.0367580498868830e-07, 2.6427000811552811e-08, 7.7724821315595607e-09,
     2.5906622112305498e-09, 9.6498350097305825e-10, 3.9718098150352457e-10,
     1.7895566883973714e-10, 8.7558201122332563e-11, 4.6189175814676659e-11,
     2.6099267350414787e-11, 1.5699094590360585e-11, 9.9930717201457044e-12,
     6.6928244402084453e-12, 4.6904125594753703e-12, 3.4216201189229559e-12,
     2.5855482572818158e-12, 2.0148061040911660e-12, 1.6126041358751725e-12,
     1.3209594241502843e-12, 1.1039911378214769e-12, 9.3882518185150368e-13,
     8.1047589967962882e-13, 7.0887668093943024e-13, 6.2710798155056705e-13,
     5.6031190503950625e-13, 5.0501102848904426e-13, 4.5867119686943510e-13,
     4.1941498121656176e-13, 3.8583019582433345e-13, 3.5683960308647783e-13,
     3.3161074636610506e-13, 3.0949256333837284e-13, 2.8997016658645758e-13,
     2.7263213868016867e-13},
    // x = 25.6000
    {1.7515597551228915e-01, 3.4210151465755320e-03, 2.0045010609579555e-04,
     1.9575205524552978e-05, 2.6762974814454230e-06, 4.7044276779577364e-07,
     1.0107153952906864e-07, 2.5662546718998012e-08, 7.5181753695268676e-09,
     2.4961203011086366e-09, 9.2614577843495285e-10, 3.7971561488163088e-10,
     1.7042650931802724e-10, 8.3067204448362658e-11, 4.3656106541235915e-11,
     2.4578227041041572e-11, 1.4732483849175312e-11, 9.3466816763334459e-12,
     6.2404686226007343e-12, 4.3608490984709807e-12, 3.1728759696456124e-12,
     2.3919150304874460e-12, 1.8599703343056114e-12, 1.4858749970554608e-12,
     1.2151222591229254e-12, 1.0140454199709078e-12, 8.6121975046881602e-13,
     7.4263245274090531e-13, 6.4888515051243910e-13, 5.7352711688859677e-13,
     5.1203583402176361e-13, 4.6117813829716239e-13, 4.1860073277750676e-13,
     3.8256215695361381e-13, 3.5175389299568804e-13, 3.2517877777714004e-13,
     3.0206695366531378e-13, 2.8181684420029271e-13, 2.6395308829119383e-13,
     2.4809614460759724e-13},
    // x = 25.7000
    {1.7481487299673998e-01, 3.4010675678957862e-03, 1.9850588904262278e-04,
     1.9309911251295041e-05, 2.6297543163913700e-06, 4.6046268387107987e-07,
     9.8542463541499106e-08, 2.4923056993981814e-08, 7.2731314856206989e-09,
     2.4053762394305184e-09, 8.8901268483184088e-10, 3.6308112514874393e-10,
     1.6233403364976437e-10, 7.8822067945931660e-11, 4.1270414119006484e-11,
     2.3150689895485729e-11, 1.3828304240016277e-11, 8.7439200602590669e-12,
     5.8198570678180146e-12, 4.0552171728802610e-12, 2.9427416521227411e-12,
     2.2131490061052771e-12, 1.7172929657452476e-12, 1.3692925026325865e-12,
     1.1179027003990347e-12, 9.3153080732162483e-13, 7.9010743871948797e-13,
     6.8052812118505206e-13, 5.9401746774234758e-13, 5.2456122253098516e-13,
     4.6794870245344213e-13, 4.2117358028091041e-13, 3.8205032557346660e-13,
     3.4896346963140360e-13, 3.2070045996270112e-13, 2.9633818899231919e-13,
     2.7516464193061202e-13, 2.5662393069481456e-13, 2.4027715912237199e-13,
     2.2577417177434871e-13},
    // x = 25.8000
    {1.7447575556199613e-01, 3.3813130921658114e-03, 1.9658797035382127e-04,
     1.9049221812574628e-05, 2.5841966365846309e-06, 4.5073185056213643e-07,
     9.6086126277675742e-08, 2.4207624057254924e-08, 7.0369790814611094e-09,
     2.3182636459187654e-09, 8.5350327552375919e-10, 3.4723504904202286e-10,
     1.5465437737472793e-10, 7.4808511219721619e-11, 3.9023053246161256e-11,
     2.1810625750157240e-11, 1.2982347729153337e-11, 8.1817290626259257e-12,
     5.4286872434369148e-12, 3.7717281507093698e-12, 2.7297895823081972e-12,
     2.0480836713967778e-12, 1.5858012100678110e-12, 1.2620309888079278e-12,
     1.0285891675079506e-12, 8.5582584839781399e-13, 7.2493920228151396e-13,
     6.2367291260383436e-13, 5.4383245664047481e-13, 4.7980997282838527e-13,
     4.2768474740915366e-13, 3.8466112715883713e-13, 3.4870921667527859e-13,
     3.1833030335912219e-13, 2.9240067624997677e-13, 2.7006572152492320e-13,
     2.5066687592455714e-13, 2.3369043627771810e-13, 2.1873114903617308e-13,
     2.0546595048881701e-13},
    // x = 25.9000
    {1.7413860402849110e-01, 3.3617491124101288e-03, 1.9469589443212298e-04,
     1.8793039894096494e-05, 2.5395998766845333e-06, 4.4124311281359422e-07,
     9.3700165917624013e-08, 2.3515376650820471e-08, 6.8093629989516485e-09,
     2.2346240263622849e-09, 8.1954073521760094e-10, 3.3213723982544298e-10,
     1.4736505983346551e-10, 7.1013128942566820e-11, 3.6905561097251536e-11,
     2.0552410332212728e-11, 1.2190700843376827e-11, 7.6572724173698929e-12,
     5.0648285327465443e-12, 3.5087304845549664e-12, 2.5327040981847249e-12,
     1.8956457801910958e-12, 1.4646013887363354e-12, 1.1633329320745195e-12,
     9.4652955976500945e-13, 7.8636161808514512e-13, 6.6521285541829037e-13,
     5.7161901014468023e-13, 4.9792719973518243e-13, 4.3890798790020178e-13,
     3.9091032190032704e-13, 3.5133422826727794e-13, 3.1829420403790665e-13,
     2.9039974229322347e-13, 2.6660938459403536e-13, 2.4613187114504800e-13,
     2.2835803551396603e-13, 2.1281312704333154e-13, 1.9912293275241070e-13,
     1.8698936699344936e-13},
    // x = 26.0000
    {1.7380339947509396e-01, 3.3423730667304775e-03, 1.9282921529004506e-04,
     1.8541270602714157e-05, 2.4959401751905780e-06, 4.3198954745434947e-07,
     9.1382306017477204e-08, 2.2845478252657223e-08, 6.5899435519390429e-09,
     2.1543063710372236e-09, 7.8705215308998431e-10, 3.1774973318964629e-10,
     1.4044489950641923e-10, 6.7423334589084955e-11, 3.4910018170715961e-11,
     1.9370796883128832e-11, 1.1449723352864050e-11, 7.1679188772394301e-12,
     4.7263090706791653e-12, 3.2646989728281887e-12, 2.3502725175430004e-12,
     1.7548477729076873e-12, 1.3528724078262927e-12, 1.0725032562330737e-12,
     8.7112623105559952e-13, 7.2261723641655892e-13, 6.1046903902271528e-13,
     5.2395711615654995e-13, 4.5593369924128693e-13, 4.0152176593634727e-13,
     3.5732106081117622e-13, 3.2091337848881636e-13, 2.9054718878331009e-13,
     2.6493227390099687e-13, 2.4310333314045167e-13, 2.2432771074284347e-13,
     2.0804189297458802e-13, 1.9380709921310784e-13, 1.8127775794076510e-13,
     1.7017881410337669e-13},
    // x = 26.1000
    {1.7347012323464947e-01, 3.3231824373568305e-03, 1.9098749631125665e-04,
     1.8293821397191352e-05, 2.4531943516751823e-06, 4.2296445483106726e-07,
     8.9130352112007942e-08, 2.2197125566306102e-08, 6.3783957969284845e-09,
     2.0771667749589950e-09, 7.5596831090605822e-10, 3.0403662134296087e-10,
     1.3387393478857145e-10, 6.4027307946161251e-11, 3.3029011870147222e-11,
     1.8260889833517255e-11, 1.0756028542420811e-11, 6.7112269535482414e-12,
     4.4113036101440532e-12, 3.0382248783651593e-12, 2.1813769220607769e-12,
     1.6247808214525521e-12, 1.2498597777354612e-12, 9.8890412018059525e-13,
     8.0183139317661543e-13, 6.6411577281581403e-13, 5.6028753810225748e-13,
     4.8031311484197296e-13, 4.1751583125746824e-13, 3.6734688614955996e-13,
     3.2663929804895342e-13, 2.9314372138539276e-13, 2.6523294104618917e-13,
     2.4170969814815417e-13, 2.2167921170991649e-13, 2.0446303991026017e-13,
     1.8953986415631842e-13, 1.7650412181499100e-13, 1.6503667071754251e-13,
     1.5488369196517489e-13},
    // x = 26.2000
    {1.7313875688961164e-01, 3.3041747497219208e-03, 1.8917031001875560e-04,
     1.8050602021198658e-05, 2.4113398848363027e-06, 4.1416135077401550e-07,
     8.6942188464615696e-08, 2.1569547081520931e-08, 6.1744088407348542e-09,
     2.0030680787058213e-09, 7.2622348334166243e-10, 2.9096393476846227e-10,
     1.2763334983525081e-10, 6.0813946136904218e-11, 3.1255602621926056e-11,
     1.7218120376495544e-11, 1.0106464949692236e-11, 6.2849308212281530e-12,
     4.1181223365552749e-12, 2.8280068348293214e-12, 2.0249866078407403e-12,
     1.5046084468317109e-12, 1.1548701317991378e-12, 9.1195014548611544e-13,
     7.3814291088470841e-13, 6.1042050240194612e-13, 5.1428391822539154e-13,
     4.4034502592475232e-13, 3.8236656714451605e-13, 3.3610545839215047e-13,
     2.9861171255053709e-13, 2.6779286561248167e-13, 2.4213706538354216e-13,
     2.2053322409239520e-13, 2.0215186463171287e-13, 1.8636470137594370e-13,
     1.7268941775197081e-13, 1.6075112954971689e-13, 1.5025508413626017e-13,
     1.4096704524815456e-13},
    // x = 26.3000
    {1.7280928226776870e-01, 3.2853475715586269e-03, 1.8737723784963828e-04,
     1.7811524438465497e-05, 2.3703548913374941e-06, 4.0557395888082131e-07,
     8.4815774957847724e-08, 2.0962001702391205e-08, 5.9776851830727178e-09,
     1.9318795286310479e-09, 6.9775524976705003e-10, 2.7849953115712874e-10,
     1.2170540514013090e-10, 5.7772817913691978e-11, 2.9583292346513471e-11,
     1.6238223798589660e-11, 9.4980994584259551e-12, 5.8869272981935776e-12,
     3.8452005550133221e-12, 2.6328424770421774e-12, 1.8801511475583758e-12,
     1.3935606617265982e-12, 1.0672662017332580e-12, 8.4110404629129749e-13,
     6.7960045499271194e-13, 5.6113148387012428e-13, 4.7210645312361177e-13,
     4.0374022381135139e-13, 3.5020543852756605e-13, 3.0754379758549285e-13,
     2.7300718278459553e-13, 2.4464889556627349e-13, 2.2106414786313463e-13,
     2.0122173703477979e-13, 1.8435266422539727e-13, 1.6987504921060951e-13,
     1.5734262834370566e-13, 1.4640885220563063e-13, 1.3680147666060726e-13,
     1.2830432433065507e-13},
    // x = 26.4000
    {1.7248168143805750e-01, 3.2666985120195597e-03, 1.8560786993624914e-04,
     1.7576502770009865e-05, 2.3302181054042474e-06, 3.9719620310442103e-07,
     8.2749144117875714e-08, 2.0373777439537727e-08, 5.7879400922021471e-09,
     1.8634764553531059e-09, 6.7050431679766138e-10, 2.6661299105981813e-10,
     1.1607337252998721e-10, 5.4894120993507354e-11, 2.8005995114246341e-11,
     1.5317218437891142e-11, 8.9282016436010205e-12, 5.5152638152716317e-12,
     3.5910891807199892e-12, 2.4516207366315904e-12, 1.7459940139420539e-12,
     1.2909285943017547e-12, 9.8646221141067935e-13, 7.7587262729145772e-13,
     6.2578198276907087e-13, 5.1588243112107757e-13, 4.3343331808112441e-13,
     3.7021289897363658e-13, 3.2077622445186412e-13, 2.8143030305425517e-13,
     2.4961483120661886e-13, 2.2351855892257897e-13, 2.0183606855361544e-13,
     1.8361022089949661e-13, 1.6812803045027809e-13, 1.5485055508669725e-13,
     1.4336486020196825e-13, 1.3335056822447163e-13, 1.2455620879438945e-13,
     1.1678225917947786e-13},
    // x = 26.5000
    {1.7215593670646437e-01, 3.2482252208182170e-03, 1.8386180489350626e-04,
     1.7345453233371926e-05, 2.2909088591468743e-06, 3.8902220063212683e-07,
     8.0740398266684083e-08, 1.9804190163165172e-08, 5.6049010118554498e-09,
     1.7977399694793193e-09, 6.4441435096166663e-10, 2.5527551983125057e-10,
     1.1072147428264223e-10, 5.2168642225044047e-11, 2.6518009829008811e-11,
     1.4451386149104407e-11, 8.3942292732738670e-12, 5.1681272996097321e-12,
     3.3544459688229934e-12, 2.2833147492024333e-12, 1.6217067184938312e-12,
     1.1960595531985901e-12, 9.1191965375127480e-13, 7.1580311849217968e-13,
     5.7630051755491776e-13, 4.7433785323526710e-13, 3.9797002596748677e-13,
     3.3950174033122604e-13, 2.9384484112255874e-13, 2.5755352462762328e-13,
     2.2824224177939530e-13, 2.0422561527964712e-13, 1.8429046460181058e-13,
     1.6754832189426160e-13, 1.5333808355082464e-13, 1.4116054011858660e-13,
     1.3063356999429887e-13, 1.2146097114833955e-13, 1.1341044712082355e-13,
     1.0629783565248313e-13},
    // x = 26.6000
    {1.7183203061201127e-01, 3.2299253873911151e-03, 1.8213864961220711e-04,
     1.7118294083780227e-05, 2.2524070635814675e-06, 3.8104625504334020e-07,
     7.8787706796038212e-08, 1.9252582413928351e-08, 5.4283069977709932e-09,
     1.7345566735826773e-09, 6.1943182087523365e-10, 2.4445985546750876e-10,
     1.0563482609481689e-10, 4.9587720392142504e-11, 2.5113994794725674e-11,
     1.3637254164259452e-11, 7.8938148793910065e-12, 4.8438339005913883e-12,
     3.1340274250424244e-12, 2.1269753236490275e-12, 1.5065434231930564e-12,
     1.1083524970389824e-12, 8.4314341850065365e-13, 6.6047981812195345e-13,
     5.3080120194154810e-13, 4.3619043959951852e-13, 3.6544708479575140e-13,
     3.1136781928070217e-13, 2.6919741639092379e-13, 2.3572039898197721e-13,
     2.0871383517848088e-13, 1.8660933153093734e-13, 1.6827933041367646e-13,
     1.5289904076029120e-13, 1.3985541771612864e-13, 1.2868602093931227e-13,
     1.1903721763627232e-13, 1.1063513870312820e-13, 1.0326518598295368e-13,
     9.6757364891841412e-14},
    // x = 26.7000
    {1.7150994592282459e-01, 3.2117967400802894e-03, 1.8043801905812305e-04,
     1.6894945557181962e-05, 2.2146931903216326e-06, 3.7326284973401049e-07,
     7.6889303557595273e-08, 1.8718322268730556e-08, 5.2579081822552656e-09,
     1.6738183895095954e-09, 5.9550584835496654e-10, 2.3414018196492883e-10,
     1.0079938364543564e-10, 4.7143211472074777e-11, 2.3788944030316865e-11,
     1.2871578246293836e-11, 7.4247533166294138e-12, 4.5408194929152152e-12,
     2.9286813431635638e-12, 1.9817249282762861e-12, 1.3998159864358983e-12,
     1.0272538748459488e-12, 7.7967824140969315e-13, 6.0952101753441941e-13,
     4.8895860106778169e-13, 4.0115866901166909e-13, 3.3561785753739602e-13,
     2.8559265772254322e-13, 2.4663853272737250e-13, 2.1575464116782889e-13,
     1.9086938767422691e-13, 1.7052311088070544e-13, 1.5366774266640955e-13,
     1.3953754206401583e-13, 1.2756398485376162e-13, 1.1731865969684228e-13,
     1.0847427549396316e-13, 1.0077759520526446e-13, 9.4030357926903937e-14,
     8.8075637327217331e-14},
    // x = 26.8000
    {1.7118966563228447e-01, 3.1938370453356117e-03, 1.7875953607670100e-04,
     1.6675329815071699e-05, 2.1777482539149294e-06, 3.6566664159650517e-07,
     7.5043484363803630e-08, 1.8200802258726990e-08, 5.0934652652840336e-09,
     1.6154219001520151e-09, 5.7258806793297995e-10, 2.2429204785283612e-10,
     9.6201892517437812e-11, 4.4827456180137739e-11, 2.2538165208640979e-11,
     1.2151327040485555e-11, 6.9849902345986917e-12, 4.2576308966640759e-12,
     2.7373399198261235e-12, 1.8467521521082875e-12, 1.3008894076802401e-12,
     9.5225380661058762e-13, 7.2110544783009174e-13, 5.6257618417521966e-13,
     4.5047423462076450e-13, 3.6898462343013272e-13, 3.0825660661146298e-13,
     2.6197646398964913e-13, 2.2598962385948403e-13, 1.9749527760127327e-13,
     1.7456268093120841e-13, 1.5583324358659273e-13, 1.4033269898121253e-13,
     1.2735007002269532e-13, 1.1635807856265357e-13, 1.0695980858695446e-13,
     9.8852327035660660e-14, 9.1801458825389431e-14, 8.5624024838467357e-14,
     8.0175153582083082e-14},
    // x = 26.9000
    {1.7087117295525325e-01, 3.1760441069363576e-03, 1.7710283120319468e-04,
     1.6459370891055254e-05, 2.1415537947989993e-06, 3.5825245494405954e-07,
     7.3248604594499381e-08, 1.7699438336949898e-08, 4.9347490307371873e-09,
     1.5592687048688559e-09, 5.5063249427361594e-10, 2.1489228957543144e-10,
     9.1829841257382870e-11, 4.2633249643993447e-11, 2.1357259104091056e-11,
     1.1473667535890639e-11, 6.5726113945146033e-12, 3.9929177589453135e-12,
     2.5590134010290280e-12, 1.7213066031663558e-12, 1.2091776382112216e-12,
     8.8288257581665379e-13, 6.6703996602959061e-13, 5.1932338073619784e-13,
     4.1507431797169467e-13, 3.3943198871540714e-13, 2.8315670700300441e-13,
     2.4033652199693495e-13, 2.0708751156333249e-13, 1.8079530801832385e-13,
     1.5966027110328949e-13, 1.4241776857948938e-13, 1.2816206008891645e-13,
     1.1623296130214563e-13, 1.0614140928475154e-13, 9.7519640388916642e-14,
     9.0087246838084767e-14, 8.3627666012194183e-14, 7.7971642438483641e-14,
     7.2985425279537516e-14},
    // x = 27.0000
    {1.7055445132438052e-01, 3.1584157652314996e-03, 1.7546754247805502e-04,
     1.6246994639087889e-05, 2.1060918628534521e-06, 3.5101527566948599e-07,
     7.1503076904361645e-08, 1.7213668893108980e-08, 4.7815398864410762e-09,
     1.5052647877904025e-09, 5.2959539702224285e-10, 2.0591895941945478e-10,
     8.7671417367239270e-11, 4.0553813062304487e-11, 2.0242100441957073e-11,
     1.0835951555559554e-11, 6.1858327667742067e-12, 3.7454250460557352e-12,
     2.3927842184335481e-12, 1.6046942086204110e-12, 1.1241397281417955e-12,
     8.1870740809767639e-13, 6.1712758762335203e-13, 4.7946690049095835e-13,
     3.8250769456548092e-13, 3.1228422624387948e-13, 2.6012901336849573e-13,
     2.2050572022205919e-13, 1.8978307029026248e-13, 1.6552048500010902e-13,
     1.4604036663828427e-13, 1.3016543608141232e-13, 1.1705358623314633e-13,
     1.0609174608547084e-13, 9.6826262429397709e-14, 8.8916357242395557e-14,
     8.2102454586500070e-14, 7.6184266079174636e-14, 7.1005391470352353e-14,
     6.6442339382926596e-14},
    // x = 27.1000
    {1.7023948438648359e-01, 3.1409498963982079e-03, 1.7385331526741619e-04,
     1.6038128683328653e-05, 2.0713450015245858e-06, 3.4395024562828912e-07,
     6.9805369026626915e-08, 1.6742953813249969e-08, 4.6336274267665694e-09,
     1.4533203982846899e-09, 5.0943518264589703e-10, 1.9735125770410257e-10,
     8.3715466036768943e-11, 3.8582767212504469e-11, 1.9188820050482275e-11,
     1.0235703200415210e-11, 5.8229913507636569e-12, 3.5139860991461127e-12,
     2.2378015769120386e-12, 1.4962728845805164e-12, 1.0452762822354012e-12,
     7.5932951236476110e-13, 5.7104245443179579e-13, 4.4273510051567132e-13,
     3.5254394322426481e-13, 2.8734290067357152e-13, 2.3900036776514278e-13,
     2.0233120830389595e-13, 1.7394000839913041e-13, 1.5154820068762053e-13,
     1.3359180514922882e-13, 1.1897476222548993e-13, 1.0691405938606025e-13,
     9.6840329494695789e-14, 8.8332732004856317e-14, 8.1075470606156367e-14,
     7.4828236375566312e-14, 6.9405779592895450e-14, 6.4663569520032426e-14,
     6.0487580287105166e-14},
    // x = 27.2000
    {1.6992625599900069e-01, 3.1236444117180482e-03, 1.7225980208851685e-04,
     1.5832702369554343e-05, 2.0372962325009618e-06, 3.3705265723677590e-07,
     6.8154001668685508e-08, 1.6286773582074781e-08, 4.4908100166007036e-09,
     1.4033498429075105e-09, 4.9011228285292835e-10, 1.8916946896816355e-10,
     7.9951451437947864e-11, 3.6714107682082717e-11, 1.8193788223765374e-11,
     9.6706071773124800e-12, 5.4825366627443037e-12, 3.2975162100581847e-12,
     2.0932764578794036e-12, 1.3954485459217123e-12, 9.7212619940349175e-13,
     7.0438136272327884e-13, 5.2848475282544450e-13, 4.0887841606154751e-13,
     3.2497164543191044e-13, 2.6442615041850751e-13, 2.1961223580785059e-13,
     1.8567317011800133e-13, 1.5943375575539083e-13, 1.3876647122537628e-13,
     1.2221312056802189e-13, 1.0875316748625971e-13, 9.7658483607307987e-14,
     8.8400246159525977e-14, 8.0588022964916007e-14, 7.3929145977617551e-14,
     6.8201127224522796e-14, 6.3232614779289531e-14, 5.8890037960476208e-14,
     5.5068104403674116e-14},
    // x = 27.3000
    {1.6961475022651631e-01, 3.1064972568704014e-03, 1.7068666243990501e-04,
     1.5630646718079410e-05, 2.0039290409188032e-06, 3.3031794827619106e-07,
     6.6547546495401224e-08, 1.5844628425842900e-08, 4.3528943955760061e-09,
     1.3552712881977744e-09, 4.7158904920447107e-10, 1.8135490190659900e-10,
     7.6369420414985106e-11, 3.4942181706612443e-11, 1.7253599210347017e-11,
     9.1384979470050334e-12, 5.1630228418235822e-12, 3.0950066774100681e-12,
     1.9584770057832202e-12, 1.3016714289448838e-12, 9.0426367280699569e-13,
     6.5352420130949328e-13, 4.8917859821830975e-13, 3.7766754077354283e-13,
     2.9959679895179533e-13, 2.4336728855401970e-13, 2.0181946012209725e-13,
     1.7040370316634909e-13, 1.4615044836191763e-13, 1.2707301031210279e-13,
     1.1181169257922286e-13, 9.9416191460514711e-14, 8.9209356564540944e-14,
     8.0699981375579955e-14, 7.3525816091123872e-14, 6.7415606532949695e-14,
     6.2163349271995132e-14, 5.7610536689874549e-14, 5.3633719013774717e-14,
     5.0135662462860103e-14},
    // x = 27.4000
    {1.6930495133735596e-01, 3.0895064112426295e-03, 1.6913356263627738e-04,
     1.5431894378129583e-05, 1.9712273610770295e-06, 3.2374169689429603e-07,
     6.4984624196193657e-08, 1.5416037493880929e-08, 4.2196953015031216e-09,
     1.3090065737173400e-09, 4.5382965355450057e-10, 1.7388983282484826e-10,
     7.2959968414756961e-11, 3.3261666505950659e-11, 1.6365056748500731e-11,
     8.6373496325491487e-12, 4.8631013278502884e-12, 2.9055193061356158e-12,
     1.8327242677357426e-12, 1.2144327018867645e-12, 8.4129542939335823e-13,
     6.0644574383758953e-13, 4.5287009383555778e-13, 3.4889175875496901e-13,
     2.7624136530393727e-13, 2.2401352293914492e-13, 1.8548912093190878e-13,
     1.5640579501028063e-13, 1.3398600150981102e-13, 1.1637438400057379e-13,
     1.0230297104390627e-13, 9.0886777205301048e-14, 8.1496005859601849e-14,
     7.3674352884057158e-14, 6.7085689788394955e-14, 6.1478590375391688e-14,
     5.6662300617244869e-14, 5.2490184347282745e-14, 4.8848138379898135e-14,
     4.5646365190225696e-14},
    // x = 27.5000
    {1.6899684380024907e-01, 3.0726698872565275e-03, 1.6760017564781073e-04,
     1.5236379583619307e-05, 1.9391755626426050e-06, 3.1731961679621073e-07,
     6.3463902632117534e-08, 1.5000538076830449e-08, 4.0910351120109756e-09,
     1.2644810347697006e-09, 4.3679999397765937e-10, 1.6675745239414187e-10,
     6.9714207512949409e-11, 3.1667549017648925e-11, 1.5525160574972298e-11,
     8.1652666331118594e-12, 4.5815140686076963e-12, 2.7281813165638113e-12,
     1.7153882586670725e-12, 1.1332613403206783e-12, 7.8285818944476566e-13,
     5.6285807116710988e-13, 4.1932554922075234e-13, 3.2235741567071796e-13,
     2.5474193969962541e-13, 2.0622478531340539e-13, 1.7049949450798763e-13,
     1.4357238828870894e-13, 1.2284526368790249e-13, 1.0658523958483804e-13,
     9.3609768772019806e-14, 8.3094618946342878e-14, 7.4453984374095452e-14,
     6.7263947841306383e-14, 6.1212593678603248e-14, 5.6066856559623069e-14,
     5.1650090230707025e-14, 4.7826631523586519e-14, 4.4491009294993396e-14,
     4.1560288412184430e-14},
    // x = 27.6000
    {1.6869041228105858e-01, 3.0559857297106350e-03, 1.6608618094384779e-04,
     1.5044038110284997e-05, 1.9077584373276752e-06, 3.1104755261669910e-07,
     6.1984095059355154e-08, 1.4597684859861094e-08, 3.9667435034534685e-09,
     1.2216233342649289e-09, 4.2046760586501376e-10, 1.5994181550507967e-10,
     6.6623736400940954e-11, 3.0155106933670896e-11, 1.4731093839206944e-11,
     7.7204748922483836e-12, 4.3170872169349484e-12, 2.5621806317746862e-12,
     1.6058843265801313e-12, 1.0577212463509814e-12, 7.2861632831155072e-13,
     5.2249569159208165e-13, 3.8832984517334825e-13, 2.9788651717228235e-13,
     2.3494853301098900e-13, 1.8987266001173520e-13, 1.5673910100919946e-13,
     1.3180552660501608e-13, 1.1264124417364491e-13, 9.7627602047229695e-14,
     8.5661616592316460e-14, 7.5975567522475065e-14, 6.8024519366399370e-14,
     6.1414610034878195e-14, 5.5856369337792731e-14, 5.1133735496685016e-14,
     4.7083114771267269e-14, 4.3578987199489665e-14, 4.0523854258533329e-14,
     3.7841114003919964e-14},
    // x = 27.7000
    {1.6838564163957570e-01, 3.0394520151379484e-03, 1.6459126434080322e-04,
     1.4854807234127613e-05, 1.8769611860207017e-06, 3.0492147546642463e-07,
     6.0543958425709067e-08, 1.4207049209166597e-08, 3.8466571261933321e-09,
     1.1803653032292110e-09, 4.0480157788748488e-10, 1.5342779402976238e-10,
     6.3680612207512420e-11, 2.8719890953058935e-11, 1.3980211360004427e-11,
     7.3013137734978803e-12, 4.0687252814095401e-12, 2.4067615144073838e-12,
     1.5036697945110581e-12, 9.8740859221233498e-13, 6.7825972396995155e-13,
     4.8511375982711639e-13, 3.5968493292816317e-13, 2.7531544392457709e-13,
     2.1672345624746134e-13, 1.7483940375053865e-13, 1.4410583398793685e-13,
     1.2101557424214594e-13, 1.0329440785148480e-13, 8.9430232115322540e-14,
     7.8394175206827812e-14, 6.9471088434489717e-14, 6.2154010523954638e-14,
     5.6076972846774095e-14, 5.0971313942023399e-14, 4.6636719741044172e-14,
     4.2921673524699816e-14, 3.9710032111142912e-14, 3.6911661010648948e-14,
     3.4455803420112019e-14},
    // x = 27.8000
    {1.6808251692637777e-01, 3.0230668511786545e-03, 1.6311511785416252e-04,
     1.4668625691120250e-05, 1.8467694063545860e-06, 2.9893747864504557e-07,
     5.9142291736848292e-08, 1.3828218490149129e-08, 3.7306192954226752e-09,
     1.1406417894864983e-09, 3.8977247254509524e-10, 1.4720103231494312e-10,
     6.0877324037905145e-11, 2.7357708169270949e-11, 1.3270028666081863e-11,
     6.9062284996440636e-12, 3.8354056969928419e-12, 2.2612205263417231e-12,
     1.4082408576582067e-12, 9.2194937045305175e-13, 6.3150177538678125e-13,
     4.5048643983262706e-13, 3.3320845641437041e-13, 2.5449377342594766e-13,
     1.9994029882833764e-13, 1.6101704867670097e-13, 1.3250616450265784e-13,
     1.1112050328195857e-13, 9.4732031373088974e-14, 8.1928040505433479e-14,
     7.1748698806773454e-14, 6.3527767824626079e-14, 5.6793572604091257e-14,
     5.1206033868740296e-14, 4.6515782968374587e-14, 4.2537091522551745e-14,
     3.9129618020453269e-14, 3.6185888117430229e-14, 3.3622569651857827e-14,
     3.1374300845846758e-14},
    // x = 27.9000
    {1.6778102337974826e-01, 3.0068283759674568e-03, 1.6165743955444821e-04,
     1.4485433638137665e-05, 1.8171690806954168e-06, 2.9309177351432541e-07,
     5.7777934489214832e-08, 1.3460795415777802e-08, 3.6184796967253694e-09,
     1.1023905140670227e-09, 3.7535225103731214e-10, 1.4124790524000253e-10,
     5.8206768120585040e-11, 2.6064606516545032e-11, 1.2598211766224067e-11,
     6.5337631152165762e-12, 3.6161737845932066e-12, 2.1249027867412702e-12,
     1.3191297158518185e-12, 8.6099713432086655e-13, 5.8807757790503283e-13,
     4.1840539968984605e-13, 3.0873248737007939e-13, 2.3528319964588941e-13,
     1.8448299268716596e-13, 1.4830658154584563e-13, 1.2185441339471650e-13,
     1.0204524226715109e-13, 8.6887615291762771e-14, 7.5061553408916390e-14,
     6.5671545872994996e-14, 5.8096862129744657e-14, 5.1898618687472540e-14,
     4.6760767442164365e-14, 4.2451828444660701e-14, 3.8799583714491815e-14,
     3.5674033195626796e-14, 3.2975717416985832e-14, 3.0627588079946127e-14,
     2.8569263274962453e-14},
    // x = 28.0000
    {1.6748114642265752e-01, 2.9907347575351085e-03, 1.6021793342703365e-04,
     1.4305172615066574e-05, 1.7881465645361786e-06, 2.8738068552474275e-07,
     5.6449765166645699e-08, 1.3104397423685417e-08, 3.5100941056298308e-09,
     1.0655519349231505e-09, 3.6151420220587782e-10, 1.3555547868415610e-10,
     5.5662224459373134e-11, 2.4836860204886326e-11, 1.1962567598593511e-11,
     6.1825539347949599e-12, 3.4101380708562455e-12, 1.9971985058493228e-12,
     1.2359019231077184e-12, 8.0423091329092070e-13, 5.4774224299378388e-13,
     3.8867842771519857e-13, 2.8610236394749141e-13, 2.1755654226684089e-13,
     1.7024495492584837e-13, 1.3661719251200983e-13, 1.1207208584675863e-13,
     9.3721081056860518e-14, 7.9700347275594765e-14, 6.8776424714551519e-14,
     6.0113732990437818e-14, 5.3133887530762351e-14, 4.7428480424000173e-14,
     4.2703771729749866e-14, 3.8744869557128970e-14, 3.5392071227640721e-14,
     3.2524947258365599e-14, 3.0051448913690558e-14, 2.7900333175585225e-14,
     2.6015815068322295e-14},
    // x = 28.1000
    {1.6718287165980245e-01, 2.9747841932237871e-03, 1.5879630923568895e-04,
     1.4127785508057014e-05, 1.7596885752803963e-06, 2.8180065038937318e-07,
     5.5156699797905872e-08, 1.2758656080640258e-08, 3.4053241204414569e-09,
     1.0300691175571309e-09, 3.4823287531657441e-10, 1.3011147225718968e-10,
     5.3237334895397843e-11, 2.3670956078130663e-11, 1.1361035112375884e-11,
     5.8513234424358979e-12, 3.2164659416638737e-12, 1.8775397736815787e-12,
     1.1581539374528841e-12, 7.5135328988885389e-13, 5.1026935074131956e-13,
     3.6112815990019526e-13, 2.6517562422963104e-13, 2.0119683801299388e-13,
     1.5712830235956558e-13, 1.2586558757323096e-13, 1.0308726285109689e-13,
     8.6085126994750947e-14, 7.3114611934309009e-14, 6.3022990858585087e-14,
     5.5030527876394766e-14, 4.8598245666493751e-14, 4.3346061974839403e-14,
     3.9000947230630381e-14, 3.5363392712617488e-14, 3.2285290091947932e-14,
     2.9655067660590205e-14, 2.7387529274376372e-14, 2.5416795401692085e-14,
     2.3691324767387142e-14},
    // x = 28.2000
    {1.6688618487470400e-01, 2.9589749091159198e-03, 1.5739228238974566e-04,
     1.3953216513876330e-05, 1.7317821814012608e-06, 2.7634821039908055e-07,
     5.3897690572458943e-08, 1.2423216513098684e-08, 3.3040369076822616e-09,
     9.9588761218790395e-10, 3.3548401646045227e-10, 1.2490422415724883e-10,
     5.0926082489646422e-11, 2.2563580834051365e-11, 1.0791676938275231e-11,
     5.5388746100942886e-12, 3.0343796048272908e-12, 1.7653975843729933e-12,
     1.0855108555388472e-12, 7.0208862507577266e-13, 4.7544952442005984e-13,
     3.3559090958437389e-13, 2.4582102679901276e-13, 1.8609650718602493e-13,
     1.4504313186311016e-13, 1.1597535923100421e-13, 9.4834044682933536e-14,
     7.9079807934959503e-14, 6.7079543187216462e-14, 5.7755864556871944e-14,
     5.0381078159426988e-14, 4.4452882394703633e-14, 3.9617524648981702e-14,
     3.5621203927660440e-14, 3.2278678412934189e-14, 2.9452581734551346e-14,
     2.7039540824279837e-14, 2.4960696445189142e-14, 2.3155124693713189e-14,
     2.1575202140995628e-14},
    // x = 28.3000
    {1.6659107202686105e-01, 2.9433051594761270e-03, 1.5600557381477293e-04,
     1.3781411105329211e-05, 1.7044147919624225e-06, 2.7102001087332928e-07,
     5.2671724511930449e-08, 1.2097736862610200e-08, 3.2061049595016701e-09,
     9.6295533710547633e-10, 3.2324450836829716e-10, 1.1992265802784115e-10,
     4.8722772142790016e-11, 2.1511609049669966e-11, 1.0252671607467463e-11,
     5.2440856052672406e-12, 2.8631523393083749e-12, 1.6602790784230718e-12,
     1.0176243177805121e-12, 6.5618142048495484e-13, 4.4308911732301684e-13,
     3.1191559102143472e-13, 2.2791765115708836e-13, 1.7215658911022050e-13,
     1.3390686094964446e-13, 1.0687641039049627e-13, 8.7252041899025052e-14,
     7.2652418060385818e-14, 6.1548615457433343e-14, 5.2933564004297023e-14,
     4.6128072753057123e-14, 4.0663976669717215e-14, 3.6211999501090024e-14,
     3.2536194507780219e-14, 2.9464552536939654e-14, 2.6869660187603311e-14,
     2.4655733478610321e-14, 2.2749773595548806e-14, 2.1095435710877573e-14,
     1.9648713627333612e-14},
    // x = 28.4000
    {1.6629751924895986e-01, 2.9277732262059223e-03, 1.5463590982665987e-04,
     1.3612315997707980e-05, 1.6775741464871437e-06, 2.6581279674116532e-07,
     5.1477822194836579e-08, 1.1781887764908025e-08, 3.1114058624563483e-09,
     9.3122246788130482e-10, 3.1149231344445408e-10, 1.1515625169425593e-10,
     4.6622012374438136e-11, 2.0512091958823135e-11, 9.7423062814863604e-12,
     4.9659048602694566e-12, 2.7021050100097518e-12, 1.5617249864528610e-12,
     9.5417057088117641e-13, 6.1339480673951091e-13, 4.1300900287045118e-13,
     2.8996272918819019e-13, 2.1135407139600144e-13, 1.5928604072219656e-13,
     1.2364362348844967e-13, 9.8504426956597493e-14, 8.0285909770900905e-14,
     6.6754702782719218e-14, 5.6479270303701725e-14, 4.8518174428022894e-14,
     4.2237432948134730e-14, 3.7200656866847206e-14, 3.3101325401621053e-14,
     2.9720071325811095e-14, 2.6897159842078280e-14, 2.4514400163804652e-14,
     2.2483033658993504e-14, 2.0735481642732902e-14, 1.9219630693761760e-14,
     1.7894814500358447e-14},
    // x = 28.5000
    {1.6600551284413689e-01, 2.9123774183108337e-03, 1.5328302200900209e-04,
     1.3445879116239138e-05, 1.6512483051630595e-06, 2.6072340924715728e-07,
     5.0315036532266030e-08, 1.1475351851577065e-08, 3.0198220770892997e-09,
     9.0064133212196925e-10, 3.0020641983774694e-10, 1.1059500766713770e-10,
     4.4618698189093476e-11, 1.9562246932657879e-11, 9.2589699581741169e-12,
     4.7033464775651114e-12, 2.5506028287487810e-12, 1.4693072593486286e-12,
     8.9484867563796518e-13, 5.7350914794327636e-13, 3.8504345971841806e-13,
     2.6960354881784542e-13, 1.9602759707595433e-13, 1.4740109302771911e-13,
     1.1418371590289204e-13, 9.0800394971995741e-14, 7.3884922416382956e-14,
     6.1342479337374436e-14, 5.1832575398555086e-14, 4.4475039155783987e-14,
     3.8678030602689498e-14, 3.4034742121510643e-14, 3.0259810312582756e-14,
     2.7149264990875896e-14, 2.4554757693872493e-14, 2.2366644123495593e-14,
     2.0502669595267251e-14, 1.8900268677483429e-14, 1.7511238332864987e-14,
     1.6297996241976351e-14},
    // x = 28.6000
    {1.6571503928329454e-01, 2.8971160713796343e-03, 1.5194664709369636e-04,
     1.3282049564493210e-05, 1.6254256393703170e-06, 2.5574878277731692e-07,
     4.9182451592310524e-08, 1.1177823273246740e-08, 2.9312407277688379e-09,
     8.7116630947109627e-10, 2.8936679037809968e-10, 1.0622942530798186e-10,
     4.2707994960840080e-11, 1.8659447617088677e-11, 8.8011471212913360e-12,
     4.4554859474461333e-12, 2.4080523434842710e-12, 1.3826268708295643e-12,
     8.3937884887246311e-13, 5.3632075322991735e-13, 3.5903914427337280e-13,
     2.5071913621815432e-13, 1.8184357576662208e-13, 1.3642466059384846e-13,
     1.0546308958708086e-13, 8.3710158500648706e-14, 6.8002583216839633e-14,
     5.6375289981864420e-14, 4.7572913032739266e-14, 4.0772477610774175e-14,
     3.5421431040311534e-14, 3.1140506467292786e-14, 2.7664013704713124e-14,
     2.4802282644693995e-14, 2.2417528210920297e-14, 2.0408026626287509e-14,
     1.8697544876513253e-14, 1.7228154763209957e-14, 1.5955267197742857e-14,
     1.4844147739446596e-14},
    // x = 28.7000
    {1.6542608520246849e-01, 2.8819875470753502e-03, 1.5062652684464620e-04,
     1.3120777593725995e-05, 1.6000948225213064e-06, 2.5088594180022314e-07,
     4.8079181471141230e-08, 1.0889007243308156e-08, 2.8455534022765609e-09,
     8.4275373658107592e-10, 2.7895431421764249e-10, 1.0205047455788724e-10,
     4.0885323273234179e-11, 1.7801214684374402e-11, 8.3674118036645422e-12,
     4.2214561560715613e-12, 2.2738986392065227e-12, 1.3013117795479307e-12,
     7.8750092920581058e-13, 5.0164068698346626e-13, 3.3485414363399236e-13,
     2.3319966796567356e-13, 1.6871475217351809e-13, 1.2628579965233949e-13,
     9.7422885642786289e-14, 7.7184014786280550e-14, 6.2596268403071114e-14,
     5.1816084975587890e-14, 4.3667695573054149e-14, 3.7381527872823389e-14,
     3.2441658479661222e-14, 2.8494463754273701e-14, 2.5292548236153635e-14,
     2.2659524173971461e-14, 2.0467407175819917e-14, 1.8621814414068790e-14,
     1.7052088425694421e-14, 1.5704590719715463e-14, 1.4538072407920454e-14,
     1.3520429053063632e-14},
    // x = 28.8000
    {1.6513863740024515e-01, 2.8669902326377505e-03, 1.4932240794448903e-04,
     1.2962014573120850e-05, 1.5752448212007278e-06, 2.4613199791879378e-07,
     4.7004369208727557e-08, 1.0608619601205459e-08, 2.7626559606607310e-09,
     8.1536181679181702e-10, 2.6895076102466612e-10, 9.8049571137052678e-11,
     3.9146344655007923e-11, 1.6985207158879233e-11, 7.9564220358677922e-12,
     4.0004436634793677e-12, 2.1476227351407303e-12, 1.2250150388175304e-12,
     7.3897295720241651e-13, 4.6929367001276180e-13, 3.1235710254762767e-13,
     2.1694370105379155e-13, 1.5656067919372730e-13, 1.1691921076325317e-13,
     9.0009008370113823e-14, 7.1176343513563105e-14, 5.7626900962454385e-14,
     4.7630932764856585e-14, 4.0087105526263322e-14, 3.4275711653518510e-14,
     2.9714982010473043e-14, 2.6075171508411671e-14, 2.3125898980479036e-14,
     2.0703114756541555e-14, 1.8687928210658679e-14, 1.6992760812171985e-14,
     1.5552117949824071e-14, 1.4316329628173852e-14, 1.3247234346505116e-14,
     1.2315156613308151e-14},
    // x = 28.9000
    {1.6485268283522841e-01, 2.8521225403970118e-03, 1.4803404188425462e-04,
     1.2805712960902327e-05, 1.5508648865951166e-06, 2.4148414702833079e-07,
     4.5957185747285608e-08, 1.0336386394397631e-08, 2.6824483528967922e-09,
     7.8895053426843563e-10, 2.5933873758772314e-10, 9.4218553128240625e-11,
     3.7486948156049387e-11, 1.6209214279781244e-11, 7.5669146543981779e-12,
     3.7916852326586190e-12, 2.0287391640577564e-12, 1.1534130429762464e-12,
     6.9356986114688061e-13, 4.3911706457708810e-13, 2.9142641848836561e-13,
     2.0185751951806375e-13, 1.4530717663318055e-13, 1.0826478232896573e-13,
     8.3171734248831952e-14, 6.5645267271866134e-14, 5.3058652266198417e-14,
     4.3788755021273571e-14, 3.6803858067320813e-14, 3.1430819722463369e-14,
     2.7219720307031742e-14, 2.3863052056228943e-14, 2.1146258644697583e-14,
     1.8916752284194444e-14, 1.7064080850555788e-14, 1.5506973162793174e-14,
     1.4184715651892943e-14, 1.3051309908465062e-14, 1.2071448326227959e-14,
     1.1217698810391860e-14},
    // x = 29.0000
    {1.6456820862355448e-01, 2.8373829072982776e-03, 1.4676118485587011e-04,
     1.2651826276292828e-05, 1.5269445462014348e-06, 2.3693966657666110e-07,
     4.4936828930632881e-08, 1.0072043478132115e-08, 2.6048344449209822e-09,
     7.6348157236365443e-10, 2.5010164669559818e-10, 9.0549658862807304e-11,
     3.5903237711721206e-11, 1.5471147866005041e-11, 7.1977004451275586e-12,
     3.5944645921372656e-12, 1.9167937205434062e-12, 1.0862039002274953e-12,
     6.5108224040042437e-13, 4.1095993672548290e-13, 2.7194949944062303e-13,
     1.8785453297116999e-13, 1.3488583367280374e-13, 1.0026717156100816e-13,
     7.6865353424649898e-14, 6.0552340570175068e-14, 4.8858669040366138e-14,
     4.0261084386236451e-14, 3.3792984077677782e-14, 2.8824715994654187e-14,
     2.4936063430119054e-14, 2.1840229387923642e-14, 1.9337377356922377e-14,
     1.7285568335556262e-14, 1.5582181271458233e-14, 1.4151792120567701e-14,
     1.2938115100389915e-14, 1.1898548923978834e-14, 1.1000424216564152e-14,
     1.0218381033409459e-14},
    // x = 29.1000
    {1.6428520203645383e-01, 2.8227697944368329e-03, 1.4550359764742846e-04,
     1.2500309072284913e-05, 1.5034735958047081e-06, 2.3249591292237088e-07,
     4.3942522542709506e-08, 9.8153361322137120e-09, 2.5297218526257732e-09,
     7.3891823598371934e-10, 2.4122364816675259e-10, 8.7035506032923231e-11,
     3.4391520246934056e-11, 1.4769035150991054e-11, 6.8476595995032050e-12,
     3.4081094158062023e-12, 1.8113613660536144e-12, 1.0231059225789923e-12,
     6.1131523891052888e-13, 3.8468218993157469e-13, 2.5382207940995653e-13,
     1.7485472285457550e-13, 1.2523355149613089e-13, 9.2875419787868710e-14,
     7.1047840968348781e-14, 5.5862264978989028e-14, 4.4996823509562028e-14,
     3.7021842961080019e-14, 3.1031631910437640e-14, 2.6437158646443573e-14,
     2.2845910214230930e-14, 1.9990380367631557e-14, 1.7684425730049118e-14,
     1.5796001491086986e-14, 1.4229754540049789e-14, 1.2915681745558517e-14,
     1.1801598251936856e-14, 1.0848046163564608e-14, 1.0024795130505419e-14,
     9.3083993020101072e-15},
    // x = 29.2000
    {1.6400365049785870e-01, 2.8082816866036031e-03, 1.4426104554113954e-04,
     1.2351116909202743e-05, 1.4804420917150933e-06, 2.2815031878729638e-07,
     4.2973515383604982e-08, 9.5660186939920487e-09, 2.4570217834272536e-09,
     7.1522537774887337e-10, 2.3268962190923779e-10, 8.3669071958302873e-11,
     3.2948294474518153e-11, 1.4101012057086441e-11, 6.5157374625381569e-12,
     3.2319885048715915e-12, 1.7120442794848255e-12, 9.6385622421050835e-13,
     5.7408750202503811e-13, 3.6015376448323386e-13, 2.3694758708459886e-13,
     1.6278413255758570e-13, 1.1629212278888073e-13, 8.6042599253323316e-14,
     6.5680555407545333e-14, 5.1542628182605681e-14, 4.1445484729577732e-14,
     3.4047139752775178e-14, 2.8498886255087057e-14, 2.4249636771137214e-14,
     2.0932719869718247e-14, 1.8298599023069582e-14, 1.6173870023079359e-14,
     1.4435681890318639e-14, 1.2995427355236291e-14, 1.1788129424756068e-14,
     1.0765401713221434e-14, 9.8906951362947760e-15, 9.1360343513109277e-15,
     8.4797417113937055e-15},
    // x = 29.3000
    {1.6372354158205613e-01, 2.7939170918407446e-03, 1.4303329821388727e-04,
     1.2204206329027250e-05, 1.4578403432551426e-06, 2.2390039079960572e-07,
     4.2029080381506430e-08, 9.3238542068293102e-09, 2.3866488850350465e-09,
     6.9236932775102234e-10, 2.2448513289903634e-10, 8.0443674940168492e-11,
     3.1570240345358100e-11, 1.3465316881366258e-11, 6.2009405530887202e-12,
     3.0655091579126775e-12, 1.6184700428225788e-12, 9.0820942026032759e-13,
     5.3923021029388319e-13, 3.3725389752532045e-13, 2.2123656344023209e-13,
     1.5157439786961806e-13, 1.0800784509391943e-13, 7.9725488794137775e-14,
     6.0727962242910907e-14, 4.7563664914893127e-14, 3.8179309296410427e-14,
     3.1315085440611866e-14, 2.6175602616787618e-14, 2.2245221216160357e-14,
     1.9181376550572751e-14, 1.6751272759301031e-14, 1.4793358326724699e-14,
     1.3193326029815833e-14, 1.1868830343135897e-14, 1.0759554740383607e-14,
     9.8206314108760799e-15, 9.0182031929019865e-15, 8.3263797538865341e-15,
     7.7251169822417543e-15},
    // x = 29.4000
    {1.6344486301138444e-01, 2.7796745410070393e-03, 1.4182012964031652e-04,
     1.2059534830460472e-05, 1.4356589054883721e-06, 2.1974370712395132e-07,
     4.1108513739055385e-08, 9.0886140833454551e-09, 2.3185211010743950e-09,
     6.7031782672206342e-10, 2.1659639797128937e-10, 7.7352956639280510e-11,
     3.0254209110477317e-11, 1.2860284366573946e-11, 5.9023328382672051e-12,
     2.9081147160265986e-12, 1.5302899522013549e-12, 8.5593641862686756e-13,
     5.0658618443121525e-13, 3.1587043905731411e-13, 2.0660612441922366e-13,
     1.4116231452046385e-13, 1.0033116525501784e-13, 7.3884276004947341e-14,
     5.6157380354551523e-14, 4.3898037924968158e-14, 3.5175049775314254e-14,
     2.8805622971203376e-14, 2.4044256054953876e-14, 2.0408428363052283e-14,
     1.7578065748897742e-14, 1.5335969443841255e-14, 1.3531616795528947e-14,
     1.2058640889872185e-14, 1.0840509051641824e-14, 9.8212264826256615e-15,
     8.9591849129511070e-15, 8.2230185588526108e-15, 7.5887650430110306e-15,
     7.0378894715468156e-15},
    // x = 29.5000
    {1.6316760265397315e-01, 2.7655525873528613e-03, 1.4062131799837799e-04,
     1.1917060844705238e-05, 1.4138885721806020e-06, 2.1567791517532705e-07,
     4.0211134112669098e-08, 8.8600777827725075e-09, 2.2525595332282590e-09,
     6.4903996243661155e-10, 2.0901025432500507e-10, 7.4390865418725201e-11,
     2.8997213957802707e-11, 1.2284340132622858e-11, 5.6190322450931850e-12,
     2.7592822709725958e-12, 1.4471774454208691e-12, 8.0682329794185587e-13,
     4.7600905606179235e-13, 2.9589921955876482e-13, 1.9297946512665485e-13,
     1.3148943982836722e-13, 9.3216352411874597e-14, 6.8482283698562144e-14,
     5.1938749381580226e-14, 4.0520637277597331e-14, 3.2411379336241954e-14,
     2.6500372618671788e-14, 2.2088802944199310e-14, 1.8725095720341907e-14,
     1.6110161477643795e-14, 1.4041334403679377e-14, 1.2378355035806994e-14,
     1.1022236559462140e-14, 9.9018428671995756e-15, 8.9651870765691924e-15,
     8.1736807138512613e-15, 7.4982639286771826e-15, 6.9167571920610172e-15,
     6.4120200588052635e-15},
    // x = 29.6000
    {1.6289174852152435e-01, 2.7515498061044712e-03, 1.3943664557726038e-04,
     1.1776743711937544e-05, 1.3925203689858697e-06, 2.1170072941340361e-07,
     3.9336281823447787e-08, 8.6380325017819387e-09, 2.1886883095844339e-09,
     6.2850610918251381e-10, 2.0171412964758783e-10, 7.1551640595756484e-11,
     2.7796421188713472e-11, 1.1735995445731005e-11, 5.3502073936516172e-12,
     2.6185205250901283e-12, 1.3688266376221737e-12, 7.6067026538801385e-13,
     4.4736249929086418e-13, 2.7724346525676813e-13, 1.8028540227115955e-13,
     1.2250172571885526e-13, 8.6621197218740987e-14, 6.3485718653412709e-14,
     4.8044416320085970e-14, 3.7408396437886875e-14, 2.9868731212346618e-14,
     2.4382490265085495e-14, 2.0294554628689049e-14, 1.7182268298546730e-14,
     1.4766123294813290e-14, 1.2856996459836166e-14, 1.1324179837979204e-14,
     1.0075546603164495e-14, 9.0449711461268852e-15, 8.1841837588081910e-15,
     7.4573938566566240e-15, 6.8376760299767422e-15, 6.3044995217389530e-15,
     5.8420123752126651e-15},
    // x = 29.7000
    {1.6261728876713571e-01, 2.7376647940573970e-03, 1.3826589868764101e-04,
     1.1638543658449335e-05, 1.3715455468490004e-06, 2.0780992921424752e-07,
     3.8483318098350580e-08, 8.4222728781794587e-09, 2.1268344588887885e-09,
     6.0868787014160900e-10, 1.9469601377092161e-10, 6.8829797550362451e-11,
     2.6649141901692107e-11, 1.1213842303779093e-11, 5.0950745371079263e-12,
     2.4853677925640606e-12, 1.2949509574353745e-12, 7.1729068851082002e-13,
     4.2051951851942839e-13, 2.5981325736353878e-13, 1.6845795184096580e-13,
     1.1414918060025178e-13, 8.0506735151282535e-14, 5.8863440807795763e-14,
     4.4448939733026224e-14, 3.4540123722115182e-14, 2.7529151717226086e-14,
     2.2436537764113448e-14, 1.8648061939420799e-14, 1.5768094826975450e-14,
     1.3535402305840191e-14, 1.1773482202552321e-14, 1.0360516514076647e-14,
     9.2107554813188183e-15, 8.2627259162241938e-15, 7.4716058989027843e-15,
     6.8041973231894610e-15, 6.2355506189930396e-15, 5.7466599090881133e-15,
     5.3228638918373919e-15},
    // x = 29.8000
    {1.6234421168316363e-01, 2.7238961691786801e-03, 1.3710886757419035e-04,
     1.1502421774440561e-05, 1.3509555756172761e-06, 2.0400335681646337e-07,
     3.7651624340383073e-08, 8.2126007068911910e-09, 2.0669277904211883e-09,
     5.8955802253179787e-10, 1.8794443177582415e-10, 6.6220113641397100e-11,
     2.5552824152468005e-11, 1.0716548817896986e-11, 4.8528946949358882e-12,
     2.3593901323506898e-12, 1.2252818764736868e-12, 6.7651019561396015e-13,
     3.9536178827934476e-13, 2.4352503190094155e-13, 1.5743593924659518e-13,
     1.0638555778642215e-13, 7.4836991942558896e-14, 5.4586751214354102e-14,
     4.1128910101258392e-14, 3.1896347817483540e-14, 2.5376165665302298e-14,
     2.0648364349351987e-14, 1.7137009633803957e-14, 1.4471732954482932e-14,
     1.2408355356960563e-14, 1.0782137780787778e-14, 9.4795371669460914e-15,
     8.4207323961720251e-15, 7.5485705621456814e-15, 6.8214279152417642e-15,
     6.2085086735322523e-15, 5.6866924182585782e-15, 5.2383836632508381e-15,
     4.8500214266891228e-15},
    // x = 29.9000
    {1.6207250569912537e-01, 2.7102425702177583e-03, 1.3596534633026516e-04,
     1.1368339992439938e-05, 1.3307421378538766e-06, 2.0027891533891572e-07,
     3.6840601426594394e-08, 8.0088246676922984e-09, 2.0089007792250303e-09,
     5.7109046536952902e-10, 1.8144841846644442e-10, 6.3717614884074554e-11,
     2.4505045561973907e-11, 1.0242854871599881e-11, 4.6229709666475238e-12,
     2.2401796037642712e-12, 1.1595677255688565e-12, 6.3816583973407147e-13,
     3.7177904118758703e-13, 2.2830111699441359e-13, 1.4716263938306589e-13,
     9.9168068344998575e-14, 6.9578749350504175e-14, 5.0629197208994965e-14,
     3.8062784969029675e-14, 2.9459176184074689e-14, 2.3394653139013719e-14,
     1.9004998138677901e-14, 1.5750119898871165e-14, 1.3283262651519533e-14,
     1.1376166701753576e-14, 9.8750575440163840e-15, 8.6740952775626063e-15,
     7.6989709927555357e-15, 6.8965439605323114e-15, 6.2281572843133947e-15,
     5.6652414634023850e-15, 5.1863695510633072e-15, 4.7752506464040039e-15,
     4.4193406705404719e-15},
    // x = 30.0000
    {1.6180215937964007e-01, 2.6967026563257742e-03, 1.3483513281472911e-04,
     1.1236261066334487e-05, 1.3108971228460863e-06, 1.9663456686730906e-07,
     3.6049669032736161e-08, 7.8107600641556742e-09, 1.9526884564350898e-09,
     5.5326016971944726e-10, 1.7519749414066347e-10, 6.1317563345404073e-11,
     2.3503506345243414e-11, 9.7915680400232839e-12, 4.4046460141823363e-12,
     2.1273526363599893e-12, 1.0975725916245211e-12, 6.0210532156534654e-13,
     3.4966850041831192e-13, 2.1406930476315236e-13, 1.3758544426790896e-13,
     9.2457116421597806e-14, 6.4701329607338380e-14, 4.6966393377363737e-14,
     3.5230737650794916e-14, 2.7212165253342493e-14, 2.1570736637201090e-14,
     1.7494546868054268e-14, 1.4477064134243051e-14, 1.2193607099390873e-14,
     1.0430776486260998e-14, 9.0450189328919931e-15, 7.9376660513965703e-15,
     7.0395343942062586e-15, 6.3012095787236323e-15, 5.6867871873921528e-15,
     5.1697610102740184e-15, 4.7302720676933706e-15, 4.3532362564766849e-15,
     4.0270493676717204e-15},
};
