﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/auditmanager/AuditManagerRequest.h>
#include <aws/auditmanager/AuditManager_EXPORTS.h>
#include <aws/auditmanager/model/AssessmentReportsDestination.h>
#include <aws/auditmanager/model/Role.h>
#include <aws/auditmanager/model/Scope.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace AuditManager {
namespace Model {

/**
 */
class CreateAssessmentRequest : public AuditManagerRequest {
 public:
  AWS_AUDITMANAGER_API CreateAssessmentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateAssessment"; }

  AWS_AUDITMANAGER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> The name of the assessment to be created. </p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateAssessmentRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The optional description of the assessment to be created. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateAssessmentRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The assessment report storage destination for the assessment that's being
   * created. </p>
   */
  inline const AssessmentReportsDestination& GetAssessmentReportsDestination() const { return m_assessmentReportsDestination; }
  inline bool AssessmentReportsDestinationHasBeenSet() const { return m_assessmentReportsDestinationHasBeenSet; }
  template <typename AssessmentReportsDestinationT = AssessmentReportsDestination>
  void SetAssessmentReportsDestination(AssessmentReportsDestinationT&& value) {
    m_assessmentReportsDestinationHasBeenSet = true;
    m_assessmentReportsDestination = std::forward<AssessmentReportsDestinationT>(value);
  }
  template <typename AssessmentReportsDestinationT = AssessmentReportsDestination>
  CreateAssessmentRequest& WithAssessmentReportsDestination(AssessmentReportsDestinationT&& value) {
    SetAssessmentReportsDestination(std::forward<AssessmentReportsDestinationT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Scope& GetScope() const { return m_scope; }
  inline bool ScopeHasBeenSet() const { return m_scopeHasBeenSet; }
  template <typename ScopeT = Scope>
  void SetScope(ScopeT&& value) {
    m_scopeHasBeenSet = true;
    m_scope = std::forward<ScopeT>(value);
  }
  template <typename ScopeT = Scope>
  CreateAssessmentRequest& WithScope(ScopeT&& value) {
    SetScope(std::forward<ScopeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The list of roles for the assessment. </p>
   */
  inline const Aws::Vector<Role>& GetRoles() const { return m_roles; }
  inline bool RolesHasBeenSet() const { return m_rolesHasBeenSet; }
  template <typename RolesT = Aws::Vector<Role>>
  void SetRoles(RolesT&& value) {
    m_rolesHasBeenSet = true;
    m_roles = std::forward<RolesT>(value);
  }
  template <typename RolesT = Aws::Vector<Role>>
  CreateAssessmentRequest& WithRoles(RolesT&& value) {
    SetRoles(std::forward<RolesT>(value));
    return *this;
  }
  template <typename RolesT = Role>
  CreateAssessmentRequest& AddRoles(RolesT&& value) {
    m_rolesHasBeenSet = true;
    m_roles.emplace_back(std::forward<RolesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The identifier for the framework that the assessment will be created from.
   * </p>
   */
  inline const Aws::String& GetFrameworkId() const { return m_frameworkId; }
  inline bool FrameworkIdHasBeenSet() const { return m_frameworkIdHasBeenSet; }
  template <typename FrameworkIdT = Aws::String>
  void SetFrameworkId(FrameworkIdT&& value) {
    m_frameworkIdHasBeenSet = true;
    m_frameworkId = std::forward<FrameworkIdT>(value);
  }
  template <typename FrameworkIdT = Aws::String>
  CreateAssessmentRequest& WithFrameworkId(FrameworkIdT&& value) {
    SetFrameworkId(std::forward<FrameworkIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The tags that are associated with the assessment. </p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateAssessmentRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateAssessmentRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_description;

  AssessmentReportsDestination m_assessmentReportsDestination;

  Scope m_scope;

  Aws::Vector<Role> m_roles;

  Aws::String m_frameworkId;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_nameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_assessmentReportsDestinationHasBeenSet = false;
  bool m_scopeHasBeenSet = false;
  bool m_rolesHasBeenSet = false;
  bool m_frameworkIdHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace AuditManager
}  // namespace Aws
