/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include <libgnomedbmm.h>
#include <libgdamm.h>

#include <iostream>

#ifdef GLIBMM_EXCEPTIONS_ENABLED
void do_example()
#else
void do_example(std::auto_ptr<Glib::Error>& error)
#endif // GLIBMM_EXCEPTIONS_ENABLED
{
  Glib::RefPtr<Gnome::Gda::Dict> dict = Gnome::Gda::Dict::create();
  Glib::RefPtr<Gnome::Gda::Client> client = Gnome::Gda::Client::create();
  Glib::ustring connection_string = "DB_DIR=" LIBGNOMEDB_DATADIR ";DB_NAME=demo_db";

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  Glib::RefPtr<Gnome::Gda::Connection> cnc = client->open_connection_from_string("SQLite", connection_string, "" /* username */, "" /* password */);
  dict->set_connection(cnc);
  dict->update_dbms_meta_data();
#else
  Glib::RefPtr<Gnome::Gda::Connection> cnc = client->open_connection_from_string("SQLite", connection_string, "", "", Gnome::Gda::ConnectionOptions(0), error);
  if(error.get() != NULL) return;

  dict->set_connection(cnc);
  dict->update_dbms_meta_data(error);
  if(error.get() != NULL) return;
#endif
  
  // Specifies that this is an INSERT query
  Glib::RefPtr<Gnome::Gda::Query> query = Gnome::Gda::Query::create(dict, Gnome::Gda::QUERY_TYPE_INSERT);

  // Insert queries may only have one target
  Glib::RefPtr<Gnome::Gda::QueryTarget> products = Gnome::Gda::QueryTarget::create(query, "products");

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  query->add_target(products);
#else
  query->add_target(products, error);
  if(error.get() != NULL) return;
#endif

  // This specifies the field's values
  query->add_field( Gnome::Gda::QueryFieldField::create( query, "category", products, Gnome::Gda::QueryFieldValue::create(query, Gnome::Gda::Value(12)) ) );
  query->add_field( Gnome::Gda::QueryFieldField::create( query, "name", products, Gnome::Gda::QueryFieldValue::create(query, Gnome::Gda::Value("Highest Hopes")) ) );
  query->add_field( Gnome::Gda::QueryFieldField::create( query, "price", products, Gnome::Gda::QueryFieldValue::create(query, Gnome::Gda::Value(16.90)) ) );
  query->add_field( Gnome::Gda::QueryFieldField::create( query, "wh_stored", products, Gnome::Gda::QueryFieldValue::create(query, Gnome::Gda::Value(4)) ) );

  // Prints "INSERT INTO products (category, name, price, wh_stored) VALUES (12, 'Highest Hopes', 16.900000, 4)"
  std::cout << query->get_sql_text() << std::endl;
}

int main(int argc, char* argv[])
{
  Gnome::Db::init("Queries example", "1.0", argc, argv);

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  try
  {
    do_example();
  }
  catch(const Glib::Error& err)
  {
    std::cerr << "Exception caught: " << err.what() << std::endl;
    return 1;
  }
#else
  std::auto_ptr<Glib::Error> error;
  do_example(error);
  if(error.get() != NULL)
  {
    std::cerr << "Exception caught: " << error->what() << std::endl;
    return 1;
  }
#endif // GLIBMM_EXCEPTIONS_ENABLED

  return 0;
}
