/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */


#include <libgnomedbmm.h>
#include <libgdamm.h>
#include <gtkmm.h>

#include <iostream>

#include "examplewindow.h"

#ifdef GLIBMM_EXCEPTIONS_ENABLED
void do_example(const Glib::RefPtr<Gnome::Gda::Dict>& dict)
#else
void do_example(const Glib::RefPtr<Gnome::Gda::Dict>& dict, std::auto_ptr<Glib::Error>& error)
#endif // GLIBMM_EXCEPTIONS_ENABLED
{
  // File to try to read dictionary from
  std::string filename = "my_dict.xml";

  Glib::RefPtr<Gnome::Gda::Client> client = Gnome::Gda::Client::create();
  Glib::ustring connection_string = "DB_DIR=" LIBGNOMEDB_DATADIR ";DB_NAME=demo_db";

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  Glib::RefPtr<Gnome::Gda::Connection> cnc = client->open_connection_from_string("SQLite", connection_string, "" /* username */, "" /* password */);
#else
  Glib::RefPtr<Gnome::Gda::Connection> cnc = client->open_connection_from_string("SQLite", connection_string, "" /* username */, "" /* password */, Gnome::Gda::ConnectionOptions(0), error);
  if(error.get() != NULL) return;
#endif // GLIBMM_EXCEPTIONS_ENABLED
  dict->set_connection(cnc);

  // Load from XML file if present, query database otherwise
  if(Glib::file_test(filename, Glib::FILE_TEST_IS_REGULAR))
  {
#ifdef GLIBMM_EXCEPTIONS_ENABLED
    dict->load_xml_file(filename);
#else
    dict->load_xml_file(filename, error);
    if(error.get() != NULL) return;
#endif // GLIBMM_EXCEPTIONS_ENABLED
  }
  else
  {
#ifdef GLIBMM_EXCEPTIONS_ENABLED
    dict->update_dbms_meta_data();
#else
    dict->update_dbms_meta_data(error);
    if(error.get() != NULL) return;
#endif // GLIBMM_EXCEPTIONS_ENABLED
  }

  ExampleWindow window(dict);
  Gtk::Main::run(window);

  // Save dict back to XML after program termination
#ifdef GLIBMM_EXCEPTIONS_ENABLED
  dict->save_xml_file(filename);
#else
  dict->save_xml_file(filename, error);
#endif // GLIBMM_EXCEPTIONS_ENABLED
}


int main(int argc, char* argv[])
{
  Gtk::Main kit(argc, argv);
  Gnome::Db::init("Grid example", "1.0", argc, argv);

  Glib::RefPtr<Gnome::Gda::Dict> dict = Gnome::Gda::Dict::create();

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  try
  {
    do_example(dict);
  }
  catch(const Glib::Error& err)
  {
    std::cerr << "Exception caught: " << err.what() << std::endl;
    return 1;
  }
#else
  std::auto_ptr<Glib::Error> error;
  do_example(dict, error);
  if(error.get() != NULL)
  {
    std::cerr << "Exception caught: " << error->what() << std::endl;
    return 1;
  }
#endif // GLIBMM_EXCEPTIONS_ENABLED

  return 0;
}
