/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include "examplewindow.h" 

#include <libgdamm.h>
#include <iostream>

#ifdef GLIBMM_EXCEPTIONS_ENABLED
void ExampleWindow::create_model(const Glib::RefPtr<Gnome::Gda::Dict>& dict)
#else
void ExampleWindow::create_model(const Glib::RefPtr<Gnome::Gda::Dict>& dict, std::auto_ptr<Glib::Error>& error)
#endif // GLIBMM_EXCEPTIONS_ENABLED
{
  Glib::RefPtr<Gnome::Gda::Query> query = Gnome::Gda::Query::create(dict);

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  query->set_sql_text("SELECT ref, category, name, price, wh_stored FROM products");
#else
  query->set_sql_text("SELECT ref, category, name, price, wh_stored FROM products", error);
  if(error.get() != NULL) return;
#endif // GLIBMM_EXCEPTIONS_ENABLED
  
  m_model = Gnome::Gda::DataModelQuery::create(query);

  const Glib::ustring update_query =
                 "UPDATE products set "
                 "ref=##/*name:'+0' type:gchararray*/, "
                 "category=##/*name:'+1' type:gint*/,"
                 "name=##/*name:'+2' type:gchararray*/, "
		 "price=##/*name:'+3' type:gdouble*/, "
                 "wh_stored=##/*name:'+4' type:gint*/ "
                 "WHERE ref=##/*name:'-0' type:gchararray*/";

  const Glib::ustring delete_query =
                 "DELETE FROM products WHERE ref=##/*name:'-0' type:gchararray*/";

  const Glib::ustring insert_query =
                 "INSERT INTO products (ref, category, name, price, wh_stored) "
                 "VALUES (##/*name:'+0' type:gchararray*/, "
                 "##/*name:'+1' type:gint*/, "
                 "##/*name:'+2' type:gchararray*/, "
                 "1.0, "
                 "##/*name:'+4' type:gint*/)";

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  m_model->set_modification_query(update_query);
  m_model->set_modification_query(delete_query);
  m_model->set_modification_query(insert_query);
#else
  m_model->set_modification_query(update_query, error);
  m_model->set_modification_query(delete_query, error);
  m_model->set_modification_query(insert_query, error);
#endif
}

ExampleWindow::ExampleWindow(const Glib::RefPtr<Gnome::Gda::Dict>& dict) :
  m_label("The following Gnome::Db::Form widget displays data from the 'products' table.\n\n"
          "The 'price' field has special formatting rules:\nIt always shows two decimal places and the 'EUR' currency symbol."),
  m_form(0)
{    
  m_box.set_border_width(6);
  m_box.pack_start(m_label, Gtk::PACK_SHRINK);

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  try
  {
    create_model(dict);
  }
  catch(const Glib::Error& err)
  {
    std::cerr << "Exception caught: " << err.what() << std::endl;
    exit(1);
  }
#else
  std::auto_ptr<Glib::Error> error;
  create_model(dict, error);
  if(error.get() != NULL)
  {
    std::cerr << "Exception caught: " << error->what() << std::endl;
    exit(1);
  }
#endif

  /* Create the demo widget */
  m_form = Gtk::manage(new Gnome::Db::Form(m_model));
  /* Get the underlaying RawForm */
  Gnome::Db::RawForm* rawform = m_form->get_raw_form();
  /* Find the Gda::Parameter for the price field */
  Glib::RefPtr<Gnome::Gda::Parameter> param = rawform->get_current_data()->find_param("price");
  /* Set options for the plugin */
  param->set_entry_plugin("textual:CURRENCY=EUR;NB_DECIMALS=2");

  m_box.pack_start(*m_form);
  add(m_box);
  show_all();
}
