import gtk
import gobject

from gazpacho.l10n import _
from gazpacho.widget import get_widget_from_gtk_widget
from gazpacho.util import unselect_when_clicked_on_empty_space, select_iter
from gazpacho.util import unselect_when_press_escape, xml_filter_nodes

import xml.dom.minidom

class UIEditor(gtk.Dialog):
    def __init__(self):
        gtk.Dialog.__init__(self,
                            title='',
                            parent=None,
                            flags=gtk.DIALOG_DESTROY_WITH_PARENT,
                            buttons=(gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
        self.widget = None
        self.gwidget = None
        self.proxy = None
        # this boolean tells if we are editing a tree like UI
        # as a menubar, or a list like UI, as a toolbar
        self.is_tree = True
        
        # avoid destrcution when clicking the x button
        self.connect('delete-event', self._on_delete_event)
        self.connect('response', self._on_response)

        self.set_border_width(6)
        self.set_resizable(False)
        self.set_has_separator(False)
        self.vbox.set_spacing(6)
        
        self._create_widgets()

        self.vbox.show_all()
        
    def set_widget(self, widget, proxy):
        self.widget = widget
        self.gwidget = get_widget_from_gtk_widget(widget)
        if isinstance(widget, gtk.MenuBar):
            self.set_title(_('Editing menubar %s') % self.gwidget.name)
            self.is_tree = True
        elif isinstance(widget, gtk.Toolbar):
            self.set_title(_('Editing toolbar %s') % self.gwidget.name)
            self.is_tree = False
        else:
            raise ValueError(_('UIEditor is only for toolbars and menubars'))
        self.proxy = proxy
        self._load_ui()
        
    # internal methods
    def _create_widgets(self):
        h_button_box = gtk.HButtonBox()
        h_button_box.set_layout(gtk.BUTTONBOX_START)
        self.add = gtk.Button(stock=gtk.STOCK_ADD)
        self.add.connect('clicked', self._on_add_clicked)
        h_button_box.pack_start(self.add)
        self.remove = gtk.Button(stock=gtk.STOCK_REMOVE)
        self.remove.connect('clicked', self._on_remove_clicked)
        h_button_box.pack_start(self.remove)
        self.vbox.pack_start(h_button_box, False, False)

        hbox = gtk.HBox(spacing=6)

        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        hbox.pack_start(sw, True, True)

        self.model = gtk.TreeStore(object)
        self.treeview = gtk.TreeView(self.model)
        self.treeview.connect('button-press-event',
                              unselect_when_clicked_on_empty_space)
        self.treeview.connect('key-press-event',
                              unselect_when_press_escape)
        selection = self.treeview.get_selection()
        selection.connect('changed', self._on_selection_changed)
        self.treeview.set_size_request(300, 300)
        sw.add(self.treeview)
        self.treeview.set_headers_visible(False)
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn()
        self.treeview.append_column(column)
        column.pack_start(renderer)
        column.set_cell_data_func(renderer, self._cell_data_function)
        
        v_button_box = gtk.VButtonBox()
        v_button_box.set_layout(gtk.BUTTONBOX_START)
        self.up = gtk.Button(stock=gtk.STOCK_GO_UP)
        self.up.connect('clicked', self._on_up_clicked)
        v_button_box.pack_start(self.up)
        self.down = gtk.Button(stock=gtk.STOCK_GO_DOWN)
        self.down.connect('clicked', self._on_down_clicked)
        v_button_box.pack_start(self.down)
        self.left = gtk.Button(stock=gtk.STOCK_GO_BACK)
        self.left.connect('clicked', self._on_left_clicked)
        self.right = gtk.Button(stock=gtk.STOCK_GO_FORWARD)
        self.right.connect('clicked', self._on_right_clicked)
        if self.is_tree:
            v_button_box.pack_start(self.left)
            v_button_box.pack_start(self.right)
        
#        hbox.pack_start(v_button_box, False, False)

        self.vbox.pack_start(hbox, True, True)

        label = gtk.Label()
        label.set_markup('<small>%s</small>' % \
                         _('Press Escape to unselected an item'))
        self.vbox.pack_start(label, False, False)

    def _load_node(self, xml_node, parent_iter):
        new_iter = self.model.append(parent_iter, (xml_node,))
        for child_node in xml_filter_nodes(xml_node.childNodes,
                                           xml_node.ELEMENT_NODE):
            self._load_node(child_node, new_iter)
    
    def _load_ui(self):
        ui = self.gwidget.get_glade_property('ui').value
        self.model.clear()
        if ui:
            self.xml_doc = xml.dom.minidom.parseString(ui)
            self.xml_doc.normalize()
            self.root_node = self.xml_doc.documentElement
            typename = self.root_node.tagName
        else:
            # create empty XML document
            self.xml_doc = xml.dom.getDOMImplementation().createDocument(None,
                                                                         None,
                                                                         None)
            if self.is_tree:
                typename = 'menubar'
            else:
                typename = 'toolbar'
            self.root_node = self._create_xml_node(self.xml_doc, typename,
                                                   self.gwidget.name)
        for xml_node in xml_filter_nodes(self.root_node.childNodes,
                                         self.root_node.ELEMENT_NODE):
            self._load_node(xml_node, None)
            
        self._set_buttons_sensitiviness(typename)

    def _cell_data_function(self, column, cell, model, iter):
        xml_node = model.get_value(iter, 0)
        typename = xml_node.tagName
        name = xml_node.getAttribute('name')
        if typename == 'separator':
            name = '<separator>'
        cell.set_property('text', name)

    def _get_selected_iter(self):
        selection = self.treeview.get_selection()
        model, iter = selection.get_selected()
        return iter

    def _create_xml_node(self, parent, typename, name):
        element = self.xml_doc.createElement(typename)
        element.setAttribute('name', name)
        if typename != 'separator':
            element.setAttribute('action', name)
        return parent.appendChild(element)
    
    def _set_buttons_sensitiviness(self, typename):
        buttons = (self.add, self.remove, self.up, self.down, self.left,
                   self.right)
        sensitivity = {
            'menubar': (True, False, False, False, False, False),
            'menu': (True, True, True, True, False, True),
            'menuitem': (False, True, True, True, True, False),
            'toolbar':(True, False, False, False, False, False),
            'toolitem': (False, True, True, True, False, False ),
            'separator': (False, True, True, True, False, False),
            }
        for i, s in enumerate(sensitivity[typename]):
            buttons[i].set_sensitive(s)
    
    def _set_ui_property(self):
        project = self.gwidget.project
        app = project._app
        ui_string = self.root_node.toxml()
        self.proxy.set_value(ui_string)
        
    # callbacks
    def _on_delete_event(self, dialog, event):
        self.hide()
        return True
    
    def _on_response(self, dialog, response_id):
        self.hide()

    def _on_selection_changed(self, selection):
        model, selected_iter = selection.get_selected()
        if selected_iter is not None:
            xml_node = model.get_value(selected_iter, 0)
            name = xml_node.tagName
        else:
            name = self.root_node.tagName
        
        self._set_buttons_sensitiviness(name)
            
            
    def _on_add_clicked(self, button):
        project = self.gwidget.project
        parent_iter = self._get_selected_iter()
        if parent_iter is None:
            # we are adding a toplevel
            if self.is_tree:
                dialog = AddMenuDialog(project.action_groups, self)
                typename = 'menu'
            else:
                dialog = AddToolitemDialog(project.action_groups, self)
                typename = 'toolitem'
                
            if dialog.run() == gtk.RESPONSE_OK:
                name = dialog.get_name()
                if dialog.use_separator():
                    typename = 'separator'
                    name = 'sep%d' % project.separator_counter
                node = self._create_xml_node(self.root_node, typename, name)
                new_iter = self.model.append(None, (node,))
                select_iter(self.treeview, new_iter)
                self._set_ui_property()
        else:
            # we are adding a child in case this is a menu or just another
            # node if this is a toolbar
            if self.is_tree:
                dialog = AddMenuitemDialog(project.action_groups, self)
                typename = 'menuitem'
                parent_node = self.model.get_value(parent_iter, 0)
            else:
                dialog = AddToolitemDialog(project.action_groups, self)
                typename = 'toolitem'
                parent_node = self.root_node
                parent_iter = None
                
            if dialog.run() == gtk.RESPONSE_OK:
                name = dialog.get_name()
                if dialog.use_separator():
                    typename = 'separator'
                node = self._create_xml_node(parent_node, typename, name)
                new_iter = self.model.append(parent_iter, (node,))
                select_iter(self.treeview, new_iter)
                self._set_ui_property()
                
        dialog.destroy()
        
    def _on_remove_clicked(self, button):
        iter = self._get_selected_iter()
        if iter is None:
            return
        
        xml_node = self.model.get_value(iter, 0)
        self.model.remove(iter)
        
        parent_node = xml_node.parentNode
        parent_node.removeChild(xml_node)

        self._set_ui_property()

    def _on_up_clicked(self, button):
        select_iter = self._get_selected_iter()
        if select_iter is None:
            return
        
        xml_node = self.model.get_value(select_iter, 0)
        parent_node = xml_node.parentNode
        previous_sibling = xml_node.previousSibling
        if previous_sibling is not None:
            print previous_sibling
            tmp_node = parent_node.removeChild(xml_node)
            parent_node.insertBefore(tmp_node, previous_sibling)
            # not update the TreeView
            selected_path = self.model.get_path(select_iter)
            print selected_path, type(selected_path)
            sibling_iter = self.model.get_iter((selected_path[0] - 1,))
            self.model.move_before(select_iter, sibling_iter)
            
            self._set_ui_property()

    def _on_down_clicked(self, button):
        print 'not implemented yet'

    def _on_left_clicked(self, button):
        print 'not implemented yet'

    def _on_right_clicked(self, button):
        print 'not implemented yet'
    

class ChooseActionDialog(gtk.Dialog):
    """Dialog to choose an action from a list"""
    def __init__(self, action_groups=[], toplevel=None):
        gtk.Dialog.__init__(self,
                            title='',
                            parent=toplevel,
                            flags=gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                            buttons=(gtk.STOCK_OK, gtk.RESPONSE_OK,
                                     gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
        self.set_border_width(6)
        self.set_resizable(False)
        self.set_has_separator(False)
        self.vbox.set_spacing(6)

        self.label = gtk.Label(_('Choose an action')+':')
        self.vbox.pack_start(self.label, False)
        
        # create a combo with all the actions
        model = gtk.ListStore(str)
        self.actions = gtk.ComboBox(model)
        renderer = gtk.CellRendererText()
        self.actions.pack_start(renderer)
        self.actions.add_attribute(renderer, 'text', 0)
        
        for action_group in action_groups:
            for action in action_group.actions:
                model.append(('%s/%s' % (action_group.name, action.name),))
        
        self.actions.set_active(0)
        
        self.vbox.pack_start(self.actions)
        
        self.set_default_response(gtk.RESPONSE_OK)

    def get_selected_action_name(self):
        model = self.actions.get_model()
        active_iter = self.actions.get_active_iter()
        if active_iter is not None:
            name = model.get_value(active_iter, 0)
            action_name = name.split('/')[1]
            return action_name
        
    def use_separator(self):
        """Subclasses will probably override this"""
        return False

gobject.type_register(ChooseActionDialog)

class AddMenuDialog(ChooseActionDialog):
    def __init__(self, action_groups=[], toplevel=None):
        ChooseActionDialog.__init__(self, action_groups, toplevel)

        self.set_title(_('Adding a menu'))
        self.vbox.show_all()
        
    def get_name(self):
        return self.get_selected_action_name()

gobject.type_register(AddMenuDialog)

class AddToolitemDialog(AddMenuDialog):
    def __init__(self, action_groups=[], toplevel=None):
        AddMenuDialog.__init__(self, action_groups, toplevel)
        
        # put the combo in other place
        self.vbox.remove(self.actions)
        self.vbox.remove(self.label)
        
        hbox = gtk.HBox(spacing=6)
        self.select_action = gtk.RadioButton(None, _('Action'))
        self.select_separator = gtk.RadioButton(self.select_action, _('Separator'))

        hbox.pack_start(self.select_action, False, False)
        hbox.pack_start(self.actions)
        
        self.vbox.pack_start(hbox, False, False)
        self.vbox.pack_start(self.select_separator, False, False)
            
        self.set_title(_('Addind a toolitem'))
        self.vbox.show_all()

    def use_separator(self):
        return self.select_separator.get_active()
        
gobject.type_register(AddToolitemDialog)

class AddMenuitemDialog(AddToolitemDialog):
    def __init__(self, action_groups=[], toplevel=None):
        AddToolitemDialog.__init__(self, action_groups, toplevel)
        
        self.set_title(_('Adding a menuitem'))
        self.vbox.show_all()
    
gobject.type_register(AddMenuitemDialog)
