/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* vim: set sw=8: */

/*
 * guppi-gnumeric-vector.c: Handlers for sequence change notification.
 *
 * Copyright (C) 2000-2001 Jody Goldberg (jgoldberg@home.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "guppi-gnumeric-vector.h"
#include "guppi-gnumeric-manager.h"
#include "GNOME_Gnumeric_Graph.h"

#include <bonobo.h>
#include <gtk/gtkobject.h>
#include <guppi-seq-scalar.h>
#include <guppi-seq-date.h>
#include <guppi-seq-string.h>
#include <guppi-useful.h>

struct _GupGnmVector {
	GtkObject obj;

	GNOME_Gnumeric_VectorType type;
	GupGnmManager *manager;
	GuppiData    *data;
	GuppiData    *scalar_data;
	long id;

	gboolean initialized : 1;
	gboolean activated   : 1;
	gboolean has_header  : 1;

	GSList *name_indicies;

	CORBA_Object    corba_obj;	/* local CORBA object */
	union {
		POA_GNOME_Gnumeric_Scalar_Vector	scalar;
		POA_GNOME_Gnumeric_Date_Vector		date;
		POA_GNOME_Gnumeric_String_Vector	string;
		PortableServer_POA			any;
	} servant;

	/* The remote server monitoring this series */
	union {
		GNOME_Gnumeric_Scalar_Vector	scalar;
		GNOME_Gnumeric_Date_Vector	date;
		GNOME_Gnumeric_String_Vector	string;
		CORBA_Object			any;
	} subscriber;
};

typedef struct {
	GtkObjectClass parent_class;
} GupGnmVectorClass;

#define GNM_GRAPH_VECTOR_TYPE        (gup_gnm_vector_get_type ())
#define GNM_GRAPH_VECTOR(o)          (GTK_CHECK_CAST ((o), GNM_GRAPH_VECTOR_TYPE, GupGnmVector))
#define GNM_GRAPH_VECTOR_CLASS(k)    (GTK_CHECK_CLASS_CAST((k), GNM_GRAPH_VECTOR_TYPE, GupGnmVectorClass))
#define IS_GNM_GRAPH_VECTOR(o)       (GTK_CHECK_TYPE ((o), GNM_GRAPH_VECTOR_TYPE))
#define IS_GNM_GRAPH_VECTOR_CLASS(k) (GTK_CHECK_CLASS_TYPE ((k), GNM_GRAPH_VECTOR_TYPE))
#define SERVANT_TO_GNM_GRAPH_VECTOR(ptr) (GupGnmVector *)(((void *)ptr) - GTK_STRUCT_OFFSET(GupGnmVector, servant))

extern int debug_guppi_gnumeric;

#ifndef DISABLE_DEBUG
#define d(level, code)	do { if (debug_guppi_gnumeric >= level) { code; } } while (0)
#else
#define d(level, code)	
#endif

static GtkType gup_gnm_vector_get_type (void);

static GtkObjectClass *gup_gnm_vector_parent_class = NULL;
static POA_GNOME_Gnumeric_VectorSelection__vepv	vector_selection_vepv;
static POA_GNOME_Gnumeric_Scalar_Vector__vepv	scalar_vector_vepv;
static POA_GNOME_Gnumeric_Date_Vector__vepv	date_vector_vepv;
static POA_GNOME_Gnumeric_String_Vector__vepv	string_vector_vepv;

static void
impl_vector_selection_selected (PortableServer_Servant servant,
				const GNOME_Gnumeric_SeqPair * ranges,
				CORBA_Environment * ev)
{
	GupGnmVector *vector = SERVANT_TO_GNM_GRAPH_VECTOR (servant);

	g_warning ("Guppi : VectorSelection::selected (%p) placeholder\n", vector);
}

static void
impl_scalar_vector_changed (PortableServer_Servant servant,
			    const CORBA_short start,
			    const GNOME_Gnumeric_Scalar_Seq *vals,
			    CORBA_Environment *ev)
{
	GupGnmVector *vector = SERVANT_TO_GNM_GRAPH_VECTOR (servant);

	d(1, printf ("Guppi : scalar changed %p\n", vector));
	guppi_seq_scalar_set_many (GUPPI_SEQ_SCALAR (vector->data), 0,
		vals->_buffer, sizeof(double), vals->_length);

	d(5, unsigned i; for (i = 0; i < vals->_length; i++)
	  printf ("scalar[%d] = %f\n", i, vals->_buffer[i]););
#if 0
	if (vector->name_indicies != NULL) {
		/* TODO : What to do if someone uses a number as a name ?
		 * do we convert it to a string ? or just use the index as we do now.
		 */
		guppi_seq_scalar_set (GUPPI_SEQ_SCALAR (vector->container),
			vector->element, vals->_buffer[0]);
	}
#endif
}

static void
impl_date_vector_changed (PortableServer_Servant servant,
			  const CORBA_short start,
			  const GNOME_Gnumeric_Date_Seq *vals,
			  CORBA_Environment *ev)
{
	GupGnmVector *vector = SERVANT_TO_GNM_GRAPH_VECTOR (servant);

	d(1, printf ("Guppi : date changed %p\n", vector));
#if 0
	if (vector->name_indicies != NULL)
		/* TODO : What to do if someone uses a date as a name ?
		 * We need gnumerics format routines to do that.
		 */
		guppi_seq_date_set (GUPPI_SEQ_DATE (vector->container),
			vector->element, vals->_buffer[0]);
#endif
}

static void
impl_string_vector_changed (PortableServer_Servant servant,
			    const CORBA_short start,
			    const GNOME_Gnumeric_String_Seq *vals,
			    CORBA_Environment *ev)
{
	int i;
	GupGnmVector *vector = SERVANT_TO_GNM_GRAPH_VECTOR (servant);

	d(1, printf ("Guppi : string changed %p\n", vector));

	/* TODO : Do we need a set_many */
	i = vals->_length;
	while (i-- > 0 )
		guppi_seq_string_set (GUPPI_SEQ_STRING (vector->data),
			i, vals->_buffer[i]);

	if (vector->name_indicies != NULL) {
		GSList *ptr = vector->name_indicies;

		for (; ptr != NULL ; ptr = ptr->next)
			gup_gnm_manager_store_series_name (vector->manager,
				GPOINTER_TO_INT (ptr->data), vals->_buffer[0]);
	}
}

static void
gup_gnm_vector_corba_destroy (GupGnmVector *vector)
{
	CORBA_Environment ev;

	CORBA_exception_init (&ev);
	if (vector->subscriber.any != CORBA_OBJECT_NIL) {
		CORBA_Object_release(vector->subscriber.any, &ev);
		if (ev._major != CORBA_NO_EXCEPTION) {
			g_warning ("'%s' : while releasing a vector",
				   bonobo_exception_get_text (&ev));
		}
	}
	if (vector->activated) {
		PortableServer_ObjectId *oid;
		PortableServer_POA poa = bonobo_poa ();

		oid = PortableServer_POA_servant_to_id (poa,
			&vector->servant.any, &ev);
		PortableServer_POA_deactivate_object (poa, oid, &ev);
		vector->activated = FALSE;
		CORBA_free (oid);

		g_return_if_fail (ev._major == CORBA_NO_EXCEPTION);
	}
	if (vector->initialized) {
		switch (vector->type) {
		case GNOME_Gnumeric_VECTOR_TYPE_SCALAR :
			POA_GNOME_Gnumeric_Scalar_Vector__fini (
				&vector->servant.scalar, &ev);
			break;
		case GNOME_Gnumeric_VECTOR_TYPE_DATE :
			POA_GNOME_Gnumeric_Date_Vector__fini (
				&vector->servant.date, &ev);
			break;
		case GNOME_Gnumeric_VECTOR_TYPE_STRING :
			POA_GNOME_Gnumeric_String_Vector__fini (
				&vector->servant.string, &ev);
			break;
		default :
			g_warning ("Should not be reached");
		};

		g_return_if_fail (ev._major == CORBA_NO_EXCEPTION);
	}
	CORBA_exception_free (&ev);

}
static void
gup_gnm_vector_destroy (GtkObject *obj)
{
	GupGnmVector *vector = GNM_GRAPH_VECTOR (obj);

	d(1, printf ("guppi : graph-vector destroy %p\n", obj));

	if (vector->data != NULL) {
		guppi_unref (GTK_OBJECT (vector->data));
		vector->data = NULL;
	}
	if (vector->scalar_data != NULL) {
		guppi_unref (GTK_OBJECT (vector->scalar_data));
		vector->scalar_data = NULL;
	}

	gup_gnm_vector_clear_names (vector);
	gup_gnm_vector_corba_destroy (vector);

	if (gup_gnm_vector_parent_class->destroy)
		gup_gnm_vector_parent_class->destroy (obj);
}

static void
corba_implementation_classes_init (void)
{
	static POA_GNOME_Gnumeric_VectorSelection__epv	selection_epv;
	static POA_GNOME_Gnumeric_Scalar_Vector__epv	scalar_epv;
	static POA_GNOME_Gnumeric_Date_Vector__epv	date_epv;
	static POA_GNOME_Gnumeric_String_Vector__epv	string_epv;

	selection_epv.selected = &impl_vector_selection_selected;
	vector_selection_vepv.GNOME_Gnumeric_VectorSelection_epv =
		&selection_epv;

	scalar_epv.changed = &impl_scalar_vector_changed;
	scalar_vector_vepv.GNOME_Gnumeric_Scalar_Vector_epv =
		&scalar_epv;
	scalar_vector_vepv.GNOME_Gnumeric_VectorSelection_epv =
		&selection_epv;

	date_epv.changed = & impl_date_vector_changed;
	date_vector_vepv.GNOME_Gnumeric_Date_Vector_epv =
		&date_epv;
	date_vector_vepv.GNOME_Gnumeric_VectorSelection_epv =
		&selection_epv;

	string_epv.changed = & impl_string_vector_changed;
	string_vector_vepv.GNOME_Gnumeric_String_Vector_epv =
		&string_epv;
	string_vector_vepv.GNOME_Gnumeric_VectorSelection_epv =
		&selection_epv;
}

static void
gup_gnm_vector_class_init (GtkObjectClass *object_class)
{
	gup_gnm_vector_parent_class = gtk_type_class (gtk_object_get_type ());

	object_class->destroy = &gup_gnm_vector_destroy;

	corba_implementation_classes_init ();
}

static void
gup_gnm_vector_init (GtkObject *object)
{
	GupGnmVector *vector = GNM_GRAPH_VECTOR (object);

	vector->corba_obj = CORBA_OBJECT_NIL;
	vector->subscriber.any = CORBA_OBJECT_NIL;
	vector->activated = FALSE;
	vector->initialized = FALSE;
	vector->has_header = FALSE;
	vector->name_indicies = NULL;
	vector->data = NULL;
	vector->scalar_data = NULL;
}

static GtkType
gup_gnm_vector_get_type (void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"GupGnmVector",
			sizeof (GupGnmVector),
			sizeof (GupGnmVectorClass),
			(GtkClassInitFunc) gup_gnm_vector_class_init,
			(GtkObjectInitFunc) gup_gnm_vector_init,
			NULL, /* reserved 1 */
			NULL, /* reserved 2 */
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (gtk_object_get_type (), &info);
	}

	return type;
}

static GupGnmVector *
gup_gnm_vector_corba_init (GupGnmVector *vector)
{
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	switch (vector->type) {
	case GNOME_Gnumeric_VECTOR_TYPE_SCALAR :
		vector->servant.scalar.vepv = &scalar_vector_vepv;
		POA_GNOME_Gnumeric_Scalar_Vector__init (
			&vector->servant.scalar, &ev);
		break;

	case GNOME_Gnumeric_VECTOR_TYPE_DATE :
		vector->servant.date.vepv = &date_vector_vepv;
		POA_GNOME_Gnumeric_Date_Vector__init (
			&vector->servant.date, &ev);
		break;

	case GNOME_Gnumeric_VECTOR_TYPE_STRING :
		vector->servant.string.vepv = &string_vector_vepv;
		POA_GNOME_Gnumeric_String_Vector__init (
			&vector->servant.string, &ev);
		break;

	default :
		g_assert_not_reached ();
	};

	if (ev._major == CORBA_NO_EXCEPTION) {
		PortableServer_ObjectId *oid;
		PortableServer_POA poa = bonobo_poa ();
		
		vector->initialized = TRUE;

		oid = PortableServer_POA_activate_object (poa,
			&vector->servant.any, &ev);
		vector->activated = (ev._major == CORBA_NO_EXCEPTION);

		vector->corba_obj = PortableServer_POA_servant_to_reference (poa,
			&vector->servant.any, &ev);
		CORBA_free (oid);
	} else {
		g_warning ("'%s' : while creating a vector",
			   bonobo_exception_get_text (&ev));
		gtk_object_unref (GTK_OBJECT (vector));
		vector = NULL;
	}
	CORBA_exception_free (&ev);

	return vector;
}

GupGnmVector *
gup_gnm_vector_new (GupGnmManager *manager,
		    GNOME_Gnumeric_VectorSelection subscriber,
		    GNOME_Gnumeric_VectorType type,
		    GNOME_Gnumeric_VectorID id)
{
	GupGnmVector *vector = NULL;
	CORBA_Environment ev;
	CORBA_Object dup;

	CORBA_exception_init (&ev);
	dup = CORBA_Object_duplicate (subscriber, &ev);
	if (ev._major == CORBA_NO_EXCEPTION) {
		vector = guppi_type_new (gup_gnm_vector_get_type ());
		vector->subscriber.any = dup;
		vector->id = id;
		vector->type = type;

		d(2, printf ("Guppi : new vector (type = %d, id = %d) = %p\n", type, id, vector));

		vector = gup_gnm_vector_corba_init (vector);
		if (vector != NULL) {
			gup_gnm_manager_add_vector (manager, vector, vector->id);
			vector->manager = manager;
		}
	} else {
		g_warning ("'%s' : while initializing vector %p",
			   bonobo_exception_get_text (&ev), vector);
	}
	CORBA_exception_free (&ev);

	return vector;
}

/****************************************************************************/

GuppiData *
gup_gnm_vector_data_get (GupGnmVector *vector, gboolean scalar)
{
	g_return_val_if_fail (IS_GNM_GRAPH_VECTOR (vector), NULL);

	if (vector->data == NULL) {
		CORBA_Environment ev;
		CORBA_exception_init (&ev);
		switch (vector->type) {
		case GNOME_Gnumeric_VECTOR_TYPE_SCALAR : {
			GNOME_Gnumeric_Scalar_Seq *values;

			GNOME_Gnumeric_Scalar_Vector_value (vector->subscriber.scalar, &values, &ev);
			vector->data = guppi_seq_scalar_new ();
			d(2, printf ("Guppi : vector scalar init %p\n", vector));
			if (ev._major == CORBA_NO_EXCEPTION) {
				d(1, printf ("Guppi : scalar load %p\n", vector));
				d(5, unsigned i; for (i = 0; i < values->_length; i++)
				  printf ("scalar[%d] = %f\n", i, values->_buffer[i]););
				guppi_seq_scalar_append_many (GUPPI_SEQ_SCALAR (vector->data),
					values->_buffer, sizeof(double), values->_length);
			}
			break;
		}

		case GNOME_Gnumeric_VECTOR_TYPE_DATE : {
#if 0
			GNOME_Gnumeric_Date_Seq *values;
			GNOME_Gnumeric_Date_Vector_value (vector->subscriber.date, &values, &ev);
			vector->data = guppi_seq_date_new ();
			if (ev._major == CORBA_NO_EXCEPTION) {
				guppi_seq_date_append_many (GUPPI_SEQ_DATE (vector->data),
				       values->_buffer, sizeof(CORBA_long), values->_length);
				d(2, printf ("Guppi : vector date init %p\n", vector));
			}
#endif
			break;
		}

		case GNOME_Gnumeric_VECTOR_TYPE_STRING : {
			int i;
			GNOME_Gnumeric_String_Seq *values;

			GNOME_Gnumeric_String_Vector_value (vector->subscriber.string, &values, &ev);
			vector->data = guppi_seq_string_new ();
			d(2, printf ("Guppi : vector string init %p\n", vector));
			if (ev._major == CORBA_NO_EXCEPTION) {
				for (i = 0; i < values->_length; i++)
					guppi_seq_string_append (GUPPI_SEQ_STRING (vector->data),
								 values->_buffer [i]);
			}
			break;
		}

		default :
			g_assert_not_reached ();
		};
		if (ev._major != CORBA_NO_EXCEPTION) {
			g_warning ("'%s' : while initializing vector %p",
				   bonobo_exception_get_text (&ev), vector);
		}
		CORBA_exception_free (&ev);
	}

	if (scalar && vector->type != GNOME_Gnumeric_VECTOR_TYPE_SCALAR) {
		if (vector->scalar_data == NULL) {
			int i, n = guppi_seq_size (GUPPI_SEQ (vector->data));
			vector->scalar_data = guppi_seq_scalar_new ();
			for (i = 0; i < n ; )
				guppi_seq_scalar_append (GUPPI_SEQ_SCALAR (vector->scalar_data), ++i);
		}
		return vector->scalar_data;
	}

	return vector->data;
}

CORBA_Object
gup_gnm_vector_servant_get (GupGnmVector *vector)
{
	g_return_val_if_fail (IS_GNM_GRAPH_VECTOR (vector), CORBA_OBJECT_NIL);

	return vector->corba_obj;
}

void
gup_gnm_vector_mark_as_name (GupGnmVector *vector, unsigned element)
{
	g_return_if_fail (IS_GNM_GRAPH_VECTOR (vector));

	vector->name_indicies = g_slist_prepend (vector->name_indicies,
						 GINT_TO_POINTER (element));
}

void
gup_gnm_vector_clear_names (GupGnmVector *vector)
{
	g_return_if_fail (IS_GNM_GRAPH_VECTOR (vector));

	if (vector->name_indicies != NULL) {
		g_slist_free (vector->name_indicies);
		vector->name_indicies = NULL;
	}
}

GNOME_Gnumeric_VectorType
gup_gnm_vector_type (GupGnmVector *vec)
{
	g_return_val_if_fail (IS_GNM_GRAPH_VECTOR (vec),
			      GNOME_Gnumeric_VECTOR_TYPE_SCALAR);

	return vec->type;
}
