/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.glib;

/**
 * Value is a polymorphic type that can hold values of any other
 * type.  This is used internally and should not be exposed outside
 * of the library.
 */ 
public class Value extends Boxed {
	
    private Object javaval = null;
	
	/**
	 * Construct a new Value from a given org.gnu.glib.Type.
	 */
	public Value(Type type) {
		handle = Value.g_value_init(type.getTypeHandle());
	}
	
	/**
	 * Construct a new Value object using a handle to a native object.
	 * @param aHandle The handle
	 */
	public Value(Handle aHandle) {
		handle = aHandle;
	}
	
	/**
	 * Create a copy of this Value object.
	 * 
	 * @return An object of type Value that contains the same
	 * data as the current object.
	 */
	public Value copy() {
		Handle dest = Value.g_value_copy(handle);
		return new Value(dest);
	}

	/**
	 * Used internally by Java-Gnome to set a string value 
	 */
	public void setString( String value){
		if (value == null)
			value = "";
		g_value_set_string(handle, value);
	}

	/**
	 * Used internally by Java-Gnome
	 */
	public String getString(){
		return g_value_get_string(handle);
	}

	/**
	 * Used internally by Java-Gnome to set a boolean value 
	 */
	public void setBoolean(boolean value){
		g_value_set_boolean(handle, value );
	}

	/**
	 * Used internally by Java-Gnome
	 */
	public boolean getBoolean(){
		return g_value_get_boolean(handle);
	}
	
	/**
	 * Used internally by Java-Gnome to set an integer value 
	 */
	public void setInteger(int value){
		g_value_set_int(handle, value );
	}

	/**
	 * Used internally by Java-Gnome
	 */
	public int getInt(){
		return g_value_get_int(handle);
	}
	
	/**
	 * Used internally by Java-Gnome to set a long value 
	 */
	public void setLong(long value){
		g_value_set_long(handle, value );
	}

	/**
	 * Used internally by Java-Gnome
	 */
	public long getLong(){
		return g_value_get_long(handle);
	}
	
	/**
	 * Used internally by Java-Gnome to set a float value 
	 */
	public void setFloat(float value){
		g_value_set_float(handle, value );
	}
	
	/**
	 * Used internally by Java-Gnome
	 */
	public double getFloat(){
		return g_value_get_float(handle);
	}

	/**
	 * Used internally by Java-Gnome to set a double value 
	 */
	public void setDouble(double value){
		g_value_set_double(handle, value );
	}

	/**
	 * Used internally by Java-Gnome
	 */
	public double getDouble(){
		return g_value_get_double(handle);
	}


	/**
	 * Used internally by Java-Gnome
	 */
	public void setPixbuf(org.gnu.gdk.Pixbuf obj){
		g_value_set_pixbuf(handle, obj.getHandle());
	}

	/**
	 * Used internally
	 */
	public org.gnu.gdk.Pixbuf getPixbuf(){
		return new org.gnu.gdk.Pixbuf( g_value_get_pixbuf(handle));
	}

	/**
	 * Set the data held by this Value object with the given Object.
         * Objects that inherit from either {@link GObject} or {@link Boxed}
         * will be set directly in the C GValue structure.  Other "non-Glib"
         * objects will be kept locally in the Java object.
         * <p>
         * <b>NOTE</b>: You probably don't want to use this directly.  Prefer
         * using a convenience method such as: 
         * {@link GObject#setJavaObjectProperty}.
         * 
         * @param obj The object to set as the data value for this Value 
         * instance.
	 */
	public void setJavaObject(Object obj){
            if ( obj instanceof GObject ) {
                g_value_set_java_object( getHandle(), 
                                         ((GObject)obj).getHandle() );
            } else if ( obj instanceof Boxed ) {
                g_value_set_java_object( getHandle(), 
                                         ((Boxed)obj).getHandle() );
            } else {
                javaval = obj;
            }
	}

	/**
         * Get the data held by this Value object.
         * <p>
         * <b>NOTE</b>: You probably don't want to use this directly.  Prefer
         * using a convenience method such as: 
         * {@link GObject#getJavaObjectProperty}.
         *
         * @return The data value held by this Value instance.  If the data
         * is held in the C GValue structure, the returned object is an 
         * instance of {@link Handle}.  If the data is held locally in the 
         * Java object, that is returned, otherwise <tt>null</tt> is returned.
	 */
	public Object getJavaObject() {
        Object obj = (Object) g_value_get_java_object( getHandle() );
        if (obj instanceof Handle) {
            Handle hndl = (Handle) obj;
            if ( hndl == null || hndl.isNull() ) {
                return javaval;
            } else {
                return hndl;
            }
        }
        return obj;
	}
	
    native static final protected Handle g_value_init (int type);
    native static final protected int g_value_type(Handle value);
    native static final protected Handle g_value_copy (Handle srcValue);
	native static final protected Handle g_value_reset(Handle value);
	native static final protected void g_value_unset(Handle value);
	native static final protected void g_value_set_char(Handle value, byte vChar);
	native static final protected byte g_value_get_char(Handle value);
	native static final protected void g_value_set_boolean(Handle value, boolean vBoolean);
	native static final protected boolean g_value_get_boolean(Handle value);
	native static final protected void g_value_set_int(Handle value, int vInt);
	native static final protected int g_value_get_int(Handle value);
	native static final protected void g_value_set_long(Handle value, long vLong);
	native static final protected long g_value_get_long(Handle value);
	native static final protected void g_value_set_float(Handle value, double vFloat);
	native static final protected double g_value_get_float(Handle value);
	native static final protected void g_value_set_double(Handle value, double vDouble);
	native static final protected double g_value_get_double(Handle value);
	native static final protected void g_value_set_string(Handle value, String vString);
	native static final protected String g_value_get_string(Handle value);
	native static final protected void g_value_set_pixbuf(Handle value, Handle obj);
	native static final protected Handle g_value_get_pixbuf(Handle value);
	native static final protected void g_value_set_java_object(Handle value, Object obj);
	native static final protected Handle g_value_get_java_object(Handle value);
    native static final protected void g_value_set_pointer(Handle value, Handle ptr);
    native static final protected Handle g_value_get_pointer(Handle value);
    /**
     * Test if the value contains an {@link Enum}-derived class.
     */
//    native protected boolean g_value_is_enum(Handle value);
    /**
     * Gets the
     */
//    native protected String g_value_get_type_name(Handle value);
}
