#!/usr/bin/python2
## monitorDialog.py - UI code for selecting the monitor
## Copyright (C) 2001-2003 Red Hat, Inc.

## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.

## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

import string
import gtk
import gobject
import rhpxl.monitor

TRUE = 1
FALSE = 0

class MonitorDialog:
    def __init__(self, xml, monitor, probed_id, hardware_state, monitor_name=None):
        self.xml = xml
        self.monitor = monitor
        self.probed_id = probed_id
        self.hardware_state = hardware_state
        self.monitor_name = monitor_name

        self.db = self.monitor.readMonitorsDB()
        list = self.db.keys()
        list.sort()

        #put Generic LCD and Generic CRT at the front of the list
        tmp = list[list.index("Generic LCD Display")]
        list.remove(list[list.index("Generic LCD Display")])
        self.short_list = [tmp]

        tmp = list[list.index("Generic CRT Display")]
        list.remove(list[list.index("Generic CRT Display")])
        self.short_list = [tmp] + self.short_list

        self.long_list = self.short_list + list

        xml.get_widget("monitor_icon").set_from_file("/usr/share/system-config-display/pixmaps/monitor.png")

        self.ok_button = self.xml.get_widget("monitor_ok_button")
        self.ok_button.set_sensitive(False)

        self.probed_path = None

        self.changing_selection = 0
        
        dialog = self.xml.get_widget("monitor_dialog")
        dialog.set_transient_for(self.xml.get_widget("display_dialog"))
        dialog.set_modal(TRUE)

        self.monitor_store = gtk.TreeStore(gobject.TYPE_STRING, gobject.TYPE_STRING,
                                      gobject.TYPE_STRING, gobject.TYPE_STRING)

        self.tree_view = self.xml.get_widget("monitor_tree")
        self.tree_view.set_model(self.monitor_store)
        self.tree_view.set_direction(gtk.TEXT_DIR_LTR)

        for col in self.tree_view.get_columns():
            self.tree_view.remove_column(col)
        
        selection = self.tree_view.get_selection()
        selection.connect("changed", self.selection_changed)

        self.col = gtk.TreeViewColumn(None, gtk.CellRendererText(), text=0)
        self.tree_view.append_column(self.col)

        button = self.xml.get_widget("monitor_probe_button")
        button.connect("clicked", self.goto_path)
        button.set_sensitive(self.probed_id != "Unprobed Monitor")

        self.show_all_button = xml.get_widget("monitor_show_all")
        self.show_all_button.connect("toggled", self.enable_toggled)

    # terrible way to update this, but it works
    def enable_toggled(self, *args):
        self.hydrate_primary_monitor()

    def selection_changed(self, selection):
        if self.changing_selection:
            return
        (store, iter) = selection.get_selected()

        if iter == None:
            return
        else:
            if store.iter_depth(iter) == 0:
                self.ok_button.set_sensitive(False)
            else:
                self.ok_button.set_sensitive(True)

        if store.get_value(iter, 1) != None:
            model = store.get_value(iter, 0)
            vsync = store.get_value(iter, 2)
            hsync = store.get_value(iter, 3)

    def goto_path(self, *args):
        if self.probed_path:
            self.tree_view = self.xml.get_widget("monitor_tree")
            self.tree_view.expand_row(self.probed_path[:-1], TRUE)
            self.changing_selection = 1
            selection = self.tree_view.get_selection()
            selection.unselect_all()
            iter = self.monitor_store.get_iter(self.probed_path)
            selection.select_iter(iter)
            if self.monitor_store.iter_depth(iter) == 0:
                self.ok_button.set_sensitive(False)
            else:
                self.ok_button.set_sensitive(True)

            self.changing_selection = 0
            self.tree_view.set_cursor(self.probed_path, self.col, FALSE)
            self.tree_view.scroll_to_cell(self.probed_path, self.col, TRUE, 0.5, 0.0)
            self.tree_view.grab_focus ()

    def hydrate_primary_monitor(self):
        if self.show_all_button.get_active() == True:
            list = self.long_list
        else:
            list = self.short_list

        prev_iter = None

        self.monitor_store.clear()
        for maker in list:
            parent = self.monitor_store.append(None)
            self.monitor_store.set_value(parent, 0, maker)
            for mon in self.db[maker]:
                #Try to filter out entries with different EISA IDs but similar names
                if prev_iter:
                    if self.monitor_store.get_value(prev_iter, 0) == mon[0]:
                        continue

                iter = self.monitor_store.append(parent)
                
                self.monitor_store.set_value(iter, 0, mon[0])
                self.monitor_store.set_value(iter, 1, mon[1])
                self.monitor_store.set_value(iter, 2, mon[2])
                self.monitor_store.set_value(iter, 3, mon[3])

                if self.monitor_name == "Unknown monitor":
                    #The name in the file is "Unknown monitor", so let's try to probe
                    if self.probed_id and string.upper(mon[1]) == string.upper(self.probed_id):
                        self.probed_path = self.monitor_store.get_path(iter) 
                    elif self.hardware_state.get_monitor_name() == mon[0]:
                        #We couldn't probe the monitor (no self.probed_id), but maybe we have the name
                        #if the user has previously selected it in the list
                        self.probed_path = self.monitor_store.get_path(iter)
                else:
                    #We've found a name in the file, so use that
                    if self.monitor_name == mon[0]:
                        self.probed_path = self.monitor_store.get_path(iter)
                prev_iter = iter
                
    def dehydrate_primary_monitor(self, state):
        (store, iter) = self.tree_view.get_selection().get_selected()
        if iter == None:
            return

        self.probed_path = store.get_path(iter)

        if store.get_value(iter, 1) != None:
            model = store.get_value(iter, 0)
            vsync = store.get_value(iter, 2)
            hsync = store.get_value(iter, 3)

        state.set_monitor_name(model)
        state.set_hsync(hsync)
        state.set_vsync(vsync)

    def hydrate_second_monitor(self):
        if self.show_all_button.get_active() == True:
            list = self.long_list
        else:
            list = self.short_list

        self.monitor_store.clear()
        for maker in list:
            parent = self.monitor_store.append(None)
            self.monitor_store.set_value(parent, 0, maker)
            for mon in self.db[maker]:
                iter = self.monitor_store.append(parent)
                self.monitor_store.set_value(iter, 0, mon[0])
                self.monitor_store.set_value(iter, 1, mon[1])
                self.monitor_store.set_value(iter, 2, mon[2])
                self.monitor_store.set_value(iter, 3, mon[3])
                
                if self.monitor_name == mon[0]:
                    self.probed_path = self.monitor_store.get_path(iter)
                    
    def dehydrate_second_monitor(self, state):
        (store, iter) = self.tree_view.get_selection().get_selected()
        if iter == None:
            return

        self.probed_path = store.get_path(iter)

        if store.get_value(iter, 1) != None:
            model = store.get_value(iter, 0)
            vsync = store.get_value(iter, 2)
            hsync = store.get_value(iter, 3)

        return model, vsync, hsync

    def run(self):
        dialog = self.xml.get_widget("monitor_dialog")
        dialog.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
        dialog.present()
        while 1:
            res = dialog.run()
            if res != gtk.RESPONSE_OK:
                dialog.hide()
                return FALSE
            
            break
        
        dialog.hide()
        return TRUE
