# xserver.py - initial xserver startup for GUI mode.
#
# Michael Fulbright <msf@redhat.com>
# Matt Wilson <msw@redhat.com>
# Brent Fox <bfox@redhat.com>
# Chris Lumens <clumens@redhat.com>
#
# Copyright 1999-2008 Red Hat, Inc.
#
# This software may be freely redistributed under the terms of the GNU
# library public license.
#
# You should have received a copy of the GNU Library Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#

import os
import string
import subprocess
import sys
import time

import videocard
import monitor
import rhpl.translate as translate
import rhpl.arch
from rhpl.translate import _
from xhwstate import *
import xf86config

translate.textdomain('rhpxl')

END = 0
PROBED = 1
VESA = 2
FB = 3

class XServer:
    def __init__(self):
        self.config = None
        self.defaultdepth = 24
        self.display = ":9"
        self.hwstate = None
        self.keyboard = None
        self.logfile = "/dev/null"
        self.monitorhw = None
        self.mousehw = None
        self.resolution = "800x600"
        self.root = "/"
        self.serverflags = ["vt6", "-config", "/tmp/XConfig.test",
                            "-s", "1440", "-dpms",
                            "-v", "-ac", "-nolisten", "tcp"]
        self.videohw = None

    def generateConfig(self):
        if self.hwstate.videocard == None:
            return

        self.config = self.hwstate.generate_xconfig(keyboard = self.keyboard)

    def addExtraScreen(self, identifier):
        screen = xf86config.XF86ConfScreen()
        self.config.screen.insert(screen)
        screen.identifier = identifier
        # Might need to make this a parameter in the future.
        screen.device = "Videocard0"
        screen.defaultdepth = self.defaultdepth

        display = xf86config.XF86ConfDisplay()
        screen.display.insert(display)
        display.modes.insert(xf86config.XF86Mode(self.resolution))

    def writeConfig(self, filename="/tmp/XConfig.test"):
        if self.config:
            self.config.write(filename)

    def setHWState(self, confFile=None):
        self.hwstate = XF86HardwareState(defcard=self.videohw, defmon=self.monitorhw,
                                         probeflags=XF86HW_PROBE_NONE, xconfig=confFile)
        self.hwstate.set_videocard_driver(self.videohw.primaryCard().getDriver())

    def probeHW(self, skipMouseProbe=0, logFunc=None, forceDriver=None):
        self.videohw = videocard.VideoCardInfo(forceDriver=forceDriver)

        if self.videohw and self.videohw.primaryCard():
            cardstr = self.videohw.primaryCard().getDescription().encode("utf-8")
        else:
            cardstr = _("Unable to probe")

        cardstr = _("Probing for video card:   %s") % (cardstr,)
        if logFunc is None:
            print cardstr
        else:
            logFunc(cardstr)

        self.monitorhw = monitor.MonitorInfo()

    def startX(self, xStartedCB=None, useFB=False):
        if not os.access("/usr/bin/Xorg", os.X_OK):
            raise RuntimeError, "No X server binaries found to run"

        os.environ['DISPLAY'] = self.display

        # use the framebuffer server by default on ppc
        if rhpl.arch.getCanonArch() == "ppc64pseries" or useFB:
            attempt = FB
        else:
            attempt = PROBED

        while attempt != END:
            (found_card, attempt) = self._getAttempt(attempt, self.videohw.primaryCard())

            if found_card:
                try:
                    self.hwstate.set_resolution(self.resolution)
                    self.writeConfig()

                    self.hwstate.set_resolution(self.resolution, change=True)
                    pid = self._run(xStartedCB=xStartedCB)
                    return pid
                except (RuntimeError, IOError):
                    pass

        # If we make it here, no card was defined so we have no choice but to error
        raise RuntimeError, "No X server binaries found to run"

    def _getAttempt(self, attempt, primary_card):
        if attempt == END:
            return (False, END)
        elif attempt == PROBED:
            if primary_card is not None:
                print _("Attempting to start native X server")
                return (True, VESA)

            return (False, VESA)
        elif attempt == VESA:
            if primary_card and primary_card.getDescription() != "VESA driver (generic)":
                print _("Attempting to start VESA X server")
                if primary_card.getVideoRam():
                    self.videohw = videocard.VideoCardInfo(forceDriver="vesa")
                    self.hwstate.set_videocard_driver(self.videohw.primaryCard().getDriver())
                    self.hwstate.merge_into(self.config)
                    return (True, END)

            return (False, END)
        elif attempt == FB:
            print _("Attempting to start frame buffer X server")
            self.videohw = videocard.VideoCardInfo(forceDriver="fbdev")
            self.hwstate.set_videocard_driver(self.videohw.primaryCard().getDriver())
            self.hwstate.merge_into(self.config)
            return (True, END)

    def _run(self, xStartedCB = None):
        def chroot():
            if self.root != "/":
                os.chroot(self.root)
                os.chdir("/")

        args = ['-logfile', self.logfile, self.display] + self.serverflags

        if self.hwstate.videocard == None:
            sys.stderr.write("X SERVER FAILED\n")
            raise RuntimeError, "X server failed to start"

        noOutput = os.open("/dev/null", os.O_RDWR)

        try:
            proc = subprocess.Popen(["/usr/bin/Xorg"] + args, preexec_fn=chroot,
                                    stdout=noOutput, stderr=noOutput)

            sys.stdout.write(_("Waiting for X server to start...log located in %s\n") % self.logfile)
            sys.stdout.flush()

            for i in range(5):
                time.sleep(1)
                sys.stdout.write("%s..." % (i+1))
                sys.stdout.flush()

                # If the X process failed for some reason, raise an error now.
                if proc.poll() is not None:
                    raise OSError
        except OSError:
            sys.stderr.write("X SERVER FAILED\n");
            raise RuntimeError, "X server failed to start"
        except:
            import traceback
            (ty, value, tb) = sys.exc_info()
            lst = traceback.format_exception (ty, value, tb)
            text = string.joinfields (lst, "")
            print text

            sys.stderr.write("X SERVER FAILED\n")
            raise RuntimeError, "X server failed to start"

        print _(" X server started successfully.")

        if xStartedCB is not None:
            xStartedCB()

        return proc.pid
