"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdtDebugClient = void 0;
/*********************************************************************
 * Copyright (c) 2018, 2023 Ericsson and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
const cp = require("child_process");
const debugadapter_testsupport_1 = require("@vscode/debugadapter-testsupport");
const path = require("path");
const utils_1 = require("./utils");
const os = require("os");
const chai_1 = require("chai");
function getAdapterAndArgs(adapter) {
    const chosenAdapter = adapter !== undefined ? adapter : utils_1.defaultAdapter;
    const adapterPath = path.join(__dirname, '../../dist', chosenAdapter);
    if (process.env.INSPECT_DEBUG_ADAPTER) {
        return ['--inspect-brk', adapterPath];
    }
    return [adapterPath];
}
/**
 * Extend the standard DebugClient to support additional client features
 */
class CdtDebugClient extends debugadapter_testsupport_1.DebugClient {
    constructor(adapter, extraArgs) {
        // The unused are as such because we do override process launching
        super('unused', 'unused', 'gdb');
        /**
         * Reverse Request Handlers:
         */
        this.reverseRequestHandlers = {
            runInTerminal: (args) => __awaiter(this, void 0, void 0, function* () {
                const process = yield new Promise((resolve, reject) => {
                    const child = cp.spawn(args.args[0], args.args.slice(1), {
                        cwd: args.cwd,
                        env: sanitizeEnv(args.env),
                    });
                    if (typeof child.pid !== 'undefined') {
                        return resolve(child);
                    }
                    child.once('error', (error) => {
                        reject(error);
                    });
                });
                return {
                    processId: process.pid,
                };
            }),
        };
        this._cdt_args = getAdapterAndArgs(adapter);
        if (extraArgs) {
            this._cdt_args.push(...extraArgs);
        }
        // These timeouts should be smaller than what is in .mocharc.json and .mocharc-windows-ci.json
        // to allow the individual timeouts to fail before the whole test timesout.
        // This will mean error message on things such as waitForEvent will not
        // be hidden by overall test failure
        this.defaultTimeout = os.platform() === 'win32' ? 25000 / 2 : 5000 / 2;
    }
    /**
     * Start a debug session allowing command line arguments to be supplied
     */
    start(port) {
        if (typeof port === 'number') {
            return super.start(port);
        }
        return new Promise((resolve, reject) => {
            this._cdt_adapterProcess = cp.spawn('node', this._cdt_args);
            this._cdt_adapterProcess.on('error', (err) => {
                console.log(err);
                reject(err);
            });
            if (this._cdt_adapterProcess.stdout === null ||
                this._cdt_adapterProcess.stdin === null) {
                reject('Missing stdout/stdin');
                return;
            }
            this.connect(this._cdt_adapterProcess.stdout, this._cdt_adapterProcess.stdin);
            resolve();
        });
    }
    stop() {
        return super
            .stop()
            .then(() => {
            this.killAdapter();
        })
            .catch(() => {
            this.killAdapter();
        });
    }
    killAdapter() {
        if (this._cdt_adapterProcess) {
            this._cdt_adapterProcess.kill();
            this._cdt_adapterProcess = undefined;
        }
    }
    /**
     * Notify the Debug Adapter by default that this client supports `runInTerminal`.
     */
    initializeRequest(args) {
        if (!args) {
            args = {
                supportsRunInTerminalRequest: true,
                adapterID: this['_debugType'],
                linesStartAt1: true,
                columnsStartAt1: true,
                pathFormat: 'path',
            };
        }
        return super.initializeRequest(args);
    }
    /**
     * Send a continueRequest and wait for target to stop
     */
    continue(args, reason, expected) {
        return __awaiter(this, void 0, void 0, function* () {
            const waitForStopped = this.assertStoppedLocation(reason, expected);
            const continueResp = this.continueRequest(args);
            yield Promise.all([waitForStopped, continueResp]);
            return waitForStopped;
        });
    }
    /**
     * Send a nextRequest and wait for target to stop
     */
    next(args, expected) {
        return __awaiter(this, void 0, void 0, function* () {
            const waitForStopped = this.assertStoppedLocation('step', expected);
            const next = this.nextRequest(args);
            yield Promise.all([waitForStopped, next]);
            return waitForStopped;
        });
    }
    /**
     * Send a stepInRequest and wait for target to stop
     */
    stepIn(args, expected) {
        return __awaiter(this, void 0, void 0, function* () {
            const waitForStopped = this.assertStoppedLocation('step', expected);
            const next = this.stepInRequest(args);
            yield Promise.all([waitForStopped, next]);
            return waitForStopped;
        });
    }
    /**
     * Send a stepOutRequest and wait for target to stop
     */
    stepOut(args, expected) {
        return __awaiter(this, void 0, void 0, function* () {
            const waitForStopped = this.assertStoppedLocation('step', expected);
            const next = this.stepOutRequest(args);
            yield Promise.all([waitForStopped, next]);
            return waitForStopped;
        });
    }
    /**
     * Send a stepBackRequest and wait for target to stop
     */
    stepBack(args, expected) {
        return __awaiter(this, void 0, void 0, function* () {
            const waitForStopped = this.assertStoppedLocation('step', expected);
            const next = this.stepBackRequest(args);
            yield Promise.all([waitForStopped, next]);
            return waitForStopped;
        });
    }
    /*
     * Returns a promise that will resolve if an output event
     * with a specific category and optional output message was received.
     * The promise will be rejected if a timeout occurs.
     */
    waitForOutputEvent(category, output, timeout = this.defaultTimeout) {
        return __awaiter(this, void 0, void 0, function* () {
            const isOutputEvent = (event) => {
                return (!!event.body &&
                    !!event.body.output);
            };
            return new Promise((resolve, reject) => {
                const outputProcessor = (event) => {
                    if (isOutputEvent(event) && event.body.category === category) {
                        if (output === undefined || output === event.body.output) {
                            clearTimeout(timer);
                            this.off('output', outputProcessor);
                            resolve(event);
                        }
                    }
                };
                const timer = setTimeout(() => {
                    this.off('output', outputProcessor);
                    reject(new Error(`no output event with category '${category}' ${output === undefined
                        ? ''
                        : `and output message '${output}'`} received after ${timeout} ms`));
                }, timeout);
                this.on('output', outputProcessor);
            });
        });
    }
    /**
     * Send a response following a Debug Adapter Reverse Request.
     * @param request original request to respond to.
     * @param handler processes the request and returns the response body.
     */
    doRespond(request) {
        return __awaiter(this, void 0, void 0, function* () {
            const { command } = request;
            const handler = this['reverseRequestHandlers'][command];
            const response = {
                type: 'response',
                request_seq: request.seq,
                command,
                success: true,
            };
            if (!handler) {
                response.success = false;
                response.message = `Unknown command: ${command}`;
            }
            else {
                try {
                    response.body = yield handler(request.arguments);
                }
                catch (error) {
                    response.success = false;
                    response.message =
                        error instanceof Error ? error.message : String(error);
                }
            }
            const json = JSON.stringify(response);
            this['outputStream'].write(`Content-Length: ${Buffer.byteLength(json, 'utf-8')}\r\n\r\n${json}`);
        });
    }
    readMemoryRequest(args) {
        return this.send('readMemory', args);
    }
    writeMemoryRequest(args) {
        return this.send('writeMemory', args);
    }
    attachHitBreakpoint(attachArgs, breakpoint) {
        return Promise.all([
            this.waitForEvent('initialized')
                .then((_event) => {
                return this.setBreakpointsRequest({
                    breakpoints: [{ line: breakpoint.line }],
                    source: { path: breakpoint.path },
                });
            })
                .then((response) => {
                const bp = response.body.breakpoints[0];
                (0, chai_1.expect)(bp.verified).to.be.true;
                (0, chai_1.expect)(bp.line).to.equal(breakpoint.line);
                return Promise.all([
                    this.configurationDoneRequest(),
                    this.assertStoppedLocation('breakpoint', breakpoint),
                ]);
            }),
            this.initializeRequest().then((_response) => {
                return this.attachRequest(attachArgs);
            }),
        ]);
    }
    /**
     * Obtain the value of the expression in the context of the
     * top frame, of the first returned thread.
     * @param name name of the variable
     */
    evaluate(expression) {
        return __awaiter(this, void 0, void 0, function* () {
            const threads = yield this.threadsRequest();
            const stack = yield this.stackTraceRequest({
                threadId: threads.body.threads[0].id,
            });
            const evalResponse = yield this.evaluateRequest({
                expression,
                frameId: stack.body.stackFrames[0].id,
            });
            return evalResponse.body.result;
        });
    }
}
exports.CdtDebugClient = CdtDebugClient;
/**
 * DebugProtocol.dispatch is private, overriding manually.
 */
CdtDebugClient.prototype['dispatch'] = function dispatch(raw) {
    const message = JSON.parse(raw);
    if (isRequest(message)) {
        this['doRespond'](message);
    }
    else {
        debugadapter_testsupport_1.DebugClient.prototype['dispatch'].apply(this, [raw]);
    }
};
function isRequest(message) {
    return message.type === 'request';
}
function sanitizeEnv(env) {
    if (!env) {
        return {};
    }
    const sanitized = {};
    for (const key of Object.keys(env)) {
        if (typeof env[key] === 'string') {
            sanitized[key] = env[key];
        }
    }
    return sanitized;
}
//# sourceMappingURL=debugClient.js.map