# This file contains tests for the tclProc.c source file. Tests appear in
# the same order as the C code that they test. The set of tests is
# currently incomplete since it includes only new tests, in particular
# tests for code changed for the addition of Tcl namespaces. Other
# procedure-related tests appear in other test files such as proc-old.test.
#
# Sourcing this file into Tcl runs the tests and generates output for
# errors.  No output means no errors were found.
#
# Copyright (c) 1997 Sun Microsystems, Inc.
# Copyright (c) 1998-1999 by Scriptics Corporation.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# RCS: @(#) $Id: proc.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $

if {[lsearch [namespace children] ::tcltest] == -1} {
    package require tcltest
    namespace import -force ::tcltest::*
}

catch {eval namespace delete [namespace children :: test_ns_*]}
catch {rename p ""}
catch {rename {} ""}
catch {unset msg}

test proc-1.1 {Tcl_ProcObjCmd, put proc in namespace specified in name, if any} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    namespace eval test_ns_1 {
        namespace eval baz {}
    }
    proc test_ns_1::baz::p {} {
        return "p in [namespace current]"
    }
    list [test_ns_1::baz::p] \
         [namespace eval test_ns_1 {baz::p}] \
         [info commands test_ns_1::baz::*]
} {{p in ::test_ns_1::baz} {p in ::test_ns_1::baz} ::test_ns_1::baz::p}
test proc-1.2 {Tcl_ProcObjCmd, namespace specified in proc name must exist} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    list [catch {proc test_ns_1::baz::p {} {}} msg] $msg
} {1 {can't create procedure "test_ns_1::baz::p": unknown namespace}}
test proc-1.3 {Tcl_ProcObjCmd, empty proc name} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    proc :: {} {
        return "empty called"
    }
    list [::] \
         [info body {}]
} {{empty called} {
        return "empty called"
    }}
test proc-1.4 {Tcl_ProcObjCmd, simple proc name and proc defined in namespace} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    namespace eval test_ns_1 {
        namespace eval baz {
            proc p {} {
                return "p in [namespace current]"
            }
        }
    }
    list [test_ns_1::baz::p] \
         [info commands test_ns_1::baz::*]
} {{p in ::test_ns_1::baz} ::test_ns_1::baz::p}
test proc-1.5 {Tcl_ProcObjCmd, qualified proc name and proc defined in namespace} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    namespace eval test_ns_1::baz {}
    namespace eval test_ns_1 {
        proc baz::p {} {
            return "p in [namespace current]"
        }
    }
    list [test_ns_1::baz::p] \
         [info commands test_ns_1::baz::*] \
         [namespace eval test_ns_1::baz {namespace which p}]
} {{p in ::test_ns_1::baz} ::test_ns_1::baz::p ::test_ns_1::baz::p}
test proc-1.6 {Tcl_ProcObjCmd, namespace code ignores single ":"s in middle or end of command names} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    namespace eval test_ns_1 {
        proc q: {} {return "q:"}
        proc value:at: {} {return "value:at:"}
    }
    list [namespace eval test_ns_1 {q:}] \
         [namespace eval test_ns_1 {value:at:}] \
         [test_ns_1::q:] \
         [test_ns_1::value:at:] \
         [lsort [info commands test_ns_1::*]] \
         [namespace eval test_ns_1 {namespace which q:}] \
         [namespace eval test_ns_1 {namespace which value:at:}]
} {q: value:at: q: value:at: {::test_ns_1::q: ::test_ns_1::value:at:} ::test_ns_1::q: ::test_ns_1::value:at:}
test proc-1.7 {Tcl_ProcObjCmd, check that formal parameter names are not array elements} {
    catch {rename p ""}
    list [catch {proc p {a(1) a(2)} { 
            set z [expr $a(1)+$a(2)]
            puts "$z=z, $a(1)=$a(1)"
        }} msg] $msg
} {1 {procedure "p" has formal parameter "a(1)" that is an array element}}
test proc-1.8 {Tcl_ProcObjCmd, check that formal parameter names are simple names} {
    catch {rename p ""}
    list [catch {proc p {b:a b::a} { 
    }} msg] $msg
} {1 {procedure "p" has formal parameter "b::a" that is not a simple name}}

test proc-2.1 {TclFindProc, simple proc name and proc not in namespace} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    catch {rename p ""}
    proc p {} {return "p in [namespace current]"}
    info body p
} {return "p in [namespace current]"}
test proc-2.2 {TclFindProc, simple proc name and proc defined in namespace} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    namespace eval test_ns_1 {
        namespace eval baz {
            proc p {} {return "p in [namespace current]"}
        }
    }
    namespace eval test_ns_1::baz {info body p}
} {return "p in [namespace current]"}
test proc-2.3 {TclFindProc, qualified proc name and proc defined in namespace} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    namespace eval test_ns_1::baz {}
    namespace eval test_ns_1 {
        proc baz::p {} {return "p in [namespace current]"}
    }
    namespace eval test_ns_1 {info body baz::p}
} {return "p in [namespace current]"}
test proc-2.4 {TclFindProc, global proc and executing in namespace} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    catch {rename p ""}
    proc p {} {return "global p"}
    namespace eval test_ns_1::baz {info body p}
} {return "global p"}

test proc-3.1 {TclObjInterpProc, proc defined and executing in same namespace} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    proc p {} {return "p in [namespace current]"}
    p
} {p in ::}
test proc-3.2 {TclObjInterpProc, proc defined and executing in same namespace} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    namespace eval test_ns_1::baz {
        proc p {} {return "p in [namespace current]"}
        p
    }
} {p in ::test_ns_1::baz}
test proc-3.3 {TclObjInterpProc, proc defined and executing in different namespaces} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    catch {rename p ""}
    proc p {} {return "p in [namespace current]"}
    namespace eval test_ns_1::baz {
        p
    }
} {p in ::}
test proc-3.4 {TclObjInterpProc, procs execute in the namespace in which they were defined unless renamed into new namespace} {
    catch {eval namespace delete [namespace children :: test_ns_*]}
    catch {rename p ""}
    namespace eval test_ns_1::baz {
        proc p {} {return "p in [namespace current]"}
        rename ::test_ns_1::baz::p ::p
        list [p] [namespace which p]
    }
} {{p in ::} ::p}
test proc-3.5 {TclObjInterpProc, any old result is reset before appending error msg about missing arguments} {
    proc p {x} {info commands 3m}
    list [catch {p} msg] $msg
} {1 {wrong # args: should be "p x"}}

test proc-3.6 {TclObjInterpProc, proper quoting of proc name, Bug 942757} {
    proc {a b  c} {x} {info commands 3m}
    list [catch {{a b  c}} msg] $msg
} {1 {wrong # args: should be "{a b  c} x"}}

catch {eval namespace delete [namespace children :: test_ns_*]}
catch {rename p ""}
catch {rename {} ""}
catch {rename {a b  c} {}}
catch {unset msg}

if {[catch {package require procbodytest}]} {
    puts "This application couldn't load the \"procbodytest\" package, so I"
    puts "can't test creation of procs whose bodies have type \"procbody\"."
    ::tcltest::cleanupTests
    return
}

catch {rename p ""}
catch {rename t ""}

# Note that the test require that procedures whose body is used to create
# procbody objects must be executed before the procbodytest::proc command
# is executed, so that the Proc struct is populated correctly (CompiledLocals
# are added at compile time).

test proc-4.1 {TclCreateProc, procbody obj} {
    catch {
	proc p x {return "$x:$x"}
	set rv [p P]
	procbodytest::proc t x p
	lappend rv [t T]
	set rv
    } result
    catch {rename p ""}
    catch {rename t ""}
    set result
} {P:P T:T}

test proc-4.2 {TclCreateProc, procbody obj, use compiled locals} {
    catch {
	proc p x {
	    set y [string tolower $x]
	    return "$x:$y"
	}
	set rv [p P]
	procbodytest::proc t x p
	lappend rv [t T]
	set rv
    } result
    catch {rename p ""}
    catch {rename t ""}
    set result
} {P:p T:t}

test proc-4.3 {TclCreateProc, procbody obj, too many args} {
    catch {
	proc p x {
	    set y [string tolower $x]
	    return "$x:$y"
	}
	set rv [p P]
	procbodytest::proc t {x x1 x2} p
	lappend rv [t T]
	set rv
    } result
    catch {rename p ""}
    catch {rename t ""}
    set result
} {procedure "t": arg list contains 3 entries, precompiled header expects 1}

test proc-4.4 {TclCreateProc, procbody obj, inconsitent arg name} {
    catch {
	proc p {x y z} {
	    set v [join [list $x $y $z]]
	    set w [string tolower $v]
	    return "$v:$w"
	}
	set rv [p P Q R]
	procbodytest::proc t {x x1 z} p
	lappend rv [t S T U]
	set rv
    } result
    catch {rename p ""}
    catch {rename t ""}
    set result
} {procedure "t": formal parameter 1 is inconsistent with precompiled body}

test proc-4.5 {TclCreateProc, procbody obj, inconsitent arg default type} {
    catch {
	proc p {x y {z Z}} {
	    set v [join [list $x $y $z]]
	    set w [string tolower $v]
	    return "$v:$w"
	}
	set rv [p P Q R]
	procbodytest::proc t {x y z} p
	lappend rv [t S T U]
	set rv
    } result
    catch {rename p ""}
    catch {rename t ""}
    set result
} {procedure "t": formal parameter 2 is inconsistent with precompiled body}

test proc-4.6 {TclCreateProc, procbody obj, inconsitent arg default type} {
    catch {
	proc p {x y z} {
	    set v [join [list $x $y $z]]
	    set w [string tolower $v]
	    return "$v:$w"
	}
	set rv [p P Q R]
	procbodytest::proc t {x y {z Z}} p
	lappend rv [t S T U]
	set rv
    } result
    catch {rename p ""}
    catch {rename t ""}
    set result
} {procedure "t": formal parameter 2 is inconsistent with precompiled body}

test proc-4.7 {TclCreateProc, procbody obj, inconsitent arg default value} {
    catch {
	proc p {x y {z Z}} {
	    set v [join [list $x $y $z]]
	    set w [string tolower $v]
	    return "$v:$w"
	}
	set rv [p P Q R]
	procbodytest::proc t {x y {z ZZ}} p
	lappend rv [t S T U]
	set rv
    } result
    catch {rename p ""}
    catch {rename t ""}
    set result
} {procedure "t": formal parameter "z" has default value inconsistent with precompiled body}

test proc-5.1 {Bytecompiling noop; test for correct argument substitution} {
    proc p args {} ; # this will be bytecompiled into t
    proc t {} {
	set res {}
	set a 0
	set b 0
	trace add variable a read {append res a ;#}
	trace add variable b write {append res b ;#}
	p $a ccccccw {bfe} {$a} [incr b] [incr a] {[incr b]} {$a} hello
	set res
    }
    set result [t]
    catch {rename p ""}
    catch {rename t ""}
    set result
} {aba}    

test proc-6.1 {ProcessProcResultCode: Bug 647307 (negative return code)} {
    proc a {} {return -code -5}
    proc b {} a
    set result [catch b]
    rename a {}
    rename b {}
    set result
} -5

# cleanup
catch {rename p ""}
catch {rename t ""}
::tcltest::cleanupTests
return












