/*
 * Decompiled with CFR 0.152.
 */
package oracle.kv.util;

import com.sleepycat.je.CacheMode;
import com.sleepycat.je.Database;
import com.sleepycat.je.DatabaseConfig;
import com.sleepycat.je.DatabaseEntry;
import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.DiskOrderedCursor;
import com.sleepycat.je.DiskOrderedCursorConfig;
import com.sleepycat.je.DiskOrderedCursorProducerException;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.EnvironmentNotFoundException;
import com.sleepycat.je.ForwardCursor;
import com.sleepycat.je.OperationStatus;
import com.sleepycat.je.PreloadConfig;
import com.sleepycat.je.StatsConfig;
import com.sleepycat.je.ThreadInterruptedException;
import com.sleepycat.je.utilint.JVMSystemUtils;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.io.RandomAccessFile;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.rmi.RemoteException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Logger;
import oracle.kv.BulkWriteOptions;
import oracle.kv.EntryStream;
import oracle.kv.KVStore;
import oracle.kv.KVStoreConfig;
import oracle.kv.KVStoreFactory;
import oracle.kv.Key;
import oracle.kv.KeyValue;
import oracle.kv.LoginCredentials;
import oracle.kv.Value;
import oracle.kv.impl.admin.CommandServiceAPI;
import oracle.kv.impl.admin.SecurityStore;
import oracle.kv.impl.admin.TableStore;
import oracle.kv.impl.admin.client.CommandShell;
import oracle.kv.impl.api.KVStoreImpl;
import oracle.kv.impl.api.table.TableAPIImpl;
import oracle.kv.impl.api.table.TableImpl;
import oracle.kv.impl.api.table.TableMetadata;
import oracle.kv.impl.metadata.Metadata;
import oracle.kv.impl.security.metadata.SecurityMetadata;
import oracle.kv.impl.security.util.KVStoreLogin;
import oracle.kv.impl.test.TestHook;
import oracle.kv.impl.test.TestHookExecute;
import oracle.kv.impl.topo.PartitionId;
import oracle.kv.impl.util.CommandParser;
import oracle.kv.impl.util.server.LoggerUtils;
import oracle.kv.table.Table;
import oracle.kv.util.shell.ShellException;

public class Load
implements KVStoreLogin.CredentialsProvider {
    public static final String COMMAND_NAME = "load";
    public static final String COMMAND_DESC = "Loads Admin metadata or data into a store from backup directories.";
    private static final String LOAD_ADMIN_COMMAND_DESC = "Loads Admin metadata into a store from backup directory.";
    private static final String LOAD_ADMIN_COMMAND_ARGS = CommandParser.getStoreUsage() + " " + CommandParser.getHostUsage() + " " + CommandParser.getPortUsage() + "\n\t-load-admin -source <admin-backup-dir> " + CommandParser.optional((String)"-force") + "\n\t" + CommandParser.optional((String)CommandParser.getUserUsage()) + " " + CommandParser.optional((String)CommandParser.getSecurityUsage());
    private static final String LOAD_DATA_COMMAND_DESC = "Loads data into a store from backup directories.";
    private static final String LOAD_DATA_COMMAND_ARGS = CommandParser.getStoreUsage() + " " + CommandParser.getHostUsage() + " " + CommandParser.getPortUsage() + "\n\t-source <shard-backup-dir>[, <shard-backup-dir>]*\n\t" + CommandParser.optional((String)"-checkpoint <checkpoint-files-directory>") + "\n\t" + CommandParser.optional((String)CommandParser.getUserUsage()) + " " + CommandParser.optional((String)CommandParser.getSecurityUsage());
    public static final String COMMAND_ARGS = LOAD_ADMIN_COMMAND_ARGS + "\n\t" + "Loads Admin metadata into a store from backup directory." + "\n\n" + "Usage: java -jar KVHOME/lib/kvstore.jar " + "load" + " " + CommandParser.optional((String)"-verbose") + "\n\t" + LOAD_DATA_COMMAND_ARGS + "\n\t" + "Loads data into a store from backup directories.";
    private static final String LOCK_FILE = "cp.lck";
    private static final int STREAM_PARALLELISM_ENV_DOC = 1;
    private static final int STREAM_PARALLELISM_DB_DOC = 3;
    private static final SimpleDateFormat utcDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS z");
    private static final Key SESSION_PARENT_KEY;
    private final List<SnapshotEnvironment> environments;
    private PrintStream output;
    private boolean verboseOutput;
    private KVStoreLogin storeLogin;
    private LoginCredentials loginCreds;
    private final KVStore userStore;
    private final KVStore internalStore;
    private final String storeName;
    private final String hostname;
    private AtomicLong loadedCount;
    private RandomAccessFile checkpointLockFile;
    private FileChannel checkpointLockChannel;
    private FileLock checkpointEnvLock;
    private final int bulkputStreamParallelism;
    private final int bulkputHeapPercent;
    private final int bulkputPerShardParallelism;
    private final int bulkputMaxRequestSize;
    private final long requestTimeoutMs;
    private static final int BULKPUT_HEAP_PERCENT_DEF = 20;
    private static final int STREAM_CACHE_PERCENT_DEF = 60;
    private static final int ENV_CACHE_PERCENT_DEF = 66;
    private final int streamCachePercent;
    private final int envCachePercent;
    private long envCacheSize;
    private long docCacheSize;
    private final boolean useEnvDOC;
    private final Map<String, String> envConfigProps;
    private final Set<String> sysTableIds;
    private TestHook<LoadStream> streamGetNext;

    public Load(File[] envDirs, String storeName, String targetHost, int targetPort, String user, String securityFile, String checkpointDir, String statsDir, boolean useEnvDOC, Map<String, String> envConfigProps, int bulkputStreamParallelism, int bulkputHeapPercent, int bulkputPerShardParallelism, int bulkputMaxRequestSize, long requestTimeoutMs, int streamCachePercent, int envCachePercent, boolean verboseOutput, PrintStream output) throws Exception {
        this.verboseOutput = verboseOutput;
        this.output = output;
        this.storeName = storeName;
        this.hostname = this.getLocalHostName();
        this.useEnvDOC = useEnvDOC;
        this.envConfigProps = envConfigProps;
        this.bulkputStreamParallelism = bulkputStreamParallelism == 0 ? (useEnvDOC ? 1 : 3) : bulkputStreamParallelism;
        this.bulkputHeapPercent = bulkputHeapPercent == 0 ? 20 : bulkputHeapPercent;
        this.bulkputPerShardParallelism = bulkputPerShardParallelism;
        this.bulkputMaxRequestSize = bulkputMaxRequestSize;
        this.requestTimeoutMs = requestTimeoutMs;
        this.streamCachePercent = streamCachePercent == 0 ? 60 : streamCachePercent;
        int n = this.envCachePercent = envCachePercent == 0 ? 66 : envCachePercent;
        if (this.bulkputHeapPercent + this.streamCachePercent > 85) {
            throw new IllegalArgumentException("The sum of bulkput-heap-percent and stream-cache-percent must typically be less than 85, leaving 15% as room for the GC to collect objects. -bulkput-heap-percent: " + this.bulkputHeapPercent + ", -stream-cache-percent: " + this.streamCachePercent);
        }
        this.verbose(String.format("\nArguments:\n\tstoreName: %s\n\thelper-host: %s\n\tbulkputStreamParallelism: %d\n\tbulkputHeapPercent: %d\n\tbulkputPerShardParallelism: %d\n\tbulkputMaxRequestSize: %d\n\trequestTimeoutMs: %d\n\tstreamCachePercent: %d\n\tenvCachePercent: %d", this.storeName, targetHost + ":" + targetPort, this.bulkputStreamParallelism, this.bulkputHeapPercent, this.bulkputPerShardParallelism, this.bulkputMaxRequestSize, this.requestTimeoutMs, this.streamCachePercent, this.envCachePercent));
        this.computeStreamCacheSizes();
        if (checkpointDir != null) {
            this.lockCheckpointDir(checkpointDir);
        }
        this.environments = new ArrayList<SnapshotEnvironment>(envDirs.length);
        for (File envDir : envDirs) {
            this.environments.add(new SnapshotEnvironment(envDir, checkpointDir, statsDir));
        }
        String[] hosts = new String[]{targetHost + ":" + targetPort};
        this.prepareAuthentication(user, securityFile);
        KVStoreConfig kvConfig = new KVStoreConfig(storeName, hosts[0]);
        kvConfig.setSecurityProperties(this.storeLogin.getSecurityProperties());
        this.userStore = KVStoreFactory.getStore(kvConfig, this.loginCreds, KVStoreLogin.makeReauthenticateHandler(this));
        this.internalStore = KVStoreImpl.makeInternalHandle(this.userStore);
        this.sysTableIds = this.getSysTableIds();
        this.verbose("Opened store " + storeName);
    }

    public Load(File[] envDirs, String storeName, String targetHost, int targetPort, String user, String securityFile, String checkpointDir, boolean verboseOutput, PrintStream output) throws Exception {
        this(envDirs, storeName, targetHost, targetPort, user, securityFile, checkpointDir, null, false, null, 0, 0, 0, 0, 0L, 0, 0, verboseOutput, output);
    }

    public Load(File envDir, String storeName, String targetHost, int targetPort, String user, String securityFile, String checkpointDir, boolean verboseOutput, PrintStream output) throws Exception {
        this(new File[]{envDir}, storeName, targetHost, targetPort, user, securityFile, checkpointDir, verboseOutput, output);
    }

    @Override
    public LoginCredentials getCredentials() {
        return this.loginCreds;
    }

    public void setOutput(PrintStream output) {
        this.output = output;
    }

    public PrintStream getOutput() {
        return this.output;
    }

    public void setVerbose(boolean verboseOutput) {
        this.verboseOutput = verboseOutput;
    }

    public boolean getVerbose() {
        return this.verboseOutput;
    }

    private void message(String msg) {
        if (this.output != null) {
            String now = utcDateFormat.format(new Date());
            this.output.println(String.format("%s [LOAD] %s", now, msg));
        }
    }

    private void verbose(String msg) {
        if (this.verboseOutput) {
            this.message(msg);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long run() throws Exception {
        this.loadedCount = new AtomicLong();
        List<EntryStream<KeyValue>> streams = null;
        try {
            streams = this.createLoadStreams();
            if (streams.isEmpty()) {
                this.message("No more database to load.");
                long l = 0L;
                return l;
            }
            BulkWriteOptions wro = new BulkWriteOptions(null, 0L, null);
            wro.setStreamParallelism(this.bulkputStreamParallelism);
            if (this.bulkputHeapPercent > 0) {
                wro.setBulkHeapPercent(this.bulkputHeapPercent);
            }
            if (this.bulkputPerShardParallelism > 0) {
                wro.setPerShardParallelism(this.bulkputPerShardParallelism);
            }
            if (this.bulkputMaxRequestSize > 0) {
                wro.setMaxRequestSize(this.bulkputMaxRequestSize);
            }
            if (this.requestTimeoutMs > 0L) {
                wro.setTimeout(this.requestTimeoutMs, TimeUnit.MILLISECONDS);
            }
            this.verbose(String.format("Bulk put write options: %d streamParallelism; %d heapPercent; %d perShardParallelism; %d maxRequestSize; %,d requestTimeoutMs\n", wro.getStreamParallelism(), wro.getBulkHeapPercent(), wro.getPerShardParallelism(), wro.getMaxRequestSize(), wro.getTimeoutUnit() == null ? (long)((KVStoreImpl)this.internalStore).getDefaultRequestTimeoutMs() : wro.getTimeout()));
            try {
                this.verbose("Starting loading..");
                this.internalStore.put(streams, wro);
                this.verbose("Loading is done.");
            }
            catch (RuntimeException re) {
                this.message("Loading failed: " + re.getMessage());
                throw re;
            }
        }
        finally {
            if (streams != null) {
                for (EntryStream<KeyValue> stream : streams) {
                    ((LoadStream)stream).close();
                }
            }
            this.close();
        }
        return this.loadedCount.get();
    }

    private void computeStreamCacheSizes() {
        long streamCacheSize = JVMSystemUtils.getRuntimeMaxMemory() * (long)this.streamCachePercent / 100L / (long)this.bulkputStreamParallelism;
        if (this.useEnvDOC) {
            this.envCacheSize = streamCacheSize * (long)this.envCachePercent / 100L;
            this.docCacheSize = streamCacheSize - this.envCacheSize;
            this.verbose("JE Snapshot preloaded env internal memory limit: " + Load.toMB(this.envCacheSize) + "MB");
            this.verbose("DiskOrderedCursor internal memory limit: " + Load.toMB(this.docCacheSize) + "MB");
        } else {
            this.docCacheSize = streamCacheSize;
            this.verbose("DiskOrderedCursor internal memory limit: " + Load.toMB(this.docCacheSize) + "MB");
        }
    }

    private long getStreamEnvCacheSize() {
        return this.envCacheSize;
    }

    private long getStreamDocCacheSize() {
        return this.docCacheSize;
    }

    private void prepareAuthentication(String user, String securityFile) throws Exception {
        this.storeLogin = new KVStoreLogin(user, securityFile);
        try {
            this.storeLogin.loadSecurityProperties();
        }
        catch (IllegalArgumentException iae) {
            this.message(iae.getMessage());
        }
        if (this.storeLogin.foundTransportSettings()) {
            this.loginCreds = this.storeLogin.makeShellLoginCredentials();
        }
    }

    private void close() {
        if (this.environments != null) {
            for (SnapshotEnvironment se : this.environments) {
                se.close();
            }
        }
        if (this.userStore != null) {
            this.userStore.close();
        }
        this.releaseCheckpointDirLock();
    }

    private List<EntryStream<KeyValue>> createLoadStreams() {
        ArrayList<EntryStream<KeyValue>> streams = new ArrayList<EntryStream<KeyValue>>();
        if (this.useEnvDOC) {
            for (SnapshotEnvironment se : this.environments) {
                streams.add(new SnapshotStream(se));
            }
        } else {
            ArrayList<SnapshotEnvironment> envs = new ArrayList<SnapshotEnvironment>();
            for (SnapshotEnvironment se : this.environments) {
                this.verbose("Opened source backup directory " + se.getEnvDir());
                List<String> dbToLoads = se.getDatabasesToLoad();
                if (dbToLoads.isEmpty()) {
                    this.message("No more database to load in: " + se.getName());
                    continue;
                }
                envs.add(se);
            }
            int ind = 0;
            while (!envs.isEmpty()) {
                Iterator envIter = envs.iterator();
                while (envIter.hasNext()) {
                    SnapshotEnvironment se = (SnapshotEnvironment)envIter.next();
                    List<String> dbsToLoad = se.getDatabasesToLoad();
                    if (ind < dbsToLoad.size()) {
                        streams.add(new DatabaseStream(se, dbsToLoad.get(ind)));
                        continue;
                    }
                    envIter.remove();
                }
                ++ind;
            }
        }
        return streams;
    }

    private void tallyLoadedCount(long count) {
        this.loadedCount.addAndGet(count);
    }

    private synchronized void displayLoadProgress(String envDir, String database, long count, long skipCount, long dupCount, long elapseTimeMs) {
        String info;
        if (database == null) {
            String fmt = "Load %,d records, %,d records skipped, %,d pre-existing records from %s: ";
            info = String.format(fmt, count, skipCount, dupCount, envDir);
        } else {
            String fmt = "Load %,d records, %,d records skipped, %,d pre-existing records from %s of %s: ";
            info = String.format(fmt, count, skipCount, dupCount, database, envDir);
        }
        info = elapseTimeMs > 60000L ? info + String.format("%,dm%.3fs", elapseTimeMs / 60000L, Float.valueOf((float)(elapseTimeMs % 60000L) / 1000.0f)) : info + String.format("%.3fs", Float.valueOf((float)elapseTimeMs / 1000.0f));
        this.message(info);
    }

    private void lockCheckpointDir(String checkpointDir) {
        try {
            this.checkpointLockFile = new RandomAccessFile(new File(checkpointDir, LOCK_FILE), "rwd");
            this.checkpointLockChannel = this.checkpointLockFile.getChannel();
            String msg = "Another load is already running. Failed to acquire a lock on checkpoint files directory: " + checkpointDir;
            try {
                this.checkpointEnvLock = this.checkpointLockChannel.tryLock(1L, 1L, false);
                if (this.checkpointEnvLock == null) {
                    throw new IllegalStateException(msg);
                }
            }
            catch (OverlappingFileLockException ofle) {
                throw new IllegalStateException(msg, ofle);
            }
        }
        catch (IOException ioe) {
            throw new IllegalArgumentException("Failed to open checkpoint files directory: " + checkpointDir, ioe);
        }
        catch (SecurityException se) {
            throw new IllegalArgumentException("Failed to open checkpoint files directory: " + checkpointDir, se);
        }
    }

    private void releaseCheckpointDirLock() {
        try {
            if (this.checkpointEnvLock != null) {
                this.checkpointEnvLock.release();
            }
            if (this.checkpointLockChannel != null) {
                this.checkpointLockChannel.close();
            }
            if (this.checkpointLockFile != null) {
                this.checkpointLockFile.close();
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private String getLocalHostName() {
        try {
            return InetAddress.getLocalHost().getHostName();
        }
        catch (UnknownHostException e) {
            return "localhost";
        }
    }

    public void setLoadStreamGetNext(TestHook<LoadStream> testHook) {
        this.streamGetNext = testHook;
    }

    public static void main(String[] args) throws Exception {
        LoadParser lp = new LoadParser(args);
        lp.parseArgs();
        if (lp.getLoadAdmin()) {
            Load.loadAdmin(lp);
            return;
        }
        try {
            Load load = new Load(lp.getSourceDirs(), lp.getStoreName(), lp.getHostname(), lp.getRegistryPort(), lp.getUserName(), lp.getSecurityFile(), lp.getCheckpointDir(), lp.getStatsDir(), lp.useEnvDOC(), lp.getJEProps(), lp.getBulkputStreamParallelism(), lp.getBulkputHeapPercent(), lp.getBulkputPerShardParallelism(), lp.getBulkputMaxRequestSize(), lp.getRequestTimeoutMs(), lp.getStreamCachePercent(), lp.getEnvCachePercent(), lp.getVerbose(), System.out);
            long total = load.run();
            System.out.println("Load succeeded, wrote " + total + " records");
        }
        catch (Exception e) {
            System.err.println("Load operation failed with exception: " + LoggerUtils.getStackTrace((Throwable)e));
            System.exit(-1);
        }
    }

    private static void loadAdmin(LoadParser lp) {
        Load.loadAdmin(lp.getSourceDirs()[0], lp.getHostname(), lp.getRegistryPort(), lp.getUserName(), lp.getSecurityFile(), lp.getVerbose(), lp.getForceLoadAdmin(), System.out);
    }

    public static void loadAdmin(File envDir, String targetHost, int targetPort, String user, String securityFile, boolean verboseOutput, boolean forceLoad, PrintStream output) {
        try {
            LoadAdmin load = new LoadAdmin(envDir, targetHost, targetPort, user, securityFile, verboseOutput, forceLoad, output);
            load.loadMetadata();
        }
        catch (Exception e) {
            System.err.println("Admin load operation failed with exception: " + LoggerUtils.getStackTrace((Throwable)e));
        }
    }

    private static Environment openEnvironment(File envDir, Map<String, String> configProps) {
        if (!envDir.isDirectory()) {
            System.err.println("Environment path is not a directory or does not exist: " + envDir);
            throw new IllegalArgumentException("Bad environment directory: " + envDir);
        }
        EnvironmentConfig envConfig = new EnvironmentConfig();
        if (configProps != null) {
            for (Map.Entry<String, String> prop : configProps.entrySet()) {
                try {
                    envConfig.setConfigParam(prop.getKey(), prop.getValue());
                }
                catch (IllegalArgumentException iae) {
                    throw new IllegalArgumentException("Invalid environment configuration parameter '-" + prop.getKey() + " " + prop.getValue() + "': " + iae.getMessage());
                }
            }
        }
        envConfig.setTransactional(false);
        envConfig.setAllowCreate(false);
        envConfig.setReadOnly(true);
        envConfig.setSharedCache(true);
        try {
            return new Environment(envDir, envConfig);
        }
        catch (EnvironmentNotFoundException e) {
            throw new IllegalArgumentException("Cannot find valid Environment in directory: " + envDir);
        }
    }

    private Set<String> getSysTableIds() {
        HashSet<String> tableIds = new HashSet<String>();
        for (TableImpl table : ((TableAPIImpl)this.internalStore.getTableAPI()).getSystemTables()) {
            tableIds.add(table.getIdString());
        }
        return tableIds;
    }

    private boolean isSysTableRecord(Key key) {
        return this.sysTableIds.contains(key.getMajorPath().get(0));
    }

    private boolean isSessionRecord(Key key) {
        return SESSION_PARENT_KEY.isPrefix(key);
    }

    private static String toMB(long size) {
        return String.format("%.2f", (double)size / 1048576.0);
    }

    static {
        utcDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        SESSION_PARENT_KEY = Key.fromString("///sess");
    }

    abstract class LoadStream
    implements EntryStream<KeyValue> {
        private long readCount;
        private long skipCount;
        private final AtomicLong keyExists;
        final SnapshotEnvironment env;
        private ForwardCursor cursor;
        private final DatabaseEntry key;
        private final DatabaseEntry data;
        private long startTimeMs;
        private long endTimeMs;

        LoadStream(SnapshotEnvironment env) {
            this.env = env;
            this.cursor = null;
            this.key = new DatabaseEntry();
            this.data = new DatabaseEntry();
            this.readCount = 0L;
            this.skipCount = 0L;
            this.keyExists = new AtomicLong();
            this.startTimeMs = 0L;
        }

        abstract DiskOrderedCursor openCursor();

        abstract String getDatabase();

        public String toString() {
            String fmt = "%s %,d rows read; %,d row skipped; %,d pre-existing rows.";
            return String.format("%s %,d rows read; %,d row skipped; %,d pre-existing rows.", this.name(), this.readCount, this.skipCount, this.keyExists.get());
        }

        @Override
        public KeyValue getNext() {
            KeyValue entry;
            assert (TestHookExecute.doHookIfSet(Load.this.streamGetNext, this));
            if (this.startTimeMs == 0L) {
                this.startTimeMs = System.currentTimeMillis();
            }
            if (this.cursor == null) {
                this.cursor = this.openCursor();
                if (this.cursor == null) {
                    return null;
                }
            }
            if ((entry = this.readNextEntry()) == null) {
                this.close();
                Load.this.verbose(this.name() + " read EOS: " + this.readCount);
                return null;
            }
            return entry;
        }

        @Override
        public void completed() {
            this.endTimeMs = System.currentTimeMillis();
            long nLoaded = this.readCount - this.keyExists.get() - this.skipCount;
            Load.this.tallyLoadedCount(nLoaded);
            this.env.setLoaded(this);
        }

        @Override
        public void keyExists(KeyValue entry) {
            this.keyExists.incrementAndGet();
        }

        @Override
        public void catchException(RuntimeException rte, KeyValue entry) {
            throw rte;
        }

        Database openDatabase(String name) {
            DatabaseConfig dbConfig = new DatabaseConfig();
            dbConfig.setAllowCreate(false);
            dbConfig.setReadOnly(true);
            dbConfig.setCacheMode(CacheMode.EVICT_LN);
            return this.env.getJEEnv().openDatabase(null, name, dbConfig);
        }

        long getReadCount() {
            return this.readCount;
        }

        long getSkippedCount() {
            return this.skipCount;
        }

        long getKeyExistCount() {
            return this.keyExists.get();
        }

        long getElapsedTime() {
            return this.endTimeMs == 0L ? 0L : this.endTimeMs - this.startTimeMs;
        }

        private KeyValue readNextEntry() {
            try {
                while (this.cursor.getNext(this.key, this.data, null) == OperationStatus.SUCCESS) {
                    ++this.readCount;
                    Key k = Key.fromByteArray(this.key.getData());
                    if (Load.this.isSysTableRecord(k) || Load.this.isSessionRecord(k)) {
                        ++this.skipCount;
                        continue;
                    }
                    Value v = this.data.getData().length == 0 ? Value.EMPTY_VALUE : Value.fromByteArray(this.data.getData());
                    return new KeyValue(k, v);
                }
            }
            catch (DiskOrderedCursorProducerException dcpe) {
                Load.this.message("Failed to read entry from cursor:" + dcpe.getMessage());
                throw dcpe;
            }
            catch (ThreadInterruptedException tie) {
                Load.this.message("Failed to read entry from cursor:" + tie.getMessage());
                throw tie;
            }
            return null;
        }

        void close() {
            if (this.cursor != null) {
                try {
                    this.cursor.close();
                }
                catch (ThreadInterruptedException threadInterruptedException) {
                    // empty catch block
                }
                this.cursor = null;
            }
        }
    }

    class SnapshotStream
    extends LoadStream {
        private Database[] dbsToLoad;

        SnapshotStream(SnapshotEnvironment env) {
            super(env);
        }

        @Override
        public String name() {
            return "LoadStream_" + this.env.getName();
        }

        @Override
        DiskOrderedCursor openCursor() {
            List<String> dbNames = this.env.getDatabasesToLoad();
            if (dbNames.isEmpty()) {
                return null;
            }
            long startMs = System.currentTimeMillis();
            this.dbsToLoad = this.openDatabases(dbNames);
            PreloadConfig preConfig = new PreloadConfig().setLoadLNs(false).setInternalMemoryLimit(Load.this.getStreamEnvCacheSize());
            this.env.getJEEnv().preload(this.dbsToLoad, preConfig);
            Load.this.verbose(String.format("Environment preload time:%,d ms", System.currentTimeMillis() - startMs));
            DiskOrderedCursorConfig docConfig = new DiskOrderedCursorConfig().setInternalMemoryLimit(Load.this.getStreamDocCacheSize());
            return this.env.getJEEnv().openDiskOrderedCursor(this.dbsToLoad, docConfig);
        }

        private Database[] openDatabases(List<String> dbNames) {
            ArrayList<Database> dbs = new ArrayList<Database>();
            for (String name : dbNames) {
                Database db = this.openDatabase(name);
                dbs.add(db);
            }
            return dbs.toArray(new Database[dbs.size()]);
        }

        @Override
        String getDatabase() {
            List<String> dbs = this.env.getDatabasesToLoad();
            return dbs.isEmpty() ? null : dbs.toString();
        }

        @Override
        void close() {
            super.close();
            if (this.dbsToLoad != null) {
                for (Database db : this.dbsToLoad) {
                    try {
                        db.close();
                    }
                    catch (ThreadInterruptedException threadInterruptedException) {
                        // empty catch block
                    }
                }
                this.dbsToLoad = null;
            }
            this.env.close();
        }
    }

    class DatabaseStream
    extends LoadStream {
        private final String database;
        private Database db;

        DatabaseStream(SnapshotEnvironment env, String database) {
            super(env);
            this.database = database;
        }

        @Override
        public String name() {
            return "LoadStream_" + this.database;
        }

        @Override
        DiskOrderedCursor openCursor() {
            this.db = this.openDatabase(this.database);
            DiskOrderedCursorConfig docConfig = new DiskOrderedCursorConfig().setInternalMemoryLimit(Load.this.getStreamDocCacheSize());
            return this.db.openCursor(docConfig);
        }

        @Override
        String getDatabase() {
            return this.database;
        }

        @Override
        void close() {
            super.close();
            if (this.db != null) {
                try {
                    this.db.close();
                }
                catch (ThreadInterruptedException threadInterruptedException) {
                    // empty catch block
                }
                this.db = null;
            }
        }
    }

    abstract class Checkpoint
    implements LoadCheckpoint {
        private static final String STORE = "store";
        private static final String MACHINE = "machine";
        private static final String LOADTIME = "loadTime";
        final File baseDir;

        Checkpoint(File baseDir) {
            assert (baseDir != null);
            this.baseDir = baseDir;
        }

        @Override
        public void setLoaded(String name, long timestampMs) {
            this.writeToCheckpointFile(name, timestampMs);
        }

        boolean isValidCheckpointFile(File file) {
            try {
                String loadedTime;
                Properties props = new Properties();
                props.load(new FileReader(file));
                String store = props.getProperty(STORE);
                if (store != null && store.equalsIgnoreCase(Load.this.storeName) && (loadedTime = (String)props.get(LOADTIME)) != null) {
                    Load.this.verbose("Loaded checkpoint information from " + file + ": " + props.toString());
                    return true;
                }
            }
            catch (IOException e) {
                throw new IllegalStateException("Failed to load from checkpoint file " + file, e);
            }
            return false;
        }

        private void writeToCheckpointFile(String name, long timestampMs) {
            File file = new File(this.baseDir, name);
            if (!file.exists()) {
                try {
                    file.createNewFile();
                }
                catch (IOException ioe) {
                    throw new IllegalStateException("Failed to create checkpoint file: " + file, ioe);
                }
            }
            FileOutputStream fos = null;
            try {
                fos = new FileOutputStream(file);
                Properties props = new Properties();
                props.put(STORE, Load.this.storeName);
                props.put(MACHINE, Load.this.hostname);
                props.put(LOADTIME, utcDateFormat.format(new Date(timestampMs)));
                props.store(fos, null);
            }
            catch (IOException ioe) {
                throw new IllegalStateException("Failed to write checkpoint information to " + file, ioe);
            }
            finally {
                if (fos != null) {
                    try {
                        fos.close();
                    }
                    catch (IOException iOException) {}
                }
            }
        }
    }

    class SnapshotCheckpoint
    extends Checkpoint {
        private boolean isLoaded;

        SnapshotCheckpoint(File baseDir, String name) {
            super(baseDir);
            if (!baseDir.isDirectory()) {
                throw new IllegalArgumentException("A file but not a directory exists :" + baseDir);
            }
            File cpFile = new File(baseDir, name);
            this.isLoaded = cpFile.exists() && cpFile.isFile() ? this.isValidCheckpointFile(cpFile) : false;
        }

        @Override
        public boolean isLoaded(String name) {
            return this.isLoaded;
        }
    }

    class DatabaseCheckpoint
    extends Checkpoint {
        private final Set<String> loaded;

        DatabaseCheckpoint(File baseDir) {
            super(baseDir);
            this.loaded = new HashSet<String>();
            if (!baseDir.exists()) {
                if (!baseDir.mkdirs()) {
                    throw new IllegalStateException("Failed to create checkpoint directory: " + baseDir);
                }
            } else {
                if (!baseDir.isDirectory()) {
                    throw new IllegalArgumentException("A file but not a directory exists :" + baseDir);
                }
                this.loadCheckpointFiles();
            }
        }

        @Override
        public boolean isLoaded(String name) {
            return this.loaded.contains(name);
        }

        private void loadCheckpointFiles() {
            File[] files;
            for (File file : files = this.baseDir.listFiles()) {
                if (file.getName().equals(Load.LOCK_FILE)) continue;
                if (!file.isFile()) {
                    Load.this.message("Skip an invalid checkpoint file: " + file);
                    continue;
                }
                if (this.isValidCheckpointFile(file)) {
                    this.loaded.add(file.getName());
                    continue;
                }
                Load.this.message("The checkpoint file is not for store \"" + Load.this.storeName + "\" or not a valid checkpoint file: " + file.getAbsolutePath());
            }
        }
    }

    private class SnapshotEnvironment {
        private final File envDir;
        private Environment jeEnv;
        private final LoadCheckpoint checkpoint;
        private List<String> dbsToLoad;
        private static final String SUFFIX_STATS_FILE = ".stats";
        private final File envStatsFile;

        SnapshotEnvironment(File sourceDir, String checkpointDir, String statsDir) {
            this.envDir = sourceDir;
            if (checkpointDir != null) {
                if (Load.this.useEnvDOC) {
                    if (!this.envDir.exists() || !this.envDir.isDirectory()) {
                        String msg = this.envDir + " is not a valid snapshot directory.";
                        throw new IllegalArgumentException(msg);
                    }
                    this.checkpoint = new SnapshotCheckpoint(new File(checkpointDir), this.getName());
                } else {
                    File cpDir = new File(new File(checkpointDir), this.getName());
                    this.checkpoint = new DatabaseCheckpoint(cpDir);
                }
            } else {
                this.checkpoint = null;
            }
            this.envStatsFile = statsDir != null ? new File(statsDir, this.getName() + SUFFIX_STATS_FILE) : null;
        }

        String getName() {
            return "snapshot" + this.envDir.getAbsolutePath().replace("/", "_").replace(".", "");
        }

        File getEnvDir() {
            return this.envDir;
        }

        Environment getJEEnv() {
            if (this.jeEnv == null) {
                this.jeEnv = this.openJEEnvironment();
            }
            return this.jeEnv;
        }

        private Environment openJEEnvironment() {
            long startMs = System.currentTimeMillis();
            Environment env = Load.openEnvironment(this.envDir, Load.this.envConfigProps);
            Load.this.verbose(String.format("Environment open time:%,d ms", System.currentTimeMillis() - startMs));
            if (Load.this.envConfigProps != null && !Load.this.envConfigProps.isEmpty()) {
                String fmt = "Opened JE envrionement '%s' with properties: %s";
                Load.this.verbose(String.format(fmt, this.envDir.getAbsolutePath(), Load.this.envConfigProps));
            }
            return env;
        }

        List<String> getDatabasesToLoad() {
            if (this.dbsToLoad != null) {
                return this.dbsToLoad;
            }
            List dbs = this.getJEEnv().getDatabaseNames();
            this.dbsToLoad = new ArrayList<String>();
            if (Load.this.useEnvDOC) {
                if (this.isLoaded(this.getName())) {
                    this.dbsToLoad = Collections.emptyList();
                    Load.this.message("Skipping already loaded snapshot: " + this.getName());
                } else {
                    for (String db : dbs) {
                        if (!PartitionId.isPartitionName(db)) {
                            Load.this.verbose("Skipping non-partition database: " + db);
                            continue;
                        }
                        this.dbsToLoad.add(db);
                    }
                }
            } else {
                for (String db : dbs) {
                    if (!PartitionId.isPartitionName(db)) {
                        Load.this.verbose("Skipping non-partition database: " + db);
                        continue;
                    }
                    if (this.isLoaded(db)) {
                        Load.this.verbose("Skipping already loaded database: " + db);
                        continue;
                    }
                    this.dbsToLoad.add(db);
                }
                if (!this.dbsToLoad.isEmpty()) {
                    Collections.sort(this.dbsToLoad);
                }
            }
            return this.dbsToLoad;
        }

        void setLoaded(LoadStream stream) {
            if (this.checkpoint != null) {
                String name = Load.this.useEnvDOC ? this.getName() : stream.getDatabase();
                this.checkpoint.setLoaded(name, stream.getElapsedTime());
            }
            Load.this.displayLoadProgress(this.envDir.toString(), stream.getDatabase(), stream.getReadCount(), stream.getSkippedCount(), stream.getKeyExistCount(), stream.getElapsedTime());
        }

        boolean isLoaded(String name) {
            return this.checkpoint != null ? this.checkpoint.isLoaded(name) : false;
        }

        void close() {
            if (this.jeEnv != null) {
                if (!this.jeEnv.isClosed()) {
                    if (this.envStatsFile != null) {
                        this.dumpEnvStatsToFile();
                    }
                    this.jeEnv.close();
                    Load.this.verbose("Environment closed: " + this.getName());
                }
                this.jeEnv = null;
            }
        }

        private void dumpEnvStatsToFile() {
            if (!this.envStatsFile.exists()) {
                try {
                    this.envStatsFile.createNewFile();
                }
                catch (IOException ioe) {
                    throw new IllegalStateException("Failed to create environment stats file: " + this.envStatsFile, ioe);
                }
            }
            FileOutputStream fos = null;
            try {
                fos = new FileOutputStream(this.envStatsFile);
                String info = this.jeEnv.getStats(StatsConfig.DEFAULT).toString();
                fos.write(info.getBytes());
                fos.flush();
            }
            catch (IOException ioe) {
                throw new IllegalStateException("Failed to write stats information to " + this.envStatsFile, ioe);
            }
            catch (DatabaseException de) {
                throw new IllegalStateException("Failed to get stats information:" + de.getMessage(), de);
            }
            finally {
                if (fos != null) {
                    try {
                        fos.close();
                    }
                    catch (IOException iOException) {}
                }
            }
            Load.this.verbose("Dump environment statistic to " + this.envStatsFile.getAbsolutePath());
        }
    }

    static interface LoadCheckpoint {
        public void setLoaded(String var1, long var2);

        public boolean isLoaded(String var1);
    }

    public static class LoadParser
    extends CommandParser {
        private static final String SOURCE_FLAG = "-source";
        private static final String CHECKPOINT_FLAG = "-checkpoint";
        private static final String ADMIN_LOAD_FLAG = "-load-admin";
        private static final String FORCE_ADMIN_FLAG = "-force";
        private static final String BULKPUT_STREAM_PARALLELISM = "-bulkput-stream-parallelism";
        private static final String BULKPUT_HEAP_PERCENT = "-bulkput-heap-percent";
        private static final String BULKPUT_PER_SHARD_PARALLELISM = "-bulkput-per-shard-parallelism";
        private static final String BULKPUT_MAX_REQUEST_SIZE = "-bulkput-max-request-size";
        private static final String REQUEST_TIMEOUT_MS = "-request-timeout-ms";
        private static final String STREAM_CACHE_PERCENT = "-stream-cache-percent";
        private static final String ENV_CACHE_PERCENT = "-env-cache-percent";
        private static final String STATS_FLAG = "-collect-stats";
        private static final String JE_PROP_FLAG_PREFIX = "-je.";
        private static final String DB_DOC_FLAG = "-single-database-cursor";
        private File[] sourceDirs = null;
        private String checkpointDir = null;
        private boolean loadAdmin = false;
        private boolean forceLoadAdmin = false;
        private int bulkputStreamParallelism = 0;
        private int bulkputHeapPercent = 0;
        private int bulkputPerShardParallelism = 0;
        private int bulkputMaxRequestSize = 0;
        private long requestTimeoutMs = 0L;
        private String statsDir = null;
        private boolean useEnvDOC = true;
        private Map<String, String> jeProps = null;
        private int streamCachePercent = 0;
        private int envCachePercent = 0;

        LoadParser(String[] args) {
            super(args);
        }

        public void usage(String errorMsg) {
            if (errorMsg != null) {
                System.err.println(errorMsg);
            }
            System.err.println("Usage: java -jar KVHOME/lib/kvstore.jar load\n\t" + COMMAND_ARGS);
            System.exit(-1);
        }

        protected boolean checkArg(String arg) {
            if (arg.equals(SOURCE_FLAG)) {
                String source = this.nextArg(arg);
                String[] paths = source.split(",");
                this.sourceDirs = new File[paths.length];
                for (int i = 0; i < paths.length; ++i) {
                    this.sourceDirs[i] = new File(paths[i]);
                    if (this.sourceDirs[i].exists() && this.sourceDirs[i].isDirectory()) continue;
                    throw new IllegalArgumentException("Could not access backup source directory: " + this.sourceDirs[i]);
                }
                return true;
            }
            if (arg.equals(CHECKPOINT_FLAG)) {
                this.checkpointDir = this.nextArg(arg);
                File file = new File(this.checkpointDir);
                if (!file.exists() || !file.isDirectory()) {
                    throw new IllegalArgumentException("Could not access checkpoint files directory: " + this.checkpointDir);
                }
                return true;
            }
            if (arg.equals(ADMIN_LOAD_FLAG)) {
                this.loadAdmin = true;
                return true;
            }
            if (arg.equals(FORCE_ADMIN_FLAG)) {
                this.forceLoadAdmin = true;
                return true;
            }
            if (arg.equals(BULKPUT_STREAM_PARALLELISM)) {
                this.bulkputStreamParallelism = this.nextIntArg(arg);
                if (this.bulkputStreamParallelism < 1) {
                    throw new IllegalArgumentException("-bulkput-stream-parallelism requires a positive integer: " + this.bulkputStreamParallelism);
                }
                return true;
            }
            if (arg.equals(BULKPUT_HEAP_PERCENT)) {
                this.bulkputHeapPercent = this.checkPercentArg(arg);
                return true;
            }
            if (arg.equals(BULKPUT_PER_SHARD_PARALLELISM)) {
                this.bulkputPerShardParallelism = this.nextIntArg(arg);
                if (this.bulkputPerShardParallelism < 1) {
                    throw new IllegalArgumentException("-bulkput-per-shard-parallelism requires a positive integer: " + this.bulkputPerShardParallelism);
                }
                return true;
            }
            if (arg.equals(BULKPUT_MAX_REQUEST_SIZE)) {
                this.bulkputMaxRequestSize = this.nextIntArg(arg);
                if (this.bulkputMaxRequestSize < 1) {
                    throw new IllegalArgumentException("-bulkput-max-request-size requires a positive integer: " + this.bulkputMaxRequestSize);
                }
                return true;
            }
            if (arg.equals(STATS_FLAG)) {
                this.statsDir = this.nextArg(arg);
                File file = new File(this.statsDir);
                if (!file.exists() || !file.isDirectory()) {
                    throw new IllegalArgumentException("Could not access stats files directory: " + this.statsDir);
                }
                return true;
            }
            if (arg.equals(DB_DOC_FLAG)) {
                this.useEnvDOC = false;
                return true;
            }
            if (arg.startsWith(JE_PROP_FLAG_PREFIX) && arg.length() > JE_PROP_FLAG_PREFIX.length()) {
                if (this.jeProps == null) {
                    this.jeProps = new HashMap<String, String>();
                }
                this.jeProps.put(arg.substring(1), this.nextArg(arg));
                return true;
            }
            if (arg.equals(REQUEST_TIMEOUT_MS)) {
                this.requestTimeoutMs = this.nextLongArg(arg);
                if (this.requestTimeoutMs < 1L) {
                    throw new IllegalArgumentException("-request-timeout-ms requires a positive integer: " + this.requestTimeoutMs);
                }
                return true;
            }
            if (arg.equals(STREAM_CACHE_PERCENT)) {
                this.streamCachePercent = this.checkPercentArg(arg);
                return true;
            }
            if (arg.equals(ENV_CACHE_PERCENT)) {
                this.envCachePercent = this.checkPercentArg(arg);
                return true;
            }
            return false;
        }

        private int checkPercentArg(String arg) {
            int percent = this.nextIntArg(arg);
            if (percent < 1 || percent > 99) {
                throw new IllegalArgumentException(arg + " is a percentage and must be in the range 1 ~ 99: " + percent);
            }
            return percent;
        }

        protected void verifyArgs() {
            if (this.getHostname() == null) {
                this.missingArg("-host");
            }
            if (this.getRegistryPort() == 0) {
                this.missingArg("-port");
            }
            if (this.getStoreName() == null && !this.loadAdmin) {
                this.missingArg("-store");
            }
            if (this.getSourceDirs() == null) {
                this.missingArg(SOURCE_FLAG);
            } else if (this.loadAdmin && this.getSourceDirs().length > 1) {
                throw new IllegalArgumentException("There must be exactly one source dir if loading Admin metadata");
            }
        }

        public String getCheckpointDir() {
            return this.checkpointDir;
        }

        public File[] getSourceDirs() {
            return this.sourceDirs;
        }

        public boolean getLoadAdmin() {
            return this.loadAdmin;
        }

        public boolean getForceLoadAdmin() {
            return this.forceLoadAdmin;
        }

        public int getBulkputStreamParallelism() {
            return this.bulkputStreamParallelism;
        }

        public int getBulkputHeapPercent() {
            return this.bulkputHeapPercent;
        }

        public int getBulkputPerShardParallelism() {
            return this.bulkputPerShardParallelism;
        }

        public int getBulkputMaxRequestSize() {
            return this.bulkputMaxRequestSize;
        }

        public long getRequestTimeoutMs() {
            return this.requestTimeoutMs;
        }

        public String getStatsDir() {
            return this.statsDir;
        }

        public boolean useEnvDOC() {
            return this.useEnvDOC;
        }

        public Map<String, String> getJEProps() {
            return this.jeProps;
        }

        public int getStreamCachePercent() {
            return this.streamCachePercent;
        }

        public int getEnvCachePercent() {
            return this.envCachePercent;
        }
    }

    private static class LoadAdmin {
        private final Environment env;
        private final boolean verboseOutput;
        private final PrintStream output;
        private final String securityFile;
        private final String user;
        private final String targetHost;
        private final int targetPort;
        private final boolean forceLoad;

        public LoadAdmin(File envDir, String targetHost, int targetPort, String user, String securityFile, boolean verboseOutput, boolean forceLoad, PrintStream output) {
            this.env = Load.openEnvironment(envDir, null);
            this.output = output;
            this.verboseOutput = verboseOutput;
            this.securityFile = securityFile;
            this.user = user;
            this.targetHost = targetHost;
            this.targetPort = targetPort;
            this.forceLoad = forceLoad;
            this.verbose("Opened environment for admin load: " + envDir);
        }

        public void loadMetadata() {
            Logger logger = Logger.getLogger(LoadAdmin.class.getName());
            TableStore ts = TableStore.getReadOnlyInstance((Logger)logger, (Environment)this.env);
            TableMetadata tmd = ts.getTableMetadata(null);
            ts.close();
            if (tmd != null) {
                List<String> tablesToLoad;
                this.verbose("Found table metadata");
                if (this.verboseOutput && !(tablesToLoad = tmd.listTables(null, true)).isEmpty()) {
                    this.message("Writing tables:");
                    for (String s : tablesToLoad) {
                        this.message("\t" + s);
                    }
                }
            } else {
                this.message("No tables to write");
            }
            SecurityStore ss = SecurityStore.getReadOnlyInstance((Logger)logger, (Environment)this.env);
            SecurityMetadata smd = ss.getSecurityMetadata(null);
            ss.close();
            if (smd != null) {
                this.verbose("Found security metadata");
            }
            if (tmd != null || smd != null) {
                this.writeMetadata(tmd, smd);
            }
            this.env.close();
        }

        private void writeMetadata(TableMetadata tmd, SecurityMetadata smd) {
            ArrayList<String> argList = new ArrayList<String>();
            argList.add("-host");
            argList.add(this.targetHost);
            argList.add("-port");
            argList.add(Integer.toString(this.targetPort));
            if (this.securityFile != null) {
                argList.add("-admin-security");
                argList.add(this.securityFile);
            }
            if (this.user != null) {
                argList.add("-admin-username");
                argList.add(this.user);
            }
            CommandShell shell = new CommandShell(null, this.output);
            shell.parseArgs(argList.toArray(new String[argList.size()]));
            shell.init();
            try {
                TableMetadata existingTableMD;
                CommandServiceAPI cs = shell.getAdmin();
                boolean tableMDExist = false;
                boolean secMDExist = false;
                if (tmd != null && (existingTableMD = (TableMetadata)cs.getMetadata(TableMetadata.class, Metadata.MetadataType.TABLE)) != null) {
                    for (Table table : existingTableMD.getTables().values()) {
                        if (((TableImpl)table).isSystemTable()) continue;
                        tableMDExist = true;
                    }
                }
                boolean bl = secMDExist = cs.getUsersDescription() != null;
                if ((tableMDExist || secMDExist) && !this.forceLoad) {
                    this.message("Metadata exists, use -force flag to overwrite");
                    return;
                }
                if (tmd != null) {
                    cs.putMetadata((Metadata)tmd);
                    this.verbose("Wrote table metadata");
                }
                if (smd != null) {
                    cs.putMetadata((Metadata)smd);
                    this.verbose("Wrote security metadata");
                }
            }
            catch (RemoteException re) {
                this.message("Failed to acquire admin interface or write metadata: " + re.getMessage());
            }
            catch (ShellException se) {
                this.message("Failed to acquire admin interface or write metadata: " + se.getMessage());
            }
        }

        private void message(String msg) {
            this.output.println(msg);
        }

        private void verbose(String msg) {
            if (this.verboseOutput) {
                this.message(msg);
            }
        }
    }
}

