package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Device}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableDevice.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableDevice implements Device {
  private final @Nullable String pathOnHost;
  private final @Nullable String pathInContainer;
  private final @Nullable String cgroupPermissions;

  private ImmutableDevice(
      @Nullable String pathOnHost,
      @Nullable String pathInContainer,
      @Nullable String cgroupPermissions) {
    this.pathOnHost = pathOnHost;
    this.pathInContainer = pathInContainer;
    this.cgroupPermissions = cgroupPermissions;
  }

  /**
   * @return The value of the {@code pathOnHost} attribute
   */
  @JsonProperty("PathOnHost")
  @Override
  public @Nullable String pathOnHost() {
    return pathOnHost;
  }

  /**
   * @return The value of the {@code pathInContainer} attribute
   */
  @JsonProperty("PathInContainer")
  @Override
  public @Nullable String pathInContainer() {
    return pathInContainer;
  }

  /**
   * @return The value of the {@code cgroupPermissions} attribute
   */
  @JsonProperty("CgroupPermissions")
  @Override
  public @Nullable String cgroupPermissions() {
    return cgroupPermissions;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Device#pathOnHost() pathOnHost} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for pathOnHost (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableDevice withPathOnHost(@Nullable String value) {
    if (Objects.equals(this.pathOnHost, value)) return this;
    return new ImmutableDevice(value, this.pathInContainer, this.cgroupPermissions);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Device#pathInContainer() pathInContainer} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for pathInContainer (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableDevice withPathInContainer(@Nullable String value) {
    if (Objects.equals(this.pathInContainer, value)) return this;
    return new ImmutableDevice(this.pathOnHost, value, this.cgroupPermissions);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Device#cgroupPermissions() cgroupPermissions} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for cgroupPermissions (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableDevice withCgroupPermissions(@Nullable String value) {
    if (Objects.equals(this.cgroupPermissions, value)) return this;
    return new ImmutableDevice(this.pathOnHost, this.pathInContainer, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableDevice} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableDevice
        && equalsByValue((ImmutableDevice) another);
  }

  private boolean equalsByValue(ImmutableDevice another) {
    return Objects.equals(pathOnHost, another.pathOnHost)
        && Objects.equals(pathInContainer, another.pathInContainer)
        && Objects.equals(cgroupPermissions, another.cgroupPermissions);
  }

  /**
   * Computes a hash code from attributes: {@code pathOnHost}, {@code pathInContainer}, {@code cgroupPermissions}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(pathOnHost);
    h += (h << 5) + Objects.hashCode(pathInContainer);
    h += (h << 5) + Objects.hashCode(cgroupPermissions);
    return h;
  }

  /**
   * Prints the immutable value {@code Device} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Device{"
        + "pathOnHost=" + pathOnHost
        + ", pathInContainer=" + pathInContainer
        + ", cgroupPermissions=" + cgroupPermissions
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Device} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Device instance
   */
  public static ImmutableDevice copyOf(Device instance) {
    if (instance instanceof ImmutableDevice) {
      return (ImmutableDevice) instance;
    }
    return ImmutableDevice.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableDevice ImmutableDevice}.
   * <pre>
   * ImmutableDevice.builder()
   *    .pathOnHost(String | null) // nullable {@link Device#pathOnHost() pathOnHost}
   *    .pathInContainer(String | null) // nullable {@link Device#pathInContainer() pathInContainer}
   *    .cgroupPermissions(String | null) // nullable {@link Device#cgroupPermissions() cgroupPermissions}
   *    .build();
   * </pre>
   * @return A new ImmutableDevice builder
   */
  public static ImmutableDevice.Builder builder() {
    return new ImmutableDevice.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableDevice ImmutableDevice}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements Device.Builder {
    private String pathOnHost;
    private String pathInContainer;
    private String cgroupPermissions;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Device} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Device instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable String pathOnHostValue = instance.pathOnHost();
      if (pathOnHostValue != null) {
        pathOnHost(pathOnHostValue);
      }
      @Nullable String pathInContainerValue = instance.pathInContainer();
      if (pathInContainerValue != null) {
        pathInContainer(pathInContainerValue);
      }
      @Nullable String cgroupPermissionsValue = instance.cgroupPermissions();
      if (cgroupPermissionsValue != null) {
        cgroupPermissions(cgroupPermissionsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Device#pathOnHost() pathOnHost} attribute.
     * @param pathOnHost The value for pathOnHost (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("PathOnHost")
    public final Builder pathOnHost(@Nullable String pathOnHost) {
      this.pathOnHost = pathOnHost;
      return this;
    }

    /**
     * Initializes the value for the {@link Device#pathInContainer() pathInContainer} attribute.
     * @param pathInContainer The value for pathInContainer (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("PathInContainer")
    public final Builder pathInContainer(@Nullable String pathInContainer) {
      this.pathInContainer = pathInContainer;
      return this;
    }

    /**
     * Initializes the value for the {@link Device#cgroupPermissions() cgroupPermissions} attribute.
     * @param cgroupPermissions The value for cgroupPermissions (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("CgroupPermissions")
    public final Builder cgroupPermissions(@Nullable String cgroupPermissions) {
      this.cgroupPermissions = cgroupPermissions;
      return this;
    }

    /**
     * Builds a new {@link ImmutableDevice ImmutableDevice}.
     * @return An immutable instance of Device
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableDevice build() {
      return new ImmutableDevice(pathOnHost, pathInContainer, cgroupPermissions);
    }
  }
}
