/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.kura.core.identity;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Dictionary;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.eclipse.kura.KuraErrorCode;
import org.eclipse.kura.KuraException;
import org.eclipse.kura.audit.AuditContext;
import org.eclipse.kura.configuration.ComponentConfiguration;
import org.eclipse.kura.core.identity.PasswordHashImpl;
import org.eclipse.kura.core.identity.ValidationUtil;
import org.eclipse.kura.crypto.CryptoService;
import org.eclipse.kura.identity.AdditionalConfigurations;
import org.eclipse.kura.identity.AssignedPermissions;
import org.eclipse.kura.identity.IdentityConfiguration;
import org.eclipse.kura.identity.IdentityConfigurationComponent;
import org.eclipse.kura.identity.IdentityService;
import org.eclipse.kura.identity.PasswordConfiguration;
import org.eclipse.kura.identity.PasswordHash;
import org.eclipse.kura.identity.PasswordStrengthVerificationService;
import org.eclipse.kura.identity.Permission;
import org.eclipse.kura.identity.configuration.extension.IdentityConfigurationExtension;
import org.eclipse.kura.util.useradmin.UserAdminHelper;
import org.osgi.service.useradmin.Role;
import org.osgi.service.useradmin.User;
import org.osgi.service.useradmin.UserAdmin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IdentityServiceImpl
implements IdentityService {
    private static final String IDENTITY_SERVICE_FAILURE_FORMAT_STRING = "{} IdentityService - Failure - {}";
    private static final String IDENTITY_SERVICE_SUCCESS_FORMAT_STRING = "{} IdentityService - Success - {}";
    private static final Logger auditLogger = LoggerFactory.getLogger((String)"AuditLogger");
    private static final Logger logger = LoggerFactory.getLogger(IdentityServiceImpl.class);
    private static final String PASSWORD_PROPERTY = "kura.password";
    private static final String KURA_NEED_PASSWORD_CHANGE = "kura.need.password.change";
    private UserAdmin userAdmin;
    private CryptoService cryptoService;
    private UserAdminHelper userAdminHelper;
    private PasswordStrengthVerificationService passwordStrengthVerificationService;
    private final Map<String, IdentityConfigurationExtension> extensions = new ConcurrentHashMap<String, IdentityConfigurationExtension>();

    public void setCryptoService(CryptoService cryptoService) {
        this.cryptoService = cryptoService;
    }

    public void setUserAdmin(UserAdmin userAdmin) {
        this.userAdmin = userAdmin;
    }

    public void setPasswordStrengthVerificationService(PasswordStrengthVerificationService passwordStrengthVerificationService) {
        this.passwordStrengthVerificationService = passwordStrengthVerificationService;
    }

    public synchronized void setIdentityConfigurationExtension(IdentityConfigurationExtension identityConfigurationExtension, Map<String, Object> properties) {
        Object kuraServicePid = properties.get("kura.service.pid");
        if (!(kuraServicePid instanceof String)) {
            logger.warn("found {} registered without setting the {} service property, service will not be tracked", (Object)IdentityConfigurationExtension.class.getSimpleName(), (Object)"kura.service.pid");
            return;
        }
        this.extensions.put((String)kuraServicePid, identityConfigurationExtension);
    }

    public synchronized void unsetIdentityConfigurationExtension(IdentityConfigurationExtension identityConfigurationExtension, Map<String, Object> properties) {
        Object kuraServicePid = properties.get("kura.service.pid");
        if (kuraServicePid instanceof String) {
            this.extensions.remove(kuraServicePid);
        }
    }

    public void activate() {
        this.userAdminHelper = new UserAdminHelper(this.userAdmin, this.cryptoService);
    }

    public synchronized boolean createIdentity(String name) throws KuraException {
        if (this.userAdminHelper.getUser(name).isPresent()) {
            return false;
        }
        IdentityServiceImpl.audit(() -> {
            ValidationUtil.validateNewIdentityName(name);
            this.userAdminHelper.createUser(name);
        }, "Create identity " + name);
        return true;
    }

    public synchronized boolean deleteIdentity(String name) throws KuraException {
        if (!this.userAdminHelper.getUser(name).isPresent()) {
            return false;
        }
        IdentityServiceImpl.audit(() -> this.userAdminHelper.deleteUser(name), "Delete identity " + name);
        return true;
    }

    public synchronized List<IdentityConfiguration> getIdentitiesConfiguration(Set<Class<? extends IdentityConfigurationComponent>> componentsToReturn) throws KuraException {
        ArrayList<IdentityConfiguration> result = new ArrayList<IdentityConfiguration>();
        this.userAdminHelper.foreachUser((name, user) -> {
            boolean bl = result.add(this.buildIdentity(name, user, componentsToReturn));
        });
        return result;
    }

    public synchronized Optional<IdentityConfiguration> getIdentityConfiguration(String name, Set<Class<? extends IdentityConfigurationComponent>> componentsToReturn) throws KuraException {
        return this.userAdminHelper.getUser(name).map(u -> this.buildIdentity(name, (User)u, componentsToReturn));
    }

    public IdentityConfiguration getIdentityDefaultConfiguration(String identityName, Set<Class<? extends IdentityConfigurationComponent>> componentsToReturn) throws KuraException {
        ArrayList<Object> components = new ArrayList<Object>();
        if (componentsToReturn.contains(PasswordConfiguration.class)) {
            components.add(new PasswordConfiguration(false, false, Optional.empty(), Optional.empty()));
        }
        if (componentsToReturn.contains(AssignedPermissions.class)) {
            components.add(new AssignedPermissions(Collections.emptySet()));
        }
        if (componentsToReturn.contains(AdditionalConfigurations.class)) {
            components.add(this.getAdditionalConfigurationsDefaults(identityName));
        }
        return new IdentityConfiguration(identityName, components);
    }

    public void validateIdentityConfiguration(IdentityConfiguration identityConfiguration) throws KuraException {
        IdentityServiceImpl.audit(() -> {
            Optional assignedPermissions;
            Optional additionalConfigurations;
            Optional passwordCofiguration = identityConfiguration.getComponent(PasswordConfiguration.class);
            if (passwordCofiguration.isPresent()) {
                this.validatePasswordConfiguration(identityConfiguration, (PasswordConfiguration)passwordCofiguration.get());
            }
            if ((additionalConfigurations = identityConfiguration.getComponent(AdditionalConfigurations.class)).isPresent()) {
                this.validateAdditionalConfigurations(identityConfiguration, (AdditionalConfigurations)additionalConfigurations.get());
            }
            if ((assignedPermissions = identityConfiguration.getComponent(AssignedPermissions.class)).isPresent()) {
                this.validateAssignedPermissions((AssignedPermissions)assignedPermissions.get());
            }
        }, "Validate configuration for identity" + identityConfiguration.getName());
    }

    public synchronized void updateIdentityConfiguration(IdentityConfiguration identityConfiguration) throws KuraException {
        IdentityServiceImpl.audit(() -> {
            Optional user = this.userAdminHelper.getUser(identityConfiguration.getName());
            if (!user.isPresent()) {
                throw new KuraException(KuraErrorCode.INVALID_PARAMETER, new Object[]{"Identity does not exist"});
            }
            this.validateIdentityConfiguration(identityConfiguration);
            this.updateIdentityConfigurationInternal((User)user.get(), identityConfiguration);
        }, "Update configuration for identity " + identityConfiguration.getName());
    }

    public synchronized boolean createPermission(Permission permission) throws KuraException {
        if (this.userAdminHelper.getPermission(permission.getName()).isPresent()) {
            return false;
        }
        ValidationUtil.validateNewPermissionName(permission.getName());
        this.userAdminHelper.getOrCreatePermission(permission.getName());
        return true;
    }

    public synchronized boolean deletePermission(Permission permission) throws KuraException {
        if (!this.userAdminHelper.getPermission(permission.getName()).isPresent()) {
            return false;
        }
        this.userAdminHelper.deletePremission(permission.getName());
        return true;
    }

    public synchronized Set<Permission> getPermissions() {
        return new UserAdminHelper(this.userAdmin, this.cryptoService).getDefinedPermissions().stream().map(Permission::new).collect(Collectors.toSet());
    }

    public PasswordHash computePasswordHash(char[] password) throws KuraException {
        try {
            String result = this.cryptoService.sha256Hash(new String(password));
            int i = 0;
            while (i < password.length) {
                password[i] = 32;
                ++i;
            }
            return new PasswordHashImpl(result);
        }
        catch (Exception exception) {
            throw new KuraException(KuraErrorCode.SERVICE_UNAVAILABLE, new Object[]{"failed to compute password hash"});
        }
    }

    public void checkPassword(String identityName, char[] password) throws KuraException {
        PasswordConfiguration passwordConfiguration = (PasswordConfiguration)this.getIdentityConfiguration(identityName, Collections.singleton(PasswordConfiguration.class)).flatMap(i -> i.getComponent(PasswordConfiguration.class)).orElseThrow(() -> new KuraException(KuraErrorCode.SECURITY_EXCEPTION));
        if (!passwordConfiguration.isPasswordAuthEnabled() || !Objects.equals(passwordConfiguration.getPasswordHash(), Optional.of(this.computePasswordHash(password)))) {
            throw new KuraException(KuraErrorCode.SECURITY_EXCEPTION, new Object[]{"Password authentication is not enabled or password does not match"});
        }
    }

    public void checkPermission(String identityName, Permission permission) throws KuraException {
        try {
            this.userAdminHelper.requirePermissions(identityName, new String[]{permission.getName()});
        }
        catch (UserAdminHelper.AuthenticationException authenticationException) {
            throw new KuraException(KuraErrorCode.SECURITY_EXCEPTION, new Object[]{"The specified permission is not assigned to the given identity"});
        }
    }

    private void updateIdentityConfigurationInternal(User user, IdentityConfiguration identity) throws KuraException {
        Optional additionalConfigurations;
        Optional permissions;
        String identityName = identity.getName();
        Optional passwordData = identity.getComponent(PasswordConfiguration.class);
        if (passwordData.isPresent()) {
            this.updatePassword(identityName, (PasswordConfiguration)passwordData.get(), user);
        }
        if ((permissions = identity.getComponent(AssignedPermissions.class)).isPresent()) {
            this.updateAssignedPermissions(identityName, (AssignedPermissions)permissions.get(), user);
        }
        if ((additionalConfigurations = identity.getComponent(AdditionalConfigurations.class)).isPresent()) {
            this.updateAdditionalConfigurations(identity.getName(), (AdditionalConfigurations)additionalConfigurations.get());
        }
    }

    private IdentityConfiguration buildIdentity(String name, User user, Set<Class<? extends IdentityConfigurationComponent>> componentsToReturn) {
        ArrayList<Object> components = new ArrayList<Object>();
        if (componentsToReturn.contains(PasswordConfiguration.class)) {
            components.add(this.getPasswordData(user));
        }
        if (componentsToReturn.contains(AssignedPermissions.class)) {
            Set permissions = this.userAdminHelper.getIdentityPermissions(name).stream().map(Permission::new).collect(Collectors.toSet());
            components.add(new AssignedPermissions(permissions));
        }
        if (componentsToReturn.contains(AdditionalConfigurations.class)) {
            components.add(this.getAdditionalConfigurations(name));
        }
        return new IdentityConfiguration(name, components);
    }

    private AdditionalConfigurations getAdditionalConfigurations(String name) {
        ArrayList additionalConfigurations = new ArrayList();
        for (IdentityConfigurationExtension extension : this.extensions.values()) {
            try {
                extension.getConfiguration(name).ifPresent(additionalConfigurations::add);
            }
            catch (Exception e) {
                logger.warn("Failed to get identity additional configuration from extension", (Throwable)e);
            }
        }
        return new AdditionalConfigurations(additionalConfigurations);
    }

    private AdditionalConfigurations getAdditionalConfigurationsDefaults(String name) {
        ArrayList additionalConfigurations = new ArrayList();
        for (IdentityConfigurationExtension extension : this.extensions.values()) {
            try {
                extension.getDefaultConfiguration(name).ifPresent(additionalConfigurations::add);
            }
            catch (Exception e) {
                logger.warn("Failed to get identity additional configuration defaults from extension", (Throwable)e);
            }
        }
        return new AdditionalConfigurations(additionalConfigurations);
    }

    private PasswordConfiguration getPasswordData(User user) {
        Optional<String> passwordHash = Optional.ofNullable(user.getCredentials().get(PASSWORD_PROPERTY)).filter(String.class::isInstance).map(String.class::cast);
        boolean isPasswordChangeNeeded = Objects.equals("true", user.getProperties().get(KURA_NEED_PASSWORD_CHANGE));
        return new PasswordConfiguration(isPasswordChangeNeeded, passwordHash.isPresent(), Optional.empty(), passwordHash.map(PasswordHashImpl::new));
    }

    private void setProperty(Dictionary<String, Object> properties, String key, Object value) {
        if (!Objects.equals(properties.get(key), value)) {
            properties.put(key, value);
        }
    }

    private void removeProperty(Dictionary<String, Object> properties, String key) {
        if (properties.get(key) != null) {
            properties.remove(key);
        }
    }

    private void updateAssignedPermissions(String identityName, AssignedPermissions assignedPermissions, User user) {
        this.userAdminHelper.foreachPermission((name, group) -> {
            Permission permission = new Permission(name);
            List members = Optional.ofNullable(group.getMembers()).map(roleArray -> Arrays.asList(roleArray)).orElse(Collections.emptyList());
            if (assignedPermissions.getPermissions().contains(permission) && !members.contains(user)) {
                IdentityServiceImpl.audit(() -> group.addMember((Role)user), "Add permission " + permission.getName() + " to identity " + identityName);
            } else if (!assignedPermissions.getPermissions().contains(permission) && members.contains(user)) {
                IdentityServiceImpl.audit(() -> group.removeMember((Role)user), "Remove permission " + permission.getName() + " from identity " + identityName);
            }
        });
    }

    private void updatePassword(String identityName, PasswordConfiguration passwordData, User user) throws KuraException {
        Dictionary properties = user.getProperties();
        Object currentIsPasswordChangeNeeded = properties.get(KURA_NEED_PASSWORD_CHANGE);
        if (passwordData.isPasswordChangeNeeded()) {
            if (!"true".equals(currentIsPasswordChangeNeeded)) {
                IdentityServiceImpl.audit(() -> this.setProperty(properties, KURA_NEED_PASSWORD_CHANGE, "true"), "Enable password change at next login for identity " + identityName);
            }
        } else if (currentIsPasswordChangeNeeded != null) {
            IdentityServiceImpl.audit(() -> this.removeProperty(properties, KURA_NEED_PASSWORD_CHANGE), "Disable password change at next login for identity " + identityName);
        }
        Dictionary credentials = user.getCredentials();
        Optional newPassword = passwordData.getNewPassword();
        Object currentPasswordHash = credentials.get(PASSWORD_PROPERTY);
        if (passwordData.isPasswordAuthEnabled() && newPassword.isPresent()) {
            IdentityServiceImpl.audit(() -> this.setProperty(credentials, PASSWORD_PROPERTY, this.computePasswordHash((char[])newPassword.get()).toString()), "Update Kura password for identity " + identityName);
        } else if (!passwordData.isPasswordAuthEnabled() && currentPasswordHash != null) {
            IdentityServiceImpl.audit(() -> this.removeProperty(credentials, PASSWORD_PROPERTY), "Disable Kura password for identity " + identityName);
        }
    }

    private void validatePasswordConfiguration(IdentityConfiguration identityConfiguration, PasswordConfiguration passwordCofiguration) throws KuraException {
        if (!passwordCofiguration.isPasswordAuthEnabled()) {
            return;
        }
        Optional newPassword = passwordCofiguration.getNewPassword();
        if (newPassword.isPresent()) {
            ValidationUtil.validateNewPassword((char[])newPassword.get(), this.passwordStrengthVerificationService);
        } else if (!this.userAdminHelper.getUser(identityConfiguration.getName()).filter(u -> u.getCredentials().get(PASSWORD_PROPERTY) != null).isPresent()) {
            throw new KuraException(KuraErrorCode.INVALID_PARAMETER, new Object[]{"Password authentication is enabled but no password has been provided or is currently assigned"});
        }
    }

    private void validateAssignedPermissions(AssignedPermissions assignedPermissions) throws KuraException {
        for (Permission permission : assignedPermissions.getPermissions()) {
            if (this.userAdminHelper.getPermission(permission.getName()).isPresent()) continue;
            throw new KuraException(KuraErrorCode.INVALID_PARAMETER, new Object[]{"Permission does not exist"});
        }
    }

    private void validateAdditionalConfigurations(IdentityConfiguration identityConfiguration, AdditionalConfigurations additionalConfigurations) throws KuraException {
        for (ComponentConfiguration config : additionalConfigurations.getConfigurations()) {
            Optional<IdentityConfigurationExtension> extension = Optional.ofNullable(this.extensions.get(config.getPid()));
            if (!extension.isPresent()) {
                throw new KuraException(KuraErrorCode.INVALID_PARAMETER, new Object[]{"Configuration extension pid is not registered"});
            }
            extension.get().validateConfiguration(identityConfiguration.getName(), config);
        }
    }

    private void updateAdditionalConfigurations(String identityName, AdditionalConfigurations additionalConfigurations) throws KuraException {
        FailureHandler failureHandler = new FailureHandler();
        for (ComponentConfiguration config : additionalConfigurations.getConfigurations()) {
            String pid = config.getPid();
            Optional<IdentityConfigurationExtension> extension = Optional.ofNullable(this.extensions.get(pid));
            if (!extension.isPresent()) {
                failureHandler.addError("Configuration extension pid is not registered");
                continue;
            }
            try {
                IdentityServiceImpl.audit(() -> ((IdentityConfigurationExtension)extension.get()).updateConfiguration(identityName, config), "Update configuration for extension " + pid + " for identity " + identityName);
            }
            catch (KuraException e) {
                failureHandler.addError(e.getMessage());
            }
        }
        failureHandler.throwIfFailuresOccurred(KuraErrorCode.CONFIGURATION_ERROR);
    }

    private static <T, E extends Throwable> T audit(FallibleSupplier<T, E> task, String message) throws E {
        try {
            T result = task.get();
            auditLogger.info(IDENTITY_SERVICE_SUCCESS_FORMAT_STRING, (Object)AuditContext.currentOrInternal(), (Object)message);
            return result;
        }
        catch (Exception e) {
            auditLogger.warn(IDENTITY_SERVICE_FAILURE_FORMAT_STRING, (Object)AuditContext.currentOrInternal(), (Object)message);
            throw e;
        }
    }

    private static <E extends Throwable> void audit(FallibleTask<E> task, String message) throws E {
        try {
            task.run();
            auditLogger.info(IDENTITY_SERVICE_SUCCESS_FORMAT_STRING, (Object)AuditContext.currentOrInternal(), (Object)message);
        }
        catch (Exception e) {
            auditLogger.warn(IDENTITY_SERVICE_FAILURE_FORMAT_STRING, (Object)AuditContext.currentOrInternal(), (Object)message);
            throw e;
        }
    }

    private static class FailureHandler {
        final Set<String> errors = new HashSet<String>();

        private FailureHandler() {
        }

        public void addError(String message) {
            this.errors.add(message);
            logger.error(message);
        }

        public void throwIfFailuresOccurred(KuraErrorCode errorCode) throws KuraException {
            if (!this.errors.isEmpty()) {
                throw new KuraException(errorCode, new Object[]{this.errors.stream().collect(Collectors.joining("; "))});
            }
        }
    }

    private static interface FallibleSupplier<T, E extends Throwable> {
        public T get() throws E;
    }

    private static interface FallibleTask<E extends Throwable> {
        public void run() throws E;
    }
}

