/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.broker;

import io.moquette.broker.BrokerConfiguration;
import io.moquette.broker.DebugUtils;
import io.moquette.broker.InflightResender;
import io.moquette.broker.NettyUtils;
import io.moquette.broker.PostOffice;
import io.moquette.broker.Session;
import io.moquette.broker.SessionCorruptedException;
import io.moquette.broker.SessionRegistry;
import io.moquette.broker.security.IAuthenticator;
import io.moquette.broker.subscriptions.Topic;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufHolder;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelPipeline;
import io.netty.handler.codec.mqtt.MqttConnAckMessage;
import io.netty.handler.codec.mqtt.MqttConnectMessage;
import io.netty.handler.codec.mqtt.MqttConnectPayload;
import io.netty.handler.codec.mqtt.MqttConnectReturnCode;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageBuilders;
import io.netty.handler.codec.mqtt.MqttMessageIdVariableHeader;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttPubAckMessage;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttPublishVariableHeader;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.handler.codec.mqtt.MqttSubAckMessage;
import io.netty.handler.codec.mqtt.MqttSubscribeMessage;
import io.netty.handler.codec.mqtt.MqttUnsubAckMessage;
import io.netty.handler.codec.mqtt.MqttUnsubscribeMessage;
import io.netty.handler.codec.mqtt.MqttVersion;
import io.netty.handler.timeout.IdleStateHandler;
import java.net.InetSocketAddress;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class MQTTConnection {
    private static final Logger LOG = LoggerFactory.getLogger(MQTTConnection.class);
    final Channel channel;
    private final BrokerConfiguration brokerConfig;
    private final IAuthenticator authenticator;
    private final SessionRegistry sessionRegistry;
    private final PostOffice postOffice;
    private volatile boolean connected;
    private final AtomicInteger lastPacketId = new AtomicInteger(0);
    private Session bindedSession;

    MQTTConnection(Channel channel, BrokerConfiguration brokerConfig, IAuthenticator authenticator, SessionRegistry sessionRegistry, PostOffice postOffice) {
        this.channel = channel;
        this.brokerConfig = brokerConfig;
        this.authenticator = authenticator;
        this.sessionRegistry = sessionRegistry;
        this.postOffice = postOffice;
        this.connected = false;
    }

    void handleMessage(MqttMessage msg) {
        MqttMessageType messageType = msg.fixedHeader().messageType();
        LOG.debug("Received MQTT message, type: {}", (Object)messageType);
        switch (messageType) {
            case CONNECT: {
                this.processConnect((MqttConnectMessage)msg);
                break;
            }
            case SUBSCRIBE: {
                this.processSubscribe((MqttSubscribeMessage)msg);
                break;
            }
            case UNSUBSCRIBE: {
                this.processUnsubscribe((MqttUnsubscribeMessage)msg);
                break;
            }
            case PUBLISH: {
                this.processPublish((MqttPublishMessage)msg);
                break;
            }
            case PUBREC: {
                this.processPubRec(msg);
                break;
            }
            case PUBCOMP: {
                this.processPubComp(msg);
                break;
            }
            case PUBREL: {
                this.processPubRel(msg);
                break;
            }
            case DISCONNECT: {
                this.processDisconnect(msg);
                break;
            }
            case PUBACK: {
                this.processPubAck(msg);
                break;
            }
            case PINGREQ: {
                MqttFixedHeader pingHeader = new MqttFixedHeader(MqttMessageType.PINGRESP, false, MqttQoS.AT_MOST_ONCE, false, 0);
                MqttMessage pingResp = new MqttMessage(pingHeader);
                this.channel.writeAndFlush(pingResp).addListener(ChannelFutureListener.CLOSE_ON_FAILURE);
                break;
            }
            default: {
                LOG.error("Unknown MessageType: {}", (Object)messageType);
            }
        }
    }

    private void processPubComp(MqttMessage msg) {
        int messageID = ((MqttMessageIdVariableHeader)msg.variableHeader()).messageId();
        this.bindedSession.processPubComp(messageID);
    }

    private void processPubRec(MqttMessage msg) {
        int messageID = ((MqttMessageIdVariableHeader)msg.variableHeader()).messageId();
        this.bindedSession.processPubRec(messageID);
    }

    static MqttMessage pubrel(int messageID) {
        MqttFixedHeader pubRelHeader = new MqttFixedHeader(MqttMessageType.PUBREL, false, MqttQoS.AT_LEAST_ONCE, false, 0);
        return new MqttMessage(pubRelHeader, MqttMessageIdVariableHeader.from(messageID));
    }

    private void processPubAck(MqttMessage msg) {
        int messageID = ((MqttMessageIdVariableHeader)msg.variableHeader()).messageId();
        this.bindedSession.pubAckReceived(messageID);
    }

    void processConnect(final MqttConnectMessage msg) {
        SessionRegistry.SessionCreationResult result;
        MqttConnectPayload payload = msg.payload();
        String clientId = payload.clientIdentifier();
        String username = payload.userName();
        LOG.trace("Processing CONNECT message. CId: {} username: {}", (Object)clientId, (Object)username);
        if (this.isNotProtocolVersion(msg, MqttVersion.MQTT_3_1) && this.isNotProtocolVersion(msg, MqttVersion.MQTT_3_1_1)) {
            LOG.warn("MQTT protocol version is not valid. CId: {}", (Object)clientId);
            this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_UNACCEPTABLE_PROTOCOL_VERSION);
            return;
        }
        boolean cleanSession = msg.variableHeader().isCleanSession();
        if (clientId == null || clientId.length() == 0) {
            if (!this.brokerConfig.isAllowZeroByteClientId()) {
                LOG.info("Broker doesn't permit MQTT empty client ID. Username: {}", (Object)username);
                this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_IDENTIFIER_REJECTED);
                return;
            }
            if (!cleanSession) {
                LOG.info("MQTT client ID cannot be empty for persistent session. Username: {}", (Object)username);
                this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_IDENTIFIER_REJECTED);
                return;
            }
            clientId = UUID.randomUUID().toString().replace("-", "");
            LOG.debug("Client has connected with integration generated id: {}, username: {}", (Object)clientId, (Object)username);
        }
        if (!this.login(msg, clientId)) {
            this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_BAD_USER_NAME_OR_PASSWORD);
            this.channel.close().addListener(ChannelFutureListener.CLOSE_ON_FAILURE);
            return;
        }
        try {
            LOG.trace("Binding MQTTConnection to session");
            result = this.sessionRegistry.createOrReopenSession(msg, clientId, this.getUsername());
            result.session.bind(this);
            this.bindedSession = result.session;
        }
        catch (SessionCorruptedException scex) {
            LOG.warn("MQTT session for client ID {} cannot be created", (Object)clientId);
            this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_SERVER_UNAVAILABLE);
            return;
        }
        boolean msgCleanSessionFlag = msg.variableHeader().isCleanSession();
        boolean isSessionAlreadyPresent = !msgCleanSessionFlag && result.alreadyStored;
        final String clientIdUsed = clientId;
        MqttConnAckMessage ackMessage = MqttMessageBuilders.connAck().returnCode(MqttConnectReturnCode.CONNECTION_ACCEPTED).sessionPresent(isSessionAlreadyPresent).build();
        this.channel.writeAndFlush(ackMessage).addListener(new ChannelFutureListener(){

            @Override
            public void operationComplete(ChannelFuture future) throws Exception {
                if (future.isSuccess()) {
                    LOG.trace("CONNACK sent, channel: {}", (Object)MQTTConnection.this.channel);
                    if (!result.session.completeConnection()) {
                        MqttMessage disconnectMsg = MqttMessageBuilders.disconnect().build();
                        MQTTConnection.this.channel.writeAndFlush(disconnectMsg).addListener(CLOSE);
                        LOG.warn("CONNACK is sent but the session created can't transition in CONNECTED state");
                    } else {
                        NettyUtils.clientID(MQTTConnection.this.channel, clientIdUsed);
                        MQTTConnection.this.connected = true;
                        if (result.mode == SessionRegistry.CreationModeEnum.REOPEN_EXISTING) {
                            result.session.sendQueuedMessagesWhileOffline();
                        }
                        MQTTConnection.this.initializeKeepAliveTimeout(MQTTConnection.this.channel, msg, clientIdUsed);
                        MQTTConnection.this.setupInflightResender(MQTTConnection.this.channel);
                        MQTTConnection.this.postOffice.dispatchConnection(msg);
                        LOG.trace("dispatch connection: {}", (Object)msg.toString());
                    }
                } else {
                    MQTTConnection.this.bindedSession.disconnect();
                    MQTTConnection.this.sessionRegistry.remove(MQTTConnection.this.bindedSession);
                    LOG.error("CONNACK send failed, cleanup session and close the connection", future.cause());
                    MQTTConnection.this.channel.close();
                }
            }
        });
    }

    private void setupInflightResender(Channel channel) {
        channel.pipeline().addFirst("inflightResender", (ChannelHandler)new InflightResender(5000L, TimeUnit.MILLISECONDS));
    }

    private void initializeKeepAliveTimeout(Channel channel, MqttConnectMessage msg, String clientId) {
        int keepAlive = msg.variableHeader().keepAliveTimeSeconds();
        NettyUtils.keepAlive(channel, keepAlive);
        NettyUtils.cleanSession(channel, msg.variableHeader().isCleanSession());
        NettyUtils.clientID(channel, clientId);
        int idleTime = Math.round((float)keepAlive * 1.5f);
        this.setIdleTime(channel.pipeline(), idleTime);
        LOG.debug("Connection has been configured CId={}, keepAlive={}, removeTemporaryQoS2={}, idleTime={}", clientId, keepAlive, msg.variableHeader().isCleanSession(), idleTime);
    }

    private void setIdleTime(ChannelPipeline pipeline, int idleTime) {
        if (pipeline.names().contains("idleStateHandler")) {
            pipeline.remove("idleStateHandler");
        }
        pipeline.addFirst("idleStateHandler", (ChannelHandler)new IdleStateHandler(idleTime, 0, 0));
    }

    private boolean isNotProtocolVersion(MqttConnectMessage msg, MqttVersion version) {
        return msg.variableHeader().version() != version.protocolLevel();
    }

    private void abortConnection(MqttConnectReturnCode returnCode) {
        MqttConnAckMessage badProto = MqttMessageBuilders.connAck().returnCode(returnCode).sessionPresent(false).build();
        this.channel.writeAndFlush(badProto).addListener(ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
        this.channel.close().addListener(ChannelFutureListener.CLOSE_ON_FAILURE);
    }

    private boolean login(MqttConnectMessage msg, String clientId) {
        if (msg.variableHeader().hasUserName()) {
            byte[] pwd = null;
            if (msg.variableHeader().hasPassword()) {
                pwd = msg.payload().passwordInBytes();
            } else if (!this.brokerConfig.isAllowAnonymous()) {
                LOG.info("Client didn't supply any password and MQTT anonymous mode is disabled CId={}", (Object)clientId);
                return false;
            }
            String login = msg.payload().userName();
            if (!this.authenticator.checkValid(clientId, login, pwd)) {
                LOG.info("Authenticator has rejected the MQTT credentials CId={}, username={}", (Object)clientId, (Object)login);
                return false;
            }
            NettyUtils.userName(this.channel, login);
        } else if (!this.brokerConfig.isAllowAnonymous()) {
            LOG.info("Client didn't supply any credentials and MQTT anonymous mode is disabled. CId={}", (Object)clientId);
            return false;
        }
        return true;
    }

    void handleConnectionLost() {
        String clientID = NettyUtils.clientID(this.channel);
        if (clientID == null || clientID.isEmpty()) {
            return;
        }
        LOG.info("Notifying connection lost event");
        if (this.bindedSession.hasWill()) {
            this.postOffice.fireWill(this.bindedSession.getWill());
        }
        if (this.bindedSession.isClean()) {
            LOG.debug("Remove session for client");
            this.sessionRegistry.remove(this.bindedSession);
        } else {
            this.bindedSession.disconnect();
        }
        this.connected = false;
        String userName = NettyUtils.userName(this.channel);
        this.postOffice.dispatchConnectionLost(clientID, userName);
        LOG.trace("dispatch disconnection: userName={}", (Object)userName);
    }

    boolean isConnected() {
        return this.connected;
    }

    void dropConnection() {
        this.channel.close().addListener(ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
    }

    void processDisconnect(MqttMessage msg) {
        String clientID = NettyUtils.clientID(this.channel);
        LOG.trace("Start DISCONNECT");
        if (!this.connected) {
            LOG.info("DISCONNECT received on already closed connection");
            return;
        }
        this.bindedSession.disconnect();
        this.connected = false;
        this.channel.close().addListener(ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
        LOG.trace("Processed DISCONNECT");
        String userName = NettyUtils.userName(this.channel);
        this.postOffice.dispatchDisconnection(clientID, userName);
        LOG.trace("dispatch disconnection userName={}", (Object)userName);
    }

    void processSubscribe(MqttSubscribeMessage msg) {
        String clientID = NettyUtils.clientID(this.channel);
        if (!this.connected) {
            LOG.warn("SUBSCRIBE received on already closed connection");
            this.dropConnection();
            return;
        }
        this.postOffice.subscribeClientToTopics(msg, clientID, NettyUtils.userName(this.channel), this);
    }

    void sendSubAckMessage(int messageID, MqttSubAckMessage ackMessage) {
        LOG.trace("Sending SUBACK response messageId: {}", (Object)messageID);
        this.channel.writeAndFlush(ackMessage).addListener(ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
    }

    private void processUnsubscribe(MqttUnsubscribeMessage msg) {
        List<String> topics = msg.payload().topics();
        String clientID = NettyUtils.clientID(this.channel);
        LOG.trace("Processing UNSUBSCRIBE message. topics: {}", (Object)topics);
        this.postOffice.unsubscribe(topics, this, msg.variableHeader().messageId());
    }

    void sendUnsubAckMessage(List<String> topics, String clientID, int messageID) {
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.UNSUBACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttUnsubAckMessage ackMessage = new MqttUnsubAckMessage(fixedHeader, MqttMessageIdVariableHeader.from(messageID));
        LOG.trace("Sending UNSUBACK message. messageId: {}, topics: {}", (Object)messageID, (Object)topics);
        this.channel.writeAndFlush(ackMessage).addListener(ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
        LOG.trace("Client unsubscribed from topics <{}>", (Object)topics);
    }

    void processPublish(MqttPublishMessage msg) {
        MqttQoS qos = msg.fixedHeader().qosLevel();
        String username = NettyUtils.userName(this.channel);
        String topicName = msg.variableHeader().topicName();
        String clientId = this.getClientId();
        int messageID = msg.variableHeader().packetId();
        LOG.trace("Processing PUBLISH message, topic: {}, messageId: {}, qos: {}", new Object[]{topicName, messageID, qos});
        Topic topic = new Topic(topicName);
        if (!topic.isValid()) {
            LOG.debug("Drop connection because of invalid topic format");
            this.dropConnection();
        }
        switch (qos) {
            case AT_MOST_ONCE: {
                this.postOffice.receivedPublishQos0(topic, username, clientId, msg);
                break;
            }
            case AT_LEAST_ONCE: {
                this.postOffice.receivedPublishQos1(this, topic, username, messageID, msg);
                break;
            }
            case EXACTLY_ONCE: {
                this.bindedSession.receivedPublishQos2(messageID, msg);
                this.postOffice.receivedPublishQos2(this, msg, username);
                break;
            }
            default: {
                LOG.error("Unknown QoS-Type:{}", (Object)qos);
            }
        }
    }

    void sendPublishReceived(int messageID) {
        LOG.trace("sendPubRec invoked, messageID: {}", (Object)messageID);
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBREC, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttPubAckMessage pubRecMessage = new MqttPubAckMessage(fixedHeader, MqttMessageIdVariableHeader.from(messageID));
        this.sendIfWritableElseDrop(pubRecMessage);
    }

    private void processPubRel(MqttMessage msg) {
        int messageID = ((MqttMessageIdVariableHeader)msg.variableHeader()).messageId();
        this.bindedSession.receivedPubRelQos2(messageID);
        this.sendPubCompMessage(messageID);
    }

    void sendPublish(MqttPublishMessage publishMsg) {
        int packetId = publishMsg.variableHeader().packetId();
        String topicName = publishMsg.variableHeader().topicName();
        String clientId = this.getClientId();
        MqttQoS qos = publishMsg.fixedHeader().qosLevel();
        if (LOG.isTraceEnabled()) {
            LOG.trace("Sending PUBLISH({}) message. MessageId={}, topic={}, payload={}", new Object[]{qos, packetId, topicName, DebugUtils.payload2Str(publishMsg.payload())});
        } else {
            LOG.debug("Sending PUBLISH({}) message. MessageId={}, topic={}", new Object[]{qos, packetId, topicName});
        }
        this.sendIfWritableElseDrop(publishMsg);
    }

    void sendIfWritableElseDrop(MqttMessage msg) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("OUT {}", (Object)msg.fixedHeader().messageType());
        }
        if (this.channel.isWritable()) {
            Object retainedDup = msg;
            if (msg instanceof ByteBufHolder) {
                retainedDup = ((ByteBufHolder)((Object)msg)).retainedDuplicate();
            }
            ChannelFuture channelFuture = this.brokerConfig.isImmediateBufferFlush() ? this.channel.writeAndFlush(retainedDup) : this.channel.write(retainedDup);
            channelFuture.addListener(ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
        }
    }

    public void writabilityChanged() {
        if (this.channel.isWritable()) {
            LOG.debug("Channel is again writable");
            this.bindedSession.writabilityChanged();
        }
    }

    void sendPubAck(int messageID) {
        LOG.trace("sendPubAck for messageID: {}", (Object)messageID);
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttPubAckMessage pubAckMessage = new MqttPubAckMessage(fixedHeader, MqttMessageIdVariableHeader.from(messageID));
        this.sendIfWritableElseDrop(pubAckMessage);
    }

    private void sendPubCompMessage(int messageID) {
        LOG.trace("Sending PUBCOMP message messageId: {}", (Object)messageID);
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBCOMP, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttMessage pubCompMessage = new MqttMessage(fixedHeader, MqttMessageIdVariableHeader.from(messageID));
        this.sendIfWritableElseDrop(pubCompMessage);
    }

    String getClientId() {
        return NettyUtils.clientID(this.channel);
    }

    String getUsername() {
        return NettyUtils.userName(this.channel);
    }

    public void sendPublishRetainedQos0(Topic topic, MqttQoS qos, ByteBuf payload) {
        MqttPublishMessage publishMsg = MQTTConnection.retainedPublish(topic.toString(), qos, payload);
        this.sendPublish(publishMsg);
    }

    public void sendPublishRetainedWithPacketId(Topic topic, MqttQoS qos, ByteBuf payload) {
        int packetId = this.nextPacketId();
        MqttPublishMessage publishMsg = MQTTConnection.retainedPublishWithMessageId(topic.toString(), qos, payload, packetId);
        this.sendPublish(publishMsg);
    }

    private static MqttPublishMessage retainedPublish(String topic, MqttQoS qos, ByteBuf message) {
        return MQTTConnection.retainedPublishWithMessageId(topic, qos, message, 0);
    }

    private static MqttPublishMessage retainedPublishWithMessageId(String topic, MqttQoS qos, ByteBuf message, int messageId) {
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBLISH, false, qos, true, 0);
        MqttPublishVariableHeader varHeader = new MqttPublishVariableHeader(topic, messageId);
        return new MqttPublishMessage(fixedHeader, varHeader, message);
    }

    void sendPublishNotRetainedQos0(Topic topic, MqttQoS qos, ByteBuf payload) {
        MqttPublishMessage publishMsg = MQTTConnection.notRetainedPublish(topic.toString(), qos, payload);
        this.sendPublish(publishMsg);
    }

    static MqttPublishMessage notRetainedPublish(String topic, MqttQoS qos, ByteBuf message) {
        return MQTTConnection.notRetainedPublishWithMessageId(topic, qos, message, 0);
    }

    static MqttPublishMessage notRetainedPublishWithMessageId(String topic, MqttQoS qos, ByteBuf message, int messageId) {
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBLISH, false, qos, false, 0);
        MqttPublishVariableHeader varHeader = new MqttPublishVariableHeader(topic, messageId);
        return new MqttPublishMessage(fixedHeader, varHeader, message);
    }

    public void resendNotAckedPublishes() {
        this.bindedSession.resendInflightNotAcked();
    }

    int nextPacketId() {
        return this.lastPacketId.updateAndGet(v -> v == 65535 ? 1 : v + 1);
    }

    public String toString() {
        return "MQTTConnection{channel=" + this.channel + ", connected=" + this.connected + '}';
    }

    InetSocketAddress remoteAddress() {
        return (InetSocketAddress)this.channel.remoteAddress();
    }

    public void readCompleted() {
        LOG.debug("readCompleted client CId: {}", (Object)this.getClientId());
        if (this.getClientId() != null) {
            this.bindedSession.flushAllQueuedMessages();
        }
    }

    public void flush() {
        this.channel.flush();
    }
}

