## -*-Tcl-*-
 # ###################################################################
 #	Vince's	Additions -	an extension package for Alpha
 # 
 #	FILE: "elecTemplates.tcl"
 #					created: 24/2/97 {1:34:29 pm}	
 #				  last update: 4/7/1999 {2:52:11 pm}	
 #	Author:	Vince Darley
 #	E-mail:	<darley@fas.harvard.edu>
 #	  mail:	Division of	Applied	Sciences, Harvard University
 #			Oxford Street, Cambridge MA	02138, USA
 #	   www:	<http://www.fas.harvard.edu/~darley/>
 #	
 #  Routines for electric insertions, and keeping track of template
 #  positions.	
 # ###################################################################
 ##

alpha::feature betterTemplates 9.2.2 global {
    alpha::package require elecBindings 9.0
    alpha::useElectricTemplates
} {
    lunion varPrefs(Electrics) [list "Better Templates:" stopNavigationMsgOff \
      templateStopColor maxTemplateNesting \
      TemplatePrompts TemplateWrappers]
    # The colour used for template stops inserted into the text.
    newPref var templateStopColor 4 global "" alpha::basiccolors varindex
    # If the level of nesting of template stops exceeds this value,
    # we clear all template stops.
    newPref var maxTemplateNesting 5
    ## 
     # The format of the template stops:
     #     (a) just use bullets
     #     (b) use bullets but signal the name in the status window
     #     (c) insert names into the window with the bullets
     #     (d) insert names and highlight into the window with the bullets
     ##
    newPref var TemplatePrompts 1 global "" [list {Just use bullets} \
      {Use bullets and status window prompt} {Put prompts in the text} \
      {Highlight prompts in the text}] index
    # Visual appearance of templates in the text
    newPref var TemplateWrappers 0 global ring::_changeTemplateWrappers \
      [list {<Angle brackets>} {Curly quotes} {Curly brackets} ] index
    # Don't bother with the basic 'hit tab to go to next stop...' message
    newPref flag stopNavigationMsgOff 0 global ring::setTemplateMessage
    # so we force a reload of this file when necessary
    if {[info commands ring::setTemplateMessage] != ""} {
	rename ring::setTemplateMessage ""
    }
    ring::setTemplateMessage
    # setup template wrappers
    ring::_changeTemplateWrappers
    # call on close to clear the stop ring.
    hook::register closeHook ring::unsetName	
} {
    hook::deregister closeHook ring::unsetName
    # source old code since we over-rode it below.
    source [file join $HOME Tcl SystemCode templates.tcl]
} maintainer {
    "Vince Darley" darley@fas.harvard.edu <http://www.fas.harvard.edu/~darley/>
} uninstall this-file help {file "ElecCompletions Help"}

# we don't want to be auto-loaded unless we're active.
#if {![package::active betterTemplates]} { 
#	alertnote "Something's trying to auto-load the betterTemplates extension\
#	  but it's not active!"
#	return 
#}

# indicates we're a better ring
proc ring::type {} { return 1 }

proc ring::isNested {p} {
    if {![catch {ring::minmax} mm] \
      && [pos::compare $p >= [lindex $mm 0]] \
      && [pos::compare $p <= [lindex $mm 1]]} {
	return 1
    } else {
	ring::clear
	return 0
    }
}

proc ring::nestedPos {pos} {
    if {[catch {
	set p [tmark::getPositions {nestStart nestEnd}]
	if {[pos::compare $pos < [lindex $p 0]]} { return -1 }
	if {[pos::compare $pos > [lindex $p 1]]} { return -1 }
    }]} { return -1 }
    set positions [ring::orderAndPositions]
    if {$positions == "" || [pos::compare $pos < [lindex $positions 0]] \
      || [pos::compare $pos >= [lindex $positions end]]} {
	return -1
    } else {
	set i 0
	while {[pos::compare $pos >= [lindex $positions $i]]} {incr i}
	return $i
    }
}

proc ring::minmax {} {
    return [tmark::getPositions {nestStart nestEnd}]
}
proc ring::getlist {} {
    # get a local reference to the window's stopRing
    upvar \#0 __elecRing([ring::winName]) s
    if {![info exists s]} {
	return [ring::clear]
    }
    set s
}

proc ring::clear {} {
    set x [ring::winName]
    # get a local reference to the window's stopRing
    upvar \#0 __elecRing($x) s
    if {[info exists s] && $s != ""} {
	ring::_ensure_no_bullets $s
    }
    set s ""
    upvar \#0 __elecRingPrompts$x w
    if {[info exists w]} {unset w}
    global __elecNestingLevel __elecLastStop
    set __elecNestingLevel($x) 0
    set __elecLastStop($x) ""
	
    removeTMark "nestStart"
    removeTMark "nestEnd"
}

proc ring::unsetName {name} {
    ring::unseti [join [file tail $name] ""]
}

proc ring::unseti {x} {
    global __elecRing __elecNestingLevel __elecLastStop __elecRingPrompts$x
    if {[info exists __elecRing($x)]} {
	unset __elecRing($x)
    }
    if {[info exists __elecNestingLevel($x)]} {
	unset __elecNestingLevel($x)
    }
    if {[info exists __elecLastStop($x)]} {
	unset __elecLastStop($x)
    }
    if {[info exists __elecRingPrompts$x]} {
	unset __elecRingPrompts$x
    }
}

proc ring::_ensure_no_bullets {stops} {
    message "Deleting non-nested prompts"
    createTMark "_deleting_" [getPos]
    foreach stop $stops {
	if {![catch {tmark::getPos $stop} p]} {
	    ring::_deleteBullet $p
	    removeTMark $stop
	}	
    }
    message ""
    gotoTMark "_deleting_"
    removeTMark "_deleting_"
}

## 
 # -------------------------------------------------------------------------
 # 
 # "ring::replaceStopMatches" --
 # 
 #  Replace all stops which match 'stoppat' (a simple glob like pattern)
 #  with the text '$text'.  The stops are permanently deleted.
 # -------------------------------------------------------------------------
 ##
proc ring::replaceStopMatches {stoppat text} {
    # get a local reference to the window's stopRing
    set x [ring::winName]
    upvar \#0 __elecRing($x) s
    if {[info exists s]} {
	placeBookmark
	upvar \#0 __elecRingPrompts$x w
	set i 0
	foreach stop $s {
	    if {[string match $stoppat $w($stop)]} {
		if {![catch {tmark::getPos $stop} p]} {
		    if {[ring::_deleteBullet $p]} {
			insertText $text
		    }
		    removeTMark $stop
		    set s [lreplace $s $i $i]
		    incr i -1
		}	
	    }
	    incr i
	}	
	returnToBookmark
    } else {
	ring::clear
    } 
}

proc ring::winName {} { return [join [win::CurrentTail] ""] }

proc ring::order {} {
    # get a local reference to the window's stopRing
    upvar \#0 __elecRing([ring::winName]) s
    if {[info exists s]} {
	for {set i 0} {$i <100} {incr i} {
	    if {[set lpos [lsearch -exact $s stop0:${i}]] != -1 } {
		set s [concat [lrange $s $lpos end] [lrange $s 0 [incr lpos -1]]]
		return $s
	    }
	}
    } else {
	ring::clear
    } 
}

proc ring::orderAndPositions {} {
    # get a local reference to the window's stopRing
    upvar \#0 __elecRing([ring::winName]) s
    if {[info exists s] && ([string trim $s] != {}) } {
	set positions [tmark::getPositions $s]
	set max -1
	set idx 0
	set lpos -1
	foreach st $s {
	    if {[pos::compare [set p [lindex $positions $idx]] > $max]} {
		set max $p
		set lpos $idx
	    }
	    incr idx
	}
	set s [concat [lrange $s [expr {$lpos +1}] end] [lrange $s 0 $lpos]]
	set positions [concat [lrange $positions [expr {$lpos +1}] end] \
	  [lrange $positions 0 $lpos]]
	return $positions
    } else {
	ring::clear
	return ""
    } 
}

## 
 # -------------------------------------------------------------------------
 # 
 # "ring::_deleteBullet" --
 # 
 #  Deletes the bullet and a following tag-prompt.  The mark moves to the
 #  location of the deleted text (side-effect).  Returns '1' if the deletion
 #  was successful, else '0'.
 # -------------------------------------------------------------------------
 ##
proc ring::_deleteBullet {p {h 0}} {
    global elecStopMarker
    if {[lookAt $p] == $elecStopMarker} {
	global ring::_tstart ring::_tmatch
	if {[lookAt [pos::math $p + 1]] == ${ring::_tstart} } {
	    set	ppos [search -s -f 1 -r 1 -l [pos::math $p + 80] -n ${ring::_tmatch} $p]
	    if {[pos::compare [lindex $ppos 0] == $p]} {
		if {$h} {
		    eval select $ppos
		} else {
		    eval deleteText $ppos
		}
		return 1
	    }
	}
	deleteText $p [pos::math $p + 1]
	return 1
    }
    return 0
}

proc ring::_goto {rest} {
    global __elecLastStop ring::_templateMessage TemplatePrompts
    set x [ring::winName]
    gotoTMark [set __elecLastStop($x) $rest]
    # remove the stop '' plus optional prompt-tag.
    ring::_deleteBullet [getPos] [expr {$TemplatePrompts == 3}]
    if {$TemplatePrompts} {
	upvar \#0 __elecRingPrompts$x w
	if {$w($rest) != ""} {
	    message "Fill in '$w($rest)'${ring::_templateMessage}"
	} else {
	    message "Fill in template stop${ring::_templateMessage}"
	}
    }
}

proc ring::nth {} {
    # get a local reference to the window's stopRing
    set x [ring::winName]
    upvar \#0 __elecRing($x) s
    upvar \#0 __elecRingPrompts$x w
    foreach f $s {
	if {$w($f) != ""} {
	    lappend l "$f -- $w($f)"
	} else {
	    lappend l "$f -- (no prompt)"
	}
    }
    if {![info exists l]} { beep; message "No template stops exist." }
    set item [lindex [listpick -p "Pick a stop (listed from current pos)" $l] 0]
    ring::goTo $item
}
proc ring::goTo {stop} {
    # get a local reference to the window's stopRing
    upvar \#0 __elecRing([ring::winName]) s
    if {[info exists s]} {
	if { [set lpos [lsearch -exact $s $stop]] != -1 } {
	    set s [concat [lrange $s $lpos end] [lrange $s 0 [incr lpos -1]]]
	    ring::_goto $stop
	}
    } else {
	ring::clear
    } 
}

## 
 # -------------------------------------------------------------------------
 # 
 # "ring::TMarkAt" --
 # 
 #  Is the template stop with prompt 'name' at position 'pos'.  The 'name'
 #  is the name of the enclosed prompt as in 'environment name', but
 #  without the bullets.  It is matched via 'string match'.
 # -------------------------------------------------------------------------
 ##
proc ring::TMarkAt {name pos} {
    set stop [tmark::isAt $pos]
    if {$stop != ""} {
	set x [ring::winName]
	upvar \#0 __elecRingPrompts$x w
	return [string match $name $w($stop)]
    } else {
	return 0
    }
}

proc ring::+ {} {
    # get a local reference to the window's stopRing
    upvar \#0 __elecRing([ring::winName]) s
    set first [lindex $s 0]
    set s [lreplace $s 0 0]
    lappend s $first
    set next [lindex $s 0]
    ring::_goto $next
}
proc ring::- {} {
    # get a local reference to the window's stopRing
    upvar \#0 __elecRing([ring::winName]) s
    #set end [expr {[llength $s] - 1}]
    set last [lindex $s end]
    set s [lreplace $s end end]
    set s [linsert $s 0 $last]
    ring::_goto $last
}

proc ring::deleteBulletAndMove {} {
    ring::_deleteBullet [getPos]
    ring::+
}

proc ring::deleteStopAndMove {} {
    ring::_deleteStop
    upvar \#0 __elecRing([ring::winName]) s
    ring::_goto [lindex $s 0]
}

proc ring::deleteStop {} {
    ring::_deleteStop
}

proc ring::_deleteStop {} {
    global __elecLastStop
    set x [ring::winName]
    # get a local reference to the window's stopRing
    upvar \#0 __elecRing($x) s
    set l [lsearch -exact $s $__elecLastStop($x)]
    if {$l != -1 } {
	global TemplatePrompts
	if {$TemplatePrompts == 3} {
	    ring::_deleteBullet [getPos]
	}
	set s [lreplace $s $l $l]
	removeTMark $__elecLastStop($x)
	set __elecLastStop($x) ""
    }
}

proc ring::insert {rest {goto 1}} {
    global __elecNestingLevel __elecCurrentNesting maxTemplateNesting \
      elecStopMarker
    # get a local reference to the window's stopRing
    set x [ring::winName]
    upvar \#0 __elecRing($x) s
    # if not nested, clear everything
    if {[set p [ring::nestedPos [getPos]]] == "-1" \
      || [incr __elecNestingLevel($x)] > $maxTemplateNesting } {
	ring::clear
	set p 0
    }
    set _level $__elecNestingLevel($x)
    # preliminaries
    set pos [getPos]
    set ii [set i 0] 
    upvar \#0 __elecRingPrompts$x w
    global __elecPrompts
    if {![info exists __elecPrompts]} {
	set __elecPrompts ""
    }
    # do the stop ring, extracting prompts from '__elecPrompts'
    while {[regexp -indices $elecStopMarker $rest I] == 1} {
	regsub $elecStopMarker $rest "o" rest
	createTMark "stop${_level}:$i" [pos::math $pos + [lindex $I 0]]
	lappend ss "stop${_level}:$i"
	set w(stop${_level}:$i) [lindex $__elecPrompts $i]
	#set __elecPrompts [lrange $__elecPrompts 1 end]
	incr i
    }
    if {$i > 2 || ($i == 2 && $_level == 0)} {
	# store insertion's min and max, if we have more than two stops
	createTMark "nestStart" $pos
	createTMark "nestEnd" [pos::math $pos + [string length $rest]]
    }
    # put the stop ring together
    set s [concat $ss [lrange $s $p end] [lrange $s 0 [expr {$p -1}]]]
    # forget the prompt list (we've stored them in an array)
    unset __elecPrompts
    # goto the first stop we just inserted
    if {$goto} {
	ring::_goto "stop${_level}:${ii}"
    }
}


proc ring::_changeTemplateWrappers {{flag ""}} {
    global flag::list TemplateWrappers elecStopMarker
    set wrap [lindex [lindex [set flag::list(TemplateWrappers)] 1] $TemplateWrappers]
    global ring::_tstart ring::_tend ring::_tmatch
    set a [string index $wrap 0]
    set b [string index $wrap [expr {[string length $wrap] -1}]]
	
    set "ring::_tstart" $a
    set "ring::_tend" $b
    # 	set "ring::_tmatch" "${a}\[^${a}${b}\]*${b}"
    set "ring::_tmatch" "(${elecStopMarker}${a}\[^${a}${b}]*${b}|${elecStopMarker}${a}(\[^${a}${b}\]*(${a}\[^${a}${b}\]*${b})\[^${a}${b}\]*)*${b})"
}

proc ring::setTemplateMessage {} {
    global electricBindings ring::_templateMessage stopNavigationMsgOff
    set ring::_templateMessage [lindex \
      {", press Tab (shift-Tab) to move to the next (previous) stop." \
      ", press ctrl-j (shift-ctrl-j) to move to the next (previous) stop." \
      ", press user-defined keys to move from stop to stop." } \
      $electricBindings]
    if {$stopNavigationMsgOff} {
	set ring::_templateMessage ""
    } 
}



## 
 # -------------------------------------------------------------------------
 #	 
 #	"elec::_Insertion" --
 #	
 #  Insert a piece of text, padding on the left appropriately.  The text
 #  should already be correctly indented within itself. 
 # -------------------------------------------------------------------------
 ##
proc elec::_Insertion { center args } {
    global __elecPrompts TemplatePrompts elecStopMarker
    set text [join $args ""]
    set pos [getPos]
    regsub -all "\t" $text [text::Tab] text
    if {[regexp "\[\n\r\]" $text]} {
	regsub -all "\[\n\r\]" $text "\r[text::indentTo $pos]" text
    }
    if {[regexp "" $text]} {
	regsub -all "" $text [text::halfTab] text
    }
    if {![regexp "" $text] || ([regexp {^([^]*)$} $text "" text])} {
	setMark
	insertText $text
	if {$center} { centerRedraw }
	return
    }
    switch -- $TemplatePrompts {
	0 {
	    set t $text
	    regsub -all {[^]*} $text $elecStopMarker text
	    insertText $text
	    while {[regexp {^([^]*)([^]*)(.*)$} $t "" tt hyper t]} {
		lappend __elecPrompts $hyper
	    }
	}
	1 {
	    while {[regexp {^([^]*)([^]*)(.*)$} $text "" tt hyper text]} {
		lappend __elecPrompts $hyper
		append t "${tt}$elecStopMarker"
		lappend colours [list [string length $tt] 1]
	    }
	    append t $text
	}
	2 -
	3 {
	    global ring::_tstart ring::_tend
	    while {[regexp {^([^]*)([^]*)(.*)$} $text "" tt hyper text]} {
		lappend __elecPrompts $hyper
		if {$hyper != ""} {
		    append t "${tt}${elecStopMarker}${ring::_tstart}${hyper}${ring::_tend}"
		    lappend colours [list [string length $tt] \
		      [expr {3 + [string length $hyper]}]]
		} else {
		    append t "${tt}${elecStopMarker}"
		    lappend colours [list [string length $tt] 1]
		}
	    }
	    append t $text
	}
    }
    if {$TemplatePrompts} {
	set p $pos
	# we insert in one chunk so undoing is easy.
	insertText $t
	global templateStopColor
	if {$templateStopColor} {
	    foreach col $colours {
		set p [pos::math $p + [lindex $col 0]]
		insertColorEscape $p $templateStopColor
		set p [pos::math $p + [lindex $col 1]]
		insertColorEscape $p 0
	    }
	}
	
	set text $t
    }
    
    goto $pos
    if {$center} { centerRedraw }
    ring::insert $text
}


#  possible tab key bindings  #
# note: Also provided by the base Alpha system, these overide when 
# Univs Completions package is in use (these may be more intricate).

## 
 # -------------------------------------------------------------------------
 #	 
 #	"bind::IndentOrNextstop" --
 #	
 #  Either insert a real tab if your mode hasn't defined its electricTab
 #  variable, or jump to the next template stop (if we're mid-template), or
 #  indent the current line correctly. 
 # -------------------------------------------------------------------------
 ##
proc bind::IndentOrNextstop {{hard 0}} {
    global electricTab
    if {$hard || !$electricTab} {
	insertActualTab 
    } else {
	global tabNeverIndents
	if {[info exists tabNeverIndents] && $tabNeverIndents} { return [ring::+] }
	if {[ring::isNested [getPos]]} {
	    ring::+
	} else {
	    bind::IndentLine
	}
    }
}

## 
 # -------------------------------------------------------------------------
 #	 
 #	"bind::TabOrComplete" --
 #	
 #  Either insert a real tab if your mode hasn't defined its electricTab
 #  variable, or invoke the completion mechanism, or indent the current
 #  line correctly. 
 # -------------------------------------------------------------------------
 ##
proc bind::TabOrComplete {{hard 0}} {
    global electricTab
    if {$hard || !$electricTab} {
	insertActualTab 
    } else {
	bind::Completion
    }
}
