﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/rum/CloudWatchRUMRequest.h>
#include <aws/rum/CloudWatchRUM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace CloudWatchRUM {
namespace Model {

/**
 */
class PutResourcePolicyRequest : public CloudWatchRUMRequest {
 public:
  AWS_CLOUDWATCHRUM_API PutResourcePolicyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutResourcePolicy"; }

  AWS_CLOUDWATCHRUM_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the app monitor that you want to apply this resource-based policy
   * to. To find the names of your app monitors, you can use the <a
   * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_ListAppMonitors.html">ListAppMonitors</a>
   * operation.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  PutResourcePolicyRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The JSON to use as the resource policy. The document can be up to 4 KB in
   * size. For more information about the contents and syntax for this policy, see <a
   * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-resource-policies.html">Using
   * resource-based policies with CloudWatch RUM</a>.</p>
   */
  inline const Aws::String& GetPolicyDocument() const { return m_policyDocument; }
  inline bool PolicyDocumentHasBeenSet() const { return m_policyDocumentHasBeenSet; }
  template <typename PolicyDocumentT = Aws::String>
  void SetPolicyDocument(PolicyDocumentT&& value) {
    m_policyDocumentHasBeenSet = true;
    m_policyDocument = std::forward<PolicyDocumentT>(value);
  }
  template <typename PolicyDocumentT = Aws::String>
  PutResourcePolicyRequest& WithPolicyDocument(PolicyDocumentT&& value) {
    SetPolicyDocument(std::forward<PolicyDocumentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A string value that you can use to conditionally update your policy. You can
   * provide the revision ID of your existing policy to make mutating requests
   * against that policy.</p> <p>When you assign a policy revision ID, then later
   * requests about that policy will be rejected with an
   * <code>InvalidPolicyRevisionIdException</code> error if they don't provide the
   * correct current revision ID.</p>
   */
  inline const Aws::String& GetPolicyRevisionId() const { return m_policyRevisionId; }
  inline bool PolicyRevisionIdHasBeenSet() const { return m_policyRevisionIdHasBeenSet; }
  template <typename PolicyRevisionIdT = Aws::String>
  void SetPolicyRevisionId(PolicyRevisionIdT&& value) {
    m_policyRevisionIdHasBeenSet = true;
    m_policyRevisionId = std::forward<PolicyRevisionIdT>(value);
  }
  template <typename PolicyRevisionIdT = Aws::String>
  PutResourcePolicyRequest& WithPolicyRevisionId(PolicyRevisionIdT&& value) {
    SetPolicyRevisionId(std::forward<PolicyRevisionIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_policyDocument;
  bool m_policyDocumentHasBeenSet = false;

  Aws::String m_policyRevisionId;
  bool m_policyRevisionIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchRUM
}  // namespace Aws
