/*******************************************************************************
* Copyright 2014-2020 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

//@HEADER
// ***************************************************
//
// HPCG: High Performance Conjugate Gradient Benchmark
//
// Contact:
// Michael A. Heroux ( maherou@sandia.gov)
// Jack Dongarra     (dongarra@eecs.utk.edu)
// Piotr Luszczek    (luszczek@eecs.utk.edu)
//
// ***************************************************
//@HEADER

/*!
 @file TestCG.cpp

 HPCG routine
 */

// Changelog
//
// Version 0.4
// - Added timing of setup time for sparse MV
// - Corrected percentages reported for sparse MV with overhead
//
/////////////////////////////////////////////////////////////////////////

#include <fstream>
#include <iostream>
using std::endl;
#include <vector>
#include "hpcg.hpp"
#include "WriteProblem.hpp"

#include "TestCG.hpp"
#include "CG.hpp"
#include "CG_ref.hpp"

/*!
  Test the correctness of the Preconditined CG implementation by using a system matrix with a dominant diagonal.

  @param[in]    geom The description of the problem's geometry.
  @param[in]    A    The known system matrix
  @param[in]    data the data structure with all necessary CG vectors preallocated
  @param[in]    b    The known right hand side vector
  @param[inout] x    On entry: the initial guess; on exit: the new approximate solution
  @param[out]   testcg_data the data structure with the results of the test including pass/fail information

  @return Returns zero on success and a non-zero value otherwise.

  @see CG()
 */
int TestCG(SparseMatrix & A, CGData & data, Vector & b, Vector & x, TestCGData & testcg_data,
           sycl::queue & main_queue) {

    // Use this array for collecting timing information
    std::vector< double > times(8,0.0);

    Vector origDiagA, exaggeratedDiagA, origB;
    InitializeVectorShared(origDiagA, A.localNumberOfRows, main_queue);
    InitializeVectorShared(exaggeratedDiagA, A.localNumberOfRows, main_queue);
    InitializeVectorShared(origB, A.localNumberOfRows, main_queue);
    CopyMatrixDiagonal(A, origDiagA, main_queue).wait();
    CopyVector(origDiagA, exaggeratedDiagA, main_queue).wait();
    CopyVector(b, origB, main_queue).wait();

    // Modify the matrix diagonal and RHS to greatly exaggerate diagonal values.
    // CG should converge in about 10 iterations for this problem, regardless of problem size
    global_int_t * A_localToGlobalMap = A.localToGlobalMap;
    struct optData *optData = (struct optData *)A.optimizationData;
    local_int_t *invperm = optData->invperm;

    double * exaggeratedDiag_values = exaggeratedDiagA.values;
    double * b_values = b.values;
    auto ev = main_queue.submit([&](sycl::handler& cgh) {
        cgh.parallel_for(
            sycl::range<1>(A.localNumberOfRows),
            [=](sycl::id<1> id) {
                global_int_t globalRowID = A_localToGlobalMap[invperm[id]];
                if (globalRowID < 9) {
                    double scale = (globalRowID+2) * 1.0e6;
                    exaggeratedDiag_values[id] *= scale;
                    b_values[id] *= scale;
                }
                else {
                    exaggeratedDiag_values[id] *= 1.0e6;
                    b_values[id] *= 1.0e6;
                }
            });
    });
    ev.wait();

#ifndef HPCG_LOCAL_LONG_LONG
    ReplaceSparseMatrixDiagonal(A, exaggeratedDiagA, main_queue).wait();
#else
    ReplaceMatrixDiagonal(A, exaggeratedDiagA);
#endif

    int niters = 0;
    double normr = 0.0;
    double normr0 = 0.0;
    int maxIters = 50;
    int numberOfCgCalls = 2;
    double tolerance = 1.0e-12; // Set tolerance to reasonable value for grossly scaled diagonal terms
    testcg_data.expected_niters_no_prec = 12; // For the unpreconditioned CG call, we should take about 10 iterations, permit 12
    testcg_data.expected_niters_prec = 2;   // For the preconditioned case, we should take about 1 iteration, permit 2
    testcg_data.niters_max_no_prec = 0;
    testcg_data.niters_max_prec = 0;
    for (int k=0; k<2; ++k) { // This loop tests both unpreconditioned and preconditioned runs
        int expected_niters = testcg_data.expected_niters_no_prec;
        if (k==1) expected_niters = testcg_data.expected_niters_prec;
        for (int i=0; i< numberOfCgCalls; ++i) {
            ZeroVector(x, main_queue).wait(); // Zero out x
            int ierr = CG(A, data, b, x, maxIters, tolerance, niters, normr, normr0, &times[0],
                          k==1, main_queue);
            if (ierr) HPCG_fout << "Error in call to CG: " << ierr << ".\n" << endl;
            if (niters <= expected_niters) {
                ++testcg_data.count_pass;
            } else {
                ++testcg_data.count_fail;
            }
            if (k==0 && niters>testcg_data.niters_max_no_prec) testcg_data.niters_max_no_prec = niters; // Keep track of largest iter count
            if (k==1 && niters>testcg_data.niters_max_prec) testcg_data.niters_max_prec = niters; // Same for preconditioned run
            if (A.geom->rank==0) {
                HPCG_fout << "Call [" << i << "] Number of Iterations [" << niters <<"] Scaled Residual [" << normr/normr0 << "]" << endl;
                if (niters > expected_niters) {
                    HPCG_fout << " Expected " << expected_niters << " iterations.  Performed " << niters << "." << endl;
                }
            }
        }
    }

    // Restore matrix diagonal and RHS
#ifndef HPCG_LOCAL_LONG_LONG
    ReplaceSparseMatrixDiagonal(A, origDiagA, main_queue).wait();
#else
    ReplaceMatrixDiagonal(A, origDiagA);
#endif

    CopyVector(origB, b, main_queue).wait();

    // Delete vectors
    DeleteVector(origDiagA, main_queue); 
    DeleteVector(exaggeratedDiagA, main_queue);
    DeleteVector(origB, main_queue);

    testcg_data.normr = normr;

    return 0;
}
