// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
/**
 * This class is the KMS H-keyring. This class is within the kms-keyring-node
 * module because it is a KMS keyring variation. However, the KDF used in this
 * keyring's operations will only work in Node.js runtimes and not browser JS.
 * Thus, this H-keyring implementation is only Node compatible, and thus,
 * resides in a node module, not a browser module
 */
import { EncryptedDataKey, immutableClass, KeyringNode, needs, readOnlyProperty, isDecryptionMaterial, } from '@aws-crypto/material-management';
import { getLocalCryptographicMaterialsCache, } from '@aws-crypto/cache-material';
import { destructureCiphertext, getBranchKeyId, getBranchKeyMaterials, getCacheEntryId, getPlaintextDataKey, wrapPlaintextDataKey, unwrapEncryptedDataKey, filterEdk, modifyEncryptionMaterial, modifyDencryptionMaterial, decompressBytesToUuidv4, stringToUtf8Bytes, } from './kms_hkeyring_node_helpers';
import { isIBranchKeyStoreNode, } from '@aws-crypto/branch-keystore-node';
import { isBranchKeyIdSupplier, } from '@aws-crypto/kms-keyring';
import { randomBytes } from 'crypto';
export class KmsHierarchicalKeyRingNode extends KeyringNode {
    constructor({ branchKeyId, branchKeyIdSupplier, keyStore, cacheLimitTtl, cache, maxCacheSize, partitionId, }) {
        super();
        needs(!partitionId || typeof partitionId === 'string', 'Partition id must be a string.');
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#partition-id
        //= type=implication
        //# The Partition ID MUST NOT be changed after initialization.
        readOnlyProperty(this, '_partition', 
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#partition-id-1
        //# It can either be a String provided by the user, which MUST be interpreted as the bytes of
        //# UTF-8 Encoding of the String, or a v4 UUID, which SHOULD be interpreted as the 16 byte representation of the UUID.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#partition-id-1
        //# The constructor of the Hierarchical Keyring MUST record these bytes at construction time.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#partition-id
        //# If provided, it MUST be interpreted as UTF8 bytes.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#partition-id
        //= type=exception
        //# If the PartitionId is NOT provided by the user, it MUST be set to the 16 byte representation of a v4 UUID.
        partitionId ? stringToUtf8Bytes(partitionId) : randomBytes(64));
        /* Precondition: The branch key id must be a string */
        if (branchKeyId) {
            needs(typeof branchKeyId === 'string', 'The branch key id must be a string');
        }
        else {
            branchKeyId = undefined;
        }
        /* Precondition: The branch key id supplier must be a BranchKeyIdSupplier */
        if (branchKeyIdSupplier) {
            needs(isBranchKeyIdSupplier(branchKeyIdSupplier), 'The branch key id supplier must be a BranchKeyIdSupplier');
        }
        else {
            branchKeyIdSupplier = undefined;
        }
        /* Precondition: The keystore must be a BranchKeyStore */
        needs(isIBranchKeyStoreNode(keyStore), 'The keystore must be a BranchKeyStore');
        readOnlyProperty(this, '_logicalKeyStoreName', 
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#logical-key-store-name
        //# Logical Key Store Name MUST be converted to UTF8 Bytes to be used in
        //# the cache identifiers.
        stringToUtf8Bytes(keyStore.getKeyStoreInfo().logicalKeyStoreName));
        /* Precondition: The cache limit TTL must be a number */
        needs(typeof cacheLimitTtl === 'number', 'The cache limit TTL must be a number');
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#cache-limit-ttl
        //# The maximum amount of time in seconds that an entry within the cache may be used before it MUST be evicted.
        //# The client MUST set a time-to-live (TTL) for [branch key materials](../structures.md#branch-key-materials) in the underlying cache.
        //# This value MUST be greater than zero.
        /* Precondition: Cache limit TTL must be non-negative and less than or equal to (Number.MAX_SAFE_INTEGER / 1000) seconds */
        // In the MPL, TTL can be a non-negative signed 64-bit integer.
        // However, JavaScript numbers cannot safely represent integers beyond
        // Number.MAX_SAFE_INTEGER. Thus, we will cap TTL in seconds such that TTL
        // in ms is <= Number.MAX_SAFE_INTEGER. TTL could be a BigInt type but this
        // would require casting back to a number in order to configure the CMC,
        // which leads to a lossy conversion
        needs(0 <= cacheLimitTtl && cacheLimitTtl * 1000 <= Number.MAX_SAFE_INTEGER, 'Cache limit TTL must be non-negative and less than or equal to (Number.MAX_SAFE_INTEGER / 1000) seconds');
        /* Precondition: Must provide a branch key identifier or supplier */
        needs(branchKeyId || branchKeyIdSupplier, 'Must provide a branch key identifier or supplier');
        readOnlyProperty(this, 'keyStore', Object.freeze(keyStore));
        /* Postcondition: The keystore object is frozen */
        // convert seconds to milliseconds
        readOnlyProperty(this, 'cacheLimitTtl', cacheLimitTtl * 1000);
        readOnlyProperty(this, 'branchKeyId', branchKeyId);
        readOnlyProperty(this, 'branchKeyIdSupplier', branchKeyIdSupplier
            ? Object.freeze(branchKeyIdSupplier)
            : branchKeyIdSupplier);
        /* Postcondition: Provided branch key supplier must be frozen */
        if (cache) {
            needs(!maxCacheSize, 'Max cache size not supported when passing a cache.');
        }
        else {
            /* Precondition: The max cache size must be a number */
            needs(
            // Order is important, 0 is a number but also false.
            typeof maxCacheSize === 'number' || !maxCacheSize, 'The max cache size must be a number');
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#initialization
            //# If no max cache size is provided, the cryptographic materials cache MUST be configured to a
            //# max cache size of 1000.
            maxCacheSize = maxCacheSize === 0 || maxCacheSize ? maxCacheSize : 1000;
            /* Precondition: Max cache size must be non-negative and less than or equal Number.MAX_SAFE_INTEGER */
            needs(0 <= maxCacheSize && maxCacheSize <= Number.MAX_SAFE_INTEGER, 'Max cache size must be non-negative and less than or equal Number.MAX_SAFE_INTEGER');
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#initialization
            //# On initialization the Hierarchical Keyring MUST initialize a [cryptographic-materials-cache](../local-cryptographic-materials-cache.md) with the configured cache limit TTL and the max cache size.
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#initialization
            //# If the Hierarchical Keyring does NOT get a `Shared` cache on initialization,
            //# it MUST initialize a [cryptographic-materials-cache](../local-cryptographic-materials-cache.md)
            //# with the user provided cache limit TTL and the entry capacity.
            cache = getLocalCryptographicMaterialsCache(maxCacheSize);
        }
        readOnlyProperty(this, 'maxCacheSize', maxCacheSize);
        readOnlyProperty(this, '_cmc', cache);
        Object.freeze(this);
        /* Postcondition: The HKR object must be frozen */
    }
    async _onEncrypt(
    //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
    //= type=implication
    //# OnEncrypt MUST take [encryption materials](../structures.md#encryption-materials) as input.
    encryptionMaterial) {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# The `branchKeyId` used in this operation is either the configured branchKeyId, if supplied, or the result of the `branchKeySupplier`'s
        //# `getBranchKeyId` operation, using the encryption material's encryption context as input.
        const branchKeyId = getBranchKeyId(this, encryptionMaterial);
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# The hierarchical keyring MUST use the formulas specified in [Appendix A](#appendix-a-cache-entry-identifier-formulas)
        //# to compute the [cache entry identifier](../cryptographic-materials-cache.md#cache-identifier).
        const cacheEntryId = getCacheEntryId(this._logicalKeyStoreName, this._partition, branchKeyId);
        const branchKeyMaterials = await getBranchKeyMaterials(this, this._cmc, branchKeyId, cacheEntryId);
        // get a pdk (generate it if not already set)
        const pdk = getPlaintextDataKey(encryptionMaterial);
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# If the keyring is unable to wrap a plaintext data key, OnEncrypt MUST fail
        //# and MUST NOT modify the [decryption materials](structures.md#decryption-materials).
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# - MUST wrap a data key with the branch key materials according to the [branch key wrapping](#branch-key-wrapping) section.
        const edk = wrapPlaintextDataKey(pdk, branchKeyMaterials, encryptionMaterial);
        // return the modified encryption material with the new edk and newly
        // generated pdk (if applicable)
        return modifyEncryptionMaterial(encryptionMaterial, pdk, edk, branchKeyId);
    }
    async onDecrypt(
    //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
    //= type=implication
    //# OnDecrypt MUST take [decryption materials](../structures.md#decryption-materials) and a list of [encrypted data keys](../structures.md#encrypted-data-keys) as input.
    material, encryptedDataKeys) {
        needs(isDecryptionMaterial(material), 'Unsupported material type.');
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
        //# If the decryption materials already contain a `PlainTextDataKey`, OnDecrypt MUST fail.
        /* Precondition: If the decryption materials already contain a PlainTextDataKey, OnDecrypt MUST fail */
        needs(!material.hasUnencryptedDataKey, 'Decryption materials already contain a plaintext data key');
        needs(encryptedDataKeys.every((edk) => edk instanceof EncryptedDataKey), 'Unsupported EncryptedDataKey type');
        const _material = await this._onDecrypt(material, encryptedDataKeys);
        needs(material === _material, 'New DecryptionMaterial instances can not be created.');
        return material;
    }
    cacheEntryHasExceededLimits({ now }) {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# There MUST be a check (cacheEntryWithinLimits) to make sure that for the cache entry found, who's TTL has NOT expired,
        //# `time.now() - cacheEntryCreationTime <= ttlSeconds` is true and
        //# valid for TTL of the Hierarchical Keyring getting the cache entry.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
        //# There MUST be a check (cacheEntryWithinLimits) to make sure that for the cache entry found, who's TTL has NOT expired,
        //# `time.now() - cacheEntryCreationTime <= ttlSeconds` is true and
        //# valid for TTL of the Hierarchical Keyring getting the cache entry.
        const age = Date.now() - now;
        return age > this.cacheLimitTtl;
    }
    async _onDecrypt(decryptionMaterial, encryptedDataKeyObjs) {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
        //# The `branchKeyId` used in this operation is either the configured branchKeyId, if supplied, or the result of the `branchKeySupplier`'s
        //# `getBranchKeyId` operation, using the decryption material's encryption context as input.
        const branchKeyId = getBranchKeyId(this, decryptionMaterial);
        // filter out edk objects that don't match this keyring's configuration
        const filteredEdkObjs = encryptedDataKeyObjs.filter((edkObj) => filterEdk(branchKeyId, edkObj));
        /* Precondition: There must be an encrypted data key that matches this keyring configuration */
        needs(filteredEdkObjs.length > 0, "There must be an encrypted data key that matches this keyring's configuration");
        const errors = [];
        for (const { encryptedDataKey: ciphertext } of filteredEdkObjs) {
            let udk = undefined;
            try {
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
                //# - Deserialize the UUID string representation of the `version` from the [encrypted data key](../structures.md#encrypted-data-key) [ciphertext](#ciphertext).
                // get the branch key version (as compressed bytes) from the
                // destructured ciphertext of the edk
                const { branchKeyVersionAsBytesCompressed } = destructureCiphertext(ciphertext, decryptionMaterial.suite);
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
                //# - The deserialized UUID string representation of the `version`
                // uncompress the branch key version into regular utf8 bytes
                const branchKeyVersionAsBytes = stringToUtf8Bytes(decompressBytesToUuidv4(branchKeyVersionAsBytesCompressed));
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
                //# The hierarchical keyring MUST use the OnDecrypt formula specified in [Appendix A](#decryption-materials)
                //# in order to compute the [cache entry identifier](cryptographic-materials-cache.md#cache-identifier).
                const cacheEntryId = getCacheEntryId(this._logicalKeyStoreName, this._partition, 
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
                //# OnDecrypt MUST calculate the following values:
                branchKeyId, branchKeyVersionAsBytes);
                // get the string representation of the branch key version
                const branchKeyVersionAsString = decompressBytesToUuidv4(branchKeyVersionAsBytesCompressed);
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
                //# To decrypt each encrypted data key in the filtered set, the hierarchical keyring MUST attempt
                //# to find the corresponding [branch key materials](../structures.md#branch-key-materials)
                //# from the underlying [cryptographic materials cache](../local-cryptographic-materials-cache.md).
                const branchKeyMaterials = await getBranchKeyMaterials(this, this._cmc, branchKeyId, cacheEntryId, branchKeyVersionAsString);
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
                //# - MUST unwrap the encrypted data key with the branch key materials according to the [branch key unwrapping](#branch-key-unwrapping) section.
                udk = unwrapEncryptedDataKey(ciphertext, branchKeyMaterials, decryptionMaterial);
            }
            catch (e) {
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
                //# For each encrypted data key in the filtered set, one at a time, OnDecrypt MUST attempt to decrypt the encrypted data key.
                //# If this attempt results in an error, then these errors MUST be collected.
                errors.push({ errPlus: e });
            }
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
            //# If a decryption succeeds, this keyring MUST
            //# add the resulting plaintext data key to the decryption materials and return the modified materials.
            if (udk) {
                return modifyDencryptionMaterial(decryptionMaterial, udk, branchKeyId);
            }
        }
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
        //# If OnDecrypt fails to successfully decrypt any [encrypted data key](../structures.md#encrypted-data-key),
        //# then it MUST yield an error that includes all the collected errors
        //# and MUST NOT modify the [decryption materials](structures.md#decryption-materials).
        throw new Error(errors.reduce((m, e, i) => `${m} Error #${i + 1} \n ${e.errPlus.stack} \n`, 'Unable to decrypt data key'));
    }
}
immutableClass(KmsHierarchicalKeyRingNode);
// The JS version has not been released with a Storm Tracking CMC
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#initialization
//= type=exception
//# If the cache to initialize is a [Storm Tracking Cryptographic Materials Cache](../storm-tracking-cryptographic-materials-cache.md#overview)
//# then the [Grace Period](../storm-tracking-cryptographic-materials-cache.md#grace-period) MUST be less than the [cache limit TTL](#cache-limit-ttl).
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#initialization
//= type=exception
//# If no `cache` is provided, a `DefaultCache` MUST be configured with entry capacity of 1000.
// These are not something we can enforce
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#logical-key-store-name
//= type=exception
//# > Note: Users MUST NEVER have two different physical Key Stores with the same Logical Key Store Name.
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#shared-cache-considerations
//= type=exception
//# Any keyring that has access to the `Shared` cache MAY be able to use materials
//# that it MAY or MAY NOT have direct access to.
//#
//# Users MUST make sure that all of Partition ID, Logical Key Store Name of the Key Store for the Hierarchical Keyring
//# and Branch Key ID are set to be the same for two Hierarchical Keyrings if and only they want the keyrings to share
//# cache entries.
//# sourceMappingURL=data:application/json;base64,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