/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.shiro.spring.web;

import org.apache.shiro.web.filter.InvalidRequestFilter;
import org.apache.shiro.web.filter.mgt.DefaultFilterChainManager;
import org.apache.shiro.web.filter.mgt.NamedFilterList;
import org.apache.shiro.web.filter.mgt.PathMatchingFilterChainResolver;
import org.apache.shiro.web.servlet.AbstractShiroFilter;
import org.junit.jupiter.api.Test;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import jakarta.servlet.Filter;
import jakarta.servlet.FilterChain;
import jakarta.servlet.FilterConfig;
import jakarta.servlet.ServletContext;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;

import static org.assertj.core.api.Assertions.assertThat;
import static org.easymock.EasyMock.createNiceMock;
import static org.easymock.EasyMock.expect;
import static org.easymock.EasyMock.replay;
import static org.easymock.EasyMock.verify;


/**
 * Unit tests for the {@link ShiroFilterFactoryBean} implementation.
 *
 * @since 1.0
 */
//@RunWith(SpringJUnit4ClassRunner.class)
//@ContextConfiguration(locations = {"/org/apache/shiro/spring/web/ShiroFilterFactoryBeanTest.xml"})
public class ShiroFilterFactoryBeanTest {

    @Test
    void testFilterDefinition() {

        ClassPathXmlApplicationContext context =
                new ClassPathXmlApplicationContext("org/apache/shiro/spring/web/ShiroFilterFactoryBeanTest.xml");

        AbstractShiroFilter shiroFilter = (AbstractShiroFilter) context.getBean("shiroFilter");

        PathMatchingFilterChainResolver resolver = (PathMatchingFilterChainResolver) shiroFilter.getFilterChainResolver();
        DefaultFilterChainManager fcManager = (DefaultFilterChainManager) resolver.getFilterChainManager();
        NamedFilterList chain = fcManager.getChain("/test");
        assertThat(chain).isNotNull();
        assertThat(chain).hasSize(2);
        Filter[] filters = new Filter[chain.size()];
        filters = chain.toArray(filters);
        // global filter
        assertThat(filters[0] instanceof InvalidRequestFilter).isTrue();
        assertThat(filters[1] instanceof DummyFilter).isTrue();
    }

    /**
     * Verifies fix for <a href="https://issues.apache.org/jira/browse/SHIRO-167">SHIRO-167</a>
     *
     * @throws Exception if there is any unexpected error
     */
    @Test
    void testFilterDefinitionWithInit() throws Exception {

        ClassPathXmlApplicationContext context =
                new ClassPathXmlApplicationContext("org/apache/shiro/spring/web/ShiroFilterFactoryBeanTest.xml");

        AbstractShiroFilter shiroFilter = (AbstractShiroFilter) context.getBean("shiroFilter");

        FilterConfig mockFilterConfig = createNiceMock(FilterConfig.class);
        ServletContext mockServletContext = createNiceMock(ServletContext.class);
        expect(mockFilterConfig.getServletContext()).andReturn(mockServletContext).anyTimes();
        HttpServletRequest mockRequest = createNiceMock(HttpServletRequest.class);
        expect(mockRequest.getContextPath()).andReturn("/").anyTimes();
        expect(mockRequest.getRequestURI()).andReturn("/test").anyTimes();
        expect(mockRequest.getServletPath()).andReturn("/test").anyTimes();
        HttpServletResponse mockResponse = createNiceMock(HttpServletResponse.class);

        replay(mockFilterConfig);
        replay(mockServletContext);
        shiroFilter.init(mockFilterConfig);
        verify(mockServletContext);
        verify(mockFilterConfig);

        FilterChain filterChain = new FilterChain() {
            public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse)
                    throws IOException, ServletException {
                HttpServletRequest request = (HttpServletRequest) servletRequest;
                assertThat(request.getSession()).isNotNull();
                //this line asserts the fix for the user-reported issue:
                assertThat(request.getSession().getServletContext()).isNotNull();
            }
        };

        replay(mockRequest);
        replay(mockResponse);

        shiroFilter.doFilter(mockRequest, mockResponse, filterChain);

        verify(mockResponse);
        verify(mockRequest);
    }
}
