﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/gamelift/GameLiftRequest.h>
#include <aws/gamelift/GameLift_EXPORTS.h>
#include <aws/gamelift/model/RoutingStrategy.h>
#include <aws/gamelift/model/Tag.h>

#include <utility>

namespace Aws {
namespace GameLift {
namespace Model {

/**
 */
class CreateAliasRequest : public GameLiftRequest {
 public:
  AWS_GAMELIFT_API CreateAliasRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateAlias"; }

  AWS_GAMELIFT_API Aws::String SerializePayload() const override;

  AWS_GAMELIFT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A descriptive label that is associated with an alias. Alias names do not need
   * to be unique.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateAliasRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A human-readable description of the alias.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateAliasRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The routing configuration, including routing type and fleet target, for the
   * alias. </p>
   */
  inline const RoutingStrategy& GetRoutingStrategy() const { return m_routingStrategy; }
  inline bool RoutingStrategyHasBeenSet() const { return m_routingStrategyHasBeenSet; }
  template <typename RoutingStrategyT = RoutingStrategy>
  void SetRoutingStrategy(RoutingStrategyT&& value) {
    m_routingStrategyHasBeenSet = true;
    m_routingStrategy = std::forward<RoutingStrategyT>(value);
  }
  template <typename RoutingStrategyT = RoutingStrategy>
  CreateAliasRequest& WithRoutingStrategy(RoutingStrategyT&& value) {
    SetRoutingStrategy(std::forward<RoutingStrategyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of labels to assign to the new alias resource. Tags are
   * developer-defined key-value pairs. Tagging Amazon Web Services resources are
   * useful for resource management, access management and cost allocation. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging
   * Amazon Web Services Resources</a> in the <i>Amazon Web Services General
   * Reference</i>.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateAliasRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateAliasRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  RoutingStrategy m_routingStrategy;
  bool m_routingStrategyHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace GameLift
}  // namespace Aws
