﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/dynamodb/DynamoDB_EXPORTS.h>
#include <aws/dynamodb/model/AttributeValue.h>
#include <aws/dynamodb/model/ReturnValuesOnConditionCheckFailure.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace DynamoDB {
namespace Model {

/**
 * <p> A PartiQL batch statement request. </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/BatchStatementRequest">AWS
 * API Reference</a></p>
 */
class BatchStatementRequest {
 public:
  AWS_DYNAMODB_API BatchStatementRequest() = default;
  AWS_DYNAMODB_API BatchStatementRequest(Aws::Utils::Json::JsonView jsonValue);
  AWS_DYNAMODB_API BatchStatementRequest& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_DYNAMODB_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p> A valid PartiQL statement. </p>
   */
  inline const Aws::String& GetStatement() const { return m_statement; }
  inline bool StatementHasBeenSet() const { return m_statementHasBeenSet; }
  template <typename StatementT = Aws::String>
  void SetStatement(StatementT&& value) {
    m_statementHasBeenSet = true;
    m_statement = std::forward<StatementT>(value);
  }
  template <typename StatementT = Aws::String>
  BatchStatementRequest& WithStatement(StatementT&& value) {
    SetStatement(std::forward<StatementT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The parameters associated with a PartiQL statement in the batch request.
   * </p>
   */
  inline const Aws::Vector<AttributeValue>& GetParameters() const { return m_parameters; }
  inline bool ParametersHasBeenSet() const { return m_parametersHasBeenSet; }
  template <typename ParametersT = Aws::Vector<AttributeValue>>
  void SetParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters = std::forward<ParametersT>(value);
  }
  template <typename ParametersT = Aws::Vector<AttributeValue>>
  BatchStatementRequest& WithParameters(ParametersT&& value) {
    SetParameters(std::forward<ParametersT>(value));
    return *this;
  }
  template <typename ParametersT = AttributeValue>
  BatchStatementRequest& AddParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters.emplace_back(std::forward<ParametersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The read consistency of the PartiQL batch request. </p>
   */
  inline bool GetConsistentRead() const { return m_consistentRead; }
  inline bool ConsistentReadHasBeenSet() const { return m_consistentReadHasBeenSet; }
  inline void SetConsistentRead(bool value) {
    m_consistentReadHasBeenSet = true;
    m_consistentRead = value;
  }
  inline BatchStatementRequest& WithConsistentRead(bool value) {
    SetConsistentRead(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional parameter that returns the item attributes for a PartiQL batch
   * request operation that failed a condition check.</p> <p>There is no additional
   * cost associated with requesting a return value aside from the small network and
   * processing overhead of receiving a larger response. No read capacity units are
   * consumed.</p>
   */
  inline ReturnValuesOnConditionCheckFailure GetReturnValuesOnConditionCheckFailure() const {
    return m_returnValuesOnConditionCheckFailure;
  }
  inline bool ReturnValuesOnConditionCheckFailureHasBeenSet() const { return m_returnValuesOnConditionCheckFailureHasBeenSet; }
  inline void SetReturnValuesOnConditionCheckFailure(ReturnValuesOnConditionCheckFailure value) {
    m_returnValuesOnConditionCheckFailureHasBeenSet = true;
    m_returnValuesOnConditionCheckFailure = value;
  }
  inline BatchStatementRequest& WithReturnValuesOnConditionCheckFailure(ReturnValuesOnConditionCheckFailure value) {
    SetReturnValuesOnConditionCheckFailure(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_statement;

  Aws::Vector<AttributeValue> m_parameters;

  bool m_consistentRead{false};

  ReturnValuesOnConditionCheckFailure m_returnValuesOnConditionCheckFailure{ReturnValuesOnConditionCheckFailure::NOT_SET};
  bool m_statementHasBeenSet = false;
  bool m_parametersHasBeenSet = false;
  bool m_consistentReadHasBeenSet = false;
  bool m_returnValuesOnConditionCheckFailureHasBeenSet = false;
};

}  // namespace Model
}  // namespace DynamoDB
}  // namespace Aws
