"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BasicAuthRoutes = void 0;
var _configSchema = require("@osd/config-schema");
var _security_cookie = require("../../../session/security_cookie");
var _common = require("../../../../common");
var _tenant_resolver = require("../../../multitenancy/tenant_resolver");
var _encode_uri_query = require("../../../../../../src/plugins/opensearch_dashboards_utils/common/url/encode_uri_query");
var _next_url = require("../../../utils/next_url");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

class BasicAuthRoutes {
  constructor(router, config, sessionStorageFactory, securityClient, coreSetup) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.securityClient = securityClient;
    this.coreSetup = coreSetup;
  }
  setupRoutes() {
    // bootstrap an empty page so that browser app can render the login page
    // using client side routing.
    this.coreSetup.http.resources.register({
      path: _common.LOGIN_PAGE_URI,
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: nexturl => {
              return (0, _next_url.validateNextUrl)(nexturl, this.coreSetup.http.basePath.serverBasePath);
            }
          }))
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return response.renderAnonymousCoreApp({
        headers: {
          'set-cookie': clearOldVersionCookie
        }
      });
    });

    // login using username and password
    this.router.post({
      path: _common.API_AUTH_LOGIN,
      validate: {
        body: _configSchema.schema.object({
          username: _configSchema.schema.string(),
          password: _configSchema.schema.string()
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$multiten;
      const forbiddenUsernames = this.config.auth.forbidden_usernames;
      if (forbiddenUsernames.indexOf(request.body.username) > -1) {
        context.security_plugin.logger.error(`Denied login for forbidden username ${request.body.username}`);
        return response.badRequest({
          // Cannot login using forbidden user name.
          body: 'Invalid username or password'
        });
      }
      let user;
      try {
        user = await this.securityClient.authenticate(request, {
          username: request.body.username,
          password: request.body.password
        });
      } catch (error) {
        context.security_plugin.logger.error(`Failed authentication: ${error}`);
        return response.unauthorized({
          headers: {
            'www-authenticate': error.message
          }
        });
      }
      this.sessionStorageFactory.asScoped(request).clear();
      const encodedCredentials = Buffer.from(`${request.body.username}:${request.body.password}`).toString('base64');
      const sessionStorage = {
        username: user.username,
        credentials: {
          authHeaderValue: `Basic ${encodedCredentials}`
        },
        authType: _common.AuthType.BASIC,
        isAnonymousAuth: false,
        expiryTime: Date.now() + this.config.session.ttl
      };
      if (user.multitenancy_enabled) {
        const selectTenant = (0, _tenant_resolver.resolveTenant)({
          request,
          username: user.username,
          roles: user.roles,
          availableTenants: user.tenants,
          config: this.config,
          cookie: sessionStorage,
          multitenancyEnabled: user.multitenancy_enabled,
          privateTenantEnabled: user.private_tenant_enabled,
          defaultTenant: user.default_tenant
        });
        // const selectTenant = user.default_tenant;
        sessionStorage.tenant = selectTenant;
      }
      this.sessionStorageFactory.asScoped(request).set(sessionStorage);
      return response.ok({
        body: {
          username: user.username,
          tenants: user.tenants,
          roles: user.roles,
          backendroles: user.backendRoles,
          selectedTenants: (_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled ? sessionStorage.tenant : undefined
        }
      });
    });

    // logout
    this.router.post({
      path: _common.API_AUTH_LOGOUT,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.ok({
        body: {}
      });
    });

    // anonymous auth
    this.router.get({
      path: _common.ANONYMOUS_AUTH_LOGIN,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      if (this.config.auth.anonymous_auth_enabled) {
        let user;
        // If the request contains no redirect path, simply redirect to basepath.
        let redirectUrl = this.coreSetup.http.basePath.serverBasePath ? this.coreSetup.http.basePath.serverBasePath : '/';
        const requestQuery = request.url.searchParams;
        const nextUrl = requestQuery === null || requestQuery === void 0 ? void 0 : requestQuery.get('nextUrl');
        if (nextUrl) {
          redirectUrl = nextUrl;
        }
        context.security_plugin.logger.info('The Redirect Path is ' + redirectUrl);
        try {
          user = await this.securityClient.authenticateWithHeaders(request, {});
        } catch (error) {
          context.security_plugin.logger.error(`Failed authentication: ${error}. Redirecting to Login Page`);
          return response.redirected({
            headers: {
              location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}${nextUrl ? '?nextUrl=' + (0, _encode_uri_query.encodeUriQuery)(redirectUrl) : ''}`
            }
          });
        }
        this.sessionStorageFactory.asScoped(request).clear();
        const sessionStorage = {
          username: user.username,
          authType: _common.AuthType.BASIC,
          isAnonymousAuth: true,
          expiryTime: Date.now() + this.config.session.ttl
        };
        if (user.multitenancy_enabled) {
          const selectTenant = (0, _tenant_resolver.resolveTenant)({
            request,
            username: user.username,
            roles: user.roles,
            availableTenants: user.tenants,
            config: this.config,
            cookie: sessionStorage,
            multitenancyEnabled: user.multitenancy_enabled,
            privateTenantEnabled: user.private_tenant_enabled,
            defaultTenant: user.default_tenant
          });
          sessionStorage.tenant = selectTenant;
        }
        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        return response.redirected({
          headers: {
            location: `${redirectUrl}`
          }
        });
      } else {
        context.security_plugin.logger.error('Anonymous auth is disabled. Redirecting to Login Page');
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}`
          }
        });
      }
    });
  }
}
exports.BasicAuthRoutes = BasicAuthRoutes;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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