/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.LongMath;
import com.google.common.math.MathPreconditions;
import com.google.common.primitives.Ints;
import com.google.errorprone.annotations.InlineMe;
import java.math.RoundingMode;

@GwtCompatible
public final class IntMath {
    @VisibleForTesting
    static final int MAX_SIGNED_POWER_OF_TWO = 0x40000000;
    @VisibleForTesting
    static final int MAX_POWER_OF_SQRT2_UNSIGNED = -1257966797;
    @VisibleForTesting
    static final byte[] maxLog10ForLeadingZeros = new byte[]{9, 9, 9, 8, 8, 8, 7, 7, 7, 6, 6, 6, 6, 5, 5, 5, 4, 4, 4, 3, 3, 3, 3, 2, 2, 2, 1, 1, 1, 0, 0, 0, 0};
    @VisibleForTesting
    static final int[] powersOf10 = new int[]{1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000, 1000000000};
    @VisibleForTesting
    static final int[] halfPowersOf10 = new int[]{3, 31, 316, 3162, 31622, 316227, 3162277, 31622776, 316227766, Integer.MAX_VALUE};
    @VisibleForTesting
    static final int FLOOR_SQRT_MAX_INT = 46340;
    private static final int[] factorials = new int[]{1, 1, 2, 6, 24, 120, 720, 5040, 40320, 362880, 3628800, 39916800, 479001600};
    @VisibleForTesting
    static final int[] biggestBinomials = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, 65536, 2345, 477, 193, 110, 75, 58, 49, 43, 39, 37, 35, 34, 34, 33};

    public static int ceilingPowerOfTwo(int x15) {
        MathPreconditions.checkPositive("x", x15);
        if (x15 > 0x40000000) {
            throw new ArithmeticException("ceilingPowerOfTwo(" + x15 + ") not representable as an int");
        }
        return 1 << -Integer.numberOfLeadingZeros(x15 - 1);
    }

    public static int floorPowerOfTwo(int x15) {
        MathPreconditions.checkPositive("x", x15);
        return Integer.highestOneBit(x15);
    }

    public static boolean isPowerOfTwo(int x15) {
        return x15 > 0 & (x15 & x15 - 1) == 0;
    }

    @VisibleForTesting
    static int lessThanBranchFree(int x15, int y15) {
        return ~(~(x15 - y15)) >>> 31;
    }

    public static int log2(int x15, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x15);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(IntMath.isPowerOfTwo(x15));
            }
            case DOWN: 
            case FLOOR: {
                return 31 - Integer.numberOfLeadingZeros(x15);
            }
            case UP: 
            case CEILING: {
                return 32 - Integer.numberOfLeadingZeros(x15 - 1);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int leadingZeros = Integer.numberOfLeadingZeros(x15);
                int cmp = -1257966797 >>> leadingZeros;
                int logFloor = 31 - leadingZeros;
                return logFloor + IntMath.lessThanBranchFree(cmp, x15);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static int log10(int x15, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x15);
        int logFloor = IntMath.log10Floor(x15);
        int floorPow = powersOf10[logFloor];
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(x15 == floorPow);
            }
            case DOWN: 
            case FLOOR: {
                return logFloor;
            }
            case UP: 
            case CEILING: {
                return logFloor + IntMath.lessThanBranchFree(floorPow, x15);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                return logFloor + IntMath.lessThanBranchFree(halfPowersOf10[logFloor], x15);
            }
        }
        throw new AssertionError();
    }

    private static int log10Floor(int x15) {
        byte y15 = maxLog10ForLeadingZeros[Integer.numberOfLeadingZeros(x15)];
        return y15 - IntMath.lessThanBranchFree(x15, powersOf10[y15]);
    }

    @GwtIncompatible
    public static int pow(int b15, int k15) {
        MathPreconditions.checkNonNegative("exponent", k15);
        switch (b15) {
            case 0: {
                return k15 == 0 ? 1 : 0;
            }
            case 1: {
                return 1;
            }
            case -1: {
                return (k15 & 1) == 0 ? 1 : -1;
            }
            case 2: {
                return k15 < 32 ? 1 << k15 : 0;
            }
            case -2: {
                if (k15 < 32) {
                    return (k15 & 1) == 0 ? 1 << k15 : -(1 << k15);
                }
                return 0;
            }
        }
        int accum = 1;
        while (true) {
            switch (k15) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return b15 * accum;
                }
            }
            accum *= (k15 & 1) == 0 ? 1 : b15;
            b15 *= b15;
            k15 >>= 1;
        }
    }

    @GwtIncompatible
    public static int sqrt(int x15, RoundingMode mode) {
        MathPreconditions.checkNonNegative("x", x15);
        int sqrtFloor = IntMath.sqrtFloor(x15);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(sqrtFloor * sqrtFloor == x15);
            }
            case DOWN: 
            case FLOOR: {
                return sqrtFloor;
            }
            case UP: 
            case CEILING: {
                return sqrtFloor + IntMath.lessThanBranchFree(sqrtFloor * sqrtFloor, x15);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int halfSquare = sqrtFloor * sqrtFloor + sqrtFloor;
                return sqrtFloor + IntMath.lessThanBranchFree(halfSquare, x15);
            }
        }
        throw new AssertionError();
    }

    private static int sqrtFloor(int x15) {
        return (int)Math.sqrt(x15);
    }

    public static int divide(int p15, int q15, RoundingMode mode) {
        boolean increment;
        Preconditions.checkNotNull(mode);
        if (q15 == 0) {
            throw new ArithmeticException("/ by zero");
        }
        int div = p15 / q15;
        int rem = p15 - q15 * div;
        if (rem == 0) {
            return div;
        }
        int signum = 1 | (p15 ^ q15) >> 31;
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(rem == 0);
            }
            case DOWN: {
                increment = false;
                break;
            }
            case UP: {
                increment = true;
                break;
            }
            case CEILING: {
                increment = signum > 0;
                break;
            }
            case FLOOR: {
                increment = signum < 0;
                break;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int absRem = Math.abs(rem);
                int cmpRemToHalfDivisor = absRem - (Math.abs(q15) - absRem);
                if (cmpRemToHalfDivisor == 0) {
                    increment = mode == RoundingMode.HALF_UP || mode == RoundingMode.HALF_EVEN & (div & 1) != 0;
                    break;
                }
                increment = cmpRemToHalfDivisor > 0;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return increment ? div + signum : div;
    }

    public static int mod(int x15, int m15) {
        if (m15 <= 0) {
            throw new ArithmeticException("Modulus " + m15 + " must be > 0");
        }
        return Math.floorMod(x15, m15);
    }

    public static int gcd(int a15, int b15) {
        MathPreconditions.checkNonNegative("a", a15);
        MathPreconditions.checkNonNegative("b", b15);
        if (a15 == 0) {
            return b15;
        }
        if (b15 == 0) {
            return a15;
        }
        int aTwos = Integer.numberOfTrailingZeros(a15);
        a15 >>= aTwos;
        int bTwos = Integer.numberOfTrailingZeros(b15);
        b15 >>= bTwos;
        while (a15 != b15) {
            int delta = a15 - b15;
            int minDeltaOrZero = delta & delta >> 31;
            a15 = delta - minDeltaOrZero - minDeltaOrZero;
            b15 += minDeltaOrZero;
            a15 >>= Integer.numberOfTrailingZeros(a15);
        }
        return a15 << Math.min(aTwos, bTwos);
    }

    @InlineMe(replacement="Math.addExact(a, b)")
    public static int checkedAdd(int a15, int b15) {
        return Math.addExact(a15, b15);
    }

    @InlineMe(replacement="Math.subtractExact(a, b)")
    public static int checkedSubtract(int a15, int b15) {
        return Math.subtractExact(a15, b15);
    }

    @InlineMe(replacement="Math.multiplyExact(a, b)")
    public static int checkedMultiply(int a15, int b15) {
        return Math.multiplyExact(a15, b15);
    }

    public static int checkedPow(int b15, int k15) {
        MathPreconditions.checkNonNegative("exponent", k15);
        switch (b15) {
            case 0: {
                return k15 == 0 ? 1 : 0;
            }
            case 1: {
                return 1;
            }
            case -1: {
                return (k15 & 1) == 0 ? 1 : -1;
            }
            case 2: {
                MathPreconditions.checkNoOverflow(k15 < 31, "checkedPow", b15, k15);
                return 1 << k15;
            }
            case -2: {
                MathPreconditions.checkNoOverflow(k15 < 32, "checkedPow", b15, k15);
                return (k15 & 1) == 0 ? 1 << k15 : -1 << k15;
            }
        }
        int accum = 1;
        while (true) {
            switch (k15) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return Math.multiplyExact(accum, b15);
                }
            }
            if ((k15 & 1) != 0) {
                accum = Math.multiplyExact(accum, b15);
            }
            if ((k15 >>= 1) <= 0) continue;
            MathPreconditions.checkNoOverflow(-46340 <= b15 & b15 <= 46340, "checkedPow", b15, k15);
            b15 *= b15;
        }
    }

    public static int saturatedAdd(int a15, int b15) {
        return Ints.saturatedCast((long)a15 + (long)b15);
    }

    public static int saturatedSubtract(int a15, int b15) {
        return Ints.saturatedCast((long)a15 - (long)b15);
    }

    public static int saturatedMultiply(int a15, int b15) {
        return Ints.saturatedCast((long)a15 * (long)b15);
    }

    public static int saturatedPow(int b15, int k15) {
        MathPreconditions.checkNonNegative("exponent", k15);
        switch (b15) {
            case 0: {
                return k15 == 0 ? 1 : 0;
            }
            case 1: {
                return 1;
            }
            case -1: {
                return (k15 & 1) == 0 ? 1 : -1;
            }
            case 2: {
                if (k15 >= 31) {
                    return Integer.MAX_VALUE;
                }
                return 1 << k15;
            }
            case -2: {
                if (k15 >= 32) {
                    return Integer.MAX_VALUE + (k15 & 1);
                }
                return (k15 & 1) == 0 ? 1 << k15 : -1 << k15;
            }
        }
        int accum = 1;
        int limit = Integer.MAX_VALUE + (b15 >>> 31 & (k15 & 1));
        while (true) {
            switch (k15) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return IntMath.saturatedMultiply(accum, b15);
                }
            }
            if ((k15 & 1) != 0) {
                accum = IntMath.saturatedMultiply(accum, b15);
            }
            if ((k15 >>= 1) <= 0) continue;
            if (-46340 > b15 | b15 > 46340) {
                return limit;
            }
            b15 *= b15;
        }
    }

    public static int factorial(int n15) {
        MathPreconditions.checkNonNegative("n", n15);
        return n15 < factorials.length ? factorials[n15] : Integer.MAX_VALUE;
    }

    public static int binomial(int n15, int k15) {
        MathPreconditions.checkNonNegative("n", n15);
        MathPreconditions.checkNonNegative("k", k15);
        Preconditions.checkArgument(k15 <= n15, "k (%s) > n (%s)", k15, n15);
        if (k15 > n15 >> 1) {
            k15 = n15 - k15;
        }
        if (k15 >= biggestBinomials.length || n15 > biggestBinomials[k15]) {
            return Integer.MAX_VALUE;
        }
        switch (k15) {
            case 0: {
                return 1;
            }
            case 1: {
                return n15;
            }
        }
        long result = 1L;
        for (int i15 = 0; i15 < k15; ++i15) {
            result *= (long)(n15 - i15);
            result /= (long)(i15 + 1);
        }
        return (int)result;
    }

    public static int mean(int x15, int y15) {
        return (x15 & y15) + ((x15 ^ y15) >> 1);
    }

    @GwtIncompatible
    public static boolean isPrime(int n15) {
        return LongMath.isPrime(n15);
    }

    public static int saturatedAbs(int x15) {
        return x15 == Integer.MIN_VALUE ? Integer.MAX_VALUE : Math.abs(x15);
    }

    private IntMath() {
    }
}

