/*
 * Decompiled with CFR 0.152.
 */
package generic.util.image;

import generic.theme.GThemeDefaults;
import ghidra.util.MathUtilities;
import ghidra.util.Msg;
import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.awt.image.LookupOp;
import java.awt.image.LookupTable;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.util.Objects;
import javax.imageio.ImageIO;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import resources.ResourceManager;

public class ImageUtils {
    private static final float DEFAULT_TRANSPARENCY_ALPHA = 0.4f;
    private static JComponent mediaTrackerComponent;

    private ImageUtils() {
    }

    public static Image createImage(Component c) {
        Rectangle bounds = c.getBounds();
        int w = Math.max(bounds.width, 1);
        int h = Math.max(bounds.height, 1);
        BufferedImage bufferedImage = new BufferedImage(w, h, 2);
        Graphics g = bufferedImage.getGraphics();
        c.paint(g);
        g.dispose();
        return bufferedImage;
    }

    public static Image padImage(Image i, Color c, int top, int left, int right, int bottom) {
        int width = i.getWidth(null) + left + right;
        int height = i.getHeight(null) + top + bottom;
        BufferedImage newImage = new BufferedImage(width, height, 2);
        Graphics g = newImage.getGraphics();
        g.setColor(c);
        g.fillRect(0, 0, width, height);
        g.drawImage(i, left, top, null);
        g.dispose();
        ImageUtils.waitForImage(null, newImage);
        return newImage;
    }

    public static Image padImage(Image i, Color c, Padding padding) {
        return ImageUtils.padImage(i, c, padding.top, padding.left, padding.right, padding.bottom);
    }

    public static Image crop(Image i, Rectangle bounds) {
        BufferedImage newImage = new BufferedImage(bounds.width, bounds.height, 2);
        Graphics g = newImage.getGraphics();
        g.drawImage(i, -bounds.x, -bounds.y, null);
        ImageUtils.waitForImage(null, newImage);
        return newImage;
    }

    public static BufferedImage createEmptyImage(int width, int height) {
        BufferedImage newImage = new BufferedImage(width, height, 2);
        Graphics g = newImage.getGraphics();
        g.setColor(GThemeDefaults.Colors.BACKGROUND);
        g.fillRect(0, 0, width, height);
        return newImage;
    }

    public static Image placeImagesSideBySide(Image left, Image right) {
        int leftHeight = left.getHeight(null);
        int leftWidth = left.getWidth(null);
        int rightHeight = right.getHeight(null);
        int rightWidth = right.getWidth(null);
        int width = leftWidth + rightWidth;
        int height = Math.max(leftHeight, rightHeight);
        BufferedImage newImage = ImageUtils.createEmptyImage(width, height);
        Graphics g = newImage.getGraphics();
        int y = 0;
        if (leftHeight < rightHeight) {
            y = (rightHeight - leftHeight) / 2;
        }
        g.drawImage(left, 0, y, null);
        y = 0;
        if (leftHeight > rightHeight) {
            y = (leftHeight - rightHeight) / 2;
        }
        g.drawImage(right, leftWidth, y, null);
        g.dispose();
        ImageUtils.waitForImage(null, newImage);
        return newImage;
    }

    public static RenderedImage toRenderedImage(Image image) {
        if (image instanceof RenderedImage) {
            return (RenderedImage)((Object)image);
        }
        return ImageUtils.getBufferedImage(image);
    }

    public static BufferedImage getBufferedImage(Image image) {
        if (image instanceof BufferedImage) {
            return (BufferedImage)image;
        }
        boolean success = ImageUtils.waitForImage("<unknown name>", image);
        if (!success) {
            return null;
        }
        int width = image.getWidth(null);
        int height = image.getHeight(null);
        BufferedImage bufferedImage = new BufferedImage(width, height, 2);
        Graphics graphics = bufferedImage.getGraphics();
        graphics.drawImage(image, 0, 0, null);
        graphics.dispose();
        return bufferedImage;
    }

    public static boolean waitForImage(String imageName, Image image) {
        if (image instanceof BufferedImage) {
            return true;
        }
        if (image.getWidth(null) > 0 && image.getHeight(null) > 0) {
            return true;
        }
        MediaTracker tracker = new MediaTracker(ImageUtils.getMediaTrackerComponent());
        int maxWaits = 20;
        int waitTime = 100;
        tracker.addImage(image, 0);
        for (int i = 0; i < maxWaits; ++i) {
            try {
                tracker.waitForID(0, waitTime);
                int width = image.getWidth(null);
                int height = image.getHeight(null);
                if (width < 0 || height < 0) continue;
                tracker.removeImage(image, 0);
                return true;
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        Msg.debug(ImageUtils.class, (Object)("Timed-out waiting for image to load after " + maxWaits * waitTime / 1000 + " seconds; " + imageName));
        tracker.removeImage(image, 0);
        return false;
    }

    public static void writeFile(Image i, File imageFile) throws IOException {
        ImageIO.write(ImageUtils.toRenderedImage(i), "png", imageFile);
    }

    public static void writeFile(RenderedImage i, File imageFile) throws IOException {
        ImageIO.write(i, "png", imageFile);
    }

    public static BufferedImage readFile(File imageFile) throws IOException {
        return ImageIO.read(imageFile);
    }

    public static void writeIconToPNG(Icon icon, String filename) throws IOException {
        BufferedImage buffi = new BufferedImage(icon.getIconWidth(), icon.getIconHeight(), 2);
        Graphics graphics = buffi.getGraphics();
        icon.paintIcon(null, graphics, 0, 0);
        ImageIO.write((RenderedImage)buffi, "png", new File(filename));
        graphics.dispose();
    }

    public static Icon makeTransparent(Icon icon) {
        return ImageUtils.makeTransparent(icon, 0.4f);
    }

    public static Icon makeTransparent(Icon icon, float alpha) {
        BufferedImage newImage = new BufferedImage(icon.getIconWidth(), icon.getIconHeight(), 2);
        Graphics2D g = newImage.createGraphics();
        g.setComposite(AlphaComposite.getInstance(3, alpha));
        icon.paintIcon(null, g, 0, 0);
        g.dispose();
        ImageIcon imageIcon = new ImageIcon(newImage);
        String iconName = ResourceManager.getIconName(icon);
        if (iconName != null) {
            imageIcon.setDescription(iconName);
        }
        return imageIcon;
    }

    public static Image createScaledImage(Image image, int width, int height, int hints) {
        BufferedImage scaledImage = new BufferedImage(width, height, 2);
        Graphics graphics = scaledImage.getGraphics();
        Graphics2D g2 = (Graphics2D)graphics;
        g2.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
        graphics.drawImage(image, 0, 0, width, height, null);
        graphics.dispose();
        return scaledImage;
    }

    public static Image createDisabledImage(Image image, final int brightnessPercent) {
        Objects.requireNonNull(image);
        BufferedImage srcImage = new BufferedImage(image.getWidth(null), image.getHeight(null), 2);
        Graphics graphics = srcImage.getGraphics();
        graphics.drawImage(image, 0, 0, null);
        graphics.dispose();
        BufferedImage destImage = new BufferedImage(image.getWidth(null), image.getHeight(null), 2);
        LookupTable table = new LookupTable(0, 4){

            @Override
            public int[] lookupPixel(int[] src, int[] dest) {
                return ImageUtils.filterRgbDisabledImage(src, dest, brightnessPercent);
            }
        };
        LookupOp lookupOp = new LookupOp(table, null);
        lookupOp.filter(srcImage, destImage);
        return destImage;
    }

    public static Image changeColor(Image image, Color oldColor, Color newColor) {
        Objects.requireNonNull(image);
        BufferedImage srcImage = new BufferedImage(image.getWidth(null), image.getHeight(null), 2);
        Graphics graphics = srcImage.getGraphics();
        graphics.drawImage(image, 0, 0, null);
        graphics.dispose();
        final int[] oldRgb = new int[]{oldColor.getRed(), oldColor.getGreen(), oldColor.getBlue()};
        final int[] newRgb = new int[]{newColor.getRed(), newColor.getGreen(), newColor.getBlue()};
        BufferedImage destImage = new BufferedImage(image.getWidth(null), image.getHeight(null), 2);
        LookupTable table = new LookupTable(0, 4){

            @Override
            public int[] lookupPixel(int[] src, int[] dest) {
                return ImageUtils.filterRgbChangeColor(src, dest, oldRgb, newRgb);
            }
        };
        LookupOp lookupOp = new LookupOp(table, null);
        lookupOp.filter(srcImage, destImage);
        return destImage;
    }

    private static synchronized JComponent getMediaTrackerComponent() {
        if (mediaTrackerComponent == null) {
            mediaTrackerComponent = new JComponent(){};
        }
        return mediaTrackerComponent;
    }

    private static int[] filterRgbDisabledImage(int[] rgbPixels, int[] destination, int brightnessPercent) {
        int r = (int)(0.3 * (double)(rgbPixels[0] & 0xFF));
        int g = (int)(0.59 * (double)(rgbPixels[1] & 0xFF));
        int b = (int)(0.11 * (double)(rgbPixels[2] & 0xFF));
        int gray = (r + g + b) / 3;
        gray = 255 - (255 - gray) * (100 - brightnessPercent) / 100;
        destination[0] = gray = MathUtilities.clamp((int)gray, (int)0, (int)255);
        destination[1] = gray;
        destination[2] = gray;
        destination[3] = rgbPixels[3];
        return destination;
    }

    private static int[] filterRgbChangeColor(int[] rgbPixels, int[] destination, int[] oldRgb, int[] newRgb) {
        int r = rgbPixels[0] & 0xFF;
        int g = rgbPixels[1] & 0xFF;
        int b = rgbPixels[2] & 0xFF;
        int oldR = oldRgb[0];
        int oldG = oldRgb[1];
        int oldB = oldRgb[2];
        if (r == oldR && g == oldG && b == oldB) {
            destination[0] = newRgb[0];
            destination[1] = newRgb[1];
            destination[2] = newRgb[2];
        } else {
            destination[0] = r;
            destination[1] = g;
            destination[2] = b;
        }
        destination[3] = rgbPixels[3];
        return destination;
    }

    public record Padding(int top, int left, int right, int bottom) {
    }
}

