//    SPDX-License-Identifier: LGPL-3.0-or-later
/*
   This file is part of FlexiBLAS, a BLAS/LAPACK interface wrapper library.
   Copyright (C) 2013-2025 Martin Koehler

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <complex.h>

#include "flexiblas_config.h"
#include "flexiblas.h"
#include "flexiblas_fortran_char_len.h"
#include "flexiblas_fortran_mangle.h"
#include "helper.h"

#if __GNUC__ >= 7
#define likely(x)      __builtin_expect(!!(x), 1)
#define unlikely(x)    __builtin_expect(!!(x), 0)
#else
#define likely(x) x
#define unlikely(x) x
#endif

#include "lapacke.h"

#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cbbcsd = 0;
#endif

/* Wrapper for LAPACKE_cbbcsd */
lapack_int  LAPACKE_cbbcsd (int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, float *theta, float *phi, lapack_complex_float *u1, lapack_int ldu1, lapack_complex_float *u2, lapack_int ldu2, lapack_complex_float *v1t, lapack_int ldv1t, lapack_complex_float *v2t, lapack_int ldv2t, float *b11d, float *b11e, float *b12d, float *b12e, float *b21d, float *b21e, float *b22d, float *b22e)
{
    lapack_int (*fn) (int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, void *theta, void *phi, void *u1, lapack_int ldu1, void *u2, lapack_int ldu2, void *v1t, lapack_int ldv1t, void *v2t, lapack_int ldv2t, void *b11d, void *b11e, void *b12d, void *b12e, void *b21d, void *b21e, void *b22d, void *b22e);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, void *theta, void *phi, void *u1, lapack_int ldu1, void *u2, lapack_int ldu2, void *v1t, lapack_int ldv1t, void *v2t, lapack_int ldv2t, void *b11d, void *b11e, void *b12d, void *b12e, void *b21d, void *b21e, void *b22d, void *b22e);
    hook_pos_LAPACKE_cbbcsd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cbbcsd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cbbcsd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu1, jobu2, jobv1t, jobv2t, trans, m, p, q, theta, phi, u1, ldu1, u2, ldu2, v1t, ldv1t, v2t, ldv2t, b11d, b11e, b12d, b12e, b21d, b21e, b22d, b22e);
    } else {
        return fn(matrix_layout, jobu1, jobu2, jobv1t, jobv2t, trans, m, p, q, theta, phi, u1, ldu1, u2, ldu2, v1t, ldv1t, v2t, ldv2t, b11d, b11e, b12d, b12e, b21d, b21e, b22d, b22e);
    }
    #else
    return fn(matrix_layout, jobu1, jobu2, jobv1t, jobv2t, trans, m, p, q, theta, phi, u1, ldu1, u2, ldu2, v1t, ldv1t, v2t, ldv2t, b11d, b11e, b12d, b12e, b21d, b21e, b22d, b22e);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cbbcsd(int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, float *theta, float *phi, lapack_complex_float *u1, lapack_int ldu1, lapack_complex_float *u2, lapack_int ldu2, lapack_complex_float *v1t, lapack_int ldv1t, lapack_complex_float *v2t, lapack_int ldv2t, float *b11d, float *b11e, float *b12d, float *b12e, float *b21d, float *b21e, float *b22d, float *b22e)
{
    lapack_int (*fn) (int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, void *theta, void *phi, void *u1, lapack_int ldu1, void *u2, lapack_int ldu2, void *v1t, lapack_int ldv1t, void *v2t, lapack_int ldv2t, void *b11d, void *b11e, void *b12d, void *b12e, void *b21d, void *b21e, void *b22d, void *b22e);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cbbcsd;
    return fn(matrix_layout, jobu1, jobu2, jobv1t, jobv2t, trans, m, p, q, theta, phi, u1, ldu1, u2, ldu2, v1t, ldv1t, v2t, ldv2t, b11d, b11e, b12d, b12e, b21d, b21e, b22d, b22e);
}

lapack_int flexiblas_chain_LAPACKE_cbbcsd(int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, void *theta, void *phi, void *u1, lapack_int ldu1, void *u2, lapack_int ldu2, void *v1t, lapack_int ldv1t, void *v2t, lapack_int ldv2t, void *b11d, void *b11e, void *b12d, void *b12e, void *b21d, void *b21e, void *b22d, void *b22e)
{
    lapack_int (*fn) (int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, void *theta, void *phi, void *u1, lapack_int ldu1, void *u2, lapack_int ldu2, void *v1t, lapack_int ldv1t, void *v2t, lapack_int ldv2t, void *b11d, void *b11e, void *b12d, void *b12e, void *b21d, void *b21e, void *b22d, void *b22e);
    hook_pos_LAPACKE_cbbcsd++;
    if ( hook_pos_LAPACKE_cbbcsd < __flexiblas_hooks->LAPACKE_cbbcsd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cbbcsd.hook_function[hook_pos_LAPACKE_cbbcsd];
    } else {
        hook_pos_LAPACKE_cbbcsd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cbbcsd;
    }
    return fn(matrix_layout, jobu1, jobu2, jobv1t, jobv2t, trans, m, p, q, theta, phi, u1, ldu1, u2, ldu2, v1t, ldv1t, v2t, ldv2t, b11d, b11e, b12d, b12e, b21d, b21e, b22d, b22e);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cbbcsd_work = 0;
#endif

/* Wrapper for LAPACKE_cbbcsd_work */
lapack_int  LAPACKE_cbbcsd_work (int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, float *theta, float *phi, lapack_complex_float *u1, lapack_int ldu1, lapack_complex_float *u2, lapack_int ldu2, lapack_complex_float *v1t, lapack_int ldv1t, lapack_complex_float *v2t, lapack_int ldv2t, float *b11d, float *b11e, float *b12d, float *b12e, float *b21d, float *b21e, float *b22d, float *b22e, float *rwork, lapack_int lrwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, void *theta, void *phi, void *u1, lapack_int ldu1, void *u2, lapack_int ldu2, void *v1t, lapack_int ldv1t, void *v2t, lapack_int ldv2t, void *b11d, void *b11e, void *b12d, void *b12e, void *b21d, void *b21e, void *b22d, void *b22e, void *rwork, lapack_int lrwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, void *theta, void *phi, void *u1, lapack_int ldu1, void *u2, lapack_int ldu2, void *v1t, lapack_int ldv1t, void *v2t, lapack_int ldv2t, void *b11d, void *b11e, void *b12d, void *b12e, void *b21d, void *b21e, void *b22d, void *b22e, void *rwork, lapack_int lrwork);
    hook_pos_LAPACKE_cbbcsd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cbbcsd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cbbcsd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu1, jobu2, jobv1t, jobv2t, trans, m, p, q, theta, phi, u1, ldu1, u2, ldu2, v1t, ldv1t, v2t, ldv2t, b11d, b11e, b12d, b12e, b21d, b21e, b22d, b22e, rwork, lrwork);
    } else {
        return fn(matrix_layout, jobu1, jobu2, jobv1t, jobv2t, trans, m, p, q, theta, phi, u1, ldu1, u2, ldu2, v1t, ldv1t, v2t, ldv2t, b11d, b11e, b12d, b12e, b21d, b21e, b22d, b22e, rwork, lrwork);
    }
    #else
    return fn(matrix_layout, jobu1, jobu2, jobv1t, jobv2t, trans, m, p, q, theta, phi, u1, ldu1, u2, ldu2, v1t, ldv1t, v2t, ldv2t, b11d, b11e, b12d, b12e, b21d, b21e, b22d, b22e, rwork, lrwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cbbcsd_work(int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, float *theta, float *phi, lapack_complex_float *u1, lapack_int ldu1, lapack_complex_float *u2, lapack_int ldu2, lapack_complex_float *v1t, lapack_int ldv1t, lapack_complex_float *v2t, lapack_int ldv2t, float *b11d, float *b11e, float *b12d, float *b12e, float *b21d, float *b21e, float *b22d, float *b22e, float *rwork, lapack_int lrwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, void *theta, void *phi, void *u1, lapack_int ldu1, void *u2, lapack_int ldu2, void *v1t, lapack_int ldv1t, void *v2t, lapack_int ldv2t, void *b11d, void *b11e, void *b12d, void *b12e, void *b21d, void *b21e, void *b22d, void *b22e, void *rwork, lapack_int lrwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cbbcsd_work;
    return fn(matrix_layout, jobu1, jobu2, jobv1t, jobv2t, trans, m, p, q, theta, phi, u1, ldu1, u2, ldu2, v1t, ldv1t, v2t, ldv2t, b11d, b11e, b12d, b12e, b21d, b21e, b22d, b22e, rwork, lrwork);
}

lapack_int flexiblas_chain_LAPACKE_cbbcsd_work(int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, void *theta, void *phi, void *u1, lapack_int ldu1, void *u2, lapack_int ldu2, void *v1t, lapack_int ldv1t, void *v2t, lapack_int ldv2t, void *b11d, void *b11e, void *b12d, void *b12e, void *b21d, void *b21e, void *b22d, void *b22e, void *rwork, lapack_int lrwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu1, char jobu2, char jobv1t, char jobv2t, char trans, lapack_int m, lapack_int p, lapack_int q, void *theta, void *phi, void *u1, lapack_int ldu1, void *u2, lapack_int ldu2, void *v1t, lapack_int ldv1t, void *v2t, lapack_int ldv2t, void *b11d, void *b11e, void *b12d, void *b12e, void *b21d, void *b21e, void *b22d, void *b22e, void *rwork, lapack_int lrwork);
    hook_pos_LAPACKE_cbbcsd_work++;
    if ( hook_pos_LAPACKE_cbbcsd_work < __flexiblas_hooks->LAPACKE_cbbcsd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cbbcsd_work.hook_function[hook_pos_LAPACKE_cbbcsd_work];
    } else {
        hook_pos_LAPACKE_cbbcsd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cbbcsd_work;
    }
    return fn(matrix_layout, jobu1, jobu2, jobv1t, jobv2t, trans, m, p, q, theta, phi, u1, ldu1, u2, ldu2, v1t, ldv1t, v2t, ldv2t, b11d, b11e, b12d, b12e, b21d, b21e, b22d, b22e, rwork, lrwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cbdsqr = 0;
#endif

/* Wrapper for LAPACKE_cbdsqr */
lapack_int  LAPACKE_cbdsqr (int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, float *d, float *e, lapack_complex_float *vt, lapack_int ldvt, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, void *d, void *e, void *vt, lapack_int ldvt, void *u, lapack_int ldu, void *c, lapack_int ldc);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, void *d, void *e, void *vt, lapack_int ldvt, void *u, lapack_int ldu, void *c, lapack_int ldc);
    hook_pos_LAPACKE_cbdsqr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cbdsqr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cbdsqr.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc);
    } else {
        return fn(matrix_layout, uplo, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc);
    }
    #else
    return fn(matrix_layout, uplo, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cbdsqr(int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, float *d, float *e, lapack_complex_float *vt, lapack_int ldvt, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, void *d, void *e, void *vt, lapack_int ldvt, void *u, lapack_int ldu, void *c, lapack_int ldc);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cbdsqr;
    return fn(matrix_layout, uplo, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc);
}

lapack_int flexiblas_chain_LAPACKE_cbdsqr(int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, void *d, void *e, void *vt, lapack_int ldvt, void *u, lapack_int ldu, void *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, void *d, void *e, void *vt, lapack_int ldvt, void *u, lapack_int ldu, void *c, lapack_int ldc);
    hook_pos_LAPACKE_cbdsqr++;
    if ( hook_pos_LAPACKE_cbdsqr < __flexiblas_hooks->LAPACKE_cbdsqr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cbdsqr.hook_function[hook_pos_LAPACKE_cbdsqr];
    } else {
        hook_pos_LAPACKE_cbdsqr = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cbdsqr;
    }
    return fn(matrix_layout, uplo, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cbdsqr_work = 0;
#endif

/* Wrapper for LAPACKE_cbdsqr_work */
lapack_int  LAPACKE_cbdsqr_work (int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, float *d, float *e, lapack_complex_float *vt, lapack_int ldvt, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *c, lapack_int ldc, float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, void *d, void *e, void *vt, lapack_int ldvt, void *u, lapack_int ldu, void *c, lapack_int ldc, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, void *d, void *e, void *vt, lapack_int ldvt, void *u, lapack_int ldu, void *c, lapack_int ldc, void *work);
    hook_pos_LAPACKE_cbdsqr_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cbdsqr_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cbdsqr_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc, work);
    } else {
        return fn(matrix_layout, uplo, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc, work);
    }
    #else
    return fn(matrix_layout, uplo, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cbdsqr_work(int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, float *d, float *e, lapack_complex_float *vt, lapack_int ldvt, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *c, lapack_int ldc, float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, void *d, void *e, void *vt, lapack_int ldvt, void *u, lapack_int ldu, void *c, lapack_int ldc, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cbdsqr_work;
    return fn(matrix_layout, uplo, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc, work);
}

lapack_int flexiblas_chain_LAPACKE_cbdsqr_work(int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, void *d, void *e, void *vt, lapack_int ldvt, void *u, lapack_int ldu, void *c, lapack_int ldc, void *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int ncvt, lapack_int nru, lapack_int ncc, void *d, void *e, void *vt, lapack_int ldvt, void *u, lapack_int ldu, void *c, lapack_int ldc, void *work);
    hook_pos_LAPACKE_cbdsqr_work++;
    if ( hook_pos_LAPACKE_cbdsqr_work < __flexiblas_hooks->LAPACKE_cbdsqr_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cbdsqr_work.hook_function[hook_pos_LAPACKE_cbdsqr_work];
    } else {
        hook_pos_LAPACKE_cbdsqr_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cbdsqr_work;
    }
    return fn(matrix_layout, uplo, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbbrd = 0;
#endif

/* Wrapper for LAPACKE_cgbbrd */
lapack_int  LAPACKE_cgbbrd (int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, lapack_complex_float *ab, lapack_int ldab, float *d, float *e, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *pt, lapack_int ldpt, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *pt, lapack_int ldpt, void *c, lapack_int ldc);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *pt, lapack_int ldpt, void *c, lapack_int ldc);
    hook_pos_LAPACKE_cgbbrd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbbrd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbbrd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, vect, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc);
    } else {
        return fn(matrix_layout, vect, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc);
    }
    #else
    return fn(matrix_layout, vect, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbbrd(int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, lapack_complex_float *ab, lapack_int ldab, float *d, float *e, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *pt, lapack_int ldpt, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *pt, lapack_int ldpt, void *c, lapack_int ldc);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbbrd;
    return fn(matrix_layout, vect, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc);
}

lapack_int flexiblas_chain_LAPACKE_cgbbrd(int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *pt, lapack_int ldpt, void *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *pt, lapack_int ldpt, void *c, lapack_int ldc);
    hook_pos_LAPACKE_cgbbrd++;
    if ( hook_pos_LAPACKE_cgbbrd < __flexiblas_hooks->LAPACKE_cgbbrd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbbrd.hook_function[hook_pos_LAPACKE_cgbbrd];
    } else {
        hook_pos_LAPACKE_cgbbrd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbbrd;
    }
    return fn(matrix_layout, vect, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbbrd_work = 0;
#endif

/* Wrapper for LAPACKE_cgbbrd_work */
lapack_int  LAPACKE_cgbbrd_work (int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, lapack_complex_float *ab, lapack_int ldab, float *d, float *e, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *pt, lapack_int ldpt, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *pt, lapack_int ldpt, void *c, lapack_int ldc, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *pt, lapack_int ldpt, void *c, lapack_int ldc, void *work, void *rwork);
    hook_pos_LAPACKE_cgbbrd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbbrd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbbrd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, vect, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc, work, rwork);
    } else {
        return fn(matrix_layout, vect, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc, work, rwork);
    }
    #else
    return fn(matrix_layout, vect, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbbrd_work(int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, lapack_complex_float *ab, lapack_int ldab, float *d, float *e, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *pt, lapack_int ldpt, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *pt, lapack_int ldpt, void *c, lapack_int ldc, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbbrd_work;
    return fn(matrix_layout, vect, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgbbrd_work(int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *pt, lapack_int ldpt, void *c, lapack_int ldc, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char vect, lapack_int m, lapack_int n, lapack_int ncc, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *pt, lapack_int ldpt, void *c, lapack_int ldc, void *work, void *rwork);
    hook_pos_LAPACKE_cgbbrd_work++;
    if ( hook_pos_LAPACKE_cgbbrd_work < __flexiblas_hooks->LAPACKE_cgbbrd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbbrd_work.hook_function[hook_pos_LAPACKE_cgbbrd_work];
    } else {
        hook_pos_LAPACKE_cgbbrd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbbrd_work;
    }
    return fn(matrix_layout, vect, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbcon = 0;
#endif

/* Wrapper for LAPACKE_cgbcon */
lapack_int  LAPACKE_cgbcon (int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const lapack_complex_float *ab, lapack_int ldab, const lapack_int *ipiv, float anorm, float *rcond)
{
    lapack_int (*fn) (int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, const void *ipiv, float anorm, void *rcond);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, const void *ipiv, float anorm, void *rcond);
    hook_pos_LAPACKE_cgbcon = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbcon;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbcon.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, norm, n, kl, ku, ab, ldab, ipiv, anorm, rcond);
    } else {
        return fn(matrix_layout, norm, n, kl, ku, ab, ldab, ipiv, anorm, rcond);
    }
    #else
    return fn(matrix_layout, norm, n, kl, ku, ab, ldab, ipiv, anorm, rcond);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbcon(int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const lapack_complex_float *ab, lapack_int ldab, const lapack_int *ipiv, float anorm, float *rcond)
{
    lapack_int (*fn) (int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, const void *ipiv, float anorm, void *rcond);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbcon;
    return fn(matrix_layout, norm, n, kl, ku, ab, ldab, ipiv, anorm, rcond);
}

lapack_int flexiblas_chain_LAPACKE_cgbcon(int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, const void *ipiv, float anorm, void *rcond)
{
    lapack_int (*fn) (int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, const void *ipiv, float anorm, void *rcond);
    hook_pos_LAPACKE_cgbcon++;
    if ( hook_pos_LAPACKE_cgbcon < __flexiblas_hooks->LAPACKE_cgbcon.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbcon.hook_function[hook_pos_LAPACKE_cgbcon];
    } else {
        hook_pos_LAPACKE_cgbcon = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbcon;
    }
    return fn(matrix_layout, norm, n, kl, ku, ab, ldab, ipiv, anorm, rcond);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbcon_work = 0;
#endif

/* Wrapper for LAPACKE_cgbcon_work */
lapack_int  LAPACKE_cgbcon_work (int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const lapack_complex_float *ab, lapack_int ldab, const lapack_int *ipiv, float anorm, float *rcond, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, const void *ipiv, float anorm, void *rcond, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, const void *ipiv, float anorm, void *rcond, void *work, void *rwork);
    hook_pos_LAPACKE_cgbcon_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbcon_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbcon_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, norm, n, kl, ku, ab, ldab, ipiv, anorm, rcond, work, rwork);
    } else {
        return fn(matrix_layout, norm, n, kl, ku, ab, ldab, ipiv, anorm, rcond, work, rwork);
    }
    #else
    return fn(matrix_layout, norm, n, kl, ku, ab, ldab, ipiv, anorm, rcond, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbcon_work(int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const lapack_complex_float *ab, lapack_int ldab, const lapack_int *ipiv, float anorm, float *rcond, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, const void *ipiv, float anorm, void *rcond, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbcon_work;
    return fn(matrix_layout, norm, n, kl, ku, ab, ldab, ipiv, anorm, rcond, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgbcon_work(int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, const void *ipiv, float anorm, void *rcond, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char norm, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, const void *ipiv, float anorm, void *rcond, void *work, void *rwork);
    hook_pos_LAPACKE_cgbcon_work++;
    if ( hook_pos_LAPACKE_cgbcon_work < __flexiblas_hooks->LAPACKE_cgbcon_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbcon_work.hook_function[hook_pos_LAPACKE_cgbcon_work];
    } else {
        hook_pos_LAPACKE_cgbcon_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbcon_work;
    }
    return fn(matrix_layout, norm, n, kl, ku, ab, ldab, ipiv, anorm, rcond, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbequ = 0;
#endif

/* Wrapper for LAPACKE_cgbequ */
lapack_int  LAPACKE_cgbequ (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const lapack_complex_float *ab, lapack_int ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgbequ = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbequ;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbequ.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
    } else {
        return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
    }
    #else
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbequ(int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const lapack_complex_float *ab, lapack_int ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbequ;
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
}

lapack_int flexiblas_chain_LAPACKE_cgbequ(int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgbequ++;
    if ( hook_pos_LAPACKE_cgbequ < __flexiblas_hooks->LAPACKE_cgbequ.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbequ.hook_function[hook_pos_LAPACKE_cgbequ];
    } else {
        hook_pos_LAPACKE_cgbequ = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbequ;
    }
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbequ_work = 0;
#endif

/* Wrapper for LAPACKE_cgbequ_work */
lapack_int  LAPACKE_cgbequ_work (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const lapack_complex_float *ab, lapack_int ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgbequ_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbequ_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbequ_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
    } else {
        return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
    }
    #else
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbequ_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const lapack_complex_float *ab, lapack_int ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbequ_work;
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
}

lapack_int flexiblas_chain_LAPACKE_cgbequ_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgbequ_work++;
    if ( hook_pos_LAPACKE_cgbequ_work < __flexiblas_hooks->LAPACKE_cgbequ_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbequ_work.hook_function[hook_pos_LAPACKE_cgbequ_work];
    } else {
        hook_pos_LAPACKE_cgbequ_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbequ_work;
    }
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbequb = 0;
#endif

/* Wrapper for LAPACKE_cgbequb */
lapack_int  LAPACKE_cgbequb (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const lapack_complex_float *ab, lapack_int ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgbequb = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbequb;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbequb.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
    } else {
        return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
    }
    #else
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbequb(int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const lapack_complex_float *ab, lapack_int ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbequb;
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
}

lapack_int flexiblas_chain_LAPACKE_cgbequb(int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgbequb++;
    if ( hook_pos_LAPACKE_cgbequb < __flexiblas_hooks->LAPACKE_cgbequb.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbequb.hook_function[hook_pos_LAPACKE_cgbequb];
    } else {
        hook_pos_LAPACKE_cgbequb = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbequb;
    }
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbequb_work = 0;
#endif

/* Wrapper for LAPACKE_cgbequb_work */
lapack_int  LAPACKE_cgbequb_work (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const lapack_complex_float *ab, lapack_int ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgbequb_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbequb_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbequb_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
    } else {
        return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
    }
    #else
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbequb_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const lapack_complex_float *ab, lapack_int ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbequb_work;
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
}

lapack_int flexiblas_chain_LAPACKE_cgbequb_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, const void *ab, lapack_int ldab, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgbequb_work++;
    if ( hook_pos_LAPACKE_cgbequb_work < __flexiblas_hooks->LAPACKE_cgbequb_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbequb_work.hook_function[hook_pos_LAPACKE_cgbequb_work];
    } else {
        hook_pos_LAPACKE_cgbequb_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbequb_work;
    }
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, r, c, rowcnd, colcnd, amax);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbrfs = 0;
#endif

/* Wrapper for LAPACKE_cgbrfs */
lapack_int  LAPACKE_cgbrfs (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const lapack_complex_float *ab, lapack_int ldab, const lapack_complex_float *afb, lapack_int ldafb, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *afb, lapack_int ldafb, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *afb, lapack_int ldafb, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    hook_pos_LAPACKE_cgbrfs = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbrfs;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbrfs.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr);
    } else {
        return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr);
    }
    #else
    return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbrfs(int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const lapack_complex_float *ab, lapack_int ldab, const lapack_complex_float *afb, lapack_int ldafb, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *afb, lapack_int ldafb, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbrfs;
    return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr);
}

lapack_int flexiblas_chain_LAPACKE_cgbrfs(int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *afb, lapack_int ldafb, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *afb, lapack_int ldafb, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    hook_pos_LAPACKE_cgbrfs++;
    if ( hook_pos_LAPACKE_cgbrfs < __flexiblas_hooks->LAPACKE_cgbrfs.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbrfs.hook_function[hook_pos_LAPACKE_cgbrfs];
    } else {
        hook_pos_LAPACKE_cgbrfs = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbrfs;
    }
    return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbrfs_work = 0;
#endif

/* Wrapper for LAPACKE_cgbrfs_work */
lapack_int  LAPACKE_cgbrfs_work (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const lapack_complex_float *ab, lapack_int ldab, const lapack_complex_float *afb, lapack_int ldafb, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *afb, lapack_int ldafb, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *afb, lapack_int ldafb, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cgbrfs_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbrfs_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbrfs_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    } else {
        return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    }
    #else
    return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbrfs_work(int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const lapack_complex_float *ab, lapack_int ldab, const lapack_complex_float *afb, lapack_int ldafb, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *afb, lapack_int ldafb, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbrfs_work;
    return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgbrfs_work(int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *afb, lapack_int ldafb, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *afb, lapack_int ldafb, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cgbrfs_work++;
    if ( hook_pos_LAPACKE_cgbrfs_work < __flexiblas_hooks->LAPACKE_cgbrfs_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbrfs_work.hook_function[hook_pos_LAPACKE_cgbrfs_work];
    } else {
        hook_pos_LAPACKE_cgbrfs_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbrfs_work;
    }
    return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbsv = 0;
#endif

/* Wrapper for LAPACKE_cgbsv */
lapack_int  LAPACKE_cgbsv (int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, lapack_complex_float *ab, lapack_int ldab, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgbsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbsv.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbsv(int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, lapack_complex_float *ab, lapack_int ldab, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbsv;
    return fn(matrix_layout, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgbsv(int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgbsv++;
    if ( hook_pos_LAPACKE_cgbsv < __flexiblas_hooks->LAPACKE_cgbsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbsv.hook_function[hook_pos_LAPACKE_cgbsv];
    } else {
        hook_pos_LAPACKE_cgbsv = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbsv;
    }
    return fn(matrix_layout, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbsv_work = 0;
#endif

/* Wrapper for LAPACKE_cgbsv_work */
lapack_int  LAPACKE_cgbsv_work (int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, lapack_complex_float *ab, lapack_int ldab, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgbsv_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbsv_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbsv_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbsv_work(int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, lapack_complex_float *ab, lapack_int ldab, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbsv_work;
    return fn(matrix_layout, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgbsv_work(int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgbsv_work++;
    if ( hook_pos_LAPACKE_cgbsv_work < __flexiblas_hooks->LAPACKE_cgbsv_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbsv_work.hook_function[hook_pos_LAPACKE_cgbsv_work];
    } else {
        hook_pos_LAPACKE_cgbsv_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbsv_work;
    }
    return fn(matrix_layout, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbsvx = 0;
#endif

/* Wrapper for LAPACKE_cgbsvx */
lapack_int  LAPACKE_cgbsvx (int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *afb, lapack_int ldafb, lapack_int *ipiv, char *equed, float *r, float *c, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, float *rpivot)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *afb, lapack_int ldafb, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *rpivot);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *afb, lapack_int ldafb, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *rpivot);
    hook_pos_LAPACKE_cgbsvx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbsvx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbsvx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, fact, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
    } else {
        return fn(matrix_layout, fact, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
    }
    #else
    return fn(matrix_layout, fact, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbsvx(int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *afb, lapack_int ldafb, lapack_int *ipiv, char *equed, float *r, float *c, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, float *rpivot)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *afb, lapack_int ldafb, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *rpivot);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbsvx;
    return fn(matrix_layout, fact, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
}

lapack_int flexiblas_chain_LAPACKE_cgbsvx(int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *afb, lapack_int ldafb, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *rpivot)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *afb, lapack_int ldafb, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *rpivot);
    hook_pos_LAPACKE_cgbsvx++;
    if ( hook_pos_LAPACKE_cgbsvx < __flexiblas_hooks->LAPACKE_cgbsvx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbsvx.hook_function[hook_pos_LAPACKE_cgbsvx];
    } else {
        hook_pos_LAPACKE_cgbsvx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbsvx;
    }
    return fn(matrix_layout, fact, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbsvx_work = 0;
#endif

/* Wrapper for LAPACKE_cgbsvx_work */
lapack_int  LAPACKE_cgbsvx_work (int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *afb, lapack_int ldafb, lapack_int *ipiv, char *equed, float *r, float *c, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *afb, lapack_int ldafb, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *afb, lapack_int ldafb, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cgbsvx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbsvx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbsvx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, fact, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
    } else {
        return fn(matrix_layout, fact, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
    }
    #else
    return fn(matrix_layout, fact, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbsvx_work(int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *afb, lapack_int ldafb, lapack_int *ipiv, char *equed, float *r, float *c, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *afb, lapack_int ldafb, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbsvx_work;
    return fn(matrix_layout, fact, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgbsvx_work(int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *afb, lapack_int ldafb, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, void *ab, lapack_int ldab, void *afb, lapack_int ldafb, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cgbsvx_work++;
    if ( hook_pos_LAPACKE_cgbsvx_work < __flexiblas_hooks->LAPACKE_cgbsvx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbsvx_work.hook_function[hook_pos_LAPACKE_cgbsvx_work];
    } else {
        hook_pos_LAPACKE_cgbsvx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbsvx_work;
    }
    return fn(matrix_layout, fact, trans, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbtrf = 0;
#endif

/* Wrapper for LAPACKE_cgbtrf */
lapack_int  LAPACKE_cgbtrf (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, lapack_complex_float *ab, lapack_int ldab, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *ipiv);
    hook_pos_LAPACKE_cgbtrf = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbtrf;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbtrf.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, kl, ku, ab, ldab, ipiv);
    } else {
        return fn(matrix_layout, m, n, kl, ku, ab, ldab, ipiv);
    }
    #else
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbtrf(int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, lapack_complex_float *ab, lapack_int ldab, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbtrf;
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_cgbtrf(int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *ipiv);
    hook_pos_LAPACKE_cgbtrf++;
    if ( hook_pos_LAPACKE_cgbtrf < __flexiblas_hooks->LAPACKE_cgbtrf.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbtrf.hook_function[hook_pos_LAPACKE_cgbtrf];
    } else {
        hook_pos_LAPACKE_cgbtrf = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbtrf;
    }
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbtrf_work = 0;
#endif

/* Wrapper for LAPACKE_cgbtrf_work */
lapack_int  LAPACKE_cgbtrf_work (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, lapack_complex_float *ab, lapack_int ldab, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *ipiv);
    hook_pos_LAPACKE_cgbtrf_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbtrf_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbtrf_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, kl, ku, ab, ldab, ipiv);
    } else {
        return fn(matrix_layout, m, n, kl, ku, ab, ldab, ipiv);
    }
    #else
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbtrf_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, lapack_complex_float *ab, lapack_int ldab, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbtrf_work;
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_cgbtrf_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int kl, lapack_int ku, void *ab, lapack_int ldab, void *ipiv);
    hook_pos_LAPACKE_cgbtrf_work++;
    if ( hook_pos_LAPACKE_cgbtrf_work < __flexiblas_hooks->LAPACKE_cgbtrf_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbtrf_work.hook_function[hook_pos_LAPACKE_cgbtrf_work];
    } else {
        hook_pos_LAPACKE_cgbtrf_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbtrf_work;
    }
    return fn(matrix_layout, m, n, kl, ku, ab, ldab, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbtrs = 0;
#endif

/* Wrapper for LAPACKE_cgbtrs */
lapack_int  LAPACKE_cgbtrs (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const lapack_complex_float *ab, lapack_int ldab, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgbtrs = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbtrs;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbtrs.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbtrs(int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const lapack_complex_float *ab, lapack_int ldab, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbtrs;
    return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgbtrs(int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgbtrs++;
    if ( hook_pos_LAPACKE_cgbtrs < __flexiblas_hooks->LAPACKE_cgbtrs.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbtrs.hook_function[hook_pos_LAPACKE_cgbtrs];
    } else {
        hook_pos_LAPACKE_cgbtrs = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbtrs;
    }
    return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgbtrs_work = 0;
#endif

/* Wrapper for LAPACKE_cgbtrs_work */
lapack_int  LAPACKE_cgbtrs_work (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const lapack_complex_float *ab, lapack_int ldab, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgbtrs_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbtrs_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgbtrs_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgbtrs_work(int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const lapack_complex_float *ab, lapack_int ldab, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgbtrs_work;
    return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgbtrs_work(int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int kl, lapack_int ku, lapack_int nrhs, const void *ab, lapack_int ldab, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgbtrs_work++;
    if ( hook_pos_LAPACKE_cgbtrs_work < __flexiblas_hooks->LAPACKE_cgbtrs_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgbtrs_work.hook_function[hook_pos_LAPACKE_cgbtrs_work];
    } else {
        hook_pos_LAPACKE_cgbtrs_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgbtrs_work;
    }
    return fn(matrix_layout, trans, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgebak = 0;
#endif

/* Wrapper for LAPACKE_cgebak */
lapack_int  LAPACKE_cgebak (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const float *scale, lapack_int m, lapack_complex_float *v, lapack_int ldv)
{
    lapack_int (*fn) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *scale, lapack_int m, void *v, lapack_int ldv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *scale, lapack_int m, void *v, lapack_int ldv);
    hook_pos_LAPACKE_cgebak = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgebak;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgebak.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, side, n, ilo, ihi, scale, m, v, ldv);
    } else {
        return fn(matrix_layout, job, side, n, ilo, ihi, scale, m, v, ldv);
    }
    #else
    return fn(matrix_layout, job, side, n, ilo, ihi, scale, m, v, ldv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgebak(int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const float *scale, lapack_int m, lapack_complex_float *v, lapack_int ldv)
{
    lapack_int (*fn) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *scale, lapack_int m, void *v, lapack_int ldv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgebak;
    return fn(matrix_layout, job, side, n, ilo, ihi, scale, m, v, ldv);
}

lapack_int flexiblas_chain_LAPACKE_cgebak(int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *scale, lapack_int m, void *v, lapack_int ldv)
{
    lapack_int (*fn) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *scale, lapack_int m, void *v, lapack_int ldv);
    hook_pos_LAPACKE_cgebak++;
    if ( hook_pos_LAPACKE_cgebak < __flexiblas_hooks->LAPACKE_cgebak.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgebak.hook_function[hook_pos_LAPACKE_cgebak];
    } else {
        hook_pos_LAPACKE_cgebak = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgebak;
    }
    return fn(matrix_layout, job, side, n, ilo, ihi, scale, m, v, ldv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgebak_work = 0;
#endif

/* Wrapper for LAPACKE_cgebak_work */
lapack_int  LAPACKE_cgebak_work (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const float *scale, lapack_int m, lapack_complex_float *v, lapack_int ldv)
{
    lapack_int (*fn) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *scale, lapack_int m, void *v, lapack_int ldv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *scale, lapack_int m, void *v, lapack_int ldv);
    hook_pos_LAPACKE_cgebak_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgebak_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgebak_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, side, n, ilo, ihi, scale, m, v, ldv);
    } else {
        return fn(matrix_layout, job, side, n, ilo, ihi, scale, m, v, ldv);
    }
    #else
    return fn(matrix_layout, job, side, n, ilo, ihi, scale, m, v, ldv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgebak_work(int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const float *scale, lapack_int m, lapack_complex_float *v, lapack_int ldv)
{
    lapack_int (*fn) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *scale, lapack_int m, void *v, lapack_int ldv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgebak_work;
    return fn(matrix_layout, job, side, n, ilo, ihi, scale, m, v, ldv);
}

lapack_int flexiblas_chain_LAPACKE_cgebak_work(int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *scale, lapack_int m, void *v, lapack_int ldv)
{
    lapack_int (*fn) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *scale, lapack_int m, void *v, lapack_int ldv);
    hook_pos_LAPACKE_cgebak_work++;
    if ( hook_pos_LAPACKE_cgebak_work < __flexiblas_hooks->LAPACKE_cgebak_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgebak_work.hook_function[hook_pos_LAPACKE_cgebak_work];
    } else {
        hook_pos_LAPACKE_cgebak_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgebak_work;
    }
    return fn(matrix_layout, job, side, n, ilo, ihi, scale, m, v, ldv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgebal = 0;
#endif

/* Wrapper for LAPACKE_cgebal */
lapack_int  LAPACKE_cgebal (int matrix_layout, char job, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ilo, lapack_int *ihi, float *scale)
{
    lapack_int (*fn) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *ilo, void *ihi, void *scale);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *ilo, void *ihi, void *scale);
    hook_pos_LAPACKE_cgebal = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgebal;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgebal.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, n, a, lda, ilo, ihi, scale);
    } else {
        return fn(matrix_layout, job, n, a, lda, ilo, ihi, scale);
    }
    #else
    return fn(matrix_layout, job, n, a, lda, ilo, ihi, scale);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgebal(int matrix_layout, char job, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ilo, lapack_int *ihi, float *scale)
{
    lapack_int (*fn) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *ilo, void *ihi, void *scale);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgebal;
    return fn(matrix_layout, job, n, a, lda, ilo, ihi, scale);
}

lapack_int flexiblas_chain_LAPACKE_cgebal(int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *ilo, void *ihi, void *scale)
{
    lapack_int (*fn) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *ilo, void *ihi, void *scale);
    hook_pos_LAPACKE_cgebal++;
    if ( hook_pos_LAPACKE_cgebal < __flexiblas_hooks->LAPACKE_cgebal.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgebal.hook_function[hook_pos_LAPACKE_cgebal];
    } else {
        hook_pos_LAPACKE_cgebal = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgebal;
    }
    return fn(matrix_layout, job, n, a, lda, ilo, ihi, scale);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgebal_work = 0;
#endif

/* Wrapper for LAPACKE_cgebal_work */
lapack_int  LAPACKE_cgebal_work (int matrix_layout, char job, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ilo, lapack_int *ihi, float *scale)
{
    lapack_int (*fn) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *ilo, void *ihi, void *scale);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *ilo, void *ihi, void *scale);
    hook_pos_LAPACKE_cgebal_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgebal_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgebal_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, n, a, lda, ilo, ihi, scale);
    } else {
        return fn(matrix_layout, job, n, a, lda, ilo, ihi, scale);
    }
    #else
    return fn(matrix_layout, job, n, a, lda, ilo, ihi, scale);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgebal_work(int matrix_layout, char job, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ilo, lapack_int *ihi, float *scale)
{
    lapack_int (*fn) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *ilo, void *ihi, void *scale);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgebal_work;
    return fn(matrix_layout, job, n, a, lda, ilo, ihi, scale);
}

lapack_int flexiblas_chain_LAPACKE_cgebal_work(int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *ilo, void *ihi, void *scale)
{
    lapack_int (*fn) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *ilo, void *ihi, void *scale);
    hook_pos_LAPACKE_cgebal_work++;
    if ( hook_pos_LAPACKE_cgebal_work < __flexiblas_hooks->LAPACKE_cgebal_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgebal_work.hook_function[hook_pos_LAPACKE_cgebal_work];
    } else {
        hook_pos_LAPACKE_cgebal_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgebal_work;
    }
    return fn(matrix_layout, job, n, a, lda, ilo, ihi, scale);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgebrd = 0;
#endif

/* Wrapper for LAPACKE_cgebrd */
lapack_int  LAPACKE_cgebrd (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *d, float *e, lapack_complex_float *tauq, lapack_complex_float *taup)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tauq, void *taup);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tauq, void *taup);
    hook_pos_LAPACKE_cgebrd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgebrd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgebrd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, d, e, tauq, taup);
    } else {
        return fn(matrix_layout, m, n, a, lda, d, e, tauq, taup);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, d, e, tauq, taup);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgebrd(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *d, float *e, lapack_complex_float *tauq, lapack_complex_float *taup)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tauq, void *taup);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgebrd;
    return fn(matrix_layout, m, n, a, lda, d, e, tauq, taup);
}

lapack_int flexiblas_chain_LAPACKE_cgebrd(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tauq, void *taup)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tauq, void *taup);
    hook_pos_LAPACKE_cgebrd++;
    if ( hook_pos_LAPACKE_cgebrd < __flexiblas_hooks->LAPACKE_cgebrd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgebrd.hook_function[hook_pos_LAPACKE_cgebrd];
    } else {
        hook_pos_LAPACKE_cgebrd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgebrd;
    }
    return fn(matrix_layout, m, n, a, lda, d, e, tauq, taup);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgebrd_work = 0;
#endif

/* Wrapper for LAPACKE_cgebrd_work */
lapack_int  LAPACKE_cgebrd_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *d, float *e, lapack_complex_float *tauq, lapack_complex_float *taup, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tauq, void *taup, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tauq, void *taup, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgebrd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgebrd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgebrd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, d, e, tauq, taup, work, lwork);
    } else {
        return fn(matrix_layout, m, n, a, lda, d, e, tauq, taup, work, lwork);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, d, e, tauq, taup, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgebrd_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *d, float *e, lapack_complex_float *tauq, lapack_complex_float *taup, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tauq, void *taup, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgebrd_work;
    return fn(matrix_layout, m, n, a, lda, d, e, tauq, taup, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgebrd_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tauq, void *taup, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tauq, void *taup, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgebrd_work++;
    if ( hook_pos_LAPACKE_cgebrd_work < __flexiblas_hooks->LAPACKE_cgebrd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgebrd_work.hook_function[hook_pos_LAPACKE_cgebrd_work];
    } else {
        hook_pos_LAPACKE_cgebrd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgebrd_work;
    }
    return fn(matrix_layout, m, n, a, lda, d, e, tauq, taup, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgecon = 0;
#endif

/* Wrapper for LAPACKE_cgecon */
lapack_int  LAPACKE_cgecon (int matrix_layout, char norm, lapack_int n, const lapack_complex_float *a, lapack_int lda, float anorm, float *rcond)
{
    lapack_int (*fn) (int matrix_layout, char norm, lapack_int n, const void *a, lapack_int lda, float anorm, void *rcond);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char norm, lapack_int n, const void *a, lapack_int lda, float anorm, void *rcond);
    hook_pos_LAPACKE_cgecon = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgecon;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgecon.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, norm, n, a, lda, anorm, rcond);
    } else {
        return fn(matrix_layout, norm, n, a, lda, anorm, rcond);
    }
    #else
    return fn(matrix_layout, norm, n, a, lda, anorm, rcond);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgecon(int matrix_layout, char norm, lapack_int n, const lapack_complex_float *a, lapack_int lda, float anorm, float *rcond)
{
    lapack_int (*fn) (int matrix_layout, char norm, lapack_int n, const void *a, lapack_int lda, float anorm, void *rcond);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgecon;
    return fn(matrix_layout, norm, n, a, lda, anorm, rcond);
}

lapack_int flexiblas_chain_LAPACKE_cgecon(int matrix_layout, char norm, lapack_int n, const void *a, lapack_int lda, float anorm, void *rcond)
{
    lapack_int (*fn) (int matrix_layout, char norm, lapack_int n, const void *a, lapack_int lda, float anorm, void *rcond);
    hook_pos_LAPACKE_cgecon++;
    if ( hook_pos_LAPACKE_cgecon < __flexiblas_hooks->LAPACKE_cgecon.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgecon.hook_function[hook_pos_LAPACKE_cgecon];
    } else {
        hook_pos_LAPACKE_cgecon = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgecon;
    }
    return fn(matrix_layout, norm, n, a, lda, anorm, rcond);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgecon_work = 0;
#endif

/* Wrapper for LAPACKE_cgecon_work */
lapack_int  LAPACKE_cgecon_work (int matrix_layout, char norm, lapack_int n, const lapack_complex_float *a, lapack_int lda, float anorm, float *rcond, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char norm, lapack_int n, const void *a, lapack_int lda, float anorm, void *rcond, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char norm, lapack_int n, const void *a, lapack_int lda, float anorm, void *rcond, void *work, void *rwork);
    hook_pos_LAPACKE_cgecon_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgecon_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgecon_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, norm, n, a, lda, anorm, rcond, work, rwork);
    } else {
        return fn(matrix_layout, norm, n, a, lda, anorm, rcond, work, rwork);
    }
    #else
    return fn(matrix_layout, norm, n, a, lda, anorm, rcond, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgecon_work(int matrix_layout, char norm, lapack_int n, const lapack_complex_float *a, lapack_int lda, float anorm, float *rcond, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char norm, lapack_int n, const void *a, lapack_int lda, float anorm, void *rcond, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgecon_work;
    return fn(matrix_layout, norm, n, a, lda, anorm, rcond, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgecon_work(int matrix_layout, char norm, lapack_int n, const void *a, lapack_int lda, float anorm, void *rcond, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char norm, lapack_int n, const void *a, lapack_int lda, float anorm, void *rcond, void *work, void *rwork);
    hook_pos_LAPACKE_cgecon_work++;
    if ( hook_pos_LAPACKE_cgecon_work < __flexiblas_hooks->LAPACKE_cgecon_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgecon_work.hook_function[hook_pos_LAPACKE_cgecon_work];
    } else {
        hook_pos_LAPACKE_cgecon_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgecon_work;
    }
    return fn(matrix_layout, norm, n, a, lda, anorm, rcond, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeequ = 0;
#endif

/* Wrapper for LAPACKE_cgeequ */
lapack_int  LAPACKE_cgeequ (int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgeequ = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeequ;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeequ.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
    } else {
        return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeequ(int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeequ;
    return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
}

lapack_int flexiblas_chain_LAPACKE_cgeequ(int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgeequ++;
    if ( hook_pos_LAPACKE_cgeequ < __flexiblas_hooks->LAPACKE_cgeequ.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeequ.hook_function[hook_pos_LAPACKE_cgeequ];
    } else {
        hook_pos_LAPACKE_cgeequ = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeequ;
    }
    return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeequ_work = 0;
#endif

/* Wrapper for LAPACKE_cgeequ_work */
lapack_int  LAPACKE_cgeequ_work (int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgeequ_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeequ_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeequ_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
    } else {
        return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeequ_work(int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeequ_work;
    return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
}

lapack_int flexiblas_chain_LAPACKE_cgeequ_work(int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgeequ_work++;
    if ( hook_pos_LAPACKE_cgeequ_work < __flexiblas_hooks->LAPACKE_cgeequ_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeequ_work.hook_function[hook_pos_LAPACKE_cgeequ_work];
    } else {
        hook_pos_LAPACKE_cgeequ_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeequ_work;
    }
    return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeequb = 0;
#endif

/* Wrapper for LAPACKE_cgeequb */
lapack_int  LAPACKE_cgeequb (int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgeequb = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeequb;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeequb.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
    } else {
        return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeequb(int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeequb;
    return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
}

lapack_int flexiblas_chain_LAPACKE_cgeequb(int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgeequb++;
    if ( hook_pos_LAPACKE_cgeequb < __flexiblas_hooks->LAPACKE_cgeequb.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeequb.hook_function[hook_pos_LAPACKE_cgeequb];
    } else {
        hook_pos_LAPACKE_cgeequb = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeequb;
    }
    return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeequb_work = 0;
#endif

/* Wrapper for LAPACKE_cgeequb_work */
lapack_int  LAPACKE_cgeequb_work (int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgeequb_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeequb_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeequb_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
    } else {
        return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeequb_work(int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeequb_work;
    return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
}

lapack_int flexiblas_chain_LAPACKE_cgeequb_work(int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *r, void *c, void *rowcnd, void *colcnd, void *amax);
    hook_pos_LAPACKE_cgeequb_work++;
    if ( hook_pos_LAPACKE_cgeequb_work < __flexiblas_hooks->LAPACKE_cgeequb_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeequb_work.hook_function[hook_pos_LAPACKE_cgeequb_work];
    } else {
        hook_pos_LAPACKE_cgeequb_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeequb_work;
    }
    return fn(matrix_layout, m, n, a, lda, r, c, rowcnd, colcnd, amax);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgees = 0;
#endif

/* Wrapper for LAPACKE_cgees */
lapack_int  LAPACKE_cgees (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *sdim, lapack_complex_float *w, lapack_complex_float *vs, lapack_int ldvs)
{
    lapack_int (*fn) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs);
    hook_pos_LAPACKE_cgees = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgees;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgees.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvs, sort, select, n, a, lda, sdim, w, vs, ldvs);
    } else {
        return fn(matrix_layout, jobvs, sort, select, n, a, lda, sdim, w, vs, ldvs);
    }
    #else
    return fn(matrix_layout, jobvs, sort, select, n, a, lda, sdim, w, vs, ldvs);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgees(int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *sdim, lapack_complex_float *w, lapack_complex_float *vs, lapack_int ldvs)
{
    lapack_int (*fn) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgees;
    return fn(matrix_layout, jobvs, sort, select, n, a, lda, sdim, w, vs, ldvs);
}

lapack_int flexiblas_chain_LAPACKE_cgees(int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs)
{
    lapack_int (*fn) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs);
    hook_pos_LAPACKE_cgees++;
    if ( hook_pos_LAPACKE_cgees < __flexiblas_hooks->LAPACKE_cgees.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgees.hook_function[hook_pos_LAPACKE_cgees];
    } else {
        hook_pos_LAPACKE_cgees = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgees;
    }
    return fn(matrix_layout, jobvs, sort, select, n, a, lda, sdim, w, vs, ldvs);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgees_work = 0;
#endif

/* Wrapper for LAPACKE_cgees_work */
lapack_int  LAPACKE_cgees_work (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *sdim, lapack_complex_float *w, lapack_complex_float *vs, lapack_int ldvs, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *work, lapack_int lwork, void *rwork, void *bwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *work, lapack_int lwork, void *rwork, void *bwork);
    hook_pos_LAPACKE_cgees_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgees_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgees_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvs, sort, select, n, a, lda, sdim, w, vs, ldvs, work, lwork, rwork, bwork);
    } else {
        return fn(matrix_layout, jobvs, sort, select, n, a, lda, sdim, w, vs, ldvs, work, lwork, rwork, bwork);
    }
    #else
    return fn(matrix_layout, jobvs, sort, select, n, a, lda, sdim, w, vs, ldvs, work, lwork, rwork, bwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgees_work(int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *sdim, lapack_complex_float *w, lapack_complex_float *vs, lapack_int ldvs, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *work, lapack_int lwork, void *rwork, void *bwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgees_work;
    return fn(matrix_layout, jobvs, sort, select, n, a, lda, sdim, w, vs, ldvs, work, lwork, rwork, bwork);
}

lapack_int flexiblas_chain_LAPACKE_cgees_work(int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *work, lapack_int lwork, void *rwork, void *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *work, lapack_int lwork, void *rwork, void *bwork);
    hook_pos_LAPACKE_cgees_work++;
    if ( hook_pos_LAPACKE_cgees_work < __flexiblas_hooks->LAPACKE_cgees_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgees_work.hook_function[hook_pos_LAPACKE_cgees_work];
    } else {
        hook_pos_LAPACKE_cgees_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgees_work;
    }
    return fn(matrix_layout, jobvs, sort, select, n, a, lda, sdim, w, vs, ldvs, work, lwork, rwork, bwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeesx = 0;
#endif

/* Wrapper for LAPACKE_cgeesx */
lapack_int  LAPACKE_cgeesx (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *sdim, lapack_complex_float *w, lapack_complex_float *vs, lapack_int ldvs, float *rconde, float *rcondv)
{
    lapack_int (*fn) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *rconde, void *rcondv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *rconde, void *rcondv);
    hook_pos_LAPACKE_cgeesx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeesx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeesx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvs, sort, select, sense, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv);
    } else {
        return fn(matrix_layout, jobvs, sort, select, sense, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv);
    }
    #else
    return fn(matrix_layout, jobvs, sort, select, sense, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeesx(int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *sdim, lapack_complex_float *w, lapack_complex_float *vs, lapack_int ldvs, float *rconde, float *rcondv)
{
    lapack_int (*fn) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *rconde, void *rcondv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeesx;
    return fn(matrix_layout, jobvs, sort, select, sense, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv);
}

lapack_int flexiblas_chain_LAPACKE_cgeesx(int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *rconde, void *rcondv)
{
    lapack_int (*fn) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *rconde, void *rcondv);
    hook_pos_LAPACKE_cgeesx++;
    if ( hook_pos_LAPACKE_cgeesx < __flexiblas_hooks->LAPACKE_cgeesx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeesx.hook_function[hook_pos_LAPACKE_cgeesx];
    } else {
        hook_pos_LAPACKE_cgeesx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeesx;
    }
    return fn(matrix_layout, jobvs, sort, select, sense, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeesx_work = 0;
#endif

/* Wrapper for LAPACKE_cgeesx_work */
lapack_int  LAPACKE_cgeesx_work (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *sdim, lapack_complex_float *w, lapack_complex_float *vs, lapack_int ldvs, float *rconde, float *rcondv, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *bwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *bwork);
    hook_pos_LAPACKE_cgeesx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeesx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeesx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvs, sort, select, sense, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv, work, lwork, rwork, bwork);
    } else {
        return fn(matrix_layout, jobvs, sort, select, sense, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv, work, lwork, rwork, bwork);
    }
    #else
    return fn(matrix_layout, jobvs, sort, select, sense, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv, work, lwork, rwork, bwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeesx_work(int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *sdim, lapack_complex_float *w, lapack_complex_float *vs, lapack_int ldvs, float *rconde, float *rcondv, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *bwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeesx_work;
    return fn(matrix_layout, jobvs, sort, select, sense, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv, work, lwork, rwork, bwork);
}

lapack_int flexiblas_chain_LAPACKE_cgeesx_work(int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvs, char sort, LAPACK_C_SELECT1 select, char sense, lapack_int n, void *a, lapack_int lda, void *sdim, void *w, void *vs, lapack_int ldvs, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *bwork);
    hook_pos_LAPACKE_cgeesx_work++;
    if ( hook_pos_LAPACKE_cgeesx_work < __flexiblas_hooks->LAPACKE_cgeesx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeesx_work.hook_function[hook_pos_LAPACKE_cgeesx_work];
    } else {
        hook_pos_LAPACKE_cgeesx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeesx_work;
    }
    return fn(matrix_layout, jobvs, sort, select, sense, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv, work, lwork, rwork, bwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeev = 0;
#endif

/* Wrapper for LAPACKE_cgeev */
lapack_int  LAPACKE_cgeev (int matrix_layout, char jobvl, char jobvr, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *w, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr);
    hook_pos_LAPACKE_cgeev = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeev;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeev.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvl, jobvr, n, a, lda, w, vl, ldvl, vr, ldvr);
    } else {
        return fn(matrix_layout, jobvl, jobvr, n, a, lda, w, vl, ldvl, vr, ldvr);
    }
    #else
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, w, vl, ldvl, vr, ldvr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeev(int matrix_layout, char jobvl, char jobvr, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *w, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeev;
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, w, vl, ldvl, vr, ldvr);
}

lapack_int flexiblas_chain_LAPACKE_cgeev(int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr);
    hook_pos_LAPACKE_cgeev++;
    if ( hook_pos_LAPACKE_cgeev < __flexiblas_hooks->LAPACKE_cgeev.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeev.hook_function[hook_pos_LAPACKE_cgeev];
    } else {
        hook_pos_LAPACKE_cgeev = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeev;
    }
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, w, vl, ldvl, vr, ldvr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeev_work = 0;
#endif

/* Wrapper for LAPACKE_cgeev_work */
lapack_int  LAPACKE_cgeev_work (int matrix_layout, char jobvl, char jobvr, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *w, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cgeev_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeev_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeev_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvl, jobvr, n, a, lda, w, vl, ldvl, vr, ldvr, work, lwork, rwork);
    } else {
        return fn(matrix_layout, jobvl, jobvr, n, a, lda, w, vl, ldvl, vr, ldvr, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, w, vl, ldvl, vr, ldvr, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeev_work(int matrix_layout, char jobvl, char jobvr, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *w, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeev_work;
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, w, vl, ldvl, vr, ldvr, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgeev_work(int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cgeev_work++;
    if ( hook_pos_LAPACKE_cgeev_work < __flexiblas_hooks->LAPACKE_cgeev_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeev_work.hook_function[hook_pos_LAPACKE_cgeev_work];
    } else {
        hook_pos_LAPACKE_cgeev_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeev_work;
    }
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, w, vl, ldvl, vr, ldvr, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeevx = 0;
#endif

/* Wrapper for LAPACKE_cgeevx */
lapack_int  LAPACKE_cgeevx (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *w, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_int *ilo, lapack_int *ihi, float *scale, float *abnrm, float *rconde, float *rcondv)
{
    lapack_int (*fn) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *scale, void *abnrm, void *rconde, void *rcondv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *scale, void *abnrm, void *rconde, void *rcondv);
    hook_pos_LAPACKE_cgeevx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeevx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeevx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv);
    } else {
        return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv);
    }
    #else
    return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeevx(int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *w, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_int *ilo, lapack_int *ihi, float *scale, float *abnrm, float *rconde, float *rcondv)
{
    lapack_int (*fn) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *scale, void *abnrm, void *rconde, void *rcondv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeevx;
    return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv);
}

lapack_int flexiblas_chain_LAPACKE_cgeevx(int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *scale, void *abnrm, void *rconde, void *rcondv)
{
    lapack_int (*fn) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *scale, void *abnrm, void *rconde, void *rcondv);
    hook_pos_LAPACKE_cgeevx++;
    if ( hook_pos_LAPACKE_cgeevx < __flexiblas_hooks->LAPACKE_cgeevx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeevx.hook_function[hook_pos_LAPACKE_cgeevx];
    } else {
        hook_pos_LAPACKE_cgeevx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeevx;
    }
    return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeevx_work = 0;
#endif

/* Wrapper for LAPACKE_cgeevx_work */
lapack_int  LAPACKE_cgeevx_work (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *w, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_int *ilo, lapack_int *ihi, float *scale, float *abnrm, float *rconde, float *rcondv, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *scale, void *abnrm, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *scale, void *abnrm, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cgeevx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeevx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeevx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv, work, lwork, rwork);
    } else {
        return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeevx_work(int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *w, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_int *ilo, lapack_int *ihi, float *scale, float *abnrm, float *rconde, float *rcondv, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *scale, void *abnrm, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeevx_work;
    return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgeevx_work(int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *scale, void *abnrm, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *scale, void *abnrm, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cgeevx_work++;
    if ( hook_pos_LAPACKE_cgeevx_work < __flexiblas_hooks->LAPACKE_cgeevx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeevx_work.hook_function[hook_pos_LAPACKE_cgeevx_work];
    } else {
        hook_pos_LAPACKE_cgeevx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeevx_work;
    }
    return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgehrd = 0;
#endif

/* Wrapper for LAPACKE_cgehrd */
lapack_int  LAPACKE_cgehrd (int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgehrd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgehrd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgehrd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, ilo, ihi, a, lda, tau);
    } else {
        return fn(matrix_layout, n, ilo, ihi, a, lda, tau);
    }
    #else
    return fn(matrix_layout, n, ilo, ihi, a, lda, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgehrd(int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgehrd;
    return fn(matrix_layout, n, ilo, ihi, a, lda, tau);
}

lapack_int flexiblas_chain_LAPACKE_cgehrd(int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgehrd++;
    if ( hook_pos_LAPACKE_cgehrd < __flexiblas_hooks->LAPACKE_cgehrd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgehrd.hook_function[hook_pos_LAPACKE_cgehrd];
    } else {
        hook_pos_LAPACKE_cgehrd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgehrd;
    }
    return fn(matrix_layout, n, ilo, ihi, a, lda, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgehrd_work = 0;
#endif

/* Wrapper for LAPACKE_cgehrd_work */
lapack_int  LAPACKE_cgehrd_work (int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgehrd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgehrd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgehrd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, ilo, ihi, a, lda, tau, work, lwork);
    } else {
        return fn(matrix_layout, n, ilo, ihi, a, lda, tau, work, lwork);
    }
    #else
    return fn(matrix_layout, n, ilo, ihi, a, lda, tau, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgehrd_work(int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgehrd_work;
    return fn(matrix_layout, n, ilo, ihi, a, lda, tau, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgehrd_work(int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgehrd_work++;
    if ( hook_pos_LAPACKE_cgehrd_work < __flexiblas_hooks->LAPACKE_cgehrd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgehrd_work.hook_function[hook_pos_LAPACKE_cgehrd_work];
    } else {
        hook_pos_LAPACKE_cgehrd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgehrd_work;
    }
    return fn(matrix_layout, n, ilo, ihi, a, lda, tau, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgejsv = 0;
#endif

/* Wrapper for LAPACKE_cgejsv */
lapack_int  LAPACKE_cgejsv (int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *sva, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, float *stat, lapack_int *istat)
{
    lapack_int (*fn) (int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, void *u, lapack_int ldu, void *v, lapack_int ldv, void *stat, void *istat);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, void *u, lapack_int ldu, void *v, lapack_int ldv, void *stat, void *istat);
    hook_pos_LAPACKE_cgejsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgejsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgejsv.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, joba, jobu, jobv, jobr, jobt, jobp, m, n, a, lda, sva, u, ldu, v, ldv, stat, istat);
    } else {
        return fn(matrix_layout, joba, jobu, jobv, jobr, jobt, jobp, m, n, a, lda, sva, u, ldu, v, ldv, stat, istat);
    }
    #else
    return fn(matrix_layout, joba, jobu, jobv, jobr, jobt, jobp, m, n, a, lda, sva, u, ldu, v, ldv, stat, istat);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgejsv(int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *sva, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, float *stat, lapack_int *istat)
{
    lapack_int (*fn) (int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, void *u, lapack_int ldu, void *v, lapack_int ldv, void *stat, void *istat);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgejsv;
    return fn(matrix_layout, joba, jobu, jobv, jobr, jobt, jobp, m, n, a, lda, sva, u, ldu, v, ldv, stat, istat);
}

lapack_int flexiblas_chain_LAPACKE_cgejsv(int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, void *u, lapack_int ldu, void *v, lapack_int ldv, void *stat, void *istat)
{
    lapack_int (*fn) (int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, void *u, lapack_int ldu, void *v, lapack_int ldv, void *stat, void *istat);
    hook_pos_LAPACKE_cgejsv++;
    if ( hook_pos_LAPACKE_cgejsv < __flexiblas_hooks->LAPACKE_cgejsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgejsv.hook_function[hook_pos_LAPACKE_cgejsv];
    } else {
        hook_pos_LAPACKE_cgejsv = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgejsv;
    }
    return fn(matrix_layout, joba, jobu, jobv, jobr, jobt, jobp, m, n, a, lda, sva, u, ldu, v, ldv, stat, istat);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgejsv_work = 0;
#endif

/* Wrapper for LAPACKE_cgejsv_work */
lapack_int  LAPACKE_cgejsv_work (int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *sva, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *cwork, lapack_int lwork, float *work, lapack_int lrwork, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, void *u, lapack_int ldu, void *v, lapack_int ldv, void *cwork, lapack_int lwork, void *work, lapack_int lrwork, void *iwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, void *u, lapack_int ldu, void *v, lapack_int ldv, void *cwork, lapack_int lwork, void *work, lapack_int lrwork, void *iwork);
    hook_pos_LAPACKE_cgejsv_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgejsv_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgejsv_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, joba, jobu, jobv, jobr, jobt, jobp, m, n, a, lda, sva, u, ldu, v, ldv, cwork, lwork, work, lrwork, iwork);
    } else {
        return fn(matrix_layout, joba, jobu, jobv, jobr, jobt, jobp, m, n, a, lda, sva, u, ldu, v, ldv, cwork, lwork, work, lrwork, iwork);
    }
    #else
    return fn(matrix_layout, joba, jobu, jobv, jobr, jobt, jobp, m, n, a, lda, sva, u, ldu, v, ldv, cwork, lwork, work, lrwork, iwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgejsv_work(int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *sva, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *cwork, lapack_int lwork, float *work, lapack_int lrwork, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, void *u, lapack_int ldu, void *v, lapack_int ldv, void *cwork, lapack_int lwork, void *work, lapack_int lrwork, void *iwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgejsv_work;
    return fn(matrix_layout, joba, jobu, jobv, jobr, jobt, jobp, m, n, a, lda, sva, u, ldu, v, ldv, cwork, lwork, work, lrwork, iwork);
}

lapack_int flexiblas_chain_LAPACKE_cgejsv_work(int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, void *u, lapack_int ldu, void *v, lapack_int ldv, void *cwork, lapack_int lwork, void *work, lapack_int lrwork, void *iwork)
{
    lapack_int (*fn) (int matrix_layout, char joba, char jobu, char jobv, char jobr, char jobt, char jobp, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, void *u, lapack_int ldu, void *v, lapack_int ldv, void *cwork, lapack_int lwork, void *work, lapack_int lrwork, void *iwork);
    hook_pos_LAPACKE_cgejsv_work++;
    if ( hook_pos_LAPACKE_cgejsv_work < __flexiblas_hooks->LAPACKE_cgejsv_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgejsv_work.hook_function[hook_pos_LAPACKE_cgejsv_work];
    } else {
        hook_pos_LAPACKE_cgejsv_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgejsv_work;
    }
    return fn(matrix_layout, joba, jobu, jobv, jobr, jobt, jobp, m, n, a, lda, sva, u, ldu, v, ldv, cwork, lwork, work, lrwork, iwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgelq = 0;
#endif

/* Wrapper for LAPACKE_cgelq */
lapack_int  LAPACKE_cgelq (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int tsize)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize);
    hook_pos_LAPACKE_cgelq = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelq;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgelq.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, t, tsize);
    } else {
        return fn(matrix_layout, m, n, a, lda, t, tsize);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, t, tsize);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgelq(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int tsize)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelq;
    return fn(matrix_layout, m, n, a, lda, t, tsize);
}

lapack_int flexiblas_chain_LAPACKE_cgelq(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize);
    hook_pos_LAPACKE_cgelq++;
    if ( hook_pos_LAPACKE_cgelq < __flexiblas_hooks->LAPACKE_cgelq.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgelq.hook_function[hook_pos_LAPACKE_cgelq];
    } else {
        hook_pos_LAPACKE_cgelq = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgelq;
    }
    return fn(matrix_layout, m, n, a, lda, t, tsize);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgelq2 = 0;
#endif

/* Wrapper for LAPACKE_cgelq2 */
lapack_int  LAPACKE_cgelq2 (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgelq2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelq2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgelq2.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgelq2(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelq2;
    return fn(matrix_layout, m, n, a, lda, tau);
}

lapack_int flexiblas_chain_LAPACKE_cgelq2(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgelq2++;
    if ( hook_pos_LAPACKE_cgelq2 < __flexiblas_hooks->LAPACKE_cgelq2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgelq2.hook_function[hook_pos_LAPACKE_cgelq2];
    } else {
        hook_pos_LAPACKE_cgelq2 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgelq2;
    }
    return fn(matrix_layout, m, n, a, lda, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgelq2_work = 0;
#endif

/* Wrapper for LAPACKE_cgelq2_work */
lapack_int  LAPACKE_cgelq2_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work);
    hook_pos_LAPACKE_cgelq2_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelq2_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgelq2_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau, work);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau, work);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgelq2_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelq2_work;
    return fn(matrix_layout, m, n, a, lda, tau, work);
}

lapack_int flexiblas_chain_LAPACKE_cgelq2_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work);
    hook_pos_LAPACKE_cgelq2_work++;
    if ( hook_pos_LAPACKE_cgelq2_work < __flexiblas_hooks->LAPACKE_cgelq2_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgelq2_work.hook_function[hook_pos_LAPACKE_cgelq2_work];
    } else {
        hook_pos_LAPACKE_cgelq2_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgelq2_work;
    }
    return fn(matrix_layout, m, n, a, lda, tau, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgelq_work = 0;
#endif

/* Wrapper for LAPACKE_cgelq_work */
lapack_int  LAPACKE_cgelq_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int tsize, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgelq_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelq_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgelq_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, t, tsize, work, lwork);
    } else {
        return fn(matrix_layout, m, n, a, lda, t, tsize, work, lwork);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, t, tsize, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgelq_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int tsize, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelq_work;
    return fn(matrix_layout, m, n, a, lda, t, tsize, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgelq_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgelq_work++;
    if ( hook_pos_LAPACKE_cgelq_work < __flexiblas_hooks->LAPACKE_cgelq_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgelq_work.hook_function[hook_pos_LAPACKE_cgelq_work];
    } else {
        hook_pos_LAPACKE_cgelq_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgelq_work;
    }
    return fn(matrix_layout, m, n, a, lda, t, tsize, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgelqf = 0;
#endif

/* Wrapper for LAPACKE_cgelqf */
lapack_int  LAPACKE_cgelqf (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgelqf = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelqf;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgelqf.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgelqf(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelqf;
    return fn(matrix_layout, m, n, a, lda, tau);
}

lapack_int flexiblas_chain_LAPACKE_cgelqf(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgelqf++;
    if ( hook_pos_LAPACKE_cgelqf < __flexiblas_hooks->LAPACKE_cgelqf.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgelqf.hook_function[hook_pos_LAPACKE_cgelqf];
    } else {
        hook_pos_LAPACKE_cgelqf = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgelqf;
    }
    return fn(matrix_layout, m, n, a, lda, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgelqf_work = 0;
#endif

/* Wrapper for LAPACKE_cgelqf_work */
lapack_int  LAPACKE_cgelqf_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgelqf_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelqf_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgelqf_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau, work, lwork);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgelqf_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelqf_work;
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgelqf_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgelqf_work++;
    if ( hook_pos_LAPACKE_cgelqf_work < __flexiblas_hooks->LAPACKE_cgelqf_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgelqf_work.hook_function[hook_pos_LAPACKE_cgelqf_work];
    } else {
        hook_pos_LAPACKE_cgelqf_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgelqf_work;
    }
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgels = 0;
#endif

/* Wrapper for LAPACKE_cgels */
lapack_int  LAPACKE_cgels (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgels = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgels;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgels.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb);
    } else {
        return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb);
    }
    #else
    return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgels(int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgels;
    return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgels(int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgels++;
    if ( hook_pos_LAPACKE_cgels < __flexiblas_hooks->LAPACKE_cgels.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgels.hook_function[hook_pos_LAPACKE_cgels];
    } else {
        hook_pos_LAPACKE_cgels = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgels;
    }
    return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgels_work = 0;
#endif

/* Wrapper for LAPACKE_cgels_work */
lapack_int  LAPACKE_cgels_work (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgels_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgels_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgels_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb, work, lwork);
    } else {
        return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb, work, lwork);
    }
    #else
    return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgels_work(int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgels_work;
    return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgels_work(int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgels_work++;
    if ( hook_pos_LAPACKE_cgels_work < __flexiblas_hooks->LAPACKE_cgels_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgels_work.hook_function[hook_pos_LAPACKE_cgels_work];
    } else {
        hook_pos_LAPACKE_cgels_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgels_work;
    }
    return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgelsd = 0;
#endif

/* Wrapper for LAPACKE_cgelsd */
lapack_int  LAPACKE_cgelsd (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *s, float rcond, lapack_int *rank)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank);
    hook_pos_LAPACKE_cgelsd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelsd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgelsd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank);
    } else {
        return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank);
    }
    #else
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgelsd(int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *s, float rcond, lapack_int *rank)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelsd;
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank);
}

lapack_int flexiblas_chain_LAPACKE_cgelsd(int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank);
    hook_pos_LAPACKE_cgelsd++;
    if ( hook_pos_LAPACKE_cgelsd < __flexiblas_hooks->LAPACKE_cgelsd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgelsd.hook_function[hook_pos_LAPACKE_cgelsd];
    } else {
        hook_pos_LAPACKE_cgelsd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgelsd;
    }
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgelsd_work = 0;
#endif

/* Wrapper for LAPACKE_cgelsd_work */
lapack_int  LAPACKE_cgelsd_work (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *s, float rcond, lapack_int *rank, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank, void *work, lapack_int lwork, void *rwork, void *iwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank, void *work, lapack_int lwork, void *rwork, void *iwork);
    hook_pos_LAPACKE_cgelsd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelsd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgelsd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank, work, lwork, rwork, iwork);
    } else {
        return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank, work, lwork, rwork, iwork);
    }
    #else
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank, work, lwork, rwork, iwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgelsd_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *s, float rcond, lapack_int *rank, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank, void *work, lapack_int lwork, void *rwork, void *iwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelsd_work;
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank, work, lwork, rwork, iwork);
}

lapack_int flexiblas_chain_LAPACKE_cgelsd_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank, void *work, lapack_int lwork, void *rwork, void *iwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank, void *work, lapack_int lwork, void *rwork, void *iwork);
    hook_pos_LAPACKE_cgelsd_work++;
    if ( hook_pos_LAPACKE_cgelsd_work < __flexiblas_hooks->LAPACKE_cgelsd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgelsd_work.hook_function[hook_pos_LAPACKE_cgelsd_work];
    } else {
        hook_pos_LAPACKE_cgelsd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgelsd_work;
    }
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank, work, lwork, rwork, iwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgelss = 0;
#endif

/* Wrapper for LAPACKE_cgelss */
lapack_int  LAPACKE_cgelss (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *s, float rcond, lapack_int *rank)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank);
    hook_pos_LAPACKE_cgelss = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelss;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgelss.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank);
    } else {
        return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank);
    }
    #else
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgelss(int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *s, float rcond, lapack_int *rank)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelss;
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank);
}

lapack_int flexiblas_chain_LAPACKE_cgelss(int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank);
    hook_pos_LAPACKE_cgelss++;
    if ( hook_pos_LAPACKE_cgelss < __flexiblas_hooks->LAPACKE_cgelss.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgelss.hook_function[hook_pos_LAPACKE_cgelss];
    } else {
        hook_pos_LAPACKE_cgelss = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgelss;
    }
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgelss_work = 0;
#endif

/* Wrapper for LAPACKE_cgelss_work */
lapack_int  LAPACKE_cgelss_work (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *s, float rcond, lapack_int *rank, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cgelss_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelss_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgelss_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank, work, lwork, rwork);
    } else {
        return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgelss_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *s, float rcond, lapack_int *rank, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelss_work;
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgelss_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *s, float rcond, void *rank, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cgelss_work++;
    if ( hook_pos_LAPACKE_cgelss_work < __flexiblas_hooks->LAPACKE_cgelss_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgelss_work.hook_function[hook_pos_LAPACKE_cgelss_work];
    } else {
        hook_pos_LAPACKE_cgelss_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgelss_work;
    }
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, s, rcond, rank, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgelsy = 0;
#endif

/* Wrapper for LAPACKE_cgelsy */
lapack_int  LAPACKE_cgelsy (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *jpvt, float rcond, lapack_int *rank)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *jpvt, float rcond, void *rank);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *jpvt, float rcond, void *rank);
    hook_pos_LAPACKE_cgelsy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelsy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgelsy.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, nrhs, a, lda, b, ldb, jpvt, rcond, rank);
    } else {
        return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, jpvt, rcond, rank);
    }
    #else
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, jpvt, rcond, rank);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgelsy(int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *jpvt, float rcond, lapack_int *rank)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *jpvt, float rcond, void *rank);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelsy;
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, jpvt, rcond, rank);
}

lapack_int flexiblas_chain_LAPACKE_cgelsy(int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *jpvt, float rcond, void *rank)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *jpvt, float rcond, void *rank);
    hook_pos_LAPACKE_cgelsy++;
    if ( hook_pos_LAPACKE_cgelsy < __flexiblas_hooks->LAPACKE_cgelsy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgelsy.hook_function[hook_pos_LAPACKE_cgelsy];
    } else {
        hook_pos_LAPACKE_cgelsy = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgelsy;
    }
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, jpvt, rcond, rank);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgelsy_work = 0;
#endif

/* Wrapper for LAPACKE_cgelsy_work */
lapack_int  LAPACKE_cgelsy_work (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *jpvt, float rcond, lapack_int *rank, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *jpvt, float rcond, void *rank, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *jpvt, float rcond, void *rank, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cgelsy_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelsy_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgelsy_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, nrhs, a, lda, b, ldb, jpvt, rcond, rank, work, lwork, rwork);
    } else {
        return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, jpvt, rcond, rank, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, jpvt, rcond, rank, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgelsy_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *jpvt, float rcond, lapack_int *rank, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *jpvt, float rcond, void *rank, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgelsy_work;
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, jpvt, rcond, rank, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgelsy_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *jpvt, float rcond, void *rank, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *jpvt, float rcond, void *rank, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cgelsy_work++;
    if ( hook_pos_LAPACKE_cgelsy_work < __flexiblas_hooks->LAPACKE_cgelsy_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgelsy_work.hook_function[hook_pos_LAPACKE_cgelsy_work];
    } else {
        hook_pos_LAPACKE_cgelsy_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgelsy_work;
    }
    return fn(matrix_layout, m, n, nrhs, a, lda, b, ldb, jpvt, rcond, rank, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgemlq = 0;
#endif

/* Wrapper for LAPACKE_cgemlq */
lapack_int  LAPACKE_cgemlq (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *t, lapack_int tsize, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc);
    hook_pos_LAPACKE_cgemlq = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgemlq;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgemlq.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc);
    } else {
        return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc);
    }
    #else
    return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgemlq(int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *t, lapack_int tsize, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgemlq;
    return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc);
}

lapack_int flexiblas_chain_LAPACKE_cgemlq(int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc);
    hook_pos_LAPACKE_cgemlq++;
    if ( hook_pos_LAPACKE_cgemlq < __flexiblas_hooks->LAPACKE_cgemlq.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgemlq.hook_function[hook_pos_LAPACKE_cgemlq];
    } else {
        hook_pos_LAPACKE_cgemlq = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgemlq;
    }
    return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgemlq_work = 0;
#endif

/* Wrapper for LAPACKE_cgemlq_work */
lapack_int  LAPACKE_cgemlq_work (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *t, lapack_int tsize, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgemlq_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgemlq_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgemlq_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc, work, lwork);
    } else {
        return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc, work, lwork);
    }
    #else
    return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgemlq_work(int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *t, lapack_int tsize, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgemlq_work;
    return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgemlq_work(int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgemlq_work++;
    if ( hook_pos_LAPACKE_cgemlq_work < __flexiblas_hooks->LAPACKE_cgemlq_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgemlq_work.hook_function[hook_pos_LAPACKE_cgemlq_work];
    } else {
        hook_pos_LAPACKE_cgemlq_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgemlq_work;
    }
    return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgemqr = 0;
#endif

/* Wrapper for LAPACKE_cgemqr */
lapack_int  LAPACKE_cgemqr (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *t, lapack_int tsize, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc);
    hook_pos_LAPACKE_cgemqr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgemqr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgemqr.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc);
    } else {
        return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc);
    }
    #else
    return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgemqr(int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *t, lapack_int tsize, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgemqr;
    return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc);
}

lapack_int flexiblas_chain_LAPACKE_cgemqr(int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc);
    hook_pos_LAPACKE_cgemqr++;
    if ( hook_pos_LAPACKE_cgemqr < __flexiblas_hooks->LAPACKE_cgemqr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgemqr.hook_function[hook_pos_LAPACKE_cgemqr];
    } else {
        hook_pos_LAPACKE_cgemqr = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgemqr;
    }
    return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgemqr_work = 0;
#endif

/* Wrapper for LAPACKE_cgemqr_work */
lapack_int  LAPACKE_cgemqr_work (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *t, lapack_int tsize, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgemqr_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgemqr_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgemqr_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc, work, lwork);
    } else {
        return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc, work, lwork);
    }
    #else
    return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgemqr_work(int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *t, lapack_int tsize, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgemqr_work;
    return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgemqr_work(int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, const void *a, lapack_int lda, const void *t, lapack_int tsize, void *c, lapack_int ldc, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgemqr_work++;
    if ( hook_pos_LAPACKE_cgemqr_work < __flexiblas_hooks->LAPACKE_cgemqr_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgemqr_work.hook_function[hook_pos_LAPACKE_cgemqr_work];
    } else {
        hook_pos_LAPACKE_cgemqr_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgemqr_work;
    }
    return fn(matrix_layout, side, trans, m, n, k, a, lda, t, tsize, c, ldc, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgemqrt = 0;
#endif

/* Wrapper for LAPACKE_cgemqrt */
lapack_int  LAPACKE_cgemqrt (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const lapack_complex_float *v, lapack_int ldv, const lapack_complex_float *t, lapack_int ldt, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc);
    hook_pos_LAPACKE_cgemqrt = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgemqrt;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgemqrt.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, side, trans, m, n, k, nb, v, ldv, t, ldt, c, ldc);
    } else {
        return fn(matrix_layout, side, trans, m, n, k, nb, v, ldv, t, ldt, c, ldc);
    }
    #else
    return fn(matrix_layout, side, trans, m, n, k, nb, v, ldv, t, ldt, c, ldc);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgemqrt(int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const lapack_complex_float *v, lapack_int ldv, const lapack_complex_float *t, lapack_int ldt, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgemqrt;
    return fn(matrix_layout, side, trans, m, n, k, nb, v, ldv, t, ldt, c, ldc);
}

lapack_int flexiblas_chain_LAPACKE_cgemqrt(int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc);
    hook_pos_LAPACKE_cgemqrt++;
    if ( hook_pos_LAPACKE_cgemqrt < __flexiblas_hooks->LAPACKE_cgemqrt.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgemqrt.hook_function[hook_pos_LAPACKE_cgemqrt];
    } else {
        hook_pos_LAPACKE_cgemqrt = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgemqrt;
    }
    return fn(matrix_layout, side, trans, m, n, k, nb, v, ldv, t, ldt, c, ldc);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgemqrt_work = 0;
#endif

/* Wrapper for LAPACKE_cgemqrt_work */
lapack_int  LAPACKE_cgemqrt_work (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const lapack_complex_float *v, lapack_int ldv, const lapack_complex_float *t, lapack_int ldt, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc, void *work);
    hook_pos_LAPACKE_cgemqrt_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgemqrt_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgemqrt_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, side, trans, m, n, k, nb, v, ldv, t, ldt, c, ldc, work);
    } else {
        return fn(matrix_layout, side, trans, m, n, k, nb, v, ldv, t, ldt, c, ldc, work);
    }
    #else
    return fn(matrix_layout, side, trans, m, n, k, nb, v, ldv, t, ldt, c, ldc, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgemqrt_work(int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const lapack_complex_float *v, lapack_int ldv, const lapack_complex_float *t, lapack_int ldt, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgemqrt_work;
    return fn(matrix_layout, side, trans, m, n, k, nb, v, ldv, t, ldt, c, ldc, work);
}

lapack_int flexiblas_chain_LAPACKE_cgemqrt_work(int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc, void *work)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, lapack_int m, lapack_int n, lapack_int k, lapack_int nb, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc, void *work);
    hook_pos_LAPACKE_cgemqrt_work++;
    if ( hook_pos_LAPACKE_cgemqrt_work < __flexiblas_hooks->LAPACKE_cgemqrt_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgemqrt_work.hook_function[hook_pos_LAPACKE_cgemqrt_work];
    } else {
        hook_pos_LAPACKE_cgemqrt_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgemqrt_work;
    }
    return fn(matrix_layout, side, trans, m, n, k, nb, v, ldv, t, ldt, c, ldc, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqlf = 0;
#endif

/* Wrapper for LAPACKE_cgeqlf */
lapack_int  LAPACKE_cgeqlf (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgeqlf = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqlf;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqlf.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqlf(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqlf;
    return fn(matrix_layout, m, n, a, lda, tau);
}

lapack_int flexiblas_chain_LAPACKE_cgeqlf(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgeqlf++;
    if ( hook_pos_LAPACKE_cgeqlf < __flexiblas_hooks->LAPACKE_cgeqlf.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqlf.hook_function[hook_pos_LAPACKE_cgeqlf];
    } else {
        hook_pos_LAPACKE_cgeqlf = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqlf;
    }
    return fn(matrix_layout, m, n, a, lda, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqlf_work = 0;
#endif

/* Wrapper for LAPACKE_cgeqlf_work */
lapack_int  LAPACKE_cgeqlf_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgeqlf_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqlf_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqlf_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau, work, lwork);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqlf_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqlf_work;
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgeqlf_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgeqlf_work++;
    if ( hook_pos_LAPACKE_cgeqlf_work < __flexiblas_hooks->LAPACKE_cgeqlf_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqlf_work.hook_function[hook_pos_LAPACKE_cgeqlf_work];
    } else {
        hook_pos_LAPACKE_cgeqlf_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqlf_work;
    }
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqp3 = 0;
#endif

/* Wrapper for LAPACKE_cgeqp3 */
lapack_int  LAPACKE_cgeqp3 (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *jpvt, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau);
    hook_pos_LAPACKE_cgeqp3 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqp3;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqp3.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, jpvt, tau);
    } else {
        return fn(matrix_layout, m, n, a, lda, jpvt, tau);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, jpvt, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqp3(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *jpvt, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqp3;
    return fn(matrix_layout, m, n, a, lda, jpvt, tau);
}

lapack_int flexiblas_chain_LAPACKE_cgeqp3(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau);
    hook_pos_LAPACKE_cgeqp3++;
    if ( hook_pos_LAPACKE_cgeqp3 < __flexiblas_hooks->LAPACKE_cgeqp3.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqp3.hook_function[hook_pos_LAPACKE_cgeqp3];
    } else {
        hook_pos_LAPACKE_cgeqp3 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqp3;
    }
    return fn(matrix_layout, m, n, a, lda, jpvt, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqp3_work = 0;
#endif

/* Wrapper for LAPACKE_cgeqp3_work */
lapack_int  LAPACKE_cgeqp3_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *jpvt, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cgeqp3_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqp3_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqp3_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, jpvt, tau, work, lwork, rwork);
    } else {
        return fn(matrix_layout, m, n, a, lda, jpvt, tau, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, jpvt, tau, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqp3_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *jpvt, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqp3_work;
    return fn(matrix_layout, m, n, a, lda, jpvt, tau, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgeqp3_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cgeqp3_work++;
    if ( hook_pos_LAPACKE_cgeqp3_work < __flexiblas_hooks->LAPACKE_cgeqp3_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqp3_work.hook_function[hook_pos_LAPACKE_cgeqp3_work];
    } else {
        hook_pos_LAPACKE_cgeqp3_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqp3_work;
    }
    return fn(matrix_layout, m, n, a, lda, jpvt, tau, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqpf = 0;
#endif

/* Wrapper for LAPACKE_cgeqpf */
lapack_int  LAPACKE_cgeqpf (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *jpvt, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau);
    hook_pos_LAPACKE_cgeqpf = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqpf;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqpf.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, jpvt, tau);
    } else {
        return fn(matrix_layout, m, n, a, lda, jpvt, tau);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, jpvt, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqpf(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *jpvt, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqpf;
    return fn(matrix_layout, m, n, a, lda, jpvt, tau);
}

lapack_int flexiblas_chain_LAPACKE_cgeqpf(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau);
    hook_pos_LAPACKE_cgeqpf++;
    if ( hook_pos_LAPACKE_cgeqpf < __flexiblas_hooks->LAPACKE_cgeqpf.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqpf.hook_function[hook_pos_LAPACKE_cgeqpf];
    } else {
        hook_pos_LAPACKE_cgeqpf = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqpf;
    }
    return fn(matrix_layout, m, n, a, lda, jpvt, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqpf_work = 0;
#endif

/* Wrapper for LAPACKE_cgeqpf_work */
lapack_int  LAPACKE_cgeqpf_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *jpvt, lapack_complex_float *tau, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau, void *work, void *rwork);
    hook_pos_LAPACKE_cgeqpf_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqpf_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqpf_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, jpvt, tau, work, rwork);
    } else {
        return fn(matrix_layout, m, n, a, lda, jpvt, tau, work, rwork);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, jpvt, tau, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqpf_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *jpvt, lapack_complex_float *tau, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqpf_work;
    return fn(matrix_layout, m, n, a, lda, jpvt, tau, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgeqpf_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *jpvt, void *tau, void *work, void *rwork);
    hook_pos_LAPACKE_cgeqpf_work++;
    if ( hook_pos_LAPACKE_cgeqpf_work < __flexiblas_hooks->LAPACKE_cgeqpf_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqpf_work.hook_function[hook_pos_LAPACKE_cgeqpf_work];
    } else {
        hook_pos_LAPACKE_cgeqpf_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqpf_work;
    }
    return fn(matrix_layout, m, n, a, lda, jpvt, tau, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqr = 0;
#endif

/* Wrapper for LAPACKE_cgeqr */
lapack_int  LAPACKE_cgeqr (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int tsize)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize);
    hook_pos_LAPACKE_cgeqr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqr.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, t, tsize);
    } else {
        return fn(matrix_layout, m, n, a, lda, t, tsize);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, t, tsize);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqr(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int tsize)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqr;
    return fn(matrix_layout, m, n, a, lda, t, tsize);
}

lapack_int flexiblas_chain_LAPACKE_cgeqr(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize);
    hook_pos_LAPACKE_cgeqr++;
    if ( hook_pos_LAPACKE_cgeqr < __flexiblas_hooks->LAPACKE_cgeqr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqr.hook_function[hook_pos_LAPACKE_cgeqr];
    } else {
        hook_pos_LAPACKE_cgeqr = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqr;
    }
    return fn(matrix_layout, m, n, a, lda, t, tsize);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqr2 = 0;
#endif

/* Wrapper for LAPACKE_cgeqr2 */
lapack_int  LAPACKE_cgeqr2 (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgeqr2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqr2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqr2.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqr2(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqr2;
    return fn(matrix_layout, m, n, a, lda, tau);
}

lapack_int flexiblas_chain_LAPACKE_cgeqr2(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgeqr2++;
    if ( hook_pos_LAPACKE_cgeqr2 < __flexiblas_hooks->LAPACKE_cgeqr2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqr2.hook_function[hook_pos_LAPACKE_cgeqr2];
    } else {
        hook_pos_LAPACKE_cgeqr2 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqr2;
    }
    return fn(matrix_layout, m, n, a, lda, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqr2_work = 0;
#endif

/* Wrapper for LAPACKE_cgeqr2_work */
lapack_int  LAPACKE_cgeqr2_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work);
    hook_pos_LAPACKE_cgeqr2_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqr2_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqr2_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau, work);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau, work);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqr2_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqr2_work;
    return fn(matrix_layout, m, n, a, lda, tau, work);
}

lapack_int flexiblas_chain_LAPACKE_cgeqr2_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work);
    hook_pos_LAPACKE_cgeqr2_work++;
    if ( hook_pos_LAPACKE_cgeqr2_work < __flexiblas_hooks->LAPACKE_cgeqr2_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqr2_work.hook_function[hook_pos_LAPACKE_cgeqr2_work];
    } else {
        hook_pos_LAPACKE_cgeqr2_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqr2_work;
    }
    return fn(matrix_layout, m, n, a, lda, tau, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqr_work = 0;
#endif

/* Wrapper for LAPACKE_cgeqr_work */
lapack_int  LAPACKE_cgeqr_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int tsize, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgeqr_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqr_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqr_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, t, tsize, work, lwork);
    } else {
        return fn(matrix_layout, m, n, a, lda, t, tsize, work, lwork);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, t, tsize, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqr_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int tsize, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqr_work;
    return fn(matrix_layout, m, n, a, lda, t, tsize, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgeqr_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int tsize, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgeqr_work++;
    if ( hook_pos_LAPACKE_cgeqr_work < __flexiblas_hooks->LAPACKE_cgeqr_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqr_work.hook_function[hook_pos_LAPACKE_cgeqr_work];
    } else {
        hook_pos_LAPACKE_cgeqr_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqr_work;
    }
    return fn(matrix_layout, m, n, a, lda, t, tsize, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqrf = 0;
#endif

/* Wrapper for LAPACKE_cgeqrf */
lapack_int  LAPACKE_cgeqrf (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgeqrf = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrf;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqrf.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqrf(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrf;
    return fn(matrix_layout, m, n, a, lda, tau);
}

lapack_int flexiblas_chain_LAPACKE_cgeqrf(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgeqrf++;
    if ( hook_pos_LAPACKE_cgeqrf < __flexiblas_hooks->LAPACKE_cgeqrf.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqrf.hook_function[hook_pos_LAPACKE_cgeqrf];
    } else {
        hook_pos_LAPACKE_cgeqrf = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrf;
    }
    return fn(matrix_layout, m, n, a, lda, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqrf_work = 0;
#endif

/* Wrapper for LAPACKE_cgeqrf_work */
lapack_int  LAPACKE_cgeqrf_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgeqrf_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrf_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqrf_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau, work, lwork);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqrf_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrf_work;
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgeqrf_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgeqrf_work++;
    if ( hook_pos_LAPACKE_cgeqrf_work < __flexiblas_hooks->LAPACKE_cgeqrf_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqrf_work.hook_function[hook_pos_LAPACKE_cgeqrf_work];
    } else {
        hook_pos_LAPACKE_cgeqrf_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrf_work;
    }
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqrfp = 0;
#endif

/* Wrapper for LAPACKE_cgeqrfp */
lapack_int  LAPACKE_cgeqrfp (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgeqrfp = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrfp;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqrfp.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqrfp(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrfp;
    return fn(matrix_layout, m, n, a, lda, tau);
}

lapack_int flexiblas_chain_LAPACKE_cgeqrfp(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgeqrfp++;
    if ( hook_pos_LAPACKE_cgeqrfp < __flexiblas_hooks->LAPACKE_cgeqrfp.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqrfp.hook_function[hook_pos_LAPACKE_cgeqrfp];
    } else {
        hook_pos_LAPACKE_cgeqrfp = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrfp;
    }
    return fn(matrix_layout, m, n, a, lda, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqrfp_work = 0;
#endif

/* Wrapper for LAPACKE_cgeqrfp_work */
lapack_int  LAPACKE_cgeqrfp_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgeqrfp_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrfp_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqrfp_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau, work, lwork);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqrfp_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrfp_work;
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgeqrfp_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgeqrfp_work++;
    if ( hook_pos_LAPACKE_cgeqrfp_work < __flexiblas_hooks->LAPACKE_cgeqrfp_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqrfp_work.hook_function[hook_pos_LAPACKE_cgeqrfp_work];
    } else {
        hook_pos_LAPACKE_cgeqrfp_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrfp_work;
    }
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqrt = 0;
#endif

/* Wrapper for LAPACKE_cgeqrt */
lapack_int  LAPACKE_cgeqrt (int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, void *a, lapack_int lda, void *t, lapack_int ldt);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, void *a, lapack_int lda, void *t, lapack_int ldt);
    hook_pos_LAPACKE_cgeqrt = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqrt.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, nb, a, lda, t, ldt);
    } else {
        return fn(matrix_layout, m, n, nb, a, lda, t, ldt);
    }
    #else
    return fn(matrix_layout, m, n, nb, a, lda, t, ldt);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqrt(int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, void *a, lapack_int lda, void *t, lapack_int ldt);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt;
    return fn(matrix_layout, m, n, nb, a, lda, t, ldt);
}

lapack_int flexiblas_chain_LAPACKE_cgeqrt(int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, void *a, lapack_int lda, void *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, void *a, lapack_int lda, void *t, lapack_int ldt);
    hook_pos_LAPACKE_cgeqrt++;
    if ( hook_pos_LAPACKE_cgeqrt < __flexiblas_hooks->LAPACKE_cgeqrt.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqrt.hook_function[hook_pos_LAPACKE_cgeqrt];
    } else {
        hook_pos_LAPACKE_cgeqrt = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt;
    }
    return fn(matrix_layout, m, n, nb, a, lda, t, ldt);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqrt2 = 0;
#endif

/* Wrapper for LAPACKE_cgeqrt2 */
lapack_int  LAPACKE_cgeqrt2 (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    hook_pos_LAPACKE_cgeqrt2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqrt2.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, t, ldt);
    } else {
        return fn(matrix_layout, m, n, a, lda, t, ldt);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, t, ldt);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqrt2(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt2;
    return fn(matrix_layout, m, n, a, lda, t, ldt);
}

lapack_int flexiblas_chain_LAPACKE_cgeqrt2(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    hook_pos_LAPACKE_cgeqrt2++;
    if ( hook_pos_LAPACKE_cgeqrt2 < __flexiblas_hooks->LAPACKE_cgeqrt2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqrt2.hook_function[hook_pos_LAPACKE_cgeqrt2];
    } else {
        hook_pos_LAPACKE_cgeqrt2 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt2;
    }
    return fn(matrix_layout, m, n, a, lda, t, ldt);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqrt2_work = 0;
#endif

/* Wrapper for LAPACKE_cgeqrt2_work */
lapack_int  LAPACKE_cgeqrt2_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    hook_pos_LAPACKE_cgeqrt2_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt2_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqrt2_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, t, ldt);
    } else {
        return fn(matrix_layout, m, n, a, lda, t, ldt);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, t, ldt);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqrt2_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt2_work;
    return fn(matrix_layout, m, n, a, lda, t, ldt);
}

lapack_int flexiblas_chain_LAPACKE_cgeqrt2_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    hook_pos_LAPACKE_cgeqrt2_work++;
    if ( hook_pos_LAPACKE_cgeqrt2_work < __flexiblas_hooks->LAPACKE_cgeqrt2_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqrt2_work.hook_function[hook_pos_LAPACKE_cgeqrt2_work];
    } else {
        hook_pos_LAPACKE_cgeqrt2_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt2_work;
    }
    return fn(matrix_layout, m, n, a, lda, t, ldt);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqrt3 = 0;
#endif

/* Wrapper for LAPACKE_cgeqrt3 */
lapack_int  LAPACKE_cgeqrt3 (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    hook_pos_LAPACKE_cgeqrt3 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt3;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqrt3.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, t, ldt);
    } else {
        return fn(matrix_layout, m, n, a, lda, t, ldt);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, t, ldt);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqrt3(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt3;
    return fn(matrix_layout, m, n, a, lda, t, ldt);
}

lapack_int flexiblas_chain_LAPACKE_cgeqrt3(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    hook_pos_LAPACKE_cgeqrt3++;
    if ( hook_pos_LAPACKE_cgeqrt3 < __flexiblas_hooks->LAPACKE_cgeqrt3.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqrt3.hook_function[hook_pos_LAPACKE_cgeqrt3];
    } else {
        hook_pos_LAPACKE_cgeqrt3 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt3;
    }
    return fn(matrix_layout, m, n, a, lda, t, ldt);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqrt3_work = 0;
#endif

/* Wrapper for LAPACKE_cgeqrt3_work */
lapack_int  LAPACKE_cgeqrt3_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    hook_pos_LAPACKE_cgeqrt3_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt3_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqrt3_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, t, ldt);
    } else {
        return fn(matrix_layout, m, n, a, lda, t, ldt);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, t, ldt);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqrt3_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt3_work;
    return fn(matrix_layout, m, n, a, lda, t, ldt);
}

lapack_int flexiblas_chain_LAPACKE_cgeqrt3_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *t, lapack_int ldt);
    hook_pos_LAPACKE_cgeqrt3_work++;
    if ( hook_pos_LAPACKE_cgeqrt3_work < __flexiblas_hooks->LAPACKE_cgeqrt3_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqrt3_work.hook_function[hook_pos_LAPACKE_cgeqrt3_work];
    } else {
        hook_pos_LAPACKE_cgeqrt3_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt3_work;
    }
    return fn(matrix_layout, m, n, a, lda, t, ldt);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgeqrt_work = 0;
#endif

/* Wrapper for LAPACKE_cgeqrt_work */
lapack_int  LAPACKE_cgeqrt_work (int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int ldt, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, void *a, lapack_int lda, void *t, lapack_int ldt, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, void *a, lapack_int lda, void *t, lapack_int ldt, void *work);
    hook_pos_LAPACKE_cgeqrt_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgeqrt_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, nb, a, lda, t, ldt, work);
    } else {
        return fn(matrix_layout, m, n, nb, a, lda, t, ldt, work);
    }
    #else
    return fn(matrix_layout, m, n, nb, a, lda, t, ldt, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgeqrt_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, lapack_complex_float *a, lapack_int lda, lapack_complex_float *t, lapack_int ldt, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, void *a, lapack_int lda, void *t, lapack_int ldt, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt_work;
    return fn(matrix_layout, m, n, nb, a, lda, t, ldt, work);
}

lapack_int flexiblas_chain_LAPACKE_cgeqrt_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, void *a, lapack_int lda, void *t, lapack_int ldt, void *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int nb, void *a, lapack_int lda, void *t, lapack_int ldt, void *work);
    hook_pos_LAPACKE_cgeqrt_work++;
    if ( hook_pos_LAPACKE_cgeqrt_work < __flexiblas_hooks->LAPACKE_cgeqrt_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgeqrt_work.hook_function[hook_pos_LAPACKE_cgeqrt_work];
    } else {
        hook_pos_LAPACKE_cgeqrt_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgeqrt_work;
    }
    return fn(matrix_layout, m, n, nb, a, lda, t, ldt, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgerfs = 0;
#endif

/* Wrapper for LAPACKE_cgerfs */
lapack_int  LAPACKE_cgerfs (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *af, lapack_int ldaf, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    hook_pos_LAPACKE_cgerfs = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgerfs;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgerfs.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
    } else {
        return fn(matrix_layout, trans, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
    }
    #else
    return fn(matrix_layout, trans, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgerfs(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *af, lapack_int ldaf, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgerfs;
    return fn(matrix_layout, trans, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

lapack_int flexiblas_chain_LAPACKE_cgerfs(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    hook_pos_LAPACKE_cgerfs++;
    if ( hook_pos_LAPACKE_cgerfs < __flexiblas_hooks->LAPACKE_cgerfs.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgerfs.hook_function[hook_pos_LAPACKE_cgerfs];
    } else {
        hook_pos_LAPACKE_cgerfs = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgerfs;
    }
    return fn(matrix_layout, trans, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgerfs_work = 0;
#endif

/* Wrapper for LAPACKE_cgerfs_work */
lapack_int  LAPACKE_cgerfs_work (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *af, lapack_int ldaf, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cgerfs_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgerfs_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgerfs_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    } else {
        return fn(matrix_layout, trans, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    }
    #else
    return fn(matrix_layout, trans, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgerfs_work(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *af, lapack_int ldaf, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgerfs_work;
    return fn(matrix_layout, trans, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgerfs_work(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cgerfs_work++;
    if ( hook_pos_LAPACKE_cgerfs_work < __flexiblas_hooks->LAPACKE_cgerfs_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgerfs_work.hook_function[hook_pos_LAPACKE_cgerfs_work];
    } else {
        hook_pos_LAPACKE_cgerfs_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgerfs_work;
    }
    return fn(matrix_layout, trans, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgerqf = 0;
#endif

/* Wrapper for LAPACKE_cgerqf */
lapack_int  LAPACKE_cgerqf (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgerqf = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgerqf;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgerqf.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgerqf(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgerqf;
    return fn(matrix_layout, m, n, a, lda, tau);
}

lapack_int flexiblas_chain_LAPACKE_cgerqf(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau);
    hook_pos_LAPACKE_cgerqf++;
    if ( hook_pos_LAPACKE_cgerqf < __flexiblas_hooks->LAPACKE_cgerqf.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgerqf.hook_function[hook_pos_LAPACKE_cgerqf];
    } else {
        hook_pos_LAPACKE_cgerqf = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgerqf;
    }
    return fn(matrix_layout, m, n, a, lda, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgerqf_work = 0;
#endif

/* Wrapper for LAPACKE_cgerqf_work */
lapack_int  LAPACKE_cgerqf_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgerqf_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgerqf_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgerqf_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, tau, work, lwork);
    } else {
        return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgerqf_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgerqf_work;
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgerqf_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgerqf_work++;
    if ( hook_pos_LAPACKE_cgerqf_work < __flexiblas_hooks->LAPACKE_cgerqf_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgerqf_work.hook_function[hook_pos_LAPACKE_cgerqf_work];
    } else {
        hook_pos_LAPACKE_cgerqf_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgerqf_work;
    }
    return fn(matrix_layout, m, n, a, lda, tau, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgesdd = 0;
#endif

/* Wrapper for LAPACKE_cgesdd */
lapack_int  LAPACKE_cgesdd (int matrix_layout, char jobz, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *s, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *vt, lapack_int ldvt)
{
    lapack_int (*fn) (int matrix_layout, char jobz, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt);
    hook_pos_LAPACKE_cgesdd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesdd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgesdd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, m, n, a, lda, s, u, ldu, vt, ldvt);
    } else {
        return fn(matrix_layout, jobz, m, n, a, lda, s, u, ldu, vt, ldvt);
    }
    #else
    return fn(matrix_layout, jobz, m, n, a, lda, s, u, ldu, vt, ldvt);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgesdd(int matrix_layout, char jobz, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *s, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *vt, lapack_int ldvt)
{
    lapack_int (*fn) (int matrix_layout, char jobz, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesdd;
    return fn(matrix_layout, jobz, m, n, a, lda, s, u, ldu, vt, ldvt);
}

lapack_int flexiblas_chain_LAPACKE_cgesdd(int matrix_layout, char jobz, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt)
{
    lapack_int (*fn) (int matrix_layout, char jobz, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt);
    hook_pos_LAPACKE_cgesdd++;
    if ( hook_pos_LAPACKE_cgesdd < __flexiblas_hooks->LAPACKE_cgesdd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgesdd.hook_function[hook_pos_LAPACKE_cgesdd];
    } else {
        hook_pos_LAPACKE_cgesdd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgesdd;
    }
    return fn(matrix_layout, jobz, m, n, a, lda, s, u, ldu, vt, ldvt);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgesdd_work = 0;
#endif

/* Wrapper for LAPACKE_cgesdd_work */
lapack_int  LAPACKE_cgesdd_work (int matrix_layout, char jobz, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *s, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *vt, lapack_int ldvt, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork, void *iwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork, void *iwork);
    hook_pos_LAPACKE_cgesdd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesdd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgesdd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork, iwork);
    } else {
        return fn(matrix_layout, jobz, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork, iwork);
    }
    #else
    return fn(matrix_layout, jobz, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork, iwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgesdd_work(int matrix_layout, char jobz, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *s, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *vt, lapack_int ldvt, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork, void *iwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesdd_work;
    return fn(matrix_layout, jobz, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork, iwork);
}

lapack_int flexiblas_chain_LAPACKE_cgesdd_work(int matrix_layout, char jobz, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork, void *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork, void *iwork);
    hook_pos_LAPACKE_cgesdd_work++;
    if ( hook_pos_LAPACKE_cgesdd_work < __flexiblas_hooks->LAPACKE_cgesdd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgesdd_work.hook_function[hook_pos_LAPACKE_cgesdd_work];
    } else {
        hook_pos_LAPACKE_cgesdd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgesdd_work;
    }
    return fn(matrix_layout, jobz, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork, iwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgesv = 0;
#endif

/* Wrapper for LAPACKE_cgesv */
lapack_int  LAPACKE_cgesv (int matrix_layout, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgesv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgesv.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, nrhs, a, lda, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, n, nrhs, a, lda, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, n, nrhs, a, lda, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgesv(int matrix_layout, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesv;
    return fn(matrix_layout, n, nrhs, a, lda, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgesv(int matrix_layout, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgesv++;
    if ( hook_pos_LAPACKE_cgesv < __flexiblas_hooks->LAPACKE_cgesv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgesv.hook_function[hook_pos_LAPACKE_cgesv];
    } else {
        hook_pos_LAPACKE_cgesv = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgesv;
    }
    return fn(matrix_layout, n, nrhs, a, lda, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgesv_work = 0;
#endif

/* Wrapper for LAPACKE_cgesv_work */
lapack_int  LAPACKE_cgesv_work (int matrix_layout, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgesv_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesv_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgesv_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, nrhs, a, lda, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, n, nrhs, a, lda, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, n, nrhs, a, lda, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgesv_work(int matrix_layout, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesv_work;
    return fn(matrix_layout, n, nrhs, a, lda, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgesv_work(int matrix_layout, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgesv_work++;
    if ( hook_pos_LAPACKE_cgesv_work < __flexiblas_hooks->LAPACKE_cgesv_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgesv_work.hook_function[hook_pos_LAPACKE_cgesv_work];
    } else {
        hook_pos_LAPACKE_cgesv_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgesv_work;
    }
    return fn(matrix_layout, n, nrhs, a, lda, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgesvd = 0;
#endif

/* Wrapper for LAPACKE_cgesvd */
lapack_int  LAPACKE_cgesvd (int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *s, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *vt, lapack_int ldvt, float *superb)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *superb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *superb);
    hook_pos_LAPACKE_cgesvd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgesvd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu, jobvt, m, n, a, lda, s, u, ldu, vt, ldvt, superb);
    } else {
        return fn(matrix_layout, jobu, jobvt, m, n, a, lda, s, u, ldu, vt, ldvt, superb);
    }
    #else
    return fn(matrix_layout, jobu, jobvt, m, n, a, lda, s, u, ldu, vt, ldvt, superb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgesvd(int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *s, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *vt, lapack_int ldvt, float *superb)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *superb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvd;
    return fn(matrix_layout, jobu, jobvt, m, n, a, lda, s, u, ldu, vt, ldvt, superb);
}

lapack_int flexiblas_chain_LAPACKE_cgesvd(int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *superb)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *superb);
    hook_pos_LAPACKE_cgesvd++;
    if ( hook_pos_LAPACKE_cgesvd < __flexiblas_hooks->LAPACKE_cgesvd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgesvd.hook_function[hook_pos_LAPACKE_cgesvd];
    } else {
        hook_pos_LAPACKE_cgesvd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvd;
    }
    return fn(matrix_layout, jobu, jobvt, m, n, a, lda, s, u, ldu, vt, ldvt, superb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgesvd_work = 0;
#endif

/* Wrapper for LAPACKE_cgesvd_work */
lapack_int  LAPACKE_cgesvd_work (int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *s, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *vt, lapack_int ldvt, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cgesvd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgesvd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu, jobvt, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork);
    } else {
        return fn(matrix_layout, jobu, jobvt, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, jobu, jobvt, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgesvd_work(int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *s, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *vt, lapack_int ldvt, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvd_work;
    return fn(matrix_layout, jobu, jobvt, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgesvd_work(int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobvt, lapack_int m, lapack_int n, void *a, lapack_int lda, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cgesvd_work++;
    if ( hook_pos_LAPACKE_cgesvd_work < __flexiblas_hooks->LAPACKE_cgesvd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgesvd_work.hook_function[hook_pos_LAPACKE_cgesvd_work];
    } else {
        hook_pos_LAPACKE_cgesvd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvd_work;
    }
    return fn(matrix_layout, jobu, jobvt, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgesvdx = 0;
#endif

/* Wrapper for LAPACKE_cgesvdx */
lapack_int  LAPACKE_cgesvdx (int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, lapack_int *ns, float *s, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *vt, lapack_int ldvt, lapack_int *superb)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, void *ns, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *superb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, void *ns, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *superb);
    hook_pos_LAPACKE_cgesvdx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvdx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgesvdx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu, jobvt, range, m, n, a, lda, vl, vu, il, iu, ns, s, u, ldu, vt, ldvt, superb);
    } else {
        return fn(matrix_layout, jobu, jobvt, range, m, n, a, lda, vl, vu, il, iu, ns, s, u, ldu, vt, ldvt, superb);
    }
    #else
    return fn(matrix_layout, jobu, jobvt, range, m, n, a, lda, vl, vu, il, iu, ns, s, u, ldu, vt, ldvt, superb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgesvdx(int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, lapack_int *ns, float *s, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *vt, lapack_int ldvt, lapack_int *superb)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, void *ns, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *superb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvdx;
    return fn(matrix_layout, jobu, jobvt, range, m, n, a, lda, vl, vu, il, iu, ns, s, u, ldu, vt, ldvt, superb);
}

lapack_int flexiblas_chain_LAPACKE_cgesvdx(int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, void *ns, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *superb)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, void *ns, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *superb);
    hook_pos_LAPACKE_cgesvdx++;
    if ( hook_pos_LAPACKE_cgesvdx < __flexiblas_hooks->LAPACKE_cgesvdx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgesvdx.hook_function[hook_pos_LAPACKE_cgesvdx];
    } else {
        hook_pos_LAPACKE_cgesvdx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvdx;
    }
    return fn(matrix_layout, jobu, jobvt, range, m, n, a, lda, vl, vu, il, iu, ns, s, u, ldu, vt, ldvt, superb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgesvdx_work = 0;
#endif

/* Wrapper for LAPACKE_cgesvdx_work */
lapack_int  LAPACKE_cgesvdx_work (int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, lapack_int *ns, float *s, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *vt, lapack_int ldvt, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, void *ns, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork, void *iwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, void *ns, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork, void *iwork);
    hook_pos_LAPACKE_cgesvdx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvdx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgesvdx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu, jobvt, range, m, n, a, lda, vl, vu, il, iu, ns, s, u, ldu, vt, ldvt, work, lwork, rwork, iwork);
    } else {
        return fn(matrix_layout, jobu, jobvt, range, m, n, a, lda, vl, vu, il, iu, ns, s, u, ldu, vt, ldvt, work, lwork, rwork, iwork);
    }
    #else
    return fn(matrix_layout, jobu, jobvt, range, m, n, a, lda, vl, vu, il, iu, ns, s, u, ldu, vt, ldvt, work, lwork, rwork, iwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgesvdx_work(int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, lapack_int *ns, float *s, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *vt, lapack_int ldvt, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, void *ns, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork, void *iwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvdx_work;
    return fn(matrix_layout, jobu, jobvt, range, m, n, a, lda, vl, vu, il, iu, ns, s, u, ldu, vt, ldvt, work, lwork, rwork, iwork);
}

lapack_int flexiblas_chain_LAPACKE_cgesvdx_work(int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, void *ns, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork, void *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobvt, char range, lapack_int m, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, void *ns, void *s, void *u, lapack_int ldu, void *vt, lapack_int ldvt, void *work, lapack_int lwork, void *rwork, void *iwork);
    hook_pos_LAPACKE_cgesvdx_work++;
    if ( hook_pos_LAPACKE_cgesvdx_work < __flexiblas_hooks->LAPACKE_cgesvdx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgesvdx_work.hook_function[hook_pos_LAPACKE_cgesvdx_work];
    } else {
        hook_pos_LAPACKE_cgesvdx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvdx_work;
    }
    return fn(matrix_layout, jobu, jobvt, range, m, n, a, lda, vl, vu, il, iu, ns, s, u, ldu, vt, ldvt, work, lwork, rwork, iwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgesvj = 0;
#endif

/* Wrapper for LAPACKE_cgesvj */
lapack_int  LAPACKE_cgesvj (int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *sva, lapack_int mv, lapack_complex_float *v, lapack_int ldv, float *stat)
{
    lapack_int (*fn) (int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, lapack_int mv, void *v, lapack_int ldv, void *stat);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, lapack_int mv, void *v, lapack_int ldv, void *stat);
    hook_pos_LAPACKE_cgesvj = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvj;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgesvj.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, joba, jobu, jobv, m, n, a, lda, sva, mv, v, ldv, stat);
    } else {
        return fn(matrix_layout, joba, jobu, jobv, m, n, a, lda, sva, mv, v, ldv, stat);
    }
    #else
    return fn(matrix_layout, joba, jobu, jobv, m, n, a, lda, sva, mv, v, ldv, stat);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgesvj(int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *sva, lapack_int mv, lapack_complex_float *v, lapack_int ldv, float *stat)
{
    lapack_int (*fn) (int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, lapack_int mv, void *v, lapack_int ldv, void *stat);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvj;
    return fn(matrix_layout, joba, jobu, jobv, m, n, a, lda, sva, mv, v, ldv, stat);
}

lapack_int flexiblas_chain_LAPACKE_cgesvj(int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, lapack_int mv, void *v, lapack_int ldv, void *stat)
{
    lapack_int (*fn) (int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, lapack_int mv, void *v, lapack_int ldv, void *stat);
    hook_pos_LAPACKE_cgesvj++;
    if ( hook_pos_LAPACKE_cgesvj < __flexiblas_hooks->LAPACKE_cgesvj.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgesvj.hook_function[hook_pos_LAPACKE_cgesvj];
    } else {
        hook_pos_LAPACKE_cgesvj = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvj;
    }
    return fn(matrix_layout, joba, jobu, jobv, m, n, a, lda, sva, mv, v, ldv, stat);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgesvj_work = 0;
#endif

/* Wrapper for LAPACKE_cgesvj_work */
lapack_int  LAPACKE_cgesvj_work (int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *sva, lapack_int mv, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *cwork, lapack_int lwork, float *rwork, lapack_int lrwork)
{
    lapack_int (*fn) (int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, lapack_int mv, void *v, lapack_int ldv, void *cwork, lapack_int lwork, void *rwork, lapack_int lrwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, lapack_int mv, void *v, lapack_int ldv, void *cwork, lapack_int lwork, void *rwork, lapack_int lrwork);
    hook_pos_LAPACKE_cgesvj_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvj_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgesvj_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, joba, jobu, jobv, m, n, a, lda, sva, mv, v, ldv, cwork, lwork, rwork, lrwork);
    } else {
        return fn(matrix_layout, joba, jobu, jobv, m, n, a, lda, sva, mv, v, ldv, cwork, lwork, rwork, lrwork);
    }
    #else
    return fn(matrix_layout, joba, jobu, jobv, m, n, a, lda, sva, mv, v, ldv, cwork, lwork, rwork, lrwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgesvj_work(int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, float *sva, lapack_int mv, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *cwork, lapack_int lwork, float *rwork, lapack_int lrwork)
{
    lapack_int (*fn) (int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, lapack_int mv, void *v, lapack_int ldv, void *cwork, lapack_int lwork, void *rwork, lapack_int lrwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvj_work;
    return fn(matrix_layout, joba, jobu, jobv, m, n, a, lda, sva, mv, v, ldv, cwork, lwork, rwork, lrwork);
}

lapack_int flexiblas_chain_LAPACKE_cgesvj_work(int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, lapack_int mv, void *v, lapack_int ldv, void *cwork, lapack_int lwork, void *rwork, lapack_int lrwork)
{
    lapack_int (*fn) (int matrix_layout, char joba, char jobu, char jobv, lapack_int m, lapack_int n, void *a, lapack_int lda, void *sva, lapack_int mv, void *v, lapack_int ldv, void *cwork, lapack_int lwork, void *rwork, lapack_int lrwork);
    hook_pos_LAPACKE_cgesvj_work++;
    if ( hook_pos_LAPACKE_cgesvj_work < __flexiblas_hooks->LAPACKE_cgesvj_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgesvj_work.hook_function[hook_pos_LAPACKE_cgesvj_work];
    } else {
        hook_pos_LAPACKE_cgesvj_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvj_work;
    }
    return fn(matrix_layout, joba, jobu, jobv, m, n, a, lda, sva, mv, v, ldv, cwork, lwork, rwork, lrwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgesvx = 0;
#endif

/* Wrapper for LAPACKE_cgesvx */
lapack_int  LAPACKE_cgesvx (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *af, lapack_int ldaf, lapack_int *ipiv, char *equed, float *r, float *c, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, float *rpivot)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *rpivot);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *rpivot);
    hook_pos_LAPACKE_cgesvx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgesvx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, fact, trans, n, nrhs, a, lda, af, ldaf, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
    } else {
        return fn(matrix_layout, fact, trans, n, nrhs, a, lda, af, ldaf, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
    }
    #else
    return fn(matrix_layout, fact, trans, n, nrhs, a, lda, af, ldaf, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgesvx(int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *af, lapack_int ldaf, lapack_int *ipiv, char *equed, float *r, float *c, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, float *rpivot)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *rpivot);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvx;
    return fn(matrix_layout, fact, trans, n, nrhs, a, lda, af, ldaf, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
}

lapack_int flexiblas_chain_LAPACKE_cgesvx(int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *rpivot)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *rpivot);
    hook_pos_LAPACKE_cgesvx++;
    if ( hook_pos_LAPACKE_cgesvx < __flexiblas_hooks->LAPACKE_cgesvx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgesvx.hook_function[hook_pos_LAPACKE_cgesvx];
    } else {
        hook_pos_LAPACKE_cgesvx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvx;
    }
    return fn(matrix_layout, fact, trans, n, nrhs, a, lda, af, ldaf, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgesvx_work = 0;
#endif

/* Wrapper for LAPACKE_cgesvx_work */
lapack_int  LAPACKE_cgesvx_work (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *af, lapack_int ldaf, lapack_int *ipiv, char *equed, float *r, float *c, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cgesvx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgesvx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, fact, trans, n, nrhs, a, lda, af, ldaf, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
    } else {
        return fn(matrix_layout, fact, trans, n, nrhs, a, lda, af, ldaf, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
    }
    #else
    return fn(matrix_layout, fact, trans, n, nrhs, a, lda, af, ldaf, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgesvx_work(int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *af, lapack_int ldaf, lapack_int *ipiv, char *equed, float *r, float *c, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvx_work;
    return fn(matrix_layout, fact, trans, n, nrhs, a, lda, af, ldaf, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgesvx_work(int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, void *equed, void *r, void *c, void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cgesvx_work++;
    if ( hook_pos_LAPACKE_cgesvx_work < __flexiblas_hooks->LAPACKE_cgesvx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgesvx_work.hook_function[hook_pos_LAPACKE_cgesvx_work];
    } else {
        hook_pos_LAPACKE_cgesvx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgesvx_work;
    }
    return fn(matrix_layout, fact, trans, n, nrhs, a, lda, af, ldaf, ipiv, equed, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgetf2 = 0;
#endif

/* Wrapper for LAPACKE_cgetf2 */
lapack_int  LAPACKE_cgetf2 (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_cgetf2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetf2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgetf2.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, ipiv);
    } else {
        return fn(matrix_layout, m, n, a, lda, ipiv);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgetf2(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetf2;
    return fn(matrix_layout, m, n, a, lda, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_cgetf2(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_cgetf2++;
    if ( hook_pos_LAPACKE_cgetf2 < __flexiblas_hooks->LAPACKE_cgetf2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgetf2.hook_function[hook_pos_LAPACKE_cgetf2];
    } else {
        hook_pos_LAPACKE_cgetf2 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgetf2;
    }
    return fn(matrix_layout, m, n, a, lda, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgetf2_work = 0;
#endif

/* Wrapper for LAPACKE_cgetf2_work */
lapack_int  LAPACKE_cgetf2_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_cgetf2_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetf2_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgetf2_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, ipiv);
    } else {
        return fn(matrix_layout, m, n, a, lda, ipiv);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgetf2_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetf2_work;
    return fn(matrix_layout, m, n, a, lda, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_cgetf2_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_cgetf2_work++;
    if ( hook_pos_LAPACKE_cgetf2_work < __flexiblas_hooks->LAPACKE_cgetf2_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgetf2_work.hook_function[hook_pos_LAPACKE_cgetf2_work];
    } else {
        hook_pos_LAPACKE_cgetf2_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgetf2_work;
    }
    return fn(matrix_layout, m, n, a, lda, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgetrf = 0;
#endif

/* Wrapper for LAPACKE_cgetrf */
lapack_int  LAPACKE_cgetrf (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_cgetrf = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrf;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgetrf.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, ipiv);
    } else {
        return fn(matrix_layout, m, n, a, lda, ipiv);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgetrf(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrf;
    return fn(matrix_layout, m, n, a, lda, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_cgetrf(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_cgetrf++;
    if ( hook_pos_LAPACKE_cgetrf < __flexiblas_hooks->LAPACKE_cgetrf.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgetrf.hook_function[hook_pos_LAPACKE_cgetrf];
    } else {
        hook_pos_LAPACKE_cgetrf = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrf;
    }
    return fn(matrix_layout, m, n, a, lda, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgetrf2 = 0;
#endif

/* Wrapper for LAPACKE_cgetrf2 */
lapack_int  LAPACKE_cgetrf2 (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_cgetrf2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrf2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgetrf2.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, ipiv);
    } else {
        return fn(matrix_layout, m, n, a, lda, ipiv);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgetrf2(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrf2;
    return fn(matrix_layout, m, n, a, lda, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_cgetrf2(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_cgetrf2++;
    if ( hook_pos_LAPACKE_cgetrf2 < __flexiblas_hooks->LAPACKE_cgetrf2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgetrf2.hook_function[hook_pos_LAPACKE_cgetrf2];
    } else {
        hook_pos_LAPACKE_cgetrf2 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrf2;
    }
    return fn(matrix_layout, m, n, a, lda, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgetrf2_work = 0;
#endif

/* Wrapper for LAPACKE_cgetrf2_work */
lapack_int  LAPACKE_cgetrf2_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_cgetrf2_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrf2_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgetrf2_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, ipiv);
    } else {
        return fn(matrix_layout, m, n, a, lda, ipiv);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgetrf2_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrf2_work;
    return fn(matrix_layout, m, n, a, lda, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_cgetrf2_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_cgetrf2_work++;
    if ( hook_pos_LAPACKE_cgetrf2_work < __flexiblas_hooks->LAPACKE_cgetrf2_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgetrf2_work.hook_function[hook_pos_LAPACKE_cgetrf2_work];
    } else {
        hook_pos_LAPACKE_cgetrf2_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrf2_work;
    }
    return fn(matrix_layout, m, n, a, lda, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgetrf_work = 0;
#endif

/* Wrapper for LAPACKE_cgetrf_work */
lapack_int  LAPACKE_cgetrf_work (int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_cgetrf_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrf_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgetrf_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, ipiv);
    } else {
        return fn(matrix_layout, m, n, a, lda, ipiv);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgetrf_work(int matrix_layout, lapack_int m, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrf_work;
    return fn(matrix_layout, m, n, a, lda, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_cgetrf_work(int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_cgetrf_work++;
    if ( hook_pos_LAPACKE_cgetrf_work < __flexiblas_hooks->LAPACKE_cgetrf_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgetrf_work.hook_function[hook_pos_LAPACKE_cgetrf_work];
    } else {
        hook_pos_LAPACKE_cgetrf_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrf_work;
    }
    return fn(matrix_layout, m, n, a, lda, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgetri = 0;
#endif

/* Wrapper for LAPACKE_cgetri */
lapack_int  LAPACKE_cgetri (int matrix_layout, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, void *a, lapack_int lda, const void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, void *a, lapack_int lda, const void *ipiv);
    hook_pos_LAPACKE_cgetri = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetri;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgetri.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, a, lda, ipiv);
    } else {
        return fn(matrix_layout, n, a, lda, ipiv);
    }
    #else
    return fn(matrix_layout, n, a, lda, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgetri(int matrix_layout, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, void *a, lapack_int lda, const void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetri;
    return fn(matrix_layout, n, a, lda, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_cgetri(int matrix_layout, lapack_int n, void *a, lapack_int lda, const void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, void *a, lapack_int lda, const void *ipiv);
    hook_pos_LAPACKE_cgetri++;
    if ( hook_pos_LAPACKE_cgetri < __flexiblas_hooks->LAPACKE_cgetri.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgetri.hook_function[hook_pos_LAPACKE_cgetri];
    } else {
        hook_pos_LAPACKE_cgetri = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgetri;
    }
    return fn(matrix_layout, n, a, lda, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgetri_work = 0;
#endif

/* Wrapper for LAPACKE_cgetri_work */
lapack_int  LAPACKE_cgetri_work (int matrix_layout, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgetri_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetri_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgetri_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, a, lda, ipiv, work, lwork);
    } else {
        return fn(matrix_layout, n, a, lda, ipiv, work, lwork);
    }
    #else
    return fn(matrix_layout, n, a, lda, ipiv, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgetri_work(int matrix_layout, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetri_work;
    return fn(matrix_layout, n, a, lda, ipiv, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgetri_work(int matrix_layout, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgetri_work++;
    if ( hook_pos_LAPACKE_cgetri_work < __flexiblas_hooks->LAPACKE_cgetri_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgetri_work.hook_function[hook_pos_LAPACKE_cgetri_work];
    } else {
        hook_pos_LAPACKE_cgetri_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgetri_work;
    }
    return fn(matrix_layout, n, a, lda, ipiv, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgetrs = 0;
#endif

/* Wrapper for LAPACKE_cgetrs */
lapack_int  LAPACKE_cgetrs (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgetrs = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrs;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgetrs.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, n, nrhs, a, lda, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, trans, n, nrhs, a, lda, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, trans, n, nrhs, a, lda, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgetrs(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrs;
    return fn(matrix_layout, trans, n, nrhs, a, lda, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgetrs(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgetrs++;
    if ( hook_pos_LAPACKE_cgetrs < __flexiblas_hooks->LAPACKE_cgetrs.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgetrs.hook_function[hook_pos_LAPACKE_cgetrs];
    } else {
        hook_pos_LAPACKE_cgetrs = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrs;
    }
    return fn(matrix_layout, trans, n, nrhs, a, lda, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgetrs_work = 0;
#endif

/* Wrapper for LAPACKE_cgetrs_work */
lapack_int  LAPACKE_cgetrs_work (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgetrs_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrs_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgetrs_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, n, nrhs, a, lda, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, trans, n, nrhs, a, lda, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, trans, n, nrhs, a, lda, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgetrs_work(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrs_work;
    return fn(matrix_layout, trans, n, nrhs, a, lda, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgetrs_work(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgetrs_work++;
    if ( hook_pos_LAPACKE_cgetrs_work < __flexiblas_hooks->LAPACKE_cgetrs_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgetrs_work.hook_function[hook_pos_LAPACKE_cgetrs_work];
    } else {
        hook_pos_LAPACKE_cgetrs_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgetrs_work;
    }
    return fn(matrix_layout, trans, n, nrhs, a, lda, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgetsls = 0;
#endif

/* Wrapper for LAPACKE_cgetsls */
lapack_int  LAPACKE_cgetsls (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgetsls = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetsls;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgetsls.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb);
    } else {
        return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb);
    }
    #else
    return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgetsls(int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetsls;
    return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgetsls(int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgetsls++;
    if ( hook_pos_LAPACKE_cgetsls < __flexiblas_hooks->LAPACKE_cgetsls.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgetsls.hook_function[hook_pos_LAPACKE_cgetsls];
    } else {
        hook_pos_LAPACKE_cgetsls = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgetsls;
    }
    return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgetsls_work = 0;
#endif

/* Wrapper for LAPACKE_cgetsls_work */
lapack_int  LAPACKE_cgetsls_work (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgetsls_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetsls_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgetsls_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb, work, lwork);
    } else {
        return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb, work, lwork);
    }
    #else
    return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgetsls_work(int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgetsls_work;
    return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgetsls_work(int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int m, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgetsls_work++;
    if ( hook_pos_LAPACKE_cgetsls_work < __flexiblas_hooks->LAPACKE_cgetsls_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgetsls_work.hook_function[hook_pos_LAPACKE_cgetsls_work];
    } else {
        hook_pos_LAPACKE_cgetsls_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgetsls_work;
    }
    return fn(matrix_layout, trans, m, n, nrhs, a, lda, b, ldb, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggbak = 0;
#endif

/* Wrapper for LAPACKE_cggbak */
lapack_int  LAPACKE_cggbak (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const float *lscale, const float *rscale, lapack_int m, lapack_complex_float *v, lapack_int ldv)
{
    lapack_int (*fn) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *lscale, const void *rscale, lapack_int m, void *v, lapack_int ldv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *lscale, const void *rscale, lapack_int m, void *v, lapack_int ldv);
    hook_pos_LAPACKE_cggbak = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggbak;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggbak.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, side, n, ilo, ihi, lscale, rscale, m, v, ldv);
    } else {
        return fn(matrix_layout, job, side, n, ilo, ihi, lscale, rscale, m, v, ldv);
    }
    #else
    return fn(matrix_layout, job, side, n, ilo, ihi, lscale, rscale, m, v, ldv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggbak(int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const float *lscale, const float *rscale, lapack_int m, lapack_complex_float *v, lapack_int ldv)
{
    lapack_int (*fn) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *lscale, const void *rscale, lapack_int m, void *v, lapack_int ldv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggbak;
    return fn(matrix_layout, job, side, n, ilo, ihi, lscale, rscale, m, v, ldv);
}

lapack_int flexiblas_chain_LAPACKE_cggbak(int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *lscale, const void *rscale, lapack_int m, void *v, lapack_int ldv)
{
    lapack_int (*fn) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *lscale, const void *rscale, lapack_int m, void *v, lapack_int ldv);
    hook_pos_LAPACKE_cggbak++;
    if ( hook_pos_LAPACKE_cggbak < __flexiblas_hooks->LAPACKE_cggbak.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggbak.hook_function[hook_pos_LAPACKE_cggbak];
    } else {
        hook_pos_LAPACKE_cggbak = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggbak;
    }
    return fn(matrix_layout, job, side, n, ilo, ihi, lscale, rscale, m, v, ldv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggbak_work = 0;
#endif

/* Wrapper for LAPACKE_cggbak_work */
lapack_int  LAPACKE_cggbak_work (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const float *lscale, const float *rscale, lapack_int m, lapack_complex_float *v, lapack_int ldv)
{
    lapack_int (*fn) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *lscale, const void *rscale, lapack_int m, void *v, lapack_int ldv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *lscale, const void *rscale, lapack_int m, void *v, lapack_int ldv);
    hook_pos_LAPACKE_cggbak_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggbak_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggbak_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, side, n, ilo, ihi, lscale, rscale, m, v, ldv);
    } else {
        return fn(matrix_layout, job, side, n, ilo, ihi, lscale, rscale, m, v, ldv);
    }
    #else
    return fn(matrix_layout, job, side, n, ilo, ihi, lscale, rscale, m, v, ldv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggbak_work(int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const float *lscale, const float *rscale, lapack_int m, lapack_complex_float *v, lapack_int ldv)
{
    lapack_int (*fn) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *lscale, const void *rscale, lapack_int m, void *v, lapack_int ldv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggbak_work;
    return fn(matrix_layout, job, side, n, ilo, ihi, lscale, rscale, m, v, ldv);
}

lapack_int flexiblas_chain_LAPACKE_cggbak_work(int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *lscale, const void *rscale, lapack_int m, void *v, lapack_int ldv)
{
    lapack_int (*fn) (int matrix_layout, char job, char side, lapack_int n, lapack_int ilo, lapack_int ihi, const void *lscale, const void *rscale, lapack_int m, void *v, lapack_int ldv);
    hook_pos_LAPACKE_cggbak_work++;
    if ( hook_pos_LAPACKE_cggbak_work < __flexiblas_hooks->LAPACKE_cggbak_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggbak_work.hook_function[hook_pos_LAPACKE_cggbak_work];
    } else {
        hook_pos_LAPACKE_cggbak_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggbak_work;
    }
    return fn(matrix_layout, job, side, n, ilo, ihi, lscale, rscale, m, v, ldv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggbal = 0;
#endif

/* Wrapper for LAPACKE_cggbal */
lapack_int  LAPACKE_cggbal (int matrix_layout, char job, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *ilo, lapack_int *ihi, float *lscale, float *rscale)
{
    lapack_int (*fn) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *ilo, void *ihi, void *lscale, void *rscale);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *ilo, void *ihi, void *lscale, void *rscale);
    hook_pos_LAPACKE_cggbal = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggbal;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggbal.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, n, a, lda, b, ldb, ilo, ihi, lscale, rscale);
    } else {
        return fn(matrix_layout, job, n, a, lda, b, ldb, ilo, ihi, lscale, rscale);
    }
    #else
    return fn(matrix_layout, job, n, a, lda, b, ldb, ilo, ihi, lscale, rscale);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggbal(int matrix_layout, char job, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *ilo, lapack_int *ihi, float *lscale, float *rscale)
{
    lapack_int (*fn) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *ilo, void *ihi, void *lscale, void *rscale);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggbal;
    return fn(matrix_layout, job, n, a, lda, b, ldb, ilo, ihi, lscale, rscale);
}

lapack_int flexiblas_chain_LAPACKE_cggbal(int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *ilo, void *ihi, void *lscale, void *rscale)
{
    lapack_int (*fn) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *ilo, void *ihi, void *lscale, void *rscale);
    hook_pos_LAPACKE_cggbal++;
    if ( hook_pos_LAPACKE_cggbal < __flexiblas_hooks->LAPACKE_cggbal.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggbal.hook_function[hook_pos_LAPACKE_cggbal];
    } else {
        hook_pos_LAPACKE_cggbal = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggbal;
    }
    return fn(matrix_layout, job, n, a, lda, b, ldb, ilo, ihi, lscale, rscale);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggbal_work = 0;
#endif

/* Wrapper for LAPACKE_cggbal_work */
lapack_int  LAPACKE_cggbal_work (int matrix_layout, char job, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *ilo, lapack_int *ihi, float *lscale, float *rscale, float *work)
{
    lapack_int (*fn) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *ilo, void *ihi, void *lscale, void *rscale, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *ilo, void *ihi, void *lscale, void *rscale, void *work);
    hook_pos_LAPACKE_cggbal_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggbal_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggbal_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, n, a, lda, b, ldb, ilo, ihi, lscale, rscale, work);
    } else {
        return fn(matrix_layout, job, n, a, lda, b, ldb, ilo, ihi, lscale, rscale, work);
    }
    #else
    return fn(matrix_layout, job, n, a, lda, b, ldb, ilo, ihi, lscale, rscale, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggbal_work(int matrix_layout, char job, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *ilo, lapack_int *ihi, float *lscale, float *rscale, float *work)
{
    lapack_int (*fn) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *ilo, void *ihi, void *lscale, void *rscale, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggbal_work;
    return fn(matrix_layout, job, n, a, lda, b, ldb, ilo, ihi, lscale, rscale, work);
}

lapack_int flexiblas_chain_LAPACKE_cggbal_work(int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *ilo, void *ihi, void *lscale, void *rscale, void *work)
{
    lapack_int (*fn) (int matrix_layout, char job, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *ilo, void *ihi, void *lscale, void *rscale, void *work);
    hook_pos_LAPACKE_cggbal_work++;
    if ( hook_pos_LAPACKE_cggbal_work < __flexiblas_hooks->LAPACKE_cggbal_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggbal_work.hook_function[hook_pos_LAPACKE_cggbal_work];
    } else {
        hook_pos_LAPACKE_cggbal_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggbal_work;
    }
    return fn(matrix_layout, job, n, a, lda, b, ldb, ilo, ihi, lscale, rscale, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgges = 0;
#endif

/* Wrapper for LAPACKE_cgges */
lapack_int  LAPACKE_cgges (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, lapack_int ldvsl, lapack_complex_float *vsr, lapack_int ldvsr)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr);
    hook_pos_LAPACKE_cgges = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgges;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgges.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr);
    } else {
        return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr);
    }
    #else
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgges(int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, lapack_int ldvsl, lapack_complex_float *vsr, lapack_int ldvsr)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgges;
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr);
}

lapack_int flexiblas_chain_LAPACKE_cgges(int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr);
    hook_pos_LAPACKE_cgges++;
    if ( hook_pos_LAPACKE_cgges < __flexiblas_hooks->LAPACKE_cgges.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgges.hook_function[hook_pos_LAPACKE_cgges];
    } else {
        hook_pos_LAPACKE_cgges = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgges;
    }
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgges3 = 0;
#endif

/* Wrapper for LAPACKE_cgges3 */
lapack_int  LAPACKE_cgges3 (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, lapack_int ldvsl, lapack_complex_float *vsr, lapack_int ldvsr)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr);
    hook_pos_LAPACKE_cgges3 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgges3;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgges3.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr);
    } else {
        return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr);
    }
    #else
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgges3(int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, lapack_int ldvsl, lapack_complex_float *vsr, lapack_int ldvsr)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgges3;
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr);
}

lapack_int flexiblas_chain_LAPACKE_cgges3(int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr);
    hook_pos_LAPACKE_cgges3++;
    if ( hook_pos_LAPACKE_cgges3 < __flexiblas_hooks->LAPACKE_cgges3.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgges3.hook_function[hook_pos_LAPACKE_cgges3];
    } else {
        hook_pos_LAPACKE_cgges3 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgges3;
    }
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgges3_work = 0;
#endif

/* Wrapper for LAPACKE_cgges3_work */
lapack_int  LAPACKE_cgges3_work (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, lapack_int ldvsl, lapack_complex_float *vsr, lapack_int ldvsr, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *work, lapack_int lwork, void *rwork, void *bwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *work, lapack_int lwork, void *rwork, void *bwork);
    hook_pos_LAPACKE_cgges3_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgges3_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgges3_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, rwork, bwork);
    } else {
        return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, rwork, bwork);
    }
    #else
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, rwork, bwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgges3_work(int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, lapack_int ldvsl, lapack_complex_float *vsr, lapack_int ldvsr, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *work, lapack_int lwork, void *rwork, void *bwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgges3_work;
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, rwork, bwork);
}

lapack_int flexiblas_chain_LAPACKE_cgges3_work(int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *work, lapack_int lwork, void *rwork, void *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *work, lapack_int lwork, void *rwork, void *bwork);
    hook_pos_LAPACKE_cgges3_work++;
    if ( hook_pos_LAPACKE_cgges3_work < __flexiblas_hooks->LAPACKE_cgges3_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgges3_work.hook_function[hook_pos_LAPACKE_cgges3_work];
    } else {
        hook_pos_LAPACKE_cgges3_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgges3_work;
    }
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, rwork, bwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgges_work = 0;
#endif

/* Wrapper for LAPACKE_cgges_work */
lapack_int  LAPACKE_cgges_work (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, lapack_int ldvsl, lapack_complex_float *vsr, lapack_int ldvsr, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *work, lapack_int lwork, void *rwork, void *bwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *work, lapack_int lwork, void *rwork, void *bwork);
    hook_pos_LAPACKE_cgges_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgges_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgges_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, rwork, bwork);
    } else {
        return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, rwork, bwork);
    }
    #else
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, rwork, bwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgges_work(int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, lapack_int ldvsl, lapack_complex_float *vsr, lapack_int ldvsr, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *work, lapack_int lwork, void *rwork, void *bwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgges_work;
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, rwork, bwork);
}

lapack_int flexiblas_chain_LAPACKE_cgges_work(int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *work, lapack_int lwork, void *rwork, void *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *work, lapack_int lwork, void *rwork, void *bwork);
    hook_pos_LAPACKE_cgges_work++;
    if ( hook_pos_LAPACKE_cgges_work < __flexiblas_hooks->LAPACKE_cgges_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgges_work.hook_function[hook_pos_LAPACKE_cgges_work];
    } else {
        hook_pos_LAPACKE_cgges_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgges_work;
    }
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, rwork, bwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggesx = 0;
#endif

/* Wrapper for LAPACKE_cggesx */
lapack_int  LAPACKE_cggesx (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, lapack_int ldvsl, lapack_complex_float *vsr, lapack_int ldvsr, float *rconde, float *rcondv)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *rconde, void *rcondv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *rconde, void *rcondv);
    hook_pos_LAPACKE_cggesx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggesx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggesx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvsl, jobvsr, sort, selctg, sense, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv);
    } else {
        return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, sense, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv);
    }
    #else
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, sense, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggesx(int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, lapack_int ldvsl, lapack_complex_float *vsr, lapack_int ldvsr, float *rconde, float *rcondv)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *rconde, void *rcondv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggesx;
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, sense, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv);
}

lapack_int flexiblas_chain_LAPACKE_cggesx(int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *rconde, void *rcondv)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *rconde, void *rcondv);
    hook_pos_LAPACKE_cggesx++;
    if ( hook_pos_LAPACKE_cggesx < __flexiblas_hooks->LAPACKE_cggesx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggesx.hook_function[hook_pos_LAPACKE_cggesx];
    } else {
        hook_pos_LAPACKE_cggesx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggesx;
    }
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, sense, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggesx_work = 0;
#endif

/* Wrapper for LAPACKE_cggesx_work */
lapack_int  LAPACKE_cggesx_work (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, lapack_int ldvsl, lapack_complex_float *vsr, lapack_int ldvsr, float *rconde, float *rcondv, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork, lapack_int liwork, lapack_int *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *iwork, lapack_int liwork, void *bwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *iwork, lapack_int liwork, void *bwork);
    hook_pos_LAPACKE_cggesx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggesx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggesx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvsl, jobvsr, sort, selctg, sense, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv, work, lwork, rwork, iwork, liwork, bwork);
    } else {
        return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, sense, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv, work, lwork, rwork, iwork, liwork, bwork);
    }
    #else
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, sense, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv, work, lwork, rwork, iwork, liwork, bwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggesx_work(int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_int *sdim, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vsl, lapack_int ldvsl, lapack_complex_float *vsr, lapack_int ldvsr, float *rconde, float *rcondv, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork, lapack_int liwork, lapack_int *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *iwork, lapack_int liwork, void *bwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggesx_work;
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, sense, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv, work, lwork, rwork, iwork, liwork, bwork);
}

lapack_int flexiblas_chain_LAPACKE_cggesx_work(int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *iwork, lapack_int liwork, void *bwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvsl, char jobvsr, char sort, LAPACK_C_SELECT2 selctg, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *sdim, void *alpha, void *beta, void *vsl, lapack_int ldvsl, void *vsr, lapack_int ldvsr, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *iwork, lapack_int liwork, void *bwork);
    hook_pos_LAPACKE_cggesx_work++;
    if ( hook_pos_LAPACKE_cggesx_work < __flexiblas_hooks->LAPACKE_cggesx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggesx_work.hook_function[hook_pos_LAPACKE_cggesx_work];
    } else {
        hook_pos_LAPACKE_cggesx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggesx_work;
    }
    return fn(matrix_layout, jobvsl, jobvsr, sort, selctg, sense, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv, work, lwork, rwork, iwork, liwork, bwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggev = 0;
#endif

/* Wrapper for LAPACKE_cggev */
lapack_int  LAPACKE_cggev (int matrix_layout, char jobvl, char jobvr, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr);
    hook_pos_LAPACKE_cggev = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggev;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggev.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr);
    } else {
        return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr);
    }
    #else
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggev(int matrix_layout, char jobvl, char jobvr, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggev;
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr);
}

lapack_int flexiblas_chain_LAPACKE_cggev(int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr);
    hook_pos_LAPACKE_cggev++;
    if ( hook_pos_LAPACKE_cggev < __flexiblas_hooks->LAPACKE_cggev.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggev.hook_function[hook_pos_LAPACKE_cggev];
    } else {
        hook_pos_LAPACKE_cggev = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggev;
    }
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggev3 = 0;
#endif

/* Wrapper for LAPACKE_cggev3 */
lapack_int  LAPACKE_cggev3 (int matrix_layout, char jobvl, char jobvr, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr);
    hook_pos_LAPACKE_cggev3 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggev3;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggev3.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr);
    } else {
        return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr);
    }
    #else
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggev3(int matrix_layout, char jobvl, char jobvr, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggev3;
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr);
}

lapack_int flexiblas_chain_LAPACKE_cggev3(int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr);
    hook_pos_LAPACKE_cggev3++;
    if ( hook_pos_LAPACKE_cggev3 < __flexiblas_hooks->LAPACKE_cggev3.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggev3.hook_function[hook_pos_LAPACKE_cggev3];
    } else {
        hook_pos_LAPACKE_cggev3 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggev3;
    }
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggev3_work = 0;
#endif

/* Wrapper for LAPACKE_cggev3_work */
lapack_int  LAPACKE_cggev3_work (int matrix_layout, char jobvl, char jobvr, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cggev3_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggev3_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggev3_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, work, lwork, rwork);
    } else {
        return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggev3_work(int matrix_layout, char jobvl, char jobvr, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggev3_work;
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cggev3_work(int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cggev3_work++;
    if ( hook_pos_LAPACKE_cggev3_work < __flexiblas_hooks->LAPACKE_cggev3_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggev3_work.hook_function[hook_pos_LAPACKE_cggev3_work];
    } else {
        hook_pos_LAPACKE_cggev3_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggev3_work;
    }
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggev_work = 0;
#endif

/* Wrapper for LAPACKE_cggev_work */
lapack_int  LAPACKE_cggev_work (int matrix_layout, char jobvl, char jobvr, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cggev_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggev_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggev_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, work, lwork, rwork);
    } else {
        return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggev_work(int matrix_layout, char jobvl, char jobvr, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggev_work;
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cggev_work(int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobvl, char jobvr, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cggev_work++;
    if ( hook_pos_LAPACKE_cggev_work < __flexiblas_hooks->LAPACKE_cggev_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggev_work.hook_function[hook_pos_LAPACKE_cggev_work];
    } else {
        hook_pos_LAPACKE_cggev_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggev_work;
    }
    return fn(matrix_layout, jobvl, jobvr, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggevx = 0;
#endif

/* Wrapper for LAPACKE_cggevx */
lapack_int  LAPACKE_cggevx (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_int *ilo, lapack_int *ihi, float *lscale, float *rscale, float *abnrm, float *bbnrm, float *rconde, float *rcondv)
{
    lapack_int (*fn) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *lscale, void *rscale, void *abnrm, void *bbnrm, void *rconde, void *rcondv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *lscale, void *rscale, void *abnrm, void *bbnrm, void *rconde, void *rcondv);
    hook_pos_LAPACKE_cggevx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggevx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggevx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv);
    } else {
        return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv);
    }
    #else
    return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggevx(int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_int *ilo, lapack_int *ihi, float *lscale, float *rscale, float *abnrm, float *bbnrm, float *rconde, float *rcondv)
{
    lapack_int (*fn) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *lscale, void *rscale, void *abnrm, void *bbnrm, void *rconde, void *rcondv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggevx;
    return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv);
}

lapack_int flexiblas_chain_LAPACKE_cggevx(int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *lscale, void *rscale, void *abnrm, void *bbnrm, void *rconde, void *rcondv)
{
    lapack_int (*fn) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *lscale, void *rscale, void *abnrm, void *bbnrm, void *rconde, void *rcondv);
    hook_pos_LAPACKE_cggevx++;
    if ( hook_pos_LAPACKE_cggevx < __flexiblas_hooks->LAPACKE_cggevx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggevx.hook_function[hook_pos_LAPACKE_cggevx];
    } else {
        hook_pos_LAPACKE_cggevx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggevx;
    }
    return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggevx_work = 0;
#endif

/* Wrapper for LAPACKE_cggevx_work */
lapack_int  LAPACKE_cggevx_work (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_int *ilo, lapack_int *ihi, float *lscale, float *rscale, float *abnrm, float *bbnrm, float *rconde, float *rcondv, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork, lapack_int *bwork)
{
    lapack_int (*fn) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *lscale, void *rscale, void *abnrm, void *bbnrm, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *iwork, void *bwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *lscale, void *rscale, void *abnrm, void *bbnrm, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *iwork, void *bwork);
    hook_pos_LAPACKE_cggevx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggevx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggevx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv, work, lwork, rwork, iwork, bwork);
    } else {
        return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv, work, lwork, rwork, iwork, bwork);
    }
    #else
    return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv, work, lwork, rwork, iwork, bwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggevx_work(int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_int *ilo, lapack_int *ihi, float *lscale, float *rscale, float *abnrm, float *bbnrm, float *rconde, float *rcondv, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork, lapack_int *bwork)
{
    lapack_int (*fn) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *lscale, void *rscale, void *abnrm, void *bbnrm, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *iwork, void *bwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggevx_work;
    return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv, work, lwork, rwork, iwork, bwork);
}

lapack_int flexiblas_chain_LAPACKE_cggevx_work(int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *lscale, void *rscale, void *abnrm, void *bbnrm, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *iwork, void *bwork)
{
    lapack_int (*fn) (int matrix_layout, char balanc, char jobvl, char jobvr, char sense, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, void *ilo, void *ihi, void *lscale, void *rscale, void *abnrm, void *bbnrm, void *rconde, void *rcondv, void *work, lapack_int lwork, void *rwork, void *iwork, void *bwork);
    hook_pos_LAPACKE_cggevx_work++;
    if ( hook_pos_LAPACKE_cggevx_work < __flexiblas_hooks->LAPACKE_cggevx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggevx_work.hook_function[hook_pos_LAPACKE_cggevx_work];
    } else {
        hook_pos_LAPACKE_cggevx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggevx_work;
    }
    return fn(matrix_layout, balanc, jobvl, jobvr, sense, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv, work, lwork, rwork, iwork, bwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggglm = 0;
#endif

/* Wrapper for LAPACKE_cggglm */
lapack_int  LAPACKE_cggglm (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *d, lapack_complex_float *x, lapack_complex_float *y)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *d, void *x, void *y);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *d, void *x, void *y);
    hook_pos_LAPACKE_cggglm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggglm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggglm.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, m, p, a, lda, b, ldb, d, x, y);
    } else {
        return fn(matrix_layout, n, m, p, a, lda, b, ldb, d, x, y);
    }
    #else
    return fn(matrix_layout, n, m, p, a, lda, b, ldb, d, x, y);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggglm(int matrix_layout, lapack_int n, lapack_int m, lapack_int p, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *d, lapack_complex_float *x, lapack_complex_float *y)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *d, void *x, void *y);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggglm;
    return fn(matrix_layout, n, m, p, a, lda, b, ldb, d, x, y);
}

lapack_int flexiblas_chain_LAPACKE_cggglm(int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *d, void *x, void *y)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *d, void *x, void *y);
    hook_pos_LAPACKE_cggglm++;
    if ( hook_pos_LAPACKE_cggglm < __flexiblas_hooks->LAPACKE_cggglm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggglm.hook_function[hook_pos_LAPACKE_cggglm];
    } else {
        hook_pos_LAPACKE_cggglm = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggglm;
    }
    return fn(matrix_layout, n, m, p, a, lda, b, ldb, d, x, y);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggglm_work = 0;
#endif

/* Wrapper for LAPACKE_cggglm_work */
lapack_int  LAPACKE_cggglm_work (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *d, lapack_complex_float *x, lapack_complex_float *y, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *d, void *x, void *y, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *d, void *x, void *y, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cggglm_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggglm_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggglm_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, m, p, a, lda, b, ldb, d, x, y, work, lwork);
    } else {
        return fn(matrix_layout, n, m, p, a, lda, b, ldb, d, x, y, work, lwork);
    }
    #else
    return fn(matrix_layout, n, m, p, a, lda, b, ldb, d, x, y, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggglm_work(int matrix_layout, lapack_int n, lapack_int m, lapack_int p, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *d, lapack_complex_float *x, lapack_complex_float *y, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *d, void *x, void *y, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggglm_work;
    return fn(matrix_layout, n, m, p, a, lda, b, ldb, d, x, y, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cggglm_work(int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *d, void *x, void *y, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *d, void *x, void *y, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cggglm_work++;
    if ( hook_pos_LAPACKE_cggglm_work < __flexiblas_hooks->LAPACKE_cggglm_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggglm_work.hook_function[hook_pos_LAPACKE_cggglm_work];
    } else {
        hook_pos_LAPACKE_cggglm_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggglm_work;
    }
    return fn(matrix_layout, n, m, p, a, lda, b, ldb, d, x, y, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgghd3 = 0;
#endif

/* Wrapper for LAPACKE_cgghd3 */
lapack_int  LAPACKE_cgghd3 (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz);
    hook_pos_LAPACKE_cgghd3 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgghd3;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgghd3.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
    } else {
        return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
    }
    #else
    return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgghd3(int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgghd3;
    return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_cgghd3(int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz);
    hook_pos_LAPACKE_cgghd3++;
    if ( hook_pos_LAPACKE_cgghd3 < __flexiblas_hooks->LAPACKE_cgghd3.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgghd3.hook_function[hook_pos_LAPACKE_cgghd3];
    } else {
        hook_pos_LAPACKE_cgghd3 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgghd3;
    }
    return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgghd3_work = 0;
#endif

/* Wrapper for LAPACKE_cgghd3_work */
lapack_int  LAPACKE_cgghd3_work (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgghd3_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgghd3_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgghd3_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz, work, lwork);
    } else {
        return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz, work, lwork);
    }
    #else
    return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgghd3_work(int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgghd3_work;
    return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgghd3_work(int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgghd3_work++;
    if ( hook_pos_LAPACKE_cgghd3_work < __flexiblas_hooks->LAPACKE_cgghd3_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgghd3_work.hook_function[hook_pos_LAPACKE_cgghd3_work];
    } else {
        hook_pos_LAPACKE_cgghd3_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgghd3_work;
    }
    return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgghrd = 0;
#endif

/* Wrapper for LAPACKE_cgghrd */
lapack_int  LAPACKE_cgghrd (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz);
    hook_pos_LAPACKE_cgghrd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgghrd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgghrd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
    } else {
        return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
    }
    #else
    return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgghrd(int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgghrd;
    return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_cgghrd(int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz);
    hook_pos_LAPACKE_cgghrd++;
    if ( hook_pos_LAPACKE_cgghrd < __flexiblas_hooks->LAPACKE_cgghrd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgghrd.hook_function[hook_pos_LAPACKE_cgghrd];
    } else {
        hook_pos_LAPACKE_cgghrd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgghrd;
    }
    return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgghrd_work = 0;
#endif

/* Wrapper for LAPACKE_cgghrd_work */
lapack_int  LAPACKE_cgghrd_work (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz);
    hook_pos_LAPACKE_cgghrd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgghrd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgghrd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
    } else {
        return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
    }
    #else
    return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgghrd_work(int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgghrd_work;
    return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_cgghrd_work(int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *a, lapack_int lda, void *b, lapack_int ldb, void *q, lapack_int ldq, void *z, lapack_int ldz);
    hook_pos_LAPACKE_cgghrd_work++;
    if ( hook_pos_LAPACKE_cgghrd_work < __flexiblas_hooks->LAPACKE_cgghrd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgghrd_work.hook_function[hook_pos_LAPACKE_cgghrd_work];
    } else {
        hook_pos_LAPACKE_cgghrd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgghrd_work;
    }
    return fn(matrix_layout, compq, compz, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgglse = 0;
#endif

/* Wrapper for LAPACKE_cgglse */
lapack_int  LAPACKE_cgglse (int matrix_layout, lapack_int m, lapack_int n, lapack_int p, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *c, lapack_complex_float *d, lapack_complex_float *x)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *c, void *d, void *x);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *c, void *d, void *x);
    hook_pos_LAPACKE_cgglse = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgglse;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgglse.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, p, a, lda, b, ldb, c, d, x);
    } else {
        return fn(matrix_layout, m, n, p, a, lda, b, ldb, c, d, x);
    }
    #else
    return fn(matrix_layout, m, n, p, a, lda, b, ldb, c, d, x);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgglse(int matrix_layout, lapack_int m, lapack_int n, lapack_int p, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *c, lapack_complex_float *d, lapack_complex_float *x)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *c, void *d, void *x);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgglse;
    return fn(matrix_layout, m, n, p, a, lda, b, ldb, c, d, x);
}

lapack_int flexiblas_chain_LAPACKE_cgglse(int matrix_layout, lapack_int m, lapack_int n, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *c, void *d, void *x)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *c, void *d, void *x);
    hook_pos_LAPACKE_cgglse++;
    if ( hook_pos_LAPACKE_cgglse < __flexiblas_hooks->LAPACKE_cgglse.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgglse.hook_function[hook_pos_LAPACKE_cgglse];
    } else {
        hook_pos_LAPACKE_cgglse = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgglse;
    }
    return fn(matrix_layout, m, n, p, a, lda, b, ldb, c, d, x);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgglse_work = 0;
#endif

/* Wrapper for LAPACKE_cgglse_work */
lapack_int  LAPACKE_cgglse_work (int matrix_layout, lapack_int m, lapack_int n, lapack_int p, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *c, lapack_complex_float *d, lapack_complex_float *x, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *c, void *d, void *x, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *c, void *d, void *x, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgglse_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgglse_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgglse_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, p, a, lda, b, ldb, c, d, x, work, lwork);
    } else {
        return fn(matrix_layout, m, n, p, a, lda, b, ldb, c, d, x, work, lwork);
    }
    #else
    return fn(matrix_layout, m, n, p, a, lda, b, ldb, c, d, x, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgglse_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int p, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *c, lapack_complex_float *d, lapack_complex_float *x, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *c, void *d, void *x, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgglse_work;
    return fn(matrix_layout, m, n, p, a, lda, b, ldb, c, d, x, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cgglse_work(int matrix_layout, lapack_int m, lapack_int n, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *c, void *d, void *x, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, lapack_int p, void *a, lapack_int lda, void *b, lapack_int ldb, void *c, void *d, void *x, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cgglse_work++;
    if ( hook_pos_LAPACKE_cgglse_work < __flexiblas_hooks->LAPACKE_cgglse_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgglse_work.hook_function[hook_pos_LAPACKE_cgglse_work];
    } else {
        hook_pos_LAPACKE_cgglse_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgglse_work;
    }
    return fn(matrix_layout, m, n, p, a, lda, b, ldb, c, d, x, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggqrf = 0;
#endif

/* Wrapper for LAPACKE_cggqrf */
lapack_int  LAPACKE_cggqrf (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, lapack_complex_float *a, lapack_int lda, lapack_complex_float *taua, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *taub)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub);
    hook_pos_LAPACKE_cggqrf = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggqrf;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggqrf.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, m, p, a, lda, taua, b, ldb, taub);
    } else {
        return fn(matrix_layout, n, m, p, a, lda, taua, b, ldb, taub);
    }
    #else
    return fn(matrix_layout, n, m, p, a, lda, taua, b, ldb, taub);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggqrf(int matrix_layout, lapack_int n, lapack_int m, lapack_int p, lapack_complex_float *a, lapack_int lda, lapack_complex_float *taua, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *taub)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggqrf;
    return fn(matrix_layout, n, m, p, a, lda, taua, b, ldb, taub);
}

lapack_int flexiblas_chain_LAPACKE_cggqrf(int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub);
    hook_pos_LAPACKE_cggqrf++;
    if ( hook_pos_LAPACKE_cggqrf < __flexiblas_hooks->LAPACKE_cggqrf.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggqrf.hook_function[hook_pos_LAPACKE_cggqrf];
    } else {
        hook_pos_LAPACKE_cggqrf = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggqrf;
    }
    return fn(matrix_layout, n, m, p, a, lda, taua, b, ldb, taub);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggqrf_work = 0;
#endif

/* Wrapper for LAPACKE_cggqrf_work */
lapack_int  LAPACKE_cggqrf_work (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, lapack_complex_float *a, lapack_int lda, lapack_complex_float *taua, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *taub, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cggqrf_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggqrf_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggqrf_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, m, p, a, lda, taua, b, ldb, taub, work, lwork);
    } else {
        return fn(matrix_layout, n, m, p, a, lda, taua, b, ldb, taub, work, lwork);
    }
    #else
    return fn(matrix_layout, n, m, p, a, lda, taua, b, ldb, taub, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggqrf_work(int matrix_layout, lapack_int n, lapack_int m, lapack_int p, lapack_complex_float *a, lapack_int lda, lapack_complex_float *taua, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *taub, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggqrf_work;
    return fn(matrix_layout, n, m, p, a, lda, taua, b, ldb, taub, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cggqrf_work(int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int m, lapack_int p, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cggqrf_work++;
    if ( hook_pos_LAPACKE_cggqrf_work < __flexiblas_hooks->LAPACKE_cggqrf_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggqrf_work.hook_function[hook_pos_LAPACKE_cggqrf_work];
    } else {
        hook_pos_LAPACKE_cggqrf_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggqrf_work;
    }
    return fn(matrix_layout, n, m, p, a, lda, taua, b, ldb, taub, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggrqf = 0;
#endif

/* Wrapper for LAPACKE_cggrqf */
lapack_int  LAPACKE_cggrqf (int matrix_layout, lapack_int m, lapack_int p, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *taua, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *taub)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub);
    hook_pos_LAPACKE_cggrqf = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggrqf;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggrqf.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, p, n, a, lda, taua, b, ldb, taub);
    } else {
        return fn(matrix_layout, m, p, n, a, lda, taua, b, ldb, taub);
    }
    #else
    return fn(matrix_layout, m, p, n, a, lda, taua, b, ldb, taub);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggrqf(int matrix_layout, lapack_int m, lapack_int p, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *taua, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *taub)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggrqf;
    return fn(matrix_layout, m, p, n, a, lda, taua, b, ldb, taub);
}

lapack_int flexiblas_chain_LAPACKE_cggrqf(int matrix_layout, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub);
    hook_pos_LAPACKE_cggrqf++;
    if ( hook_pos_LAPACKE_cggrqf < __flexiblas_hooks->LAPACKE_cggrqf.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggrqf.hook_function[hook_pos_LAPACKE_cggrqf];
    } else {
        hook_pos_LAPACKE_cggrqf = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggrqf;
    }
    return fn(matrix_layout, m, p, n, a, lda, taua, b, ldb, taub);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggrqf_work = 0;
#endif

/* Wrapper for LAPACKE_cggrqf_work */
lapack_int  LAPACKE_cggrqf_work (int matrix_layout, lapack_int m, lapack_int p, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *taua, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *taub, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cggrqf_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggrqf_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggrqf_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, p, n, a, lda, taua, b, ldb, taub, work, lwork);
    } else {
        return fn(matrix_layout, m, p, n, a, lda, taua, b, ldb, taub, work, lwork);
    }
    #else
    return fn(matrix_layout, m, p, n, a, lda, taua, b, ldb, taub, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggrqf_work(int matrix_layout, lapack_int m, lapack_int p, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *taua, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *taub, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggrqf_work;
    return fn(matrix_layout, m, p, n, a, lda, taua, b, ldb, taub, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cggrqf_work(int matrix_layout, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *taua, void *b, lapack_int ldb, void *taub, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cggrqf_work++;
    if ( hook_pos_LAPACKE_cggrqf_work < __flexiblas_hooks->LAPACKE_cggrqf_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggrqf_work.hook_function[hook_pos_LAPACKE_cggrqf_work];
    } else {
        hook_pos_LAPACKE_cggrqf_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggrqf_work;
    }
    return fn(matrix_layout, m, p, n, a, lda, taua, b, ldb, taub, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggsvd = 0;
#endif

/* Wrapper for LAPACKE_cggsvd */
lapack_int  LAPACKE_cggsvd (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, lapack_int *k, lapack_int *l, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *alpha, float *beta, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork);
    hook_pos_LAPACKE_cggsvd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggsvd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
    } else {
        return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
    }
    #else
    return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggsvd(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, lapack_int *k, lapack_int *l, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *alpha, float *beta, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvd;
    return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
}

lapack_int flexiblas_chain_LAPACKE_cggsvd(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork);
    hook_pos_LAPACKE_cggsvd++;
    if ( hook_pos_LAPACKE_cggsvd < __flexiblas_hooks->LAPACKE_cggsvd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggsvd.hook_function[hook_pos_LAPACKE_cggsvd];
    } else {
        hook_pos_LAPACKE_cggsvd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvd;
    }
    return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggsvd3 = 0;
#endif

/* Wrapper for LAPACKE_cggsvd3 */
lapack_int  LAPACKE_cggsvd3 (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, lapack_int *k, lapack_int *l, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *alpha, float *beta, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork);
    hook_pos_LAPACKE_cggsvd3 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvd3;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggsvd3.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
    } else {
        return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
    }
    #else
    return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggsvd3(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, lapack_int *k, lapack_int *l, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *alpha, float *beta, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvd3;
    return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
}

lapack_int flexiblas_chain_LAPACKE_cggsvd3(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork);
    hook_pos_LAPACKE_cggsvd3++;
    if ( hook_pos_LAPACKE_cggsvd3 < __flexiblas_hooks->LAPACKE_cggsvd3.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggsvd3.hook_function[hook_pos_LAPACKE_cggsvd3];
    } else {
        hook_pos_LAPACKE_cggsvd3 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvd3;
    }
    return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggsvd3_work = 0;
#endif

/* Wrapper for LAPACKE_cggsvd3_work */
lapack_int  LAPACKE_cggsvd3_work (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, lapack_int *k, lapack_int *l, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *alpha, float *beta, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *work, lapack_int lwork, void *rwork, void *iwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *work, lapack_int lwork, void *rwork, void *iwork);
    hook_pos_LAPACKE_cggsvd3_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvd3_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggsvd3_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, lwork, rwork, iwork);
    } else {
        return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, lwork, rwork, iwork);
    }
    #else
    return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, lwork, rwork, iwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggsvd3_work(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, lapack_int *k, lapack_int *l, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *alpha, float *beta, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *work, lapack_int lwork, void *rwork, void *iwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvd3_work;
    return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, lwork, rwork, iwork);
}

lapack_int flexiblas_chain_LAPACKE_cggsvd3_work(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *work, lapack_int lwork, void *rwork, void *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *work, lapack_int lwork, void *rwork, void *iwork);
    hook_pos_LAPACKE_cggsvd3_work++;
    if ( hook_pos_LAPACKE_cggsvd3_work < __flexiblas_hooks->LAPACKE_cggsvd3_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggsvd3_work.hook_function[hook_pos_LAPACKE_cggsvd3_work];
    } else {
        hook_pos_LAPACKE_cggsvd3_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvd3_work;
    }
    return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, lwork, rwork, iwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggsvd_work = 0;
#endif

/* Wrapper for LAPACKE_cggsvd_work */
lapack_int  LAPACKE_cggsvd_work (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, lapack_int *k, lapack_int *l, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *alpha, float *beta, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *work, float *rwork, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *work, void *rwork, void *iwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *work, void *rwork, void *iwork);
    hook_pos_LAPACKE_cggsvd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggsvd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, rwork, iwork);
    } else {
        return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, rwork, iwork);
    }
    #else
    return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, rwork, iwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggsvd_work(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, lapack_int *k, lapack_int *l, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *alpha, float *beta, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *work, float *rwork, lapack_int *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *work, void *rwork, void *iwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvd_work;
    return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, rwork, iwork);
}

lapack_int flexiblas_chain_LAPACKE_cggsvd_work(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *work, void *rwork, void *iwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int n, lapack_int p, void *k, void *l, void *a, lapack_int lda, void *b, lapack_int ldb, void *alpha, void *beta, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *work, void *rwork, void *iwork);
    hook_pos_LAPACKE_cggsvd_work++;
    if ( hook_pos_LAPACKE_cggsvd_work < __flexiblas_hooks->LAPACKE_cggsvd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggsvd_work.hook_function[hook_pos_LAPACKE_cggsvd_work];
    } else {
        hook_pos_LAPACKE_cggsvd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvd_work;
    }
    return fn(matrix_layout, jobu, jobv, jobq, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, rwork, iwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggsvp = 0;
#endif

/* Wrapper for LAPACKE_cggsvp */
lapack_int  LAPACKE_cggsvp (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float tola, float tolb, lapack_int *k, lapack_int *l, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq);
    hook_pos_LAPACKE_cggsvp = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvp;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggsvp.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
    } else {
        return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
    }
    #else
    return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggsvp(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float tola, float tolb, lapack_int *k, lapack_int *l, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvp;
    return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
}

lapack_int flexiblas_chain_LAPACKE_cggsvp(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq);
    hook_pos_LAPACKE_cggsvp++;
    if ( hook_pos_LAPACKE_cggsvp < __flexiblas_hooks->LAPACKE_cggsvp.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggsvp.hook_function[hook_pos_LAPACKE_cggsvp];
    } else {
        hook_pos_LAPACKE_cggsvp = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvp;
    }
    return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggsvp3 = 0;
#endif

/* Wrapper for LAPACKE_cggsvp3 */
lapack_int  LAPACKE_cggsvp3 (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float tola, float tolb, lapack_int *k, lapack_int *l, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq);
    hook_pos_LAPACKE_cggsvp3 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvp3;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggsvp3.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
    } else {
        return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
    }
    #else
    return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggsvp3(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float tola, float tolb, lapack_int *k, lapack_int *l, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvp3;
    return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
}

lapack_int flexiblas_chain_LAPACKE_cggsvp3(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq);
    hook_pos_LAPACKE_cggsvp3++;
    if ( hook_pos_LAPACKE_cggsvp3 < __flexiblas_hooks->LAPACKE_cggsvp3.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggsvp3.hook_function[hook_pos_LAPACKE_cggsvp3];
    } else {
        hook_pos_LAPACKE_cggsvp3 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvp3;
    }
    return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggsvp3_work = 0;
#endif

/* Wrapper for LAPACKE_cggsvp3_work */
lapack_int  LAPACKE_cggsvp3_work (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float tola, float tolb, lapack_int *k, lapack_int *l, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq, lapack_int *iwork, float *rwork, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork, void *rwork, void *tau, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork, void *rwork, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cggsvp3_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvp3_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggsvp3_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, rwork, tau, work, lwork);
    } else {
        return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, rwork, tau, work, lwork);
    }
    #else
    return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, rwork, tau, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggsvp3_work(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float tola, float tolb, lapack_int *k, lapack_int *l, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq, lapack_int *iwork, float *rwork, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork, void *rwork, void *tau, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvp3_work;
    return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, rwork, tau, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_cggsvp3_work(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork, void *rwork, void *tau, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork, void *rwork, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_cggsvp3_work++;
    if ( hook_pos_LAPACKE_cggsvp3_work < __flexiblas_hooks->LAPACKE_cggsvp3_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggsvp3_work.hook_function[hook_pos_LAPACKE_cggsvp3_work];
    } else {
        hook_pos_LAPACKE_cggsvp3_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvp3_work;
    }
    return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, rwork, tau, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cggsvp_work = 0;
#endif

/* Wrapper for LAPACKE_cggsvp_work */
lapack_int  LAPACKE_cggsvp_work (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float tola, float tolb, lapack_int *k, lapack_int *l, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq, lapack_int *iwork, float *rwork, lapack_complex_float *tau, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork, void *rwork, void *tau, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork, void *rwork, void *tau, void *work);
    hook_pos_LAPACKE_cggsvp_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvp_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cggsvp_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, rwork, tau, work);
    } else {
        return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, rwork, tau, work);
    }
    #else
    return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, rwork, tau, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cggsvp_work(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float tola, float tolb, lapack_int *k, lapack_int *l, lapack_complex_float *u, lapack_int ldu, lapack_complex_float *v, lapack_int ldv, lapack_complex_float *q, lapack_int ldq, lapack_int *iwork, float *rwork, lapack_complex_float *tau, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork, void *rwork, void *tau, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvp_work;
    return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, rwork, tau, work);
}

lapack_int flexiblas_chain_LAPACKE_cggsvp_work(int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork, void *rwork, void *tau, void *work)
{
    lapack_int (*fn) (int matrix_layout, char jobu, char jobv, char jobq, lapack_int m, lapack_int p, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float tola, float tolb, void *k, void *l, void *u, lapack_int ldu, void *v, lapack_int ldv, void *q, lapack_int ldq, void *iwork, void *rwork, void *tau, void *work);
    hook_pos_LAPACKE_cggsvp_work++;
    if ( hook_pos_LAPACKE_cggsvp_work < __flexiblas_hooks->LAPACKE_cggsvp_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cggsvp_work.hook_function[hook_pos_LAPACKE_cggsvp_work];
    } else {
        hook_pos_LAPACKE_cggsvp_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cggsvp_work;
    }
    return fn(matrix_layout, jobu, jobv, jobq, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, rwork, tau, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgtcon = 0;
#endif

/* Wrapper for LAPACKE_cgtcon */
lapack_int  LAPACKE_cgtcon (char norm, lapack_int n, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, const lapack_complex_float *du2, const lapack_int *ipiv, float anorm, float *rcond)
{
    lapack_int (*fn) (char norm, lapack_int n, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, float anorm, void *rcond);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (char norm, lapack_int n, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, float anorm, void *rcond);
    hook_pos_LAPACKE_cgtcon = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtcon;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgtcon.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(norm, n, dl, d, du, du2, ipiv, anorm, rcond);
    } else {
        return fn(norm, n, dl, d, du, du2, ipiv, anorm, rcond);
    }
    #else
    return fn(norm, n, dl, d, du, du2, ipiv, anorm, rcond);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgtcon(char norm, lapack_int n, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, const lapack_complex_float *du2, const lapack_int *ipiv, float anorm, float *rcond)
{
    lapack_int (*fn) (char norm, lapack_int n, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, float anorm, void *rcond);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtcon;
    return fn(norm, n, dl, d, du, du2, ipiv, anorm, rcond);
}

lapack_int flexiblas_chain_LAPACKE_cgtcon(char norm, lapack_int n, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, float anorm, void *rcond)
{
    lapack_int (*fn) (char norm, lapack_int n, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, float anorm, void *rcond);
    hook_pos_LAPACKE_cgtcon++;
    if ( hook_pos_LAPACKE_cgtcon < __flexiblas_hooks->LAPACKE_cgtcon.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgtcon.hook_function[hook_pos_LAPACKE_cgtcon];
    } else {
        hook_pos_LAPACKE_cgtcon = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgtcon;
    }
    return fn(norm, n, dl, d, du, du2, ipiv, anorm, rcond);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgtcon_work = 0;
#endif

/* Wrapper for LAPACKE_cgtcon_work */
lapack_int  LAPACKE_cgtcon_work (char norm, lapack_int n, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, const lapack_complex_float *du2, const lapack_int *ipiv, float anorm, float *rcond, lapack_complex_float *work)
{
    lapack_int (*fn) (char norm, lapack_int n, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, float anorm, void *rcond, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (char norm, lapack_int n, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, float anorm, void *rcond, void *work);
    hook_pos_LAPACKE_cgtcon_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtcon_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgtcon_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(norm, n, dl, d, du, du2, ipiv, anorm, rcond, work);
    } else {
        return fn(norm, n, dl, d, du, du2, ipiv, anorm, rcond, work);
    }
    #else
    return fn(norm, n, dl, d, du, du2, ipiv, anorm, rcond, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgtcon_work(char norm, lapack_int n, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, const lapack_complex_float *du2, const lapack_int *ipiv, float anorm, float *rcond, lapack_complex_float *work)
{
    lapack_int (*fn) (char norm, lapack_int n, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, float anorm, void *rcond, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtcon_work;
    return fn(norm, n, dl, d, du, du2, ipiv, anorm, rcond, work);
}

lapack_int flexiblas_chain_LAPACKE_cgtcon_work(char norm, lapack_int n, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, float anorm, void *rcond, void *work)
{
    lapack_int (*fn) (char norm, lapack_int n, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, float anorm, void *rcond, void *work);
    hook_pos_LAPACKE_cgtcon_work++;
    if ( hook_pos_LAPACKE_cgtcon_work < __flexiblas_hooks->LAPACKE_cgtcon_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgtcon_work.hook_function[hook_pos_LAPACKE_cgtcon_work];
    } else {
        hook_pos_LAPACKE_cgtcon_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgtcon_work;
    }
    return fn(norm, n, dl, d, du, du2, ipiv, anorm, rcond, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgtrfs = 0;
#endif

/* Wrapper for LAPACKE_cgtrfs */
lapack_int  LAPACKE_cgtrfs (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, const lapack_complex_float *dlf, const lapack_complex_float *df, const lapack_complex_float *duf, const lapack_complex_float *du2, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *dlf, const void *df, const void *duf, const void *du2, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *dlf, const void *df, const void *duf, const void *du2, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    hook_pos_LAPACKE_cgtrfs = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtrfs;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgtrfs.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr);
    } else {
        return fn(matrix_layout, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr);
    }
    #else
    return fn(matrix_layout, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgtrfs(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, const lapack_complex_float *dlf, const lapack_complex_float *df, const lapack_complex_float *duf, const lapack_complex_float *du2, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *dlf, const void *df, const void *duf, const void *du2, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtrfs;
    return fn(matrix_layout, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr);
}

lapack_int flexiblas_chain_LAPACKE_cgtrfs(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *dlf, const void *df, const void *duf, const void *du2, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *dlf, const void *df, const void *duf, const void *du2, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    hook_pos_LAPACKE_cgtrfs++;
    if ( hook_pos_LAPACKE_cgtrfs < __flexiblas_hooks->LAPACKE_cgtrfs.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgtrfs.hook_function[hook_pos_LAPACKE_cgtrfs];
    } else {
        hook_pos_LAPACKE_cgtrfs = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgtrfs;
    }
    return fn(matrix_layout, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgtrfs_work = 0;
#endif

/* Wrapper for LAPACKE_cgtrfs_work */
lapack_int  LAPACKE_cgtrfs_work (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, const lapack_complex_float *dlf, const lapack_complex_float *df, const lapack_complex_float *duf, const lapack_complex_float *du2, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *dlf, const void *df, const void *duf, const void *du2, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *dlf, const void *df, const void *duf, const void *du2, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cgtrfs_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtrfs_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgtrfs_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    } else {
        return fn(matrix_layout, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    }
    #else
    return fn(matrix_layout, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgtrfs_work(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, const lapack_complex_float *dlf, const lapack_complex_float *df, const lapack_complex_float *duf, const lapack_complex_float *du2, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *dlf, const void *df, const void *duf, const void *du2, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtrfs_work;
    return fn(matrix_layout, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgtrfs_work(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *dlf, const void *df, const void *duf, const void *du2, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *dlf, const void *df, const void *duf, const void *du2, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cgtrfs_work++;
    if ( hook_pos_LAPACKE_cgtrfs_work < __flexiblas_hooks->LAPACKE_cgtrfs_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgtrfs_work.hook_function[hook_pos_LAPACKE_cgtrfs_work];
    } else {
        hook_pos_LAPACKE_cgtrfs_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgtrfs_work;
    }
    return fn(matrix_layout, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgtsv = 0;
#endif

/* Wrapper for LAPACKE_cgtsv */
lapack_int  LAPACKE_cgtsv (int matrix_layout, lapack_int n, lapack_int nrhs, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int nrhs, void *dl, void *d, void *du, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, lapack_int nrhs, void *dl, void *d, void *du, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgtsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgtsv.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, nrhs, dl, d, du, b, ldb);
    } else {
        return fn(matrix_layout, n, nrhs, dl, d, du, b, ldb);
    }
    #else
    return fn(matrix_layout, n, nrhs, dl, d, du, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgtsv(int matrix_layout, lapack_int n, lapack_int nrhs, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int nrhs, void *dl, void *d, void *du, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtsv;
    return fn(matrix_layout, n, nrhs, dl, d, du, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgtsv(int matrix_layout, lapack_int n, lapack_int nrhs, void *dl, void *d, void *du, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int nrhs, void *dl, void *d, void *du, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgtsv++;
    if ( hook_pos_LAPACKE_cgtsv < __flexiblas_hooks->LAPACKE_cgtsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgtsv.hook_function[hook_pos_LAPACKE_cgtsv];
    } else {
        hook_pos_LAPACKE_cgtsv = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgtsv;
    }
    return fn(matrix_layout, n, nrhs, dl, d, du, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgtsv_work = 0;
#endif

/* Wrapper for LAPACKE_cgtsv_work */
lapack_int  LAPACKE_cgtsv_work (int matrix_layout, lapack_int n, lapack_int nrhs, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int nrhs, void *dl, void *d, void *du, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int n, lapack_int nrhs, void *dl, void *d, void *du, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgtsv_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtsv_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgtsv_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, n, nrhs, dl, d, du, b, ldb);
    } else {
        return fn(matrix_layout, n, nrhs, dl, d, du, b, ldb);
    }
    #else
    return fn(matrix_layout, n, nrhs, dl, d, du, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgtsv_work(int matrix_layout, lapack_int n, lapack_int nrhs, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int nrhs, void *dl, void *d, void *du, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtsv_work;
    return fn(matrix_layout, n, nrhs, dl, d, du, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgtsv_work(int matrix_layout, lapack_int n, lapack_int nrhs, void *dl, void *d, void *du, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int n, lapack_int nrhs, void *dl, void *d, void *du, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgtsv_work++;
    if ( hook_pos_LAPACKE_cgtsv_work < __flexiblas_hooks->LAPACKE_cgtsv_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgtsv_work.hook_function[hook_pos_LAPACKE_cgtsv_work];
    } else {
        hook_pos_LAPACKE_cgtsv_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgtsv_work;
    }
    return fn(matrix_layout, n, nrhs, dl, d, du, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgtsvx = 0;
#endif

/* Wrapper for LAPACKE_cgtsvx */
lapack_int  LAPACKE_cgtsvx (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, lapack_complex_float *dlf, lapack_complex_float *df, lapack_complex_float *duf, lapack_complex_float *du2, lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, void *dlf, void *df, void *duf, void *du2, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, void *dlf, void *df, void *duf, void *du2, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr);
    hook_pos_LAPACKE_cgtsvx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtsvx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgtsvx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, fact, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
    } else {
        return fn(matrix_layout, fact, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
    }
    #else
    return fn(matrix_layout, fact, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgtsvx(int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, lapack_complex_float *dlf, lapack_complex_float *df, lapack_complex_float *duf, lapack_complex_float *du2, lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, void *dlf, void *df, void *duf, void *du2, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtsvx;
    return fn(matrix_layout, fact, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

lapack_int flexiblas_chain_LAPACKE_cgtsvx(int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, void *dlf, void *df, void *duf, void *du2, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, void *dlf, void *df, void *duf, void *du2, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr);
    hook_pos_LAPACKE_cgtsvx++;
    if ( hook_pos_LAPACKE_cgtsvx < __flexiblas_hooks->LAPACKE_cgtsvx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgtsvx.hook_function[hook_pos_LAPACKE_cgtsvx];
    } else {
        hook_pos_LAPACKE_cgtsvx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgtsvx;
    }
    return fn(matrix_layout, fact, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgtsvx_work = 0;
#endif

/* Wrapper for LAPACKE_cgtsvx_work */
lapack_int  LAPACKE_cgtsvx_work (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, lapack_complex_float *dlf, lapack_complex_float *df, lapack_complex_float *duf, lapack_complex_float *du2, lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, void *dlf, void *df, void *duf, void *du2, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, void *dlf, void *df, void *duf, void *du2, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cgtsvx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtsvx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgtsvx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, fact, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
    } else {
        return fn(matrix_layout, fact, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
    }
    #else
    return fn(matrix_layout, fact, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgtsvx_work(int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, lapack_complex_float *dlf, lapack_complex_float *df, lapack_complex_float *duf, lapack_complex_float *du2, lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, void *dlf, void *df, void *duf, void *du2, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgtsvx_work;
    return fn(matrix_layout, fact, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cgtsvx_work(int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, void *dlf, void *df, void *duf, void *du2, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, void *dlf, void *df, void *duf, void *du2, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cgtsvx_work++;
    if ( hook_pos_LAPACKE_cgtsvx_work < __flexiblas_hooks->LAPACKE_cgtsvx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgtsvx_work.hook_function[hook_pos_LAPACKE_cgtsvx_work];
    } else {
        hook_pos_LAPACKE_cgtsvx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgtsvx_work;
    }
    return fn(matrix_layout, fact, trans, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgttrf = 0;
#endif

/* Wrapper for LAPACKE_cgttrf */
lapack_int  LAPACKE_cgttrf (lapack_int n, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *du2, lapack_int *ipiv)
{
    lapack_int (*fn) (lapack_int n, void *dl, void *d, void *du, void *du2, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (lapack_int n, void *dl, void *d, void *du, void *du2, void *ipiv);
    hook_pos_LAPACKE_cgttrf = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgttrf;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgttrf.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(n, dl, d, du, du2, ipiv);
    } else {
        return fn(n, dl, d, du, du2, ipiv);
    }
    #else
    return fn(n, dl, d, du, du2, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgttrf(lapack_int n, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *du2, lapack_int *ipiv)
{
    lapack_int (*fn) (lapack_int n, void *dl, void *d, void *du, void *du2, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgttrf;
    return fn(n, dl, d, du, du2, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_cgttrf(lapack_int n, void *dl, void *d, void *du, void *du2, void *ipiv)
{
    lapack_int (*fn) (lapack_int n, void *dl, void *d, void *du, void *du2, void *ipiv);
    hook_pos_LAPACKE_cgttrf++;
    if ( hook_pos_LAPACKE_cgttrf < __flexiblas_hooks->LAPACKE_cgttrf.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgttrf.hook_function[hook_pos_LAPACKE_cgttrf];
    } else {
        hook_pos_LAPACKE_cgttrf = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgttrf;
    }
    return fn(n, dl, d, du, du2, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgttrf_work = 0;
#endif

/* Wrapper for LAPACKE_cgttrf_work */
lapack_int  LAPACKE_cgttrf_work (lapack_int n, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *du2, lapack_int *ipiv)
{
    lapack_int (*fn) (lapack_int n, void *dl, void *d, void *du, void *du2, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (lapack_int n, void *dl, void *d, void *du, void *du2, void *ipiv);
    hook_pos_LAPACKE_cgttrf_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgttrf_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgttrf_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(n, dl, d, du, du2, ipiv);
    } else {
        return fn(n, dl, d, du, du2, ipiv);
    }
    #else
    return fn(n, dl, d, du, du2, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgttrf_work(lapack_int n, lapack_complex_float *dl, lapack_complex_float *d, lapack_complex_float *du, lapack_complex_float *du2, lapack_int *ipiv)
{
    lapack_int (*fn) (lapack_int n, void *dl, void *d, void *du, void *du2, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgttrf_work;
    return fn(n, dl, d, du, du2, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_cgttrf_work(lapack_int n, void *dl, void *d, void *du, void *du2, void *ipiv)
{
    lapack_int (*fn) (lapack_int n, void *dl, void *d, void *du, void *du2, void *ipiv);
    hook_pos_LAPACKE_cgttrf_work++;
    if ( hook_pos_LAPACKE_cgttrf_work < __flexiblas_hooks->LAPACKE_cgttrf_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgttrf_work.hook_function[hook_pos_LAPACKE_cgttrf_work];
    } else {
        hook_pos_LAPACKE_cgttrf_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgttrf_work;
    }
    return fn(n, dl, d, du, du2, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgttrs = 0;
#endif

/* Wrapper for LAPACKE_cgttrs */
lapack_int  LAPACKE_cgttrs (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, const lapack_complex_float *du2, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgttrs = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgttrs;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgttrs.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, trans, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, trans, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgttrs(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, const lapack_complex_float *du2, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgttrs;
    return fn(matrix_layout, trans, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgttrs(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgttrs++;
    if ( hook_pos_LAPACKE_cgttrs < __flexiblas_hooks->LAPACKE_cgttrs.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgttrs.hook_function[hook_pos_LAPACKE_cgttrs];
    } else {
        hook_pos_LAPACKE_cgttrs = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgttrs;
    }
    return fn(matrix_layout, trans, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cgttrs_work = 0;
#endif

/* Wrapper for LAPACKE_cgttrs_work */
lapack_int  LAPACKE_cgttrs_work (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, const lapack_complex_float *du2, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgttrs_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cgttrs_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cgttrs_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, trans, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, trans, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, trans, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cgttrs_work(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const lapack_complex_float *dl, const lapack_complex_float *d, const lapack_complex_float *du, const lapack_complex_float *du2, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cgttrs_work;
    return fn(matrix_layout, trans, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_cgttrs_work(int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char trans, lapack_int n, lapack_int nrhs, const void *dl, const void *d, const void *du, const void *du2, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_cgttrs_work++;
    if ( hook_pos_LAPACKE_cgttrs_work < __flexiblas_hooks->LAPACKE_cgttrs_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cgttrs_work.hook_function[hook_pos_LAPACKE_cgttrs_work];
    } else {
        hook_pos_LAPACKE_cgttrs_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cgttrs_work;
    }
    return fn(matrix_layout, trans, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbev = 0;
#endif

/* Wrapper for LAPACKE_chbev */
lapack_int  LAPACKE_chbev (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chbev = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbev;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbev.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
    } else {
        return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbev(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbev;
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_chbev(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chbev++;
    if ( hook_pos_LAPACKE_chbev < __flexiblas_hooks->LAPACKE_chbev.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbev.hook_function[hook_pos_LAPACKE_chbev];
    } else {
        hook_pos_LAPACKE_chbev = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbev;
    }
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbev_2stage = 0;
#endif

/* Wrapper for LAPACKE_chbev_2stage */
lapack_int  LAPACKE_chbev_2stage (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chbev_2stage = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbev_2stage;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbev_2stage.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
    } else {
        return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbev_2stage(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbev_2stage;
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_chbev_2stage(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chbev_2stage++;
    if ( hook_pos_LAPACKE_chbev_2stage < __flexiblas_hooks->LAPACKE_chbev_2stage.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbev_2stage.hook_function[hook_pos_LAPACKE_chbev_2stage];
    } else {
        hook_pos_LAPACKE_chbev_2stage = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbev_2stage;
    }
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbev_2stage_work = 0;
#endif

/* Wrapper for LAPACKE_chbev_2stage_work */
lapack_int  LAPACKE_chbev_2stage_work (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_chbev_2stage_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbev_2stage_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbev_2stage_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork);
    } else {
        return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbev_2stage_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbev_2stage_work;
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_chbev_2stage_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_chbev_2stage_work++;
    if ( hook_pos_LAPACKE_chbev_2stage_work < __flexiblas_hooks->LAPACKE_chbev_2stage_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbev_2stage_work.hook_function[hook_pos_LAPACKE_chbev_2stage_work];
    } else {
        hook_pos_LAPACKE_chbev_2stage_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbev_2stage_work;
    }
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbev_work = 0;
#endif

/* Wrapper for LAPACKE_chbev_work */
lapack_int  LAPACKE_chbev_work (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    hook_pos_LAPACKE_chbev_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbev_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbev_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, rwork);
    } else {
        return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, rwork);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbev_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbev_work;
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_chbev_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    hook_pos_LAPACKE_chbev_work++;
    if ( hook_pos_LAPACKE_chbev_work < __flexiblas_hooks->LAPACKE_chbev_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbev_work.hook_function[hook_pos_LAPACKE_chbev_work];
    } else {
        hook_pos_LAPACKE_chbev_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbev_work;
    }
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbevd = 0;
#endif

/* Wrapper for LAPACKE_chbevd */
lapack_int  LAPACKE_chbevd (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chbevd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbevd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
    } else {
        return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbevd(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevd;
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_chbevd(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chbevd++;
    if ( hook_pos_LAPACKE_chbevd < __flexiblas_hooks->LAPACKE_chbevd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbevd.hook_function[hook_pos_LAPACKE_chbevd];
    } else {
        hook_pos_LAPACKE_chbevd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbevd;
    }
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbevd_2stage = 0;
#endif

/* Wrapper for LAPACKE_chbevd_2stage */
lapack_int  LAPACKE_chbevd_2stage (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chbevd_2stage = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevd_2stage;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbevd_2stage.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
    } else {
        return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbevd_2stage(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevd_2stage;
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_chbevd_2stage(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chbevd_2stage++;
    if ( hook_pos_LAPACKE_chbevd_2stage < __flexiblas_hooks->LAPACKE_chbevd_2stage.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbevd_2stage.hook_function[hook_pos_LAPACKE_chbevd_2stage];
    } else {
        hook_pos_LAPACKE_chbevd_2stage = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbevd_2stage;
    }
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbevd_2stage_work = 0;
#endif

/* Wrapper for LAPACKE_chbevd_2stage_work */
lapack_int  LAPACKE_chbevd_2stage_work (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_chbevd_2stage_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevd_2stage_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbevd_2stage_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    } else {
        return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbevd_2stage_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevd_2stage_work;
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

lapack_int flexiblas_chain_LAPACKE_chbevd_2stage_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_chbevd_2stage_work++;
    if ( hook_pos_LAPACKE_chbevd_2stage_work < __flexiblas_hooks->LAPACKE_chbevd_2stage_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbevd_2stage_work.hook_function[hook_pos_LAPACKE_chbevd_2stage_work];
    } else {
        hook_pos_LAPACKE_chbevd_2stage_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbevd_2stage_work;
    }
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbevd_work = 0;
#endif

/* Wrapper for LAPACKE_chbevd_work */
lapack_int  LAPACKE_chbevd_work (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_chbevd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbevd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    } else {
        return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbevd_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevd_work;
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

lapack_int flexiblas_chain_LAPACKE_chbevd_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_chbevd_work++;
    if ( hook_pos_LAPACKE_chbevd_work < __flexiblas_hooks->LAPACKE_chbevd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbevd_work.hook_function[hook_pos_LAPACKE_chbevd_work];
    } else {
        hook_pos_LAPACKE_chbevd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbevd_work;
    }
    return fn(matrix_layout, jobz, uplo, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbevx = 0;
#endif

/* Wrapper for LAPACKE_chbevx */
lapack_int  LAPACKE_chbevx (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_chbevx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbevx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbevx(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevx;
    return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

lapack_int flexiblas_chain_LAPACKE_chbevx(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_chbevx++;
    if ( hook_pos_LAPACKE_chbevx < __flexiblas_hooks->LAPACKE_chbevx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbevx.hook_function[hook_pos_LAPACKE_chbevx];
    } else {
        hook_pos_LAPACKE_chbevx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbevx;
    }
    return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbevx_2stage = 0;
#endif

/* Wrapper for LAPACKE_chbevx_2stage */
lapack_int  LAPACKE_chbevx_2stage (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_chbevx_2stage = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevx_2stage;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbevx_2stage.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbevx_2stage(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevx_2stage;
    return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

lapack_int flexiblas_chain_LAPACKE_chbevx_2stage(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_chbevx_2stage++;
    if ( hook_pos_LAPACKE_chbevx_2stage < __flexiblas_hooks->LAPACKE_chbevx_2stage.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbevx_2stage.hook_function[hook_pos_LAPACKE_chbevx_2stage];
    } else {
        hook_pos_LAPACKE_chbevx_2stage = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbevx_2stage;
    }
    return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbevx_2stage_work = 0;
#endif

/* Wrapper for LAPACKE_chbevx_2stage_work */
lapack_int  LAPACKE_chbevx_2stage_work (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_chbevx_2stage_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevx_2stage_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbevx_2stage_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbevx_2stage_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevx_2stage_work;
    return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
}

lapack_int flexiblas_chain_LAPACKE_chbevx_2stage_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_chbevx_2stage_work++;
    if ( hook_pos_LAPACKE_chbevx_2stage_work < __flexiblas_hooks->LAPACKE_chbevx_2stage_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbevx_2stage_work.hook_function[hook_pos_LAPACKE_chbevx_2stage_work];
    } else {
        hook_pos_LAPACKE_chbevx_2stage_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbevx_2stage_work;
    }
    return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbevx_work = 0;
#endif

/* Wrapper for LAPACKE_chbevx_work */
lapack_int  LAPACKE_chbevx_work (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_chbevx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbevx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbevx_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbevx_work;
    return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

lapack_int flexiblas_chain_LAPACKE_chbevx_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_chbevx_work++;
    if ( hook_pos_LAPACKE_chbevx_work < __flexiblas_hooks->LAPACKE_chbevx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbevx_work.hook_function[hook_pos_LAPACKE_chbevx_work];
    } else {
        hook_pos_LAPACKE_chbevx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbevx_work;
    }
    return fn(matrix_layout, jobz, range, uplo, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbgst = 0;
#endif

/* Wrapper for LAPACKE_chbgst */
lapack_int  LAPACKE_chbgst (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, const lapack_complex_float *bb, lapack_int ldbb, lapack_complex_float *x, lapack_int ldx)
{
    lapack_int (*fn) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, const void *bb, lapack_int ldbb, void *x, lapack_int ldx);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, const void *bb, lapack_int ldbb, void *x, lapack_int ldx);
    hook_pos_LAPACKE_chbgst = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgst;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbgst.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, vect, uplo, n, ka, kb, ab, ldab, bb, ldbb, x, ldx);
    } else {
        return fn(matrix_layout, vect, uplo, n, ka, kb, ab, ldab, bb, ldbb, x, ldx);
    }
    #else
    return fn(matrix_layout, vect, uplo, n, ka, kb, ab, ldab, bb, ldbb, x, ldx);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbgst(int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, const lapack_complex_float *bb, lapack_int ldbb, lapack_complex_float *x, lapack_int ldx)
{
    lapack_int (*fn) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, const void *bb, lapack_int ldbb, void *x, lapack_int ldx);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgst;
    return fn(matrix_layout, vect, uplo, n, ka, kb, ab, ldab, bb, ldbb, x, ldx);
}

lapack_int flexiblas_chain_LAPACKE_chbgst(int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, const void *bb, lapack_int ldbb, void *x, lapack_int ldx)
{
    lapack_int (*fn) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, const void *bb, lapack_int ldbb, void *x, lapack_int ldx);
    hook_pos_LAPACKE_chbgst++;
    if ( hook_pos_LAPACKE_chbgst < __flexiblas_hooks->LAPACKE_chbgst.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbgst.hook_function[hook_pos_LAPACKE_chbgst];
    } else {
        hook_pos_LAPACKE_chbgst = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbgst;
    }
    return fn(matrix_layout, vect, uplo, n, ka, kb, ab, ldab, bb, ldbb, x, ldx);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbgst_work = 0;
#endif

/* Wrapper for LAPACKE_chbgst_work */
lapack_int  LAPACKE_chbgst_work (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, const lapack_complex_float *bb, lapack_int ldbb, lapack_complex_float *x, lapack_int ldx, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, const void *bb, lapack_int ldbb, void *x, lapack_int ldx, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, const void *bb, lapack_int ldbb, void *x, lapack_int ldx, void *work, void *rwork);
    hook_pos_LAPACKE_chbgst_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgst_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbgst_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, vect, uplo, n, ka, kb, ab, ldab, bb, ldbb, x, ldx, work, rwork);
    } else {
        return fn(matrix_layout, vect, uplo, n, ka, kb, ab, ldab, bb, ldbb, x, ldx, work, rwork);
    }
    #else
    return fn(matrix_layout, vect, uplo, n, ka, kb, ab, ldab, bb, ldbb, x, ldx, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbgst_work(int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, const lapack_complex_float *bb, lapack_int ldbb, lapack_complex_float *x, lapack_int ldx, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, const void *bb, lapack_int ldbb, void *x, lapack_int ldx, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgst_work;
    return fn(matrix_layout, vect, uplo, n, ka, kb, ab, ldab, bb, ldbb, x, ldx, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_chbgst_work(int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, const void *bb, lapack_int ldbb, void *x, lapack_int ldx, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, const void *bb, lapack_int ldbb, void *x, lapack_int ldx, void *work, void *rwork);
    hook_pos_LAPACKE_chbgst_work++;
    if ( hook_pos_LAPACKE_chbgst_work < __flexiblas_hooks->LAPACKE_chbgst_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbgst_work.hook_function[hook_pos_LAPACKE_chbgst_work];
    } else {
        hook_pos_LAPACKE_chbgst_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbgst_work;
    }
    return fn(matrix_layout, vect, uplo, n, ka, kb, ab, ldab, bb, ldbb, x, ldx, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbgv = 0;
#endif

/* Wrapper for LAPACKE_chbgv */
lapack_int  LAPACKE_chbgv (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *bb, lapack_int ldbb, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chbgv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbgv.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
    } else {
        return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbgv(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *bb, lapack_int ldbb, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgv;
    return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_chbgv(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chbgv++;
    if ( hook_pos_LAPACKE_chbgv < __flexiblas_hooks->LAPACKE_chbgv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbgv.hook_function[hook_pos_LAPACKE_chbgv];
    } else {
        hook_pos_LAPACKE_chbgv = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbgv;
    }
    return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbgv_work = 0;
#endif

/* Wrapper for LAPACKE_chbgv_work */
lapack_int  LAPACKE_chbgv_work (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *bb, lapack_int ldbb, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    hook_pos_LAPACKE_chbgv_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgv_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbgv_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, rwork);
    } else {
        return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, rwork);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbgv_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *bb, lapack_int ldbb, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgv_work;
    return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_chbgv_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    hook_pos_LAPACKE_chbgv_work++;
    if ( hook_pos_LAPACKE_chbgv_work < __flexiblas_hooks->LAPACKE_chbgv_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbgv_work.hook_function[hook_pos_LAPACKE_chbgv_work];
    } else {
        hook_pos_LAPACKE_chbgv_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbgv_work;
    }
    return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbgvd = 0;
#endif

/* Wrapper for LAPACKE_chbgvd */
lapack_int  LAPACKE_chbgvd (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *bb, lapack_int ldbb, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chbgvd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgvd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbgvd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
    } else {
        return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbgvd(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *bb, lapack_int ldbb, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgvd;
    return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_chbgvd(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chbgvd++;
    if ( hook_pos_LAPACKE_chbgvd < __flexiblas_hooks->LAPACKE_chbgvd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbgvd.hook_function[hook_pos_LAPACKE_chbgvd];
    } else {
        hook_pos_LAPACKE_chbgvd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbgvd;
    }
    return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbgvd_work = 0;
#endif

/* Wrapper for LAPACKE_chbgvd_work */
lapack_int  LAPACKE_chbgvd_work (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *bb, lapack_int ldbb, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_chbgvd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgvd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbgvd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    } else {
        return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbgvd_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *bb, lapack_int ldbb, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgvd_work;
    return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

lapack_int flexiblas_chain_LAPACKE_chbgvd_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_chbgvd_work++;
    if ( hook_pos_LAPACKE_chbgvd_work < __flexiblas_hooks->LAPACKE_chbgvd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbgvd_work.hook_function[hook_pos_LAPACKE_chbgvd_work];
    } else {
        hook_pos_LAPACKE_chbgvd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbgvd_work;
    }
    return fn(matrix_layout, jobz, uplo, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbgvx = 0;
#endif

/* Wrapper for LAPACKE_chbgvx */
lapack_int  LAPACKE_chbgvx (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *bb, lapack_int ldbb, lapack_complex_float *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_chbgvx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgvx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbgvx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbgvx(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *bb, lapack_int ldbb, lapack_complex_float *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgvx;
    return fn(matrix_layout, jobz, range, uplo, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

lapack_int flexiblas_chain_LAPACKE_chbgvx(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_chbgvx++;
    if ( hook_pos_LAPACKE_chbgvx < __flexiblas_hooks->LAPACKE_chbgvx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbgvx.hook_function[hook_pos_LAPACKE_chbgvx];
    } else {
        hook_pos_LAPACKE_chbgvx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbgvx;
    }
    return fn(matrix_layout, jobz, range, uplo, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbgvx_work = 0;
#endif

/* Wrapper for LAPACKE_chbgvx_work */
lapack_int  LAPACKE_chbgvx_work (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *bb, lapack_int ldbb, lapack_complex_float *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_chbgvx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgvx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbgvx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbgvx_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, lapack_complex_float *ab, lapack_int ldab, lapack_complex_float *bb, lapack_int ldbb, lapack_complex_float *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbgvx_work;
    return fn(matrix_layout, jobz, range, uplo, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

lapack_int flexiblas_chain_LAPACKE_chbgvx_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_int ka, lapack_int kb, void *ab, lapack_int ldab, void *bb, lapack_int ldbb, void *q, lapack_int ldq, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_chbgvx_work++;
    if ( hook_pos_LAPACKE_chbgvx_work < __flexiblas_hooks->LAPACKE_chbgvx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbgvx_work.hook_function[hook_pos_LAPACKE_chbgvx_work];
    } else {
        hook_pos_LAPACKE_chbgvx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbgvx_work;
    }
    return fn(matrix_layout, jobz, range, uplo, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbtrd = 0;
#endif

/* Wrapper for LAPACKE_chbtrd */
lapack_int  LAPACKE_chbtrd (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *d, float *e, lapack_complex_float *q, lapack_int ldq)
{
    lapack_int (*fn) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq);
    hook_pos_LAPACKE_chbtrd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbtrd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbtrd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, vect, uplo, n, kd, ab, ldab, d, e, q, ldq);
    } else {
        return fn(matrix_layout, vect, uplo, n, kd, ab, ldab, d, e, q, ldq);
    }
    #else
    return fn(matrix_layout, vect, uplo, n, kd, ab, ldab, d, e, q, ldq);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbtrd(int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *d, float *e, lapack_complex_float *q, lapack_int ldq)
{
    lapack_int (*fn) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbtrd;
    return fn(matrix_layout, vect, uplo, n, kd, ab, ldab, d, e, q, ldq);
}

lapack_int flexiblas_chain_LAPACKE_chbtrd(int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq)
{
    lapack_int (*fn) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq);
    hook_pos_LAPACKE_chbtrd++;
    if ( hook_pos_LAPACKE_chbtrd < __flexiblas_hooks->LAPACKE_chbtrd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbtrd.hook_function[hook_pos_LAPACKE_chbtrd];
    } else {
        hook_pos_LAPACKE_chbtrd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbtrd;
    }
    return fn(matrix_layout, vect, uplo, n, kd, ab, ldab, d, e, q, ldq);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chbtrd_work = 0;
#endif

/* Wrapper for LAPACKE_chbtrd_work */
lapack_int  LAPACKE_chbtrd_work (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *d, float *e, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *work);
    hook_pos_LAPACKE_chbtrd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chbtrd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chbtrd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, vect, uplo, n, kd, ab, ldab, d, e, q, ldq, work);
    } else {
        return fn(matrix_layout, vect, uplo, n, kd, ab, ldab, d, e, q, ldq, work);
    }
    #else
    return fn(matrix_layout, vect, uplo, n, kd, ab, ldab, d, e, q, ldq, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chbtrd_work(int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, lapack_complex_float *ab, lapack_int ldab, float *d, float *e, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chbtrd_work;
    return fn(matrix_layout, vect, uplo, n, kd, ab, ldab, d, e, q, ldq, work);
}

lapack_int flexiblas_chain_LAPACKE_chbtrd_work(int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *work)
{
    lapack_int (*fn) (int matrix_layout, char vect, char uplo, lapack_int n, lapack_int kd, void *ab, lapack_int ldab, void *d, void *e, void *q, lapack_int ldq, void *work);
    hook_pos_LAPACKE_chbtrd_work++;
    if ( hook_pos_LAPACKE_chbtrd_work < __flexiblas_hooks->LAPACKE_chbtrd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chbtrd_work.hook_function[hook_pos_LAPACKE_chbtrd_work];
    } else {
        hook_pos_LAPACKE_chbtrd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chbtrd_work;
    }
    return fn(matrix_layout, vect, uplo, n, kd, ab, ldab, d, e, q, ldq, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_checon = 0;
#endif

/* Wrapper for LAPACKE_checon */
lapack_int  LAPACKE_checon (int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, float anorm, float *rcond)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *ipiv, float anorm, void *rcond);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *ipiv, float anorm, void *rcond);
    hook_pos_LAPACKE_checon = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_checon;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_checon.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv, anorm, rcond);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv, anorm, rcond);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv, anorm, rcond);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_checon(int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, float anorm, float *rcond)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *ipiv, float anorm, void *rcond);
    *(void **) &fn = current_backend->lapacke.LAPACKE_checon;
    return fn(matrix_layout, uplo, n, a, lda, ipiv, anorm, rcond);
}

lapack_int flexiblas_chain_LAPACKE_checon(int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *ipiv, float anorm, void *rcond)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *ipiv, float anorm, void *rcond);
    hook_pos_LAPACKE_checon++;
    if ( hook_pos_LAPACKE_checon < __flexiblas_hooks->LAPACKE_checon.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_checon.hook_function[hook_pos_LAPACKE_checon];
    } else {
        hook_pos_LAPACKE_checon = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_checon;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv, anorm, rcond);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_checon_3 = 0;
#endif

/* Wrapper for LAPACKE_checon_3 */
lapack_int  LAPACKE_checon_3 (int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *e, const lapack_int *ipiv, float anorm, float *rcond)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *e, const void *ipiv, float anorm, void *rcond);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *e, const void *ipiv, float anorm, void *rcond);
    hook_pos_LAPACKE_checon_3 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_checon_3;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_checon_3.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, e, ipiv, anorm, rcond);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, e, ipiv, anorm, rcond);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv, anorm, rcond);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_checon_3(int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *e, const lapack_int *ipiv, float anorm, float *rcond)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *e, const void *ipiv, float anorm, void *rcond);
    *(void **) &fn = current_backend->lapacke.LAPACKE_checon_3;
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv, anorm, rcond);
}

lapack_int flexiblas_chain_LAPACKE_checon_3(int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *e, const void *ipiv, float anorm, void *rcond)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *e, const void *ipiv, float anorm, void *rcond);
    hook_pos_LAPACKE_checon_3++;
    if ( hook_pos_LAPACKE_checon_3 < __flexiblas_hooks->LAPACKE_checon_3.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_checon_3.hook_function[hook_pos_LAPACKE_checon_3];
    } else {
        hook_pos_LAPACKE_checon_3 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_checon_3;
    }
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv, anorm, rcond);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_checon_3_work = 0;
#endif

/* Wrapper for LAPACKE_checon_3_work */
lapack_int  LAPACKE_checon_3_work (int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *e, const lapack_int *ipiv, float anorm, float *rcond, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *e, const void *ipiv, float anorm, void *rcond, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *e, const void *ipiv, float anorm, void *rcond, void *work);
    hook_pos_LAPACKE_checon_3_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_checon_3_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_checon_3_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, e, ipiv, anorm, rcond, work);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, e, ipiv, anorm, rcond, work);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv, anorm, rcond, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_checon_3_work(int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *e, const lapack_int *ipiv, float anorm, float *rcond, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *e, const void *ipiv, float anorm, void *rcond, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_checon_3_work;
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv, anorm, rcond, work);
}

lapack_int flexiblas_chain_LAPACKE_checon_3_work(int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *e, const void *ipiv, float anorm, void *rcond, void *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *e, const void *ipiv, float anorm, void *rcond, void *work);
    hook_pos_LAPACKE_checon_3_work++;
    if ( hook_pos_LAPACKE_checon_3_work < __flexiblas_hooks->LAPACKE_checon_3_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_checon_3_work.hook_function[hook_pos_LAPACKE_checon_3_work];
    } else {
        hook_pos_LAPACKE_checon_3_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_checon_3_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv, anorm, rcond, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_checon_work = 0;
#endif

/* Wrapper for LAPACKE_checon_work */
lapack_int  LAPACKE_checon_work (int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, float anorm, float *rcond, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *ipiv, float anorm, void *rcond, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *ipiv, float anorm, void *rcond, void *work);
    hook_pos_LAPACKE_checon_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_checon_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_checon_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv, anorm, rcond, work);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv, anorm, rcond, work);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv, anorm, rcond, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_checon_work(int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, float anorm, float *rcond, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *ipiv, float anorm, void *rcond, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_checon_work;
    return fn(matrix_layout, uplo, n, a, lda, ipiv, anorm, rcond, work);
}

lapack_int flexiblas_chain_LAPACKE_checon_work(int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *ipiv, float anorm, void *rcond, void *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, const void *ipiv, float anorm, void *rcond, void *work);
    hook_pos_LAPACKE_checon_work++;
    if ( hook_pos_LAPACKE_checon_work < __flexiblas_hooks->LAPACKE_checon_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_checon_work.hook_function[hook_pos_LAPACKE_checon_work];
    } else {
        hook_pos_LAPACKE_checon_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_checon_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv, anorm, rcond, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheequb = 0;
#endif

/* Wrapper for LAPACKE_cheequb */
lapack_int  LAPACKE_cheequb (int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *s, float *scond, float *amax)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, void *s, void *scond, void *amax);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, void *s, void *scond, void *amax);
    hook_pos_LAPACKE_cheequb = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheequb;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheequb.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, s, scond, amax);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, s, scond, amax);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, s, scond, amax);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheequb(int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *s, float *scond, float *amax)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, void *s, void *scond, void *amax);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheequb;
    return fn(matrix_layout, uplo, n, a, lda, s, scond, amax);
}

lapack_int flexiblas_chain_LAPACKE_cheequb(int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, void *s, void *scond, void *amax)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, void *s, void *scond, void *amax);
    hook_pos_LAPACKE_cheequb++;
    if ( hook_pos_LAPACKE_cheequb < __flexiblas_hooks->LAPACKE_cheequb.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheequb.hook_function[hook_pos_LAPACKE_cheequb];
    } else {
        hook_pos_LAPACKE_cheequb = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheequb;
    }
    return fn(matrix_layout, uplo, n, a, lda, s, scond, amax);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheequb_work = 0;
#endif

/* Wrapper for LAPACKE_cheequb_work */
lapack_int  LAPACKE_cheequb_work (int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *s, float *scond, float *amax, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, void *s, void *scond, void *amax, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, void *s, void *scond, void *amax, void *work);
    hook_pos_LAPACKE_cheequb_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheequb_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheequb_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, s, scond, amax, work);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, s, scond, amax, work);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, s, scond, amax, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheequb_work(int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *s, float *scond, float *amax, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, void *s, void *scond, void *amax, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheequb_work;
    return fn(matrix_layout, uplo, n, a, lda, s, scond, amax, work);
}

lapack_int flexiblas_chain_LAPACKE_cheequb_work(int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, void *s, void *scond, void *amax, void *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *a, lapack_int lda, void *s, void *scond, void *amax, void *work);
    hook_pos_LAPACKE_cheequb_work++;
    if ( hook_pos_LAPACKE_cheequb_work < __flexiblas_hooks->LAPACKE_cheequb_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheequb_work.hook_function[hook_pos_LAPACKE_cheequb_work];
    } else {
        hook_pos_LAPACKE_cheequb_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheequb_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, s, scond, amax, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheev = 0;
#endif

/* Wrapper for LAPACKE_cheev */
lapack_int  LAPACKE_cheev (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    hook_pos_LAPACKE_cheev = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheev;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheev.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, a, lda, w);
    } else {
        return fn(matrix_layout, jobz, uplo, n, a, lda, w);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, a, lda, w);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheev(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheev;
    return fn(matrix_layout, jobz, uplo, n, a, lda, w);
}

lapack_int flexiblas_chain_LAPACKE_cheev(int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    hook_pos_LAPACKE_cheev++;
    if ( hook_pos_LAPACKE_cheev < __flexiblas_hooks->LAPACKE_cheev.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheev.hook_function[hook_pos_LAPACKE_cheev];
    } else {
        hook_pos_LAPACKE_cheev = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheev;
    }
    return fn(matrix_layout, jobz, uplo, n, a, lda, w);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheev_2stage = 0;
#endif

/* Wrapper for LAPACKE_cheev_2stage */
lapack_int  LAPACKE_cheev_2stage (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    hook_pos_LAPACKE_cheev_2stage = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheev_2stage;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheev_2stage.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, a, lda, w);
    } else {
        return fn(matrix_layout, jobz, uplo, n, a, lda, w);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, a, lda, w);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheev_2stage(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheev_2stage;
    return fn(matrix_layout, jobz, uplo, n, a, lda, w);
}

lapack_int flexiblas_chain_LAPACKE_cheev_2stage(int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    hook_pos_LAPACKE_cheev_2stage++;
    if ( hook_pos_LAPACKE_cheev_2stage < __flexiblas_hooks->LAPACKE_cheev_2stage.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheev_2stage.hook_function[hook_pos_LAPACKE_cheev_2stage];
    } else {
        hook_pos_LAPACKE_cheev_2stage = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheev_2stage;
    }
    return fn(matrix_layout, jobz, uplo, n, a, lda, w);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheev_2stage_work = 0;
#endif

/* Wrapper for LAPACKE_cheev_2stage_work */
lapack_int  LAPACKE_cheev_2stage_work (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cheev_2stage_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheev_2stage_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheev_2stage_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork);
    } else {
        return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheev_2stage_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheev_2stage_work;
    return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cheev_2stage_work(int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cheev_2stage_work++;
    if ( hook_pos_LAPACKE_cheev_2stage_work < __flexiblas_hooks->LAPACKE_cheev_2stage_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheev_2stage_work.hook_function[hook_pos_LAPACKE_cheev_2stage_work];
    } else {
        hook_pos_LAPACKE_cheev_2stage_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheev_2stage_work;
    }
    return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheev_work = 0;
#endif

/* Wrapper for LAPACKE_cheev_work */
lapack_int  LAPACKE_cheev_work (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cheev_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheev_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheev_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork);
    } else {
        return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheev_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheev_work;
    return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cheev_work(int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_cheev_work++;
    if ( hook_pos_LAPACKE_cheev_work < __flexiblas_hooks->LAPACKE_cheev_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheev_work.hook_function[hook_pos_LAPACKE_cheev_work];
    } else {
        hook_pos_LAPACKE_cheev_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheev_work;
    }
    return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheevd = 0;
#endif

/* Wrapper for LAPACKE_cheevd */
lapack_int  LAPACKE_cheevd (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    hook_pos_LAPACKE_cheevd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheevd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, a, lda, w);
    } else {
        return fn(matrix_layout, jobz, uplo, n, a, lda, w);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, a, lda, w);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheevd(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevd;
    return fn(matrix_layout, jobz, uplo, n, a, lda, w);
}

lapack_int flexiblas_chain_LAPACKE_cheevd(int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    hook_pos_LAPACKE_cheevd++;
    if ( hook_pos_LAPACKE_cheevd < __flexiblas_hooks->LAPACKE_cheevd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheevd.hook_function[hook_pos_LAPACKE_cheevd];
    } else {
        hook_pos_LAPACKE_cheevd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheevd;
    }
    return fn(matrix_layout, jobz, uplo, n, a, lda, w);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheevd_2stage = 0;
#endif

/* Wrapper for LAPACKE_cheevd_2stage */
lapack_int  LAPACKE_cheevd_2stage (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    hook_pos_LAPACKE_cheevd_2stage = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevd_2stage;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheevd_2stage.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, a, lda, w);
    } else {
        return fn(matrix_layout, jobz, uplo, n, a, lda, w);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, a, lda, w);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheevd_2stage(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevd_2stage;
    return fn(matrix_layout, jobz, uplo, n, a, lda, w);
}

lapack_int flexiblas_chain_LAPACKE_cheevd_2stage(int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w);
    hook_pos_LAPACKE_cheevd_2stage++;
    if ( hook_pos_LAPACKE_cheevd_2stage < __flexiblas_hooks->LAPACKE_cheevd_2stage.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheevd_2stage.hook_function[hook_pos_LAPACKE_cheevd_2stage];
    } else {
        hook_pos_LAPACKE_cheevd_2stage = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheevd_2stage;
    }
    return fn(matrix_layout, jobz, uplo, n, a, lda, w);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheevd_2stage_work = 0;
#endif

/* Wrapper for LAPACKE_cheevd_2stage_work */
lapack_int  LAPACKE_cheevd_2stage_work (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_cheevd_2stage_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevd_2stage_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheevd_2stage_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork, lrwork, iwork, liwork);
    } else {
        return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork, lrwork, iwork, liwork);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork, lrwork, iwork, liwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheevd_2stage_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevd_2stage_work;
    return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork, lrwork, iwork, liwork);
}

lapack_int flexiblas_chain_LAPACKE_cheevd_2stage_work(int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_cheevd_2stage_work++;
    if ( hook_pos_LAPACKE_cheevd_2stage_work < __flexiblas_hooks->LAPACKE_cheevd_2stage_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheevd_2stage_work.hook_function[hook_pos_LAPACKE_cheevd_2stage_work];
    } else {
        hook_pos_LAPACKE_cheevd_2stage_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheevd_2stage_work;
    }
    return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork, lrwork, iwork, liwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheevd_work = 0;
#endif

/* Wrapper for LAPACKE_cheevd_work */
lapack_int  LAPACKE_cheevd_work (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_cheevd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheevd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork, lrwork, iwork, liwork);
    } else {
        return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork, lrwork, iwork, liwork);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork, lrwork, iwork, liwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheevd_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevd_work;
    return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork, lrwork, iwork, liwork);
}

lapack_int flexiblas_chain_LAPACKE_cheevd_work(int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_cheevd_work++;
    if ( hook_pos_LAPACKE_cheevd_work < __flexiblas_hooks->LAPACKE_cheevd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheevd_work.hook_function[hook_pos_LAPACKE_cheevd_work];
    } else {
        hook_pos_LAPACKE_cheevd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheevd_work;
    }
    return fn(matrix_layout, jobz, uplo, n, a, lda, w, work, lwork, rwork, lrwork, iwork, liwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheevr = 0;
#endif

/* Wrapper for LAPACKE_cheevr */
lapack_int  LAPACKE_cheevr (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *isuppz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz);
    hook_pos_LAPACKE_cheevr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheevr.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheevr(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *isuppz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevr;
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

lapack_int flexiblas_chain_LAPACKE_cheevr(int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz);
    hook_pos_LAPACKE_cheevr++;
    if ( hook_pos_LAPACKE_cheevr < __flexiblas_hooks->LAPACKE_cheevr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheevr.hook_function[hook_pos_LAPACKE_cheevr];
    } else {
        hook_pos_LAPACKE_cheevr = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheevr;
    }
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheevr_2stage = 0;
#endif

/* Wrapper for LAPACKE_cheevr_2stage */
lapack_int  LAPACKE_cheevr_2stage (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *isuppz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz);
    hook_pos_LAPACKE_cheevr_2stage = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevr_2stage;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheevr_2stage.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheevr_2stage(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *isuppz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevr_2stage;
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

lapack_int flexiblas_chain_LAPACKE_cheevr_2stage(int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz);
    hook_pos_LAPACKE_cheevr_2stage++;
    if ( hook_pos_LAPACKE_cheevr_2stage < __flexiblas_hooks->LAPACKE_cheevr_2stage.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheevr_2stage.hook_function[hook_pos_LAPACKE_cheevr_2stage];
    } else {
        hook_pos_LAPACKE_cheevr_2stage = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheevr_2stage;
    }
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheevr_2stage_work = 0;
#endif

/* Wrapper for LAPACKE_cheevr_2stage_work */
lapack_int  LAPACKE_cheevr_2stage_work (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *isuppz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_cheevr_2stage_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevr_2stage_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheevr_2stage_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, rwork, lrwork, iwork, liwork);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, rwork, lrwork, iwork, liwork);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, rwork, lrwork, iwork, liwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheevr_2stage_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *isuppz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevr_2stage_work;
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, rwork, lrwork, iwork, liwork);
}

lapack_int flexiblas_chain_LAPACKE_cheevr_2stage_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_cheevr_2stage_work++;
    if ( hook_pos_LAPACKE_cheevr_2stage_work < __flexiblas_hooks->LAPACKE_cheevr_2stage_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheevr_2stage_work.hook_function[hook_pos_LAPACKE_cheevr_2stage_work];
    } else {
        hook_pos_LAPACKE_cheevr_2stage_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheevr_2stage_work;
    }
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, rwork, lrwork, iwork, liwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheevr_work = 0;
#endif

/* Wrapper for LAPACKE_cheevr_work */
lapack_int  LAPACKE_cheevr_work (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *isuppz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_cheevr_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevr_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheevr_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, rwork, lrwork, iwork, liwork);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, rwork, lrwork, iwork, liwork);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, rwork, lrwork, iwork, liwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheevr_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *isuppz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevr_work;
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, rwork, lrwork, iwork, liwork);
}

lapack_int flexiblas_chain_LAPACKE_cheevr_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *isuppz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_cheevr_work++;
    if ( hook_pos_LAPACKE_cheevr_work < __flexiblas_hooks->LAPACKE_cheevr_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheevr_work.hook_function[hook_pos_LAPACKE_cheevr_work];
    } else {
        hook_pos_LAPACKE_cheevr_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheevr_work;
    }
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, rwork, lrwork, iwork, liwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheevx = 0;
#endif

/* Wrapper for LAPACKE_cheevx */
lapack_int  LAPACKE_cheevx (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_cheevx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheevx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheevx(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevx;
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

lapack_int flexiblas_chain_LAPACKE_cheevx(int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_cheevx++;
    if ( hook_pos_LAPACKE_cheevx < __flexiblas_hooks->LAPACKE_cheevx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheevx.hook_function[hook_pos_LAPACKE_cheevx];
    } else {
        hook_pos_LAPACKE_cheevx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheevx;
    }
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheevx_2stage = 0;
#endif

/* Wrapper for LAPACKE_cheevx_2stage */
lapack_int  LAPACKE_cheevx_2stage (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_cheevx_2stage = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevx_2stage;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheevx_2stage.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheevx_2stage(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevx_2stage;
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

lapack_int flexiblas_chain_LAPACKE_cheevx_2stage(int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_cheevx_2stage++;
    if ( hook_pos_LAPACKE_cheevx_2stage < __flexiblas_hooks->LAPACKE_cheevx_2stage.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheevx_2stage.hook_function[hook_pos_LAPACKE_cheevx_2stage];
    } else {
        hook_pos_LAPACKE_cheevx_2stage = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheevx_2stage;
    }
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheevx_2stage_work = 0;
#endif

/* Wrapper for LAPACKE_cheevx_2stage_work */
lapack_int  LAPACKE_cheevx_2stage_work (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_cheevx_2stage_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevx_2stage_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheevx_2stage_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheevx_2stage_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevx_2stage_work;
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
}

lapack_int flexiblas_chain_LAPACKE_cheevx_2stage_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_cheevx_2stage_work++;
    if ( hook_pos_LAPACKE_cheevx_2stage_work < __flexiblas_hooks->LAPACKE_cheevx_2stage_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheevx_2stage_work.hook_function[hook_pos_LAPACKE_cheevx_2stage_work];
    } else {
        hook_pos_LAPACKE_cheevx_2stage_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheevx_2stage_work;
    }
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheevx_work = 0;
#endif

/* Wrapper for LAPACKE_cheevx_work */
lapack_int  LAPACKE_cheevx_work (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_cheevx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheevx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheevx_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheevx_work;
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
}

lapack_int flexiblas_chain_LAPACKE_cheevx_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_cheevx_work++;
    if ( hook_pos_LAPACKE_cheevx_work < __flexiblas_hooks->LAPACKE_cheevx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheevx_work.hook_function[hook_pos_LAPACKE_cheevx_work];
    } else {
        hook_pos_LAPACKE_cheevx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheevx_work;
    }
    return fn(matrix_layout, jobz, range, uplo, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chegst = 0;
#endif

/* Wrapper for LAPACKE_chegst */
lapack_int  LAPACKE_chegst (int matrix_layout, lapack_int itype, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *a, lapack_int lda, const void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *a, lapack_int lda, const void *b, lapack_int ldb);
    hook_pos_LAPACKE_chegst = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chegst;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chegst.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, uplo, n, a, lda, b, ldb);
    } else {
        return fn(matrix_layout, itype, uplo, n, a, lda, b, ldb);
    }
    #else
    return fn(matrix_layout, itype, uplo, n, a, lda, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chegst(int matrix_layout, lapack_int itype, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *a, lapack_int lda, const void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chegst;
    return fn(matrix_layout, itype, uplo, n, a, lda, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chegst(int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *a, lapack_int lda, const void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *a, lapack_int lda, const void *b, lapack_int ldb);
    hook_pos_LAPACKE_chegst++;
    if ( hook_pos_LAPACKE_chegst < __flexiblas_hooks->LAPACKE_chegst.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chegst.hook_function[hook_pos_LAPACKE_chegst];
    } else {
        hook_pos_LAPACKE_chegst = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chegst;
    }
    return fn(matrix_layout, itype, uplo, n, a, lda, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chegst_work = 0;
#endif

/* Wrapper for LAPACKE_chegst_work */
lapack_int  LAPACKE_chegst_work (int matrix_layout, lapack_int itype, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *a, lapack_int lda, const void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *a, lapack_int lda, const void *b, lapack_int ldb);
    hook_pos_LAPACKE_chegst_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chegst_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chegst_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, uplo, n, a, lda, b, ldb);
    } else {
        return fn(matrix_layout, itype, uplo, n, a, lda, b, ldb);
    }
    #else
    return fn(matrix_layout, itype, uplo, n, a, lda, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chegst_work(int matrix_layout, lapack_int itype, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *a, lapack_int lda, const void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chegst_work;
    return fn(matrix_layout, itype, uplo, n, a, lda, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chegst_work(int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *a, lapack_int lda, const void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *a, lapack_int lda, const void *b, lapack_int ldb);
    hook_pos_LAPACKE_chegst_work++;
    if ( hook_pos_LAPACKE_chegst_work < __flexiblas_hooks->LAPACKE_chegst_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chegst_work.hook_function[hook_pos_LAPACKE_chegst_work];
    } else {
        hook_pos_LAPACKE_chegst_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chegst_work;
    }
    return fn(matrix_layout, itype, uplo, n, a, lda, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chegv = 0;
#endif

/* Wrapper for LAPACKE_chegv */
lapack_int  LAPACKE_chegv (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *w)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w);
    hook_pos_LAPACKE_chegv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chegv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chegv.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
    } else {
        return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
    }
    #else
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chegv(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *w)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chegv;
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
}

lapack_int flexiblas_chain_LAPACKE_chegv(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w);
    hook_pos_LAPACKE_chegv++;
    if ( hook_pos_LAPACKE_chegv < __flexiblas_hooks->LAPACKE_chegv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chegv.hook_function[hook_pos_LAPACKE_chegv];
    } else {
        hook_pos_LAPACKE_chegv = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chegv;
    }
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chegv_2stage = 0;
#endif

/* Wrapper for LAPACKE_chegv_2stage */
lapack_int  LAPACKE_chegv_2stage (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *w)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w);
    hook_pos_LAPACKE_chegv_2stage = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chegv_2stage;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chegv_2stage.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
    } else {
        return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
    }
    #else
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chegv_2stage(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *w)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chegv_2stage;
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
}

lapack_int flexiblas_chain_LAPACKE_chegv_2stage(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w);
    hook_pos_LAPACKE_chegv_2stage++;
    if ( hook_pos_LAPACKE_chegv_2stage < __flexiblas_hooks->LAPACKE_chegv_2stage.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chegv_2stage.hook_function[hook_pos_LAPACKE_chegv_2stage];
    } else {
        hook_pos_LAPACKE_chegv_2stage = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chegv_2stage;
    }
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chegv_2stage_work = 0;
#endif

/* Wrapper for LAPACKE_chegv_2stage_work */
lapack_int  LAPACKE_chegv_2stage_work (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_chegv_2stage_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chegv_2stage_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chegv_2stage_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork);
    } else {
        return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chegv_2stage_work(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chegv_2stage_work;
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_chegv_2stage_work(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_chegv_2stage_work++;
    if ( hook_pos_LAPACKE_chegv_2stage_work < __flexiblas_hooks->LAPACKE_chegv_2stage_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chegv_2stage_work.hook_function[hook_pos_LAPACKE_chegv_2stage_work];
    } else {
        hook_pos_LAPACKE_chegv_2stage_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chegv_2stage_work;
    }
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chegv_work = 0;
#endif

/* Wrapper for LAPACKE_chegv_work */
lapack_int  LAPACKE_chegv_work (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_chegv_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chegv_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chegv_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork);
    } else {
        return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chegv_work(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chegv_work;
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_chegv_work(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_chegv_work++;
    if ( hook_pos_LAPACKE_chegv_work < __flexiblas_hooks->LAPACKE_chegv_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chegv_work.hook_function[hook_pos_LAPACKE_chegv_work];
    } else {
        hook_pos_LAPACKE_chegv_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chegv_work;
    }
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chegvd = 0;
#endif

/* Wrapper for LAPACKE_chegvd */
lapack_int  LAPACKE_chegvd (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *w)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w);
    hook_pos_LAPACKE_chegvd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chegvd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chegvd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
    } else {
        return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
    }
    #else
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chegvd(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *w)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chegvd;
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
}

lapack_int flexiblas_chain_LAPACKE_chegvd(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w);
    hook_pos_LAPACKE_chegvd++;
    if ( hook_pos_LAPACKE_chegvd < __flexiblas_hooks->LAPACKE_chegvd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chegvd.hook_function[hook_pos_LAPACKE_chegvd];
    } else {
        hook_pos_LAPACKE_chegvd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chegvd;
    }
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chegvd_work = 0;
#endif

/* Wrapper for LAPACKE_chegvd_work */
lapack_int  LAPACKE_chegvd_work (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_chegvd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chegvd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chegvd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork, lrwork, iwork, liwork);
    } else {
        return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork, lrwork, iwork, liwork);
    }
    #else
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork, lrwork, iwork, liwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chegvd_work(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float *w, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chegvd_work;
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork, lrwork, iwork, liwork);
}

lapack_int flexiblas_chain_LAPACKE_chegvd_work(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, void *w, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_chegvd_work++;
    if ( hook_pos_LAPACKE_chegvd_work < __flexiblas_hooks->LAPACKE_chegvd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chegvd_work.hook_function[hook_pos_LAPACKE_chegvd_work];
    } else {
        hook_pos_LAPACKE_chegvd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chegvd_work;
    }
    return fn(matrix_layout, itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork, lrwork, iwork, liwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chegvx = 0;
#endif

/* Wrapper for LAPACKE_chegvx */
lapack_int  LAPACKE_chegvx (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_chegvx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chegvx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chegvx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, range, uplo, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    } else {
        return fn(matrix_layout, itype, jobz, range, uplo, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    }
    #else
    return fn(matrix_layout, itype, jobz, range, uplo, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chegvx(int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chegvx;
    return fn(matrix_layout, itype, jobz, range, uplo, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

lapack_int flexiblas_chain_LAPACKE_chegvx(int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_chegvx++;
    if ( hook_pos_LAPACKE_chegvx < __flexiblas_hooks->LAPACKE_chegvx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chegvx.hook_function[hook_pos_LAPACKE_chegvx];
    } else {
        hook_pos_LAPACKE_chegvx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chegvx;
    }
    return fn(matrix_layout, itype, jobz, range, uplo, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chegvx_work = 0;
#endif

/* Wrapper for LAPACKE_chegvx_work */
lapack_int  LAPACKE_chegvx_work (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_chegvx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chegvx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chegvx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, range, uplo, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
    } else {
        return fn(matrix_layout, itype, jobz, range, uplo, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
    }
    #else
    return fn(matrix_layout, itype, jobz, range, uplo, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chegvx_work(int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chegvx_work;
    return fn(matrix_layout, itype, jobz, range, uplo, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
}

lapack_int flexiblas_chain_LAPACKE_chegvx_work(int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *a, lapack_int lda, void *b, lapack_int ldb, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_chegvx_work++;
    if ( hook_pos_LAPACKE_chegvx_work < __flexiblas_hooks->LAPACKE_chegvx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chegvx_work.hook_function[hook_pos_LAPACKE_chegvx_work];
    } else {
        hook_pos_LAPACKE_chegvx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chegvx_work;
    }
    return fn(matrix_layout, itype, jobz, range, uplo, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cherfs = 0;
#endif

/* Wrapper for LAPACKE_cherfs */
lapack_int  LAPACKE_cherfs (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *af, lapack_int ldaf, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    hook_pos_LAPACKE_cherfs = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cherfs;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cherfs.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cherfs(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *af, lapack_int ldaf, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cherfs;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

lapack_int flexiblas_chain_LAPACKE_cherfs(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    hook_pos_LAPACKE_cherfs++;
    if ( hook_pos_LAPACKE_cherfs < __flexiblas_hooks->LAPACKE_cherfs.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cherfs.hook_function[hook_pos_LAPACKE_cherfs];
    } else {
        hook_pos_LAPACKE_cherfs = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cherfs;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cherfs_work = 0;
#endif

/* Wrapper for LAPACKE_cherfs_work */
lapack_int  LAPACKE_cherfs_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *af, lapack_int ldaf, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cherfs_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cherfs_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cherfs_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cherfs_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *af, lapack_int ldaf, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cherfs_work;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_cherfs_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *af, lapack_int ldaf, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_cherfs_work++;
    if ( hook_pos_LAPACKE_cherfs_work < __flexiblas_hooks->LAPACKE_cherfs_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cherfs_work.hook_function[hook_pos_LAPACKE_cherfs_work];
    } else {
        hook_pos_LAPACKE_cherfs_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cherfs_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chesv = 0;
#endif

/* Wrapper for LAPACKE_chesv */
lapack_int  LAPACKE_chesv (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chesv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chesv.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chesv(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chesv(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chesv++;
    if ( hook_pos_LAPACKE_chesv < __flexiblas_hooks->LAPACKE_chesv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chesv.hook_function[hook_pos_LAPACKE_chesv];
    } else {
        hook_pos_LAPACKE_chesv = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chesv;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chesv_aa = 0;
#endif

/* Wrapper for LAPACKE_chesv_aa */
lapack_int  LAPACKE_chesv_aa (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chesv_aa = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_aa;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chesv_aa.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chesv_aa(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_aa;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chesv_aa(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chesv_aa++;
    if ( hook_pos_LAPACKE_chesv_aa < __flexiblas_hooks->LAPACKE_chesv_aa.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chesv_aa.hook_function[hook_pos_LAPACKE_chesv_aa];
    } else {
        hook_pos_LAPACKE_chesv_aa = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_aa;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chesv_aa_2stage = 0;
#endif

/* Wrapper for LAPACKE_chesv_aa_2stage */
lapack_int  LAPACKE_chesv_aa_2stage (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tb, lapack_int ltb, lapack_int *ipiv, lapack_int *ipiv2, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chesv_aa_2stage = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_aa_2stage;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chesv_aa_2stage.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chesv_aa_2stage(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tb, lapack_int ltb, lapack_int *ipiv, lapack_int *ipiv2, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_aa_2stage;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chesv_aa_2stage(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chesv_aa_2stage++;
    if ( hook_pos_LAPACKE_chesv_aa_2stage < __flexiblas_hooks->LAPACKE_chesv_aa_2stage.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chesv_aa_2stage.hook_function[hook_pos_LAPACKE_chesv_aa_2stage];
    } else {
        hook_pos_LAPACKE_chesv_aa_2stage = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_aa_2stage;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chesv_aa_2stage_work = 0;
#endif

/* Wrapper for LAPACKE_chesv_aa_2stage_work */
lapack_int  LAPACKE_chesv_aa_2stage_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tb, lapack_int ltb, lapack_int *ipiv, lapack_int *ipiv2, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chesv_aa_2stage_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_aa_2stage_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chesv_aa_2stage_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chesv_aa_2stage_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tb, lapack_int ltb, lapack_int *ipiv, lapack_int *ipiv2, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_aa_2stage_work;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chesv_aa_2stage_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chesv_aa_2stage_work++;
    if ( hook_pos_LAPACKE_chesv_aa_2stage_work < __flexiblas_hooks->LAPACKE_chesv_aa_2stage_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chesv_aa_2stage_work.hook_function[hook_pos_LAPACKE_chesv_aa_2stage_work];
    } else {
        hook_pos_LAPACKE_chesv_aa_2stage_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_aa_2stage_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chesv_aa_work = 0;
#endif

/* Wrapper for LAPACKE_chesv_aa_work */
lapack_int  LAPACKE_chesv_aa_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chesv_aa_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_aa_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chesv_aa_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chesv_aa_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_aa_work;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chesv_aa_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chesv_aa_work++;
    if ( hook_pos_LAPACKE_chesv_aa_work < __flexiblas_hooks->LAPACKE_chesv_aa_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chesv_aa_work.hook_function[hook_pos_LAPACKE_chesv_aa_work];
    } else {
        hook_pos_LAPACKE_chesv_aa_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_aa_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chesv_rk = 0;
#endif

/* Wrapper for LAPACKE_chesv_rk */
lapack_int  LAPACKE_chesv_rk (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *e, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *e, void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *e, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chesv_rk = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_rk;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chesv_rk.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chesv_rk(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *e, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *e, void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_rk;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chesv_rk(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *e, void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *e, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chesv_rk++;
    if ( hook_pos_LAPACKE_chesv_rk < __flexiblas_hooks->LAPACKE_chesv_rk.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chesv_rk.hook_function[hook_pos_LAPACKE_chesv_rk];
    } else {
        hook_pos_LAPACKE_chesv_rk = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_rk;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chesv_rk_work = 0;
#endif

/* Wrapper for LAPACKE_chesv_rk_work */
lapack_int  LAPACKE_chesv_rk_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *e, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *e, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *e, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chesv_rk_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_rk_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chesv_rk_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chesv_rk_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *e, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *e, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_rk_work;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chesv_rk_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *e, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *e, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chesv_rk_work++;
    if ( hook_pos_LAPACKE_chesv_rk_work < __flexiblas_hooks->LAPACKE_chesv_rk_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chesv_rk_work.hook_function[hook_pos_LAPACKE_chesv_rk_work];
    } else {
        hook_pos_LAPACKE_chesv_rk_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_rk_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chesv_work = 0;
#endif

/* Wrapper for LAPACKE_chesv_work */
lapack_int  LAPACKE_chesv_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chesv_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chesv_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chesv_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_work;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chesv_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chesv_work++;
    if ( hook_pos_LAPACKE_chesv_work < __flexiblas_hooks->LAPACKE_chesv_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chesv_work.hook_function[hook_pos_LAPACKE_chesv_work];
    } else {
        hook_pos_LAPACKE_chesv_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chesv_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chesvx = 0;
#endif

/* Wrapper for LAPACKE_chesvx */
lapack_int  LAPACKE_chesvx (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, lapack_complex_float *af, lapack_int ldaf, lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr);
    hook_pos_LAPACKE_chesvx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chesvx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chesvx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, fact, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
    } else {
        return fn(matrix_layout, fact, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
    }
    #else
    return fn(matrix_layout, fact, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chesvx(int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, lapack_complex_float *af, lapack_int ldaf, lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chesvx;
    return fn(matrix_layout, fact, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

lapack_int flexiblas_chain_LAPACKE_chesvx(int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr)
{
    lapack_int (*fn) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr);
    hook_pos_LAPACKE_chesvx++;
    if ( hook_pos_LAPACKE_chesvx < __flexiblas_hooks->LAPACKE_chesvx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chesvx.hook_function[hook_pos_LAPACKE_chesvx];
    } else {
        hook_pos_LAPACKE_chesvx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chesvx;
    }
    return fn(matrix_layout, fact, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chesvx_work = 0;
#endif

/* Wrapper for LAPACKE_chesvx_work */
lapack_int  LAPACKE_chesvx_work (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, lapack_complex_float *af, lapack_int ldaf, lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_chesvx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chesvx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chesvx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, fact, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, lwork, rwork);
    } else {
        return fn(matrix_layout, fact, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, fact, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chesvx_work(int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, lapack_complex_float *af, lapack_int ldaf, lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chesvx_work;
    return fn(matrix_layout, fact, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_chesvx_work(int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, void *af, lapack_int ldaf, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_chesvx_work++;
    if ( hook_pos_LAPACKE_chesvx_work < __flexiblas_hooks->LAPACKE_chesvx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chesvx_work.hook_function[hook_pos_LAPACKE_chesvx_work];
    } else {
        hook_pos_LAPACKE_chesvx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chesvx_work;
    }
    return fn(matrix_layout, fact, uplo, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheswapr = 0;
#endif

/* Wrapper for LAPACKE_cheswapr */
lapack_int  LAPACKE_cheswapr (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int i1, lapack_int i2)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, lapack_int i1, lapack_int i2);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, lapack_int i1, lapack_int i2);
    hook_pos_LAPACKE_cheswapr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheswapr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheswapr.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, i1, i2);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, i1, i2);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, i1, i2);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheswapr(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int i1, lapack_int i2)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, lapack_int i1, lapack_int i2);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheswapr;
    return fn(matrix_layout, uplo, n, a, lda, i1, i2);
}

lapack_int flexiblas_chain_LAPACKE_cheswapr(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, lapack_int i1, lapack_int i2)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, lapack_int i1, lapack_int i2);
    hook_pos_LAPACKE_cheswapr++;
    if ( hook_pos_LAPACKE_cheswapr < __flexiblas_hooks->LAPACKE_cheswapr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheswapr.hook_function[hook_pos_LAPACKE_cheswapr];
    } else {
        hook_pos_LAPACKE_cheswapr = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheswapr;
    }
    return fn(matrix_layout, uplo, n, a, lda, i1, i2);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_cheswapr_work = 0;
#endif

/* Wrapper for LAPACKE_cheswapr_work */
lapack_int  LAPACKE_cheswapr_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int i1, lapack_int i2)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, lapack_int i1, lapack_int i2);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, lapack_int i1, lapack_int i2);
    hook_pos_LAPACKE_cheswapr_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_cheswapr_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_cheswapr_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, i1, i2);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, i1, i2);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, i1, i2);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_cheswapr_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int i1, lapack_int i2)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, lapack_int i1, lapack_int i2);
    *(void **) &fn = current_backend->lapacke.LAPACKE_cheswapr_work;
    return fn(matrix_layout, uplo, n, a, lda, i1, i2);
}

lapack_int flexiblas_chain_LAPACKE_cheswapr_work(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, lapack_int i1, lapack_int i2)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, lapack_int i1, lapack_int i2);
    hook_pos_LAPACKE_cheswapr_work++;
    if ( hook_pos_LAPACKE_cheswapr_work < __flexiblas_hooks->LAPACKE_cheswapr_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_cheswapr_work.hook_function[hook_pos_LAPACKE_cheswapr_work];
    } else {
        hook_pos_LAPACKE_cheswapr_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_cheswapr_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, i1, i2);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrd = 0;
#endif

/* Wrapper for LAPACKE_chetrd */
lapack_int  LAPACKE_chetrd (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *d, float *e, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tau);
    hook_pos_LAPACKE_chetrd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, d, e, tau);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, d, e, tau);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, d, e, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrd(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *d, float *e, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrd;
    return fn(matrix_layout, uplo, n, a, lda, d, e, tau);
}

lapack_int flexiblas_chain_LAPACKE_chetrd(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tau)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tau);
    hook_pos_LAPACKE_chetrd++;
    if ( hook_pos_LAPACKE_chetrd < __flexiblas_hooks->LAPACKE_chetrd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrd.hook_function[hook_pos_LAPACKE_chetrd];
    } else {
        hook_pos_LAPACKE_chetrd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrd;
    }
    return fn(matrix_layout, uplo, n, a, lda, d, e, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrd_work = 0;
#endif

/* Wrapper for LAPACKE_chetrd_work */
lapack_int  LAPACKE_chetrd_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *d, float *e, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tau, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, d, e, tau, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, d, e, tau, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, d, e, tau, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrd_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, float *d, float *e, lapack_complex_float *tau, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tau, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrd_work;
    return fn(matrix_layout, uplo, n, a, lda, d, e, tau, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chetrd_work(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tau, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *d, void *e, void *tau, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrd_work++;
    if ( hook_pos_LAPACKE_chetrd_work < __flexiblas_hooks->LAPACKE_chetrd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrd_work.hook_function[hook_pos_LAPACKE_chetrd_work];
    } else {
        hook_pos_LAPACKE_chetrd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrd_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, d, e, tau, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrf = 0;
#endif

/* Wrapper for LAPACKE_chetrf */
lapack_int  LAPACKE_chetrf (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_chetrf = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrf.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrf(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf;
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_chetrf(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_chetrf++;
    if ( hook_pos_LAPACKE_chetrf < __flexiblas_hooks->LAPACKE_chetrf.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrf.hook_function[hook_pos_LAPACKE_chetrf];
    } else {
        hook_pos_LAPACKE_chetrf = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrf_aa = 0;
#endif

/* Wrapper for LAPACKE_chetrf_aa */
lapack_int  LAPACKE_chetrf_aa (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_chetrf_aa = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_aa;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrf_aa.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrf_aa(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_aa;
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_chetrf_aa(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_chetrf_aa++;
    if ( hook_pos_LAPACKE_chetrf_aa < __flexiblas_hooks->LAPACKE_chetrf_aa.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrf_aa.hook_function[hook_pos_LAPACKE_chetrf_aa];
    } else {
        hook_pos_LAPACKE_chetrf_aa = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_aa;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrf_aa_2stage = 0;
#endif

/* Wrapper for LAPACKE_chetrf_aa_2stage */
lapack_int  LAPACKE_chetrf_aa_2stage (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tb, lapack_int ltb, lapack_int *ipiv, lapack_int *ipiv2)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2);
    hook_pos_LAPACKE_chetrf_aa_2stage = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_aa_2stage;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrf_aa_2stage.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, tb, ltb, ipiv, ipiv2);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, tb, ltb, ipiv, ipiv2);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, tb, ltb, ipiv, ipiv2);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrf_aa_2stage(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tb, lapack_int ltb, lapack_int *ipiv, lapack_int *ipiv2)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_aa_2stage;
    return fn(matrix_layout, uplo, n, a, lda, tb, ltb, ipiv, ipiv2);
}

lapack_int flexiblas_chain_LAPACKE_chetrf_aa_2stage(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2);
    hook_pos_LAPACKE_chetrf_aa_2stage++;
    if ( hook_pos_LAPACKE_chetrf_aa_2stage < __flexiblas_hooks->LAPACKE_chetrf_aa_2stage.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrf_aa_2stage.hook_function[hook_pos_LAPACKE_chetrf_aa_2stage];
    } else {
        hook_pos_LAPACKE_chetrf_aa_2stage = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_aa_2stage;
    }
    return fn(matrix_layout, uplo, n, a, lda, tb, ltb, ipiv, ipiv2);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrf_aa_2stage_work = 0;
#endif

/* Wrapper for LAPACKE_chetrf_aa_2stage_work */
lapack_int  LAPACKE_chetrf_aa_2stage_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tb, lapack_int ltb, lapack_int *ipiv, lapack_int *ipiv2, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrf_aa_2stage_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_aa_2stage_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrf_aa_2stage_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, tb, ltb, ipiv, ipiv2, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, tb, ltb, ipiv, ipiv2, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, tb, ltb, ipiv, ipiv2, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrf_aa_2stage_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tb, lapack_int ltb, lapack_int *ipiv, lapack_int *ipiv2, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_aa_2stage_work;
    return fn(matrix_layout, uplo, n, a, lda, tb, ltb, ipiv, ipiv2, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chetrf_aa_2stage_work(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrf_aa_2stage_work++;
    if ( hook_pos_LAPACKE_chetrf_aa_2stage_work < __flexiblas_hooks->LAPACKE_chetrf_aa_2stage_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrf_aa_2stage_work.hook_function[hook_pos_LAPACKE_chetrf_aa_2stage_work];
    } else {
        hook_pos_LAPACKE_chetrf_aa_2stage_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_aa_2stage_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, tb, ltb, ipiv, ipiv2, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrf_aa_work = 0;
#endif

/* Wrapper for LAPACKE_chetrf_aa_work */
lapack_int  LAPACKE_chetrf_aa_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrf_aa_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_aa_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrf_aa_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrf_aa_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_aa_work;
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chetrf_aa_work(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrf_aa_work++;
    if ( hook_pos_LAPACKE_chetrf_aa_work < __flexiblas_hooks->LAPACKE_chetrf_aa_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrf_aa_work.hook_function[hook_pos_LAPACKE_chetrf_aa_work];
    } else {
        hook_pos_LAPACKE_chetrf_aa_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_aa_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrf_rk = 0;
#endif

/* Wrapper for LAPACKE_chetrf_rk */
lapack_int  LAPACKE_chetrf_rk (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *e, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *e, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *e, void *ipiv);
    hook_pos_LAPACKE_chetrf_rk = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_rk;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrf_rk.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, e, ipiv);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, e, ipiv);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrf_rk(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *e, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *e, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_rk;
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_chetrf_rk(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *e, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *e, void *ipiv);
    hook_pos_LAPACKE_chetrf_rk++;
    if ( hook_pos_LAPACKE_chetrf_rk < __flexiblas_hooks->LAPACKE_chetrf_rk.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrf_rk.hook_function[hook_pos_LAPACKE_chetrf_rk];
    } else {
        hook_pos_LAPACKE_chetrf_rk = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_rk;
    }
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrf_rk_work = 0;
#endif

/* Wrapper for LAPACKE_chetrf_rk_work */
lapack_int  LAPACKE_chetrf_rk_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *e, lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *e, void *ipiv, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *e, void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrf_rk_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_rk_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrf_rk_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, e, ipiv, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, e, ipiv, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrf_rk_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_complex_float *e, lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *e, void *ipiv, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_rk_work;
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chetrf_rk_work(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *e, void *ipiv, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *e, void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrf_rk_work++;
    if ( hook_pos_LAPACKE_chetrf_rk_work < __flexiblas_hooks->LAPACKE_chetrf_rk_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrf_rk_work.hook_function[hook_pos_LAPACKE_chetrf_rk_work];
    } else {
        hook_pos_LAPACKE_chetrf_rk_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_rk_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrf_rook = 0;
#endif

/* Wrapper for LAPACKE_chetrf_rook */
lapack_int  LAPACKE_chetrf_rook (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_chetrf_rook = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_rook;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrf_rook.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrf_rook(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_rook;
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_chetrf_rook(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv);
    hook_pos_LAPACKE_chetrf_rook++;
    if ( hook_pos_LAPACKE_chetrf_rook < __flexiblas_hooks->LAPACKE_chetrf_rook.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrf_rook.hook_function[hook_pos_LAPACKE_chetrf_rook];
    } else {
        hook_pos_LAPACKE_chetrf_rook = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_rook;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrf_rook_work = 0;
#endif

/* Wrapper for LAPACKE_chetrf_rook_work */
lapack_int  LAPACKE_chetrf_rook_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrf_rook_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_rook_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrf_rook_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrf_rook_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_rook_work;
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chetrf_rook_work(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrf_rook_work++;
    if ( hook_pos_LAPACKE_chetrf_rook_work < __flexiblas_hooks->LAPACKE_chetrf_rook_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrf_rook_work.hook_function[hook_pos_LAPACKE_chetrf_rook_work];
    } else {
        hook_pos_LAPACKE_chetrf_rook_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_rook_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrf_work = 0;
#endif

/* Wrapper for LAPACKE_chetrf_work */
lapack_int  LAPACKE_chetrf_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrf_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrf_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrf_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_work;
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chetrf_work(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrf_work++;
    if ( hook_pos_LAPACKE_chetrf_work < __flexiblas_hooks->LAPACKE_chetrf_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrf_work.hook_function[hook_pos_LAPACKE_chetrf_work];
    } else {
        hook_pos_LAPACKE_chetrf_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrf_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetri = 0;
#endif

/* Wrapper for LAPACKE_chetri */
lapack_int  LAPACKE_chetri (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv);
    hook_pos_LAPACKE_chetri = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetri.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetri(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri;
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_chetri(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv);
    hook_pos_LAPACKE_chetri++;
    if ( hook_pos_LAPACKE_chetri < __flexiblas_hooks->LAPACKE_chetri.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetri.hook_function[hook_pos_LAPACKE_chetri];
    } else {
        hook_pos_LAPACKE_chetri = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetri;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetri2 = 0;
#endif

/* Wrapper for LAPACKE_chetri2 */
lapack_int  LAPACKE_chetri2 (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv);
    hook_pos_LAPACKE_chetri2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetri2.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetri2(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri2;
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_chetri2(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv);
    hook_pos_LAPACKE_chetri2++;
    if ( hook_pos_LAPACKE_chetri2 < __flexiblas_hooks->LAPACKE_chetri2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetri2.hook_function[hook_pos_LAPACKE_chetri2];
    } else {
        hook_pos_LAPACKE_chetri2 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetri2;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetri2_work = 0;
#endif

/* Wrapper for LAPACKE_chetri2_work */
lapack_int  LAPACKE_chetri2_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetri2_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri2_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetri2_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetri2_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri2_work;
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chetri2_work(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetri2_work++;
    if ( hook_pos_LAPACKE_chetri2_work < __flexiblas_hooks->LAPACKE_chetri2_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetri2_work.hook_function[hook_pos_LAPACKE_chetri2_work];
    } else {
        hook_pos_LAPACKE_chetri2_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetri2_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetri2x = 0;
#endif

/* Wrapper for LAPACKE_chetri2x */
lapack_int  LAPACKE_chetri2x (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_int nb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, lapack_int nb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, lapack_int nb);
    hook_pos_LAPACKE_chetri2x = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri2x;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetri2x.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv, nb);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv, nb);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv, nb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetri2x(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_int nb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, lapack_int nb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri2x;
    return fn(matrix_layout, uplo, n, a, lda, ipiv, nb);
}

lapack_int flexiblas_chain_LAPACKE_chetri2x(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, lapack_int nb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, lapack_int nb);
    hook_pos_LAPACKE_chetri2x++;
    if ( hook_pos_LAPACKE_chetri2x < __flexiblas_hooks->LAPACKE_chetri2x.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetri2x.hook_function[hook_pos_LAPACKE_chetri2x];
    } else {
        hook_pos_LAPACKE_chetri2x = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetri2x;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv, nb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetri2x_work = 0;
#endif

/* Wrapper for LAPACKE_chetri2x_work */
lapack_int  LAPACKE_chetri2x_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *work, lapack_int nb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int nb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int nb);
    hook_pos_LAPACKE_chetri2x_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri2x_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetri2x_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv, work, nb);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv, work, nb);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, nb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetri2x_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *work, lapack_int nb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int nb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri2x_work;
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, nb);
}

lapack_int flexiblas_chain_LAPACKE_chetri2x_work(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int nb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work, lapack_int nb);
    hook_pos_LAPACKE_chetri2x_work++;
    if ( hook_pos_LAPACKE_chetri2x_work < __flexiblas_hooks->LAPACKE_chetri2x_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetri2x_work.hook_function[hook_pos_LAPACKE_chetri2x_work];
    } else {
        hook_pos_LAPACKE_chetri2x_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetri2x_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work, nb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetri_3 = 0;
#endif

/* Wrapper for LAPACKE_chetri_3 */
lapack_int  LAPACKE_chetri_3 (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_complex_float *e, const lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *e, const void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *e, const void *ipiv);
    hook_pos_LAPACKE_chetri_3 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri_3;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetri_3.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, e, ipiv);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, e, ipiv);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetri_3(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_complex_float *e, const lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *e, const void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri_3;
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_chetri_3(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *e, const void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *e, const void *ipiv);
    hook_pos_LAPACKE_chetri_3++;
    if ( hook_pos_LAPACKE_chetri_3 < __flexiblas_hooks->LAPACKE_chetri_3.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetri_3.hook_function[hook_pos_LAPACKE_chetri_3];
    } else {
        hook_pos_LAPACKE_chetri_3 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetri_3;
    }
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetri_3_work = 0;
#endif

/* Wrapper for LAPACKE_chetri_3_work */
lapack_int  LAPACKE_chetri_3_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_complex_float *e, const lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *e, const void *ipiv, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *e, const void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetri_3_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri_3_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetri_3_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, e, ipiv, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, e, ipiv, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetri_3_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_complex_float *e, const lapack_int *ipiv, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *e, const void *ipiv, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri_3_work;
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chetri_3_work(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *e, const void *ipiv, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *e, const void *ipiv, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetri_3_work++;
    if ( hook_pos_LAPACKE_chetri_3_work < __flexiblas_hooks->LAPACKE_chetri_3_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetri_3_work.hook_function[hook_pos_LAPACKE_chetri_3_work];
    } else {
        hook_pos_LAPACKE_chetri_3_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetri_3_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, e, ipiv, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetri_work = 0;
#endif

/* Wrapper for LAPACKE_chetri_work */
lapack_int  LAPACKE_chetri_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work);
    hook_pos_LAPACKE_chetri_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetri_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, a, lda, ipiv, work);
    } else {
        return fn(matrix_layout, uplo, n, a, lda, ipiv, work);
    }
    #else
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetri_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetri_work;
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work);
}

lapack_int flexiblas_chain_LAPACKE_chetri_work(int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *a, lapack_int lda, const void *ipiv, void *work);
    hook_pos_LAPACKE_chetri_work++;
    if ( hook_pos_LAPACKE_chetri_work < __flexiblas_hooks->LAPACKE_chetri_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetri_work.hook_function[hook_pos_LAPACKE_chetri_work];
    } else {
        hook_pos_LAPACKE_chetri_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetri_work;
    }
    return fn(matrix_layout, uplo, n, a, lda, ipiv, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrs = 0;
#endif

/* Wrapper for LAPACKE_chetrs */
lapack_int  LAPACKE_chetrs (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrs.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrs(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chetrs(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs++;
    if ( hook_pos_LAPACKE_chetrs < __flexiblas_hooks->LAPACKE_chetrs.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrs.hook_function[hook_pos_LAPACKE_chetrs];
    } else {
        hook_pos_LAPACKE_chetrs = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrs2 = 0;
#endif

/* Wrapper for LAPACKE_chetrs2 */
lapack_int  LAPACKE_chetrs2 (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrs2.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrs2(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs2;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chetrs2(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs2++;
    if ( hook_pos_LAPACKE_chetrs2 < __flexiblas_hooks->LAPACKE_chetrs2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrs2.hook_function[hook_pos_LAPACKE_chetrs2];
    } else {
        hook_pos_LAPACKE_chetrs2 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs2;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrs2_work = 0;
#endif

/* Wrapper for LAPACKE_chetrs2_work */
lapack_int  LAPACKE_chetrs2_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb, void *work);
    hook_pos_LAPACKE_chetrs2_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs2_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrs2_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrs2_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs2_work;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work);
}

lapack_int flexiblas_chain_LAPACKE_chetrs2_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb, void *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb, void *work);
    hook_pos_LAPACKE_chetrs2_work++;
    if ( hook_pos_LAPACKE_chetrs2_work < __flexiblas_hooks->LAPACKE_chetrs2_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrs2_work.hook_function[hook_pos_LAPACKE_chetrs2_work];
    } else {
        hook_pos_LAPACKE_chetrs2_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs2_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrs_3 = 0;
#endif

/* Wrapper for LAPACKE_chetrs_3 */
lapack_int  LAPACKE_chetrs_3 (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *e, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *e, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *e, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_3 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_3;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrs_3.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrs_3(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *e, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *e, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_3;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chetrs_3(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *e, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *e, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_3++;
    if ( hook_pos_LAPACKE_chetrs_3 < __flexiblas_hooks->LAPACKE_chetrs_3.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrs_3.hook_function[hook_pos_LAPACKE_chetrs_3];
    } else {
        hook_pos_LAPACKE_chetrs_3 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_3;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrs_3_work = 0;
#endif

/* Wrapper for LAPACKE_chetrs_3_work */
lapack_int  LAPACKE_chetrs_3_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *e, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *e, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *e, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_3_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_3_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrs_3_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrs_3_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_complex_float *e, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *e, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_3_work;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chetrs_3_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *e, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *e, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_3_work++;
    if ( hook_pos_LAPACKE_chetrs_3_work < __flexiblas_hooks->LAPACKE_chetrs_3_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrs_3_work.hook_function[hook_pos_LAPACKE_chetrs_3_work];
    } else {
        hook_pos_LAPACKE_chetrs_3_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_3_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, e, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrs_aa = 0;
#endif

/* Wrapper for LAPACKE_chetrs_aa */
lapack_int  LAPACKE_chetrs_aa (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_aa = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_aa;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrs_aa.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrs_aa(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_aa;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chetrs_aa(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_aa++;
    if ( hook_pos_LAPACKE_chetrs_aa < __flexiblas_hooks->LAPACKE_chetrs_aa.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrs_aa.hook_function[hook_pos_LAPACKE_chetrs_aa];
    } else {
        hook_pos_LAPACKE_chetrs_aa = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_aa;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrs_aa_2stage = 0;
#endif

/* Wrapper for LAPACKE_chetrs_aa_2stage */
lapack_int  LAPACKE_chetrs_aa_2stage (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tb, lapack_int ltb, lapack_int *ipiv, lapack_int *ipiv2, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_aa_2stage = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_aa_2stage;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrs_aa_2stage.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrs_aa_2stage(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tb, lapack_int ltb, lapack_int *ipiv, lapack_int *ipiv2, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_aa_2stage;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chetrs_aa_2stage(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_aa_2stage++;
    if ( hook_pos_LAPACKE_chetrs_aa_2stage < __flexiblas_hooks->LAPACKE_chetrs_aa_2stage.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrs_aa_2stage.hook_function[hook_pos_LAPACKE_chetrs_aa_2stage];
    } else {
        hook_pos_LAPACKE_chetrs_aa_2stage = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_aa_2stage;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrs_aa_2stage_work = 0;
#endif

/* Wrapper for LAPACKE_chetrs_aa_2stage_work */
lapack_int  LAPACKE_chetrs_aa_2stage_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tb, lapack_int ltb, lapack_int *ipiv, lapack_int *ipiv2, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_aa_2stage_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_aa_2stage_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrs_aa_2stage_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrs_aa_2stage_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *a, lapack_int lda, lapack_complex_float *tb, lapack_int ltb, lapack_int *ipiv, lapack_int *ipiv2, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_aa_2stage_work;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chetrs_aa_2stage_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *a, lapack_int lda, void *tb, lapack_int ltb, void *ipiv, void *ipiv2, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_aa_2stage_work++;
    if ( hook_pos_LAPACKE_chetrs_aa_2stage_work < __flexiblas_hooks->LAPACKE_chetrs_aa_2stage_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrs_aa_2stage_work.hook_function[hook_pos_LAPACKE_chetrs_aa_2stage_work];
    } else {
        hook_pos_LAPACKE_chetrs_aa_2stage_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_aa_2stage_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, tb, ltb, ipiv, ipiv2, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrs_aa_work = 0;
#endif

/* Wrapper for LAPACKE_chetrs_aa_work */
lapack_int  LAPACKE_chetrs_aa_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrs_aa_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_aa_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrs_aa_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrs_aa_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_aa_work;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chetrs_aa_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chetrs_aa_work++;
    if ( hook_pos_LAPACKE_chetrs_aa_work < __flexiblas_hooks->LAPACKE_chetrs_aa_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrs_aa_work.hook_function[hook_pos_LAPACKE_chetrs_aa_work];
    } else {
        hook_pos_LAPACKE_chetrs_aa_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_aa_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrs_rook = 0;
#endif

/* Wrapper for LAPACKE_chetrs_rook */
lapack_int  LAPACKE_chetrs_rook (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_rook = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_rook;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrs_rook.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrs_rook(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_rook;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chetrs_rook(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_rook++;
    if ( hook_pos_LAPACKE_chetrs_rook < __flexiblas_hooks->LAPACKE_chetrs_rook.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrs_rook.hook_function[hook_pos_LAPACKE_chetrs_rook];
    } else {
        hook_pos_LAPACKE_chetrs_rook = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_rook;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrs_rook_work = 0;
#endif

/* Wrapper for LAPACKE_chetrs_rook_work */
lapack_int  LAPACKE_chetrs_rook_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_rook_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_rook_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrs_rook_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrs_rook_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_rook_work;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chetrs_rook_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_rook_work++;
    if ( hook_pos_LAPACKE_chetrs_rook_work < __flexiblas_hooks->LAPACKE_chetrs_rook_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrs_rook_work.hook_function[hook_pos_LAPACKE_chetrs_rook_work];
    } else {
        hook_pos_LAPACKE_chetrs_rook_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_rook_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chetrs_work = 0;
#endif

/* Wrapper for LAPACKE_chetrs_work */
lapack_int  LAPACKE_chetrs_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chetrs_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chetrs_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *a, lapack_int lda, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_work;
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chetrs_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *a, lapack_int lda, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chetrs_work++;
    if ( hook_pos_LAPACKE_chetrs_work < __flexiblas_hooks->LAPACKE_chetrs_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chetrs_work.hook_function[hook_pos_LAPACKE_chetrs_work];
    } else {
        hook_pos_LAPACKE_chetrs_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chetrs_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, a, lda, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chfrk = 0;
#endif

/* Wrapper for LAPACKE_chfrk */
lapack_int  LAPACKE_chfrk (int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const lapack_complex_float *a, lapack_int lda, float beta, lapack_complex_float *c)
{
    lapack_int (*fn) (int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const void *a, lapack_int lda, float beta, void *c);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const void *a, lapack_int lda, float beta, void *c);
    hook_pos_LAPACKE_chfrk = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chfrk;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chfrk.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, transr, uplo, trans, n, k, alpha, a, lda, beta, c);
    } else {
        return fn(matrix_layout, transr, uplo, trans, n, k, alpha, a, lda, beta, c);
    }
    #else
    return fn(matrix_layout, transr, uplo, trans, n, k, alpha, a, lda, beta, c);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chfrk(int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const lapack_complex_float *a, lapack_int lda, float beta, lapack_complex_float *c)
{
    lapack_int (*fn) (int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const void *a, lapack_int lda, float beta, void *c);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chfrk;
    return fn(matrix_layout, transr, uplo, trans, n, k, alpha, a, lda, beta, c);
}

lapack_int flexiblas_chain_LAPACKE_chfrk(int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const void *a, lapack_int lda, float beta, void *c)
{
    lapack_int (*fn) (int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const void *a, lapack_int lda, float beta, void *c);
    hook_pos_LAPACKE_chfrk++;
    if ( hook_pos_LAPACKE_chfrk < __flexiblas_hooks->LAPACKE_chfrk.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chfrk.hook_function[hook_pos_LAPACKE_chfrk];
    } else {
        hook_pos_LAPACKE_chfrk = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chfrk;
    }
    return fn(matrix_layout, transr, uplo, trans, n, k, alpha, a, lda, beta, c);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chfrk_work = 0;
#endif

/* Wrapper for LAPACKE_chfrk_work */
lapack_int  LAPACKE_chfrk_work (int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const lapack_complex_float *a, lapack_int lda, float beta, lapack_complex_float *c)
{
    lapack_int (*fn) (int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const void *a, lapack_int lda, float beta, void *c);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const void *a, lapack_int lda, float beta, void *c);
    hook_pos_LAPACKE_chfrk_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chfrk_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chfrk_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, transr, uplo, trans, n, k, alpha, a, lda, beta, c);
    } else {
        return fn(matrix_layout, transr, uplo, trans, n, k, alpha, a, lda, beta, c);
    }
    #else
    return fn(matrix_layout, transr, uplo, trans, n, k, alpha, a, lda, beta, c);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chfrk_work(int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const lapack_complex_float *a, lapack_int lda, float beta, lapack_complex_float *c)
{
    lapack_int (*fn) (int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const void *a, lapack_int lda, float beta, void *c);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chfrk_work;
    return fn(matrix_layout, transr, uplo, trans, n, k, alpha, a, lda, beta, c);
}

lapack_int flexiblas_chain_LAPACKE_chfrk_work(int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const void *a, lapack_int lda, float beta, void *c)
{
    lapack_int (*fn) (int matrix_layout, char transr, char uplo, char trans, lapack_int n, lapack_int k, float alpha, const void *a, lapack_int lda, float beta, void *c);
    hook_pos_LAPACKE_chfrk_work++;
    if ( hook_pos_LAPACKE_chfrk_work < __flexiblas_hooks->LAPACKE_chfrk_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chfrk_work.hook_function[hook_pos_LAPACKE_chfrk_work];
    } else {
        hook_pos_LAPACKE_chfrk_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chfrk_work;
    }
    return fn(matrix_layout, transr, uplo, trans, n, k, alpha, a, lda, beta, c);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chgeqz = 0;
#endif

/* Wrapper for LAPACKE_chgeqz */
lapack_int  LAPACKE_chgeqz (int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *h, lapack_int ldh, lapack_complex_float *t, lapack_int ldt, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *t, lapack_int ldt, void *alpha, void *beta, void *q, lapack_int ldq, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *t, lapack_int ldt, void *alpha, void *beta, void *q, lapack_int ldq, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chgeqz = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chgeqz;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chgeqz.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, compq, compz, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz);
    } else {
        return fn(matrix_layout, job, compq, compz, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz);
    }
    #else
    return fn(matrix_layout, job, compq, compz, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chgeqz(int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *h, lapack_int ldh, lapack_complex_float *t, lapack_int ldt, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *t, lapack_int ldt, void *alpha, void *beta, void *q, lapack_int ldq, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chgeqz;
    return fn(matrix_layout, job, compq, compz, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_chgeqz(int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *t, lapack_int ldt, void *alpha, void *beta, void *q, lapack_int ldq, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *t, lapack_int ldt, void *alpha, void *beta, void *q, lapack_int ldq, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chgeqz++;
    if ( hook_pos_LAPACKE_chgeqz < __flexiblas_hooks->LAPACKE_chgeqz.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chgeqz.hook_function[hook_pos_LAPACKE_chgeqz];
    } else {
        hook_pos_LAPACKE_chgeqz = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chgeqz;
    }
    return fn(matrix_layout, job, compq, compz, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chgeqz_work = 0;
#endif

/* Wrapper for LAPACKE_chgeqz_work */
lapack_int  LAPACKE_chgeqz_work (int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *h, lapack_int ldh, lapack_complex_float *t, lapack_int ldt, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *t, lapack_int ldt, void *alpha, void *beta, void *q, lapack_int ldq, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *t, lapack_int ldt, void *alpha, void *beta, void *q, lapack_int ldq, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_chgeqz_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chgeqz_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chgeqz_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, compq, compz, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz, work, lwork, rwork);
    } else {
        return fn(matrix_layout, job, compq, compz, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz, work, lwork, rwork);
    }
    #else
    return fn(matrix_layout, job, compq, compz, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz, work, lwork, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chgeqz_work(int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *h, lapack_int ldh, lapack_complex_float *t, lapack_int ldt, lapack_complex_float *alpha, lapack_complex_float *beta, lapack_complex_float *q, lapack_int ldq, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *t, lapack_int ldt, void *alpha, void *beta, void *q, lapack_int ldq, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chgeqz_work;
    return fn(matrix_layout, job, compq, compz, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz, work, lwork, rwork);
}

lapack_int flexiblas_chain_LAPACKE_chgeqz_work(int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *t, lapack_int ldt, void *alpha, void *beta, void *q, lapack_int ldq, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char job, char compq, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *t, lapack_int ldt, void *alpha, void *beta, void *q, lapack_int ldq, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork);
    hook_pos_LAPACKE_chgeqz_work++;
    if ( hook_pos_LAPACKE_chgeqz_work < __flexiblas_hooks->LAPACKE_chgeqz_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chgeqz_work.hook_function[hook_pos_LAPACKE_chgeqz_work];
    } else {
        hook_pos_LAPACKE_chgeqz_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chgeqz_work;
    }
    return fn(matrix_layout, job, compq, compz, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz, work, lwork, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpcon = 0;
#endif

/* Wrapper for LAPACKE_chpcon */
lapack_int  LAPACKE_chpcon (int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *ap, const lapack_int *ipiv, float anorm, float *rcond)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *ap, const void *ipiv, float anorm, void *rcond);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, const void *ap, const void *ipiv, float anorm, void *rcond);
    hook_pos_LAPACKE_chpcon = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpcon;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpcon.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, ap, ipiv, anorm, rcond);
    } else {
        return fn(matrix_layout, uplo, n, ap, ipiv, anorm, rcond);
    }
    #else
    return fn(matrix_layout, uplo, n, ap, ipiv, anorm, rcond);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpcon(int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *ap, const lapack_int *ipiv, float anorm, float *rcond)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *ap, const void *ipiv, float anorm, void *rcond);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpcon;
    return fn(matrix_layout, uplo, n, ap, ipiv, anorm, rcond);
}

lapack_int flexiblas_chain_LAPACKE_chpcon(int matrix_layout, char uplo, lapack_int n, const void *ap, const void *ipiv, float anorm, void *rcond)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *ap, const void *ipiv, float anorm, void *rcond);
    hook_pos_LAPACKE_chpcon++;
    if ( hook_pos_LAPACKE_chpcon < __flexiblas_hooks->LAPACKE_chpcon.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpcon.hook_function[hook_pos_LAPACKE_chpcon];
    } else {
        hook_pos_LAPACKE_chpcon = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpcon;
    }
    return fn(matrix_layout, uplo, n, ap, ipiv, anorm, rcond);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpcon_work = 0;
#endif

/* Wrapper for LAPACKE_chpcon_work */
lapack_int  LAPACKE_chpcon_work (int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *ap, const lapack_int *ipiv, float anorm, float *rcond, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *ap, const void *ipiv, float anorm, void *rcond, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, const void *ap, const void *ipiv, float anorm, void *rcond, void *work);
    hook_pos_LAPACKE_chpcon_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpcon_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpcon_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, ap, ipiv, anorm, rcond, work);
    } else {
        return fn(matrix_layout, uplo, n, ap, ipiv, anorm, rcond, work);
    }
    #else
    return fn(matrix_layout, uplo, n, ap, ipiv, anorm, rcond, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpcon_work(int matrix_layout, char uplo, lapack_int n, const lapack_complex_float *ap, const lapack_int *ipiv, float anorm, float *rcond, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *ap, const void *ipiv, float anorm, void *rcond, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpcon_work;
    return fn(matrix_layout, uplo, n, ap, ipiv, anorm, rcond, work);
}

lapack_int flexiblas_chain_LAPACKE_chpcon_work(int matrix_layout, char uplo, lapack_int n, const void *ap, const void *ipiv, float anorm, void *rcond, void *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, const void *ap, const void *ipiv, float anorm, void *rcond, void *work);
    hook_pos_LAPACKE_chpcon_work++;
    if ( hook_pos_LAPACKE_chpcon_work < __flexiblas_hooks->LAPACKE_chpcon_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpcon_work.hook_function[hook_pos_LAPACKE_chpcon_work];
    } else {
        hook_pos_LAPACKE_chpcon_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpcon_work;
    }
    return fn(matrix_layout, uplo, n, ap, ipiv, anorm, rcond, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpev = 0;
#endif

/* Wrapper for LAPACKE_chpev */
lapack_int  LAPACKE_chpev (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chpev = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpev;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpev.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, ap, w, z, ldz);
    } else {
        return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpev(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpev;
    return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_chpev(int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chpev++;
    if ( hook_pos_LAPACKE_chpev < __flexiblas_hooks->LAPACKE_chpev.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpev.hook_function[hook_pos_LAPACKE_chpev];
    } else {
        hook_pos_LAPACKE_chpev = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpev;
    }
    return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpev_work = 0;
#endif

/* Wrapper for LAPACKE_chpev_work */
lapack_int  LAPACKE_chpev_work (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    hook_pos_LAPACKE_chpev_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpev_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpev_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, ap, w, z, ldz, work, rwork);
    } else {
        return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz, work, rwork);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpev_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpev_work;
    return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_chpev_work(int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    hook_pos_LAPACKE_chpev_work++;
    if ( hook_pos_LAPACKE_chpev_work < __flexiblas_hooks->LAPACKE_chpev_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpev_work.hook_function[hook_pos_LAPACKE_chpev_work];
    } else {
        hook_pos_LAPACKE_chpev_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpev_work;
    }
    return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpevd = 0;
#endif

/* Wrapper for LAPACKE_chpevd */
lapack_int  LAPACKE_chpevd (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chpevd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpevd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpevd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, ap, w, z, ldz);
    } else {
        return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpevd(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpevd;
    return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_chpevd(int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chpevd++;
    if ( hook_pos_LAPACKE_chpevd < __flexiblas_hooks->LAPACKE_chpevd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpevd.hook_function[hook_pos_LAPACKE_chpevd];
    } else {
        hook_pos_LAPACKE_chpevd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpevd;
    }
    return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpevd_work = 0;
#endif

/* Wrapper for LAPACKE_chpevd_work */
lapack_int  LAPACKE_chpevd_work (int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_chpevd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpevd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpevd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, uplo, n, ap, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    } else {
        return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    }
    #else
    return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpevd_work(int matrix_layout, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpevd_work;
    return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

lapack_int flexiblas_chain_LAPACKE_chpevd_work(int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char uplo, lapack_int n, void *ap, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_chpevd_work++;
    if ( hook_pos_LAPACKE_chpevd_work < __flexiblas_hooks->LAPACKE_chpevd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpevd_work.hook_function[hook_pos_LAPACKE_chpevd_work];
    } else {
        hook_pos_LAPACKE_chpevd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpevd_work;
    }
    return fn(matrix_layout, jobz, uplo, n, ap, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpevx = 0;
#endif

/* Wrapper for LAPACKE_chpevx */
lapack_int  LAPACKE_chpevx (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_chpevx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpevx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpevx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpevx(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpevx;
    return fn(matrix_layout, jobz, range, uplo, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

lapack_int flexiblas_chain_LAPACKE_chpevx(int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_chpevx++;
    if ( hook_pos_LAPACKE_chpevx < __flexiblas_hooks->LAPACKE_chpevx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpevx.hook_function[hook_pos_LAPACKE_chpevx];
    } else {
        hook_pos_LAPACKE_chpevx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpevx;
    }
    return fn(matrix_layout, jobz, range, uplo, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpevx_work = 0;
#endif

/* Wrapper for LAPACKE_chpevx_work */
lapack_int  LAPACKE_chpevx_work (int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_chpevx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpevx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpevx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, jobz, range, uplo, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
    } else {
        return fn(matrix_layout, jobz, range, uplo, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
    }
    #else
    return fn(matrix_layout, jobz, range, uplo, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpevx_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpevx_work;
    return fn(matrix_layout, jobz, range, uplo, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

lapack_int flexiblas_chain_LAPACKE_chpevx_work(int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, char jobz, char range, char uplo, lapack_int n, void *ap, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_chpevx_work++;
    if ( hook_pos_LAPACKE_chpevx_work < __flexiblas_hooks->LAPACKE_chpevx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpevx_work.hook_function[hook_pos_LAPACKE_chpevx_work];
    } else {
        hook_pos_LAPACKE_chpevx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpevx_work;
    }
    return fn(matrix_layout, jobz, range, uplo, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpgst = 0;
#endif

/* Wrapper for LAPACKE_chpgst */
lapack_int  LAPACKE_chpgst (int matrix_layout, lapack_int itype, char uplo, lapack_int n, lapack_complex_float *ap, const lapack_complex_float *bp)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *ap, const void *bp);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *ap, const void *bp);
    hook_pos_LAPACKE_chpgst = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgst;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpgst.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, uplo, n, ap, bp);
    } else {
        return fn(matrix_layout, itype, uplo, n, ap, bp);
    }
    #else
    return fn(matrix_layout, itype, uplo, n, ap, bp);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpgst(int matrix_layout, lapack_int itype, char uplo, lapack_int n, lapack_complex_float *ap, const lapack_complex_float *bp)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *ap, const void *bp);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgst;
    return fn(matrix_layout, itype, uplo, n, ap, bp);
}

lapack_int flexiblas_chain_LAPACKE_chpgst(int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *ap, const void *bp)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *ap, const void *bp);
    hook_pos_LAPACKE_chpgst++;
    if ( hook_pos_LAPACKE_chpgst < __flexiblas_hooks->LAPACKE_chpgst.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpgst.hook_function[hook_pos_LAPACKE_chpgst];
    } else {
        hook_pos_LAPACKE_chpgst = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpgst;
    }
    return fn(matrix_layout, itype, uplo, n, ap, bp);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpgst_work = 0;
#endif

/* Wrapper for LAPACKE_chpgst_work */
lapack_int  LAPACKE_chpgst_work (int matrix_layout, lapack_int itype, char uplo, lapack_int n, lapack_complex_float *ap, const lapack_complex_float *bp)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *ap, const void *bp);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *ap, const void *bp);
    hook_pos_LAPACKE_chpgst_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgst_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpgst_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, uplo, n, ap, bp);
    } else {
        return fn(matrix_layout, itype, uplo, n, ap, bp);
    }
    #else
    return fn(matrix_layout, itype, uplo, n, ap, bp);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpgst_work(int matrix_layout, lapack_int itype, char uplo, lapack_int n, lapack_complex_float *ap, const lapack_complex_float *bp)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *ap, const void *bp);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgst_work;
    return fn(matrix_layout, itype, uplo, n, ap, bp);
}

lapack_int flexiblas_chain_LAPACKE_chpgst_work(int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *ap, const void *bp)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char uplo, lapack_int n, void *ap, const void *bp);
    hook_pos_LAPACKE_chpgst_work++;
    if ( hook_pos_LAPACKE_chpgst_work < __flexiblas_hooks->LAPACKE_chpgst_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpgst_work.hook_function[hook_pos_LAPACKE_chpgst_work];
    } else {
        hook_pos_LAPACKE_chpgst_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpgst_work;
    }
    return fn(matrix_layout, itype, uplo, n, ap, bp);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpgv = 0;
#endif

/* Wrapper for LAPACKE_chpgv */
lapack_int  LAPACKE_chpgv (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, lapack_complex_float *bp, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chpgv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpgv.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz);
    } else {
        return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz);
    }
    #else
    return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpgv(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, lapack_complex_float *bp, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgv;
    return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_chpgv(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chpgv++;
    if ( hook_pos_LAPACKE_chpgv < __flexiblas_hooks->LAPACKE_chpgv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpgv.hook_function[hook_pos_LAPACKE_chpgv];
    } else {
        hook_pos_LAPACKE_chpgv = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpgv;
    }
    return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpgv_work = 0;
#endif

/* Wrapper for LAPACKE_chpgv_work */
lapack_int  LAPACKE_chpgv_work (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, lapack_complex_float *bp, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    hook_pos_LAPACKE_chpgv_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgv_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpgv_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz, work, rwork);
    } else {
        return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz, work, rwork);
    }
    #else
    return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpgv_work(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, lapack_complex_float *bp, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgv_work;
    return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_chpgv_work(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz, void *work, void *rwork);
    hook_pos_LAPACKE_chpgv_work++;
    if ( hook_pos_LAPACKE_chpgv_work < __flexiblas_hooks->LAPACKE_chpgv_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpgv_work.hook_function[hook_pos_LAPACKE_chpgv_work];
    } else {
        hook_pos_LAPACKE_chpgv_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpgv_work;
    }
    return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpgvd = 0;
#endif

/* Wrapper for LAPACKE_chpgvd */
lapack_int  LAPACKE_chpgvd (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, lapack_complex_float *bp, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chpgvd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgvd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpgvd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz);
    } else {
        return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz);
    }
    #else
    return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpgvd(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, lapack_complex_float *bp, float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgvd;
    return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_chpgvd(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chpgvd++;
    if ( hook_pos_LAPACKE_chpgvd < __flexiblas_hooks->LAPACKE_chpgvd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpgvd.hook_function[hook_pos_LAPACKE_chpgvd];
    } else {
        hook_pos_LAPACKE_chpgvd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpgvd;
    }
    return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpgvd_work = 0;
#endif

/* Wrapper for LAPACKE_chpgvd_work */
lapack_int  LAPACKE_chpgvd_work (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, lapack_complex_float *bp, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_chpgvd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgvd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpgvd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    } else {
        return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    }
    #else
    return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpgvd_work(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, lapack_complex_float *ap, lapack_complex_float *bp, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork, float *rwork, lapack_int lrwork, lapack_int *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgvd_work;
    return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

lapack_int flexiblas_chain_LAPACKE_chpgvd_work(int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char uplo, lapack_int n, void *ap, void *bp, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork, void *rwork, lapack_int lrwork, void *iwork, lapack_int liwork);
    hook_pos_LAPACKE_chpgvd_work++;
    if ( hook_pos_LAPACKE_chpgvd_work < __flexiblas_hooks->LAPACKE_chpgvd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpgvd_work.hook_function[hook_pos_LAPACKE_chpgvd_work];
    } else {
        hook_pos_LAPACKE_chpgvd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpgvd_work;
    }
    return fn(matrix_layout, itype, jobz, uplo, n, ap, bp, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpgvx = 0;
#endif

/* Wrapper for LAPACKE_chpgvx */
lapack_int  LAPACKE_chpgvx (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *ap, lapack_complex_float *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *ap, void *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *ap, void *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_chpgvx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgvx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpgvx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, range, uplo, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    } else {
        return fn(matrix_layout, itype, jobz, range, uplo, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    }
    #else
    return fn(matrix_layout, itype, jobz, range, uplo, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpgvx(int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *ap, lapack_complex_float *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *ap, void *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgvx;
    return fn(matrix_layout, itype, jobz, range, uplo, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

lapack_int flexiblas_chain_LAPACKE_chpgvx(int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *ap, void *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *ap, void *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *ifail);
    hook_pos_LAPACKE_chpgvx++;
    if ( hook_pos_LAPACKE_chpgvx < __flexiblas_hooks->LAPACKE_chpgvx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpgvx.hook_function[hook_pos_LAPACKE_chpgvx];
    } else {
        hook_pos_LAPACKE_chpgvx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpgvx;
    }
    return fn(matrix_layout, itype, jobz, range, uplo, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpgvx_work = 0;
#endif

/* Wrapper for LAPACKE_chpgvx_work */
lapack_int  LAPACKE_chpgvx_work (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *ap, lapack_complex_float *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *ap, void *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *ap, void *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_chpgvx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgvx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpgvx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, itype, jobz, range, uplo, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
    } else {
        return fn(matrix_layout, itype, jobz, range, uplo, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
    }
    #else
    return fn(matrix_layout, itype, jobz, range, uplo, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpgvx_work(int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, lapack_complex_float *ap, lapack_complex_float *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, lapack_int *m, float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, float *rwork, lapack_int *iwork, lapack_int *ifail)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *ap, void *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpgvx_work;
    return fn(matrix_layout, itype, jobz, range, uplo, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

lapack_int flexiblas_chain_LAPACKE_chpgvx_work(int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *ap, void *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail)
{
    lapack_int (*fn) (int matrix_layout, lapack_int itype, char jobz, char range, char uplo, lapack_int n, void *ap, void *bp, float vl, float vu, lapack_int il, lapack_int iu, float abstol, void *m, void *w, void *z, lapack_int ldz, void *work, void *rwork, void *iwork, void *ifail);
    hook_pos_LAPACKE_chpgvx_work++;
    if ( hook_pos_LAPACKE_chpgvx_work < __flexiblas_hooks->LAPACKE_chpgvx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpgvx_work.hook_function[hook_pos_LAPACKE_chpgvx_work];
    } else {
        hook_pos_LAPACKE_chpgvx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpgvx_work;
    }
    return fn(matrix_layout, itype, jobz, range, uplo, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chprfs = 0;
#endif

/* Wrapper for LAPACKE_chprfs */
lapack_int  LAPACKE_chprfs (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *ap, const lapack_complex_float *afp, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *afp, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *afp, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    hook_pos_LAPACKE_chprfs = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chprfs;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chprfs.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chprfs(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *ap, const lapack_complex_float *afp, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *afp, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chprfs;
    return fn(matrix_layout, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr);
}

lapack_int flexiblas_chain_LAPACKE_chprfs(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *afp, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *afp, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr);
    hook_pos_LAPACKE_chprfs++;
    if ( hook_pos_LAPACKE_chprfs < __flexiblas_hooks->LAPACKE_chprfs.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chprfs.hook_function[hook_pos_LAPACKE_chprfs];
    } else {
        hook_pos_LAPACKE_chprfs = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chprfs;
    }
    return fn(matrix_layout, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chprfs_work = 0;
#endif

/* Wrapper for LAPACKE_chprfs_work */
lapack_int  LAPACKE_chprfs_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *ap, const lapack_complex_float *afp, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *afp, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *afp, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_chprfs_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chprfs_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chprfs_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chprfs_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *ap, const lapack_complex_float *afp, const lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *afp, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chprfs_work;
    return fn(matrix_layout, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_chprfs_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *afp, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *afp, const void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_chprfs_work++;
    if ( hook_pos_LAPACKE_chprfs_work < __flexiblas_hooks->LAPACKE_chprfs_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chprfs_work.hook_function[hook_pos_LAPACKE_chprfs_work];
    } else {
        hook_pos_LAPACKE_chprfs_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chprfs_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpsv = 0;
#endif

/* Wrapper for LAPACKE_chpsv */
lapack_int  LAPACKE_chpsv (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *ap, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *ap, void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *ap, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chpsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpsv.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpsv(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *ap, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *ap, void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpsv;
    return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chpsv(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *ap, void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *ap, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chpsv++;
    if ( hook_pos_LAPACKE_chpsv < __flexiblas_hooks->LAPACKE_chpsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpsv.hook_function[hook_pos_LAPACKE_chpsv];
    } else {
        hook_pos_LAPACKE_chpsv = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpsv;
    }
    return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpsv_work = 0;
#endif

/* Wrapper for LAPACKE_chpsv_work */
lapack_int  LAPACKE_chpsv_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *ap, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *ap, void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *ap, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chpsv_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpsv_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpsv_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpsv_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, lapack_complex_float *ap, lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *ap, void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpsv_work;
    return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chpsv_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *ap, void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, void *ap, void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chpsv_work++;
    if ( hook_pos_LAPACKE_chpsv_work < __flexiblas_hooks->LAPACKE_chpsv_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpsv_work.hook_function[hook_pos_LAPACKE_chpsv_work];
    } else {
        hook_pos_LAPACKE_chpsv_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpsv_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpsvx = 0;
#endif

/* Wrapper for LAPACKE_chpsvx */
lapack_int  LAPACKE_chpsvx (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *ap, lapack_complex_float *afp, lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *ap, void *afp, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *ap, void *afp, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr);
    hook_pos_LAPACKE_chpsvx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpsvx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpsvx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, fact, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
    } else {
        return fn(matrix_layout, fact, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
    }
    #else
    return fn(matrix_layout, fact, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpsvx(int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *ap, lapack_complex_float *afp, lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr)
{
    lapack_int (*fn) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *ap, void *afp, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpsvx;
    return fn(matrix_layout, fact, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

lapack_int flexiblas_chain_LAPACKE_chpsvx(int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *ap, void *afp, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr)
{
    lapack_int (*fn) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *ap, void *afp, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr);
    hook_pos_LAPACKE_chpsvx++;
    if ( hook_pos_LAPACKE_chpsvx < __flexiblas_hooks->LAPACKE_chpsvx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpsvx.hook_function[hook_pos_LAPACKE_chpsvx];
    } else {
        hook_pos_LAPACKE_chpsvx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpsvx;
    }
    return fn(matrix_layout, fact, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chpsvx_work = 0;
#endif

/* Wrapper for LAPACKE_chpsvx_work */
lapack_int  LAPACKE_chpsvx_work (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *ap, lapack_complex_float *afp, lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *ap, void *afp, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *ap, void *afp, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_chpsvx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chpsvx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chpsvx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, fact, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
    } else {
        return fn(matrix_layout, fact, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
    }
    #else
    return fn(matrix_layout, fact, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chpsvx_work(int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *ap, lapack_complex_float *afp, lapack_int *ipiv, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *x, lapack_int ldx, float *rcond, float *ferr, float *berr, lapack_complex_float *work, float *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *ap, void *afp, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chpsvx_work;
    return fn(matrix_layout, fact, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

lapack_int flexiblas_chain_LAPACKE_chpsvx_work(int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *ap, void *afp, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork)
{
    lapack_int (*fn) (int matrix_layout, char fact, char uplo, lapack_int n, lapack_int nrhs, const void *ap, void *afp, void *ipiv, const void *b, lapack_int ldb, void *x, lapack_int ldx, void *rcond, void *ferr, void *berr, void *work, void *rwork);
    hook_pos_LAPACKE_chpsvx_work++;
    if ( hook_pos_LAPACKE_chpsvx_work < __flexiblas_hooks->LAPACKE_chpsvx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chpsvx_work.hook_function[hook_pos_LAPACKE_chpsvx_work];
    } else {
        hook_pos_LAPACKE_chpsvx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chpsvx_work;
    }
    return fn(matrix_layout, fact, uplo, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chptrd = 0;
#endif

/* Wrapper for LAPACKE_chptrd */
lapack_int  LAPACKE_chptrd (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *ap, float *d, float *e, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, void *d, void *e, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *ap, void *d, void *e, void *tau);
    hook_pos_LAPACKE_chptrd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chptrd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chptrd.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, ap, d, e, tau);
    } else {
        return fn(matrix_layout, uplo, n, ap, d, e, tau);
    }
    #else
    return fn(matrix_layout, uplo, n, ap, d, e, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chptrd(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *ap, float *d, float *e, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, void *d, void *e, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chptrd;
    return fn(matrix_layout, uplo, n, ap, d, e, tau);
}

lapack_int flexiblas_chain_LAPACKE_chptrd(int matrix_layout, char uplo, lapack_int n, void *ap, void *d, void *e, void *tau)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, void *d, void *e, void *tau);
    hook_pos_LAPACKE_chptrd++;
    if ( hook_pos_LAPACKE_chptrd < __flexiblas_hooks->LAPACKE_chptrd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chptrd.hook_function[hook_pos_LAPACKE_chptrd];
    } else {
        hook_pos_LAPACKE_chptrd = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chptrd;
    }
    return fn(matrix_layout, uplo, n, ap, d, e, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chptrd_work = 0;
#endif

/* Wrapper for LAPACKE_chptrd_work */
lapack_int  LAPACKE_chptrd_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *ap, float *d, float *e, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, void *d, void *e, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *ap, void *d, void *e, void *tau);
    hook_pos_LAPACKE_chptrd_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chptrd_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chptrd_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, ap, d, e, tau);
    } else {
        return fn(matrix_layout, uplo, n, ap, d, e, tau);
    }
    #else
    return fn(matrix_layout, uplo, n, ap, d, e, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chptrd_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *ap, float *d, float *e, lapack_complex_float *tau)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, void *d, void *e, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chptrd_work;
    return fn(matrix_layout, uplo, n, ap, d, e, tau);
}

lapack_int flexiblas_chain_LAPACKE_chptrd_work(int matrix_layout, char uplo, lapack_int n, void *ap, void *d, void *e, void *tau)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, void *d, void *e, void *tau);
    hook_pos_LAPACKE_chptrd_work++;
    if ( hook_pos_LAPACKE_chptrd_work < __flexiblas_hooks->LAPACKE_chptrd_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chptrd_work.hook_function[hook_pos_LAPACKE_chptrd_work];
    } else {
        hook_pos_LAPACKE_chptrd_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chptrd_work;
    }
    return fn(matrix_layout, uplo, n, ap, d, e, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chptrf = 0;
#endif

/* Wrapper for LAPACKE_chptrf */
lapack_int  LAPACKE_chptrf (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *ap, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *ap, void *ipiv);
    hook_pos_LAPACKE_chptrf = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chptrf;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chptrf.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, ap, ipiv);
    } else {
        return fn(matrix_layout, uplo, n, ap, ipiv);
    }
    #else
    return fn(matrix_layout, uplo, n, ap, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chptrf(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *ap, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chptrf;
    return fn(matrix_layout, uplo, n, ap, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_chptrf(int matrix_layout, char uplo, lapack_int n, void *ap, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, void *ipiv);
    hook_pos_LAPACKE_chptrf++;
    if ( hook_pos_LAPACKE_chptrf < __flexiblas_hooks->LAPACKE_chptrf.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chptrf.hook_function[hook_pos_LAPACKE_chptrf];
    } else {
        hook_pos_LAPACKE_chptrf = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chptrf;
    }
    return fn(matrix_layout, uplo, n, ap, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chptrf_work = 0;
#endif

/* Wrapper for LAPACKE_chptrf_work */
lapack_int  LAPACKE_chptrf_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *ap, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *ap, void *ipiv);
    hook_pos_LAPACKE_chptrf_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chptrf_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chptrf_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, ap, ipiv);
    } else {
        return fn(matrix_layout, uplo, n, ap, ipiv);
    }
    #else
    return fn(matrix_layout, uplo, n, ap, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chptrf_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *ap, lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chptrf_work;
    return fn(matrix_layout, uplo, n, ap, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_chptrf_work(int matrix_layout, char uplo, lapack_int n, void *ap, void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, void *ipiv);
    hook_pos_LAPACKE_chptrf_work++;
    if ( hook_pos_LAPACKE_chptrf_work < __flexiblas_hooks->LAPACKE_chptrf_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chptrf_work.hook_function[hook_pos_LAPACKE_chptrf_work];
    } else {
        hook_pos_LAPACKE_chptrf_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chptrf_work;
    }
    return fn(matrix_layout, uplo, n, ap, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chptri = 0;
#endif

/* Wrapper for LAPACKE_chptri */
lapack_int  LAPACKE_chptri (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *ap, const lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, const void *ipiv);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *ap, const void *ipiv);
    hook_pos_LAPACKE_chptri = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chptri;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chptri.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, ap, ipiv);
    } else {
        return fn(matrix_layout, uplo, n, ap, ipiv);
    }
    #else
    return fn(matrix_layout, uplo, n, ap, ipiv);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chptri(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *ap, const lapack_int *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, const void *ipiv);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chptri;
    return fn(matrix_layout, uplo, n, ap, ipiv);
}

lapack_int flexiblas_chain_LAPACKE_chptri(int matrix_layout, char uplo, lapack_int n, void *ap, const void *ipiv)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, const void *ipiv);
    hook_pos_LAPACKE_chptri++;
    if ( hook_pos_LAPACKE_chptri < __flexiblas_hooks->LAPACKE_chptri.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chptri.hook_function[hook_pos_LAPACKE_chptri];
    } else {
        hook_pos_LAPACKE_chptri = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chptri;
    }
    return fn(matrix_layout, uplo, n, ap, ipiv);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chptri_work = 0;
#endif

/* Wrapper for LAPACKE_chptri_work */
lapack_int  LAPACKE_chptri_work (int matrix_layout, char uplo, lapack_int n, lapack_complex_float *ap, const lapack_int *ipiv, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, const void *ipiv, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, void *ap, const void *ipiv, void *work);
    hook_pos_LAPACKE_chptri_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chptri_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chptri_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, ap, ipiv, work);
    } else {
        return fn(matrix_layout, uplo, n, ap, ipiv, work);
    }
    #else
    return fn(matrix_layout, uplo, n, ap, ipiv, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chptri_work(int matrix_layout, char uplo, lapack_int n, lapack_complex_float *ap, const lapack_int *ipiv, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, const void *ipiv, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chptri_work;
    return fn(matrix_layout, uplo, n, ap, ipiv, work);
}

lapack_int flexiblas_chain_LAPACKE_chptri_work(int matrix_layout, char uplo, lapack_int n, void *ap, const void *ipiv, void *work)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, void *ap, const void *ipiv, void *work);
    hook_pos_LAPACKE_chptri_work++;
    if ( hook_pos_LAPACKE_chptri_work < __flexiblas_hooks->LAPACKE_chptri_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chptri_work.hook_function[hook_pos_LAPACKE_chptri_work];
    } else {
        hook_pos_LAPACKE_chptri_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chptri_work;
    }
    return fn(matrix_layout, uplo, n, ap, ipiv, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chptrs = 0;
#endif

/* Wrapper for LAPACKE_chptrs */
lapack_int  LAPACKE_chptrs (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *ap, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chptrs = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chptrs;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chptrs.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chptrs(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *ap, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chptrs;
    return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chptrs(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chptrs++;
    if ( hook_pos_LAPACKE_chptrs < __flexiblas_hooks->LAPACKE_chptrs.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chptrs.hook_function[hook_pos_LAPACKE_chptrs];
    } else {
        hook_pos_LAPACKE_chptrs = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chptrs;
    }
    return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chptrs_work = 0;
#endif

/* Wrapper for LAPACKE_chptrs_work */
lapack_int  LAPACKE_chptrs_work (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *ap, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *ipiv, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chptrs_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chptrs_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chptrs_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
    } else {
        return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chptrs_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const lapack_complex_float *ap, const lapack_int *ipiv, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *ipiv, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chptrs_work;
    return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_chptrs_work(int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *ipiv, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int n, lapack_int nrhs, const void *ap, const void *ipiv, void *b, lapack_int ldb);
    hook_pos_LAPACKE_chptrs_work++;
    if ( hook_pos_LAPACKE_chptrs_work < __flexiblas_hooks->LAPACKE_chptrs_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chptrs_work.hook_function[hook_pos_LAPACKE_chptrs_work];
    } else {
        hook_pos_LAPACKE_chptrs_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chptrs_work;
    }
    return fn(matrix_layout, uplo, n, nrhs, ap, ipiv, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chsein = 0;
#endif

/* Wrapper for LAPACKE_chsein */
lapack_int  LAPACKE_chsein (int matrix_layout, char job, char eigsrc, char initv, const lapack_int *select, lapack_int n, const lapack_complex_float *h, lapack_int ldh, lapack_complex_float *w, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_int mm, lapack_int *m, lapack_int *ifaill, lapack_int *ifailr)
{
    lapack_int (*fn) (int matrix_layout, char job, char eigsrc, char initv, const void *select, lapack_int n, const void *h, lapack_int ldh, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, lapack_int mm, void *m, void *ifaill, void *ifailr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, char eigsrc, char initv, const void *select, lapack_int n, const void *h, lapack_int ldh, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, lapack_int mm, void *m, void *ifaill, void *ifailr);
    hook_pos_LAPACKE_chsein = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chsein;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chsein.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, eigsrc, initv, select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, ifaill, ifailr);
    } else {
        return fn(matrix_layout, job, eigsrc, initv, select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, ifaill, ifailr);
    }
    #else
    return fn(matrix_layout, job, eigsrc, initv, select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, ifaill, ifailr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chsein(int matrix_layout, char job, char eigsrc, char initv, const lapack_int *select, lapack_int n, const lapack_complex_float *h, lapack_int ldh, lapack_complex_float *w, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_int mm, lapack_int *m, lapack_int *ifaill, lapack_int *ifailr)
{
    lapack_int (*fn) (int matrix_layout, char job, char eigsrc, char initv, const void *select, lapack_int n, const void *h, lapack_int ldh, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, lapack_int mm, void *m, void *ifaill, void *ifailr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chsein;
    return fn(matrix_layout, job, eigsrc, initv, select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, ifaill, ifailr);
}

lapack_int flexiblas_chain_LAPACKE_chsein(int matrix_layout, char job, char eigsrc, char initv, const void *select, lapack_int n, const void *h, lapack_int ldh, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, lapack_int mm, void *m, void *ifaill, void *ifailr)
{
    lapack_int (*fn) (int matrix_layout, char job, char eigsrc, char initv, const void *select, lapack_int n, const void *h, lapack_int ldh, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, lapack_int mm, void *m, void *ifaill, void *ifailr);
    hook_pos_LAPACKE_chsein++;
    if ( hook_pos_LAPACKE_chsein < __flexiblas_hooks->LAPACKE_chsein.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chsein.hook_function[hook_pos_LAPACKE_chsein];
    } else {
        hook_pos_LAPACKE_chsein = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chsein;
    }
    return fn(matrix_layout, job, eigsrc, initv, select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, ifaill, ifailr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chsein_work = 0;
#endif

/* Wrapper for LAPACKE_chsein_work */
lapack_int  LAPACKE_chsein_work (int matrix_layout, char job, char eigsrc, char initv, const lapack_int *select, lapack_int n, const lapack_complex_float *h, lapack_int ldh, lapack_complex_float *w, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_int mm, lapack_int *m, lapack_complex_float *work, float *rwork, lapack_int *ifaill, lapack_int *ifailr)
{
    lapack_int (*fn) (int matrix_layout, char job, char eigsrc, char initv, const void *select, lapack_int n, const void *h, lapack_int ldh, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, lapack_int mm, void *m, void *work, void *rwork, void *ifaill, void *ifailr);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, char eigsrc, char initv, const void *select, lapack_int n, const void *h, lapack_int ldh, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, lapack_int mm, void *m, void *work, void *rwork, void *ifaill, void *ifailr);
    hook_pos_LAPACKE_chsein_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chsein_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chsein_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, eigsrc, initv, select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, work, rwork, ifaill, ifailr);
    } else {
        return fn(matrix_layout, job, eigsrc, initv, select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, work, rwork, ifaill, ifailr);
    }
    #else
    return fn(matrix_layout, job, eigsrc, initv, select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, work, rwork, ifaill, ifailr);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chsein_work(int matrix_layout, char job, char eigsrc, char initv, const lapack_int *select, lapack_int n, const lapack_complex_float *h, lapack_int ldh, lapack_complex_float *w, lapack_complex_float *vl, lapack_int ldvl, lapack_complex_float *vr, lapack_int ldvr, lapack_int mm, lapack_int *m, lapack_complex_float *work, float *rwork, lapack_int *ifaill, lapack_int *ifailr)
{
    lapack_int (*fn) (int matrix_layout, char job, char eigsrc, char initv, const void *select, lapack_int n, const void *h, lapack_int ldh, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, lapack_int mm, void *m, void *work, void *rwork, void *ifaill, void *ifailr);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chsein_work;
    return fn(matrix_layout, job, eigsrc, initv, select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, work, rwork, ifaill, ifailr);
}

lapack_int flexiblas_chain_LAPACKE_chsein_work(int matrix_layout, char job, char eigsrc, char initv, const void *select, lapack_int n, const void *h, lapack_int ldh, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, lapack_int mm, void *m, void *work, void *rwork, void *ifaill, void *ifailr)
{
    lapack_int (*fn) (int matrix_layout, char job, char eigsrc, char initv, const void *select, lapack_int n, const void *h, lapack_int ldh, void *w, void *vl, lapack_int ldvl, void *vr, lapack_int ldvr, lapack_int mm, void *m, void *work, void *rwork, void *ifaill, void *ifailr);
    hook_pos_LAPACKE_chsein_work++;
    if ( hook_pos_LAPACKE_chsein_work < __flexiblas_hooks->LAPACKE_chsein_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chsein_work.hook_function[hook_pos_LAPACKE_chsein_work];
    } else {
        hook_pos_LAPACKE_chsein_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chsein_work;
    }
    return fn(matrix_layout, job, eigsrc, initv, select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, work, rwork, ifaill, ifailr);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chseqr = 0;
#endif

/* Wrapper for LAPACKE_chseqr */
lapack_int  LAPACKE_chseqr (int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *h, lapack_int ldh, lapack_complex_float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *w, void *z, lapack_int ldz);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chseqr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chseqr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chseqr.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, compz, n, ilo, ihi, h, ldh, w, z, ldz);
    } else {
        return fn(matrix_layout, job, compz, n, ilo, ihi, h, ldh, w, z, ldz);
    }
    #else
    return fn(matrix_layout, job, compz, n, ilo, ihi, h, ldh, w, z, ldz);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chseqr(int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *h, lapack_int ldh, lapack_complex_float *w, lapack_complex_float *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *w, void *z, lapack_int ldz);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chseqr;
    return fn(matrix_layout, job, compz, n, ilo, ihi, h, ldh, w, z, ldz);
}

lapack_int flexiblas_chain_LAPACKE_chseqr(int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *w, void *z, lapack_int ldz)
{
    lapack_int (*fn) (int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *w, void *z, lapack_int ldz);
    hook_pos_LAPACKE_chseqr++;
    if ( hook_pos_LAPACKE_chseqr < __flexiblas_hooks->LAPACKE_chseqr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chseqr.hook_function[hook_pos_LAPACKE_chseqr];
    } else {
        hook_pos_LAPACKE_chseqr = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chseqr;
    }
    return fn(matrix_layout, job, compz, n, ilo, ihi, h, ldh, w, z, ldz);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_chseqr_work = 0;
#endif

/* Wrapper for LAPACKE_chseqr_work */
lapack_int  LAPACKE_chseqr_work (int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *h, lapack_int ldh, lapack_complex_float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chseqr_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_chseqr_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_chseqr_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, job, compz, n, ilo, ihi, h, ldh, w, z, ldz, work, lwork);
    } else {
        return fn(matrix_layout, job, compz, n, ilo, ihi, h, ldh, w, z, ldz, work, lwork);
    }
    #else
    return fn(matrix_layout, job, compz, n, ilo, ihi, h, ldh, w, z, ldz, work, lwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_chseqr_work(int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, lapack_complex_float *h, lapack_int ldh, lapack_complex_float *w, lapack_complex_float *z, lapack_int ldz, lapack_complex_float *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_chseqr_work;
    return fn(matrix_layout, job, compz, n, ilo, ihi, h, ldh, w, z, ldz, work, lwork);
}

lapack_int flexiblas_chain_LAPACKE_chseqr_work(int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork)
{
    lapack_int (*fn) (int matrix_layout, char job, char compz, lapack_int n, lapack_int ilo, lapack_int ihi, void *h, lapack_int ldh, void *w, void *z, lapack_int ldz, void *work, lapack_int lwork);
    hook_pos_LAPACKE_chseqr_work++;
    if ( hook_pos_LAPACKE_chseqr_work < __flexiblas_hooks->LAPACKE_chseqr_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_chseqr_work.hook_function[hook_pos_LAPACKE_chseqr_work];
    } else {
        hook_pos_LAPACKE_chseqr_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_chseqr_work;
    }
    return fn(matrix_layout, job, compz, n, ilo, ihi, h, ldh, w, z, ldz, work, lwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clacgv = 0;
#endif

/* Wrapper for LAPACKE_clacgv */
lapack_int  LAPACKE_clacgv (lapack_int n, lapack_complex_float *x, lapack_int incx)
{
    lapack_int (*fn) (lapack_int n, void *x, lapack_int incx);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (lapack_int n, void *x, lapack_int incx);
    hook_pos_LAPACKE_clacgv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clacgv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clacgv.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(n, x, incx);
    } else {
        return fn(n, x, incx);
    }
    #else
    return fn(n, x, incx);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clacgv(lapack_int n, lapack_complex_float *x, lapack_int incx)
{
    lapack_int (*fn) (lapack_int n, void *x, lapack_int incx);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clacgv;
    return fn(n, x, incx);
}

lapack_int flexiblas_chain_LAPACKE_clacgv(lapack_int n, void *x, lapack_int incx)
{
    lapack_int (*fn) (lapack_int n, void *x, lapack_int incx);
    hook_pos_LAPACKE_clacgv++;
    if ( hook_pos_LAPACKE_clacgv < __flexiblas_hooks->LAPACKE_clacgv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clacgv.hook_function[hook_pos_LAPACKE_clacgv];
    } else {
        hook_pos_LAPACKE_clacgv = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clacgv;
    }
    return fn(n, x, incx);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clacgv_work = 0;
#endif

/* Wrapper for LAPACKE_clacgv_work */
lapack_int  LAPACKE_clacgv_work (lapack_int n, lapack_complex_float *x, lapack_int incx)
{
    lapack_int (*fn) (lapack_int n, void *x, lapack_int incx);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (lapack_int n, void *x, lapack_int incx);
    hook_pos_LAPACKE_clacgv_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clacgv_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clacgv_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(n, x, incx);
    } else {
        return fn(n, x, incx);
    }
    #else
    return fn(n, x, incx);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clacgv_work(lapack_int n, lapack_complex_float *x, lapack_int incx)
{
    lapack_int (*fn) (lapack_int n, void *x, lapack_int incx);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clacgv_work;
    return fn(n, x, incx);
}

lapack_int flexiblas_chain_LAPACKE_clacgv_work(lapack_int n, void *x, lapack_int incx)
{
    lapack_int (*fn) (lapack_int n, void *x, lapack_int incx);
    hook_pos_LAPACKE_clacgv_work++;
    if ( hook_pos_LAPACKE_clacgv_work < __flexiblas_hooks->LAPACKE_clacgv_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clacgv_work.hook_function[hook_pos_LAPACKE_clacgv_work];
    } else {
        hook_pos_LAPACKE_clacgv_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clacgv_work;
    }
    return fn(n, x, incx);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clacn2 = 0;
#endif

/* Wrapper for LAPACKE_clacn2 */
lapack_int  LAPACKE_clacn2 (lapack_int n, lapack_complex_float *v, lapack_complex_float *x, float *est, lapack_int *kase, lapack_int *isave)
{
    lapack_int (*fn) (lapack_int n, void *v, void *x, void *est, void *kase, void *isave);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (lapack_int n, void *v, void *x, void *est, void *kase, void *isave);
    hook_pos_LAPACKE_clacn2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clacn2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clacn2.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(n, v, x, est, kase, isave);
    } else {
        return fn(n, v, x, est, kase, isave);
    }
    #else
    return fn(n, v, x, est, kase, isave);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clacn2(lapack_int n, lapack_complex_float *v, lapack_complex_float *x, float *est, lapack_int *kase, lapack_int *isave)
{
    lapack_int (*fn) (lapack_int n, void *v, void *x, void *est, void *kase, void *isave);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clacn2;
    return fn(n, v, x, est, kase, isave);
}

lapack_int flexiblas_chain_LAPACKE_clacn2(lapack_int n, void *v, void *x, void *est, void *kase, void *isave)
{
    lapack_int (*fn) (lapack_int n, void *v, void *x, void *est, void *kase, void *isave);
    hook_pos_LAPACKE_clacn2++;
    if ( hook_pos_LAPACKE_clacn2 < __flexiblas_hooks->LAPACKE_clacn2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clacn2.hook_function[hook_pos_LAPACKE_clacn2];
    } else {
        hook_pos_LAPACKE_clacn2 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clacn2;
    }
    return fn(n, v, x, est, kase, isave);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clacn2_work = 0;
#endif

/* Wrapper for LAPACKE_clacn2_work */
lapack_int  LAPACKE_clacn2_work (lapack_int n, lapack_complex_float *v, lapack_complex_float *x, float *est, lapack_int *kase, lapack_int *isave)
{
    lapack_int (*fn) (lapack_int n, void *v, void *x, void *est, void *kase, void *isave);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (lapack_int n, void *v, void *x, void *est, void *kase, void *isave);
    hook_pos_LAPACKE_clacn2_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clacn2_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clacn2_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(n, v, x, est, kase, isave);
    } else {
        return fn(n, v, x, est, kase, isave);
    }
    #else
    return fn(n, v, x, est, kase, isave);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clacn2_work(lapack_int n, lapack_complex_float *v, lapack_complex_float *x, float *est, lapack_int *kase, lapack_int *isave)
{
    lapack_int (*fn) (lapack_int n, void *v, void *x, void *est, void *kase, void *isave);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clacn2_work;
    return fn(n, v, x, est, kase, isave);
}

lapack_int flexiblas_chain_LAPACKE_clacn2_work(lapack_int n, void *v, void *x, void *est, void *kase, void *isave)
{
    lapack_int (*fn) (lapack_int n, void *v, void *x, void *est, void *kase, void *isave);
    hook_pos_LAPACKE_clacn2_work++;
    if ( hook_pos_LAPACKE_clacn2_work < __flexiblas_hooks->LAPACKE_clacn2_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clacn2_work.hook_function[hook_pos_LAPACKE_clacn2_work];
    } else {
        hook_pos_LAPACKE_clacn2_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clacn2_work;
    }
    return fn(n, v, x, est, kase, isave);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clacp2 = 0;
#endif

/* Wrapper for LAPACKE_clacp2 */
lapack_int  LAPACKE_clacp2 (int matrix_layout, char uplo, lapack_int m, lapack_int n, const float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    hook_pos_LAPACKE_clacp2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clacp2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clacp2.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, m, n, a, lda, b, ldb);
    } else {
        return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clacp2(int matrix_layout, char uplo, lapack_int m, lapack_int n, const float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clacp2;
    return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_clacp2(int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    hook_pos_LAPACKE_clacp2++;
    if ( hook_pos_LAPACKE_clacp2 < __flexiblas_hooks->LAPACKE_clacp2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clacp2.hook_function[hook_pos_LAPACKE_clacp2];
    } else {
        hook_pos_LAPACKE_clacp2 = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clacp2;
    }
    return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clacp2_work = 0;
#endif

/* Wrapper for LAPACKE_clacp2_work */
lapack_int  LAPACKE_clacp2_work (int matrix_layout, char uplo, lapack_int m, lapack_int n, const float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    hook_pos_LAPACKE_clacp2_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clacp2_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clacp2_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, m, n, a, lda, b, ldb);
    } else {
        return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clacp2_work(int matrix_layout, char uplo, lapack_int m, lapack_int n, const float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clacp2_work;
    return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_clacp2_work(int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    hook_pos_LAPACKE_clacp2_work++;
    if ( hook_pos_LAPACKE_clacp2_work < __flexiblas_hooks->LAPACKE_clacp2_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clacp2_work.hook_function[hook_pos_LAPACKE_clacp2_work];
    } else {
        hook_pos_LAPACKE_clacp2_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clacp2_work;
    }
    return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clacpy = 0;
#endif

/* Wrapper for LAPACKE_clacpy */
lapack_int  LAPACKE_clacpy (int matrix_layout, char uplo, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    hook_pos_LAPACKE_clacpy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clacpy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clacpy.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, m, n, a, lda, b, ldb);
    } else {
        return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clacpy(int matrix_layout, char uplo, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clacpy;
    return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_clacpy(int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    hook_pos_LAPACKE_clacpy++;
    if ( hook_pos_LAPACKE_clacpy < __flexiblas_hooks->LAPACKE_clacpy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clacpy.hook_function[hook_pos_LAPACKE_clacpy];
    } else {
        hook_pos_LAPACKE_clacpy = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clacpy;
    }
    return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clacpy_work = 0;
#endif

/* Wrapper for LAPACKE_clacpy_work */
lapack_int  LAPACKE_clacpy_work (int matrix_layout, char uplo, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    hook_pos_LAPACKE_clacpy_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clacpy_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clacpy_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, uplo, m, n, a, lda, b, ldb);
    } else {
        return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
    }
    #else
    return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clacpy_work(int matrix_layout, char uplo, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, lapack_complex_float *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clacpy_work;
    return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
}

lapack_int flexiblas_chain_LAPACKE_clacpy_work(int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb)
{
    lapack_int (*fn) (int matrix_layout, char uplo, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *b, lapack_int ldb);
    hook_pos_LAPACKE_clacpy_work++;
    if ( hook_pos_LAPACKE_clacpy_work < __flexiblas_hooks->LAPACKE_clacpy_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clacpy_work.hook_function[hook_pos_LAPACKE_clacpy_work];
    } else {
        hook_pos_LAPACKE_clacpy_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clacpy_work;
    }
    return fn(matrix_layout, uplo, m, n, a, lda, b, ldb);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clacrm = 0;
#endif

/* Wrapper for LAPACKE_clacrm */
lapack_int  LAPACKE_clacrm (int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, const float *b, lapack_int ldb, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc);
    hook_pos_LAPACKE_clacrm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clacrm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clacrm.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, b, ldb, c, ldc);
    } else {
        return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clacrm(int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, const float *b, lapack_int ldb, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clacrm;
    return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc);
}

lapack_int flexiblas_chain_LAPACKE_clacrm(int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc);
    hook_pos_LAPACKE_clacrm++;
    if ( hook_pos_LAPACKE_clacrm < __flexiblas_hooks->LAPACKE_clacrm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clacrm.hook_function[hook_pos_LAPACKE_clacrm];
    } else {
        hook_pos_LAPACKE_clacrm = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clacrm;
    }
    return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clacrm_work = 0;
#endif

/* Wrapper for LAPACKE_clacrm_work */
lapack_int  LAPACKE_clacrm_work (int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, const float *b, lapack_int ldb, lapack_complex_float *c, lapack_int ldc, float *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc, void *work);
    hook_pos_LAPACKE_clacrm_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clacrm_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clacrm_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, b, ldb, c, ldc, work);
    } else {
        return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc, work);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clacrm_work(int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, const float *b, lapack_int ldb, lapack_complex_float *c, lapack_int ldc, float *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clacrm_work;
    return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc, work);
}

lapack_int flexiblas_chain_LAPACKE_clacrm_work(int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc, void *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc, void *work);
    hook_pos_LAPACKE_clacrm_work++;
    if ( hook_pos_LAPACKE_clacrm_work < __flexiblas_hooks->LAPACKE_clacrm_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clacrm_work.hook_function[hook_pos_LAPACKE_clacrm_work];
    } else {
        hook_pos_LAPACKE_clacrm_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clacrm_work;
    }
    return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clag2z = 0;
#endif

/* Wrapper for LAPACKE_clag2z */
lapack_int  LAPACKE_clag2z (int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *sa, lapack_int ldsa, lapack_complex_double *a, lapack_int lda)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *sa, lapack_int ldsa, void *a, lapack_int lda);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, const void *sa, lapack_int ldsa, void *a, lapack_int lda);
    hook_pos_LAPACKE_clag2z = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clag2z;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clag2z.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, sa, ldsa, a, lda);
    } else {
        return fn(matrix_layout, m, n, sa, ldsa, a, lda);
    }
    #else
    return fn(matrix_layout, m, n, sa, ldsa, a, lda);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clag2z(int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *sa, lapack_int ldsa, lapack_complex_double *a, lapack_int lda)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *sa, lapack_int ldsa, void *a, lapack_int lda);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clag2z;
    return fn(matrix_layout, m, n, sa, ldsa, a, lda);
}

lapack_int flexiblas_chain_LAPACKE_clag2z(int matrix_layout, lapack_int m, lapack_int n, const void *sa, lapack_int ldsa, void *a, lapack_int lda)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *sa, lapack_int ldsa, void *a, lapack_int lda);
    hook_pos_LAPACKE_clag2z++;
    if ( hook_pos_LAPACKE_clag2z < __flexiblas_hooks->LAPACKE_clag2z.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clag2z.hook_function[hook_pos_LAPACKE_clag2z];
    } else {
        hook_pos_LAPACKE_clag2z = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clag2z;
    }
    return fn(matrix_layout, m, n, sa, ldsa, a, lda);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clag2z_work = 0;
#endif

/* Wrapper for LAPACKE_clag2z_work */
lapack_int  LAPACKE_clag2z_work (int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *sa, lapack_int ldsa, lapack_complex_double *a, lapack_int lda)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *sa, lapack_int ldsa, void *a, lapack_int lda);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, const void *sa, lapack_int ldsa, void *a, lapack_int lda);
    hook_pos_LAPACKE_clag2z_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clag2z_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clag2z_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, sa, ldsa, a, lda);
    } else {
        return fn(matrix_layout, m, n, sa, ldsa, a, lda);
    }
    #else
    return fn(matrix_layout, m, n, sa, ldsa, a, lda);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clag2z_work(int matrix_layout, lapack_int m, lapack_int n, const lapack_complex_float *sa, lapack_int ldsa, lapack_complex_double *a, lapack_int lda)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *sa, lapack_int ldsa, void *a, lapack_int lda);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clag2z_work;
    return fn(matrix_layout, m, n, sa, ldsa, a, lda);
}

lapack_int flexiblas_chain_LAPACKE_clag2z_work(int matrix_layout, lapack_int m, lapack_int n, const void *sa, lapack_int ldsa, void *a, lapack_int lda)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *sa, lapack_int ldsa, void *a, lapack_int lda);
    hook_pos_LAPACKE_clag2z_work++;
    if ( hook_pos_LAPACKE_clag2z_work < __flexiblas_hooks->LAPACKE_clag2z_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clag2z_work.hook_function[hook_pos_LAPACKE_clag2z_work];
    } else {
        hook_pos_LAPACKE_clag2z_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clag2z_work;
    }
    return fn(matrix_layout, m, n, sa, ldsa, a, lda);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clange = 0;
#endif

/* Wrapper for LAPACKE_clange */
float  LAPACKE_clange (int matrix_layout, char norm, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda)
{
    float (*fn) (int matrix_layout, char norm, lapack_int m, lapack_int n, const void *a, lapack_int lda);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (int matrix_layout, char norm, lapack_int m, lapack_int n, const void *a, lapack_int lda);
    hook_pos_LAPACKE_clange = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clange;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clange.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, norm, m, n, a, lda);
    } else {
        return fn(matrix_layout, norm, m, n, a, lda);
    }
    #else
    return fn(matrix_layout, norm, m, n, a, lda);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
float  flexiblas_real_LAPACKE_clange(int matrix_layout, char norm, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda)
{
    float (*fn) (int matrix_layout, char norm, lapack_int m, lapack_int n, const void *a, lapack_int lda);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clange;
    return fn(matrix_layout, norm, m, n, a, lda);
}

float flexiblas_chain_LAPACKE_clange(int matrix_layout, char norm, lapack_int m, lapack_int n, const void *a, lapack_int lda)
{
    float (*fn) (int matrix_layout, char norm, lapack_int m, lapack_int n, const void *a, lapack_int lda);
    hook_pos_LAPACKE_clange++;
    if ( hook_pos_LAPACKE_clange < __flexiblas_hooks->LAPACKE_clange.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clange.hook_function[hook_pos_LAPACKE_clange];
    } else {
        hook_pos_LAPACKE_clange = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clange;
    }
    return fn(matrix_layout, norm, m, n, a, lda);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clange_work = 0;
#endif

/* Wrapper for LAPACKE_clange_work */
float  LAPACKE_clange_work (int matrix_layout, char norm, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *work)
{
    float (*fn) (int matrix_layout, char norm, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (int matrix_layout, char norm, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *work);
    hook_pos_LAPACKE_clange_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clange_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clange_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, norm, m, n, a, lda, work);
    } else {
        return fn(matrix_layout, norm, m, n, a, lda, work);
    }
    #else
    return fn(matrix_layout, norm, m, n, a, lda, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
float  flexiblas_real_LAPACKE_clange_work(int matrix_layout, char norm, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *work)
{
    float (*fn) (int matrix_layout, char norm, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clange_work;
    return fn(matrix_layout, norm, m, n, a, lda, work);
}

float flexiblas_chain_LAPACKE_clange_work(int matrix_layout, char norm, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *work)
{
    float (*fn) (int matrix_layout, char norm, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *work);
    hook_pos_LAPACKE_clange_work++;
    if ( hook_pos_LAPACKE_clange_work < __flexiblas_hooks->LAPACKE_clange_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clange_work.hook_function[hook_pos_LAPACKE_clange_work];
    } else {
        hook_pos_LAPACKE_clange_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clange_work;
    }
    return fn(matrix_layout, norm, m, n, a, lda, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clanhe = 0;
#endif

/* Wrapper for LAPACKE_clanhe */
float  LAPACKE_clanhe (int matrix_layout, char norm, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda)
{
    float (*fn) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda);
    hook_pos_LAPACKE_clanhe = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clanhe;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clanhe.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, norm, uplo, n, a, lda);
    } else {
        return fn(matrix_layout, norm, uplo, n, a, lda);
    }
    #else
    return fn(matrix_layout, norm, uplo, n, a, lda);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
float  flexiblas_real_LAPACKE_clanhe(int matrix_layout, char norm, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda)
{
    float (*fn) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clanhe;
    return fn(matrix_layout, norm, uplo, n, a, lda);
}

float flexiblas_chain_LAPACKE_clanhe(int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda)
{
    float (*fn) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda);
    hook_pos_LAPACKE_clanhe++;
    if ( hook_pos_LAPACKE_clanhe < __flexiblas_hooks->LAPACKE_clanhe.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clanhe.hook_function[hook_pos_LAPACKE_clanhe];
    } else {
        hook_pos_LAPACKE_clanhe = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clanhe;
    }
    return fn(matrix_layout, norm, uplo, n, a, lda);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clanhe_work = 0;
#endif

/* Wrapper for LAPACKE_clanhe_work */
float  LAPACKE_clanhe_work (int matrix_layout, char norm, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *work)
{
    float (*fn) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda, void *work);
    hook_pos_LAPACKE_clanhe_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clanhe_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clanhe_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, norm, uplo, n, a, lda, work);
    } else {
        return fn(matrix_layout, norm, uplo, n, a, lda, work);
    }
    #else
    return fn(matrix_layout, norm, uplo, n, a, lda, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
float  flexiblas_real_LAPACKE_clanhe_work(int matrix_layout, char norm, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *work)
{
    float (*fn) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clanhe_work;
    return fn(matrix_layout, norm, uplo, n, a, lda, work);
}

float flexiblas_chain_LAPACKE_clanhe_work(int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda, void *work)
{
    float (*fn) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda, void *work);
    hook_pos_LAPACKE_clanhe_work++;
    if ( hook_pos_LAPACKE_clanhe_work < __flexiblas_hooks->LAPACKE_clanhe_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clanhe_work.hook_function[hook_pos_LAPACKE_clanhe_work];
    } else {
        hook_pos_LAPACKE_clanhe_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clanhe_work;
    }
    return fn(matrix_layout, norm, uplo, n, a, lda, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clansy = 0;
#endif

/* Wrapper for LAPACKE_clansy */
float  LAPACKE_clansy (int matrix_layout, char norm, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda)
{
    float (*fn) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda);
    hook_pos_LAPACKE_clansy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clansy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clansy.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, norm, uplo, n, a, lda);
    } else {
        return fn(matrix_layout, norm, uplo, n, a, lda);
    }
    #else
    return fn(matrix_layout, norm, uplo, n, a, lda);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
float  flexiblas_real_LAPACKE_clansy(int matrix_layout, char norm, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda)
{
    float (*fn) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clansy;
    return fn(matrix_layout, norm, uplo, n, a, lda);
}

float flexiblas_chain_LAPACKE_clansy(int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda)
{
    float (*fn) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda);
    hook_pos_LAPACKE_clansy++;
    if ( hook_pos_LAPACKE_clansy < __flexiblas_hooks->LAPACKE_clansy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clansy.hook_function[hook_pos_LAPACKE_clansy];
    } else {
        hook_pos_LAPACKE_clansy = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clansy;
    }
    return fn(matrix_layout, norm, uplo, n, a, lda);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clansy_work = 0;
#endif

/* Wrapper for LAPACKE_clansy_work */
float  LAPACKE_clansy_work (int matrix_layout, char norm, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *work)
{
    float (*fn) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda, void *work);
    hook_pos_LAPACKE_clansy_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clansy_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clansy_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, norm, uplo, n, a, lda, work);
    } else {
        return fn(matrix_layout, norm, uplo, n, a, lda, work);
    }
    #else
    return fn(matrix_layout, norm, uplo, n, a, lda, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
float  flexiblas_real_LAPACKE_clansy_work(int matrix_layout, char norm, char uplo, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *work)
{
    float (*fn) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clansy_work;
    return fn(matrix_layout, norm, uplo, n, a, lda, work);
}

float flexiblas_chain_LAPACKE_clansy_work(int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda, void *work)
{
    float (*fn) (int matrix_layout, char norm, char uplo, lapack_int n, const void *a, lapack_int lda, void *work);
    hook_pos_LAPACKE_clansy_work++;
    if ( hook_pos_LAPACKE_clansy_work < __flexiblas_hooks->LAPACKE_clansy_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clansy_work.hook_function[hook_pos_LAPACKE_clansy_work];
    } else {
        hook_pos_LAPACKE_clansy_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clansy_work;
    }
    return fn(matrix_layout, norm, uplo, n, a, lda, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clantr = 0;
#endif

/* Wrapper for LAPACKE_clantr */
float  LAPACKE_clantr (int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda)
{
    float (*fn) (int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const void *a, lapack_int lda);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const void *a, lapack_int lda);
    hook_pos_LAPACKE_clantr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clantr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clantr.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, norm, uplo, diag, m, n, a, lda);
    } else {
        return fn(matrix_layout, norm, uplo, diag, m, n, a, lda);
    }
    #else
    return fn(matrix_layout, norm, uplo, diag, m, n, a, lda);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
float  flexiblas_real_LAPACKE_clantr(int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda)
{
    float (*fn) (int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const void *a, lapack_int lda);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clantr;
    return fn(matrix_layout, norm, uplo, diag, m, n, a, lda);
}

float flexiblas_chain_LAPACKE_clantr(int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const void *a, lapack_int lda)
{
    float (*fn) (int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const void *a, lapack_int lda);
    hook_pos_LAPACKE_clantr++;
    if ( hook_pos_LAPACKE_clantr < __flexiblas_hooks->LAPACKE_clantr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clantr.hook_function[hook_pos_LAPACKE_clantr];
    } else {
        hook_pos_LAPACKE_clantr = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clantr;
    }
    return fn(matrix_layout, norm, uplo, diag, m, n, a, lda);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clantr_work = 0;
#endif

/* Wrapper for LAPACKE_clantr_work */
float  LAPACKE_clantr_work (int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *work)
{
    float (*fn) (int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *work);
    hook_pos_LAPACKE_clantr_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clantr_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clantr_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, norm, uplo, diag, m, n, a, lda, work);
    } else {
        return fn(matrix_layout, norm, uplo, diag, m, n, a, lda, work);
    }
    #else
    return fn(matrix_layout, norm, uplo, diag, m, n, a, lda, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
float  flexiblas_real_LAPACKE_clantr_work(int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const lapack_complex_float *a, lapack_int lda, float *work)
{
    float (*fn) (int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clantr_work;
    return fn(matrix_layout, norm, uplo, diag, m, n, a, lda, work);
}

float flexiblas_chain_LAPACKE_clantr_work(int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *work)
{
    float (*fn) (int matrix_layout, char norm, char uplo, char diag, lapack_int m, lapack_int n, const void *a, lapack_int lda, void *work);
    hook_pos_LAPACKE_clantr_work++;
    if ( hook_pos_LAPACKE_clantr_work < __flexiblas_hooks->LAPACKE_clantr_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clantr_work.hook_function[hook_pos_LAPACKE_clantr_work];
    } else {
        hook_pos_LAPACKE_clantr_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clantr_work;
    }
    return fn(matrix_layout, norm, uplo, diag, m, n, a, lda, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clapmr = 0;
#endif

/* Wrapper for LAPACKE_clapmr */
lapack_int  LAPACKE_clapmr (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, lapack_complex_float *x, lapack_int ldx, lapack_int *k)
{
    lapack_int (*fn) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    hook_pos_LAPACKE_clapmr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clapmr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clapmr.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, forwrd, m, n, x, ldx, k);
    } else {
        return fn(matrix_layout, forwrd, m, n, x, ldx, k);
    }
    #else
    return fn(matrix_layout, forwrd, m, n, x, ldx, k);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clapmr(int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, lapack_complex_float *x, lapack_int ldx, lapack_int *k)
{
    lapack_int (*fn) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clapmr;
    return fn(matrix_layout, forwrd, m, n, x, ldx, k);
}

lapack_int flexiblas_chain_LAPACKE_clapmr(int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k)
{
    lapack_int (*fn) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    hook_pos_LAPACKE_clapmr++;
    if ( hook_pos_LAPACKE_clapmr < __flexiblas_hooks->LAPACKE_clapmr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clapmr.hook_function[hook_pos_LAPACKE_clapmr];
    } else {
        hook_pos_LAPACKE_clapmr = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clapmr;
    }
    return fn(matrix_layout, forwrd, m, n, x, ldx, k);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clapmr_work = 0;
#endif

/* Wrapper for LAPACKE_clapmr_work */
lapack_int  LAPACKE_clapmr_work (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, lapack_complex_float *x, lapack_int ldx, lapack_int *k)
{
    lapack_int (*fn) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    hook_pos_LAPACKE_clapmr_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clapmr_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clapmr_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, forwrd, m, n, x, ldx, k);
    } else {
        return fn(matrix_layout, forwrd, m, n, x, ldx, k);
    }
    #else
    return fn(matrix_layout, forwrd, m, n, x, ldx, k);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clapmr_work(int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, lapack_complex_float *x, lapack_int ldx, lapack_int *k)
{
    lapack_int (*fn) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clapmr_work;
    return fn(matrix_layout, forwrd, m, n, x, ldx, k);
}

lapack_int flexiblas_chain_LAPACKE_clapmr_work(int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k)
{
    lapack_int (*fn) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    hook_pos_LAPACKE_clapmr_work++;
    if ( hook_pos_LAPACKE_clapmr_work < __flexiblas_hooks->LAPACKE_clapmr_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clapmr_work.hook_function[hook_pos_LAPACKE_clapmr_work];
    } else {
        hook_pos_LAPACKE_clapmr_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clapmr_work;
    }
    return fn(matrix_layout, forwrd, m, n, x, ldx, k);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clapmt = 0;
#endif

/* Wrapper for LAPACKE_clapmt */
lapack_int  LAPACKE_clapmt (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, lapack_complex_float *x, lapack_int ldx, lapack_int *k)
{
    lapack_int (*fn) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    hook_pos_LAPACKE_clapmt = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clapmt;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clapmt.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, forwrd, m, n, x, ldx, k);
    } else {
        return fn(matrix_layout, forwrd, m, n, x, ldx, k);
    }
    #else
    return fn(matrix_layout, forwrd, m, n, x, ldx, k);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clapmt(int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, lapack_complex_float *x, lapack_int ldx, lapack_int *k)
{
    lapack_int (*fn) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clapmt;
    return fn(matrix_layout, forwrd, m, n, x, ldx, k);
}

lapack_int flexiblas_chain_LAPACKE_clapmt(int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k)
{
    lapack_int (*fn) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    hook_pos_LAPACKE_clapmt++;
    if ( hook_pos_LAPACKE_clapmt < __flexiblas_hooks->LAPACKE_clapmt.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clapmt.hook_function[hook_pos_LAPACKE_clapmt];
    } else {
        hook_pos_LAPACKE_clapmt = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clapmt;
    }
    return fn(matrix_layout, forwrd, m, n, x, ldx, k);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clapmt_work = 0;
#endif

/* Wrapper for LAPACKE_clapmt_work */
lapack_int  LAPACKE_clapmt_work (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, lapack_complex_float *x, lapack_int ldx, lapack_int *k)
{
    lapack_int (*fn) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    hook_pos_LAPACKE_clapmt_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clapmt_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clapmt_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, forwrd, m, n, x, ldx, k);
    } else {
        return fn(matrix_layout, forwrd, m, n, x, ldx, k);
    }
    #else
    return fn(matrix_layout, forwrd, m, n, x, ldx, k);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clapmt_work(int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, lapack_complex_float *x, lapack_int ldx, lapack_int *k)
{
    lapack_int (*fn) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clapmt_work;
    return fn(matrix_layout, forwrd, m, n, x, ldx, k);
}

lapack_int flexiblas_chain_LAPACKE_clapmt_work(int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k)
{
    lapack_int (*fn) (int matrix_layout, lapack_int forwrd, lapack_int m, lapack_int n, void *x, lapack_int ldx, void *k);
    hook_pos_LAPACKE_clapmt_work++;
    if ( hook_pos_LAPACKE_clapmt_work < __flexiblas_hooks->LAPACKE_clapmt_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clapmt_work.hook_function[hook_pos_LAPACKE_clapmt_work];
    } else {
        hook_pos_LAPACKE_clapmt_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clapmt_work;
    }
    return fn(matrix_layout, forwrd, m, n, x, ldx, k);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clarcm = 0;
#endif

/* Wrapper for LAPACKE_clarcm */
lapack_int  LAPACKE_clarcm (int matrix_layout, lapack_int m, lapack_int n, const float *a, lapack_int lda, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc);
    hook_pos_LAPACKE_clarcm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clarcm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clarcm.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, b, ldb, c, ldc);
    } else {
        return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clarcm(int matrix_layout, lapack_int m, lapack_int n, const float *a, lapack_int lda, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clarcm;
    return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc);
}

lapack_int flexiblas_chain_LAPACKE_clarcm(int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc);
    hook_pos_LAPACKE_clarcm++;
    if ( hook_pos_LAPACKE_clarcm < __flexiblas_hooks->LAPACKE_clarcm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clarcm.hook_function[hook_pos_LAPACKE_clarcm];
    } else {
        hook_pos_LAPACKE_clarcm = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clarcm;
    }
    return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clarcm_work = 0;
#endif

/* Wrapper for LAPACKE_clarcm_work */
lapack_int  LAPACKE_clarcm_work (int matrix_layout, lapack_int m, lapack_int n, const float *a, lapack_int lda, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *c, lapack_int ldc, float *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc, void *work);
    hook_pos_LAPACKE_clarcm_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clarcm_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clarcm_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, m, n, a, lda, b, ldb, c, ldc, work);
    } else {
        return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc, work);
    }
    #else
    return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clarcm_work(int matrix_layout, lapack_int m, lapack_int n, const float *a, lapack_int lda, const lapack_complex_float *b, lapack_int ldb, lapack_complex_float *c, lapack_int ldc, float *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clarcm_work;
    return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc, work);
}

lapack_int flexiblas_chain_LAPACKE_clarcm_work(int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc, void *work)
{
    lapack_int (*fn) (int matrix_layout, lapack_int m, lapack_int n, const void *a, lapack_int lda, const void *b, lapack_int ldb, void *c, lapack_int ldc, void *work);
    hook_pos_LAPACKE_clarcm_work++;
    if ( hook_pos_LAPACKE_clarcm_work < __flexiblas_hooks->LAPACKE_clarcm_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clarcm_work.hook_function[hook_pos_LAPACKE_clarcm_work];
    } else {
        hook_pos_LAPACKE_clarcm_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clarcm_work;
    }
    return fn(matrix_layout, m, n, a, lda, b, ldb, c, ldc, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clarfb = 0;
#endif

/* Wrapper for LAPACKE_clarfb */
lapack_int  LAPACKE_clarfb (int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const lapack_complex_float *v, lapack_int ldv, const lapack_complex_float *t, lapack_int ldt, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc);
    hook_pos_LAPACKE_clarfb = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clarfb;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clarfb.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, side, trans, direct, storev, m, n, k, v, ldv, t, ldt, c, ldc);
    } else {
        return fn(matrix_layout, side, trans, direct, storev, m, n, k, v, ldv, t, ldt, c, ldc);
    }
    #else
    return fn(matrix_layout, side, trans, direct, storev, m, n, k, v, ldv, t, ldt, c, ldc);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clarfb(int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const lapack_complex_float *v, lapack_int ldv, const lapack_complex_float *t, lapack_int ldt, lapack_complex_float *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clarfb;
    return fn(matrix_layout, side, trans, direct, storev, m, n, k, v, ldv, t, ldt, c, ldc);
}

lapack_int flexiblas_chain_LAPACKE_clarfb(int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc);
    hook_pos_LAPACKE_clarfb++;
    if ( hook_pos_LAPACKE_clarfb < __flexiblas_hooks->LAPACKE_clarfb.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clarfb.hook_function[hook_pos_LAPACKE_clarfb];
    } else {
        hook_pos_LAPACKE_clarfb = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clarfb;
    }
    return fn(matrix_layout, side, trans, direct, storev, m, n, k, v, ldv, t, ldt, c, ldc);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clarfb_work = 0;
#endif

/* Wrapper for LAPACKE_clarfb_work */
lapack_int  LAPACKE_clarfb_work (int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const lapack_complex_float *v, lapack_int ldv, const lapack_complex_float *t, lapack_int ldt, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work, lapack_int ldwork)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc, void *work, lapack_int ldwork);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc, void *work, lapack_int ldwork);
    hook_pos_LAPACKE_clarfb_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clarfb_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clarfb_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, side, trans, direct, storev, m, n, k, v, ldv, t, ldt, c, ldc, work, ldwork);
    } else {
        return fn(matrix_layout, side, trans, direct, storev, m, n, k, v, ldv, t, ldt, c, ldc, work, ldwork);
    }
    #else
    return fn(matrix_layout, side, trans, direct, storev, m, n, k, v, ldv, t, ldt, c, ldc, work, ldwork);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clarfb_work(int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const lapack_complex_float *v, lapack_int ldv, const lapack_complex_float *t, lapack_int ldt, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work, lapack_int ldwork)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc, void *work, lapack_int ldwork);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clarfb_work;
    return fn(matrix_layout, side, trans, direct, storev, m, n, k, v, ldv, t, ldt, c, ldc, work, ldwork);
}

lapack_int flexiblas_chain_LAPACKE_clarfb_work(int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc, void *work, lapack_int ldwork)
{
    lapack_int (*fn) (int matrix_layout, char side, char trans, char direct, char storev, lapack_int m, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *t, lapack_int ldt, void *c, lapack_int ldc, void *work, lapack_int ldwork);
    hook_pos_LAPACKE_clarfb_work++;
    if ( hook_pos_LAPACKE_clarfb_work < __flexiblas_hooks->LAPACKE_clarfb_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clarfb_work.hook_function[hook_pos_LAPACKE_clarfb_work];
    } else {
        hook_pos_LAPACKE_clarfb_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clarfb_work;
    }
    return fn(matrix_layout, side, trans, direct, storev, m, n, k, v, ldv, t, ldt, c, ldc, work, ldwork);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clarfg = 0;
#endif

/* Wrapper for LAPACKE_clarfg */
lapack_int  LAPACKE_clarfg (lapack_int n, lapack_complex_float *alpha, lapack_complex_float *x, lapack_int incx, lapack_complex_float *tau)
{
    lapack_int (*fn) (lapack_int n, void *alpha, void *x, lapack_int incx, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (lapack_int n, void *alpha, void *x, lapack_int incx, void *tau);
    hook_pos_LAPACKE_clarfg = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clarfg;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clarfg.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(n, alpha, x, incx, tau);
    } else {
        return fn(n, alpha, x, incx, tau);
    }
    #else
    return fn(n, alpha, x, incx, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clarfg(lapack_int n, lapack_complex_float *alpha, lapack_complex_float *x, lapack_int incx, lapack_complex_float *tau)
{
    lapack_int (*fn) (lapack_int n, void *alpha, void *x, lapack_int incx, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clarfg;
    return fn(n, alpha, x, incx, tau);
}

lapack_int flexiblas_chain_LAPACKE_clarfg(lapack_int n, void *alpha, void *x, lapack_int incx, void *tau)
{
    lapack_int (*fn) (lapack_int n, void *alpha, void *x, lapack_int incx, void *tau);
    hook_pos_LAPACKE_clarfg++;
    if ( hook_pos_LAPACKE_clarfg < __flexiblas_hooks->LAPACKE_clarfg.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clarfg.hook_function[hook_pos_LAPACKE_clarfg];
    } else {
        hook_pos_LAPACKE_clarfg = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clarfg;
    }
    return fn(n, alpha, x, incx, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clarfg_work = 0;
#endif

/* Wrapper for LAPACKE_clarfg_work */
lapack_int  LAPACKE_clarfg_work (lapack_int n, lapack_complex_float *alpha, lapack_complex_float *x, lapack_int incx, lapack_complex_float *tau)
{
    lapack_int (*fn) (lapack_int n, void *alpha, void *x, lapack_int incx, void *tau);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (lapack_int n, void *alpha, void *x, lapack_int incx, void *tau);
    hook_pos_LAPACKE_clarfg_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clarfg_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clarfg_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(n, alpha, x, incx, tau);
    } else {
        return fn(n, alpha, x, incx, tau);
    }
    #else
    return fn(n, alpha, x, incx, tau);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clarfg_work(lapack_int n, lapack_complex_float *alpha, lapack_complex_float *x, lapack_int incx, lapack_complex_float *tau)
{
    lapack_int (*fn) (lapack_int n, void *alpha, void *x, lapack_int incx, void *tau);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clarfg_work;
    return fn(n, alpha, x, incx, tau);
}

lapack_int flexiblas_chain_LAPACKE_clarfg_work(lapack_int n, void *alpha, void *x, lapack_int incx, void *tau)
{
    lapack_int (*fn) (lapack_int n, void *alpha, void *x, lapack_int incx, void *tau);
    hook_pos_LAPACKE_clarfg_work++;
    if ( hook_pos_LAPACKE_clarfg_work < __flexiblas_hooks->LAPACKE_clarfg_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clarfg_work.hook_function[hook_pos_LAPACKE_clarfg_work];
    } else {
        hook_pos_LAPACKE_clarfg_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clarfg_work;
    }
    return fn(n, alpha, x, incx, tau);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clarft = 0;
#endif

/* Wrapper for LAPACKE_clarft */
lapack_int  LAPACKE_clarft (int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const lapack_complex_float *v, lapack_int ldv, const lapack_complex_float *tau, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *tau, void *t, lapack_int ldt);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *tau, void *t, lapack_int ldt);
    hook_pos_LAPACKE_clarft = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clarft;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clarft.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, direct, storev, n, k, v, ldv, tau, t, ldt);
    } else {
        return fn(matrix_layout, direct, storev, n, k, v, ldv, tau, t, ldt);
    }
    #else
    return fn(matrix_layout, direct, storev, n, k, v, ldv, tau, t, ldt);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clarft(int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const lapack_complex_float *v, lapack_int ldv, const lapack_complex_float *tau, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *tau, void *t, lapack_int ldt);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clarft;
    return fn(matrix_layout, direct, storev, n, k, v, ldv, tau, t, ldt);
}

lapack_int flexiblas_chain_LAPACKE_clarft(int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *tau, void *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *tau, void *t, lapack_int ldt);
    hook_pos_LAPACKE_clarft++;
    if ( hook_pos_LAPACKE_clarft < __flexiblas_hooks->LAPACKE_clarft.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clarft.hook_function[hook_pos_LAPACKE_clarft];
    } else {
        hook_pos_LAPACKE_clarft = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clarft;
    }
    return fn(matrix_layout, direct, storev, n, k, v, ldv, tau, t, ldt);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clarft_work = 0;
#endif

/* Wrapper for LAPACKE_clarft_work */
lapack_int  LAPACKE_clarft_work (int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const lapack_complex_float *v, lapack_int ldv, const lapack_complex_float *tau, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *tau, void *t, lapack_int ldt);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *tau, void *t, lapack_int ldt);
    hook_pos_LAPACKE_clarft_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clarft_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clarft_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, direct, storev, n, k, v, ldv, tau, t, ldt);
    } else {
        return fn(matrix_layout, direct, storev, n, k, v, ldv, tau, t, ldt);
    }
    #else
    return fn(matrix_layout, direct, storev, n, k, v, ldv, tau, t, ldt);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clarft_work(int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const lapack_complex_float *v, lapack_int ldv, const lapack_complex_float *tau, lapack_complex_float *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *tau, void *t, lapack_int ldt);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clarft_work;
    return fn(matrix_layout, direct, storev, n, k, v, ldv, tau, t, ldt);
}

lapack_int flexiblas_chain_LAPACKE_clarft_work(int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *tau, void *t, lapack_int ldt)
{
    lapack_int (*fn) (int matrix_layout, char direct, char storev, lapack_int n, lapack_int k, const void *v, lapack_int ldv, const void *tau, void *t, lapack_int ldt);
    hook_pos_LAPACKE_clarft_work++;
    if ( hook_pos_LAPACKE_clarft_work < __flexiblas_hooks->LAPACKE_clarft_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clarft_work.hook_function[hook_pos_LAPACKE_clarft_work];
    } else {
        hook_pos_LAPACKE_clarft_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clarft_work;
    }
    return fn(matrix_layout, direct, storev, n, k, v, ldv, tau, t, ldt);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clarfx = 0;
#endif

/* Wrapper for LAPACKE_clarfx */
lapack_int  LAPACKE_clarfx (int matrix_layout, char side, lapack_int m, lapack_int n, const lapack_complex_float *v, lapack_complex_float tau, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char side, lapack_int m, lapack_int n, const void *v, float _Complex tau, void *c, lapack_int ldc, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char side, lapack_int m, lapack_int n, const void *v, float _Complex tau, void *c, lapack_int ldc, void *work);
    hook_pos_LAPACKE_clarfx = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clarfx;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clarfx.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, side, m, n, v, tau, c, ldc, work);
    } else {
        return fn(matrix_layout, side, m, n, v, tau, c, ldc, work);
    }
    #else
    return fn(matrix_layout, side, m, n, v, tau, c, ldc, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clarfx(int matrix_layout, char side, lapack_int m, lapack_int n, const lapack_complex_float *v, lapack_complex_float tau, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char side, lapack_int m, lapack_int n, const void *v, float _Complex tau, void *c, lapack_int ldc, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clarfx;
    return fn(matrix_layout, side, m, n, v, tau, c, ldc, work);
}

lapack_int flexiblas_chain_LAPACKE_clarfx(int matrix_layout, char side, lapack_int m, lapack_int n, const void *v, float _Complex tau, void *c, lapack_int ldc, void *work)
{
    lapack_int (*fn) (int matrix_layout, char side, lapack_int m, lapack_int n, const void *v, float _Complex tau, void *c, lapack_int ldc, void *work);
    hook_pos_LAPACKE_clarfx++;
    if ( hook_pos_LAPACKE_clarfx < __flexiblas_hooks->LAPACKE_clarfx.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clarfx.hook_function[hook_pos_LAPACKE_clarfx];
    } else {
        hook_pos_LAPACKE_clarfx = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clarfx;
    }
    return fn(matrix_layout, side, m, n, v, tau, c, ldc, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clarfx_work = 0;
#endif

/* Wrapper for LAPACKE_clarfx_work */
lapack_int  LAPACKE_clarfx_work (int matrix_layout, char side, lapack_int m, lapack_int n, const lapack_complex_float *v, lapack_complex_float tau, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char side, lapack_int m, lapack_int n, const void *v, float _Complex tau, void *c, lapack_int ldc, void *work);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (int matrix_layout, char side, lapack_int m, lapack_int n, const void *v, float _Complex tau, void *c, lapack_int ldc, void *work);
    hook_pos_LAPACKE_clarfx_work = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clarfx_work;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clarfx_work.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(matrix_layout, side, m, n, v, tau, c, ldc, work);
    } else {
        return fn(matrix_layout, side, m, n, v, tau, c, ldc, work);
    }
    #else
    return fn(matrix_layout, side, m, n, v, tau, c, ldc, work);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clarfx_work(int matrix_layout, char side, lapack_int m, lapack_int n, const lapack_complex_float *v, lapack_complex_float tau, lapack_complex_float *c, lapack_int ldc, lapack_complex_float *work)
{
    lapack_int (*fn) (int matrix_layout, char side, lapack_int m, lapack_int n, const void *v, float _Complex tau, void *c, lapack_int ldc, void *work);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clarfx_work;
    return fn(matrix_layout, side, m, n, v, tau, c, ldc, work);
}

lapack_int flexiblas_chain_LAPACKE_clarfx_work(int matrix_layout, char side, lapack_int m, lapack_int n, const void *v, float _Complex tau, void *c, lapack_int ldc, void *work)
{
    lapack_int (*fn) (int matrix_layout, char side, lapack_int m, lapack_int n, const void *v, float _Complex tau, void *c, lapack_int ldc, void *work);
    hook_pos_LAPACKE_clarfx_work++;
    if ( hook_pos_LAPACKE_clarfx_work < __flexiblas_hooks->LAPACKE_clarfx_work.nhook ) {
        *(void **) &fn = __flexiblas_hooks->LAPACKE_clarfx_work.hook_function[hook_pos_LAPACKE_clarfx_work];
    } else {
        hook_pos_LAPACKE_clarfx_work = 0;
        *(void **) &fn = current_backend->lapacke.LAPACKE_clarfx_work;
    }
    return fn(matrix_layout, side, m, n, v, tau, c, ldc, work);
}

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_LAPACKE_clarnv = 0;
#endif

/* Wrapper for LAPACKE_clarnv */
lapack_int  LAPACKE_clarnv (lapack_int idist, lapack_int *iseed, lapack_int n, lapack_complex_float *x)
{
    lapack_int (*fn) (lapack_int idist, void *iseed, lapack_int n, void *x);
    #ifdef FLEXIBLAS_HOOK_API
    lapack_int (*fn_hook) (lapack_int idist, void *iseed, lapack_int n, void *x);
    hook_pos_LAPACKE_clarnv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->lapacke.LAPACKE_clarnv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->LAPACKE_clarnv.hook_function[0];

    if ( fn_hook != NULL ) {
        return fn_hook(idist, iseed, n, x);
    } else {
        return fn(idist, iseed, n, x);
    }
    #else
    return fn(idist, iseed, n, x);
    #endif
}

#ifdef FLEXIBLAS_HOOK_API
lapack_int  flexiblas_real_LAPACKE_clarnv(lapack_int idist, lapack_int *iseed, lapack_int n, lapack_complex_float *x)
{
    lapack_int (*fn) (lapack_int idist, void *iseed, lapack_int n, void *x);
    *(void **) &fn = current_backend->lapacke.LAPACKE_clarnv;
    return fn(idist, iseed, n, x);
}

lapack_int flexiblas_chain_LAPACKE_clarnv(lapack_int idist, void *iseed, lapack_int n, void *x)
{
    lapack_int (*fn) (lapack_int idist, void *iseed, lapack_int n, void *x);
    hook_pos_LAPACKE_clarnv++;
    if ( hook_pos_LAPACKE_clarnv < __flexib