"use strict";
/*********************************************************************
 * Copyright (c) 2023 Kichwa Coders Canada Inc. and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const utils_1 = require("./utils");
const chai_1 = require("chai");
/**
 * To test that cwd is set properly we remove the compilation directory from the executable,
 * see the makefile for that part, and then launch with a variety of executable/cwd locations
 * to make sure that we can insert breakpoints when we expect to, and cannot insert breakpoints
 * when we force gdb not to be able to find the source
 */
describe('gdb cwd', function () {
    let dc;
    const program = path.join(utils_1.testProgramsDir, 'cwd.exe');
    const programRelocated = path.join(utils_1.testProgramsDir, 'Debug', 'cwd.exe');
    const src = path.join(utils_1.testProgramsDir, 'cwd.c');
    const lineTags = {
        'STOP HERE': 0,
    };
    before(function () {
        (0, utils_1.resolveLineTagLocations)(src, lineTags);
    });
    beforeEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            dc = yield (0, utils_1.standardBeforeEach)();
        });
    });
    afterEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.stop();
        });
    });
    it('default cwd finds source in program directory', function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.launchRequest((0, utils_1.fillDefaults)(this.test, {
                program: program,
            }));
            const bps = yield dc.setBreakpointsRequest({
                lines: [lineTags['STOP HERE']],
                breakpoints: [{ line: lineTags['STOP HERE'], column: 1 }],
                source: { path: src },
            });
            (0, chai_1.expect)(bps.body.breakpoints[0].verified).to.eq(true);
        });
    });
    it('explicit cwd finds source in program directory', function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.launchRequest((0, utils_1.fillDefaults)(this.test, {
                program: program,
                cwd: utils_1.testProgramsDir,
            }));
            const bps = yield dc.setBreakpointsRequest({
                lines: [lineTags['STOP HERE']],
                breakpoints: [{ line: lineTags['STOP HERE'], column: 1 }],
                source: { path: src },
            });
            (0, chai_1.expect)(bps.body.breakpoints[0].verified).to.eq(true);
        });
    });
    it('default cwd does not find source with relocated program', function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.launchRequest((0, utils_1.fillDefaults)(this.test, {
                program: programRelocated,
            }));
            const bps = yield dc.setBreakpointsRequest({
                lines: [lineTags['STOP HERE']],
                breakpoints: [{ line: lineTags['STOP HERE'], column: 1 }],
                source: { path: src },
            });
            (0, chai_1.expect)(bps.body.breakpoints[0].verified).to.eq(false);
        });
    });
    it('explicitly incorrect cwd does not finds source with relocated program', function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.launchRequest((0, utils_1.fillDefaults)(this.test, {
                program: programRelocated,
                cwd: path.join(utils_1.testProgramsDir, 'EmptyDir'),
            }));
            const bps = yield dc.setBreakpointsRequest({
                lines: [lineTags['STOP HERE']],
                breakpoints: [{ line: lineTags['STOP HERE'], column: 1 }],
                source: { path: src },
            });
            (0, chai_1.expect)(bps.body.breakpoints[0].verified).to.eq(false);
        });
    });
    it('explicitly correct cwd does find source with relocated program', function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.launchRequest((0, utils_1.fillDefaults)(this.test, {
                program: programRelocated,
                cwd: utils_1.testProgramsDir,
            }));
            const bps = yield dc.setBreakpointsRequest({
                lines: [lineTags['STOP HERE']],
                breakpoints: [{ line: lineTags['STOP HERE'], column: 1 }],
                source: { path: src },
            });
            (0, chai_1.expect)(bps.body.breakpoints[0].verified).to.eq(true);
        });
    });
});
//# sourceMappingURL=gdbCwd.spec.js.map