"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.DebugClient = void 0;
const fs = require("fs");
const constants = require("constants");
const cp = require("child_process");
const assert = require("assert");
const net = require("net");
const protocolClient_1 = require("./protocolClient");
class DebugClient extends protocolClient_1.ProtocolClient {
    /**
     * Creates a DebugClient object that provides a promise-based API to write
     * debug adapter tests.
     * A simple mocha example for setting and hitting a breakpoint in line 15 of a program 'test.js' looks like this:
     *
     * var dc;
     * setup( () => {
     *     dc = new DebugClient('node', './out/node/nodeDebug.js', 'node');
     *     return dc.start();
     * });
     * teardown( () => dc.stop() );
     *
     * test('should stop on a breakpoint', () => {
     *     return dc.hitBreakpoint({ program: 'test.js' }, 'test.js', 15);
     * });
     */
    constructor(debugAdapterRuntime, debugAdapterExecutable, debugType, spawnOptions, enableStderr) {
        super();
        this.defaultTimeout = 5000;
        this._runtime = debugAdapterRuntime;
        this._executable = debugAdapterExecutable;
        this._spawnOptions = spawnOptions;
        this._enableStderr = enableStderr;
        this._debugType = debugType;
        this._supportsConfigurationDoneRequest = false;
        if (DebugClient.CASE_INSENSITIVE_FILESYSTEM === undefined) {
            try {
                fs.accessSync(process.execPath.toLowerCase(), constants.F_OK);
                fs.accessSync(process.execPath.toUpperCase(), constants.F_OK);
                DebugClient.CASE_INSENSITIVE_FILESYSTEM = true;
            }
            catch (err) {
                DebugClient.CASE_INSENSITIVE_FILESYSTEM = false;
            }
        }
    }
    // ---- life cycle --------------------------------------------------------------------------------------------------------
    /**
     * Starts a new debug adapter and sets up communication via stdin/stdout.
     * If a port number is specified the adapter is not launched but a connection to
     * a debug adapter running in server mode is established. This is useful for debugging
     * the adapter while running tests. For this reason all timeouts are disabled in server mode.
     */
    start(port) {
        return new Promise((resolve, reject) => {
            if (typeof port === 'number') {
                this._socket = net.createConnection(port, '127.0.0.1', () => {
                    this.connect(this._socket, this._socket);
                    resolve();
                });
            }
            else {
                this._adapterProcess = cp.spawn(this._runtime, [this._executable], this._spawnOptions);
                const sanitize = (s) => s.toString().replace(/\r?\n$/mg, '');
                this._adapterProcess.stderr.on('data', (data) => {
                    if (this._enableStderr) {
                        console.log(sanitize(data));
                    }
                });
                this._adapterProcess.on('error', (err) => {
                    console.log(err);
                    reject(err);
                });
                this._adapterProcess.on('exit', (code, signal) => {
                    if (code) {
                        // done(new Error('debug adapter exit code: ' + code));
                    }
                });
                this.connect(this._adapterProcess.stdout, this._adapterProcess.stdin);
                resolve();
            }
        });
    }
    /**
     * Shutdown the debuggee and the debug adapter (or disconnect if in server mode).
     */
    stop() {
        return this.disconnectRequest().then(() => {
            this.stopAdapter();
        }).catch(() => {
            this.stopAdapter();
        });
    }
    stopAdapter() {
        if (this._adapterProcess) {
            this._adapterProcess.kill();
            this._adapterProcess = null;
        }
        if (this._socket) {
            this._socket.end();
            this._socket = null;
        }
    }
    // ---- protocol requests -------------------------------------------------------------------------------------------------
    initializeRequest(args) {
        if (!args) {
            args = {
                adapterID: this._debugType,
                linesStartAt1: true,
                columnsStartAt1: true,
                pathFormat: 'path'
            };
        }
        return this.send('initialize', args);
    }
    configurationDoneRequest(args) {
        return this.send('configurationDone', args);
    }
    launchRequest(args) {
        return this.send('launch', args);
    }
    attachRequest(args) {
        return this.send('attach', args);
    }
    restartRequest(args) {
        return this.send('restart', args);
    }
    terminateRequest(args) {
        return this.send('terminate', args);
    }
    disconnectRequest(args) {
        return this.send('disconnect', args);
    }
    setBreakpointsRequest(args) {
        return this.send('setBreakpoints', args);
    }
    setFunctionBreakpointsRequest(args) {
        return this.send('setFunctionBreakpoints', args);
    }
    setExceptionBreakpointsRequest(args) {
        return this.send('setExceptionBreakpoints', args);
    }
    dataBreakpointInfoRequest(args) {
        return this.send('dataBreakpointInfo', args);
    }
    setDataBreakpointsRequest(args) {
        return this.send('setDataBreakpoints', args);
    }
    continueRequest(args) {
        return this.send('continue', args);
    }
    nextRequest(args) {
        return this.send('next', args);
    }
    stepInRequest(args) {
        return this.send('stepIn', args);
    }
    stepOutRequest(args) {
        return this.send('stepOut', args);
    }
    stepBackRequest(args) {
        return this.send('stepBack', args);
    }
    reverseContinueRequest(args) {
        return this.send('reverseContinue', args);
    }
    restartFrameRequest(args) {
        return this.send('restartFrame', args);
    }
    gotoRequest(args) {
        return this.send('goto', args);
    }
    pauseRequest(args) {
        return this.send('pause', args);
    }
    stackTraceRequest(args) {
        return this.send('stackTrace', args);
    }
    scopesRequest(args) {
        return this.send('scopes', args);
    }
    variablesRequest(args) {
        return this.send('variables', args);
    }
    setVariableRequest(args) {
        return this.send('setVariable', args);
    }
    sourceRequest(args) {
        return this.send('source', args);
    }
    threadsRequest() {
        return this.send('threads');
    }
    modulesRequest(args) {
        return this.send('modules');
    }
    evaluateRequest(args) {
        return this.send('evaluate', args);
    }
    disassembleRequest(args) {
        return this.send('disassemble', args);
    }
    stepInTargetsRequest(args) {
        return this.send('stepInTargets', args);
    }
    gotoTargetsRequest(args) {
        return this.send('gotoTargets', args);
    }
    completionsRequest(args) {
        return this.send('completions', args);
    }
    exceptionInfoRequest(args) {
        return this.send('exceptionInfo', args);
    }
    customRequest(command, args) {
        return this.send(command, args);
    }
    // ---- convenience methods -----------------------------------------------------------------------------------------------
    /*
     * Returns a promise that will resolve if an event with a specific type was received within some specified time.
     * The promise will be rejected if a timeout occurs.
     */
    waitForEvent(eventType, timeout) {
        timeout = timeout || this.defaultTimeout;
        let timeoutHandler;
        return new Promise((resolve, reject) => {
            this.once(eventType, event => {
                clearTimeout(timeoutHandler);
                resolve(event);
            });
            if (!this._socket) { // no timeouts if debugging the tests
                timeoutHandler = setTimeout(() => {
                    reject(new Error(`no event '${eventType}' received after ${timeout} ms`));
                }, timeout);
            }
        });
    }
    /*
     * Returns a promise that will resolve if an 'initialized' event was received within some specified time
     * and a subsequent 'configurationDone' request was successfully executed.
     * The promise will be rejected if a timeout occurs or if the 'configurationDone' request fails.
     */
    configurationSequence() {
        return this.waitForEvent('initialized').then(event => {
            return this.configurationDone();
        });
    }
    /**
     * Returns a promise that will resolve if a 'initialize' and a 'launch' request were successful.
     */
    launch(launchArgs) {
        return this.initializeRequest().then(response => {
            if (response.body && response.body.supportsConfigurationDoneRequest) {
                this._supportsConfigurationDoneRequest = true;
            }
            return this.launchRequest(launchArgs);
        });
    }
    configurationDone() {
        if (this._supportsConfigurationDoneRequest) {
            return this.configurationDoneRequest();
        }
        else {
            // if debug adapter doesn't support the configurationDoneRequest we have to send the setExceptionBreakpointsRequest.
            return this.setExceptionBreakpointsRequest({ filters: ['all'] });
        }
    }
    /*
     * Returns a promise that will resolve if a 'stopped' event was received within some specified time
     * and the event's reason and line number was asserted.
     * The promise will be rejected if a timeout occurs, the assertions fail, or if the 'stackTrace' request fails.
     */
    assertStoppedLocation(reason, expected) {
        return this.waitForEvent('stopped').then(event => {
            assert.equal(event.body.reason, reason);
            return this.stackTraceRequest({
                threadId: event.body.threadId
            });
        }).then(response => {
            const frame = response.body.stackFrames[0];
            if (typeof expected.path === 'string' || expected.path instanceof RegExp) {
                this.assertPath(frame.source.path, expected.path, 'stopped location: path mismatch');
            }
            if (typeof expected.line === 'number') {
                assert.equal(frame.line, expected.line, 'stopped location: line mismatch');
            }
            if (typeof expected.column === 'number') {
                assert.equal(frame.column, expected.column, 'stopped location: column mismatch');
            }
            return response;
        });
    }
    assertPartialLocationsEqual(locA, locB) {
        if (locA.path) {
            this.assertPath(locA.path, locB.path, 'breakpoint verification mismatch: path');
        }
        if (typeof locA.line === 'number') {
            assert.equal(locA.line, locB.line, 'breakpoint verification mismatch: line');
        }
        if (typeof locB.column === 'number' && typeof locA.column === 'number') {
            assert.equal(locA.column, locB.column, 'breakpoint verification mismatch: column');
        }
    }
    /*
     * Returns a promise that will resolve if enough output events with the given category have been received
     * and the concatenated data match the expected data.
     * The promise will be rejected as soon as the received data cannot match the expected data or if a timeout occurs.
     */
    assertOutput(category, expected, timeout) {
        timeout = timeout || this.defaultTimeout;
        return new Promise((resolve, reject) => {
            let output = '';
            let timeoutHandler;
            this.on('output', event => {
                const e = event;
                if (e.body.category === category) {
                    output += e.body.output;
                    if (output.indexOf(expected) === 0) {
                        clearTimeout(timeoutHandler);
                        resolve(event);
                    }
                    else if (expected.indexOf(output) !== 0) {
                        const sanitize = (s) => s.toString().replace(/\r/mg, '\\r').replace(/\n/mg, '\\n');
                        reject(new Error(`received data '${sanitize(output)}' is not a prefix of the expected data '${sanitize(expected)}'`));
                    }
                }
            });
            if (!this._socket) { // no timeouts if debugging the tests
                timeoutHandler = setTimeout(() => {
                    reject(new Error(`not enough output data received after ${timeout} ms`));
                }, timeout);
            }
        });
    }
    assertPath(path, expected, message) {
        if (expected instanceof RegExp) {
            assert.ok(expected.test(path), message);
        }
        else {
            if (DebugClient.CASE_INSENSITIVE_FILESYSTEM) {
                if (typeof path === 'string') {
                    path = path.toLowerCase();
                }
                if (typeof expected === 'string') {
                    expected = expected.toLowerCase();
                }
            }
            assert.equal(path, expected, message);
        }
    }
    // ---- scenarios ---------------------------------------------------------------------------------------------------------
    /**
     * Returns a promise that will resolve if a configurable breakpoint has been hit within some time
     * and the event's reason and line number was asserted.
     * The promise will be rejected if a timeout occurs, the assertions fail, or if the requests fails.
     */
    hitBreakpoint(launchArgs, location, expectedStopLocation, expectedBPLocation) {
        return Promise.all([
            this.waitForEvent('initialized').then(event => {
                return this.setBreakpointsRequest({
                    lines: [location.line],
                    breakpoints: [{ line: location.line, column: location.column }],
                    source: { path: location.path }
                });
            }).then(response => {
                const bp = response.body.breakpoints[0];
                const verified = (typeof location.verified === 'boolean') ? location.verified : true;
                assert.equal(bp.verified, verified, 'breakpoint verification mismatch: verified');
                const actualLocation = {
                    column: bp.column,
                    line: bp.line,
                    path: bp.source && bp.source.path
                };
                this.assertPartialLocationsEqual(actualLocation, expectedBPLocation || location);
                return this.configurationDone();
            }),
            this.launch(launchArgs),
            this.assertStoppedLocation('breakpoint', expectedStopLocation || location)
        ]);
    }
}
exports.DebugClient = DebugClient;
//# sourceMappingURL=data:application/json;base64,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