"use strict";
// *****************************************************************************
// Copyright (C) 2018 TypeFox and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebSocketChannel = void 0;
/* eslint-disable @typescript-eslint/no-explicit-any */
const event_1 = require("../event");
const uint8_array_message_buffer_1 = require("../message-rpc/uint8-array-message-buffer");
const disposable_1 = require("../disposable");
/**
 * A channel that manages the main websocket connection between frontend and backend. All service channels
 * are reusing this main channel. (multiplexing). An {@link IWebSocket} abstraction is used to keep the implementation
 * independent of the actual websocket implementation and its execution context (backend vs. frontend).
 */
class WebSocketChannel {
    constructor(socket) {
        this.socket = socket;
        this.onCloseEmitter = new event_1.Emitter();
        this.onMessageEmitter = new event_1.Emitter();
        this.onErrorEmitter = new event_1.Emitter();
        this.toDispose = new disposable_1.DisposableCollection();
        this.toDispose.pushAll([this.onCloseEmitter, this.onMessageEmitter, this.onErrorEmitter]);
        socket.onClose((reason, code) => this.onCloseEmitter.fire({ reason, code }));
        socket.onClose(() => this.close());
        socket.onError(error => this.onErrorEmitter.fire(error));
        // eslint-disable-next-line arrow-body-style
        socket.onMessage(data => this.onMessageEmitter.fire(() => {
            // In the browser context socketIO receives binary messages as ArrayBuffers.
            // So we have to convert them to a Uint8Array before delegating the message to the read buffer.
            const buffer = data instanceof ArrayBuffer ? new Uint8Array(data) : data;
            return new uint8_array_message_buffer_1.Uint8ArrayReadBuffer(buffer);
        }));
    }
    get onClose() {
        return this.onCloseEmitter.event;
    }
    get onMessage() {
        return this.onMessageEmitter.event;
    }
    get onError() {
        return this.onErrorEmitter.event;
    }
    getWriteBuffer() {
        const result = new uint8_array_message_buffer_1.Uint8ArrayWriteBuffer();
        result.onCommit(buffer => {
            if (this.socket.isConnected()) {
                this.socket.send(buffer);
            }
        });
        return result;
    }
    close() {
        this.toDispose.dispose();
        this.socket.close();
    }
}
exports.WebSocketChannel = WebSocketChannel;
WebSocketChannel.wsPath = '/services';
//# sourceMappingURL=web-socket-channel.js.map