/*******************************************************************************
 * Copyright (c) 2011 itemis AG (http://www.itemis.eu) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package org.eclipse.xtend.ide.contentassist;

import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.TextUtilities;
import org.eclipse.xtend.core.jvmmodel.IXtendJvmAssociations;
import org.eclipse.xtend.core.xtend.XtendClass;
import org.eclipse.xtend.core.xtend.XtendFile;
import org.eclipse.xtend.core.xtend.XtendFunction;
import org.eclipse.xtend.core.xtend.XtendImport;
import org.eclipse.xtend.core.xtend.XtendPackage;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.formatting.IIndentationInformation;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.xbase.compiler.ImportManager;
import org.eclipse.xtext.xbase.compiler.StringBuilderBasedAppendable;

import com.google.inject.Inject;
import com.google.inject.Provider;

/**
 * An {@link org.eclipse.xtext.xbase.compiler.IAppendable}to insert text into an Xtend document. Takes imports and
 * existing variable names into account.
 * 
 * @author Jan Koehnlein - Initial contribution and API
 */
public class ReplacingAppendable extends StringBuilderBasedAppendable {

	private static final Logger LOG = Logger.getLogger(ReplacingAppendable.class);

	public static class Factory {

		@Inject
		private IIndentationInformation indentation;

		@Inject
		private IXtendJvmAssociations associations;

		@Inject
		private Provider<WhitespaceHelper> whitespaceHelperProvider;

		public ReplacingAppendable get(IXtextDocument document, EObject context, int offset, int length) {
			return get(document, context, offset, length, getIndentationLevelAtOffset(offset, document), false);
		}

		public ReplacingAppendable get(IXtextDocument document, EObject context, int offset, int length,
				int indentationLevel, boolean ensureEmptyLinesAround) {
			try {
				XtendFile xtendFile = EcoreUtil2.getContainerOfType(context, XtendFile.class);
				if (xtendFile != null) {
					ImportManager importManager = new ImportManager(true, '$');
					for (XtendImport xImport : xtendFile.getImports()) {
						if (xImport.getImportedType() != null) {
							importManager.addImportFor(xImport.getImportedType());
						}
					}
					WhitespaceHelper whitespaceHelper = whitespaceHelperProvider.get();
					whitespaceHelper.initialize(document, offset, length, ensureEmptyLinesAround);
					ReplacingAppendable appendable = new ReplacingAppendable(importManager,
							indentation.getIndentString(), document, xtendFile, whitespaceHelper);
					for (int i = 0; i < indentationLevel; ++i)
						appendable.increaseIndentation();
					return appendable;
				}
			} catch (Exception exc) {
				LOG.error("Error initializing appendable", exc);
			}
			return null;
		}

		protected EObject getLocalVariableScopeContext(EObject context) {
			if (context instanceof XtendClass)
				return associations.getInferredType((XtendClass) context);
			else if (context instanceof XtendFunction) {
				return associations.getDirectlyInferredOperation((XtendFunction) context);
			}
			return context;
		}

		protected int getIndentationLevelAtOffset(int offset, IDocument document) {
			try {
				if (offset <= 0)
					return 0;
				int currentOffset = offset - 1;
				char currentChr = document.getChar(currentOffset);
				int indentationOffset = 0;
				while (currentChr != '\n' && currentChr != '\r' && currentOffset > 0) {
					if (Character.isWhitespace(currentChr))
						++indentationOffset;
					else
						indentationOffset = 0;
					--currentOffset;
					currentChr = document.getChar(currentOffset);
				}
				return indentationOffset / indentation.getIndentString().length();
			} catch (BadLocationException e) {
				LOG.error("Error calculating indentation at offset", e);
			}
			return 0;
		}
	}

	private List<String> existingImports;

	private IXtextDocument document;

	private XtendFile xtendFile;

	private final WhitespaceHelper whitespaceHelper;
	
	public ReplacingAppendable(ImportManager importManager, String indentString, IXtextDocument document,
			XtendFile xtendFile, WhitespaceHelper whitespaceHelper) {
		super(importManager, indentString, TextUtilities.getDefaultLineDelimiter(document));
		this.document = document;
		this.xtendFile = xtendFile;
		this.whitespaceHelper = whitespaceHelper;
		existingImports = importManager.getImports();
	}
	
	@Override
	@NonNull
	public String toString() {
		return getCode();
	}
	
	public int getTotalOffset() {
		return whitespaceHelper.getTotalOffset();
	}

	@NonNull
	public String getCode() {
		StringBuilder b = new StringBuilder();
		if (whitespaceHelper.getPrefix() != null)
			b.append(whitespaceHelper.getPrefix().replace(getLineSeparator(), getIndentationString()));
		b.append(super.toString());
		if (whitespaceHelper.getSuffix() != null)
			b.append(whitespaceHelper.getSuffix().replace(getLineSeparator(), getIndentationString()));
		return b.toString();
	}

	public void commitChanges() throws BadLocationException {
		document.replace(whitespaceHelper.getTotalOffset(), whitespaceHelper.getTotalLength(), toString());
		insertNewImports();
	}
	
	public int commitChanges(int offset, int length) throws BadLocationException {
		int actualOffset = Math.min(whitespaceHelper.getTotalOffset(), offset);
		int endOffset = Math.max(whitespaceHelper.getTotalOffset() + whitespaceHelper.getTotalLength(), offset + length);
		int actualLength = endOffset - actualOffset;
		document.replace(actualOffset, actualLength, toString());
		int shiftCursorBy = insertNewImports();
		return shiftCursorBy;
	}

	public int insertNewImports() throws BadLocationException {
		List<String> newImports = getNewImports();
		if (!newImports.isEmpty()) {
			StringBuilder importSection = new StringBuilder();
			for (String newImport : newImports) {
				importSection.append("import ");
				importSection.append(newImport);
				importSection.append(getLineSeparator());
			}
			importSection.append(getLineSeparator());
			int offset;
			if (xtendFile.getImports().isEmpty()) {
				if(xtendFile.getPackage() != null) {
					List<INode> packageDeclarationNodes = NodeModelUtils.findNodesForFeature(xtendFile, XtendPackage.Literals.XTEND_FILE__PACKAGE);
					if(packageDeclarationNodes.isEmpty())
						throw new IllegalStateException("Package declaration nodes must not be null");
					INode packageDeclNode = packageDeclarationNodes.get(packageDeclarationNodes.size()-1);
					offset = packageDeclNode.getOffset() + packageDeclNode.getLength();
					importSection.insert(0, getLineSeparator() + getLineSeparator());
				} else {
					offset = 0;
				}
			} else {
				ICompositeNode lastImportNode = NodeModelUtils.findActualNodeFor(xtendFile.getImports().get(
						xtendFile.getImports().size() - 1));
				if (lastImportNode == null) {
					throw new IllegalStateException("Last XtendImport node may not be null");
				}
				importSection.insert(0, getLineSeparator());
				offset = lastImportNode.getOffset() + lastImportNode.getLength();
			}
			int replaceLength = 0;
			while(Character.isWhitespace(document.get(offset + replaceLength, 1).charAt(0))){
				++replaceLength;
			}
			document.replace(offset, replaceLength, importSection.toString());
			return importSection.length() - replaceLength;
		}
		return 0;
	}

	protected List<String> getNewImports() {
		List<String> imports = getImportManager().getImports();
		imports.removeAll(existingImports);
		return imports;
	}
	
}
