/*******************************************************************************
 * Copyright (c) 2007 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.internal.ui.source;

import java.util.Iterator;

import org.eclipse.core.filebuffers.ITextFileBufferManager;
import org.eclipse.core.filebuffers.LocationKind;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.mint.MintCore;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.Block;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.IBinding;
import org.eclipse.jdt.core.dom.Javadoc;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.MethodInvocation;
import org.eclipse.jdt.core.dom.PrimitiveType;
import org.eclipse.jdt.core.dom.ReturnStatement;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.SingleVariableDeclaration;
import org.eclipse.jdt.core.dom.Statement;
import org.eclipse.jdt.core.dom.TagElement;
import org.eclipse.jdt.core.dom.TextElement;
import org.eclipse.jdt.core.dom.TypeDeclaration;
import org.eclipse.jdt.core.dom.rewrite.ASTRewrite;
import org.eclipse.jdt.core.dom.rewrite.ListRewrite;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.osgi.util.NLS;
import org.eclipse.text.edits.TextEdit;

public class CustomizeGeneratedMethodOperation implements IWorkspaceRunnable {

	private static final String NOT_PARAM = "NOT"; //$NON-NLS-1$
	private static final String GENERATED_TAG = "@generated"; //$NON-NLS-1$
	private static final String GEN_SUFFIX = "Gen"; //$NON-NLS-1$
	private IMethod[] methods;

	public CustomizeGeneratedMethodOperation(IMethod[] methods) {
		this.methods = methods;
	}

	public void run(IProgressMonitor monitor) throws CoreException,
			OperationCanceledException {
		if (methods.length == 0)
			return;

		if (monitor == null)
			monitor = new NullProgressMonitor();

		monitor.beginTask("", 3 * methods.length + 1); //$NON-NLS-1$
		try {
			doRun(monitor);
		} finally {
			monitor.done();
		}
	}

	@SuppressWarnings("unchecked")
	private void doRun(IProgressMonitor monitor) throws CoreException {
		ICompilationUnit cu = methods[0].getCompilationUnit();

		ASTParser parser = ASTParser.newParser(AST.JLS3);
		parser.setSource(cu);
		parser.setResolveBindings(true);
		CompilationUnit astRoot = (CompilationUnit) parser
				.createAST(new SubProgressMonitor(monitor, 1));

		parser.setProject(cu.getJavaProject());
		IBinding[] bindings = parser.createBindings(methods,
				new SubProgressMonitor(monitor, 1));

		ITextFileBufferManager manager = ITextFileBufferManager.DEFAULT;
		IPath path = cu.getPath();
		manager.connect(path, LocationKind.IFILE, new SubProgressMonitor(
				monitor, 1));
		try {
			IDocument document = manager.getTextFileBuffer(path,
					LocationKind.IFILE).getDocument();
			ASTRewrite rewrite = ASTRewrite.create(astRoot.getAST());
			for (int i = 0; i < methods.length; ++i) {
				IMethod method = methods[i];
				if (!cu.equals(method.getCompilationUnit()))
					continue;

				monitor.subTask(method.getElementName());

				MethodDeclaration decl = (MethodDeclaration) astRoot
						.findDeclaringNode(bindings[i].getKey());
				if (decl == null) {
					String msg = NLS
							.bind(
									Messages.CustomizeGeneratedMethodOperation_ErrorMethodDecl,
									method);
					Status status = new Status(IStatus.ERROR,
							MintCore.PLUGIN_ID, 0, msg, null);
					throw new CoreException(status);
				}

				MethodDeclaration genDecl = (MethodDeclaration) ASTNode
						.copySubtree(astRoot.getAST(), decl);
				SimpleName genName = astRoot.getAST().newSimpleName(
						decl.getName().getIdentifier() + GEN_SUFFIX);
				genDecl.setName(genName);

				Javadoc javadoc = genDecl.getJavadoc();
				TagElement oldGeneratedTag = findGeneratedTag(javadoc);

				TagElement generatedTag = astRoot.getAST().newTagElement();
				generatedTag.setTagName(GENERATED_TAG);
				if (oldGeneratedTag == null)
					javadoc.tags().add(generatedTag);
				else
					javadoc.tags().set(javadoc.tags().indexOf(oldGeneratedTag),
							generatedTag);

				ListRewrite listRewrite = rewrite.getListRewrite(decl
						.getParent(),
						TypeDeclaration.BODY_DECLARATIONS_PROPERTY);
				listRewrite.insertBefore(genDecl, decl, null);

				MethodInvocation inv = astRoot.getAST().newMethodInvocation();
				inv.setName(astRoot.getAST().newSimpleName(
						genDecl.getName().getIdentifier()));
				for (Iterator<?> j = decl.parameters().iterator(); j.hasNext();) {
					SingleVariableDeclaration p = (SingleVariableDeclaration) j
							.next();
					inv.arguments().add(
							astRoot.getAST().newSimpleName(
									p.getName().getIdentifier()));
				}

				Block block = astRoot.getAST().newBlock();
				if (decl.getReturnType2().isPrimitiveType()
						&& PrimitiveType.VOID.equals(((PrimitiveType) decl
								.getReturnType2()).getPrimitiveTypeCode())) {
					Statement stmt = astRoot.getAST().newExpressionStatement(
							inv);
					block.statements().add(stmt);
				} else {
					ReturnStatement ret = astRoot.getAST().newReturnStatement();
					ret.setExpression(inv);
					block.statements().add(ret);
				}

				rewrite.replace(decl.getBody(), block, null);

				javadoc = decl.getJavadoc();
				oldGeneratedTag = findGeneratedTag(javadoc);
				generatedTag = astRoot.getAST().newTagElement();
				generatedTag.setTagName(GENERATED_TAG);
				TextElement notElement = astRoot.getAST().newTextElement();
				notElement.setText(NOT_PARAM);
				generatedTag.fragments().add(notElement);
				listRewrite = rewrite.getListRewrite(javadoc,
						Javadoc.TAGS_PROPERTY);
				if (oldGeneratedTag == null)
					listRewrite.insertLast(generatedTag, null);
				else
					listRewrite.replace(oldGeneratedTag, generatedTag, null);
			}

			TextEdit edit = rewrite.rewriteAST();
			edit.apply(document, TextEdit.UPDATE_REGIONS);
		} catch (BadLocationException e) {
			Status status = new Status(
					IStatus.ERROR,
					MintCore.PLUGIN_ID,
					0,
					Messages.CustomizeGeneratedMethodOperation_ErrorBadLocation,
					e);
			throw new CoreException(status);
		} finally {
			monitor.done();
			manager.disconnect(path, LocationKind.IFILE,
					new SubProgressMonitor(monitor, 1));
		}
	}

	private TagElement findGeneratedTag(Javadoc javadoc) {
		TagElement generatedTag = null;
		for (Iterator<?> i = javadoc.tags().iterator(); i.hasNext();) {
			TagElement tag = (TagElement) i.next();
			if (GENERATED_TAG.equals(tag.getTagName())) {
				generatedTag = tag;
				break;
			}
		}

		return generatedTag;
	}
}
