/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.geometry;

import com.google.common.geometry.S1Angle;
import com.google.common.geometry.S2AreaCentroid;
import com.google.common.geometry.S2Cap;
import com.google.common.geometry.S2Cell;
import com.google.common.geometry.S2Edge;
import com.google.common.geometry.S2EdgeIndex;
import com.google.common.geometry.S2EdgeUtil;
import com.google.common.geometry.S2LatLngRect;
import com.google.common.geometry.S2Loop;
import com.google.common.geometry.S2Point;
import com.google.common.geometry.S2PolygonBuilder;
import com.google.common.geometry.S2Region;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.logging.Logger;

public strictfp final class S2Polygon
implements S2Region,
Comparable<S2Polygon> {
    private static final Logger log = Logger.getLogger(S2Polygon.class.getCanonicalName());
    private List<S2Loop> loops = new ArrayList<S2Loop>();
    private S2LatLngRect bound;
    private boolean hasHoles;
    private int numVertices;

    public S2Polygon() {
        this.bound = S2LatLngRect.empty();
        this.hasHoles = false;
        this.numVertices = 0;
    }

    public S2Polygon(List<S2Loop> loops) {
        this.bound = S2LatLngRect.empty();
        this.init(loops);
    }

    public S2Polygon(S2Loop loop) {
        this.bound = loop.getRectBound();
        this.hasHoles = false;
        this.numVertices = loop.numVertices();
        this.loops.add(loop);
    }

    public S2Polygon(S2Polygon src) {
        this.bound = src.getRectBound();
        this.hasHoles = src.hasHoles;
        this.numVertices = src.numVertices;
        for (int i = 0; i < src.numLoops(); ++i) {
            this.loops.add(new S2Loop(src.loop(i)));
        }
    }

    @Override
    public int compareTo(S2Polygon other) {
        if (this.numLoops() != other.numLoops()) {
            return this.numLoops() - other.numLoops();
        }
        for (int i = 0; i < this.numLoops(); ++i) {
            int compare = this.loops.get(i).compareTo(other.loops.get(i));
            if (compare == 0) continue;
            return compare;
        }
        return 0;
    }

    public void init(List<S2Loop> loops) {
        HashMap<S2Loop, List<S2Loop>> loopMap = new HashMap<S2Loop, List<S2Loop>>();
        loopMap.put(null, new ArrayList());
        for (S2Loop loop : loops) {
            S2Polygon.insertLoop(loop, null, loopMap);
            this.numVertices += loop.numVertices();
        }
        loops.clear();
        S2Polygon.sortValueLoops(loopMap);
        this.initLoop(null, -1, loopMap);
        this.hasHoles = false;
        this.bound = S2LatLngRect.empty();
        for (int i = 0; i < this.numLoops(); ++i) {
            if (this.loop(i).sign() < 0) {
                this.hasHoles = true;
                continue;
            }
            this.bound = this.bound.union(this.loop(i).getRectBound());
        }
    }

    public void release(List<S2Loop> loops) {
        loops.addAll(this.loops);
        this.loops.clear();
        this.bound = S2LatLngRect.empty();
        this.hasHoles = false;
        this.numVertices = 0;
    }

    public static boolean isValid(List<S2Loop> loops) {
        if (loops.size() > 1) {
            HashMap<UndirectedEdge, LoopVertexIndexPair> edges = new HashMap<UndirectedEdge, LoopVertexIndexPair>();
            for (int i = 0; i < loops.size(); ++i) {
                S2Loop lp = loops.get(i);
                for (int j = 0; j < lp.numVertices(); ++j) {
                    UndirectedEdge key = new UndirectedEdge(lp.vertex(j), lp.vertex(j + 1));
                    LoopVertexIndexPair value = new LoopVertexIndexPair(i, j);
                    if (edges.containsKey(key)) {
                        LoopVertexIndexPair other = (LoopVertexIndexPair)edges.get(key);
                        log.info("Duplicate edge: loop " + i + ", edge " + j + " and loop " + other.getLoopIndex() + ", edge " + other.getVertexIndex());
                        return false;
                    }
                    edges.put(key, value);
                }
            }
        }
        for (int i = 0; i < loops.size(); ++i) {
            if (!loops.get(i).isNormalized()) {
                log.info("Loop " + i + " encloses more than half the sphere");
                return false;
            }
            for (int j = i + 1; j < loops.size(); ++j) {
                if (loops.get(i).containsOrCrosses(loops.get(j)) >= 0) continue;
                log.info("Loop " + i + " crosses loop " + j);
                return false;
            }
        }
        return true;
    }

    public int numLoops() {
        return this.loops.size();
    }

    public S2Loop loop(int k) {
        return this.loops.get(k);
    }

    public int getParent(int k) {
        int depth = this.loop(k).depth();
        if (depth == 0) {
            return -1;
        }
        while (--k >= 0 && this.loop(k).depth() >= depth) {
        }
        return k;
    }

    public int getLastDescendant(int k) {
        if (k < 0) {
            return this.numLoops() - 1;
        }
        int depth = this.loop(k).depth();
        while (++k < this.numLoops() && this.loop(k).depth() > depth) {
        }
        return k - 1;
    }

    private S2AreaCentroid getAreaCentroid(boolean doCentroid) {
        double areaSum = 0.0;
        S2Point centroidSum = new S2Point(0.0, 0.0, 0.0);
        for (int i = 0; i < this.numLoops(); ++i) {
            if (doCentroid) {
                S2AreaCentroid areaCentroid = this.loop(i).getAreaAndCentroid();
                double loopArea = areaCentroid.getArea();
                int loopSign = this.loop(i).sign();
                areaSum += (double)loopSign * loopArea;
                S2Point currentCentroid = areaCentroid.getCentroid();
                centroidSum = new S2Point(centroidSum.x + (double)loopSign * currentCentroid.x, centroidSum.y + (double)loopSign * currentCentroid.y, centroidSum.z + (double)loopSign * currentCentroid.z);
                continue;
            }
            double loopArea = this.loop(i).getArea();
            int loopSign = this.loop(i).sign();
            areaSum += (double)loopSign * loopArea;
        }
        return new S2AreaCentroid(areaSum, doCentroid ? centroidSum : null);
    }

    public S2AreaCentroid getAreaAndCentroid() {
        return this.getAreaCentroid(true);
    }

    public double getArea() {
        return this.getAreaCentroid(false).getArea();
    }

    public S2Point getCentroid() {
        return this.getAreaCentroid(true).getCentroid();
    }

    public S1Angle getDistance(S2Point p) {
        if (this.contains(p)) {
            return S1Angle.radians(0.0);
        }
        S1Angle minDistance = S1Angle.radians(Math.PI);
        for (int i = 0; i < this.numLoops(); ++i) {
            minDistance = S1Angle.min(minDistance, this.loop(i).getDistance(p));
        }
        return minDistance;
    }

    public boolean contains(S2Polygon b) {
        if (this.numLoops() == 1 && b.numLoops() == 1) {
            return this.loop(0).contains(b.loop(0));
        }
        if (!this.bound.contains(b.getRectBound()) && !this.bound.lng().union(b.getRectBound().lng()).isFull()) {
            return false;
        }
        if (!this.hasHoles && !b.hasHoles) {
            for (int j = 0; j < b.numLoops(); ++j) {
                if (this.anyLoopContains(b.loop(j))) continue;
                return false;
            }
            return true;
        }
        return this.containsAllShells(b) && b.excludesAllHoles(this);
    }

    public boolean intersects(S2Polygon b) {
        if (this.numLoops() == 1 && b.numLoops() == 1) {
            return this.loop(0).intersects(b.loop(0));
        }
        if (!this.bound.intersects(b.getRectBound())) {
            return false;
        }
        if (!this.hasHoles && !b.hasHoles) {
            for (int i = 0; i < this.numLoops(); ++i) {
                for (int j = 0; j < b.numLoops(); ++j) {
                    if (!this.loop(i).intersects(b.loop(j))) continue;
                    return true;
                }
            }
            return false;
        }
        return this.intersectsAnyShell(b) || b.intersectsAnyShell(this);
    }

    private static void addIntersection(S2Point a0, S2Point a1, S2Point b0, S2Point b1, boolean addSharedEdges, int crossing, List<ParametrizedS2Point> intersections) {
        if (crossing > 0) {
            S2Point x = S2EdgeUtil.getIntersection(a0, a1, b0, b1);
            double t = S2EdgeUtil.getDistanceFraction(x, a0, a1);
            intersections.add(new ParametrizedS2Point(t, x));
        } else if (S2EdgeUtil.vertexCrossing(a0, a1, b0, b1)) {
            double t;
            double d = t = a0 == b0 || a0 == b1 ? 0.0 : 1.0;
            if (!addSharedEdges && a1 == b1) {
                t = 1.0;
            }
            intersections.add(new ParametrizedS2Point(t, t == 0.0 ? a0 : a1));
        }
    }

    private static void clipEdge(S2Point a0, S2Point a1, S2LoopSequenceIndex bIndex, boolean addSharedEdges, List<ParametrizedS2Point> intersections) {
        S2EdgeIndex.DataEdgeIterator it = new S2EdgeIndex.DataEdgeIterator(bIndex);
        it.getCandidates(a0, a1);
        S2EdgeUtil.EdgeCrosser crosser = new S2EdgeUtil.EdgeCrosser(a0, a1, a0);
        S2Point from = null;
        S2Point to = null;
        while (it.hasNext()) {
            int crossing;
            S2Point previousTo = to;
            S2Edge fromTo = bIndex.edgeFromTo(it.index());
            from = fromTo.getStart();
            to = fromTo.getEnd();
            if (previousTo != from) {
                crosser.restartAt(from);
            }
            if ((crossing = crosser.robustCrossing(to)) >= 0) {
                S2Polygon.addIntersection(a0, a1, from, to, addSharedEdges, crossing, intersections);
            }
            it.next();
        }
    }

    private static void clipBoundary(S2Polygon a, boolean reverseA, S2Polygon b, boolean reverseB, boolean invertB, boolean addSharedEdges, S2PolygonBuilder builder) {
        S2PolygonIndex bIndex = new S2PolygonIndex(b, reverseB);
        bIndex.predictAdditionalCalls(a.getNumVertices());
        ArrayList<ParametrizedS2Point> intersections = new ArrayList<ParametrizedS2Point>();
        for (S2Loop aLoop : a.loops) {
            int j;
            int n = aLoop.numVertices();
            int dir = aLoop.isHole() ^ reverseA ? -1 : 1;
            boolean inside = b.contains(aLoop.vertex(0)) ^ invertB;
            int n2 = j = dir > 0 ? 0 : n;
            while (n > 0) {
                S2Point a0 = aLoop.vertex(j);
                S2Point a1 = aLoop.vertex(j + dir);
                intersections.clear();
                S2Polygon.clipEdge(a0, a1, bIndex, addSharedEdges, intersections);
                if (inside) {
                    intersections.add(new ParametrizedS2Point(0.0, a0));
                }
                boolean bl = inside = (intersections.size() & 1) == 1;
                if (inside) {
                    intersections.add(new ParametrizedS2Point(1.0, a1));
                }
                Collections.sort(intersections);
                int size = intersections.size();
                for (int i = 1; i < size; i += 2) {
                    builder.addEdge(((ParametrizedS2Point)intersections.get(i - 1)).getPoint(), ((ParametrizedS2Point)intersections.get(i)).getPoint());
                }
                --n;
                j += dir;
            }
        }
    }

    public int getNumVertices() {
        return this.numVertices;
    }

    public void initToIntersection(S2Polygon a, S2Polygon b) {
        this.initToIntersectionSloppy(a, b, S2EdgeUtil.DEFAULT_INTERSECTION_TOLERANCE);
    }

    public void initToIntersectionSloppy(S2Polygon a, S2Polygon b, S1Angle vertexMergeRadius) {
        if (this.numLoops() != 0) {
            throw new IllegalStateException();
        }
        if (!a.bound.intersects(b.bound)) {
            return;
        }
        S2PolygonBuilder.Options options = S2PolygonBuilder.Options.DIRECTED_XOR;
        options.setMergeDistance(vertexMergeRadius);
        S2PolygonBuilder builder = new S2PolygonBuilder(options);
        S2Polygon.clipBoundary(a, false, b, false, false, true, builder);
        S2Polygon.clipBoundary(b, false, a, false, false, false, builder);
        if (!builder.assemblePolygon(this, null)) {
            log.severe("Bad directed edges");
        }
    }

    public void initToUnion(S2Polygon a, S2Polygon b) {
        this.initToUnionSloppy(a, b, S2EdgeUtil.DEFAULT_INTERSECTION_TOLERANCE);
    }

    public void initToUnionSloppy(S2Polygon a, S2Polygon b, S1Angle vertexMergeRadius) {
        if (this.numLoops() != 0) {
            throw new IllegalStateException();
        }
        S2PolygonBuilder.Options options = S2PolygonBuilder.Options.DIRECTED_XOR;
        options.setMergeDistance(vertexMergeRadius);
        S2PolygonBuilder builder = new S2PolygonBuilder(options);
        S2Polygon.clipBoundary(a, false, b, false, true, true, builder);
        S2Polygon.clipBoundary(b, false, a, false, true, false, builder);
        if (!builder.assemblePolygon(this, null)) {
            log.severe("Bad directed edges");
        }
    }

    public static S2Polygon destructiveUnion(List<S2Polygon> polygons) {
        return S2Polygon.destructiveUnionSloppy(polygons, S2EdgeUtil.DEFAULT_INTERSECTION_TOLERANCE);
    }

    public static S2Polygon destructiveUnionSloppy(List<S2Polygon> polygons, S1Angle vertexMergeRadius) {
        Map.Entry firstEntry;
        Iterator polygonIterator;
        TreeMap<Integer, Collection<S2Polygon>> queue = new TreeMap<Integer, Collection<S2Polygon>>();
        for (S2Polygon polygon : polygons) {
            int vertices = polygon.getNumVertices();
            TreeSet<S2Polygon> polygonList = (TreeSet<S2Polygon>)queue.get(vertices);
            if (polygonList == null) {
                polygonList = new TreeSet<S2Polygon>();
                queue.put(vertices, polygonList);
            }
            polygonList.add(polygon);
        }
        polygons.clear();
        Collection queueSet = S2Polygon.entries(queue);
        while (queueSet.size() > 1) {
            queueSet = S2Polygon.entries(queue);
            Iterator smallestIter = queueSet.iterator();
            Map.Entry<Integer, S2Polygon> smallest = smallestIter.next();
            int aSize = (Integer)smallest.getKey();
            S2Polygon aPolygon = (S2Polygon)smallest.getValue();
            smallestIter.remove();
            S2Polygon.remove(queue, smallest);
            smallest = smallestIter.next();
            int bSize = smallest.getKey();
            S2Polygon bPolygon = smallest.getValue();
            smallestIter.remove();
            S2Polygon.remove(queue, smallest);
            S2Polygon unionPolygon = new S2Polygon();
            unionPolygon.initToUnionSloppy(aPolygon, bPolygon, vertexMergeRadius);
            int unionSize = aSize + bSize;
            TreeSet<S2Polygon> polygonList = (TreeSet<S2Polygon>)queue.get(unionSize);
            if (polygonList == null) {
                polygonList = new TreeSet<S2Polygon>();
                queue.put(unionSize, polygonList);
            }
            polygonList.add(unionPolygon);
        }
        Iterator keyIterator = queue.entrySet().iterator();
        if (keyIterator.hasNext() && (polygonIterator = ((Collection)(firstEntry = keyIterator.next()).getValue()).iterator()).hasNext()) {
            return (S2Polygon)polygonIterator.next();
        }
        return new S2Polygon();
    }

    private static <K, V> Collection<Map.Entry<K, V>> entries(Map<K, Collection<V>> queue) {
        ArrayList<Map.Entry<K, V>> entries = new ArrayList<Map.Entry<K, V>>();
        for (Map.Entry<K, Collection<V>> entry : queue.entrySet()) {
            K key = entry.getKey();
            for (V value : entry.getValue()) {
                entries.add(new AbstractMap.SimpleEntry<K, V>(key, value));
            }
        }
        return entries;
    }

    private static void remove(Map<Integer, Collection<S2Polygon>> map, Map.Entry<Integer, S2Polygon> entry) {
        int key = entry.getKey();
        Collection<S2Polygon> polygons = map.get(key);
        if (polygons == null) {
            map.remove(key);
        } else {
            polygons.remove(entry.getValue());
            if (polygons.isEmpty()) {
                map.remove(key);
            }
        }
    }

    public boolean isNormalized() {
        ArrayList<S2Point> vertices = new ArrayList<S2Point>();
        S2Loop lastParent = null;
        for (int i = 0; i < this.numLoops(); ++i) {
            S2Loop child = this.loop(i);
            if (child.depth() == 0) continue;
            S2Loop parent = this.loop(this.getParent(i));
            if (parent != lastParent) {
                vertices.clear();
                for (int j = 0; j < parent.numVertices(); ++j) {
                    vertices.add(parent.vertex(j));
                }
                lastParent = parent;
            }
            int count = 0;
            for (int j = 0; j < child.numVertices(); ++j) {
                if (!vertices.contains(child.vertex(j))) continue;
                ++count;
            }
            if (count <= true) continue;
            return false;
        }
        return true;
    }

    boolean boundaryApproxEquals(S2Polygon b, double maxError) {
        if (this.numLoops() != b.numLoops()) {
            log.severe("!= loops: " + Integer.toString(this.numLoops()) + " vs. " + Integer.toString(b.numLoops()));
            return false;
        }
        for (int i = 0; i < this.numLoops(); ++i) {
            S2Loop aLoop = this.loop(i);
            boolean success = false;
            for (int j = 0; j < this.numLoops(); ++j) {
                S2Loop bLoop = b.loop(j);
                if (bLoop.depth() != aLoop.depth() || !bLoop.boundaryApproxEquals(aLoop, maxError)) continue;
                success = true;
                break;
            }
            if (success) continue;
            return false;
        }
        return true;
    }

    @Override
    public S2Cap getCapBound() {
        return this.bound.getCapBound();
    }

    @Override
    public S2LatLngRect getRectBound() {
        return this.bound;
    }

    @Override
    public boolean contains(S2Cell cell) {
        if (this.numLoops() == 1) {
            return this.loop(0).contains(cell);
        }
        S2LatLngRect cellBound = cell.getRectBound();
        if (!this.bound.contains(cellBound)) {
            return false;
        }
        S2Loop cellLoop = new S2Loop(cell, cellBound);
        S2Polygon cellPoly = new S2Polygon(cellLoop);
        return this.contains(cellPoly);
    }

    @Override
    public boolean mayIntersect(S2Cell cell) {
        if (this.numLoops() == 1) {
            return this.loop(0).mayIntersect(cell);
        }
        S2LatLngRect cellBound = cell.getRectBound();
        if (!this.bound.intersects(cellBound)) {
            return false;
        }
        S2Loop cellLoop = new S2Loop(cell, cellBound);
        S2Polygon cellPoly = new S2Polygon(cellLoop);
        return this.intersects(cellPoly);
    }

    public boolean contains(S2Point p) {
        if (this.numLoops() == 1) {
            return this.loop(0).contains(p);
        }
        if (!this.bound.contains(p)) {
            return false;
        }
        boolean inside = false;
        for (int i = 0; i < this.numLoops() && (!(inside ^= this.loop(i).contains(p)) || this.hasHoles); ++i) {
        }
        return inside;
    }

    private static void sortValueLoops(Map<S2Loop, List<S2Loop>> loopMap) {
        for (S2Loop key : loopMap.keySet()) {
            Collections.sort(loopMap.get(key));
        }
    }

    private static void insertLoop(S2Loop newLoop, S2Loop parent, Map<S2Loop, List<S2Loop>> loopMap) {
        List<S2Loop> children = loopMap.get(parent);
        if (children == null) {
            children = new ArrayList<S2Loop>();
            loopMap.put(parent, children);
        }
        for (S2Loop child : children) {
            if (!child.containsNested(newLoop)) continue;
            S2Polygon.insertLoop(newLoop, child, loopMap);
            return;
        }
        List<S2Loop> newChildren = loopMap.get(newLoop);
        int i = 0;
        while (i < children.size()) {
            S2Loop child = children.get(i);
            if (newLoop.containsNested(child)) {
                if (newChildren == null) {
                    newChildren = new ArrayList<S2Loop>();
                    loopMap.put(newLoop, newChildren);
                }
                newChildren.add(child);
                children.remove(i);
                continue;
            }
            ++i;
        }
        children.add(newLoop);
    }

    private void initLoop(S2Loop loop, int depth, Map<S2Loop, List<S2Loop>> loopMap) {
        List<S2Loop> children;
        if (loop != null) {
            loop.setDepth(depth);
            this.loops.add(loop);
        }
        if ((children = loopMap.get(loop)) != null) {
            for (S2Loop child : children) {
                this.initLoop(child, depth + 1, loopMap);
            }
        }
    }

    private int containsOrCrosses(S2Loop b) {
        boolean inside = false;
        for (int i = 0; i < this.numLoops(); ++i) {
            int result = this.loop(i).containsOrCrosses(b);
            if (result < 0) {
                return -1;
            }
            if (result <= 0) continue;
            inside ^= true;
        }
        return inside ? 1 : 0;
    }

    private boolean anyLoopContains(S2Loop b) {
        for (int i = 0; i < this.numLoops(); ++i) {
            if (!this.loop(i).contains(b)) continue;
            return true;
        }
        return false;
    }

    private boolean containsAllShells(S2Polygon b) {
        for (int j = 0; j < b.numLoops(); ++j) {
            if (b.loop(j).sign() < 0 || this.containsOrCrosses(b.loop(j)) > 0) continue;
            return false;
        }
        return true;
    }

    private boolean excludesAllHoles(S2Polygon b) {
        for (int j = 0; j < b.numLoops(); ++j) {
            if (b.loop(j).sign() > 0 || this.containsOrCrosses(b.loop(j)) == 0) continue;
            return false;
        }
        return true;
    }

    private boolean intersectsAnyShell(S2Polygon b) {
        for (int j = 0; j < b.numLoops(); ++j) {
            if (b.loop(j).sign() < 0 || this.containsOrCrosses(b.loop(j)) == 0) continue;
            return true;
        }
        return false;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Polygon: (").append(this.numLoops()).append(") loops:\n");
        for (int i = 0; i < this.numLoops(); ++i) {
            S2Loop s2Loop = this.loop(i);
            sb.append("loop <\n");
            for (int v = 0; v < s2Loop.numVertices(); ++v) {
                S2Point s2Point = s2Loop.vertex(v);
                sb.append(s2Point.toDegreesString());
                sb.append("\n");
            }
            sb.append(">\n");
        }
        return sb.toString();
    }

    private strictfp static final class ParametrizedS2Point
    implements Comparable<ParametrizedS2Point> {
        private final double time;
        private final S2Point point;

        public ParametrizedS2Point(double time, S2Point point) {
            this.time = time;
            this.point = point;
        }

        public double getTime() {
            return this.time;
        }

        public S2Point getPoint() {
            return this.point;
        }

        @Override
        public int compareTo(ParametrizedS2Point o) {
            int compareTime = Double.compare(this.time, o.time);
            if (compareTime != 0) {
                return compareTime;
            }
            return this.point.compareTo(o.point);
        }
    }

    private strictfp static final class LoopVertexIndexPair {
        private final int loopIndex;
        private final int vertexIndex;

        public LoopVertexIndexPair(int loopIndex, int vertexIndex) {
            this.loopIndex = loopIndex;
            this.vertexIndex = vertexIndex;
        }

        public int getLoopIndex() {
            return this.loopIndex;
        }

        public int getVertexIndex() {
            return this.vertexIndex;
        }
    }

    private strictfp static final class UndirectedEdge {
        private final S2Point a;
        private final S2Point b;

        public UndirectedEdge(S2Point start, S2Point end) {
            this.a = start;
            this.b = end;
        }

        public S2Point getStart() {
            return this.a;
        }

        public S2Point getEnd() {
            return this.b;
        }

        public String toString() {
            return String.format("Edge: (%s <-> %s)\n   or [%s <-> %s]", this.a.toDegreesString(), this.b.toDegreesString(), this.a, this.b);
        }

        public boolean equals(Object o) {
            if (o == null || !(o instanceof UndirectedEdge)) {
                return false;
            }
            UndirectedEdge other = (UndirectedEdge)o;
            return this.getStart().equals(other.getStart()) && this.getEnd().equals(other.getEnd()) || this.getStart().equals(other.getEnd()) && this.getEnd().equals(other.getStart());
        }

        public int hashCode() {
            return this.getStart().hashCode() + this.getEnd().hashCode();
        }
    }

    private strictfp static final class S2PolygonIndex
    extends S2LoopSequenceIndex {
        private final S2Polygon poly;
        private final boolean reverse;

        private static int[] getVertices(S2Polygon poly) {
            int[] vertices = new int[poly.numLoops()];
            for (int i = 0; i < vertices.length; ++i) {
                vertices[i] = poly.loop(i).numVertices();
            }
            return vertices;
        }

        public S2PolygonIndex(S2Polygon poly, boolean reverse) {
            super(S2PolygonIndex.getVertices(poly));
            this.poly = poly;
            this.reverse = reverse;
        }

        @Override
        public S2Edge edgeFromTo(int index) {
            int toIndex;
            int fromIndex;
            LoopVertexIndexPair indices = this.decodeIndex(index);
            int loopIndex = indices.getLoopIndex();
            int vertexInLoop = indices.getVertexIndex();
            S2Loop loop = this.poly.loop(loopIndex);
            if (loop.isHole() ^ this.reverse) {
                fromIndex = loop.numVertices() - 1 - vertexInLoop;
                toIndex = 2 * loop.numVertices() - 2 - vertexInLoop;
            } else {
                fromIndex = vertexInLoop;
                toIndex = vertexInLoop + 1;
            }
            S2Point from = loop.vertex(fromIndex);
            S2Point to = loop.vertex(toIndex);
            return new S2Edge(from, to);
        }
    }

    private strictfp static abstract class S2LoopSequenceIndex
    extends S2EdgeIndex {
        private final int[] indexToLoop;
        private final int[] loopToFirstIndex;

        public S2LoopSequenceIndex(int[] numVertices) {
            int totalEdges = 0;
            for (int edges : numVertices) {
                totalEdges += edges;
            }
            this.indexToLoop = new int[totalEdges];
            this.loopToFirstIndex = new int[numVertices.length];
            totalEdges = 0;
            for (int j = 0; j < numVertices.length; ++j) {
                this.loopToFirstIndex[j] = totalEdges;
                for (int i = 0; i < numVertices[j]; ++i) {
                    this.indexToLoop[totalEdges] = j;
                    ++totalEdges;
                }
            }
        }

        public final LoopVertexIndexPair decodeIndex(int index) {
            int loopIndex = this.indexToLoop[index];
            int vertexInLoop = index - this.loopToFirstIndex[loopIndex];
            return new LoopVertexIndexPair(loopIndex, vertexInLoop);
        }

        public abstract S2Edge edgeFromTo(int var1);

        @Override
        public final int getNumEdges() {
            return this.indexToLoop.length;
        }

        @Override
        public S2Point edgeFrom(int index) {
            S2Edge fromTo = this.edgeFromTo(index);
            S2Point from = fromTo.getStart();
            return from;
        }

        @Override
        protected S2Point edgeTo(int index) {
            S2Edge fromTo = this.edgeFromTo(index);
            S2Point to = fromTo.getEnd();
            return to;
        }
    }
}

