﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/inspector/InspectorRequest.h>
#include <aws/inspector/Inspector_EXPORTS.h>
#include <aws/inspector/model/Attribute.h>

#include <utility>

namespace Aws {
namespace Inspector {
namespace Model {

/**
 */
class CreateAssessmentTemplateRequest : public InspectorRequest {
 public:
  AWS_INSPECTOR_API CreateAssessmentTemplateRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateAssessmentTemplate"; }

  AWS_INSPECTOR_API Aws::String SerializePayload() const override;

  AWS_INSPECTOR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN that specifies the assessment target for which you want to create the
   * assessment template.</p>
   */
  inline const Aws::String& GetAssessmentTargetArn() const { return m_assessmentTargetArn; }
  inline bool AssessmentTargetArnHasBeenSet() const { return m_assessmentTargetArnHasBeenSet; }
  template <typename AssessmentTargetArnT = Aws::String>
  void SetAssessmentTargetArn(AssessmentTargetArnT&& value) {
    m_assessmentTargetArnHasBeenSet = true;
    m_assessmentTargetArn = std::forward<AssessmentTargetArnT>(value);
  }
  template <typename AssessmentTargetArnT = Aws::String>
  CreateAssessmentTemplateRequest& WithAssessmentTargetArn(AssessmentTargetArnT&& value) {
    SetAssessmentTargetArn(std::forward<AssessmentTargetArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user-defined name that identifies the assessment template that you want
   * to create. You can create several assessment templates for an assessment target.
   * The names of the assessment templates that correspond to a particular assessment
   * target must be unique.</p>
   */
  inline const Aws::String& GetAssessmentTemplateName() const { return m_assessmentTemplateName; }
  inline bool AssessmentTemplateNameHasBeenSet() const { return m_assessmentTemplateNameHasBeenSet; }
  template <typename AssessmentTemplateNameT = Aws::String>
  void SetAssessmentTemplateName(AssessmentTemplateNameT&& value) {
    m_assessmentTemplateNameHasBeenSet = true;
    m_assessmentTemplateName = std::forward<AssessmentTemplateNameT>(value);
  }
  template <typename AssessmentTemplateNameT = Aws::String>
  CreateAssessmentTemplateRequest& WithAssessmentTemplateName(AssessmentTemplateNameT&& value) {
    SetAssessmentTemplateName(std::forward<AssessmentTemplateNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The duration of the assessment run in seconds.</p>
   */
  inline int GetDurationInSeconds() const { return m_durationInSeconds; }
  inline bool DurationInSecondsHasBeenSet() const { return m_durationInSecondsHasBeenSet; }
  inline void SetDurationInSeconds(int value) {
    m_durationInSecondsHasBeenSet = true;
    m_durationInSeconds = value;
  }
  inline CreateAssessmentTemplateRequest& WithDurationInSeconds(int value) {
    SetDurationInSeconds(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARNs that specify the rules packages that you want to attach to the
   * assessment template.</p>
   */
  inline const Aws::Vector<Aws::String>& GetRulesPackageArns() const { return m_rulesPackageArns; }
  inline bool RulesPackageArnsHasBeenSet() const { return m_rulesPackageArnsHasBeenSet; }
  template <typename RulesPackageArnsT = Aws::Vector<Aws::String>>
  void SetRulesPackageArns(RulesPackageArnsT&& value) {
    m_rulesPackageArnsHasBeenSet = true;
    m_rulesPackageArns = std::forward<RulesPackageArnsT>(value);
  }
  template <typename RulesPackageArnsT = Aws::Vector<Aws::String>>
  CreateAssessmentTemplateRequest& WithRulesPackageArns(RulesPackageArnsT&& value) {
    SetRulesPackageArns(std::forward<RulesPackageArnsT>(value));
    return *this;
  }
  template <typename RulesPackageArnsT = Aws::String>
  CreateAssessmentTemplateRequest& AddRulesPackageArns(RulesPackageArnsT&& value) {
    m_rulesPackageArnsHasBeenSet = true;
    m_rulesPackageArns.emplace_back(std::forward<RulesPackageArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user-defined attributes that are assigned to every finding that is
   * generated by the assessment run that uses this assessment template. An attribute
   * is a key and value pair (an <a>Attribute</a> object). Within an assessment
   * template, each key must be unique.</p>
   */
  inline const Aws::Vector<Attribute>& GetUserAttributesForFindings() const { return m_userAttributesForFindings; }
  inline bool UserAttributesForFindingsHasBeenSet() const { return m_userAttributesForFindingsHasBeenSet; }
  template <typename UserAttributesForFindingsT = Aws::Vector<Attribute>>
  void SetUserAttributesForFindings(UserAttributesForFindingsT&& value) {
    m_userAttributesForFindingsHasBeenSet = true;
    m_userAttributesForFindings = std::forward<UserAttributesForFindingsT>(value);
  }
  template <typename UserAttributesForFindingsT = Aws::Vector<Attribute>>
  CreateAssessmentTemplateRequest& WithUserAttributesForFindings(UserAttributesForFindingsT&& value) {
    SetUserAttributesForFindings(std::forward<UserAttributesForFindingsT>(value));
    return *this;
  }
  template <typename UserAttributesForFindingsT = Attribute>
  CreateAssessmentTemplateRequest& AddUserAttributesForFindings(UserAttributesForFindingsT&& value) {
    m_userAttributesForFindingsHasBeenSet = true;
    m_userAttributesForFindings.emplace_back(std::forward<UserAttributesForFindingsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_assessmentTargetArn;

  Aws::String m_assessmentTemplateName;

  int m_durationInSeconds{0};

  Aws::Vector<Aws::String> m_rulesPackageArns;

  Aws::Vector<Attribute> m_userAttributesForFindings;
  bool m_assessmentTargetArnHasBeenSet = false;
  bool m_assessmentTemplateNameHasBeenSet = false;
  bool m_durationInSecondsHasBeenSet = false;
  bool m_rulesPackageArnsHasBeenSet = false;
  bool m_userAttributesForFindingsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Inspector
}  // namespace Aws
