"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2026.3.1"
__checksum__ = "49cd10040f53526b80e2ce8e045738261bcec48e04524d5299dada453cb3065f"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'amazon', b'android', b'app', b'audible', b'azure', b'bank', b'bing', b'boo', b'channel', b'chrome', b'dad', b'day', b'dev', b'eat', b'esq', b'fire', b'fly', b'foo', b'fujitsu', b'gle', b'gmail', b'google', b'hangout', b'hotmail', b'imdb', b'ing', b'insurance', b'kindle', b'meet', b'meme', b'microsoft', b'mov', b'new', b'nexus', b'office', b'page', b'phd', b'play', b'prime', b'prof', b'rsvp', b'search', b'silk', b'skype', b'windows', b'xbox', b'xn--cckwcxetd', b'xn--jlq480n2rg', b'youtube', b'zappos', b'zip'}  # noqa: E501
_JUMPTABLE = [[(0, 4), (4, 10), (14, 19), (33, 42), (75, 35), (110, 12), (122, 8), (130, 19), (149, 22), (171, 7), (178, 13), (191, 25), None, (216, 23), (239, 37), (276, 19), (295, 17), (312, 36), (348, 14), (362, 10), (372, 18), None, (390, 51), (441, 8), (449, 34), (483, 19), (502, 13), (515, 14), (529, 7), None, None, (536, 33), (569, 20), (589, 25), (614, 14), (628, 14), (642, 28), None, (670, 18), (688, 30), (718, 8), (726, 13), (739, 10), (749, 8), (757, 17), None, (774, 26), (800, 16), (816, 5), (821, 10), (831, 14), (845, 11), (856, 12), (868, 27), (895, 7), (902, 11), (913, 5), (918, 5), (923, 20), (943, 8), (951, 27), (978, 46), (1024, 25), (1049, 16), (1065, 8), (1073, 5), (1078, 22), (1100, 18), None, (1118, 32), (1150, 15), (1165, 24), (1189, 11), None, (1200, 10), (1210, 19), (1229, 18), (1247, 18), (1265, 5), (1270, 20), (1290, 26), (1316, 35), (1351, 19), (1370, 20), (1390, 46), (1436, 12), (1448, 14), (1462, 8), (1470, 8), (1478, 10), (1488, 13), (1501, 20), (1521, 19), None, (1540, 13), (1553, 26), (1579, 11), (1590, 4), (1594, 22), (1616, 19), (1635, 15), (1650, 9), (1659, 19), (1678, 11), (1689, 21), (1710, 12), (1722, 25), None, (1747, 10), (1757, 14), (1771, 26), (1797, 45), (1842, 15), None, (1857, 11), (1868, 30), (1898, 21), (1919, 26), (1945, 6), (1951, 6), (1957, 20), (1977, 5), (1982, 38), (2020, 23), (2043, 6), (2049, 13), (2062, 8), (2070, 19), (2089, 12), (2101, 70), (2171, 55), None, (2226, 23), (2249, 16), (2265, 14), None, (2279, 24), (2303, 32), (2335, 6), (2341, 41), (2382, 22), (2404, 23), (2427, 23), (2450, 12), (2462, 8), (2470, 24), (2494, 12), (2506, 32), (2538, 25), (2563, 15), None, (2578, 46), (2624, 28), (2652, 13), (2665, 18), (2683, 26), (2709, 5), (2714, 42), (2756, 30), (2786, 16), (2802, 33), (2835, 17), (2852, 23), (2875, 14), (2889, 25), (2914, 19), (2933, 14), (2947, 7), (2954, 37), None, (2991, 18), (3009, 32), (3041, 27), (3068, 17), (3085, 30), (3115, 12), (3127, 28), (3155, 35), (3190, 12), (3202, 57), (3259, 32), (3291, 25), None, (3316, 8), (3324, 25), (3349, 18), (3367, 6), (3373, 23), None, (3396, 36), (3432, 33), (3465, 14), (3479, 16), (3495, 22), None, (3517, 20), (3537, 45), (3582, 34), (3616, 15), (3631, 20), (3651, 16), (3667, 21), (3688, 32), (3720, 24), (3744, 20), (3764, 10), (3774, 44), (3818, 6), (3824, 9), (3833, 12), (3845, 18), (3863, 5), (3868, 10), (3878, 39), (3917, 36), None, (3953, 25), (3978, 28), None, (4006, 14), (4020, 8), (4028, 17), None, (4045, 25), (4070, 17), None, (4087, 21), (4108, 35), (4143, 21), (4164, 10), (4174, 48), (4222, 11), (4233, 37), (4270, 46), (4316, 23), (4339, 12), (4351, 14), (4365, 23), (4388, 29), (4417, 14), (4431, 8), (4439, 47), (4486, 35), None, None, (4521, 47), (4568, 42), None, (4610, 18), None, (4628, 20), (4648, 8), (4656, 34), (4690, 6), (4696, 40), (4736, 22)], [(4758, 5324), (10082, 5067), (15149, 5699), (20848, 6023), (26871, 5879), (32750, 5153), (37903, 5811), (43714, 5341), (49055, 5169), (54224, 4891), (59115, 5234), (64349, 5414), (69763, 4925), (74688, 5766), (80454, 5543), (85997, 5505), (91502, 5069), (96571, 4820), (101391, 5366), (106757, 5114), (111871, 5189), (117060, 5284), (122344, 5799), (128143, 5094), (133237, 5515), (138752, 4988), (143740, 5321), (149061, 5493), (154554, 5227), (159781, 5485), (165266, 5169), (170435, 6002), (176437, 5280), (181717, 4993), (186710, 5018), (191728, 5825), (197553, 5413), (202966, 5747), (208713, 5805), (214518, 5706), (220224, 6283), (226507, 4897), (231404, 5342), (236746, 5344), (242090, 4751), (246841, 5569), (252410, 4877), (257287, 5969), (263256, 5040), (268296, 5236), (273532, 5768), (279300, 5260), (284560, 5199), (289759, 5490), (295249, 4965), (300214, 4980), (305194, 5891), (311085, 5769), (316854, 5215), (322069, 5000), (327069, 4834), (331903, 5539), (337442, 5469), (342911, 5600), (348511, 5902), (354413, 4604), (359017, 5516), (364533, 5895), (370428, 5655), (376083, 4923), (381006, 5151), (386157, 5452), (391609, 4969), (396578, 5534), (402112, 5226), (407338, 5943), (413281, 5065), (418346, 5714), (424060, 5570), (429630, 4980), (434610, 5160), (439770, 5200), (444970, 5029), (449999, 5209), (455208, 5595), (460803, 5610), (466413, 5742), (472155, 5125), (477280, 4964), (482244, 5543), (487787, 5977), (493764, 5663), (499427, 4831), (504258, 5056), (509314, 5243), (514557, 4807), (519364, 5682), (525046, 4726), (529772, 5744), (535516, 5071), (540587, 5268), (545855, 5258), (551113, 5475), (556588, 5272), (561860, 4755), (566615, 5230), (571845, 5589), (577434, 5177), (582611, 5143), (587754, 4395), (592149, 5402), (597551, 5235), (602786, 4656), (607442, 5339), (612781, 5485), (618266, 5127), (623393, 5614), (629007, 4957), (633964, 5661), (639625, 5486), (645111, 5796), (650907, 4980), (655887, 5577), (661464, 5508), (666972, 5427), (672399, 5948), (678347, 5534), (683881, 5238), (689119, 5155), (694274, 5510), (699784, 5875), (705659, 5373), (711032, 5290), (716322, 5105), (721427, 5090), (726517, 5800), (732317, 5515), (737832, 5984), (743816, 5503), (749319, 5565), (754884, 5963), (760847, 5601), (766448, 5587), (772035, 5790), (777825, 5679), (783504, 4986), (788490, 5481), (793971, 5469), (799440, 5074), (804514, 5259), (809773, 5130), (814903, 5431), (820334, 4960), (825294, 4959), (830253, 5528), (835781, 5413), (841194, 5234), (846428, 5247), (851675, 5441), (857116, 4992), (862108, 5356), (867464, 5310), (872774, 5434), (878208, 5285), (883493, 5157), (888650, 5763), (894413, 5715), (900128, 4839), (904967, 5733), (910700, 5165), (915865, 4968), (920833, 5369), (926202, 5222), (931424, 5663), (937087, 5143), (942230, 5056), (947286, 6164), (953450, 4766), (958216, 5061), (963277, 4839), (968116, 7067), (975183, 5504), (980687, 5476), (986163, 5379), (991542, 5641), (997183, 4995), (1002178, 4783), (1006961, 6037), (1012998, 5301), (1018299, 5395), (1023694, 5672), (1029366, 4878), (1034244, 5041), (1039285, 4798), (1044083, 5389), (1049472, 5545), (1055017, 5136), (1060153, 5228), (1065381, 4850), (1070231, 5467), (1075698, 5563), (1081261, 5821), (1087082, 5636), (1092718, 4949), (1097667, 5460), (1103127, 5831), (1108958, 5752), (1114710, 5537), (1120247, 6307), (1126554, 5077), (1131631, 6464), (1138095, 5301), (1143396, 5390), (1148786, 4862), (1153648, 5910), (1159558, 5576), (1165134, 5368), (1170502, 5824), (1176326, 4970), (1181296, 5600), (1186896, 5181), (1192077, 4687), (1196764, 5277), (1202041, 5265), (1207306, 5671), (1212977, 5087), (1218064, 6224), (1224288, 5433), (1229721, 5561), (1235282, 5640), (1240922, 5679), (1246601, 5030), (1251631, 5214), (1256845, 5545), (1262390, 4959), (1267349, 5714), (1273063, 5110), (1278173, 5020), (1283193, 4467), (1287660, 6079), (1293739, 4818), (1298557, 5252), (1303809, 5017), (1308826, 5789), (1314615, 4961), (1319576, 6166), (1325742, 5562), (1331304, 5259), (1336563, 5248), (1341811, 5049), (1346860, 5340), (1352200, 5543), (1357743, 4940), (1362683, 5258), (1367941, 5397), (1373338, 4900)], [(1378238, 738), (1378976, 453), (1379429, 771), (1380200, 713), (1380913, 619), (1381532, 655), (1382187, 511), (1382698, 516), (1383214, 573), (1383787, 595), (1384382, 462), (1384844, 528), (1385372, 681), (1386053, 764), (1386817, 556), (1387373, 699), (1388072, 1161), (1389233, 692), (1389925, 586), (1390511, 538), (1391049, 677), (1391726, 492), (1392218, 400), (1392618, 691), (1393309, 607), (1393916, 580), (1394496, 529), (1395025, 1114), (1396139, 609), (1396748, 626), (1397374, 431), (1397805, 670), (1398475, 551), (1399026, 554), (1399580, 700), (1400280, 699), (1400979, 448), (1401427, 615), (1402042, 458), (1402500, 572), (1403072, 599), (1403671, 530), (1404201, 468), (1404669, 604), (1405273, 544), (1405817, 538), (1406355, 586), (1406941, 509), (1407450, 428), (1407878, 516), (1408394, 615), (1409009, 490), (1409499, 787), (1410286, 793), (1411079, 1088), (1412167, 588), (1412755, 750), (1413505, 615), (1414120, 393), (1414513, 611), (1415124, 751), (1415875, 534), (1416409, 925), (1417334, 801), (1418135, 739), (1418874, 635), (1419509, 723), (1420232, 518), (1420750, 477), (1421227, 724), (1421951, 609), (1422560, 604), (1423164, 257), (1423421, 570), (1423991, 532), (1424523, 542), (1425065, 373), (1425438, 677), (1426115, 723), (1426838, 427), (1427265, 565), (1427830, 584), (1428414, 703), (1429117, 633), (1429750, 609), (1430359, 713), (1431072, 615), (1431687, 528), (1432215, 707), (1432922, 537), (1433459, 874), (1434333, 786), (1435119, 524), (1435643, 507), (1436150, 584), (1436734, 393), (1437127, 685), (1437812, 515), (1438327, 327), (1438654, 675), (1439329, 547), (1439876, 558), (1440434, 459), (1440893, 521), (1441414, 466), (1441880, 481), (1442361, 580), (1442941, 426), (1443367, 501), (1443868, 568), (1444436, 595), (1445031, 904), (1445935, 849), (1446784, 651), (1447435, 575), (1448010, 472), (1448482, 559), (1449041, 750), (1449791, 513), (1450304, 527), (1450831, 822), (1451653, 547), (1452200, 640), (1452840, 832), (1453672, 638), (1454310, 781), (1455091, 475), (1455566, 575), (1456141, 522), (1456663, 742), (1457405, 651), (1458056, 506), (1458562, 540), (1459102, 570), (1459672, 406), (1460078, 652), (1460730, 732), (1461462, 517), (1461979, 636), (1462615, 730), (1463345, 592), (1463937, 451), (1464388, 557), (1464945, 394), (1465339, 705), (1466044, 612), (1466656, 722), (1467378, 366), (1467744, 637), (1468381, 687), (1469068, 493), (1469561, 436), (1469997, 665), (1470662, 623), (1471285, 685), (1471970, 551), (1472521, 352), (1472873, 724), (1473597, 687), (1474284, 686), (1474970, 740), (1475710, 476), (1476186, 719), (1476905, 611), (1477516, 746), (1478262, 745), (1479007, 538), (1479545, 690), (1480235, 531), (1480766, 540), (1481306, 592), (1481898, 456), (1482354, 1001), (1483355, 678), (1484033, 867), (1484900, 664), (1485564, 556), (1486120, 614), (1486734, 677), (1487411, 584), (1487995, 1688), (1489683, 553), (1490236, 586), (1490822, 616), (1491438, 725), (1492163, 512), (1492675, 495), (1493170, 496), (1493666, 589), (1494255, 811), (1495066, 451), (1495517, 438), (1495955, 422), (1496377, 870), (1497247, 492), (1497739, 675), (1498414, 756), (1499170, 699), (1499869, 657), (1500526, 649), (1501175, 556), (1501731, 747), (1502478, 582), (1503060, 657), (1503717, 602), (1504319, 651), (1504970, 787), (1505757, 758), (1506515, 625), (1507140, 670), (1507810, 650), (1508460, 621), (1509081, 675), (1509756, 1012), (1510768, 553), (1511321, 635), (1511956, 601), (1512557, 457), (1513014, 374), (1513388, 272), (1513660, 504), (1514164, 565), (1514729, 556), (1515285, 645), (1515930, 437), (1516367, 495), (1516862, 574), (1517436, 654), (1518090, 651), (1518741, 510), (1519251, 755), (1520006, 500), (1520506, 776), (1521282, 585), (1521867, 542), (1522409, 480), (1522889, 785), (1523674, 666), (1524340, 731), (1525071, 695), (1525766, 524), (1526290, 547), (1526837, 614), (1527451, 552), (1528003, 638), (1528641, 398), (1529039, 469), (1529508, 767), (1530275, 500), (1530775, 592), (1531367, 1958), (1533325, 483), (1533808, 781), (1534589, 553), (1535142, 567), (1535709, 718)], [(1536427, 48), None, None, (1536475, 42), None, (1536517, 27), (1536544, 25), None, None, None, None, None, None, None, None, None, None, (1536569, 42), (1536611, 20), None, (1536631, 16), None, None, (1536647, 18), None, (1536665, 23), None, None, None, None, None, None, (1536688, 21), (1536709, 25), None, None, (1536734, 26), None, None, None, None, (1536760, 44), (1536804, 21), (1536825, 23), None, None, None, None, None, None, None, None, None, None, (1536848, 31), (1536879, 19), None, None, None, (1536898, 42), (1536940, 19), (1536959, 16), None, (1536975, 21), None, None, (1536996, 42), None, None, (1537038, 42), (1537080, 27), None, None, None, None, (1537107, 37), (1537144, 21), None, None, (1537165, 20), (1537185, 42), None, None, None, (1537227, 25), (1537252, 17), None, (1537269, 21), (1537290, 21), None, None, (1537311, 24), None, (1537335, 24), (1537359, 21), None, None, (1537380, 22), None, None, None, (1537402, 19), None, None, None, None, (1537421, 31), None, None, None, None, None, None, None, None, (1537452, 42), (1537494, 42), (1537536, 17), (1537553, 17), None, None, (1537570, 19), None, (1537589, 25), None, None, (1537614, 20), None, None, (1537634, 42), (1537676, 63), None, None, None, (1537739, 21), None, None, None, None, (1537760, 21), (1537781, 16), (1537797, 17), (1537814, 31), None, None, None, None, (1537845, 42), None, (1537887, 25), None, (1537912, 9), None, (1537921, 21), (1537942, 42), None, None, (1537984, 65), (1538049, 56), (1538105, 21), None, (1538126, 42), None, None, (1538168, 24), None, None, None, None, None, None, (1538192, 42), (1538234, 21), (1538255, 21), None, (1538276, 42), (1538318, 25), None, (1538343, 37), (1538380, 21), (1538401, 42), None, None, (1538443, 21), (1538464, 16), None, None, (1538480, 16), None, (1538496, 51), None, None, (1538547, 21), None, (1538568, 22), (1538590, 21), (1538611, 21), None, None, (1538632, 63), (1538695, 41), (1538736, 39), (1538775, 42), None, None, None, None, None, None, (1538817, 21), (1538838, 21), None, None, (1538859, 21), None, None, (1538880, 21), None, None, None, (1538901, 50), None, None, None, (1538951, 50), None, (1539001, 21), (1539022, 21), (1539043, 19), None, (1539062, 16), (1539078, 26), None, (1539104, 58), (1539162, 42), None, None, None, None, None, None, (1539204, 41), None, None, None, (1539245, 21), None, None, (1539266, 44), None, (1539310, 254), (1539564, 21), None, None, (1539585, 21), None], [None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)

except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path),
            "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 5 deep.
            if layer > 4:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
