﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/workmail/WorkMailRequest.h>
#include <aws/workmail/WorkMail_EXPORTS.h>

#include <utility>

namespace Aws {
namespace WorkMail {
namespace Model {

/**
 */
class AssociateMemberToGroupRequest : public WorkMailRequest {
 public:
  AWS_WORKMAIL_API AssociateMemberToGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssociateMemberToGroup"; }

  AWS_WORKMAIL_API Aws::String SerializePayload() const override;

  AWS_WORKMAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The organization under which the group exists.</p>
   */
  inline const Aws::String& GetOrganizationId() const { return m_organizationId; }
  inline bool OrganizationIdHasBeenSet() const { return m_organizationIdHasBeenSet; }
  template <typename OrganizationIdT = Aws::String>
  void SetOrganizationId(OrganizationIdT&& value) {
    m_organizationIdHasBeenSet = true;
    m_organizationId = std::forward<OrganizationIdT>(value);
  }
  template <typename OrganizationIdT = Aws::String>
  AssociateMemberToGroupRequest& WithOrganizationId(OrganizationIdT&& value) {
    SetOrganizationId(std::forward<OrganizationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The group to which the member (user or group) is associated.</p> <p>The
   * identifier can accept <i>GroupId</i>, <i>Groupname</i>, or <i>email</i>. The
   * following identity formats are available:</p> <ul> <li> <p>Group ID:
   * 12345678-1234-1234-1234-123456789012 or
   * S-1-1-12-1234567890-123456789-123456789-1234</p> </li> <li> <p>Email address:
   * group@domain.tld</p> </li> <li> <p>Group name: group</p> </li> </ul>
   */
  inline const Aws::String& GetGroupId() const { return m_groupId; }
  inline bool GroupIdHasBeenSet() const { return m_groupIdHasBeenSet; }
  template <typename GroupIdT = Aws::String>
  void SetGroupId(GroupIdT&& value) {
    m_groupIdHasBeenSet = true;
    m_groupId = std::forward<GroupIdT>(value);
  }
  template <typename GroupIdT = Aws::String>
  AssociateMemberToGroupRequest& WithGroupId(GroupIdT&& value) {
    SetGroupId(std::forward<GroupIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The member (user or group) to associate to the group.</p> <p>The member ID
   * can accept <i>UserID or GroupId</i>, <i>Username or Groupname</i>, or
   * <i>email</i>.</p> <ul> <li> <p>Member: 12345678-1234-1234-1234-123456789012 or
   * S-1-1-12-1234567890-123456789-123456789-1234</p> </li> <li> <p>Email address:
   * member@domain.tld</p> </li> <li> <p>Member name: member</p> </li> </ul>
   */
  inline const Aws::String& GetMemberId() const { return m_memberId; }
  inline bool MemberIdHasBeenSet() const { return m_memberIdHasBeenSet; }
  template <typename MemberIdT = Aws::String>
  void SetMemberId(MemberIdT&& value) {
    m_memberIdHasBeenSet = true;
    m_memberId = std::forward<MemberIdT>(value);
  }
  template <typename MemberIdT = Aws::String>
  AssociateMemberToGroupRequest& WithMemberId(MemberIdT&& value) {
    SetMemberId(std::forward<MemberIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_organizationId;

  Aws::String m_groupId;

  Aws::String m_memberId;
  bool m_organizationIdHasBeenSet = false;
  bool m_groupIdHasBeenSet = false;
  bool m_memberIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkMail
}  // namespace Aws
